@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
"%~dp0perl.exe" -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
"%~dp0perl.exe" -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/opt/bin/perl
#line 15

=head1 NAME

json_xs - JSON::XS commandline utility

=head1 SYNOPSIS

   json_xs [-v] [-f inputformat] [-t outputformat]

=head1 DESCRIPTION

F<json_xs> converts between some input and output formats (one of them is
JSON).

The default input format is C<json> and the default output format is
C<json-pretty>.

=head1 OPTIONS

=over 4

=item -v

Be slightly more verbose.

=item -f fromformat

Read a file in the given format from STDIN.

C<fromformat> can be one of:

=over 4

=item json - a json text encoded, either utf-8, utf16-be/le, utf32-be/le

=item storable - a Storable frozen value

=item storable-file - a Storable file (Storable has two incompatible formats)

=item clzf - Compress::LZF format (requires that module to be installed)

=item yaml - YAML (avoid at all costs, requires the YAML module :)

=back

=item -t toformat

Write the file in the given format to STDOUT.

C<fromformat> can be one of:

=over 4

=item json, json-utf-8 - json, utf-8 encoded

=item json-pretty - as above, but pretty-printed

=item json-utf-16le, json-utf-16be - little endian/big endian utf-16

=item json-utf-32le, json-utf-32be - little endian/big endian utf-32

=item storable - a Storable frozen value in network format

=item storable-file - a Storable file in network format (Storable has two incompatible formats)

=item clzf - Compress::LZF format

=item yaml - YAML

=back

=back

=head1 EXAMPLES

   json_xs -t null <isitreally.json

"JSON Lint" - tries to parse the file F<isitreally.json> as JSON - if it
is valid JSON, the command outputs nothing, otherwise it will print an
error message and exit with non-zero exit status.

   <src.json json_xs >pretty.json

Prettify the JSON file F<src.json> to F<dst.json>.

   json_xs -f storable-file <file

Read the serialised Storable file F<file> and print a human-readable JSON
version of it to STDOUT.

   json_xs -f storable-file -t yaml <file

Same as above, but write YAML instead (not using JSON at all :)

   lwp-request http://cpantesters.perl.org/show/JSON-XS.json | json_xs

Fetch the cpan-testers result summary C<JSON::XS> and pretty-print it.

=head1 AUTHOR

Copyright (C) 2008 Marc Lehmann <json@schmorp.de>

=cut

use strict;

use Getopt::Long;
use Storable ();
use Encode;

use JSON::XS;

my $opt_verbose;
my $opt_from = "json";
my $opt_to   = "json-pretty";

Getopt::Long::Configure ("bundling", "no_ignore_case", "require_order");

GetOptions(
   "v"   => \$opt_verbose,
   "f=s" => \$opt_from,
   "t=s" => \$opt_to,
) or die "Usage: $0 [-v] -f fromformat [-t toformat]\n";

my %F = (
   "json"          => sub {
      my $enc =
         /^\x00\x00\x00/s  ? "utf-32be"
       : /^\x00.\x00/s     ? "utf-16be"
       : /^.\x00\x00\x00/s ? "utf-32le"
       : /^.\x00.\x00/s    ? "utf-16le"
       :                     "utf-8";
      warn "input text encoding is $enc\n" if $opt_verbose;
      JSON::XS->new->decode (decode $enc, $_)
   },
   "storable"      => sub { Storable::thaw $_ },
   "storable-file" => sub { open my $fh, "<", \$_; Storable::fd_retrieve $fh },
   "clzf"          => sub { require Compress::LZF; Compress::LZF::sthaw ($_) },
   "yaml"          => sub { require YAML; YAML::Load ($_) },
);

my %T = (
   "null"          => sub { "" },
   "json"          => sub { encode_json $_ },
   "json-utf-8"    => sub { encode_json $_ },
   "json-pretty"   => sub { JSON::XS->new->utf8->pretty->encode ($_) },
   "json-utf-16le" => sub { encode "utf-16le", JSON::XS->new->encode ($_) },
   "json-utf-16be" => sub { encode "utf-16be", JSON::XS->new->encode ($_) },
   "json-utf-32le" => sub { encode "utf-32le", JSON::XS->new->encode ($_) },
   "json-utf-32be" => sub { encode "utf-32be", JSON::XS->new->encode ($_) },

   "storable"      => sub { Storable::nfreeze $_ },
   "storable-file" => sub { open my $fh, ">", \my $buf; Storable::nstore_fd $_, $fh; $buf },

   "clzf"          => sub { require Compress::LZF; Compress::LZF::sfreeze_cr ($_) },
   "yaml"          => sub { require YAML; YAML::Dump ($_) },
);

$F{$opt_from}
   or die "$opt_from: not a valid fromformat\n";

$T{$opt_to}
   or die "$opt_from: not a valid toformat\n";

{
   local $/;
   binmode STDIN; # stupid perl sometimes thinks its funny
   $_ = <STDIN>;
}

$_ = $F{$opt_from}->();
$_ = $T{$opt_to}->();

binmode STDOUT;
print $_;




__END__
:endofperl
