﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using System.IO;
using System.Diagnostics;
using junit;

namespace c4ju
{
    #region フィルター
    class FilterAttribute
    {
        public string Name;
        public string Value;
        public CSharpEval Eval;
    };
    class Filter
    {
        public string TestName;
        public string strXPath;
        public List<FilterAttribute> list;
    };
    #endregion

    class Program
    {
        static int Main(string[] args)
        {
            bool input_stdin = false;
            List<string> files = new List<string>();
            string outfile = "test_default.xml";
            string filter_xmlpath = "filter_default.xml";
            XmlDocument doc_filter = new XmlDocument();

            #region コマンドライン解析
            // コマンドライン引数の解析
            for(int i=0; i < args.Length; ++i )
            {
                string argv = args[i];
                if (argv.IndexOf("--") == 0)
                {
                    string option = argv.Substring(2);
                    string opt_out_xmlfile = "out_xmlfile";
                    string opt_filter = "filter";
                    #region 出力ファイル
                    if (option.IndexOf(opt_out_xmlfile) == 0)
                    {
                        // 出力ファイル名の指定
                        if (option.Length > 1)
                        {
                            if (option.IndexOf("=") == opt_out_xmlfile.Length)
                                outfile = option.Substring(opt_out_xmlfile.Length + 1);
                        }
                        else if( i < args.Length )
                        {
                            outfile = args[++i];
                        }
                        if( outfile[0] == '\"' )
                        {
                            outfile = outfile.Substring(1, outfile.Length - 2);
                        }
                    }
                    #endregion
                    #region ヘルプ
                    else if (option == "help")
                    {
                        Console.WriteLine(
@"c4ju [options] file1.xml ...
Process files listed on command line.
If the filenames include '-', read a list of files from standard input.

Command Line Options:
--help                  : generate this help message.
--out_xmlfile=<path>    : path of xml report.
--filter=<path>         : path of test filter xml.
"
                        );
                        return 0;
                    }
                    #endregion
                    #region フィルターファイル
                    else if (option.IndexOf(opt_filter) == 0)
                    {
                        // フィルターファイル名の指定
                        if (option.Length > 1)
                        {
                            if (option.IndexOf("=") == opt_filter.Length)
                                filter_xmlpath = option.Substring(opt_filter.Length + 1);
                        }
                        else if (i < args.Length)
                        {
                            filter_xmlpath = args[++i];
                        }

                        if (filter_xmlpath[0] == '\"')
                        {
                            filter_xmlpath = filter_xmlpath.Substring(1, filter_xmlpath.Length - 2);
                        }
                    }
                    #endregion
                }
                #region 標準入力ファイル
                else if (i == args.Length - 1 && argv.Length == 1 && argv[0] == '-')
                {
                    input_stdin = true;
                }
                #endregion
                #region 入力ファイル
                else
                {
                    files.Add(argv);
                }
                #endregion
            }
            #endregion

            #region フィルター用意
            try
            {
                doc_filter.Load(filter_xmlpath);
            }
            catch (Exception e)
            {
                Console.WriteLine(e.Message);
                return 1;
            }

            if (doc_filter.ChildNodes.Count == 0)
            {
                Console.WriteLine("not found filter.");
                return 1;
            }
            List<string> filter_xpaths = XmlToXPathList.Convert(doc_filter.SelectSingleNode("//CCCC_Project"));
            if (filter_xpaths.Count == 0)
            {
                Console.WriteLine("not found CCCC_Project node");
                return 1;
            }
            List<Filter> filter_list = new List<Filter>();
            foreach (string xpath in filter_xpaths)
            {
                Filter f = new Filter();
                XmlNode filter_node = doc_filter.SelectSingleNode(xpath);
                f.TestName = filter_node.ParentNode.Name + "/" + filter_node.Name + " : ";
                f.strXPath = xpath;
                f.list = new List<FilterAttribute>();
                foreach (XmlAttribute attr in filter_node.Attributes)
                {
                    FilterAttribute a = new FilterAttribute();
                    a.Name = attr.Name;
                    a.Value = attr.Value;
                    a.Eval = new CSharpEval();
                    if (!a.Eval.Init(attr.Value))
                    {
                        Console.WriteLine("Evaluation of the filter expression is invalid xml.");
                        return 1;
                    }
                    f.list.Add(a);
                }
                filter_list.Add(f);
            }
            #endregion

            #region 標準入力からのファイルパス取得
            if (input_stdin)
            {
                string line = null;
                while ( Console.In.Peek() != 0 && (line = Console.In.ReadLine()) != null)
                {
                    files.Add(line);
                }
            }
            #endregion

            #region テストの開始処理
            Stopwatch test_suites_sw = new Stopwatch();
            Stopwatch test_case_sw = new Stopwatch();
            Stopwatch test_sw = new Stopwatch();

            JunitXmlDocument junitDoc = new JunitXmlDocument();
            XmlNode TestSuites = junitDoc.AppendTestSuites("AllTest"
                , 0 // 仮
                , 0 // 仮
                , 0
                , test_suites_sw.Elapsed    // 仮
                );
            int test_total_count = 0;
            int failure_total_count = 0;

            test_suites_sw.Start();
            #endregion

            #region xml の解析
            foreach (string f in files)
            {
                XmlDocument xml = new XmlDocument();
                try
                {
                    xml.Load(f);

                    Console.WriteLine(f);

                    foreach (Filter filter in filter_list)
                    {
                        #region テストケース
                        try
                        {
                            string test_case_name = filter.TestName + Path.GetFileNameWithoutExtension(f);
                            XmlNodeList selectNodes = xml.SelectNodes(filter.strXPath);
                            if (selectNodes != null && selectNodes.Count > 0)
                            {
                                XmlNode TestSuite = junitDoc.AppendTestSuite(test_case_name
                                    , selectNodes.Count
                                    , 0 // 仮
                                    , 0
                                    , test_case_sw.Elapsed    // 仮
                                    );
                                int failure_count = 0;

                                test_case_sw.Reset();
                                test_case_sw.Start();
                                foreach (XmlNode node in selectNodes)
                                {
                                    #region テスト
                                    List<JunitXmlDocument.FailureResult> fr_list = new List<JunitXmlDocument.FailureResult>();
                                    XmlNode sourceNode = null;
                                    string name = node.InnerText;
                                    string file = "";
                                    int line = -1;
                                    #region テスト名・ファイルパス情報の取得
                                    XmlNode parentNode = node.ParentNode;
                                    if (parentNode.Name == "member_function")
                                    {
                                        name = parentNode.SelectSingleNode("name").InnerText;
                                        sourceNode = parentNode.SelectSingleNode("extent/source_reference");
                                    }
                                    else if (parentNode.Name == "module_summary")
                                    {
                                        name = "module_summary";
                                        sourceNode = parentNode.ParentNode.SelectSingleNode("module_detail/source_reference");
                                    }
                                    if (sourceNode != null)
                                    {
                                        file = sourceNode.Attributes["file"].Value;
                                        line = int.Parse(sourceNode.Attributes["line"].Value);
                                    }
                                    #endregion

                                    test_sw.Reset();
                                    test_sw.Start();
                                    foreach (FilterAttribute filter_attr in filter.list)
                                    {
                                        #region 属性ごとにテスト
                                        try
                                        {
                                            XmlAttribute attr = node.Attributes[filter_attr.Name];
                                            if (attr != null)
                                            {
                                                // フィルタリング用のxmlは式が記述されている
                                                // value は比較対象の属性valueに置換される。
                                                // 式評価がfalseを返した場合、テスト失敗とみなす。

                                                // parse 失敗はエラーとしない
                                                float lhs = float.Parse(attr.Value);
                                                try
                                                {
                                                    string result = filter_attr.Eval.Eval(lhs);
                                                    ++test_total_count;

                                                    if (!bool.Parse(result))
                                                    {
                                                        string expr = filter_attr.Value;
                                                        expr = expr.Replace("value", attr.Value);
                                                        JunitXmlDocument.FailureResult fr = new JunitXmlDocument.FailureResult();
                                                        fr.Message = "Expected : " + filter_attr.Value + "\n  actual : " + expr;
                                                        fr.File = file;
                                                        fr.Line = line;
                                                        fr_list.Add(fr);
                                                    }
                                                }
                                                catch (Exception e)
                                                {
                                                    Console.WriteLine(e.Message);
                                                }
                                            }
                                        }
                                        catch
                                        {
                                        }
                                        #endregion
                                    }

                                    test_sw.Stop();
                                    if (fr_list.Count > 0)
                                    {
                                        ++failure_count;
                                    }
                                    junitDoc.AppendTestCase(TestSuite
                                        , name
                                        , true
                                        , test_sw.Elapsed
                                        , fr_list
                                        );
                                    #endregion
                                }
                                test_case_sw.Stop();

                                failure_total_count += failure_count;
                                TestSuite.Attributes["failures"].Value = failure_count.ToString();
                                TestSuite.Attributes["time"].Value = test_case_sw.Elapsed.TotalSeconds.ToString();
                            }
                        }
                        catch (Exception e)
                        {
                            Console.WriteLine(e.Message);
                        }
                        #endregion
                    }
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.Message);
                }
            }
            #endregion

            #region テストの終了処理
            test_suites_sw.Stop();

            TestSuites.Attributes["tests"].Value = test_total_count.ToString();
            TestSuites.Attributes["failures"].Value = failure_total_count.ToString();
            TestSuites.Attributes["time"].Value = test_suites_sw.Elapsed.TotalSeconds.ToString();

            junitDoc.Save(outfile);
            #endregion

            return 0;
        }
    }
}

#region XmlToXPathList
class XmlToXPathList
{
    public static List<string> Convert(XmlNode node)
    {
        List<string> list = new List<string>();
        if (node.ChildNodes.Count == 0)
        {
            string path = node.Name;
            XmlNode p = node.ParentNode;
            while (p != null && p.ParentNode != null)
            {
                path = p.Name + "/" + path;
                p = p.ParentNode;
            }
            path = "//" + path;
            list.Add(path);
        }
        foreach (XmlNode n in node.ChildNodes)
        {
            list.AddRange(Convert(n));            
        }
        return list;
    }
    public static List<string> Convert(XmlDocument doc)
    {
        List<string> list = new List<string>();

        foreach (XmlNode n in doc.ChildNodes)
        {
            list.AddRange(Convert(n));
        }
        return list;
    }
}

#endregion
