package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.ErrorService;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

import java.io.PrintWriter;
import java.io.StringWriter;

/**
 * @author jum
 *
 * Implement a generic error handler that catches all errors thrown
 * by ActionListeners in the AWT event dispatcher thread.
 */
public class DefaultErrorCatcher {

	static void install() {    
        // Must use reflection to access the setProperty method, as it didn't
        // exist on 1.1.8.  Note that really doesn't even make a difference
        // on 1.1.8, because the dispatch thread doesn't consult this property,
        // but we compile against 1.1.8 to ensure everything works, so we must
        // use reflection.
        try {
            Method setPropertyMethod = System.class.getDeclaredMethod(
                "setProperty",
                new Class[] { String.class, String.class });
            setPropertyMethod.invoke(null, new String[] {
                "sun.awt.exception.handler", 
                DefaultErrorCatcher.class.getName() } );
        } catch (IllegalAccessException e1) {
            // nothing we can do
        } catch (InvocationTargetException e1) {
            // nothing we can do
        } catch (SecurityException e) {
            // nothing we can do
        } catch (NoSuchMethodException e) {
            // nothing we can do
        }
    }
	
	public void handle(Throwable ex) {
	    StringWriter sw = new StringWriter();
	    PrintWriter pw = new PrintWriter(sw);
	    ex.printStackTrace(pw);
	    pw.flush();
	    String bug = sw.toString();
	    
	    if(!isIgnorable(ex, bug))
		    ErrorService.error(ex, "Uncaught event-thread error.");
	}
	
	/**
	 * Determines if the message can be ignored.
	 */
	private boolean isIgnorable(Throwable bug, String msg) {
	    // ignore all overflows & out of memory errors,
	    // since they'll give us absolutely no debugging information
	    if(bug instanceof StackOverflowError)
	        return true;
	    if(bug instanceof OutOfMemoryError)
	        return true;
	        
        // no bug?  kinda impossible, but shouldn't report.
	    if(msg == null)
	        return true;
	        
        // frickin' repaint manager stinks.
        if(msg.indexOf("javax.swing.RepaintManager") != -1)
            return true;
        if(msg.indexOf("sun.awt.RepaintArea.paint") != -1)
            return true;
            
        if(bug instanceof ArrayIndexOutOfBoundsException) {
            if(msg.indexOf("apple.awt.CWindow.displayChanged") != -1)
                return true;
        }
	        
        // various NPEs we can ignore:
        if(bug instanceof NullPointerException) {
            if(msg.indexOf("MetalFileChooserUI") != -1)
                return true;
            if(msg.indexOf("WindowsFileChooserUI") != -1)
                return true;
            if(msg.indexOf("AquaDirectoryModel") != -1)
                return true;
            if(msg.indexOf("SizeRequirements.calculateAlignedPositions") != -1)
                return true;
            if(msg.indexOf("BasicTextUI.damageRange") != -1)
                return true;
            if(msg.indexOf("null pData") != -1)
                return true;
        }
        
        // various InternalErrors we can ignore.
        if(bug instanceof InternalError) {
            if(msg.indexOf("getGraphics not implemented for this component") != -1)
                return true;
        }

        
	    
	    // if we're not somewhere in the bug, ignore it.
	    // no need for us to debug sun's internal errors.
	    if(msg.indexOf("com.limegroup.gnutella") == -1)
	        return true;
	        
        return false;
    }
}
