 
package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.ErrorService;
import javax.swing.*;
import javax.swing.plaf.basic.*;
import java.awt.*;
import java.awt.image.*;
import javax.swing.plaf.*;
import java.net.*;
import javax.swing.plaf.metal.*;
import java.io.*;
import java.util.MissingResourceException;

/**
 * Handles the visual representation of <tt>JButton</tt> instances for the
 * application.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public class LimeButtonUI extends BasicButtonUI {

    private static final LimeButtonUI limeButtonUI = new LimeButtonUI();

	private static Icon _default0; 
	private static Icon _default1; 
	private static Icon _default2; 
	private static Icon _default3; 
	private static Icon _default4;
	private static Icon _pressed0; 
	private static Icon _pressed1; 
	private static Icon _pressed2; 
	private static Icon _pressed3;
	private static Icon _pressed4;
	private static Icon _rollover0;
    private static Icon _rollover1;
    private static Icon _rollover2;
    private static Icon _rollover3;
    private static Icon _rollover4;
    private static Color _focusColor;
	
	public LimeButtonUI() {
	    themeChanged();
	}
	

	public static void themeChanged() {
		_default0    = ResourceManager.getThemeImage("button1.gif");
		_default1    = ResourceManager.getThemeImage("button2.gif");
		_default2    = ResourceManager.getThemeImage("button3.gif");
		_default3    = ResourceManager.getThemeImage("button4.gif");
		try {
		    _default4 = ResourceManager.getThemeImage("button5.gif");
		} catch(MissingResourceException mre) {
		    _default4 = _default3;
		}
		_pressed0    = ResourceManager.getThemeImage("button1_press.gif");
		_pressed1    = ResourceManager.getThemeImage("button2_press.gif");
		_pressed2    = ResourceManager.getThemeImage("button3_press.gif");
		_pressed3    = ResourceManager.getThemeImage("button4_press.gif");
		try {
		    _pressed4 = ResourceManager.getThemeImage("button5_press.gif");
		} catch(MissingResourceException mre) {
		    _pressed4 = _pressed3;
		}
		_rollover0   = ImageManipulator.brighten(_default0);
		_rollover1   = ImageManipulator.brighten(_default1);
		_rollover2   = ImageManipulator.brighten(_default2);
		_rollover3   = ImageManipulator.brighten(_default3);
		_rollover4   = ImageManipulator.brighten(_default4);
		_focusColor = createFocusColor();
	}
	
	/**
	 * Creates a focus color by looking at the pixels in the button
	 * and inverting the color.
	 */
	private static Color createFocusColor() {
	    Image img = ((ImageIcon)_default0).getImage();
	    int[] pix = new int[1];
	    PixelGrabber pg = new PixelGrabber(img, 10, 10, 1, 1, pix, 0, 1);
	    try {
	        // returns true if pixels were grabbed succesfully.
	        if(pg.grabPixels()) {
	            // invert the color of the pixel, preserving the alpha.
	            int alpha = pix[0] & 0xFF000000;
	            int invert = ~(pix[0] & 0x00FFFFFF);
	            invert = alpha | invert;
	            return new Color(invert);
            }
	    } catch(InterruptedException ignored) {}
	    // couldn't grab pixels?  use default color.
        return UIManager.getColor("Button.focus");
    }
	
	/**
	 * Creates the <tt>ComponentUI</tt> instance for this class.
	 *
	 * @param c the component to create the ui for
	 * @return the specialized <tt>ComponentUI</tt> instance for
	 *         handling the ui for this typr of component
	 */
    public static ComponentUI createUI(JComponent c) {
        return limeButtonUI;
    }

	/**
	 * Installs the user interface elements for all future
	 * paint calls on this button.
	 *
	 * @param c the <tt>JComponent</tt> to install ui elements for,
	 *          in this case a button
	 */
  	public void installUI(JComponent c) {
		super.installUI(c);
		AbstractButton b = (AbstractButton)c;
		if(isValidButton(b) && shouldInstallUI(b)) {
			setButtonValues(b);
			setIcons(b);
		}
  	}
  	
  	/**
  	 * Removes the UI stuff that was added to this component.
  	 */
  	public void uninstallUI(JComponent c) {
  	    super.uninstallUI(c);
  	    
  	    removeButtonValues((AbstractButton)c);
  	    removeIcons((AbstractButton)c);
  	}
  	
  	/**
  	 * Determines if this is a AbstractButton that we should be mutating.
  	 */
  	private boolean isValidButton(AbstractButton b) {
  	    return (b instanceof JButton)  && 
		      !(b instanceof MediaButton) &&
		      !(b instanceof MetalComboBoxButton) &&
		      !(b instanceof BasicArrowButton) &&
		      !(b instanceof MetalScrollButton) &&
		      !(b.getText() == null) &&
		      !(b.getText().equals(""));
    }       

    /**
     * This sets the button values to the appropriate value for our
     * LimeLookAndFeel.
     * It caches the previously value as a client property so that 
     * it can be undone later.
     * This is arguably better than creating distinct UI's per each button.
     * See http://java.sun.com/products/jfc/tsc/articles/architecture/ for
     * details on Stateless vs Stateful UI delegates.
     */
	private void setButtonValues(AbstractButton b) {
	    b.putClientProperty("setBorderPainted", b.isBorderPainted() ? 
	        Boolean.TRUE : Boolean.FALSE );
		b.setBorderPainted(false);
	    b.putClientProperty("setContentAreaFilled", b.isContentAreaFilled() ?
	        Boolean.TRUE : Boolean.FALSE );
		b.setContentAreaFilled(false);
	    b.putClientProperty("setHorizontalTextPosition",
	        new Integer(b.getHorizontalTextPosition()));
		b.setHorizontalTextPosition(SwingConstants.CENTER);
	    b.putClientProperty("setFocusPainted", b.isFocusPainted() ?
	        Boolean.TRUE : Boolean.FALSE );
		b.setFocusPainted(true);
	}
	
	/**
	 * This returns the values to what they were prior to installing the UI.
	 */
	private void removeButtonValues(AbstractButton b) {
	    b.setBorderPainted( getBoolean("setBorderPainted", b) );
	    b.setContentAreaFilled( getBoolean("setContentAreaFilled", b) );
	    b.setHorizontalTextPosition(getInteger("setHorizontalTextPosition", b));
	    b.setFocusPainted( getBoolean("setFocusPainted", b) );
    }
    
    private boolean getBoolean(String s, AbstractButton b) {
        Boolean bool = (Boolean)b.getClientProperty(s);
        return bool == null ? false : bool.booleanValue();
    }
    
    private int getInteger(String s, AbstractButton b) {
        Integer integr = (Integer)b.getClientProperty(s);
        return integr == null ? 0 : integr.intValue();
    }

	private boolean shouldInstallUI(AbstractButton button) {
		Icon icon = button.getIcon();
		if(icon == null) return true;
		if(icon == _default0) return false;		   
		if(icon == _default1) return false;
		if(icon == _default2) return false;
		if(icon == _default3) return false;		
		if(icon == _default4) return false;

		Icon pressedIcon = button.getPressedIcon();
		if(pressedIcon == null) return true;
		if(pressedIcon == _pressed0) return false;		   
		if(pressedIcon == _pressed1) return false;
		if(pressedIcon == _pressed2) return false;
		if(pressedIcon == _pressed3) return false;		
		if(pressedIcon == _pressed4) return false;
		return true;
	}
	
	
	/**
	 * Shifts the painted text down and right when pressed.
	 */
	protected void paintText(Graphics g, JComponent c, Rectangle r, String text) {
	    AbstractButton b = (AbstractButton)c;
	    if(isValidButton(b)) {
	        ButtonModel model = b.getModel();
	        if(model.isArmed() && model.isPressed()) {
                r.x = r.x + 1;
                r.y = r.y + 1;
            }
        }
        super.paintText(g, c, r, text);
    }

    protected void paintFocus(Graphics g, AbstractButton b,
			      Rectangle viewRect, Rectangle textRect, Rectangle iconRect){
	    if(isValidButton(b)) {
            g.setColor(_focusColor);
        	BasicGraphicsUtils.drawDashedRect(g, textRect.x-6, textRect.y-1,
					  textRect.width + 12, textRect.height+2);
        } else {
            super.paintFocus(g, b, viewRect, textRect, iconRect);
        }
    }


	/**
	 * Sets the standard icon as well as the pressed icon for the
	 * <tt>button</tt> argument.  This caches the previous icons
	 * so that when the UI is uninstalled, the values can be returned.
	 *
	 * @param button the button to use for setting the icons
	 */
	private void setIcons(AbstractButton button) {
		FontMetrics fm = button.getFontMetrics(button.getFont());
		int stringWidth = fm.stringWidth(button.getText());
		Icon buttonIcon;
		Icon pressedButtonIcon;
		Icon rolloverIcon;
		if(stringWidth < 30) {
			buttonIcon = _default0;
			pressedButtonIcon = _pressed0;
			rolloverIcon = _rollover0;
		}
		else if(stringWidth < 53) {
			buttonIcon = _default1;
			pressedButtonIcon = _pressed1;
			rolloverIcon = _rollover1;
		}
		else if(stringWidth < 72) {
			buttonIcon = _default2;
			pressedButtonIcon = _pressed2;
			rolloverIcon = _rollover2;
		}
		else if(stringWidth < 90) {
			buttonIcon = _default3;
			pressedButtonIcon = _pressed3;
			rolloverIcon = _rollover3;
		}
		else {
		    buttonIcon = _default4;
		    pressedButtonIcon = _pressed4;
		    rolloverIcon = _rollover4;
		}
	    button.putClientProperty("setIcon", button.getIcon());
		button.setIcon(buttonIcon);
	    button.putClientProperty("setDisabledIcon", button.getDisabledIcon());
        button.setDisabledIcon(buttonIcon);        
	    button.putClientProperty("setPressedIcon", button.getPressedIcon());
		button.setPressedIcon(pressedButtonIcon);
	    button.putClientProperty("setDisabledSelectedIcon",
	        button.getDisabledSelectedIcon());
        button.setDisabledSelectedIcon(pressedButtonIcon);
        button.putClientProperty("setRolloverIcon",
            button.getRolloverIcon());
        button.setRolloverIcon(rolloverIcon);
	    button.putClientProperty("setPreferredSize", button.getPreferredSize());       
		button.setPreferredSize(new Dimension(buttonIcon.getIconWidth(),
											  buttonIcon.getIconHeight()));	
	}
	
	/**
	 * Returns the icons to what they were prior to installation.
	 */
	private void removeIcons(AbstractButton b) {
	    b.setIcon((Icon)b.getClientProperty("setIcon"));
	    b.setDisabledIcon((Icon)b.getClientProperty("setDisabledIcon"));
	    b.setPressedIcon((Icon)b.getClientProperty("setPressedIcon"));
	    b.setDisabledSelectedIcon(
	        (Icon)b.getClientProperty("setDisabledSelectedIcon"));
	    b.setPreferredSize((Dimension)b.getClientProperty("setPreferredSize"));	    
	    b.setRolloverIcon((Icon)b.getClientProperty("setRolloverIcon"));
    }
}

   

