package com.limegroup.gnutella.gui;

import java.awt.event.WindowListener;
import java.awt.event.WindowEvent;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;

import com.limegroup.gnutella.ErrorService;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.SystemUtils;

/**
 * Manages aspects of LimeWire depending on whether or not
 * it is in the foreground, or if the computer has been idle
 * for a while.
 */
class PowerManager implements RefreshListener, WindowListener {
    
    // The maximum idle time before we switch back to full power.
    private static final int MAX_IDLE_TIME = 5 * 60 * 1000; // 5 minutes
    
    // On Java 1.4 and above, we can determine if the window that
    // caused us to deactivate was a child window or not.
    // Prior to that, we just have to assume that it was always
    // something else.
    private final static Method _getOpposite;
    private static final Object[] EMPTY_OBJECT_ARRAY = new Object[0];
    static {
        Method opp = null;
        if(CommonUtils.isJava14OrLater()) {
            try {
                opp = WindowEvent.class.getMethod(
                    "getOppositeWindow", new Class[] {}
                );
            } catch(NoSuchMethodException nsme) {
                ErrorService.error(nsme);
            } catch(SecurityException se) {
                ErrorService.error(se);
            }
        }
        _getOpposite = opp;
    }
    
    /**
     * Whether or not the window is currently active.
     */
    private boolean _active = true;
            
    
    /**
     * Callback for a window activation event.
     *
     * Sets LimeWire to use its full power.
     */
    public void windowActivated(WindowEvent e) {
        _active = true;
        RouterService.setFullPower(true);
    }
    
    /**
     * Callback for a window deactivation event.
     *
     * If the deactivation was not from a child Java window,
     * sets LimeWire to use less power.
     */
    public void windowDeactivated(WindowEvent e) {
        if(_getOpposite != null) {
            try {
                // If had an opposite, deactivated from another of our windows.
                Object o = _getOpposite.invoke(e, EMPTY_OBJECT_ARRAY);
                if(o != null)
                    return;
            } catch(IllegalAccessException iae) {
                ErrorService.error(iae);
            } catch(InvocationTargetException ite) {
                ErrorService.error(ite);
            } catch(ExceptionInInitializerError eiie) {
                ErrorService.error(eiie);
            }
        }
        
        _active = false;
            
        if(SystemUtils.getIdleTime() > MAX_IDLE_TIME)
            return;
            
        RouterService.setFullPower(false);
    }
    
    /** Stubbed our WindowListener method. */
    public void windowClosed(WindowEvent e) {}
    /** Stubbed our WindowListener method. */
    public void windowClosing(WindowEvent e) {}
    /** Stubbed our WindowListener method. */
    public void windowDeiconified(WindowEvent e) {}
    /** Stubbed our WindowListener method. */    
    public void windowIconified(WindowEvent e) {}
    /** Stubbed our WindowListener method. */
    public void windowOpened(WindowEvent e) {}
    
    /**
     * Callback for the every-second update.
     *
     * If the user has been idle for a specified amount of time,
     * let LimeWire use its full power.  Otherwise, if LimeWire isn't
     * the foreground application, use less power.
     */
    public void refresh() {
        if(SystemUtils.getIdleTime() > MAX_IDLE_TIME)
            RouterService.setFullPower(true);
        else if(!_active)
            RouterService.setFullPower(false);
    } 
}
