package com.limegroup.gnutella.gui;

import java.io.*;
import java.net.*;
import java.util.ResourceBundle;
import java.util.Locale;
import java.util.MissingResourceException;
import java.awt.*;
import javax.swing.*;
import javax.swing.plaf.*;
import com.limegroup.gnutella.ErrorService;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.updates.*;

/**
 * Manages application resources, including the custom <tt>LookAndFeel</tt>,
 * the locale-specific <tt>String</tt> instances, and any <tt>Icon</tt>
 * instances needed by the application.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class ResourceManager {

    /**
     * Instance of this <tt>ResourceManager</tt>, following singleton.
     */
    private static ResourceManager _instance;

    /**
     * The constant name of the native Windows library.
     */
    private static final String WINDOWS_LIBRARY_NAME = "LimeWire20";

    /**
     * Constant name for the browser component library.
     */
    private static final String BROWSER_LIBRARY_NAME = "BrowserPage";

    /**
     * Constant for the relative path of the gui directory.
     */
    private static final String GUI_PATH =
        "com/limegroup/gnutella/gui/";

    /**
     * Constant for the relative path of the resources directory.
     */
    private static final String RESOURCES_PATH =
        GUI_PATH + "resources/";

    /**
     * Constant for the relative path of the images directory.
     */
    private static final String IMAGES_PATH =
        GUI_PATH + "images/";

    /**
     * Boolean status that controls whever the shared <tt>Locale</tt> instance
     * needs to be loaded, and locale-specific options need to be setup.
     */
    private static boolean _localeOptionsSet;

    /**
     * Static variable for the loaded <tt>Locale</tt> instance.
     */
    private static Locale _locale;

    /**
     * The <tt>ResourceBundle</tt> instance to use for loading 
     * locale-specific resources.
     */
    private static ResourceBundle _resourceBundle;

    /**
     * Locale-specific option, set from the loaded resource bundle, and that
     * control the preferred appearence and layout of the GUI for the loaded
     * locale.  Complex scripts (such as Chinese, Korean, Japanese) should
     * not be displayed with bold characters if they are small (below 12
     * 12 points).
     */
    private static boolean _useBold;

    /**
     * Locale-specific option, set from the loaded resource bundle, and that
     * control the preferred appearence and layout of the GUI for the loaded
     * locale.  Semitic scripts (such as Hebrew, Arabic, Urdu, Farsi, Divehi)
     * and Thai should be used with a right-to-left directionality in the GUI
     * layout. Needed here to extend Swing with Java 1.1.8 (on Mac OS 8/9),
     * which does not handle java.awt.ComponentOrientation.
     */
    private static boolean _useLeftToRight;

    /**
     * Boolean for whether or not the browser page dll was successfully loaded.
     */
    private static boolean _browserPageLoaded = true;


    /**
     * Boolean for whether or not the installer has been shared.
     */
    private static boolean _installerShared = false;
    
    /**
     * Boolean for whether or not the font-size has been reduced.
     */
    private static boolean _fontReduced = false;

    /**
     * Statically initialize necessary resources.
     */
    static {
        resetLocaleOptions();
    }

    static void resetLocaleOptions() {
        _localeOptionsSet = false;
        setLocaleOptions();
    }

    static void setLocaleOptions() {
        if (!_localeOptionsSet) {
            if(ApplicationSettings.LANGUAGE.getValue().equals(""))
                ApplicationSettings.LANGUAGE.setValue("en");
            _locale = new Locale(
                ApplicationSettings.LANGUAGE.getValue(),
                ApplicationSettings.COUNTRY.getValue(),
                ApplicationSettings.LOCALE_VARIANT.getValue());
            _resourceBundle = ResourceBundle.getBundle(
                "MessagesBundle", _locale);
            _useBold = !
                Boolean.valueOf(getStringResource("DISABLE_BOLD_CHARACTERS")).
                booleanValue();
            _useLeftToRight = !
                Boolean.valueOf(getStringResource("LAYOUT_RIGHT_TO_LEFT")).
                booleanValue();
            _localeOptionsSet = true;
        }
    }

    /**
     * Returns the <tt>Locale</tt> instance currently in use.
     *
     * @return the <tt>Locale</tt> instance currently in use
     */
    static Locale getLocale() {
        return _locale;
    }
    
    /**
     * Determines whether or not the current locale language is English.
     * Note that the user setting may be empty, defaulting to the running
     * system locale which may be other than English. Here we check the
     * effective locale seen in the MessagesBundle.
     */
    static boolean hasLocalizedTipsOfTheDay() {
        return Boolean.valueOf(getStringResource("HAS_TIPS_OF_THE_DAY"))
            .booleanValue();
    }
    
    /**
     * Returns the TOTD resource bundle.
     */
    static ResourceBundle getTOTDResourceBundle() {
        return ResourceBundle.getBundle("totd/TOTD", _locale);
    }

    
    /**
     * returns the resource bundle for a specific schema.
     * called by DisplayManager to get translated schema names.
     * @param String schema name 
     *        (not the URI but name returned by LimeXMLSchema.getDisplayString)
     * @return ResourceBundle
     */
    static ResourceBundle getXMLResourceBundle(String name) {
        return ResourceBundle.getBundle("xml.display." + name, _locale);
    }
    

    /**
     * Determines whever or not bold style can safely be used in the current
     * locale for displaying text with a size lower than 1 pica (12 points).
     *
     * @return <tt>true</tt> if bold style can safely be used in this
     *  locale (simple scripts), <tt>false</tt> otherwise (complex scripts)
     */
    static final boolean useBold() {
        setLocaleOptions();
        return _useBold;
    }

    /**
     * Determines whever the standard left-to-right orientation should be used
     * in the current locale.
     *
     * @return <tt>true</tt> if bold characters should be used in this
     *  locale, <tt>false</tt> otherwise (Semitic scripts).
     */
    static final boolean isLeftToRight() {
        setLocaleOptions();
        return _useLeftToRight;
    }

    /**
     * Returns whether or not the browser page dll has loaded successfully.
     *
     * @return <tt>true</tt> if the browser page has successfully loaded,
     *  <tt>false</tt> otherwise
     */
    static boolean isBrowserPageLoaded() {
        return _browserPageLoaded;
    }
 
    /**
     * Returns the look and feel for when the application first loaded.
     *
     * @return  the look and feel for when the application first loaded
     */
    static LookAndFeel getInitialLookAndFeel() {
        return new LimeLookAndFeel();
    }

    /**
     * Returns the locale-specific String from the resource manager.
     *
     * @return an internationalized <tt>String</tt> instance
     *  corresponding with the <tt>resourceKey</tt>
     */
    static final String getStringResource(final String resourceKey) {
        return _resourceBundle.getString(resourceKey);
    }

    /**
     * Serves as a single point of access for any icons that should be accessed
     * directly from the file system for themes.
     *
     * @param IMAGE_PATH the relative the icon to return
     * @return a new <tt>ImageIcon</tt> instance for the specified file,
     *  or <tt>null</tt> if the resource could not be loaded
     */
    static final ImageIcon getThemeImage(final String name) {
        File image = new File(ThemeSettings.THEME_DIR.getValue(), name);
        if (image.exists())
            return new ImageIcon(toURL(image));
        else {
            URL img = getURLImage(name);
            if(img == null)
                throw new MissingResourceException(
                    "image: " + name + " doesn't exist.", null, null);
            else                    
                return new ImageIcon(img);
        }
    }
    
    /**
     * Retrieves an icon from the specified path.
     */
    static final ImageIcon getImageFromPath(String loc) {
        return new ImageIcon(loc);
    }

    /**
     * Returns a new <tt>URL</tt> instance for the specified file in the
     * "images" directory.
     *
     * @param FILE_NAME the name of the resource file
     * @return a new <tt>URL</tt> instance for the desired file, or
     *  <tt>null</tt> if the <tt>URL</tt> could not be loaded
     */
    static URL getURLImage(final String FILE_NAME) {
        return ResourceManager.getURL(IMAGES_PATH + FILE_NAME);
    }

    /**
     * Returns a new <tt>URL</tt> instance for the specified file in the
     * "resources" directory.
     *
     * @param FILE_NAME the name of the resource file
     * @return a new <tt>URL</tt> instance for the desired file, or
     *  <tt>null</tt> if the <tt>URL</tt> could not be loaded
     */
    static URL getURLResource(final String FILE_NAME) {
        return ResourceManager.getURL(RESOURCES_PATH + FILE_NAME);
    }

    /**
     * Returns a new <tt>URL</tt> instance for the resource at the
     * specified local path.  The path should be the full path within
     * the jar file, such as: <p>
     * 
     * com/limegroup/gnutella/gui/images/searching.gif<p>
     *
     * @param PATH the path to the resource file within the jar
     * @return a new <tt>URL</tt> instance for the desired file, or
     *  <tt>null</tt> if the <tt>URL</tt> could not be loaded
     */
    private static URL getURL(final String PATH) {
        ClassLoader cl = ResourceManager.class.getClassLoader();
        if (cl ==  null) {
            return ClassLoader.getSystemResource(PATH);
        }
        URL url = cl.getResource(PATH);
        if (url == null) {
            return ClassLoader.getSystemResource(PATH);
        }
        return url;
    }

    /**
     * Instance accessor following singleton.
     */
    static final ResourceManager instance() {
        if (_instance == null)
            _instance = new ResourceManager();
        return _instance;
    }


    /**
     * Private constructor to ensure that a <tt>ResourceManager</tt> 
     * cannot be constructed from outside this class.
     */
    private ResourceManager() {
        GUIMediator.setSplashScreenString(
            GUIMediator.getStringResource("SPLASH_STATUS_RESOURCE_MANAGER"));
            
        if(!ThemeFileHandler.isCurrent()) {
            ThemeSettings.THEME_FILE.revertToDefault();
            ThemeSettings.THEME_DIR.revertToDefault();
            ThemeFileHandler.reload();
        }

        try {
            themeChanged();
        } catch (Exception e) {
            // we don't care what exception it is.  if this happens,
            // LimeWire will fail, so just exit
            GUIMediator.showInternalError(e);
            GUIMediator.shutdown();
        }
        // The Windows & Browser library is not stored in any jar, and is
        // already in the appropriate location, so copying the resource
        // file is pointless (and doesn't work also).
        if (CommonUtils.isWindows()) {
            loadLibrary(WINDOWS_LIBRARY_NAME);
        }
        if (!GUIMediator.hasDonated() && CommonUtils.isWindows()) {
            loadLibrary(BROWSER_LIBRARY_NAME);
        }
        try {
            unpackWarFiles();
        } catch(IOException e) {
            // this should not happen, so display the error to the 
            // user
            GUIMediator.showInternalError(e);
        }
        unpackVersionFile();
        shareInstaller();
    }

    /**
     * Shares the installer if it's not already shared.
     */
    private void shareInstaller() {
        // don't share it if it's pro
        if(CommonUtils.isPro()) return;

        // ignore this if it was already shared (defensive)
        if(_installerShared) return;
        
        String installerName = "";
        if(CommonUtils.isWindows()) {
            installerName = "LimeWireWin.exe";
        } else if(CommonUtils.isMacOSX()) {
            installerName = "LimeWireOSX.zip";
        } else if(CommonUtils.isMacClassic()) {
            installerName = "LimeWireMac.bin";
        } else if(CommonUtils.isLinux()) {
            installerName = "LimeWireLinux.bin";
        } else if(CommonUtils.isSolaris()) {
            installerName = "LimeWireSolaris.bin";
        } else {
            _installerShared = true;
            // otherwise, don't worry about sharing the installer
            return;
        }

        // we use java.home because it's just easier from the installer
        File javaHome = new File(System.getProperty("java.home"));
        File installer = new File(javaHome, installerName);

        // let it go if the file isn't there for whatever reason.
        // it won't be there after InstallShield installs, for example
        if(!installer.isFile()) return;
        
        File[] sharedDirs = SharingSettings.DIRECTORIES_TO_SHARE.getValue();

        // if there are no shared directories, don't share the installer
        if(sharedDirs.length == 0) return;

        // just use the first shared directory -- we don't really care 
        // which one it is
        File sharedDir = sharedDirs[0];        

        File newFile = new File(sharedDir, installerName);

        // if the shared file is the same or newer than the file on disk,
        // stop
        if(newFile.isFile() && 
           (installer.lastModified() <= newFile.lastModified())) {
            _installerShared = true;
           return;
        }
        
        // otherwise, share it!
        installer.renameTo(new File(sharedDir, installerName));
        _installerShared = true;
    }

    /**
     * Updates to the current theme.
     */
    public void themeChanged() {
        try {
            if(ThemeSettings.isNativeTheme()) {
                String lafName = UIManager.getSystemLookAndFeelClassName();
                UIManager.setLookAndFeel(lafName);

                if(CommonUtils.isMacOSX()) {
                    if(!_fontReduced) {
                        _fontReduced = true;
                        reduceFont("Label.font");
                        reduceFont("Table.font");
                    }
                }
            } else {
                UIManager.setLookAndFeel(new LimeLookAndFeel());                
            }

            UIManager.put("Tree.leafIcon",
                UIManager.getIcon("Tree.closedIcon"));
                
            // remove split pane borders
            UIManager.put("SplitPane.border", BorderFactory.createEmptyBorder());
            
            // Add a bolded text version of simple text.
            Font normal = UIManager.getFont("Table.font");
            FontUIResource bold = new FontUIResource(
                normal.getName(), Font.BOLD, normal.getSize());
            UIManager.put("Table.font.bold", bold);
            
        } catch (UnsupportedLookAndFeelException e) {
            // this should never happen, so notify the gui if it does
            GUIMediator.showInternalError(e);
        } catch (ClassNotFoundException e) {
            GUIMediator.showInternalError(e);
        } catch (InstantiationException e) {
            GUIMediator.showInternalError(e);
        } catch (IllegalAccessException e) {
            GUIMediator.showInternalError(e);
        }
    }
    
    /**
     * Updates the component to use the native UI resource.
     */
    static ComponentUI getNativeUI(JComponent c) {
        ComponentUI ret = null;
        String name = UIManager.getSystemLookAndFeelClassName();
        if(name != null) {
            try {
                Class clazz = Class.forName(name);
                LookAndFeel lf = (LookAndFeel)clazz.newInstance();
                lf.initialize();
                UIDefaults def = lf.getDefaults();
                ret = def.getUI(c);
            } catch(ExceptionInInitializerError e) {
            } catch(ClassNotFoundException e) {
            } catch(LinkageError e) {
            } catch(IllegalAccessException e) {
            } catch(InstantiationException e) {
            } catch(SecurityException e) {
            } catch(ClassCastException e) {
            }
        }

        // if any of those failed, default to the current UI.
        if(ret == null)
            ret = UIManager.getUI(c);

        return ret;
    }
    
    /**
     * Reduces the size of a font in UIManager.
     */
    private static void reduceFont(String name) {
        Font oldFont = UIManager.getFont(name);
        FontUIResource newFont =
          new FontUIResource(oldFont.getName(), oldFont.getStyle(),
                            oldFont.getSize() - 2);
        UIManager.put(name, newFont);
    }

    /**
     * Loads the library specified by the name argument, displaying an 
     * internal error message to the user if it fails.
     *
     * @param libraryName the name of the library to load
     */
    private void loadLibrary(final String libraryName) {
        try {
            System.loadLibrary(libraryName);
        } catch (UnsatisfiedLinkError e) {
            if (libraryName.equals(BROWSER_LIBRARY_NAME)) {
                // this fails for some reason under some circumstances on
                // 95/98/Me/NT 4.0.  We don't know why, so just fail 
                // gracefully
                _browserPageLoaded = false;
                return;
            }
            GUIMediator.showInternalError(e);
        }       
    }

    /**
     * Unpacks any war files in the current directory.
     */
    private static void unpackWarFiles() throws IOException {
        //Unpack any .war files, and put into appropriate dirs.
        File currDir = CommonUtils.getCurrentDirectory();
        String[] warFiles = (currDir).list(
            new FilenameFilter() {
                //the files to be accepted to be returned
                public boolean accept(File dir, String name) {
                    return name.endsWith(".war");
                }
            });
        if (warFiles == null) {
            // no war files to expand -- don't worry about it
            return;
        }
        File destDir = CommonUtils.getUserSettingsDir();
        if(!destDir.isDirectory()) {
            throw new IOException("settings dir not a directory: "+destDir);
        }
        if(!destDir.canWrite()) {
            throw new IOException("cannot write to the settings dir: "+destDir);
        }
        for (int i = 0; i < warFiles.length; i++) {
            Expand.expandFile(new File(warFiles[i]), destDir);
        }
    }

    private void unpackVersionFile() {
        File userHome = CommonUtils.getUserSettingsDir();
        File updateFile = new File(userHome, "update.xml");
        File verFile = new File("update.ver");
        if (!updateFile.exists()) { // unpack the .ver file into this dir
            try {
                Expand.expandFile(verFile, userHome);
            } catch (IOException iox) {
                return; //there is nothing we can do
            }
        } else { //there is an update file in the user's home
            String myVersion = CommonUtils.getLimeWireVersion();
            String latestVersion = parseVersion(updateFile);
            //check the version of the file, and compare to this Limeversion
            //System.out.println("me="+myVersion + " latest="+latestVersion);
            if (UpdateManager.isGreaterVersion(myVersion, latestVersion) ||
                (latestVersion.equals("@version@") &&
                 !myVersion.equals("@version@"))) {
                try {
                    Expand.expandFile(verFile,userHome);
                } catch (IOException iox) {
                    return;//there is nothing we can do
                }
            }
            // latestVersion >= myVersion so do nothing
        }
    }

    private static String parseVersion(File file) {
        try {
            RandomAccessFile raf = new RandomAccessFile(file, "r");
            byte[] bytes = new byte[(int)raf.length()];
            raf.read(bytes);
            raf.close();
            String tempString = new String(bytes, "UTF8");
            int a = tempString.indexOf("<version>");
            int b = tempString.indexOf("</version>");
            if(a <0 || b < 0 || b <= a) {
                file.delete();//bad file
                return null;
            }                
            String tempVersion = tempString.substring(a + 9, b); 
            return tempVersion;
        } catch (IOException e) {
            return null;
        }
    }

    private static URL toURL(File f) {
        try {
            return new URL("file", "",
                slashify(f.getAbsolutePath(), f));
        } catch (MalformedURLException e) {
            ErrorService.error(e);
            return null;
        }
    }

    private static String slashify(String path, File f) {
        String p = path;
        if (File.separatorChar != '/')
            p = p.replace(File.separatorChar, '/');
        if (!p.startsWith("/"))
            p = "/" + p;
        if (!p.endsWith("/") && f.isDirectory())
            p = p + "/";
        return p;
    }

}
