package com.limegroup.gnutella.gui;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.awt.*;
import java.io.*;

import com.limegroup.gnutella.ErrorService;
import com.sun.java.util.collections.*;

/**
 * This reusable component is a list with buttons for adding and removing
 * elements.  The add button brings up a dialog window to retrieve
 * the element to add from the user.  The remove button is only enabled
 * when there is an item selected in the list.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class StandardListEditor {

	/**
	 * Constant handle to the main panel in the editor.
	 */
	private final BoxPanel MAIN_PANEL = new BoxPanel(BoxPanel.X_AXIS);

	/**
	 * Constant handle to the underlying <tt>JList</tt> instance.
	 */
	private final JList LIST = new JList();
	
	/**
	 * Handle to the remove button to allow us to set its state.
	 */
	private final JButton REMOVE_BUTTON;

	/**
	 * Handle to the add button.
	 */
	private final JButton ADD_BUTTON;

	/**
	 * Default listener for the add button.
	 */
	private ActionListener _defaultAddListener;

	/**
	 * Member variable for whether or not the list data has changed
	 * since the last call to reset this value.
	 */
	private boolean _listChanged = false;
	
	/**
	 * Creates a <tt>StandardListEditor</tt> with the list of elements on
	 * the left and buttons for adding and removing elements on the
	 * right.  The key allows for a custom label for the text field
	 * in the dialog window that the "Add..." button pops up.  For example, 
	 * using <p>
	 *
	 * OPTIONS_AUTO_CONNECT_INPUT_FIELD_LABEL=Enter Host Address:<p>
	 *
	 * in the messages bundle will make "Enter Host Address: " appear 
	 * as the label for the text field in dialog window popped up by
	 * the "add" button.  
	 *
	 * @param INPUT_FIELD_KEY the key for the locale-specific label for the dialog
	 * window popped up by the "Add..." button
	 */
	public StandardListEditor(final String INPUT_FIELD_KEY) {
		this("LIST_EDITOR_ADD_BUTTON", "LIST_EDITOR_REMOVE_BUTTON",INPUT_FIELD_KEY);
	}

	/**
	 * Creates a <tt>StandardListEditor</tt> with the default settings
	 * for the names of the add and remove buttons and that uses
	 * the specified <tt>ActionListener</tt> instance as the listener
	 * for the add button.
	 */
	public StandardListEditor(ActionListener listener) {
		this("LIST_EDITOR_ADD_BUTTON", "LIST_EDITOR_REMOVE_BUTTON","");
		setAddActionListener(listener);
	}

	/**
	 * More flexible constructor that allows the text for the add and
	 * remove buttons to be set and that allows a custom <tt>ActionListener</tt>
	 * for the add button.
	 *
	 * @param ADD_BUTTON_KEY the key for the locale-specific string to use for
	 *  the add button
	 * @param REMOVE_BUTTON_KEY the key for the locale-specific string to use 
	 *  for the remove button
	 * @param INPUT_FIELD_KEY the key for the locale-specific string to use
	 *  for the label in the generic text input component used by default
	 */
	public StandardListEditor(final String ADD_BUTTON_KEY, 
							  final String REMOVE_BUTTON_KEY,
							  final String INPUT_FIELD_KEY) {	
		String[] buttonLabelKeys = {
			ADD_BUTTON_KEY,
			REMOVE_BUTTON_KEY
		};

		String[] buttonLabelTips = {
			"LIST_EDITOR_ADD_BUTTON_TIP",
			"LIST_EDITOR_REMOVE_BUTTON_TIP"
		};

		_defaultAddListener = new AddActionListener(INPUT_FIELD_KEY);
		ActionListener[] buttonListeners = {
			_defaultAddListener,
			new RemoveListener()
		};

		ButtonRow buttons = new ButtonRow(buttonLabelKeys, 
										  buttonLabelTips,
										  buttonListeners,
										  ButtonRow.Y_AXIS,
										  ButtonRow.BOTTOM_GLUE);
		
		ADD_BUTTON = buttons.getButtonAtIndex(0);
		REMOVE_BUTTON = buttons.getButtonAtIndex(1);
		REMOVE_BUTTON.setEnabled(false);
		LIST.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		LIST.addListSelectionListener(new ListEditorSelectionListener());
		JScrollPane scrollPane = new JScrollPane(LIST);
		MAIN_PANEL.add(scrollPane);
		MAIN_PANEL.add(buttons);	
		
	}

	/**
	 * Provides access to the wrapped <tt>Component</tt> of the 
	 * <tt>StandardListEditor</tt>.  This is the <tt>Component</tt>
	 * that other gui elements should add.
	 *
	 * @return the underlying <tt>Component</tt> that contains all 
	 *  of the elements fo the <tt>StandardListEditor</tt>
	 */
	public Component getComponent() {
		return MAIN_PANEL;
	}

	/**
	 * Adds the specified object to the end of the list.
	 *
	 * @param element the <tt>Object</tt> to add
	 */
	private void addElement(Object element) {
		DefaultListModel model = (DefaultListModel)LIST.getModel();
		model.addElement(element);
	}

	/**
	 * Adds the specified <tt>File</tt> instance to the end of the list.
	 *
	 * @param file the <tt>File</tt> to add
	 */
	public void addFile(File file) {
		addElement(file);
	}

	/**
	 * Adds the specified <tt>String</tt> instance to the end of the list.
	 *
	 * @param string the <tt>String</tt> to add
	 */
	public void addString(String string) {
		addElement(string);
	}

	/**
	 * Sets the data in the underlying <tt>ListModel</tt>.
	 *
	 * @param data the <tt>Vector</tt> containing data for the model
	 */
	public void setListData(Vector data) {
		DefaultListModel model = new DefaultListModel();
		for (int i=0; i<data.size(); i++)
			model.addElement(data.get(i));
		LIST.setModel(model);
	}

	/**
	 * Sets the data in the underlying <tt>ListModel</tt>.
	 *
	 * @param data array of strings containing the data for the model
	 */
	private void setListDataObjects(Object[] data) {
		DefaultListModel model = new DefaultListModel();
		for (int i=0; i<data.length; i++)
			model.addElement(data[i]);
		LIST.setModel(model);
	}

	/**
	 * Sets the data in the underlying <tt>ListModel</tt>.
	 *
	 * @param data array of <tt>File</tt> instances containing the data for the 
	 *  model
	 */
	public void setListData(File[] data) {
		setListDataObjects(data);
	}

	/**
	 * Sets the data in the underlying <tt>ListModel</tt>.
	 *
	 * @param data array of <tt>String</tt> instances containing the data for the 
	 *  model
	 */
	public void setListData(String[] data) {
		setListDataObjects(data);
	}
	
	/**
	 * Clears the selections in the list.
	 */
	public void clearSelection() {
		LIST.clearSelection();
	}

	/**
	 * Returns an array of the underlying data represented as strings
	 * by calling toString() on each element.
	 *
	 * @return the list data as an array of strings.
	 */
	public String[] getDataAsStringArray() {
		DefaultListModel model = (DefaultListModel)LIST.getModel();
		Object[] dataObjects = model.toArray();
		String[] dataStrings = new String[dataObjects.length];
		for(int i=0; i<dataObjects.length; i++) {
			dataStrings[i] = dataObjects[i].toString();
		}
		return dataStrings;
	}

	/**
	 * Returns an array of the underlying data represented as <tt>File</tt>
	 * instances.
	 *
	 * @return the list data as an array of <tt>File</tt> instances.
	 */
	public File[] getDataAsFileArray() {
		DefaultListModel model = (DefaultListModel)LIST.getModel();
		Object[] dataObjects = model.toArray();
		File[] dataFiles = new File[dataObjects.length];
		for(int i=0; i<dataObjects.length; i++) {
			dataFiles[i] = (File)dataObjects[i];
		}
		return dataFiles;
	}

	/**
	 * Returns an array of the underlying data represented as objects.
	 *
	 * @return the list data as an array of objects.
	 */
	public Object[] getDataAsObjectArray() {
		DefaultListModel model = (DefaultListModel)LIST.getModel();
		return model.toArray();
	}

	/**
	 * Sets the <tt>ActionListener</tt> to use for the add button.
	 *
	 * @param addAction the <tt>ActionListener</tt> to use for the add button
	 */
  	private void setAddActionListener(ActionListener addListener) {
  		ADD_BUTTON.removeActionListener(_defaultAddListener);
  		ADD_BUTTON.addActionListener(addListener);
  	}

	/**
	 * Returns whether or not the list has changed since the last
	 * call to reset the list.  Note that this will not be 
	 * accurate if you use your own <tt>ActionListener</tt> for the
	 * add button.  In this case, this will return whether or not
	 * elements have been removed from the list.
	 *
	 * @return <tt>true</tt> if the list has changed since the last
	 *  call to reset the list, <tt>false</tt> otherwise
	 */
	public boolean getListChanged() {
		return _listChanged;
	}

	/**
	 * Resets the value for whether or not the list has changed to
	 * <tt>false</tt>.
	 */
	public void resetList() {
		_listChanged = false;
	}

	/**
	 * This class responds to a click of the add button and pops
	 * up a window for the user to enter a new element to add
	 * to the list.
	 */
	private class AddActionListener implements ActionListener {
		private final String INPUT_FIELD_KEY;
		public AddActionListener(final String key) {
			INPUT_FIELD_KEY = key;
		}
		public void actionPerformed(ActionEvent e) {
            try {
    			InputFieldDialog dialog = new InputFieldDialog(INPUT_FIELD_KEY);
    			int returnCode = dialog.showDialog();
    
    			if(returnCode == InputFieldDialog.TEXT_ENTERED) {
    				_listChanged = true;
    				addElement(dialog.getText());
    			}
            } catch(Throwable t) {
                ErrorService.error(t);
            }
		}
	}


	/**
	 * This class responds to a click of the remove button and removes
	 * the selected element from the list.
	 */
	private class RemoveListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
		    try {
    			_listChanged = true;
    			// return if nothing is selected
    			if(LIST.isSelectionEmpty()) return;
    			
    			DefaultListModel model = (DefaultListModel)LIST.getModel();
    			model.remove(LIST.getSelectedIndex());
    
    			if(LIST.isSelectionEmpty()) {
    				REMOVE_BUTTON.setEnabled(false);
    			}
            } catch(Throwable t) {
                ErrorService.error(t);
            }
		}
	}

	/**
	 * This private class handles selection of items in the list.  It
	 * controls the state of the remove button as well, disabling it
	 * if nothing is selected.
	 */
	private class ListEditorSelectionListener implements ListSelectionListener {

		/**
		 * Implements the <tt>ListSelectionListener</tt> interface.  
		 * Responds to selections in the list.
		 */
		public void valueChanged(ListSelectionEvent e) {
            try {
    			if (e.getValueIsAdjusting())
    				return;
    			if (LIST.isSelectionEmpty()) {
    				REMOVE_BUTTON.setEnabled(false);
    			} else {
    				REMOVE_BUTTON.setEnabled(true);
    			}
            } catch(Throwable t) {
                ErrorService.error(t);
            }
		}
	}
}
