package com.limegroup.gnutella.gui.connection;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Point;
import java.awt.event.MouseEvent;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;

import com.limegroup.gnutella.Assert;
import com.limegroup.gnutella.BadConnectionSettingException;
import com.limegroup.gnutella.ErrorService;
import com.limegroup.gnutella.ManagedConnection;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.gui.AutoCompleteTextField;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.PaddedPanel;
import com.limegroup.gnutella.gui.WholeNumberField;
import com.limegroup.gnutella.gui.tables.AbstractTableMediator;
import com.limegroup.gnutella.gui.tables.DataLine;
import com.limegroup.gnutella.gui.tables.LimeJTable;
import com.limegroup.gnutella.gui.tables.TableSettings;
import com.limegroup.gnutella.settings.ConnectionSettings;
import com.sun.java.util.collections.Arrays;

/**
 * This class acts as a mediator between all of the components of the
 * connection window.
 */
public final class ConnectionMediator extends AbstractTableMediator {

    /**
     * Instance of singleton access
     */
    private static final ConnectionMediator INSTANCE =
        new ConnectionMediator();

    public static ConnectionMediator instance() { return INSTANCE; }

    /**
     * Listeners so buttons and possibly future right-click menu share.
     */
    ActionListener ADD_LISTENER;
    DocumentListener KEEP_ALIVE_LISTENER;

    private static final String IS_ULTRAPEER =
        GUIMediator.getStringResource("CV_STRING_ULTRAPEER");

    private static final String IS_LEAF =
        GUIMediator.getStringResource("CV_STRING_CLIENT");
        
    private static final String CONNECTING = 
        GUIMediator.getStringResource("CV_TABLE_STRING_CONNECTINGS");
        
	private static final String LEAVES =
        GUIMediator.getStringResource("CV_TABLE_STRING_LEAVES");
        
    private static final String ULTRAPEERS =
        GUIMediator.getStringResource("CV_TABLE_STRING_ULTRAPEERS");
        
    private static final String PEERS =
        GUIMediator.getStringResource("CV_TABLE_STRING_PEERS");

    private static final String STANDARD =
        GUIMediator.getStringResource("CV_TABLE_STRING_STANDARDS");

    /**
     * Extra component constants
     */
    private AutoCompleteTextField HOST_INPUT;
    private JTextField PORT_INPUT;
    private JTextField MIN_CONNECTIONS_INPUT;
    private JLabel SERVENT_STATUS;
    
    /**
     * The label displaying the number of ultrapeers, peers & leaves.
     */
    private JLabel NEIGHBORS;

    /**
     * Boolean for whether or not keepAlive should do anything
     */
    private boolean disableKeepAliveValidate;


    /**
     * Build the listeners
     */
    protected void buildListeners() {
        super.buildListeners();
        ADD_LISTENER = new AddListener();
        KEEP_ALIVE_LISTENER = new KeepAliveListener();
    }
    
    /**
     * Overriden to have different default values for tooltips.
     */
    protected void buildSettings() {
        SETTINGS = new TableSettings(ID) {
            public boolean getDefaultTooltips() {
                return false;
            }
        };
    }

    /**
     * Add the listeners
     */
    protected void addListeners() {
        super.addListeners();
        HOST_INPUT.addActionListener(ADD_LISTENER);
        MIN_CONNECTIONS_INPUT.getDocument().addDocumentListener(KEEP_ALIVE_LISTENER);
    }

	/**
	 * Set up the necessary constants.
	 */
	protected void setupConstants() {
		MAIN_PANEL =
		    new PaddedPanel(GUIMediator.getStringResource("CV_PANEL_TITLE"));
		DATA_MODEL = new ConnectionModel();
		TABLE = new LimeJTable(DATA_MODEL);
		BUTTON_ROW = (new ConnectionButtons(this)).getComponent();

		HOST_INPUT = new AutoCompleteTextField(20);
		PORT_INPUT = new WholeNumberField(6346, 4);
		SERVENT_STATUS = new JLabel("");
		MIN_CONNECTIONS_INPUT = new WholeNumberField(0, 2);
		NEIGHBORS = new JLabel("");
    }

    /**
     * Overridden to set the size.
     */
    protected JComponent getScrolledTablePane() {
		JComponent pane = super.getScrolledTablePane();

        SCROLL_PANE.setPreferredSize(new Dimension(3000, 5000));

        return pane;
    }

    /**
     * Update the splash screen
     */
	protected void updateSplashScreen() {
		GUIMediator.setSplashScreenString(
            GUIMediator.getStringResource("SPLASH_STATUS_CONNECTION_WINDOW"));
    }

    /**
     * Override the default main panel setup so we can add two text boxes,
     * for the host & port.
     */
    protected void setupMainPanel() {
        if (MAIN_PANEL != null) {

            MAIN_PANEL.add(NEIGHBORS);
            MAIN_PANEL.add( getScrolledTablePane() );

            // Construct the line of
            // Button[REMOVE] Button[ADD] JTextField[HOST] WholeNumberField[PORT]
            JPanel line1 = new JPanel();
            line1.setLayout( new FlowLayout() );
			line1.add( SERVENT_STATUS );
            line1.add( BUTTON_ROW );
            line1.add( HOST_INPUT );
            line1.add( PORT_INPUT );
            //add it
            MAIN_PANEL.add( line1 );

            // Construct the line of
            // JLabel[STATUS] JLabel[PRE-TEXT] WholeNumberField[MIN_CONN] JLabel[POST_TEXT]

			// add the keep alive field only if the settings tell us to
			if(ConnectionSettings.SHOW_KEEP_ALIVE.getValue()) {
				JPanel line2 = new JPanel();
				line2.setLayout( new FlowLayout() );
				line2.add(new JLabel(GUIMediator.getStringResource("CV_LABEL_KEEP_BEGIN")
									 + " "));
				line2.add( MIN_CONNECTIONS_INPUT );
				// Set the text with what's in your settings file.
				setKeepAliveText( Integer.toString( ConnectionSettings.NUM_CONNECTIONS.getValue()));
				line2.add(new JLabel( " " +
									  GUIMediator.getStringResource("CV_LABEL_KEEP_END")));
				MAIN_PANEL.add(line2);
			}

            MAIN_PANEL.setMinimumSize(ZERO_DIMENSION);
        }
    }

	/**
	 * Constructor -- private for Singleton access
	 */
	private ConnectionMediator() {
	    super("CONNECTION_TABLE");
		GUIMediator.addRefreshListener(this);
		GUIMediator.addThemeObserver(this);
	}

    /**
     * Removes all selected rows from Router,
     * which will in turn remove it from the list.
     * Overrides default removeSelection
     */
    public void removeSelection() {
		int[] sel = TABLE.getSelectedRows();
		Arrays.sort(sel);
		ManagedConnection c;
		for( int counter = sel.length - 1; counter >= 0; counter--) {
			int i = sel[counter];
			c = (ManagedConnection)(
			        (DataLine)DATA_MODEL.get(i)).getInitializeObject();
			RouterService.removeConnection(c);
		}
		clearSelection();
    }

    // inherit doc comment
    protected JPopupMenu createPopupMenu() {
        return null;
    }

	/**
	 * Handles the selection of the specified row in the connection window,
	 * enabling or disabling buttons
	 *
	 * @param row the selected row
	 */
	public void handleSelection(int row) {
	    setButtonEnabled( ConnectionButtons.REMOVE_BUTTON, true );
	}

	/**
	 * Handles the deselection of all rows in the download table,
	 * disabling all necessary buttons and menu items.
	 */
	public void handleNoSelection() {
	    setButtonEnabled( ConnectionButtons.REMOVE_BUTTON, false );
	}

    public void handleMouseDoubleClick(MouseEvent e) { 
        doBrowseHost(); 
    }

    /**
     * get the first selected row and trigger a browse host
     */
    private void doBrowseHost() {
        int[] rows = TABLE.getSelectedRows();
        if(rows.length > 0) {
            ManagedConnection c = 
                (ManagedConnection)
                ((DataLine)DATA_MODEL.get(rows[0])).getInitializeObject();
            GUIMediator.instance().doBrowseHost(c.getAddress(), c.getPort());
        }
    }

	/**
	 * Override the default doRefresh so we can update the servent status label
	 * (Uses doRefresh instead of refresh so this will only get called
	 *  when the table is showing.  Small optimization.)
	 */
	public void doRefresh() {
	    super.doRefresh();
	    SERVENT_STATUS.setText(
	        ( RouterService.isSupernode() ?
                IS_ULTRAPEER : IS_LEAF ) + "      "
        );
        int[] counts = ((ConnectionModel)DATA_MODEL).getConnectionInfo();
        ConnectionModel cm = (ConnectionModel)DATA_MODEL;
        NEIGHBORS.setText("( " +
            counts[1] + " " + ULTRAPEERS + ", " +
            counts[2] + " " + PEERS + ", " + 
            counts[3] + " " + LEAVES + ", " +
            counts[0] + " " + CONNECTING + ", " +
            counts[4] + " " + STANDARD + " )");
    }

	/**
	 * Called when the user changes their keep alive settings
	 */
    public void processKeepAliveChange(final JTextField source) {
//        String val = source.getText();
//        if ( val == null || val.equals("") || disableKeepAliveValidate )
//            return;
//
//        int n = Integer.parseInt(val);
//        if ( source == MIN_CONNECTIONS_INPUT ) {
//            try {
//                //validate and set the keep alive
//                RouterService.setKeepAlive(n);
//                //store it in the settings also, for persistence
//				ConnectionSettings.NUM_CONNECTIONS.setValue(n);
//            } catch (BadConnectionSettingException e) {
//                int reason=e.getReason();
//                if (reason==BadConnectionSettingException.TOO_HIGH_FOR_SPEED)
//                    showMessage("ERROR_KEEP_ALIVE");
//                else if (reason==BadConnectionSettingException.TOO_LOW_FOR_ULTRAPEER)
//                    showMessage("ERROR_TOO_LOW_FOR_ULTRAPEER");
//                else if (reason==BadConnectionSettingException.TOO_HIGH_FOR_LEAF)
//                    showMessage("ERROR_TOO_HIGH_FOR_LEAF");
//                processKeepAliveSuggestion(e);
//            }
//        }
//		else {
//            Assert.that(false, "Unknown connection property");
//        }
    }

    /**
	 * Called when user's keepAlive/maxIncoming values didn't work.
	 */
    private void processKeepAliveSuggestion(BadConnectionSettingException e) {
//        //Update incoming/outgoing values to suggestion.  Order does
//        //not matter here since these values are not validated.
//        final int outgoing = e.getSuggestedOutgoing();
//		ConnectionSettings.NUM_CONNECTIONS.setValue(outgoing);
//  
//        //Update the display.  Unfortunately we can't just call setText directly
//        //because IllegalStateException will be thrown.  So invoke it later.
//        //This is potentially a race condition, but it's not a fatal one.
//        SwingUtilities.invokeLater(new Runnable() {
//            public void run() {
//                disableKeepAliveValidate=true;
//                MIN_CONNECTIONS_INPUT.setText( String.valueOf(outgoing) );
//                disableKeepAliveValidate=false;
//            }
//        });
//
//        //adjust the connection fetchers
//        RouterService.forceKeepAlive(outgoing);
    }
    
    /**
     * Determines the number of connections that are in connecting state.
     */
    public int getConnectingCount() {
        return ((ConnectionModel)DATA_MODEL).getConnectingCount();
    }

    /**
     * @requires val can be parsed as a number
     * @modifies GUI
     * sets the text for the max number of outgoing connections
     */
    public void setKeepAliveText(String val) {
        disableKeepAliveValidate = true;
        MIN_CONNECTIONS_INPUT.setText( val );
        disableKeepAliveValidate = false;
    }

    private void tryConnection(String hostname, int portnum) {
        RouterService.connectToHostAsynchronously(hostname, portnum);
    }

    /**
     *  Clear the connections visually
     */
    public void clearConnections() {
		DATA_MODEL.clear();
		setKeepAliveText("0");
    }

    /**
     * Adds the host & port to the dictionary of the HOST_INPUT
     */
    void addKnownHost( String host, int port ) {
	    //HOST_INPUT.addToDictionary( host + ":" + port );
	}

    /**
	 * Displays message as an informational dialog.  This method can be
     * called from a non-Swing thread.
	 */
    private void showMessage(final String message) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                GUIMediator.showError(message);
            }
        });
    }

	/**
	 * First attempts to parse out the ':' from the host.
	 * If one exists, it replaces the text in PORT_INPUT.
	 * Otherwise, it uses the text in port input.
	 */
	private final class AddListener implements ActionListener {

        public void actionPerformed(ActionEvent e) {
            try {
                String hostnamestr = HOST_INPUT.getText();
                String portstr = PORT_INPUT.getText();
    
                // look for the port in the host
                int idx = hostnamestr.lastIndexOf(':');
                // if it exists, rewrite the host & port
                if ( idx != -1 ) {
                    PORT_INPUT.setText( hostnamestr.substring(idx+1) );
                    portstr = PORT_INPUT.getText();
                    HOST_INPUT.setText( hostnamestr.substring(0, idx) );
                    hostnamestr = HOST_INPUT.getText();
                }
    
                int portnum = -1;
                try {
                    portnum = Integer.parseInt(portstr);
                } catch (Exception ee) {
                    portnum = 6346;
                }
                if ( !hostnamestr.equals("") )
                    tryConnection(hostnamestr, portnum);
            } catch(Throwable t) {
                ErrorService.error(t);
            }
    	}
    }

    private final class KeepAliveListener implements DocumentListener {
        public void insertUpdate(DocumentEvent e) {
            processKeepAliveChange(source(e));
        }
        public void removeUpdate(DocumentEvent e) {
            processKeepAliveChange(source(e));
        }
        public void changedUpdate(DocumentEvent e) {
        }

        private JTextField source(DocumentEvent e) {
            Document doc=e.getDocument();

            if (doc == MIN_CONNECTIONS_INPUT.getDocument())
                return MIN_CONNECTIONS_INPUT;
            else {
                Assert.that(false, "Unknown connection property");
                return null;
            }
        }
    }

}
