package com.limegroup.gnutella.gui.library;

import com.limegroup.gnutella.FileManager;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.FileDesc;
import com.limegroup.gnutella.settings.SharingSettings;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.FileUtils;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.gui.mp3.PlayListGUI;

import java.io.*;
import javax.swing.*;
import java.awt.*;
import java.util.Vector;


/**
 * This class functions as an initializer for all of the elements
 * of the library and as a mediator between library objects.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class LibraryMediator implements ThemeObserver {

	/**
	 * Singleton instance of this class.
	 */
	private static final LibraryMediator INSTANCE = new LibraryMediator();
    
    /**
     * Constant handle to the <tt>LibraryTree</tt> library controller.
     */
    private static final LibraryTree  LIBRARY_TREE = new LibraryTree();
    
    /**
     * Constant handle to the <tt>LibraryTable</tt> that displays the files
     * in a given directory.
     */
    private static final LibraryTableMediator LIBRARY_TABLE =
        LibraryTableMediator.instance();


    /**
     * Constant handle to the class the constructs the library components.
     */
	private static final LibraryConstructor LIBRARY_CONSTRUCTOR =
		new LibraryConstructor(LIBRARY_TABLE, LIBRARY_TREE);

    /**
     * Constant handle to the class that displays the play list UI.
     */
    private final PlayListGUI PLAY_LIST = PlayListGUI.instance();
    
    /**
     * Constant handle to the file update handler.
     */
    private final HandleFileUpdate FILE_UPDATER = new HandleFileUpdate();
    

	/**
	 * Instance accessor following singleton.  Returns the 
	 * <tt>LibraryView</tt> instance.
	 *
	 * @return the <tt>LibraryView</tt> instance
	 */
	public static LibraryMediator instance() {
		return INSTANCE;
	}

    /** 
	 * Constructs a new <tt>LibraryMediator</tt> instance to manage calls
	 * between library components.
	 */
    private LibraryMediator() {		
		GUIMediator.setSplashScreenString(
		    GUIMediator.getStringResource("SPLASH_STATUS_LIBRARY_WINDOW"));
		GUIMediator.addThemeObserver(this);
    }

	// inherit doc comment
	public void updateTheme() {
		LIBRARY_CONSTRUCTOR.updateTheme();
		LIBRARY_TREE.updateTheme();
	}

	/**
	 * Returns the <tt>JComponent</tt> that contains all of the elements of
	 * the library.
	 *
	 * @return the <tt>JComponent</tt> that contains all of the elements of
	 * the library.
	 */
	public JComponent getComponent() {
		return LIBRARY_CONSTRUCTOR.getComponent();
	}
	
    /**
	 * Tells the library to launch the application associated with the 
	 * selected row in the library. 
	 */
    public void launch() {
		LIBRARY_TABLE.launch();
    }
    
    /**
	 * Deletes the currently selected rows in the table. 
	 */
    public void deleteLibraryFile() {
        LIBRARY_TABLE.removeSelection();
    }
        
	/**
	 * Removes the gui elements of the library tree and table.
	 */
	public void clearLibrary() {
		LIBRARY_TREE.clear();
		LIBRARY_TABLE.clearTable();
	}

	/**
	 * Reloads all file and directories in the library.
	 */
    public void refresh() {
		LIBRARY_TABLE.clearTable();
		if(LIBRARY_TREE.incompleteDirectoryIsSelected()) {
			showIncompleteFiles();
		}
		else {
		    if ( LIBRARY_TREE.getSelectedDirectory() != null ) {
		        loadShareableFiles( LIBRARY_TREE.getSelectedDirectory() );
		        LIBRARY_TABLE.forceResort();
		    }
            FileManager fm = RouterService.getFileManager();
            fm.loadSettings(false);		   
		}
    }
    
    /**
     * Reloads file only if incomplete directory is showing
     */
    public void refreshIfIncomplete() {
        if ( LIBRARY_TREE.incompleteDirectoryIsSelected() )
            refresh();
    }

	/**
	 * Cancels all editing of fields in the tree and table.
	 */
	public void cancelEditing() {
		LIBRARY_TREE.cancelEditing();
	}

	/** 
	 * Launches explorer on PC in selected Shared directory
	 *
	 */
	public static void launchExplorer()
	{
        File[] sharedDirectories = LIBRARY_TREE.getSharedDirectories();
		File exploreDir = null;
		if (sharedDirectories != null && sharedDirectories.length > 0)
		    exploreDir = sharedDirectories[0];
		if ( LIBRARY_TREE.getSelectedDirectory() != null ) 
		    exploreDir = LIBRARY_TREE.getSelectedDirectory();

		if ( exploreDir == null ) return;

		try {
		    String explorePath = exploreDir.getCanonicalPath();	   
			String cmdStr = "";
			if(CommonUtils.isWindows())
			    cmdStr = "explorer"; 
			else if (CommonUtils.isMacOSX()) 	
			    cmdStr = "open"; 
			Runtime.getRuntime().exec(new String[] {cmdStr, explorePath});
		} catch(SecurityException se) {
		} catch(IOException se) {
		}
	}

    /** 
	 * Displays a file chooser for selecting a new folder to share and 
	 * adds that new folder to the settings and FileManager.
	 */
    public void addSharedLibraryFolder() {
		File dir = FileChooserHandler.getInputDirectory();

		if(dir == null || !dir.isDirectory() || !dir.canRead()) {
			GUIMediator.showError("ERROR_INVALID_SHARED_DIRECTORY");
			return;
		}
		
		try {
			SharingSettings.addDirectory(dir);
			// Rescan the directores into the FileManager.  
			// The loadSettings method is non-blocking, 
			// so this is safe to call here.  
			RouterService.getFileManager().loadSettings(false);
		} catch(IOException ioe) {				
		}			
    }

    /** 
	 * Refreshes the shared directories in the settings manager as well 
	 * as the files in the file manager based on the directories in the
	 * library tree.
	 */
    public void handleRootSharedDirectoryChange() {
        final File[] sharedDirectories = LIBRARY_TREE.getSharedDirectories();
        SharingSettings.DIRECTORIES_TO_SHARE.
            setValue(sharedDirectories);
        RouterService.getFileManager().loadSettings(true);
    }

	/**
	 * Adds a shared directory to the library.
	 * 
	 * @param dir     a <tt>File</tt> instance denoting the abstract
	 *                pathname of the shared directory to add
	 *
	 * @param parent  a <tt>File</tt> instance denoting the abstract
	 *                pathname of the shared directory's parent directory
	 */
	public void addSharedDirectory(final File dir, final File parent) {
		LIBRARY_TREE.addSharedDirectory(dir, parent);
	}

	/**
	 * Adds a shared file to the library if its parent directory is
	 * currently selected in the library tree.
	 * 
	 * @param file    a <tt>FileDesc</tt> instance denoting the abstract
	 *                pathname of the shared file to add
	 *
	 * @param parent  a <tt>File</tt> instance denoting the abstract
	 *                pathname of the shared file's parent directory
	 *
	 */
	public void addSharedFile(final FileDesc file, final File parent) {
		if(LIBRARY_TREE.parentIsSelected(parent))
			LIBRARY_TABLE.add(file);
	}
	
	/**
	 * Update the this file's statistic
	 */
	public void updateSharedFile(final File file) {
	    File selDir = LIBRARY_TREE.getSelectedDirectory();
	    // if the library table is visible, and
	    // if the selected directory is null
	    // or if we the file exists in a directory
	    // other than the one we selected, then there
	    // is no need to update.
	    // the user will see the newest stats when he/she 
	    // selects the directory.
		if( selDir != null && 
		    selDir.equals(FileUtils.getParentFile(file)) &&
		    LIBRARY_TABLE.getTable().isShowing() ) {
		    // pass the update off to the file updater
		    // this way, only one Runnable is ever created,
		    // instead of allocating a new one every single time
		    // a query is hit.
		    // Very useful for large libraries and generic searches (ala: mp3)
		    FILE_UPDATER.addFileUpdate(file);
	    }
	}
	
	public void setAnnotateEnabled(boolean enabled) {
	    LIBRARY_TABLE.setAnnotateEnabled(enabled);
	}
	

    /**
	 * Adds a new folder to the library. the new folder is a subdirectory 
	 * of the selcected folder. 
	 */
    public void addNewLibraryFolder() {
        LIBRARY_TREE.addNewLibraryFolder();
    }

    /** 
	 * Removesthe selected folder from the shared folder group.. 
	 */
    public void unshareLibraryFolder() {
        LIBRARY_TREE.unshareLibraryFolder();
    }

    /** 
	 * Renames the selected folder in the library. 
	 */
    public void renameLibraryFolder() {
        LIBRARY_TREE.renameLibraryFolder();
    }


    /** call this if you want the playlist to add a file.
     */
    void addFileToPlayList(File toAdd) {
        PLAY_LIST.addFileToPlayList(toAdd);
    }


    /** call this if you want the specified file to be played.
     */
    void launchAudio(File toPlay) {
        PLAY_LIST.launchAudio(toPlay);
    }


    /** 
	 * Obtains the shared files for the given directory and updates the 
	 * table accordingly.
	 *
	 * @param selectedDir the currently selected directory in
	 *        the library
	 */
    void updateTableFiles(File selectedDir) {
		FileDesc[] files = 
		    RouterService.getSharedFileDescriptors(selectedDir);
		LIBRARY_TABLE.clearTable();
		
		// this is called when the non-incomplete directory is selected,
		// so disable resume
		LIBRARY_TABLE.setResumeEnabled(false);
		
		// Add stuff unsorted and then sort afterwards so we don't
		// rewrite the hashmap after every add.
		if(files != null)
		    for( int i = 0; i < files.length; i++)
				LIBRARY_TABLE.addUnsorted(files[i]);
				
        //after the shared files are added, add the non-shared files.
        // any already shared files are ignored.
        //Small optimization:
        if ( RouterService.getNumPendingShared() > 0 ) {
            loadShareableFiles(selectedDir);
        }
        
        LIBRARY_TABLE.forceResort(); 
    }
    
    /**
     * Loads all sharable files in the specified directory into
     * the table.
     *
     * To maintain sorting, must force a resort after adding.
     */
    void loadShareableFiles(File dir) {
        File[] pending = FileUtils.listFiles( dir );
        if ( pending != null ) {
            for(int i = 0; i < pending.length; i++) {
                if(FileManager.isFileShareable(pending[i],
                                               pending[i].length())
                  )
                    LIBRARY_TABLE.addUnsorted(pending[i]);
            }
        } 
    }        

	/**
	 * Displays the files in the incomplete directory.
	 */
	void showIncompleteFiles() {
		LIBRARY_TABLE.clearTable();
  		File incFile = SharingSettings.INCOMPLETE_DIRECTORY.getValue();
  		if( incFile == null ) {
			// we cannot do anything if we could not get the directory,
			// so simply return
			return;
		}
		
		// this is called when the incomplete directory is selected,
		// so enable resume
		LIBRARY_TABLE.setResumeEnabled(true);		
		
		// Ask for the File Descriptors of shared incomplete files ...
		FileDesc[] files = RouterService.getIncompleteFileDescriptors();		
		// Add stuff unsorted and then sort afterwards so we don't
		// rewrite the hashmap after every add.
		if(files != null)
		    for( int i = 0; i < files.length; i++)
				LIBRARY_TABLE.addUnsorted(files[i]);		
		
		
		// Then go back and add other non-shared files.
		String[] fileNames = incFile.list();		
		if(fileNames != null) {
			File curFile;
			for( int i = 0 ; i < fileNames.length; i++ )
				// do not show the download.dat file
				if(!fileNames[i].equals("downloads.dat") &&
				   !fileNames[i].equals("downloads.bak")) {
					curFile = new File(incFile, fileNames[i]);
					if( curFile.isFile() ) LIBRARY_TABLE.addUnsorted(curFile);
			    }
		}
		
		LIBRARY_TABLE.forceResort();
	}


	/**
	 * Returns a boolean specifying whether or not a drag and drop event
	 * is currently allowable in the table.
	 *
	 * @return <tt>true</tt> if a drag and drop event is currently 
	 *         allowable, <tt>false</tt> otherwise
	 */
	boolean canDragAndDrop() {
		return LIBRARY_TABLE.canDragAndDrop();
	}

    /**
	 * Sets the flag for whether a muliple selection is active in
     * in the table. 
	 *
	 * @param multiSelection specifies whether or not there are multiple
	 *                       rows currently selected in the table
	 */
    void setMultiSelection(boolean multiSelection) {
        LIBRARY_TABLE.setMultiSelection(multiSelection);
    }

    /** 
	 * Returns a boolean for whether or not the multiple selection flag 
	 * in the table is on. 
	 *
	 * @return <tt>true</tt> if there are currently multiple rows 
	 *         selected in the table, <tt>false</tt> otherwise
	 */
    boolean getTableMultiSelection() {
        return LIBRARY_TABLE.getMultiSelection();
    }


	/**
	 * Handles a potential "drop" event  for files moved to a 
	 * new folder from the table.
	 *
	 * @param mousePoint A <tt>Point</tt> instance specifying
	 *                   the location where the mouse was released.
	 */
	void handleDropToTreePoint(Point mousePoint) {
		File file = LIBRARY_TREE.getFileForPoint(mousePoint);	

		// set the boolean for whether or not we are dropping a file
		// to the Incomplete folder
		boolean droppingToIncomplete =
		    LIBRARY_TREE.droppingToIncompleteFolder(mousePoint);

		// set the boolean for whether or not we are dragging from
		// the incomplete folder to another folder
		boolean draggingFromIncomplete =
		    LIBRARY_TREE.incompleteDirectoryIsSelected();
		
		if(file != null) {
			String newFilePath = "";
			try {
				newFilePath = file.getCanonicalPath();
			} catch(IOException ioe) {
				return;
			}
			int[] rows = LIBRARY_TABLE.getSelectedRows();
			File oldFile;
			File newFile;
			// rename the moved files to use their new
			// path values.
			for( int length = rows.length - 1; length >= 0; length--) {
				oldFile = LIBRARY_TABLE.getFile(rows[length]);					
				newFile = new File(newFilePath, oldFile.getName());
				boolean renamed = oldFile.renameTo(newFile);
				if(renamed) {
					if(droppingToIncomplete) 
						// handle dropping to the incomplete folder,
						// simply unsharing the file
						handleDropToIncomplete(oldFile);
										
					else if(draggingFromIncomplete) 
						// handle dragging from the incomplete to a
						// shared folder, simply adding it to the
						// FileManager
						handleDragFromIncomplete(newFile);
					else 
						// otherwise simple rename the file in the
						// FileManager
						handleStandardDragAndDrop(oldFile, newFile);
				}
			}
		}
	}

	/**
	 * Sets the cursor in the tree and the table.
	 *
	 * @param cursor the <tt>Cursor</tt> to set in the tree and the table
	 */
	void setCursors(java.awt.Cursor cursor) {
		LIBRARY_TREE.setCursor(cursor);
		LIBRARY_TABLE.setCursor(cursor);
	}

	/**
	 * Returns the location of the LibraryTable on the screen.
	 *
	 * @return the <tt>Point</tt> instance containing the x,y position
	 *         of the table on the screen
	 */
	Point getTableLocation() {
		return LIBRARY_TABLE.getTableLocation();
	}

	/**
	 * Returns the location of the LibraryTree on the screen.
	 *
	 * @return the <tt>Point</tt> instance containing the x,y position
	 *         of the tree on the screen
	 */
	Point getTreeLocation() {
		return LIBRARY_TREE.getTreeLocation();
	}

	/**
	 * Returns the size of the LibraryTable on the screen.
	 *
	 * @return the <tt>Dimension</tt> instance containing the width and 
	 *         height of table
	 */
	Dimension getTableSize() {
		return LIBRARY_TABLE.getTableSize();
	}

	/**
	 * Returns the size of the LibraryTree on the screen.
	 *
	 * @return the <tt>Dimension</tt> instance containing the width and 
	 *         height of tree
	 */
	Dimension getTreeSize() {
		return LIBRARY_TREE.getTreeSize();
	}

    /**
	 * Returns the stored row in the table that could be the next selected row 
	 * depending on various conditions.
	 *
	 * @return The row of the current index candidate
	 */
    int getTableIndexCandidate() {
		return LIBRARY_TABLE.getIndexCandidate();
    }

	/**
	 * Returns the <tt>ListSelectionModel</tt> of the library table.
	 *
	 * @return the <tt>ListSelectionModel</tt> for the library table.
	 */
	ListSelectionModel getTableSelectionModel() {
		return LIBRARY_TABLE.getSelectionModel();
	}

	/**
	 * Handles the drag and drop case when a file is dragged from a shared 
	 * folder to the incomplete folder, simply removing that file from the 
	 * shared files in the FileManager.
	 *
	 * @param draggedFile the <tt>File</tt> instance denoting the abstract
	 *                    path of the file that was dropped to the incomplete
	 *                    directory
	 */
	private void handleDropToIncomplete(File draggedFile) {
		RouterService.getFileManager().removeFileIfShared(draggedFile);
		updateTableFiles(LIBRARY_TREE.getSelectedDirectory());
	}

	/**
	 * Handles the drag and drop case when a file is dragged from the 
	 * incomplete folder to a shared folder, simply adding that file to the 
	 * list of shared files.
	 *
	 * @param draggedFile the <tt>File</tt> instance denoting the abstract
	 *                    pathname of the file dragged from the incomplete 
	 *                    directory
	 */
	private void handleDragFromIncomplete(File draggedFile) {
		RouterService.getFileManager().addFileIfShared(draggedFile);
		showIncompleteFiles();
	}

	/**
	 * Handles FileManger interaction when a drag and drop event does not 
	 * involve the incomplete folder.
	 *
	 * @param oldFile the <tt>File</tt> instance denoting the old abstract
	 *                pathname of the file
	 *
	 * @param newFile the <tt>File</tt> instance denoting the new abstract
	 *                pathname of the file
	 */
	private void handleStandardDragAndDrop(File oldFile, File newFile) {
		RouterService.getFileManager().
		    renameFileIfShared(oldFile, newFile);
		updateTableFiles(LIBRARY_TREE.getSelectedDirectory());
	}
	

    /** Returns true if this is showing the special incomplete directory,
     *  false if showing normal files. */
    public boolean incompleteDirectoryIsSelected() {
        return LIBRARY_TREE.incompleteDirectoryIsSelected();        
    }
    
    /**
     *  Class to handle updates to shared file stats
     *  without creating tons of runnables.
     *  Idea taken from HandleQueryString in VisualConnectionCallback
     */
    private static final class HandleFileUpdate implements Runnable 
    {
        private Vector  list;
        private boolean active;
    
        public HandleFileUpdate( ) {
            list   = new Vector();
            active = false;
        }
    
        public void addFileUpdate(File f) {
            list.addElement(f);
            if(active == false) {
                active = true;
                SwingUtilities.invokeLater(this);
            }
        }
    
        public void run() {
            try {
                File f;
                while (list.size() > 0) {
                    f = (File) list.firstElement();
                    list.removeElementAt(0);
    			    LIBRARY_TABLE.update(f);
                }
             } catch (IndexOutOfBoundsException e) {
        	    //this really should never happen, but
        	    //who really cares if we're not sharing it?
             } catch (Exception e) {
                //no other errors could happen, so if one does, something's wrong
                GUIMediator.showInternalError(e);
             }
             active = false;
        }
    }    
}
