package com.limegroup.gnutella.gui.library;


import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.io.*;
import java.awt.event.*;
import java.awt.*;
import java.net.URLEncoder;
import com.sun.java.util.collections.*;
import com.limegroup.gnutella.gui.tables.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.xml.gui.*;
import com.limegroup.gnutella.xml.*;
import com.limegroup.gnutella.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.downloader.AlreadyDownloadingException;
import com.limegroup.gnutella.downloader.CantResumeException;

/**
 * This class wraps the JTable that displays files in the library,
 * controlling access to the table and the various table properties.
 * It is the Mediator to the Table part of the Library display.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class LibraryTableMediator extends AbstractTableMediator {

	/**
	 * Reference to the <tt>LibraryMediator</tt> that acts as the hub of
	 * communication for library components.
	 */
	private static final LibraryMediator LIBRARY_MEDIATOR =
	    LibraryMediator.instance();

    /**
     * Variables so the PopupMenu & ButtonRow can have the same listeners
     */
    public static ActionListener LAUNCH_LISTENER;
    public static ActionListener ADD_PLAY_LIST_LISTENER;
    public static ActionListener ANNOTATE_LISTENER;
    public static ActionListener BITZI_LOOKUP_LISTENER;
    public static ActionListener MAGNET_LOOKUP_LISTENER;
    public static ActionListener RESUME_LISTENER;

	/**
	 * Variable for whether or not the shift key is pressed.
	 */
    private boolean _shiftPressed;

    private boolean _multiSelection;
    private int _indexCandidate;

    private boolean _allowAnnotate;
    private boolean _allowResume;

    /**
     * instance, for singelton access
     */
    private static LibraryTableMediator _instance = new LibraryTableMediator();

    public static LibraryTableMediator instance() { return _instance; }

    /**
     * Build some extra listeners
     */
    protected void buildListeners() {
        super.buildListeners();
        LAUNCH_LISTENER = new LaunchListener();
        ADD_PLAY_LIST_LISTENER = new AddPLFileListener();
        ANNOTATE_LISTENER = new AnnotateListener();
        BITZI_LOOKUP_LISTENER = new BitziLookupListener();
        MAGNET_LOOKUP_LISTENER = new MagnetLookupListener();
        RESUME_LISTENER = new ResumeListener();
    }

    /**
     * Set up the constants
     */
    protected void setupConstants() {
		MAIN_PANEL = null;
		DATA_MODEL = new LibraryTableModel();
		TABLE = new LimeJTable(DATA_MODEL);
		((LibraryTableModel)DATA_MODEL).setTable(TABLE);
		BUTTON_ROW = (new LibraryTableButtons(this)).getComponent();
    }

    // inherit doc comment
    protected JPopupMenu createPopupMenu() {
        JPopupMenu menu = (new LibraryTablePopupMenu(this)).getComponent();
        
        boolean selection = !TABLE.getSelectionModel().isSelectionEmpty();

        if(selection) {
            if ( GUIMediator.isPlaylistVisible() ) {
                menu.getComponent(LibraryTablePopupMenu.PLAYLIST_INDEX).
                    setEnabled(true);
            }

            menu.getComponent(LibraryTablePopupMenu.ANNOTATE_INDEX).
                 setEnabled(_allowAnnotate);

            menu.getComponent(LibraryTablePopupMenu.RESUME_INDEX).
                 setEnabled(_allowResume);

            menu.getComponent(LibraryTablePopupMenu.LAUNCH_INDEX).
                setEnabled(true);
            menu.getComponent(LibraryTablePopupMenu.DELETE_INDEX).
                setEnabled(true);
            menu.getComponent(LibraryTablePopupMenu.BITZI_INDEX).
                setEnabled(true);            
            menu.getComponent(LibraryTablePopupMenu.MAGNET_INDEX).
                setEnabled(true);            
        } else {
            if ( GUIMediator.isPlaylistVisible() ) {
                menu.getComponent(LibraryTablePopupMenu.PLAYLIST_INDEX).
                    setEnabled(false);
            }
            
            menu.getComponent(LibraryTablePopupMenu.ANNOTATE_INDEX).
                setEnabled(false);            
            menu.getComponent(LibraryTablePopupMenu.RESUME_INDEX).
                setEnabled(false);            
            menu.getComponent(LibraryTablePopupMenu.LAUNCH_INDEX).
                setEnabled(false);
            menu.getComponent(LibraryTablePopupMenu.DELETE_INDEX).
                setEnabled(false);
            menu.getComponent(LibraryTablePopupMenu.BITZI_INDEX).
                setEnabled(false);
            menu.getComponent(LibraryTablePopupMenu.MAGNET_INDEX).
                setEnabled(false);
        }
        return menu;
    }

    /**
     * Sets the default editors.
     */
    protected void setDefaultEditors() {
        TableColumnModel model = TABLE.getColumnModel();
        TableColumn tc = model.getColumn(LibraryTableDataLine.NAME_IDX);
        tc.setCellEditor(new LibraryTableCellEditor(this));
    }

    /**
     * Update the table appropriately.
     */
    protected void setupTable() {
        super.setupTable();

		TABLE.addKeyListener( new LibraryKeyListener() );

		ListSelectionModel selectionModel = new LibraryListSelectionModel();
		selectionModel.setSelectionMode(ListSelectionModel.SINGLE_INTERVAL_SELECTION);
        TABLE.setSelectionModel(selectionModel);
    }

    /**
     * Upgrade getScrolledTablePane to public access for the LibraryConstructor
     */
    public JComponent getScrolledTablePane() {
        return super.getScrolledTablePane();
    }

    /* Don't display anything for this.  The LibraryMediator will do it. */
	protected void updateSplashScreen() {}

    /**
     * Note: This is set up for this to work.
     * Polling is not needed though, because updates
     * already generate update events.
     */
    private LibraryTableMediator() {
        super("LIBRARY_TABLE");
        //GUIMediator.addRefreshListener(this);
        GUIMediator.addThemeObserver(this);
    }

	/**
	 * there is no actual component that holds all of this table.
	 * The LibraryMediator is real the holder.
	 */
	public JComponent getComponent() {
		return null;
	}

	/**
	 * Cancels all editing of fields in the tree and table.
	 */
	void cancelEditing() {
	}

	/**
	 * Adds the mouse listeners to the wrapped <tt>JTable</tt>.
	 *
	 * @param listener the <tt>MouseInputListener</tt> that handles mouse events
	 *                 for the library
	 */
	void addMouseInputListener(final MouseInputListener listener) {
        TABLE.addMouseListener(listener);
        TABLE.addMouseMotionListener(listener);
	}

	/**
	 * Allows or disallows annotation
	 *
	 * @param enabled whether or not annotation is allowed
	 */
	public void setAnnotateEnabled(boolean enabled) {
	    _allowAnnotate = enabled;
	    LibraryTableDataLine.setXMLEnabled(enabled);
	    //disable the annotate buttons if we are turning annotation off.
	    if ( ! enabled ) {
		    setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, false);
		//if it's turning on and something is selected, enable the annotate button.
	    } else if ( !TABLE.getSelectionModel().isSelectionEmpty() ) {
            setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, true);
        }
	}

	/**
	 * Allows or disallows resuming
	 *
	 * @param enabled whether or not resuming is allowed
	 */
	void setResumeEnabled(boolean enabled) {
	    _allowResume = enabled;
	    //disable the resume buttons if we are turning resume off.
	    if ( ! enabled ) {
	        setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, false);
	    } else if ( !TABLE.getSelectionModel().isSelectionEmpty() ) {
	        setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, true);
	    }
	}


    /**
	 * Returns the <tt>File</tt> stored at the specified row in the list.
	 *
	 * @param row the row of the desired <tt>File</tt> instance in the
	 *            list
	 *
	 * @return a <tt>File</tt> instance associated with the specified row
	 *         in the table
	 */
    File getFile(int row) {
		return ((LibraryTableModel)DATA_MODEL).getFile(row);
    }

    /**
	 * Sets the flag for whether a muliple selection of rows is active
	 * in the table.
	 *
	 * @param multiSelection specifies whether there are currently multiple
	 *                       rows selected in the table
	 */
    void setMultiSelection(boolean multiSelection) {
        _multiSelection = multiSelection;
    }

    /**
	 * Accessor for whether or not the multiple selection
	 * flag in the table is on.
	 *
	 * @return <tt>true</tt> if there are currently multiple rows selected
	 *         in the table, <tt>false</tt> otherwise
	 */
    boolean getMultiSelection() {
        return _multiSelection;
    }

    /**
	 * Returns the stored row in the table that could be the next selected row
	 * depending on various conditions.
	 *
	 * @return The row of the current index candidate
	 */
    int getIndexCandidate() {
		return _indexCandidate;
    }

    /**
	 * Accessor for the table that this class wraps.
	 *
	 * @return The <tt>JTable</tt> instance used by the library.
	 */
    JTable getTable() {
        return TABLE;
    }

    ButtonRow getButtonRow() {
        return BUTTON_ROW;
    }

	/**
	 * Accessor for the <tt>ListSelectionModel</tt> for the wrapped
	 * <tt>JTable</tt> instance.
	 */
	ListSelectionModel getSelectionModel() {
		return TABLE.getSelectionModel();
	}


    /**
     * shows the user a meta-data for the file(if any) and allow the user
     * to edit it.
     */
    void editMeta(){
        //get the selected file. If there are more than 1 we just use the
        // last one.
        int[] rows = TABLE.getSelectedRows();
        int k = rows.length;
        if(k == 0)
            return;
        int index = rows[k-1];//this is the index of the last row selected
        FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(index);
        if( fd == null ) // oh well
            return;

        String fullName = "";
        try{
            fullName = fd.getFile().getCanonicalPath();
        }catch(IOException ee){//if there is an exception return
            return;
        }
        //Now docsOfFile has all LimeXMLDocuments pertainint to selected file
        Frame mainF = GUIMediator.getAppFrame();
        MetaEditorFrame metaEditor = new MetaEditorFrame(fd, fullName,mainF);
        metaEditor.show();
    }

	/**
	 * Performs the Bitzi lookup for the selected files in the library.
	 */
    void doBitziLookup() {
        // get the selected file. If there are more than 1 we just use the
        // last one.
        int[] rows = TABLE.getSelectedRows();
        int k = rows.length;
        if(k == 0)
            return;
        int index = rows[k-1];//this is the index of the last row selected
        FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(index);
        if (fd==null) {
            // noop
            return;
        }
        URN urn = fd.getSHA1Urn();
		if(urn==null) {
            // unable to do lookup -- perhaps SHA1 not calculated yet?;
            // TODO could show dialog suggesting user try again later but won't for now
            return;
        }
        String urnStr = urn.toString();
        int hashstart = 1+urnStr.indexOf(":",4);
        // TODO: grab this lookup URL from a template somewhere
        String lookupUrl = "http://bitzi.com/lookup/"+urnStr.substring(hashstart)+"?ref=limewire";
        try {
            Launcher.openURL(lookupUrl);
        } catch (IOException ioe) {
            // do nothing
        }
    }

    /**
     * Prepare a detail page of magnet link info for selected files 
     * in the library.
     */
    void doMagnetLookup() {
        doMagnetCommand("/magcmd/detail?");
    }
    /**
     * Fire a local lookup with file/magnet details
     */
    void doMagnetCommand(String cmd) {
        // get the selected files.  Build up a url to display details.
        int[] rows = TABLE.getSelectedRows();
        int k = rows.length;
        if(k == 0)
            return;

        boolean haveValidMagnet = false;

        int    count     = 0;
        int    port      = RouterService.getHTTPAcceptor().getPort();
        int    eport     = RouterService.getAcceptor().getPort(true);
        byte[] eaddr     = RouterService.getAcceptor().getAddress(true);
        String lookupUrl = "http://localhost:"+port+
          cmd+
          "addr="+NetworkUtils.ip2string(eaddr)+":"+eport;
        for(int i=0; i<k; i++) {
            FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(rows[i]);
            if (fd==null) {
                // Only report valid files
                continue;
            }
            URN    urn       = fd.getSHA1Urn();
            String urnStr    = urn.toString();
            int    hashstart = 1+urnStr.indexOf(":",4);
            if(urn==null) {
                // Only report valid sha1s
                continue;
            } 
            String sha1=urnStr.substring(hashstart);
            lookupUrl +=
              "&n"+count+"="+URLEncoder.encode(fd.getName())+
              "&u"+count+"="+sha1;
            count++;
            haveValidMagnet = true;
        }
        if (haveValidMagnet) {
            try {
                Launcher.openURL(lookupUrl);
            } catch (IOException ioe) {
                // do nothing
            }
        }
    }

    /**
     * Override the default removal so we can actually stop sharing
     * and delete the file.
	 * Deletes the selected rows in the table.
	 * CAUTION: THIS WILL DELETE THE FILE FROM THE DISK.
	 */
	public void removeSelection() {


		String msgKey = "MESSAGE_CONFIRM_FILE_DELETE";
		int response = GUIMediator.showYesNoMessage(msgKey);
		if(response != GUIMediator.YES_OPTION) return;

        int[] rows = TABLE.getSelectedRows();
		if(rows.length <= 0) return;

		Arrays.sort(rows);

		if(TABLE.isEditing()) {
			TableCellEditor editor = TABLE.getCellEditor();
			editor.cancelCellEditing();
		}
		
		ArrayList errors = new ArrayList();

		for(int i = rows.length - 1; i >= 0; i--) {
			File file = ((LibraryTableModel)DATA_MODEL).getFile(rows[i]);
			if( file.delete() ) {
			    DATA_MODEL.remove(rows[i]);
				// make sure the file manager
				// stops sharing the file
				RouterService.getFileManager().removeFileIfShared(file);
			} else {
			    // append all errors to a list of errors...
			    errors.add( file.getName() );
			}
		}

		clearSelection();		
		
		// go through the errors and tell them what couldn't be deleted.
		for(int i = 0; i < errors.size(); i++) {
		    String name = (String)errors.get(i);
			final String key1 = "MESSAGE_UNABLE_TO_DELETE_FILE_START";
			final String key2 = "MESSAGE_UNABLE_TO_DELETE_FILE_END";
			final String msg = "'" + name + "'.";
			// notify the user that deletion failed
			GUIMediator.showError(key1, msg, key2);
		}
    }

    public void handleMouseDoubleClick(MouseEvent e) {
        launch();
    }

    /**
     * Resume incomplete downloads
     */    
    void resumeIncomplete() {        
        //For each selected row...
        int[] rows = TABLE.getSelectedRows();
        boolean startedDownload=false;
        ArrayList errors = new ArrayList();
        for (int i=0; i<rows.length; i++) {
            //...try to download the incomplete
            File incomplete=((LibraryTableModel)DATA_MODEL).getFile(rows[i]);
            try {
                RouterService.download(incomplete);
                startedDownload=true;
            } catch (AlreadyDownloadingException e) { 
                // we must cache errors to display later so we don't wait
                // while the table might change in the background.
                errors.add(e);
            } catch(CantResumeException e) {
                errors.add(e);
            }
        }
        
        // traverse back through the errors and show them.
        for(int i = 0; i < errors.size(); i++) {
            Exception e = (Exception)errors.get(i);
            if( e instanceof AlreadyDownloadingException ) {
                GUIMediator.showError("ERROR_ALREADY_DOWNLOADING",
                    "\""+((AlreadyDownloadingException)e).getFilename()+"\".",
                    QuestionsHandler.ALREADY_DOWNLOADING);
            } else if ( e instanceof CantResumeException ) {
                GUIMediator.showError("ERROR_CANT_RESUME_START",
                    "\""+((CantResumeException)e).getFilename()+"\"",
                    "ERROR_CANT_RESUME_END",
                    QuestionsHandler.CANT_RESUME);
            }
        }       

        //Switch to download tab (if we actually started anything).
        if (startedDownload)
            GUIMediator.instance().setWindow(GUIMediator.SEARCH_INDEX);
    }

    

    /**
	 * Launches the associated applications for each selected file
	 * in the library if it can.
	 */
    void launch() {
        int[] rows = TABLE.getSelectedRows();
        boolean mp3Launched = false;
		for(int i = 0, l = rows.length; i < l; i++) {
			try {
                File currFile = ((LibraryTableModel)DATA_MODEL).getFile(rows[i]);
                if (GUIMediator.isPlaylistVisible() &&
					(CommonUtils.canLaunchFileWithQuickTime(currFile) ||
					 LimeXMLUtils.isMP3File(currFile))) {
                    // start playing the first mp3 you encounter.  everything
                    // else (mp3) is dropped on the floor
                    if (!mp3Launched) {
                        LIBRARY_MEDIATOR.launchAudio(currFile);
                        mp3Launched = true;
                    }
                }
                else
                    GUIMediator.launchFile(currFile);
			} catch(IOException ioe) {
                // TODO: we should probably propagate and display this 
                // exception, although it will happen often on Linux
				// all we can do is try to launch the file.
			}
		}
    }

	/**
	 * Handles a name change of one of the files displayed.
	 *
	 * @param newName The new name of the file
	 *
	 * @return A <tt>String</tt> that is the name of the file
	 *         after this method is called. This is the new name if
	 *         the name change succeeded, and the old name otherwise.
	 */
	String handleNameChange(String newName) {
		int row = TABLE.getEditingRow();
		File oldFile = ((LibraryTableModel)DATA_MODEL).getFile(row);
		String parent = oldFile.getParent();
		String nameToReturn = newName;
		newName = newName + "." + ((LibraryTableModel)DATA_MODEL).getType(row);
		File newFile = new File(parent, newName);
		if(!(((LibraryTableModel)DATA_MODEL).getName(row)).equals(newName)) {
			boolean b = oldFile.renameTo(newFile);
			if(b) {
				((LibraryTableModel)DATA_MODEL).handleFileChangeInternal(row, newFile);			    
				RouterService.
                       getFileManager().renameFileIfShared(oldFile, newFile);
				return nameToReturn;
			}
			else {
				final String key1 = "MESSAGE_UNABLE_TO_RENAME_FILE_START";
				final String key2 = "MESSAGE_UNABLE_TO_RENAME_FILE_END";
				final String msg = "'" + ((LibraryTableModel)DATA_MODEL).getName(row) + "'.";
				// notify the user that renaming failed
				GUIMediator.showError(key1, msg, key2);
				return ((LibraryTableModel)DATA_MODEL).getName(row);
			}
		}
		return nameToReturn;
	}

	/**
	 * Determines whether or not we can perform a drag and drop based on
	 * whether or not there is a row being edited in the table and
	 * whether or not there are rows selected in the table.
	 *
     * @return  <tt>true</tt> if a drag and drop is currently allowed,,
     *          <tt>false</tt> otherwise
	 */
	boolean canDragAndDrop() {
		int[] rows = TABLE.getSelectedRows();
		if((rows.length > 0) && (!TABLE.isEditing())) {
			return true;
		}
		return false;
	}

	/**
	 * Sets the cursor in the table.
	 *
	 * @param cursor the <tt>Cursor</tt> to set in the table
	 */
	void setCursor(java.awt.Cursor cursor) {
		TABLE.setCursor(cursor);
	}

	/**
	 * Returns the location of the <tt>JTable</tt> on the screen.
	 *
	 * @return the <tt>Point</tt> instance containing the x,y position
	 *         of the <tt>JTable</tt> on the screen
	 */
	Point getTableLocation() {
		return TABLE.getLocationOnScreen();
	}



	/**
	 * Returns the size of the <tt>JTable</tt> on the screen.
	 *
	 * @return the <tt>Dimension</tt> instance containing the width and
	 *         height of <tt>JTable</tt>
	 */
	Dimension getTableSize() {
		return TABLE.getSize();
	}

	/**
	 * Returns the selected rows in the table.
	 *
	 * @return an array of integers containing the indices of all selected rows,
	 *         or an empty array if no row is selected
	 */
	int[] getSelectedRows() {
		return TABLE.getSelectedRows();
	}

	/**
	 * Handles the selection of the specified row in the library window,
	 * enabling or disabling buttons and chat menu items depending on
	 * the values in the row.
	 *
	 * @param row the selected row
	 */
	public void handleSelection(int row) {
		setButtonEnabled(LibraryTableButtons.LAUNCH_BUTTON, true);
		setButtonEnabled(LibraryTableButtons.DELETE_BUTTON, true);

		if ( GUIMediator.isPlaylistVisible() ) {
		    setButtonEnabled(LibraryTableButtons.PLAYLIST_BUTTON, true);
		}

		if ( _allowAnnotate ) {
		    setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, true);
		}

		if ( _allowResume ) {
		    setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, true);
		}
	}

	/**
	 * Handles the deselection of all rows in the library table,
	 * disabling all necessary buttons and menu items.
	 */
	public void handleNoSelection() {
		setButtonEnabled(LibraryTableButtons.LAUNCH_BUTTON, false);
		setButtonEnabled(LibraryTableButtons.DELETE_BUTTON, false);

		if ( GUIMediator.isPlaylistVisible() ) {
		    setButtonEnabled(LibraryTableButtons.PLAYLIST_BUTTON, false);
		}

        setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, false);
		setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, false);
	}

    /**
	 * Class that handles a deletion action from the keyboard as well as
	 * pressing of the shift key for file selection.
	 */
    private final class LibraryKeyListener implements KeyListener {
        public void keyTyped(KeyEvent ke) {
			try {
				if(ke.getKeyChar() != KeyEvent.VK_BACK_SPACE) {
					return;
				}
				removeSelection();
			} catch(Throwable e) {
				GUIMediator.showInternalError(e, "LibraryKeyListener");
			}
        }

        // sets the flag for whether or not shift
        // is pressed to true.  this is useful
        // for selection handling in the table,
        // as selections are handled differently
        // when shift is held down.
        public void keyPressed(KeyEvent e) {
            if(e.getKeyCode() == KeyEvent.VK_SHIFT)
                _shiftPressed = true;
        }

        // sets the shift key flag to false
        // if shift is released.
        public void keyReleased(KeyEvent e) {
            if(e.getKeyCode() == KeyEvent.VK_SHIFT)
                _shiftPressed = false;
        }
    }

    ///////////////////////
    // A collection of private listeners
    //////////////////////

    private final class AnnotateListener implements ActionListener {
    	public void actionPerformed(ActionEvent ae) {
    		try {
    			editMeta();
    		} catch(Throwable e) {
    			GUIMediator.showInternalError(e, "AnnotateListener");
    		}
    	}
    }

    private final class ResumeListener implements ActionListener {
    	public void actionPerformed(ActionEvent ae) {
    		try {
                resumeIncomplete();
    		} catch(Throwable e) {
    			GUIMediator.showInternalError(e, "ResumeListener");
    		}
    	}
    }

    private final class BitziLookupListener implements ActionListener {
        public void actionPerformed(ActionEvent e){
            try {
                doBitziLookup();
            } catch(Throwable t) {
                ErrorService.error(t);
            }
        }
    }

    private final class MagnetLookupListener implements ActionListener {
        public void actionPerformed(ActionEvent e){
            try {
                doMagnetLookup();
            } catch(Throwable t) {
                ErrorService.error(t);
            }
        }
    }

    private final class LaunchListener implements ActionListener {

    	public void actionPerformed(ActionEvent ae) {
    		try {
    			launch();
    		} catch(Throwable e) {
    			GUIMediator.showInternalError(e, "Library.LaunchListener");
    		}
    	}
    }


    private final class AddPLFileListener implements ActionListener{
        public void actionPerformed(ActionEvent ae){
			try {
				//get the selected file. If there are more than 1 we add all
				int[] rows = TABLE.getSelectedRows();
				for (int i = 0; i < rows.length; i++) {
					int index = rows[i]; // current index to add
					File file = ((LibraryTableModel)DATA_MODEL).getFile(index);
					LIBRARY_MEDIATOR.addFileToPlayList(file);
				}
			} catch(Throwable e) {
				GUIMediator.showInternalError(e, "AddPLFileListener");
			}
        }
    }



    /**
	 * Class that overrides some of the functionality of the
	 * DefaultListSelectionListener class to get around the way it
	 * handles row selection on mouse drag events.
	 */
	private final class LibraryListSelectionModel extends DefaultListSelectionModel {

		/**
		 * Overrides method from DefaultListSelectionModel to handle
		 * specialized library behavior.
		 */
		public void setSelectionInterval(int index0, int index1) {
			_indexCandidate = -1;
			if(_multiSelection) {
				_indexCandidate = index0;
			}
			if(!_multiSelection)
				super.setSelectionInterval(index0, index1);
			else if(getMinSelectionIndex() > index0 ||
					getMaxSelectionIndex() < index1)
				super.setSelectionInterval(index0, index1);

			else {
				_indexCandidate = index0;
			}
		}


		/**
		 * This method is overridden so that dragging with multiple rows
		 * selected does not continue selecting new rows/
		 */
		public void setLeadSelectionIndex(int index) {
			if(_shiftPressed) {
				super.setLeadSelectionIndex(index);
			}
		}
	}
}









