package com.limegroup.gnutella.gui.library;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import java.awt.event.*;
import java.io.*;
import java.awt.Point;
import java.awt.Dimension;
import java.util.Enumeration;
import com.sun.java.util.collections.*;

import com.limegroup.gnutella.FileManager;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.ManagedThread;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.mp3.PlayListGUI;

import java.awt.Color;

/**
 * This class forms a wrapper around the tree that controls 
 * navigation between shared folders.  It constructs the tree and
 * supplies access to it.  It also controls tree directory selection,
 * deletion, etc.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class LibraryTree {

	/**
	 * Constant name for the incomplete folder
	 */
	private final String INCOMPLETE_FILE_NAME = "Incomplete";

	/**
	 * Constant handle to the library mediator class.
	 */
	private final LibraryMediator LIBRARY_MEDIATOR = 
		LibraryMediator.instance();
	
	/**
	 * Constant for the <tt>JTree</tt>.
	 */
	private final JTree TREE = new JTree();

	/**
	 * Constant for the root node of the tree.
	 */
	private final LibraryTreeNode ROOT_NODE = 
		new LibraryTreeNode();

	/**
	 * Constant for the tree model.
	 */
	private final DefaultTreeModel TREE_MODEL =
		new DefaultTreeModel(ROOT_NODE);

	/**
	 * Constant for the popup menu.
	 */
	private final JPopupMenu DIRECTORY_POPUP = new JPopupMenu();


	/**
	 * Constructs the tree and its primary listeners,visualization options, 
	 * editors, etc. 
	 */
	LibraryTree() {
	
		TREE.setModel(TREE_MODEL);
		TREE.setRootVisible(false);
		TREE.getSelectionModel().setSelectionMode
        (TreeSelectionModel.SINGLE_TREE_SELECTION);
		TREE.setEditable(true);
		TREE.setInvokesStopCellEditing(true);
		TREE.setShowsRootHandles(true);	
		TREE.putClientProperty("JTree.lineStyle", "None");	

		makePopupMenu();
		TREE.addTreeSelectionListener(new LibraryTreeSelectionListener());
		
		TREE.setCellEditor(
            new DefaultTreeCellEditor(TREE, 
                (DefaultTreeCellRenderer)TREE.getCellRenderer(),
				new LibraryTreeCellEditor(this)));
		// add the incomplete directory to the tree
		addIncompleteDirectory();
		updateTheme();
	}

	// inherit doc comment
	public void updateTheme() {
		Color tableColor = ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue();
		TREE.setBackground(tableColor);	    
	    DefaultTreeCellRenderer rnd = new DefaultTreeCellRenderer();
	    rnd.setOpaque(false);
	    TREE.setCellRenderer(rnd);
	    TREE.setCellEditor(
            new DefaultTreeCellEditor(TREE, 
                (DefaultTreeCellRenderer)TREE.getCellRenderer(),
				new LibraryTreeCellEditor(this)));
	}


	/**
	 * Cancels all editing of shared directory folder names.
	 */
	void cancelEditing() {
		if(TREE.isEditing()) {
			TREE.getCellEditor().cancelCellEditing();
		}        
	}

	/**
	 * Adds the visual representation of this folder to the library.
	 *
	 * @param dir    the <tt>File</tt> instance denoting the abstract
	 *               pathname of the new shared directory to add to the 
	 *               library. 
	 *
	 * @param parent the <tt>File</tt> instance denoting the abstract
	 *               pathname of the parent of the new shared directory to 
	 *               add to the library. 
	 */
	void addSharedDirectory(final File dir, final File parent) {
		LibraryTreeNode curNode = null;
		
		// the node we should reload in the tree model
		LibraryTreeNode nodeToLoad = null;

		// the holder for a potential new current node
		AbstractFileHolder holder = null;

		// if this is a "root" shared directory
		if(parent == null) {
			if(!ROOT_NODE.isChild(dir)) {
				holder = new InternalNodeFileHolder(dir);
				curNode = new LibraryTreeNode(holder);

				// insert this node one behind the end, as we
				// have already inserted the incomplete directory, and
				// that should go last
				TREE_MODEL.insertNodeInto(curNode, ROOT_NODE,
										  ROOT_NODE.getChildCount()-1);
				nodeToLoad = ROOT_NODE;
			}
		}
		
		// otherwise "dir" is a subdirectory of a directory
		// that is itself shared
		else {
			AbstractFileHolder curHolder = null;
			Enumeration enum = ROOT_NODE.breadthFirstEnumeration();
			if(enum.hasMoreElements()) {
				while(enum.hasMoreElements()) {
					curNode = (LibraryTreeNode)enum.nextElement();
					curHolder = curNode.getFileHolder();
					if(curHolder.matchesFile(parent)) {
						if(!curNode.isChild(dir)) {
							holder = new InternalNodeFileHolder(dir);
							LibraryTreeNode newNode = new LibraryTreeNode(holder);
							TREE_MODEL.insertNodeInto(newNode, curNode,
													  curNode.getChildCount());
							nodeToLoad = curNode;
						}
						break;
					}
				}
			}
			// handle the case that should never occur --
			// where a subdirectory of a "root" shared
			// directory is added before its root.
			else {
				InternalNodeFileHolder newHolder = 
				    new InternalNodeFileHolder(dir);
				curNode = new LibraryTreeNode(newHolder);
				TREE_MODEL.insertNodeInto(curNode, ROOT_NODE,
										  ROOT_NODE.getChildCount()-1);
				nodeToLoad = ROOT_NODE;
			}
		}

        // make sure we're not editing the shared file names.
		cancelEditing();

        int[] selected = TREE.getSelectionRows();
		TREE_MODEL.reload(nodeToLoad);
		
        //maintain the selected row.
        //the else is needed for the case of two shared folders..
        //calling TREE_MODEL.reload() will unselect your selection
	    if ( selected == null || selected.length == 0 )
	        setSelectionRow(0);
	    else
	        setSelectionRow(selected[0]);
	}

	/** 
	 * Adds a new folder to the currently selected folder if there is one. 
	 */
	void addNewLibraryFolder() {
		LibraryTreeNode selectedNode = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();

		// return if nothing is selected
		if(selectedNode == null) return;

		if(incompleteDirectoryIsSelected()) {
			showIncompleteFolderMessage("add a new folder to");
			return;
		}

		File curFile = selectedNode.getFileHolder().getFile();
		File file = new File(curFile, "New Folder");
		int number = 2;
		boolean dirMade = false;
		while(!dirMade) {
			dirMade = file.mkdir();
			if(dirMade) {
				LibraryTreeNode node =
				new LibraryTreeNode(new InternalNodeFileHolder(file));
				
				TREE_MODEL.insertNodeInto 
				(node,selectedNode,selectedNode.getChildCount());

				TREE.expandRow(TREE.getLeadSelectionRow());
				TREE.setSelectionRow(TREE.getLeadSelectionRow()
									  +selectedNode.getChildCount());
				
				// tell the FileManager to reload its settings.
				// we don't need to notify the settings manager, as
				// the "root" shared directories have not changed.
				LIBRARY_MEDIATOR.refresh();
			}
			else {
				file = new File(curFile, "New Folder ("
								+String.valueOf(number)+")");
			}
			number++;
		}				
	}
								

	/** 
	 * Renames the selected folder in the library if there is one. 
	 */
	void renameLibraryFolder() {
		LibraryTreeNode selectedNode = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();

		// return if nothing is selected
		if(selectedNode == null) return;

		if(!incompleteDirectoryIsSelected()) {
			TREE.startEditingAtPath(TREE.getLeadSelectionPath());
		}
		else {
			showIncompleteFolderMessage("rename");
		}
	}


    /**
     * Adds files to the playlist recursively.
     */
    void addPlayListEntries() {
		LibraryTreeNode selectedNode = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();

		// return if nothing is selected
		if(selectedNode == null) return;
		if(incompleteDirectoryIsSelected())	return;

		final File currFile = selectedNode.getFileHolder().getFile();
        (new ManagedThread ("PlayListEntryAdder") {
            public void managedRun() {
                String[] filter = {"mp3"};
                  File[] filesToAdd = FileManager.getFilesRecursive(currFile,
                                                                    filter);
                  PlayListGUI plInstance = PlayListGUI.instance();
                  // synch to stop any other synch sensitive methods from running
                  // concurrently...
                  synchronized (plInstance) {
                      for (int i = 0; 
                           (filesToAdd != null) && (i < filesToAdd.length);
                           i++)
                          plInstance.addFileToPlayList(filesToAdd[i]);
                  }
            }
        }).start();
    }

	/**
	 * Returns the file object associated with the mousePoint
	 * parameter if the mousePoint is over a valid file.
	 *
	 * @return a <tt>File</tt> instance contained in the 
	 *         node wrapper object, or <tt>null</tt> if the wrapped
	 *         <tt>File</tt> object is not a directory or is null
	 */
	File getFileForPoint(Point mousePoint) {
		TreePath path = TREE.getPathForLocation(mousePoint.x,mousePoint.y);
		if(path == null) return null;

		LibraryTreeNode node = (LibraryTreeNode)path.getLastPathComponent();		
		AbstractFileHolder fh = node.getFileHolder();
		File file = fh.getFile();
		if(file.isDirectory())
			return file;
		else
			return null;
	}

	/**
	 * Returns a boolean indicating whether or not the current mouse drop 
	 * event is dropping to the incomplete folder.
	 *
	 * @param mousePoint the <tt>Point</tt> instance representing the
	 *                   location of the mouse release
	 *
     * @return  <tt>true</tt> if the mouse was released on the Incomplete
	 *          folder, <tt>false</tt> otherwise 
	 */
	boolean droppingToIncompleteFolder(Point mousePoint) {
		TreePath path = TREE.getPathForLocation(mousePoint.x,mousePoint.y);
		if(path == null) return false;
		LibraryTreeNode node = (LibraryTreeNode)path.getLastPathComponent();		
		if(node == null) return false;
		return isIncompleteDirectory(node.getFileHolder());
	}

	/** 
	 * Returns the File object associated with the currently selected 
	 * directory.
	 *
	 * @return the currently selected directory in the library, or 
	 *  <tt>null</tt> if no directory is selected
	 */ 
	File getSelectedDirectory() {
		LibraryTreeNode selectedNode = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();

		// return if nothing is selected
		if(selectedNode == null) return null;

		return selectedNode.getFileHolder().getFile();
	}	

	/** 
	 * Returns the top-level directories as an array of <tt>File</tt> objects
	 * for updating the shared directories in the <tt>SettingsManager</tt>.
	 *
	 * @return the array of top-level directories as <tt>File</tt> objects
	 */
	File[] getSharedDirectories() {
		LibraryTreeNode curNode; 

		int length = ROOT_NODE.getChildCount();
		ArrayList newFiles = new ArrayList();
		for(int i=0; i<length; i++) {
			curNode = (LibraryTreeNode)ROOT_NODE.getChildAt(i);
			AbstractFileHolder fh = curNode.getFileHolder();
			if(!isIncompleteDirectory(fh)) {
				File f = fh.getFile();
				if(f.isDirectory()) {		
					newFiles.add(f);
				}
			}
		}
		File[] files = new File[newFiles.size()];
		for(int r=0; r<newFiles.size(); r++) {
			files[r] = (File)newFiles.get(r);
		}
		return files;
	}

	/** 
	 * Sets the selected row in the tree. 
	 *
	 * @param row the row to select
	 */
	void setSelectionRow(int row) {
		TREE.setSelectionRow(row);
	}

	/**
	 * Returns a boolean specifying whether or not the File "parent" 
	 * parameter is the same as the File object associated with the 
	 * currently selected row.
	 *
	 * @param parent the <tt>File</tt> instance associated with the
	 *               parent of the node in question
	 * @return <tt>true</tt> if the <tt>parent</tt> argument is the 
	 *         currently selected <tt>File</tt> instance, <tt>false</tt>
	 *         otherwise
	 */
	boolean parentIsSelected(final File parent) {
		LibraryTreeNode node = (LibraryTreeNode)TREE.getLastSelectedPathComponent();
		if(node == null) return false;
		
		return node.getFileHolder().matchesFile(parent);		
	}

	/** 
	 * Removes all shared directories from the visual display.
	 */ 
	void clear() {
		ROOT_NODE.removeAllChildren();	
		addIncompleteDirectory();
	}  

	/**
	 * Handles a folder name change.
	 *
	 * @param newName the new name of the folder.
	 * 
	 * @return an <tt>Object</tt> that is the current CellEditor value
	 */
	AbstractFileHolder handleNameChange(String newName) {
		LibraryTreeNode selectedNode = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();

		// return if nothing is selected
		if(selectedNode == null) {
			return null;
		}

		// only rescan node if the string is really changed
		if(newName.equals(selectedNode.getFileHolder().toString())) {
			return selectedNode.getFileHolder();
		}

		File f = selectedNode.getFileHolder().getFile();
		String parent = f.getParent();
		
		if(parent == null) {
			return null;
		}
		
		File newFile = new File(parent, newName);		
		boolean b = f.renameTo(newFile);        
		if(!b) {
            showFolderRenameError(f.getPath());
			return null;
		}

		InternalNodeFileHolder fh = new InternalNodeFileHolder(newFile);
		selectedNode.setUserObject(fh);						
		TREE_MODEL.nodeChanged(selectedNode);	
		if(!selectedNode.isLeaf()) {
			// note that this node cannot be the
			// root node
			selectedNode.removeAllChildren();
		}
		handleDirectoryChange(selectedNode);
		return selectedNode.getFileHolder();
	}


	/**
	 * Stops sharing the selected folder in the library if there is a 
	 * folder selected, if the folder is not the save folder, or if the
	 * folder is not a subdirectory of a "root" shared folder.
	 */
	void unshareLibraryFolder() {
		LibraryTreeNode node = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();
		if(node == null) return;

		if(incompleteDirectoryIsSelected()) {
			showIncompleteFolderMessage("delete");
		}
		else if(!node.getLibraryTreeNodeParent().isRoot()) {
			GUIMediator.showMessage("MESSAGE_CANNOT_UNSHARE_SUBDIRS");
		}						
		else {
			String msgKey = "MESSAGE_CONFIRM_UNSHARE_DIRECTORY";				
			int response = GUIMediator.showYesNoMessage(
			    msgKey, QuestionsHandler.UNSHARE_DIRECTORY);
			if(response != GUIMediator.YES_OPTION) return;	
			TREE_MODEL.removeNodeFromParent(node); 
			LIBRARY_MEDIATOR.handleRootSharedDirectoryChange();
		}
	}


	/**
	 * Returns whether or not the incomplete directory is
	 * selected in the tree. 
	 *
	 * @return <tt>true</tt> if the incomplete directory is selected,
	 *  <tt>false</tt> otherwise
	 */
	boolean incompleteDirectoryIsSelected() {
		LibraryTreeNode node = (LibraryTreeNode)TREE.getLastSelectedPathComponent();
		if(node == null) return false;

		return isIncompleteDirectory(node.getFileHolder());
	}

	/**
	 * Sets the cursor in the tree.
	 *
	 * @param cursor the <tt>Cursor</tt> to set in the tree
	 */
	void setCursor(java.awt.Cursor cursor) {
		TREE.setCursor(cursor);
	}

	/**
	 * Returns the location of the <tt>JTree</tt> on the screen.
	 *
	 * @return the <tt>Point</tt> instance containing the x,y position
	 *         of the <tt>JTree</tt> on the screen
	 */
	Point getTreeLocation() {
		return TREE.getLocationOnScreen();
	}

	/**
	 * Returns the size of the <tt>JTree</tt> on the screen.
	 *
	 * @return the <tt>Dimension</tt> instance containing the width and 
	 *         height of <tt>JTree</tt>
	 */
	Dimension getTreeSize() {
		return TREE.getSize();
	}

    /** 
	 * Accessor for the <tt>JTree</tt> that this class wraps. 
	 *
	 * @return the <tt>JTree</tt> instance used by the library.
	 */
    JTree getTree() {
        return TREE;
    }

	/**
	 * Determines whether the AbstractFileHolder parameter is the 
	 * holder for the incomplete folder.
	 * 
	 * @param holder the <tt>AbstractFileHolder</tt> class to check
	 *  for whether or not it is the incomplete directory
	 * @return <tt>true</tt> if it does contain the incomplete
	 *  directory, <tt>false</tt> otherwise
	 */
	private boolean isIncompleteDirectory(AbstractFileHolder holder) {
		return holder.toString().equals(INCOMPLETE_FILE_NAME);
	}

	/**
	 * Adds the incomplete directory (but not any subdirectories
	 * the incomplete directory may have) to the tree.
	 */
	private void addIncompleteDirectory() {
		try {
			File incFile = SharingSettings.INCOMPLETE_DIRECTORY.getValue();
            if (incFile==null) { throw new FileNotFoundException(); }
            
			InternalNodeFileHolder fh = new InternalNodeFileHolder(incFile);
			LibraryTreeNode node = new LibraryTreeNode(fh);
			TREE_MODEL.insertNodeInto(node, ROOT_NODE, 0);
			TREE_MODEL.reload(ROOT_NODE);
		} catch(FileNotFoundException e) {
			// this should not happen -- if it does, something is broken
			GUIMediator.showInternalError(e);
		}
	}

	/**
	 * Shows a message indicating that a specific action cannot
	 * be performed on the incomplete directory (such as changing 
	 * its name).
	 *
	 * @param action the error that occurred
	 */
	private void showIncompleteFolderMessage(String action) {
		String key1 = "MESSAGE_INCOMPLETE_DIRECTORY_START";
		String key2 = "MESSAGE_INCOMPLETE_DIRECTORY_END";
		GUIMediator.showError(key1, action, key2);
	}

	/** 
	 * Displays an error deleting the folder whose name is passed
	 * in as a parameter.
	 *
	 * @param folderName the name of the folder the encountered an error
	 *  during deletion.
	 */
	private void showFolderDeletionError(String folderName) {
		final String key1 = "MESSAGE_UNABLE_TO_DELETE_DIRECTORY_START";
		final String key2 = "MESSAGE_UNABLE_TO_CHANGE_DIRECTORY_END";
		final String msg = "'"+folderName+"'.";
		GUIMediator.showError(key1, msg, key2);
	}

	/** 
	 * Displays an error deleting the folder whose name is passed
	 * in as a parameter.
	 *
	 * @param folderName the name of the folder the encountered an error
	 *  during deletion.
	 */
	private void showFolderRenameError(String folderName) {
		final String key1 = "MESSAGE_UNABLE_TO_RENAME_DIRECTORY_START";
		final String key2 = "MESSAGE_UNABLE_TO_CHANGE_DIRECTORY_END";
		final String msg = "'"+folderName+"'.";
		GUIMediator.showError(key1, msg, key2);
	}

	/**
	 * Handles a change to the LibraryTreeNode passed in as a
	 * parameter, either calling LibraryView for a "full" reload
	 * of the node and SettingsManager values when there is a 
	 * change to one of the "root" shared directories, and simply
	 * reloading the settings in the FileManager otherwise.
	 *
	 * @param node the LibraryTreeNode whose directory has changed.
	 */
	private void handleDirectoryChange(LibraryTreeNode node) {
		if(node.getLibraryTreeNodeParent().isRoot()) {
			// make sure the shared directories in
			// the settings manager and file
			// manager match the root (shared)
			// directories displayed in the tree						
			LIBRARY_MEDIATOR.handleRootSharedDirectoryChange();
		}
		else {
			// we don't need to reload SettingsManager
			// since the names of the "root" shared
			// directories have not changed, so simply 
			// reload the FileManager and clear the 
			// library table.
			LIBRARY_MEDIATOR.refresh();
		}
	}

	/**
	 * Constructs the popup menu that appears in the tree on
	 * a right mouse click.
	 */
	private void makePopupMenu() {
		JMenuItem newItem = 
		    new JMenuItem(
                GUIMediator.getStringResource("LIBRARY_TREE_NEW_FOLDER_LABEL"));
		JMenuItem unshareItem = 
		    new JMenuItem( 
                GUIMediator.getStringResource("LIBRARY_TREE_UNSHARE_FOLDER_LABEL"));
		JMenuItem renameItem = 
		    new JMenuItem(
                GUIMediator.getStringResource("LIBRARY_TREE_RENAME_FOLDER_LABEL"));
        JMenuItem addPlayListItem = 
		    new JMenuItem(
                GUIMediator.getStringResource("LIBRARY_TREE_TO_PLAYLIST_FOLDER_LABEL"));

		newItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent ae) {
				try {
					addNewLibraryFolder();
				} catch(Throwable t) {
					GUIMediator.showInternalError(t);
				}
			}
		});

		unshareItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent ae) {
				try {
					unshareLibraryFolder();
				} catch(Throwable t) {
					GUIMediator.showInternalError(t);
				}
			}
		});

		renameItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent ae) {
				try {
					renameLibraryFolder();
				} catch(Throwable t) {
					GUIMediator.showInternalError(t);
				}
			}
		});
	   
        addPlayListItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ae) {
				try {
					addPlayListEntries();
				} catch(Throwable t) {
					GUIMediator.showInternalError(t);
				}
            }
        });

		DIRECTORY_POPUP.add(newItem);
		DIRECTORY_POPUP.add(unshareItem);
		DIRECTORY_POPUP.add(renameItem);
        if (!CommonUtils.isMacClassic())
            DIRECTORY_POPUP.add(addPlayListItem);
		MouseListener ml = new MouseAdapter() {
  			public void checkPopUp(MouseEvent e) {
				try {
					int row = TREE.getRowForLocation(e.getX(), e.getY());
					if(row == -1) return;
					
					TREE.setSelectionRow(row);
					if(e.isPopupTrigger() || 
					   SwingUtilities.isRightMouseButton(e)) {
						   // only show menu options if the user clicked
						   // on a directory other than the incomplete 
						   // directory.
						   if(!incompleteDirectoryIsSelected()) {
							   DIRECTORY_POPUP.show(TREE,e.getX(),e.getY());
						   }
					   }
				} catch(Throwable t) {
					GUIMediator.showInternalError(t);
				}				
  			}
  			public void mouseClicked(MouseEvent e) {
				checkPopUp(e);
			}
  			public void mousePressed(MouseEvent e) {
				checkPopUp(e);
			}
  		};
  		TREE.addMouseListener(ml);		
	}


	/**
	 * Selection listener that changes the files displayed in the table
	 * if the user chooses a new directory in the tree.
	 */
	private class LibraryTreeSelectionListener implements TreeSelectionListener {
		public void valueChanged(TreeSelectionEvent e) {	
			try {
				LibraryTreeNode node = (LibraryTreeNode)
				TREE.getLastSelectedPathComponent();

				// return if we cannot get the selected node
				if(node == null) return;

				if(!incompleteDirectoryIsSelected()) {
					LIBRARY_MEDIATOR.updateTableFiles(
					    node.getFileHolder().getFile());
				} else {
					// call specialized method to display incomplete files
					LIBRARY_MEDIATOR.showIncompleteFiles();
				}
			} catch(Throwable t) {
				GUIMediator.showInternalError(t);
			}			
		}
	}

	/**
	 * Private class that extends a DefaultMutableTreeNode. Using
	 * this class ensures that the "UserObjects" associated with
	 * the tree nodes will always be File objects.
	 */
	private final class LibraryTreeNode extends DefaultMutableTreeNode {
		private AbstractFileHolder _holder;

		private LibraryTreeNode() {
			_holder = new RootNodeFileHolder();
		}

		private LibraryTreeNode(AbstractFileHolder holder) {
			super(holder);
			_holder = holder;
			if(_holder == null) {
				throw new NullPointerException("null holder in LibraryTreeNode");
			}
		}

		/** 
		 * Returns the <tt>AbstractFileHolder</tt> object contained in this 
		 * node.
		 *
		 * @return the <tt>AbstractFileHolder</tt> object contained in this 
		 * node
		 */
		private AbstractFileHolder getFileHolder() {
			return _holder;
		}
		
		/**
		 * Determines whether or not the File parameter is the 
		 * <tt>File</tt> contained in the FileHolder user object 
		 * of any of this node's children.
		 *
		 * @param file The <tt>File</tt> instance to check against
		 *             this node's children to see if it is a child.
		 *
		 * @return <tt>true</tt> if and only if the file parameter
		 *         is equal to a <tt>File</tt> object in one of 
		 *         this node's children.
		 */
		private boolean isChild(File file) {
			int count = getChildCount();			
			for(int i=0; i<count; i++) {
				LibraryTreeNode curNode = (LibraryTreeNode)getChildAt(i);
				File curFile = curNode.getFileHolder().getFile();
				if(curFile.equals(file)) {
					return true;
				}
			}
			return false;
		}

		/**
		 * Overrides the setUserObject method of the 
		 * DefaultMutableTreeNode superclass to ensure that
		 * the user object must be a FileHolder and to reset
		 * the "HOLDER" variable to the new holder.
		 *
		 * @param holder the AbstractFileHolder instance to set as
		 *  the user object for this node.
		 */
		public void setUserObject(Object holder) {			          
            
            // ignore attempts to set this to null..
			if(holder == null) return;
            if(!(holder instanceof AbstractFileHolder)) {
                throw new IllegalArgumentException("unexpected object: "+
                                                   holder);
            }
			super.setUserObject(holder);
			_holder = (AbstractFileHolder)holder;
		}

		/**
		 * Replaces the getParent() method so that we handle all of the
		 * casting internally for convenience.  
		 *
		 * @return the <tt>LibraryTreeNode</tt> parent for this node, or
		 *  <tt>null</tt> if this node has no parent
		 */
		private LibraryTreeNode getLibraryTreeNodeParent() {
			TreeNode parentNode = super.getParent();
			if(parentNode == null) return null;
			return (LibraryTreeNode)parentNode;
		}	  
	}

	/**
	 * A private abstract class that enables the use of 
	 * specialized root and internal node subclasses.
	 */
	private abstract class AbstractFileHolder {
		protected abstract File getFile();
		protected abstract boolean matchesFile(File file);
	}

	/**
	 * This class wraps a File object for a node in the tree.
	 * it redefines the "toString" method so that the tree will
	 * display the name of the associated File object as the 
	 * name of the node in the tree.  this class also handles 
	 * determining equality among File objects for directory
	 * insertion.
	 */
	private class InternalNodeFileHolder extends AbstractFileHolder {	
		private final File FILE;

		/**
		 * Constructs a new <tt>InternalNodeFileHolder</tt> with the 
		 * given <tt>File</tt> instance.
		 *
		 * @param file the <tt>File</tt> instance for this holder
		 */
		InternalNodeFileHolder(File file) {
			FILE = file;
		}

		/**
		 * Returns the File object that this FileHolder holds.
		 */
		protected File getFile() {
			return FILE;
		}
		
		/**
		 * Returns a boolean indicating whether the "file"
		 * parameter is the same as the File object for 
		 * this file holder.
		 */
		protected boolean matchesFile(File file) {
			return FILE.equals(file);
		}


		/** 
		 * This is the method that JTree calls to get the
		 * display name for each node in the tree.  
		 */
		public String toString() {
			return FILE.getName();
		}
	}

	/**
	 * Root node class the extends AbstractFileHolder
	 */
	private class RootNodeFileHolder extends AbstractFileHolder {
		protected File getFile() {
			return null;
		}		
		protected boolean matchesFile(File file) {
			return false;
		}
		public String toString() {
			return "";
		}
	}

}


