package com.limegroup.gnutella.gui.library;

import javax.swing.tree.TreeCellEditor;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.event.*;

import com.limegroup.gnutella.ErrorService;

import java.util.EventObject;
import java.awt.event.*;
import java.awt.Component;

/**
 * This class acts as an editor for a table cell in the library.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class LibraryTreeCellEditor implements TreeCellEditor {

	private final JTextField _textField = new JTextField();

	private final TextFieldListener _textFieldListener = 
		new TextFieldListener();
	private LibraryTree _libraryTree;
	private CellEditorListener _cellEditorListener;

	/**
	 * Constructs a new <tt>LibraryTreeCellEditor</tt> for the specified
	 * <tt>LibraryTree</tt>.
	 *
	 * @param libraryTree the <tt>LibraryTree</tt> that this will be
	 *  the editor for
	 */
	LibraryTreeCellEditor(LibraryTree libraryTree) {
		_libraryTree = libraryTree;
		_cellEditorListener = null;
		_textField.addActionListener(_textFieldListener);
	}

    /**
	 * returns the Object value (a String) of the text field cell editor.
	 * implements javax.swing.CellEditor.
	 */
    public Object getCellEditorValue() {
		return _libraryTree.handleNameChange(_textField.getText());
    }

    /** 
	 * returns a value determining whether or not the cell is editable.
	 * implements javax.swing.CellEditor.
	 */
    public boolean isCellEditable(EventObject event) {
		if (event instanceof MouseEvent) {
			boolean isEditable = ((MouseEvent)event).getClickCount() == 1;
			if(isEditable && !_libraryTree.incompleteDirectoryIsSelected()) {
				return true;
			}
			return false;
        }
		
    	return true;
    }
    
    /**
	 * Returns a boolean determining whether or not the
	 * cell should be selected.
	 * implements javax.swing.CellEditor. 
	 */
    public boolean shouldSelectCell(EventObject anEvent) { 
		return true; 
    }

    /**
	 * Stops cell editing and returns a boolean indicating
	 * that the editing has stopped.
	 * implements javax.swing.CellEditor.
	 */
    public boolean stopCellEditing() {
		fireEditingStopped();
    	return true;
    }

    /**
	 * Cancels the editing of the current cell.
	 * implements javax.swing.CellEditor.
	 */
    public void cancelCellEditing() {
		fireEditingCanceled();
    }

    /**
	 * Sets the <tt>CellEditorListener</tt> to be the passed in listener. 
	 * implements javax.swing.CellEditor.
	 */
    public void addCellEditorListener(CellEditorListener listener) {
		_cellEditorListener = listener;
    }

    /**
	 * Sets the <tt>CellEditorListener</tt> to null.
	 * implements javax.swing.CellEditor.
	 */
    public void removeCellEditorListener(CellEditorListener listener) {
		_cellEditorListener = null;
    }

    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.  The event instance 
     * is lazily created using the parameters passed into 
     * the fire method.
     */
    void fireEditingStopped() {
		if(_cellEditorListener != null) {
			_cellEditorListener.editingStopped(new ChangeEvent(this));
		}
    }
	
    /**
     * Notifies all listeners that have registered interest for
     * notification on this event type.  The event instance 
     * is lazily created using the parameters passed into 
     * the fire method.
     */
    void fireEditingCanceled() {
		if(_cellEditorListener != null) {
			_cellEditorListener.editingCanceled(new ChangeEvent(this));
		}
    }

	/** 
	 * Returns the text field editing component
	 * Implements the <tt>TreeCellEditor</tt> interface.
	 */
    public Component getTreeCellEditorComponent(JTree tree, Object value,
												boolean isSelected,
												boolean expanded,
												boolean leaf,
												int row) {
		_textField.setText((value != null) ? value.toString() : "");
		return _textField;
    }

	/** 
	 * Private class that handles action events from the textField 
	 * and interacts with the Library table to set the correct text 
	 * coming from the editor.
	 */
	private class TextFieldListener implements ActionListener {
		// Implementing ActionListener interface
        public void actionPerformed(ActionEvent e) {
            try {
                fireEditingStopped();
            } catch(Throwable t) {
                ErrorService.error(t);
            }
		}
	}
}
