package com.limegroup.gnutella.gui.mp3;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;
import com.limegroup.gnutella.gui.*;

import com.sun.java.util.collections.LinkedList;
import com.limegroup.gnutella.Assert;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.util.ManagedThread;

/** This class sets up JPanel with MediaPlayer on it, and takes care of 
 *  GUI MediaPlayer events.
 */
public final class MediaPlayerComponent
        implements AudioPlayerListener, RefreshListener, ThemeObserver {
    
    /** Actually plays the mp3 files.
     */
    private static AudioPlayer myPlayer;
    
    /** Displays the progress of the playing file, in addition to the name.
     */
    private JProgressBar myProgressBar;
    
    /** PlayList interface instance.
     */
    private static PlayListGUI myPlayList;
    
    /** Plays the latest file in its Queue */
    private static PlayThread myPlayThread = null;
    
    private File myCurrentPlayingFile = null;
    
    /** Used to display a portion of the file in the progress bar.
     */
    private int currBeginIndex = -1;
    private static final int STRING_SIZE_TO_SHOW = 24;
    
    private static int lastButtonPressed = -1;
    private static final int PLAY_BUTTON_INDEX  = 0;
    //private static final int PAUSE_BUTTON_INDEX = 1;
    private static final int STOP_BUTTON_INDEX  = 2;
    
    /**
     * Constant for the play button.
     */
    private static final MediaButton PLAY_BUTTON = 
        new MediaButton("MEDIA_PLAY_BUTTON_TIP",
            "play_up.gif", "play_dn.gif");
    
    /**
     * Constant for the pause button.
     */
    private static final MediaButton PAUSE_BUTTON = 
        new MediaButton("MEDIA_PAUSE_BUTTON_TIP",
            "pause_up.gif", "pause_dn.gif");
    
    /** Constant for the stop button.
     */
    private static final MediaButton STOP_BUTTON = 
        new MediaButton("MEDIA_STOP_BUTTON_TIP",
            "stop_up.gif", "stop_dn.gif");
    
    /** Constant for the forward button.
     */
    private static final MediaButton FORWARD_BUTTON = 
        new MediaButton("MEDIA_FORWARD_BUTTON_TIP",
            "forward_up.gif", "forward_dn.gif");
    
    /** Constant for the rewind button.
     */
    private static final MediaButton REWIND_BUTTON = 
        new MediaButton("MEDIA_REWIND_BUTTON_TIP",
            "rewind_up.gif", "rewind_dn.gif");
    
    private static MediaPlayerComponent myInstance;
    
    private JPanel myMediaPanel = null;
    
    /**
     * Timer started upon startup (setting player visible)
     * and shut down when a file is finally played.
     * Fires continuous events to update display string in player.
     * Timers also coalesc events by default, delays pending events when
     * app is busy.
     */
    private final String INITIAL_DISPLAY_STRING = 
        GUIMediator.getStringResource("MEDIA_PLAYER_DEFAULT_STRING");
    
    /**
     * Variable for the name of the current file being played.
     */
    private String currentFileName;
    
    /**
     * Lock for access to the above String.
     */
    private Object cfnLock = new Object();
    
    /**
     * Constructs a new <tt>MediaPlayerComponent</tt>.
     */
    public MediaPlayerComponent() {
        // -- IVAR (non-gui) STUFF -- BEGIN
        myPlayer = new BasicPlayer();
        myPlayer.addAudioPlayerListener(this);
        // start this guy
        myPlayThread = new PlayThread();
        myPlayThread.start();
        
        myInstance = this;
        GUIMediator.addRefreshListener(this);
        // -- IVAR (non-gui) STUFF -- END

        GUIMediator.addThemeObserver(this);
    }
    
    // inherit doc comment
    public void updateTheme() {
        PLAY_BUTTON.updateTheme();
        PAUSE_BUTTON.updateTheme();
        STOP_BUTTON.updateTheme();
        FORWARD_BUTTON.updateTheme();
        REWIND_BUTTON.updateTheme();
        Dimension progressBarDimension;
        if(CommonUtils.isMacOSX() && 
          (ThemeSettings.THEME_FILE.getValue().equals(ThemeSettings.THEME_DEFAULT.getValue()))) {
            progressBarDimension = new Dimension(110,25);
        } else {
            progressBarDimension = new Dimension(110,15);
        }
        myProgressBar.setMaximumSize(progressBarDimension);
        myProgressBar.setPreferredSize(progressBarDimension);
        myProgressBar.setString(INITIAL_DISPLAY_STRING);
    }
    
    
    /** 
     * Class that listens on components that wish to play audio files.
     */
    private class PlayListener implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            try {
                lastButtonPressed = PLAY_BUTTON_INDEX;
                if (myPlayer.getStatus() == AudioPlayer.STATUS_PAUSED)
                    playPressedAndPaused();
                else {
                    if (myPlayer.getStatus() == AudioPlayer.STATUS_STOPPED)
                        myPlayThread.addFileToPlay(myPlayList.getFileToPlay());
                    else if ((myPlayer.getStatus() ==
                            AudioPlayer.STATUS_PLAYING) &&
                            (myPlayList.songIsSelected()))
                        skipSong();
                }
            } catch(Throwable e) {
                GUIMediator.showInternalError(e);
            }
        }
    }
    
    
    /** 
     * Class that listens on components that wish to stop audio files....
     */
    private class StopListener implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            try {
                lastButtonPressed = STOP_BUTTON_INDEX;
                stopSong();
            } catch(Throwable e) {
                GUIMediator.showInternalError(e);
            }
        }
    }
    
    
    /** Class that handles asking for next audio file to be played.
     */
    private class NextListener implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            try {
                if (myPlayer.getStatus() != AudioPlayer.STATUS_STOPPED) {
                    if (myPlayer.getStatus() == AudioPlayer.STATUS_PAUSED)
                        myPlayer.unpause();
                    skipSong();
                }
            } catch(Throwable e) {
                GUIMediator.showInternalError(e);
            }
        }
    }
    
    
    /** Class that handles asking for previous audio file to be played.
     */
    private class BackListener implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            try {
                if (myPlayer.getStatus() != AudioPlayer.STATUS_STOPPED) {
                    myPlayList.setBackwardsMode();
                    if (myPlayer.getStatus() == AudioPlayer.STATUS_PAUSED)
                        myPlayer.unpause();
                    skipSong();
                }
            } catch(Throwable e) {
                GUIMediator.showInternalError(e);
            }
        }
    }
    
    
    /** 
     * Class that listens on components that wish to pause audio files....
     */
    private class PauseListener implements ActionListener {
        public void actionPerformed(ActionEvent ae) {
            try {
                pauseSong();
            } catch(Throwable e) {
                GUIMediator.showInternalError(e);
            }
        }
    }
    
    
    int iterations = 0;
    
    private class SeekChangeListener implements ChangeListener {
        public void stateChanged(ChangeEvent ce) { 
        } 
    }
    
    
    /** Used to buffer up play requests and not start the player
     *  unless absolutely necessary.
     */
    private class PlayThread extends ManagedThread {
        
        private LinkedList list = new LinkedList();
        private Object lock = new Object();
        
        PlayThread() {
            super("PlayThread");
        }
        
        private boolean shouldRun = true;
        private void shutdown() {
            shouldRun = false;
        }
        
        public void managedRun() {
            debug("PlayThread.run(): entered.");
            while (shouldRun) {
                // wait for a file
                synchronized (lock) {
                    if (list.isEmpty()) {
                        try {
                            debug("PlayThread.run(): waiting for a file.");
                            lock.wait();
                            debug("PlayThread.run(): woken up.");
                        }
                        catch (InterruptedException ignored) {}
                    }
                }
                Assert.that(!list.isEmpty());
                
                debug("PlayThread.run(): myPlayer.getStatus() = " +
                      myPlayer.getStatus());
                // if the player is ready....
                if (myPlayer.getStatus() == AudioPlayer.STATUS_STOPPED) {
                    // get the latest file
                    File playFile = null;
                    while (!list.isEmpty()) {
                        playFile = getFileToPlay();
                        
                        // give the user a few milliseconds to decide whether 
                        // or not to stop pressing the skip button
                        try {
                            if (list.isEmpty())
                                Thread.sleep(250);
                            else
                                Thread.sleep(50);
                        }
                        catch (Exception ignored) {};
                    }
                    
                    debug("PlayThread.run(): playing file " +
                          playFile);
                    // play the latest file
                    myCurrentPlayingFile = playFile;
                    try {
                        myPlayer.play(playFile);
                    }
                    catch (IOException ioe) {
                        GUIMediator.showInternalError(ioe);
                    }
                }
            }
        }
        
        public void addFileToPlay(File f) {
            if (f != null) {
                synchronized (lock) {
                    list.addLast(f);
                    lock.notify(); // try to tell someone about it...
                }
            }
        }
        
        /** ONLY CALL THIS IF THE LIST IS NOT EMPTY ELSE NULL
         *  MAY BE RETURNED!
         */
        private File getFileToPlay() {
            File retFile = null;
            synchronized (lock) {
                retFile = (File) list.removeFirst();
            }
            return retFile;
        }
        
    }
    
    
    private final boolean debugOn = false;
    private JPanel constructMediaPanel() {
        JPanel retPanel = new BoxPanel(BoxPanel.X_AXIS);
        
        JPanel buttonPanel = new BoxPanel(BoxPanel.X_AXIS);
        
        int tempWidth = 0, tempHeight = 0;
        
        tempWidth += PLAY_BUTTON.getIcon().getIconWidth()+2;
        tempHeight += PLAY_BUTTON.getIcon().getIconHeight()+2;
        tempWidth += PAUSE_BUTTON.getIcon().getIconWidth()+2;
        tempWidth += STOP_BUTTON.getIcon().getIconWidth()+2;
        tempWidth += FORWARD_BUTTON.getIcon().getIconWidth()+2;
        tempWidth += REWIND_BUTTON.getIcon().getIconWidth()+2;
        
        // create sliders
        myProgressBar = new SongProgressBar();
        Dimension progressBarDimension;
        if(CommonUtils.isMacOSX()) {
            progressBarDimension = new Dimension(110,25);
        } else {
            progressBarDimension = new Dimension(110,15);
        }
        myProgressBar.setMaximumSize(progressBarDimension);
        myProgressBar.setPreferredSize(progressBarDimension);
        myProgressBar.setString(INITIAL_DISPLAY_STRING);
        
        // setup buttons
        PLAY_BUTTON.addActionListener(new PlayListener());
        PAUSE_BUTTON.addActionListener(new PauseListener());
        STOP_BUTTON.addActionListener(new StopListener());
        FORWARD_BUTTON.addActionListener(new NextListener());
        REWIND_BUTTON.addActionListener(new BackListener());
        
        // setup sliders
        myProgressBar.addChangeListener(new SeekChangeListener());
        updatePBValue(0);
        
        // add everything
        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.setMaximumSize(new Dimension(tempWidth, tempHeight));
        buttonPanel.setMinimumSize(new Dimension(tempWidth, tempHeight));
        buttonPanel.add(REWIND_BUTTON);
        buttonPanel.add(PLAY_BUTTON);
        buttonPanel.add(PAUSE_BUTTON);
        buttonPanel.add(STOP_BUTTON);
        buttonPanel.add(FORWARD_BUTTON);
        buttonPanel.add(Box.createHorizontalGlue());
        
        retPanel.add(Box.createVerticalGlue());
        retPanel.add(buttonPanel);
        retPanel.add(myProgressBar);
        retPanel.add(Box.createVerticalGlue());
        JPanel tempPanel = retPanel;
        retPanel = new BoxPanel(BoxPanel.X_AXIS);
        retPanel.add(Box.createHorizontalGlue());
        retPanel.add(tempPanel);
        
        myPlayList = PlayListGUI.instance();
        
        return retPanel;
    }
    
    private void debug(String out) {
        if (debugOn)
            System.out.println(out);
    }
    
    public JPanel getMediaPanel() {
        if (myMediaPanel == null)
            myMediaPanel = constructMediaPanel();
        return myMediaPanel;
    }
    
    /** call this when you want to launch a file to play.
     */
    public static void launchAudio(File toPlay) {
        if ((myInstance != null) && (toPlay != null)) {
            switch(myPlayer.getStatus()) {
            case AudioPlayer.STATUS_STOPPED:
                myPlayThread.addFileToPlay(toPlay);
                break;
            case AudioPlayer.STATUS_PLAYING:
                myPlayer.stop();
                if (myPlayList.getPlayIndex() < 0)
                    myPlayThread.addFileToPlay(toPlay);
                else // playlist is playing, just queue up song 
                    myPlayList.playSongNext(toPlay);
                break;
            case AudioPlayer.STATUS_PAUSED:
                myPlayer.stop();
                myPlayThread.addFileToPlay(toPlay);
                break;
            }
        }
    }
    
    /**
     * Pauses the currently playing audio file.
     */
    void pauseSong() {
        if (myPlayer.getStatus() == AudioPlayer.STATUS_PAUSED) {
            if (myPlayer.getFrameSeek() != 0) {
                myPlayer.stop();
            }
            else
                myPlayer.unpause();
        }
        else
            myPlayer.pause();
    }
    
    /**
     * playComplete
     * - signifies when an audio file has finished playing
     * Implements one method of BasicPlayerListener Interface.
     */
    public void playComplete() {
        // complete progress bar
        updatePBValue(myProgressBar.getMaximum());
        updatePBString("");
        
        // update gui appropriately
        if (lastButtonPressed == STOP_BUTTON_INDEX ||
            !myPlayList.shouldRepeat())
            myPlayList.playComplete(true);
        else
            myPlayList.playComplete(false);
        
        // if you should continue....
        if (myPlayList.shouldRepeat() && 
            (lastButtonPressed != STOP_BUTTON_INDEX)) {
            myPlayThread.addFileToPlay(myPlayList.getFileToPlay());
        }
    }
    
    /**
     * Method that starts playing the (first of)
     * selected audio file(s).
     */
    private void playPressedAndPaused() {
        // take appropriate action
        if (myPlayer.getStatus() == AudioPlayer.STATUS_PAUSED) {
            // player was paused, play from current
            // audio file location
            if (myPlayer.getFrameSeek() != 0) {
            }
            else
                myPlayer.unpause();
        } /* if */
    }
    
    /**
     * setUpSeek
     *
     * - called before playing a audio file
     * - gets the size of the audio file in frames
     * - sets the maximum slider value
     * Implements one method of AudioPlayerListener Interface.
     *
     */
    public void setUpSeek(int lengthInFrames) {
        // try-catch block only for catching any errors that may be 
        // occuring
        try {
            // set slider max to audio file length
            updatePBValue(0);
            updatePBMaximum(lengthInFrames);
            // others may need access to this badboy....
            synchronized (cfnLock) {
                currentFileName = myCurrentPlayingFile.getName();
                if (currentFileName.length() > STRING_SIZE_TO_SHOW) {
                    currentFileName = currentFileName + " *** " + 
                    currentFileName + " *** ";
                }
                updatePBString(currentFileName);
                currBeginIndex = -5;
            }
        } catch(Exception e) {
            GUIMediator.showInternalError(e);
        }
    }
    
    void skipSong() {
        if (myPlayer.getStatus() == AudioPlayer.STATUS_PLAYING &&
                myPlayList.shouldRepeat()) {
            // player was playing, stop current audio file....
            myPlayer.stop();
        }
    }
    
    /** Call this when a audio file in the PlayList has been doubleclicked.
     */
    public static void audioFileDoubleClicked() {
        if (myInstance == null) 
            return;
        
        switch(myPlayer.getStatus()) {
        case AudioPlayer.STATUS_STOPPED:
            lastButtonPressed = PLAY_BUTTON_INDEX;
            myPlayThread.addFileToPlay(myPlayList.getFileToPlay());
            break;
        case AudioPlayer.STATUS_PLAYING:
            if (!myPlayList.shouldRepeat()) {
                myPlayer.stop();
                myPlayThread.addFileToPlay(myPlayList.getFileToPlay());
            } else
                myInstance.skipSong();
            break;
        case AudioPlayer.STATUS_PAUSED:
            if (!myPlayList.shouldRepeat()) {
                myPlayer.stop();
                myPlayThread.addFileToPlay(myPlayList.getFileToPlay());
            } else {
                myPlayer.stop();
                myInstance.skipSong();
            }
            break;
        }
    }
    
    /**
     * Stops the currently playing audio file.
     */
    void stopSong() {
        myPlayer.stop();
    }
    
    /** always modify myProgressBar on the swing thread...
     */
    private void updatePBMaximum(final int update) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                myProgressBar.setMaximum(update);
            }
        });
    }
    
    /** always modify myProgressBar on the swing thread...
     */
    private void updatePBString(final String update) {
        if (update == null) {
            new Throwable().printStackTrace();
            //return;
        }
        
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                myProgressBar.setString(update);
            }
        });
    }
    
    /** always modify myProgressBar on the swing thread...
     */
    private void updatePBValue(final int update) {
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                try {
                    myProgressBar.setValue(update);
                } catch(ClassCastException ignored) {
                    //see: http://bugs.limewire.com:8080/bugs/searching.jsp?disp1=l&disp2=c&disp3=o&disp4=j&l=152&c=188&m=315_1102
                    //who knows why it happens, but who cares about it.
                }
            }
        });
    }
    
    public void updateAudioPosition(int value) {
        updatePBValue(value);
        
        synchronized (cfnLock) {
            if (currentFileName == null) return;
            if (currentFileName.length() <= STRING_SIZE_TO_SHOW) return;
            Assert.that(currentFileName.length() > (STRING_SIZE_TO_SHOW * 2));
            currBeginIndex = currBeginIndex + 5;
            if ((currBeginIndex + STRING_SIZE_TO_SHOW) >=
                currentFileName.length()) {
                currBeginIndex = currBeginIndex - (currentFileName.length()/2);
            }
            updatePBString(currentFileName.substring(
                currBeginIndex, currBeginIndex + STRING_SIZE_TO_SHOW));
        }
    }
    
    public void refresh() {
        myPlayer.refresh();
    }
}
