package com.limegroup.gnutella.gui.mp3;

import com.limegroup.gnutella.ErrorService;
import com.limegroup.gnutella.settings.ThemeFileHandler;

import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.mp3.MP3Info;

import java.io.File;
import java.io.IOException;
import com.sun.java.util.collections.*;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.filechooser.*;

import java.awt.*;
import java.awt.event.*;


public class PlayListGUI extends PaddedPanel 
	implements MouseListener, ThemeObserver {

    public static final String TEMP_PL_NAME = ".temp.m3u.LW";

    /** I'm a singleton.
     */
    private static PlayListGUI _instance = null;

    /** Underlying table model.
     */
    private BasicTableModel    _tableModel;

    /** The JTable to show.
     */
    private JTable _table;

    /** The current instance of the playlist. */
    private PlayList _playList;
    /** a 'temporary' playlist - used when songs added but not yet saved. */
    private PlayList _tempPL;

    /** boolean for repeat option. */
    private boolean _repeatSongs = true;
    public boolean shouldRepeat() {
        return _repeatSongs;
    }

    /** boolean for shuffle option. */
    private boolean _shuffleSongs = false;

	private final JScrollPane SCROLL_PANE;


    /** index of song that is currently playing...
     */
    private int _playIndex = -1;
    public int getPlayIndex() {
        return _playIndex;
    }
    private void resetPlayIndex() { _playIndex = -1; };

    /** Used to preempt playlist playing, i.e. for launching
     */
    private File _oneTimeSongToPlay = null;
    public synchronized void playSongNext(File f) {
        _oneTimeSongToPlay = f;
    }

    /** Factory method.
     */
    public static PlayListGUI instance() {
        if (_instance == null)
            _instance = new PlayListGUI();
        return _instance;
    }

    
    /** use this ArrayList to keep around File objs of playlist entries for a 
     *  un-named playlist.
     */
    private ArrayList _unsavedFiles = new ArrayList();

	/**
	 * We save the user's last choices in the dialogs for convenience
	 */
	private File lastOpenedPlaylist;
	private File lastSavedPlaylist;


    private PlayListGUI() {
        // put a label on me
        super(GUIMediator.getStringResource("PLAYLIST_TITLE"));
        // setup table model
        _tableModel = new BasicTableModel();
        _tableModel.addColumn(GUIMediator.getStringResource("PLAYLIST_TABLE_NAME"));
        _tableModel.addColumn(GUIMediator.getStringResource("PLAYLIST_TABLE_LENGTH"));
        _tableModel.addColumn(GUIMediator.getStringResource("PLAYLIST_TABLE_BITRATE"));

        _table = new JTable(_tableModel);
        _table.setShowGrid(false);
        _table.setColumnSelectionAllowed(false);
        _table.getColumnModel().getColumn(0).setPreferredWidth(450);
        _table.getColumnModel().getColumn(1).setPreferredWidth(20);
        _table.getColumnModel().getColumn(2).setPreferredWidth(20);

        _table.setDefaultRenderer(String.class,
                                  new PlayingSongRenderer());
        _table.addMouseListener(this);

        JTableHeader th = _table.getTableHeader();
        th.addMouseListener(new SortTableListener());

        SCROLL_PANE = new JScrollPane(_table);
        add(SCROLL_PANE);
		SCROLL_PANE.setPreferredSize(new Dimension(10000, 10000));
        
  		String[] buttonLabelKeys = {
			"PLAYLIST_LOAD_BUTTON_LABEL",
            "PLAYLIST_SAVE_BUTTON_LABEL",
            "PLAYLIST_DELETE_BUTTON_LABEL"
		};
		String[] toolTipKeys = {
			"PLAYLIST_LOAD_BUTTON_TIP",
            "PLAYLIST_SAVE_BUTTON_TIP",
            "PLAYLIST_DELETE_BUTTON_TIP"
		};		
		ActionListener[] listeners = {
            new LoadListener(),
            new SaveListener(),
            new DeleteListener()
		};
		add(Box.createVerticalStrut(GUIConstants.SEPARATOR));
        JPanel optionsPanel = new JPanel(new BorderLayout());
        optionsPanel.add(new ButtonRow(buttonLabelKeys, toolTipKeys, listeners, 
                                       ButtonRow.X_AXIS, ButtonRow.NO_GLUE),
                         BorderLayout.WEST);

        JLabel playOptionsLabel = 
        new JLabel(GUIMediator.getStringResource("PLAYLIST_OPTIONS_STRING"));
        final JCheckBox shuffleCheckBox = 
        new JCheckBox(GUIMediator.getStringResource("PLAYLIST_OPTIONS_SHUFFLE"), 
                      false);
        shuffleCheckBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    _shuffleSongs = !_shuffleSongs;
                    PlayList plToToggle = getCurrentPlayList();
                    if (plToToggle != null) {
                        if (plToToggle.isShuffled() != _shuffleSongs)
                            plToToggle.toggleShuffle();
                    }
                } catch(Throwable t) {
                    ErrorService.error(t);
                }
            }
        });
        final JCheckBox repeatCheckBox = 
        new JCheckBox(GUIMediator.getStringResource("PLAYLIST_OPTIONS_CONTINUE"), 
                      true);
        repeatCheckBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                try {
                    _repeatSongs = !_repeatSongs;
                    shuffleCheckBox.setEnabled(_repeatSongs);
                } catch(Throwable t) {
                    ErrorService.error(t);
                }
            }
        });
        JPanel checkBoxPanel = new JPanel();
        checkBoxPanel.add(playOptionsLabel);
        checkBoxPanel.add(repeatCheckBox);
        checkBoxPanel.add(shuffleCheckBox);

        optionsPanel.add(checkBoxPanel, BorderLayout.EAST);
        
        add(optionsPanel);

        // create the temp playlist, should never really fail...
        try {
            _tempPL = new PlayList(TEMP_PL_NAME);
        }
        catch (Exception ignored) {
            // TODO: why would we ignore all exceptions here?  Maybe specific
            // ones that the PlayList constructor can call, but all of them??
        }
		updateTheme();
		GUIMediator.addThemeObserver(this);
    }


	// inherit doc comment
	public void updateTheme() {
        Color tableColor = ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue();
        SCROLL_PANE.getViewport().setBackground(tableColor);
	}

    public File getFileToPlay() {
        File retFile = null;
        
        if (_oneTimeSongToPlay != null) {
            retFile = _oneTimeSongToPlay;
            _oneTimeSongToPlay = null;
            return retFile;
        }

        PlayList plToUse = getCurrentPlayList();

        int[] sel = _table.getSelectedRows();
        if (plToUse != null) {
            int oldIndex = _playIndex;
            if (sel.length > 0) {
                retFile = plToUse.getSong(sel[0]);
                _playIndex = plToUse.setCurrSongIndex(sel[0]);
                _table.clearSelection();  // unhighlight that badboy
            }
            else if (shouldRepeat()) {
                retFile = plToUse.getNextSong();
                _playIndex = plToUse.getCurrSongIndex();
            }
            // change highlight of old song and now playing song.  also focus on
            // the newly playing song...
            focusChange(_playIndex);
            if (oldIndex >= 0)
                _tableModel.fireTableRowsUpdated(oldIndex,oldIndex);
            if (_playIndex >= 0)
                _tableModel.fireTableRowsUpdated(_playIndex,_playIndex);
        }

        return retFile;
    }

    /** @return true if a song is highlighted in the playlist window.
     */
    public boolean songIsSelected() {
        boolean retBool = false;
        int[] selRows = _table.getSelectedRows();
        if (selRows.length > 0)
            retBool = true;
        return retBool;
    }
    

    /** Call this method if you want the next call to getFileToPlay to 'go
        backwards' in the playlist.
    */
    public void setBackwardsMode() {
        PlayList plToModify = getCurrentPlayList();
        if (plToModify != null)
            plToModify.setBackwardsMode();
    }
        


    /** Call this method when a song has finished playing.  I'll clean up.
     *  @param hardStop true when the mp3 stop button has been pressed.
     */
    public void playComplete(boolean hardStop) {
        if (hardStop) { 
            // only do stuff when a hardstop has been performed -
            // everything else is take care of in getFileToPlay
            int temp = _playIndex;
            resetPlayIndex();
            // if i need to repeat, then the cell will be updated later...
            if (temp >= 0)
                _tableModel.fireTableRowsUpdated(temp,temp);
        }
    }


    public void addFileToPlayList(File toAdd) {

        // try to simply add to the playlist....
        if (_playList != null) 
            _playList.addSong(toAdd);
        else { // or.....
            // add to a 'temp' playlist, which is never saved...
            if (_tempPL != null) 
                _tempPL.addSong(toAdd);
            // save the File locally, to make sure you can add it later to the
            // 'real' playlist...
            _unsavedFiles.add(toAdd);
        }

        // update the JTable
        ArrayList newRow = constructRow(toAdd);
        _tableModel.addRow(newRow);
    }


    private void focusChange(int row) {
        // change the focus of the playlist
        // JDK1.3 way...
        //        _table.changeSelection(row,0,false,false);

        // unfortunately, changeSelection doesn't exist in jdk1.1.8, so use
        // the following instead. got this from the java source....
        if ((row > -1) && (row < _tableModel.getRowCount())) { // make sure..
            _table.setRowSelectionInterval(row,row);
            // Scroll after changing the selection as blit scrolling is 
            // immediate, so that if we cause the repaint after the scroll 
            // we end up painting everything!
            // Autoscrolling support.
            Rectangle cellRect = _table.getCellRect(row, 0, false);
            if (cellRect != null) 
                _table.scrollRectToVisible(cellRect);
        }
        _table.clearSelection();
    }


    private ArrayList constructRow(File constructFrom) {
        ArrayList retList = new ArrayList();
        // add song name
        retList.add(constructFrom.getName());
        try {
            MP3Info info = 
				new MP3Info(constructFrom.getCanonicalPath());
            // add length (in minutes)
            retList.add(getSecondsLength(info));
            // add bitrate
            retList.add(new String() + info.getBitRate());
        }
        catch (IOException ioe) {};        
        return retList;
    }


    /** 
	 * Listener that loads a playlist file.
	 */
    private class LoadListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
    			File parentFile = null;
    			if (lastOpenedPlaylist != null) {
    				String parent = lastOpenedPlaylist.getParent();
    				if(parent != null) {					
    					parentFile = new File(parent);
    				}
    			}
    
    			if(parentFile == null) {
    				parentFile = CommonUtils.getCurrentDirectory();
    			}
    				
    			final File selFile = 
    				FileChooserHandler.getInputFile(instance(), 
    				    "PLAYLIST_DIALOG_OPEN_TITLE", parentFile,
    					new PlayListFileFilter());
    
    			if(selFile != null) {
                	lastOpenedPlaylist = selFile;
                    (new ManagedThread("PlayListLoader") {
    
                        public void managedRun() {
                            try {
                                _tableModel.clearTable();
                                //File selFile = fileChooser.getSelectedFile();
    
                                _playList = 
    								new PlayList(selFile.getCanonicalPath());
                                if (_playList.isShuffled() != _shuffleSongs)
                                    _playList.toggleShuffle();
    
                                fillUpJTable(_playList);
                            }
    						catch (Exception ignored) {
                                // TODO: why would we ignore this??
    						};
                        }
    
                    }).start();
                    // upon loading a new playlist, clear the currently playing
                    // indicator - but don't stop the song
                    resetPlayIndex();
                }
            } catch(Throwable t) {
                ErrorService.error(t);
            }
        }
    }

	/**
	 * <tt>FileFilter</tt> class for only displaying theme file types in
	 * the directory chooser.
	 */
	private static class PlayListFileFilter extends FileFilter {
		public boolean accept(File f) {
			if(f.isDirectory()) return true;
			String name = f.getName();
			if(name.endsWith("m3u")) return true; 
			return false;
		}
		
		public String getDescription() {
			return GUIMediator.getStringResource("PLAYLIST_FILE_DESCRIPTION");
		}		
	} 


    private void fillUpJTable(PlayList plToUse) {
        int begin = 0, end = 0;
        for (int i = 0; i < plToUse.getNumSongs(); i++) {
            File currSong = plToUse.getSong(i);
            ArrayList toAdd = constructRow(currSong);
            _tableModel.addRowSansUpdate(toAdd);
            
            // update table after some inserts...
            if ((i % 19 == 0) ||
                i == (plToUse.getNumSongs()-1)) {
                end = i;
                final int beginFinal = begin;
                final int endFinal = end;
                // only the swing thread can update the
                // table.
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        _tableModel.updateRows(beginFinal,
                                               endFinal);
                    }
                });
                begin = end+1;
            }            
        }        
    }



    /** 
	 * Listener that saves a playlist file.
	 */
    private class SaveListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                if (_playList == null) {
                    // get the user to select a new one....
    				File parentFile = null;
                	if (lastSavedPlaylist != null) {
    					String parent = lastSavedPlaylist.getParent();
    					if(parent != null) {					
    						parentFile = new File(parent);
    					}
    				}
    
    				if(parentFile == null) {
    					parentFile = CommonUtils.getCurrentDirectory();
    				}
    				
    				File selFile = 
    					FileChooserHandler.getInputFile(instance(), 
    				        "PLAYLIST_DIALOG_SAVE_TITLE",
    						"PLAYLIST_DIALOG_SAVE_SAVE",
    						parentFile);
    
    				if(selFile != null) {
    
                        try {
                            // then add all the rows from existing table...
                            lastSavedPlaylist = selFile;
                            if (selFile.exists())
                                selFile.delete();
                            _playList = new PlayList(selFile.getCanonicalPath());
                            if (_playList.isShuffled() != _shuffleSongs)
                                _playList.toggleShuffle();
                            for (int i = 0; i < _unsavedFiles.size(); i++) {
                                File currFile = (File) _unsavedFiles.get(i);
                                _playList.addSong(currFile);
                            }
                            _unsavedFiles.clear();
                        }
                        catch (Exception ignored) {
                            // TODO: why would we ignore this???  At least 
                            // needs a comment
                        }
                        _tempPL = null;
                    }
                }
    
                if (_playList != null) {
                    try {
                        _playList.save();
                    }
                    catch (Exception ignored) {
                        // TODO: why would we ignore this???  At least 
                        // needs a comment
                    }
                }
            } catch(Throwable t) {
                ErrorService.error(t);
            }
        }
            
    }



    /** Listener that deletes a song from the playlist.
     */
    private class DeleteListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                PlayList plToUse = getCurrentPlayList();
    
                int[] sel = _table.getSelectedRows();
                if ((sel.length > 0) && (plToUse != null)){
                    for (int i = 0; i < sel.length; i++) {
                        int indexToDel = sel[i] - i;
                        plToUse.deleteSong(indexToDel);
                        if (_playIndex == indexToDel)
                            resetPlayIndex();
                        else if (_playIndex > indexToDel)
                            _playIndex--;
                        _tableModel.deleteRow(indexToDel);
                    }
                }   
            } catch(Throwable t) {
                ErrorService.error(t);
            } 
        }
    }


    /** uses a simple, not very accurate, formula to get length in minutes.
     *  a sample return value is 1:34.  the formula is:
     *  (FileSize / (BitRate / 8))
     */
    private String getSecondsLength(MP3Info info) {
        StringBuffer retStringB = new StringBuffer();
        
        long numSeconds = info.getLengthInSeconds();
        long numMinutes = numSeconds / 60; 
        long remSeconds = numSeconds - (60*numMinutes);

        retStringB.append(""+numMinutes);
        retStringB.append(":");
        if (remSeconds > 9)
            retStringB.append(""+remSeconds);
        else
            retStringB.append("0"+remSeconds);
        
        return retStringB.toString();
    }

    /** @return The playlist to use.  MAY RETURN NULL!
     */
    private PlayList getCurrentPlayList() {
        PlayList retPL = null;
        if (_playList != null)
            retPL = _playList;
        else
            retPL = _tempPL;
        return retPL;
    }



    private class BasicTableModel extends AbstractTableModel {
        private ArrayList names;	// the label strings
        private ArrayList data;		// arraylist of arraylists
	

        public BasicTableModel() {
            super();
            
            names = new ArrayList();
            data = new ArrayList();
        }
        

        // Basic Model overrides
        public String getColumnName(int col) {
            return (String) names.get(col);
        }
        public int getColumnCount() { return(names.size()); }
        public int getRowCount() { return(data.size()); }
        public Object getValueAt(int row, int col) {
            String result = null;
            try {
                ArrayList rowList = (ArrayList) data.get(row);
                if (col<rowList.size()) 
                    result = (String)rowList.get(col);
            }
            catch (Exception runtime) {
                // may happen, don't want to kill anything, just return null
                // harmlessly
            }
            
            // _apparently_ it's ok to return null for a "blank" cell
            return(result);
        }

        private Class stringClass = (new String()).getClass();
        public Class getColumnClass(int colIndex) {
            return stringClass;
        }
    
        
        // Support writing
        public boolean isCellEditable(int row, int col) { return false; }
        public void setValueAt(Object value, int row, int col) {
            ArrayList rowList = (ArrayList) data.get(row);
            
            if (col>=rowList.size()) {
                while (col>=rowList.size()) rowList.add(null);
            }
            
            rowList.set(col, value);
            
            fireTableCellUpdated(row, col);
        }
        
		
        public void addColumn(String name) {
            names.add(name);
            fireTableStructureChanged();
        }

        
        public int addRow() {
            // Create a new row with nothing in it
            ArrayList row = new ArrayList();
            return(addRow(row));
        }
        
        
        public int addRow(ArrayList row) {
            data.add(row);
            fireTableRowsInserted(data.size()-1, data.size()-1);
            return(data.size() - 1);
        }
        

        public int addRowSansUpdate(ArrayList row) {
            data.add(row);
            return(data.size() - 1);
        }


        public void updateRows(int begin, int end) {
            fireTableRowsInserted(begin, end);
        }
        
        public void deleteRow(int row) {
            if (row == -1) return;
            
            if (row < getRowCount()) {
                data.remove(row);
                fireTableRowsDeleted(row, row);
            }
        }


        public void clearTable() {
            data.clear();
            fireTableDataChanged();
        }

    }
	

    public void launchAudio(File audio) {
        GUIMediator.instance().launchAudio(audio);
        resetPlayIndex();
    }


    private boolean debugOn = true;
    private void debug(String out) {
        if (debugOn)
            System.out.println(out);
    }

    /************* THIS SECTION IMPLEMENTS THE MOUSELISTENER INTERFACE ********/

    public void mouseClicked(MouseEvent e) {
        if (e.getClickCount() > 1)
            GUIMediator.instance().audioFileDoubleClicked();
    }

    public void mouseEntered(MouseEvent e) {
    }

    public void mouseExited(MouseEvent e) {
    }

    public void mousePressed(MouseEvent e) {
    }

    public void mouseReleased(MouseEvent e) {
    }

    /************* THIS SECTION IMPLEMENTS THE MOUSELISTENER INTERFACE ********/

    private class SortTableListener extends MouseAdapter {
        public void mouseClicked(MouseEvent e) {
            TableColumnModel columnModel = _table.getColumnModel();
            int viewColumn = columnModel.getColumnIndexAtX(e.getX()); 
            int column = _table.convertColumnIndexToModel(viewColumn); 
            if (e.getClickCount() == 1 && column == 0) {
                synchronized (instance()) {
                    final PlayList plToSort = getCurrentPlayList();
                    if (plToSort != null) {
                        // remember this to set the current playing song 
                        //correctly
                        File currSong = null;
                        if (_playIndex > -1)
                            currSong = plToSort.getSong(_playIndex);
                        
                        // first, sort the underlying PlayList...
                        plToSort.sortByName();
                        // then clear the JTable
                        _tableModel.clearTable();
                        if ((_playIndex > -1) && (currSong != null)) {
                            _playIndex = plToSort.getIndexOfSong(currSong);
                            plToSort.setCurrSongIndex(_playIndex);
                        }
                        // do it asynch....
                        (new ManagedThread("PlayListSorter") {
                            public void managedRun() {
                                // we need to synch here too, 
                                // so threads don't interfere
                                synchronized (instance()) {
                                    // now reload the JTable
                                    fillUpJTable(plToSort);
                                    // thread this so you don't interfere with 
                                    // above...
                                    if (_playIndex > 0) {
                                        SwingUtilities.invokeLater(
                                            new Runnable() {
                                                public void run() {
                                                    focusChange(_playIndex);
                                                }
                                            });
                                    }
                                }
                            }
                        }).start();
                    }  // if (plToSort != null)
                } // sychronized
            } // if (e.getClickCount....
        } // mouseClicked()
    }


}
