package com.limegroup.gnutella.gui.mp3;

import com.limegroup.gnutella.gui.*;
import quicktime.*;
import quicktime.std.*;
import quicktime.std.movies.*;
import quicktime.app.QTFactory;
import quicktime.app.players.QTPlayer;
import quicktime.std.clocks.TimeRecord;

import java.io.*;

/**
 * This class is an audio player that utilizes the QuickTime for Java API 
 * from Apple. This class should only be used when both QuickTime is 
 * installed on the user's system and QuickTime for Java is installed.  
 * This is the case on all OS X machines, and the QuickTime 5 installer 
 * for Windows installs QuickTime for Java by default if Java is already 
 * installed.
 */
final class QuickTimeAudioPlayer extends AbstractAudioPlayer 
	implements FinalizeListener {

    /**
     * Handle to the current QuickTime player instance.
     */
    private static QTPlayer _player;
    
    /**
     * Flag for the current status of the player -- defaults to stopped.
     */
    private int _status = STATUS_STOPPED;
    
    /**
     * Handle to the abstract pathname of the last file played.
     */
    private File _curFile = new File(System.getProperty("user.dir"));
    

    /**
     * The constructor opens the QuickTime session and registers this class as
     * a <tt>Finalizable</tt> instance for closing it prior to shutdown.
     */
    QuickTimeAudioPlayer() {
        try {
            QTSession.open();
            GUIMediator.addFinalizeListener(this);
        } catch(quicktime.QTException e) {
        	GUIMediator.showInternalError(e);
        }
    }

    public int getStatus() {
        return _status;
    }

    public void unpause() {
		if(_player == null) return;
        try {
            _player.getMovieController().play(1);
            _status = STATUS_PLAYING;
        } catch(StdQTException sqte) {
        }
    }
    
    public void pause() {
		if(_player == null) return;
        try {
            _player.getMovieController().play(0);
            if (_status != STATUS_STOPPED)
                _status = STATUS_PAUSED;
        } catch(StdQTException sqte) {
        }
    }
    
    public void stop() {       
		if(_player == null) return;
        try {
            _player.getMovieController().play(0);
            _player.setTime(0);
            _status = STATUS_STOPPED;
        } catch(StdQTException sqte) {
        }
    }
    
    public void play(File file) throws IOException {
    	String soundLocation = file.getCanonicalPath();
            
        //this call works with a file://, http://, rtsp:// located movies
				
        if(!_curFile.equals(file)) {
        	try {
            	_player = (QTPlayer)QTFactory.makeDrawable (
                    	   "file://"  + soundLocation);
               	_player.setControllerVisible(false);
            } catch(QTException qte) {
            	return;
           	}
            		
            _player.startTasking();
        }
        
        try {
            _player.setRate(1);
        } catch(StdQTException sqte) {
            return;
        }
        		
        int frames = 0;
        try {
        	frames = _player.getDuration();
        } catch(QTException e) {
        	return;
        }
        this.fireSeekSetupRequired(frames);
        this.fireAudioPositionUpdated(0);
            		
       	_status = STATUS_PLAYING;
       	_curFile = file;
    }
    
    public int getFrameSeek() {
        return 0;
    }
    
    public void doFinalize() {
        // close the QuickTime session prior to exiting the virtual machine
        try {
            QTSession.close();
        } catch(NullPointerException e) {
            // the quicktime java implementation seems to throw an NPE
            // occasionally when closing.  since there's nothing we can do
            // about it, and the user is closing LimeWire anyway
            // (hence this being in a doFinalize method), we just ignore the
            // error and continue closing.
        }
    }
    
    // inherit doc comment
    public void refresh() {
    	if(_player == null) return;
        if(_status == STATUS_STOPPED) return;
    	int framesRead = 0;
    	int duration   = 0;
    	try {
    		framesRead = _player.getTime();
    		duration   = _player.getDuration();
    	} catch(StdQTException e) {
    		return;
        }
        
        if(framesRead == duration) {
        	if (_status != STATUS_STOPPED)
            	_status = STATUS_STOPPED;
			this.firePlayComplete();
		} else {
    		this.fireAudioPositionUpdated(framesRead);
    	}
    }
}
