package com.limegroup.gnutella.gui.options;

import com.limegroup.gnutella.gui.GUIMediator;

import com.sun.java.util.collections.ArrayList;
import java.io.*;
import javax.swing.*;
import java.awt.*;

/**
 * Manages the main options window that displays the various options 
 * windows.<p>
 *
 * This class also stores all of the main options panels to access
 * all of them regardless of how many there are or what their
 * specific type is.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class OptionsPaneManager {

	/**
	 * Constant for the main panel of the options window.
	 */
	private final JPanel MAIN_PANEL = new JPanel();

	/**
	 * Constant for the <tt>CardLayout</tt> used in the main panel.
	 */
	private final CardLayout CARD_LAYOUT = new CardLayout();


	/**
	 * Constant for the <tt>ArrayList</tt> containing all of the visible
	 * <tt>OptionsPane</tt> instances.
	 */
	private final ArrayList OPTIONS_PANE_LIST = new ArrayList();
	
	/**
	 * The constructor sets the layout and adds all of the <tt>OptionPane</tt>
	 * instances.
	 */
	public OptionsPaneManager() {
		MAIN_PANEL.setLayout(CARD_LAYOUT);		
    }

	/**
	 * Shows the window speficied by its title.
	 * 
	 * @param name the name of the <code>Component</code> to show
	 */
	public final void show(final String name) {
		CARD_LAYOUT.show(MAIN_PANEL, name);
	}

	/**
	 * Sets the options for each <tt>OptionPane</tt> instance in the 
	 * <tt>ArrayList</tt> of <tt>OptionPane</tt>s when the window is shown.
	 */
	public void initOptions() {
		for(int i=0, size = OPTIONS_PANE_LIST.size(); i<size; i++) {
			OptionsPane op = (OptionsPane)OPTIONS_PANE_LIST.get(i);
			op.initOptions();
		}
	}

	/**
	 * Applies the current settings in the options windows, storing them
	 * to disk.  This method delegates to the <tt>OptionsPaneManager</tt>.
	 *
	 * @throws IOException if the options could not be fully applied
	 */
	public final void applyOptions() throws IOException {
        boolean restartRequired = false;
		for(int i=0,	 size = OPTIONS_PANE_LIST.size(); i<size; i++) {
			OptionsPane op = (OptionsPane)OPTIONS_PANE_LIST.get(i);
            restartRequired |= op.applyOptions();
		}
        if(restartRequired)
            GUIMediator.showMessage("OPTIONS_RESTART_REQUIRED");
	}
	
	/**
	 * Returns the main <code>Component</code> for this class.
	 *
	 * @return a <code>Component</code> instance that is the main component
	 *         for this class.
	 */
	public final Component getComponent() {
		return MAIN_PANEL;
	}

	/**
	 * Adds the speficied window to the CardLayout based on its title.
	 *
	 * @param window the <code>OptionsPane</code> to add
	 */
	public final void addPane(final OptionsPane pane) {
		MAIN_PANEL.add(pane.getContainer(), pane.getName());
		OPTIONS_PANE_LIST.add(pane);
	}

}
