package com.limegroup.gnutella.gui.options.panes;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.IOException;

import javax.swing.JTextField;

import com.limegroup.gnutella.ErrorService;
import com.limegroup.gnutella.FileManager;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.gui.ButtonRow;
import com.limegroup.gnutella.gui.LabeledComponent;
import com.limegroup.gnutella.gui.SizedTextField;
import com.limegroup.gnutella.settings.SharingSettings;
import com.limegroup.gnutella.util.ManagedThread;

/**
 * This class defines the panel in the options window that allows the user
 * to change the directory for saving files.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class ExtensionsPaneItem extends AbstractPaneItem {

	/**
	 * Constant for the key of the locale-specific <code>String</code> for the 
	 * label on the component that allows to user to change the setting for
	 * this <tt>PaneItem</tt>.
	 */
	private final String OPTION_LABEL = "OPTIONS_SHARED_EXTENSIONS_BOX_LABEL";

	/**
	 * Handle to the <tt>JTextField</tt> that displays the save directory.
	 */
	private JTextField _extField;

	/**
	 * The stored value to allow rolling back changes.
	 */
	private String _extensions;

	/**
	 * The constructor constructs all of the elements of this 
	 * <tt>AbstractPaneItem</tt>.
	 *
	 * @param key the key for this <tt>AbstractPaneItem</tt> that the
	 *            superclass uses to generate locale-specific keys
	 */
	public ExtensionsPaneItem(final String key) {
		super(key);
		_extField = new SizedTextField();
		LabeledComponent comp = 
		    new LabeledComponent(OPTION_LABEL,_extField,
								 LabeledComponent.TOP_LEFT);

		String[] labelKeys = {
			"OPTIONS_SHARED_EXTENSIONS_DEFAULT_BUTTON_LABEL",
		};

		String[] toolTipKeys = {
			"OPTIONS_SHARED_EXTENSIONS_DEFAULT_BUTTON_TIP",
		};
        
		ActionListener[] listeners = {
			new DefaultExtensionsListener()
		};

		ButtonRow br = new ButtonRow(labelKeys, toolTipKeys, listeners, 
									 ButtonRow.X_AXIS, ButtonRow.LEFT_GLUE);

		add(comp.getComponent());
		add(getVerticalSeparator());		
		add(br);				
	}

	/**
	 * This class handles mouse clicks on the "Use Defaults" button
	 * of the extensions panel, setting the current extensions back
	 * to the defaults..
	 */
	private class DefaultExtensionsListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
            try {
    			SharingSettings.EXTENSIONS_TO_SHARE.revertToDefault();
    			_extField.setText(
                    SharingSettings.EXTENSIONS_TO_SHARE.getValue());
            } catch(Throwable t) {
                ErrorService.error(t);
            }
		}
	}

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Sets the options for the fields in this <tt>PaneItem</tt> when the 
	 * window is shown.
	 */
	public void initOptions() {
        _extensions = SharingSettings.EXTENSIONS_TO_SHARE.getValue();
        _extField.setText(_extensions);
	}

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Applies the options currently set in this window, displaying an
	 * error message to the user if a setting could not be applied.
	 *
	 * @throws IOException if the options could not be applied for some reason
	 */
	public boolean applyOptions() throws IOException {
        final String ext = _extField.getText();

        // Handle a change to the shared directories or list of extensions.
        // The loadSettings method is non-blocking, so threads are needed.
        if(!ext.equals(_extensions)) {
			SharingSettings.EXTENSIONS_TO_SHARE.setValue(_extField.getText());
            
			Thread loadFileManagerSettings =
    			new ManagedThread("ExtensionsChangedReloader") {
                    public void managedRun() {
                        FileManager fm = RouterService.getFileManager();
                        fm.loadSettings(true);
                    }
                };
			loadFileManagerSettings.setDaemon(true);
			loadFileManagerSettings.start();

            _extensions = _extField.getText();
        }
        
        return false;
    }		    
}
