package com.limegroup.gnutella.gui.options.panes;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;

import javax.swing.JTextField;

import com.limegroup.gnutella.ErrorService;
import com.limegroup.gnutella.gui.ButtonRow;
import com.limegroup.gnutella.gui.FileChooserHandler;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.LabeledComponent;
import com.limegroup.gnutella.gui.SaveDirectoryHandler;
import com.limegroup.gnutella.gui.SizedTextField;
import com.limegroup.gnutella.settings.SharingSettings;

/**
 * This class defines the panel in the options window that allows the user
 * to change the directory for saving files.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class SaveDirPaneItem extends AbstractPaneItem {

	/**
	 * Constant for the key of the locale-specific <code>String</code> for the 
	 * label on the component that allows to user to change the setting for
	 * this <tt>PaneItem</tt>.
	 */
	private final String OPTION_LABEL = "OPTIONS_SAVE_DIR_BOX_LABEL";

	/**
	 * Handle to the <tt>JTextField</tt> that displays the save directory.
	 */
	private JTextField _saveField;

	/**
	 * String for storing the initial save directory.
	 */
    private String _saveDirectory;


	/**
	 * The constructor constructs all of the elements of this 
	 * <tt>AbstractPaneItem</tt>.  This includes the row of buttons that
	 * allow the user to select the save directory.
	 *
	 * @param key the key for this <tt>AbstractPaneItem</tt> that the
	 *            superclass uses to generate strings
	 */
	public SaveDirPaneItem(final String key) {
		super(key);
		_saveField = new SizedTextField();
		LabeledComponent comp = 
		    new LabeledComponent(OPTION_LABEL, _saveField, 
								 LabeledComponent.TOP_LEFT);

		String[] labelKeys = {
			"OPTIONS_SAVE_DIR_BROWSE_BUTTON_LABEL",
			"OPTIONS_SAVE_DIR_DEFAULT_BUTTON_LABEL"
		};

		String[] toolTipKeys = {
			"OPTIONS_SAVE_DIR_BROWSE_BUTTON_TIP",
			"OPTIONS_SAVE_DIR_DEFAULT_BUTTON_TIP"
		};
        
		ActionListener[] listeners = {
			new SelectSaveDirectoryListener(),
			new DefaultListener()
		};

		ButtonRow br = new ButtonRow(labelKeys, toolTipKeys, listeners, 
									 ButtonRow.X_AXIS, ButtonRow.LEFT_GLUE);

		add(comp.getComponent());
		add(getVerticalSeparator());		
		add(br);				
	}

	/**
	 * This listener responds to the selection of the default savee 
	 * directory and sets the current save directory to be the default.
	 */
    private class DefaultListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
                _saveField.setText(
                    SharingSettings.DEFAULT_SAVE_DIR.getAbsolutePath());
            } catch(Throwable t) {
                ErrorService.error(t);
            }
        }
    }

	/**
	 * This listener displays a <tt>JFileChooser</tt> to the user,
	 * allowing the user to select the save directory.
	 */
    private class SelectSaveDirectoryListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            try {
    			File dir = 
    				FileChooserHandler.getInputDirectory(
                        MEDIATOR.getMainOptionsComponent());
                
                // If the user cancelled the file chooser, simply return.
                if(dir == null) return;  			
                
                // Otherwise, make sure they selected a valid directory that
                // they can really write to.
                if(!SaveDirectoryHandler.isSaveDirectoryValid(dir)) {
                    GUIMediator.showError(
                        "ERROR_INVALID_SAVE_DIRECTORY_SELECTION");
                    return;
                }
    		    try {
    			    String newDir = dir.getCanonicalPath();
    			    if(!newDir.equals(_saveDirectory)) {
    				    MEDIATOR.setSaveDirectoryChanged(true);
    				    _saveField.setText(newDir);
    			    }
    			} catch(IOException ioe) {
      			}
            } catch(Throwable t) {
                ErrorService.error(t);
            }
        }
    }

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Sets the options for the fields in this <tt>PaneItem</tt> when the 
	 * window is shown.
	 */
	public void initOptions() {
		try {
            File file = SharingSettings.getSaveDirectory();
            
            if (file == null) {
                throw (new FileNotFoundException());
            }
            
			_saveDirectory = file.getCanonicalPath();
		} catch(FileNotFoundException fnfe) {
			// simply use the empty string if we could not get the save
			// directory.
			_saveDirectory = "";
		} catch(IOException ioe) {
			_saveDirectory = "";
		}
        _saveField.setText(_saveDirectory);
	}

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Applies the options currently set in this window, displaying an
	 * error message to the user if a setting could not be applied.
	 *
	 * @throws IOException if the options could not be applied for some reason
	 */
	public boolean applyOptions() throws IOException {
        final String save = _saveField.getText();
        if(!save.equals(_saveDirectory)) {
			MEDIATOR.setSaveDirectoryChanged(true);
            try {
				File saveDir = new File(save);
                
                if(!saveDir.isDirectory()) {
                    if(!saveDir.mkdirs()) throw new IOException();
                }
        
                //SharingSettings.DIRECTORY_FOR_SAVING_FILES.setValue(saveDir);
                SharingSettings.setSaveDirectory(saveDir); /* updates Incomplete directory etc... */
                
                _saveDirectory = save;
            }
            catch(IOException ioe) {
                GUIMediator.showError("ERROR_INVALID_SAVE_DIRECTORY");
                _saveField.setText(_saveDirectory);
				throw new IOException();
            }

            catch(NullPointerException npe) {
                GUIMediator.showError("ERROR_INVALID_SAVE_DIRECTORY");
                _saveField.setText(_saveDirectory);
				throw new IOException();
			}
        }

        return false;
	}
}
