package com.limegroup.gnutella.gui.options.panes;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;

import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.gui.FileChooserHandler;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.StandardListEditor;
import com.limegroup.gnutella.settings.SharingSettings;
import com.limegroup.gnutella.util.FileUtils;
import com.limegroup.gnutella.util.ManagedThread;

/**
 * This class defines the panel in the options window that allows the user
 * to change the directory that are shared.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class SharedDirPaneItem extends AbstractPaneItem {

	/**
	 * Constant handle to the <tt>StandardListEditor</tt> that adds and removes
	 * remove hosts to aumotically connect to.
	 */
	private final StandardListEditor DIR_LIST = 
		new StandardListEditor(new SelectSharedDirectoryListener());

	/**
	 * Boolean for whether or not the directories in the list have
	 * changed.
	 */
	private boolean _directoryAdded = false;

	/**
	 * The constructor constructs all of the elements of this 
	 * <tt>AbstractPaneItem</tt>.
	 *
	 * @param key the key for this <tt>AbstractPaneItem</tt> that the
	 *            superclass uses to generate strings
	 */
	public SharedDirPaneItem(final String key) {
		super(key);
		add(DIR_LIST.getComponent());
	}

    /** 
	 * Adds a directory to the string of shared directories,
     * checking to make sure that the directory is not already 
	 * contained in the shared directory String. 
	 *
	 * @param dir a <tt>File</tt> instance denoting the abstract pathname 
     *  of the new shared directory
	 */
    private void addDirectory(File dir) {
		if(!dir.isDirectory()) return;
		
		File[] dirs = DIR_LIST.getDataAsFileArray();
        for(int i=0; i<dirs.length; i++) {
            if (dirs[i].equals(dir)) { 
                return; // directory is already shared
            }
        }
        DIR_LIST.addFile(dir);
        _directoryAdded = true;
    }

	/** 
	 * This class  shows the <tt>JFileChooser</tt> when the user presses 
	 * the button to add a new directory to the shared directories.  It
	 * adds the directory only if does not already exist in the list.
	 */
  	private class SelectSharedDirectoryListener implements ActionListener {	   
  		public void actionPerformed(ActionEvent ae) {
			try {				
				File dir = 
					FileChooserHandler.getInputDirectory(MEDIATOR.getMainOptionsComponent());
				if(dir == null) {
					return;
				}
				if(!dir.isDirectory() || !dir.canRead()) {
					GUIMediator.showError("ERROR_INVALID_SHARED_DIRECTORY");
					return;
				}
				SharedDirPaneItem.this.addDirectory(dir);
			} catch(Throwable e) {
				GUIMediator.showInternalError(e, "ChangeSharedDir");
			}
  		}
  	}

	/**
	 * Handles the case where the save directory is not shared,
	 * prompting the user for whether or not they would like
	 * to share it if it is not already shared.
	 */
	private void handleSaveDirectory() {	
		if(!MEDIATOR.getSaveDirectoryChanged()) return;
        
        File saveDir = SharingSettings.getSaveDirectory();
        //add if !(doesn't exist, isn't directory, or can't write)
        if ( saveDir != null &&
          saveDir.exists() &&
          saveDir.isDirectory() &&
          FileUtils.setWriteable(saveDir) ) {
            addDirectory(saveDir);
        }
	}


	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
	 * Sets the options for the fields in this <tt>PaneItem</tt> when the 
	 * window is shown.
	 */
	public void initOptions() {
		File[] dirs = (File[])SharingSettings.DIRECTORIES_TO_SHARE.getValue();
		DIR_LIST.setListData(dirs);
		DIR_LIST.resetList();
		_directoryAdded = false;
	}

	/**
	 * Defines the abstract method in <tt>AbstractPaneItem</tt>.<p>
	 *
     * This makes sure that the shared directories have, in fact, changed to
     * make sure that we don't load the <tt>FileManager</tt> twice.  This is
     * particularly relevant to the case where the save directory has changed,
     * in which case we only want to reload the <tt>FileManager</tt> once for 
     * any changes.<p>
     * 
	 * Applies the options currently set in this window, displaying an
	 * error message to the user if a setting could not be applied.
	 *
	 * @throws <tt>IOException</tt> if the options could not be applied 
     *  for some reason
	 */
	public boolean applyOptions() throws IOException {

		handleSaveDirectory();
		
        // Handle a change to the shared directories or list of extensions.
        // The loadSettings method is non-blocking, so threads are needed.
        if(DIR_LIST.getListChanged() || _directoryAdded) {
			DIR_LIST.resetList();
			_directoryAdded = false;
			SharingSettings.DIRECTORIES_TO_SHARE.setValue(DIR_LIST.getDataAsFileArray());
            
			Thread loadFileManagerSettings =
			    new ManagedThread("DirectoriesChangedReloader") {
    				public void managedRun() {
    					RouterService.getFileManager().loadSettings(true);
    				}
    			};
			loadFileManagerSettings.setDaemon(true);
			loadFileManagerSettings.start();
		}
        
        return false;
	}
}





