package com.limegroup.gnutella.gui.search;

import java.util.Properties;
import com.sun.java.util.collections.*;
import java.io.*;
import com.limegroup.gnutella.xml.*;
import com.limegroup.gnutella.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.util.*;
import java.util.ResourceBundle;
import java.util.MissingResourceException;

/**
 * The class that handles all the display details.
 * <p>
 * Has a singleton pattern,
 * <p>
 * When instantiated, loads up the display properties for each schema, and 
 * stores the URIs against the properties in a HashMap.
 * <p>
 * This class also has the responsibility of remembering which columns are 
 * visible in the search window and the width of each column
 * @author Sumeet Thadani
 */
public class DisplayManager {
    
    private static DisplayManager instance;

    /**
     * A mapping of schema description to ResourceBundle, for retrieving
     * the the localized description of an XML column.
     */
    private HashMap/* String -> ResourceBundle*/ schemaToDispProps;

    /**private constructor
     * <p>
     * loads up all the display properties for the schemas and populates the 
     * schemaToDispProps map
     * <p>
     * Next it tries to load up Display.props which contains the list
     * of columns the user has chosen to see, with prefererd widths
     */
    private DisplayManager() {
        schemaToDispProps = new HashMap();
        Collection schemas =
            LimeXMLSchemaRepository.instance().getAvailableSchemas();
        for(Iterator i = schemas.iterator(); i.hasNext(); )
            loadDisplayProperties((LimeXMLSchema)i.next());
    }

    //singleton accessor
    public static DisplayManager instance(){
        if(instance==null) {
            instance =  new DisplayManager();
            return instance;
        }
        else
            return instance;
    }

    /**
     * For given schema description and rawFieldName returns the Display name.
     * Either by looking at ResourceBundle or, by proecessing the rawName
     */
    public String getDisplayName(String rawName, String schemaDesc){
        ResourceBundle rb = (ResourceBundle)schemaToDispProps.get(schemaDesc);
        String dispName;    
        if(rb == null) { //ResourceBundle not found during load so use rawName
            return LimeXMLUtils.processColName(rawName);
        } else {
            try {
                dispName = rb.getString(rawName);
            } catch(MissingResourceException mre) {
                //couldn't find key = rawName so use rawName
                dispName = LimeXMLUtils.processColName(rawName);
                return dispName;
            }
        }

        dispName = stripPreferredSize(dispName);
        return dispName;
    }

    /**
     * Returns the translated display name for a given schemaURI.
     * @param schemaURI
     * @return the name of schema according to the resourcebundle
     */
    public String getDisplayTitleForSchema(String schemaURI) {
        String common = LimeXMLSchema.getDisplayString(schemaURI);
        LimeXMLSchema sc = 
            LimeXMLSchemaRepository.instance().getSchema(schemaURI);
        List fields = sc.getCanonicalizedFields();
        SchemaFieldInfo infoField = (SchemaFieldInfo)fields.get(0);
        String currField = infoField.getCanonicalizedFieldName();
        List currFieldList = XMLStringUtils.split(currField);
        String ancestorCanonicalized = (String)currFieldList.remove(0);
        return getDisplayName(ancestorCanonicalized 
                              + XMLStringUtils.DELIMITER
                              + common, common);
    }
    
    /////////////////////////private methods///////////////////////////
    
    //Utility method to that returns the DisplayName of a field without the
    //preferred with (which is usually ^ separated)
    public static String stripPreferredSize(String rawName){
        int i = rawName.indexOf("^");
        if(i < 0)
            return rawName;
        return rawName.substring(0,i);
    }
    
    //helper method to load properties.
    private void loadDisplayProperties(LimeXMLSchema schema) {
        ResourceBundle rb;
        try {
            String key = schema.getDescription();
            rb = GUIMediator.getXMLResourceBundle(key);
            schemaToDispProps.put(key, rb);
        } catch(MissingResourceException mre) {
            //couldn't find a display.properties file for this schema
            //will use the keys in the schemas for the values
            //see getDisplayName
        }
    }
}//end of class







