package com.limegroup.gnutella.gui.search;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import com.limegroup.gnutella.gui.*;
import com.limegroup.gnutella.xml.gui.*;
import com.limegroup.gnutella.xml.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.*;
import com.sun.java.util.collections.*;
import com.sun.java.util.collections.List;

/**
 * Inner panel that switches between the various kinds of
 * searching.
 */
class SearchInputPanel extends JPanel {
    
    /**
     * The current search label in what's new.
     */
    private final JLabel WHATSNEW_SEARCH_LABEL = new JLabel();

    /**
     * The current search label in normal search.
     */
    private final JLabel SEARCH_TYPE_LABEL = new JLabel();
    
    /**
     * The sole input text field that is at the top of all searches.
     */
    private final SearchField SEARCH_FIELD = new SearchField(15);
    
    /**
     * The input field for browse-host searches.
     */
    private final AutoCompleteTextField BROWSE_HOST_FIELD =
        new AutoCompleteTextField();
    
    /**
     * The What's New search button.
     */
    private final JButton WHATSNEW = new JButton(
            GUIMediator.getStringResource("SEARCH_TYPE_WHATSNEW"));
    
    /**
     * The JTabbedPane that switches between types of searches.
     */
    private final JTabbedPane PANE = new JTabbedPane(JTabbedPane.BOTTOM);
        
    /**
     * The CardLayout that switches between the detailed
     * search input information for each meta-type.
     */
    private final CardLayout META_CARDS = new CardLayout();
    
    /**
     * The panel that the META_CARDS layout uses to layout
     * the detailed search input fields.
     */
    private final JPanel META_PANEL = new JPanel(META_CARDS);
    
    /**
     * The name to use for the default panel that has no meta-data.
     */
    private static final String DEFAULT_PANEL_KEY = "defaultPanel";
    
    /**
     * The box that holds the schemas for searching.
     */
    private final SchemaBox SCHEMA_BOX = new SchemaBox();
    
    /**
     * The ditherer to use for the tab backgrounds.
     */
    private final Ditherer DITHERER =
            new Ditherer(62,
                        ThemeFileHandler.SEARCH_PANEL_BG_1.getValue(), 
                        ThemeFileHandler.SEARCH_PANEL_BG_2.getValue()
                        );
                    
    
    /**
     * The listener for new searches.
     */
    private final ActionListener SEARCH_LISTENER = new SearchListener();
    
    SearchInputPanel() {
        super(new BorderLayout(0, 5));

        final ActionListener schemaListener = new SchemaListener();
        SCHEMA_BOX.addSelectionListener(schemaListener);
        add(SCHEMA_BOX, BorderLayout.NORTH);

        JPanel searchEntry = createSearchEntryPanel();
        JPanel whatsnew = createWhatIsNewPanel();
        JPanel browseHost = createBrowseHostPanel();
        panelize(searchEntry);
        panelize(whatsnew);
        panelize(browseHost);
        PANE.add(GUIMediator.getStringResource("SEARCH_TYPE_KEYWORD"),
                 searchEntry);
        PANE.add(GUIMediator.getStringResource("SEARCH_TYPE_WHATSNEW"),
                 whatsnew);
       // PANE.add(GUIMediator.getStringResource("SEARCH_TYPE_BROWSEHOST"),
       //          browseHost);

        if(!ThemeSettings.isNativeTheme()) {
            PANE.setBorder(
              new LineBorder(ThemeFileHandler.SEARCH_GRID_COLOR.getValue()) {
                public void paintBorder(Component c, Graphics g,
                                        int x, int y, int width, int height) {
                    try {
                        Component sel = PANE.getSelectedComponent();
                        if(sel != null)
                            height = sel.getBounds().height + 4;                    
                    } catch(ArrayIndexOutOfBoundsException aioobe) {}
                    super.paintBorder(c, g, x, y, width, height);
                }
            });
        }
        add(PANE, BorderLayout.CENTER);

        add(Box.createVerticalStrut(20), BorderLayout.SOUTH);
        
        WHATSNEW_SEARCH_LABEL.setFont(UIManager.getFont("Table.font.bold"));
        Font bold = UIManager.getFont("Table.font.bold");
        Font bolder =
            new Font(bold.getName(), bold.getStyle(), bold.getSize() + 5);
        SEARCH_TYPE_LABEL.setFont(bolder);
        SEARCH_TYPE_LABEL.setPreferredSize(new Dimension(100, 20));
        schemaListener.actionPerformed(null);
    }
    
    /**
     * Gets the KeyProcessingTextField that key events can be forwarded to.
     */
    KeyProcessingTextField getForwardingSearchField() {
        if(isNormalSearchType())
            return SEARCH_FIELD;
        else if(isBrowseHostSearchType())
            return BROWSE_HOST_FIELD;
        else
            return null;
    }
    
    /**
     * Determines if a key event can be forwarded to the search.
     */
    boolean isKeyEventForwardable() {
        return isNormalSearchType() ||
               isBrowseHostSearchType();
    }
    
    /**
     * Determines if browse-host is selected.
     */
    boolean isBrowseHostSearchType() {
        return PANE.getSelectedIndex() == 2;
    }
    
    /**
     * Determines if what is new is selected.
     */
    boolean isWhatIsNewSearchType() {
        return PANE.getSelectedIndex() == 1;
    }
    
    /**
     * Determines if keyword is selected.
     */
    boolean isNormalSearchType() {
        return PANE.getSelectedIndex() == 0;
    }    
    
    void requestSearchFocusImmediately() {
        if(isNormalSearchType()) {
            if(SCHEMA_BOX.getSelectedSchema() != null) {
                getInputPanel().requestFirstFocus();
            } else {
                SEARCH_FIELD.requestFocus();
            }
        } else if(isWhatIsNewSearchType()) {
            WHATSNEW.requestFocus();
        } else if(isBrowseHostSearchType()) {
            BROWSE_HOST_FIELD.requestFocus();
        }
    }
    
    void requestSearchFocus() {
        // Workaround for bug manifested on Java 1.3 where FocusEvents
        // are improperly posted, causing BasicTabbedPaneUI to throw an
        // ArrayIndexOutOfBoundsException.
        // See:
        // http://developer.java.sun.com/developer/bugParade/bugs/4523606.html
        // http://developer.java.sun.com/developer/bugParade/bugs/4379600.html
        // http://developer.java.sun.com/developer/bugParade/bugs/4128120.html
        // for related problems.
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                requestSearchFocusImmediately();
            }
        });
    }
    
    /**
     * Sets all components in this component to be not opaque
     * and sets the correct background panel.
     */
    private void panelize(JComponent c) {
        GUIUtils.setOpaque(false, c);
        if(!ThemeSettings.isNativeTheme())
            c.setOpaque(true);
        c.setBackground(ThemeFileHandler.SEARCH_PANEL_BG_2.getValue());
        c.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 0));
    }
    
    /**
     * Creates the 'Direct Connect' browse host input panel.
     */
    private JPanel createBrowseHostPanel() {
        BROWSE_HOST_FIELD.addActionListener(SEARCH_LISTENER);
        
        JLabel label = new JLabel(
            GUIMediator.getStringResource("SEARCH_BROWSEHOST_TITLE"));
        label.setFont(UIManager.getFont("Table.font.bold"));
        label.setPreferredSize(new Dimension(GUIUtils.width(label), 10));
        JPanel labelPanel = new BoxPanel(BoxPanel.Y_AXIS);
        labelPanel.add(GUIUtils.center(label));
        
        JTextArea text = new JTextArea(
            GUIMediator.getStringResource("SEARCH_BROWSEHOST_DESC"));
        text.setLineWrap(true);
        text.setWrapStyleWord(true);
        text.setColumns(15);
        text.setEditable(false);
        text.setFont(UIManager.getFont("Table.font"));
        text.setForeground(label.getForeground());
        
        JButton search = new JButton(
            GUIMediator.getStringResource("SEARCH_BROWSEHOST_BUTTON"));
        search.addActionListener(SEARCH_LISTENER);
        
        JPanel panel = new BoxPanel(BoxPanel.Y_AXIS);
        panel.add(labelPanel);
        panel.add(Box.createVerticalStrut(15));
        panel.add(text);
        panel.add(Box.createVerticalStrut(15));
        panel.add(BROWSE_HOST_FIELD);        
        panel.add(Box.createVerticalStrut(15));
        panel.add(GUIUtils.center(search));

        JPanel centered = new DitherPanel(DITHERER);
        centered.setLayout(new FlowLayout(FlowLayout.CENTER, 0, 0));
        centered.add(panel);
        return centered;
    }
    
    /**
     * Creates the What's New input panel.
     */
    private JPanel createWhatIsNewPanel() {
        JLabel label = new JLabel(
            GUIMediator.getStringResource("SEARCH_WHATSNEW_TITLE_ONE"));
        label.setFont(UIManager.getFont("Table.font.bold"));
        label.setPreferredSize(new Dimension(GUIUtils.width(label), 10));
        JLabel label2 = new JLabel(
            GUIMediator.getStringResource("SEARCH_WHATSNEW_TITLE_TWO"));        
        label2.setFont(UIManager.getFont("Table.font.bold"));
        JPanel labelPanel = new BoxPanel(BoxPanel.Y_AXIS);
        labelPanel.add(GUIUtils.center(label));
        labelPanel.add(GUIUtils.center(label2));
        
        JTextArea text = new JTextArea(
            GUIMediator.getStringResource("SEARCH_WHATSNEW_DESC"));
        text.setLineWrap(true);
        text.setWrapStyleWord(true);
        text.setColumns(15);
        text.setEditable(false);
        text.setFont(UIManager.getFont("Table.font"));
        text.setForeground(label.getForeground());
        
        JTextArea type = new JTextArea(
            GUIMediator.getStringResource("SEARCH_CURRENT_SEARCH") + 
            "  ");
        type.setFont(UIManager.getFont("Table.font"));
        type.setEditable(false);
        type.setForeground(label.getForeground());
        JPanel typePanel = new JPanel(new FlowLayout(FlowLayout.CENTER, 0, 0));
        typePanel.add(type);
        typePanel.add(WHATSNEW_SEARCH_LABEL);
        
        WHATSNEW.addActionListener(SEARCH_LISTENER);
        
        final JCheckBox hideAdult = new JCheckBox(
            GUIMediator.getStringResource("SEARCH_WHATSNEW_HIDE_ADULT"),
            FilterSettings.FILTER_WHATS_NEW_ADULT.getValue());
        hideAdult.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ae) {
                FilterSettings.FILTER_WHATS_NEW_ADULT.
                    setValue(hideAdult.isSelected());
            }
        });

        JPanel panel = new BoxPanel(BoxPanel.Y_AXIS);
        panel.add(labelPanel);
        panel.add(Box.createVerticalStrut(15));
        panel.add(text);
        panel.add(Box.createVerticalStrut(30));
        panel.add(typePanel);
        panel.add(Box.createVerticalStrut(5));
        panel.add(GUIUtils.center(WHATSNEW));
        if(!FilterSettings.FILTER_ADULT.getValue()) {
            panel.add(Box.createVerticalStrut(30));
            panel.add(GUIUtils.center(hideAdult));
        }

        JPanel centered = new DitherPanel(DITHERER);
        centered.setLayout(new FlowLayout(FlowLayout.CENTER, 0, 0));
        centered.add(panel);
        return centered;
    }

    private JPanel createSearchEntryPanel() {
        SEARCH_FIELD.addActionListener(SEARCH_LISTENER);

        // add the default search input panel to the meta cards
        META_PANEL.add(createDefaultSearchPanel(), DEFAULT_PANEL_KEY);

        // add each other media to the meta cards
        List medias = NamedMediaType.getAllNamedMediaTypes();
        for(Iterator i = medias.iterator();  i.hasNext(); ) {
            NamedMediaType nmt = (NamedMediaType)i.next();
            String name = nmt.getName();
            LimeXMLSchema schema = nmt.getSchema();
            // If a schema exists, add it as a possible type.
            if(schema != null) {
                InputPanel panel = new InputPanel(schema, SEARCH_LISTENER, 
                                                  SEARCH_FIELD.getDocument());
                panel.addMoreOptionsListener(new MoreOptionsListener());
                JScrollPane pane = new JScrollPane(panel,
                    JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
                    JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
                pane.setOpaque(false);
                pane.setBorder(BorderFactory.createEmptyBorder());
                pane.getViewport().setBorder(null);
                BoxPanel box = new BoxPanel(BoxPanel.Y_AXIS);
                box.add(pane);
                box.add(Box.createVerticalStrut(5));
                box.add(createSearchButtonPanel());
                META_PANEL.add(GUIUtils.left(box), name);
            }
        }
        
        JPanel search = new DitherPanel(DITHERER);
        search.setLayout(new BoxLayout(search, BoxLayout.Y_AXIS));
        search.add(GUIUtils.left(SEARCH_TYPE_LABEL));
        search.add(Box.createVerticalStrut(5));
        search.add(META_PANEL);
        return search;
    }
    
    /**
     * Creates the default search input of:
     *    Filename
     *    [   input box  ]
     */
    private JPanel createDefaultSearchPanel() {
        
        JPanel label = createLabel(
			GUIMediator.getStringResource("SEARCH_DEFAULT_FIELD_TITLE"));
        JPanel fullPanel = new BoxPanel(BoxPanel.Y_AXIS);
        fullPanel.add(label);
        fullPanel.add(Box.createVerticalStrut(3));
        fullPanel.add(GUIUtils.left(SEARCH_FIELD));
        fullPanel.add(Box.createVerticalStrut(5));
        fullPanel.add(createSearchButtonPanel());

        return GUIUtils.left(fullPanel);
    }
    
    private JPanel createLabel(String text) {
        JPanel labelPanel = new BoxPanel(BoxPanel.X_AXIS);
        labelPanel.setOpaque(false);
        labelPanel.add(new JLabel(text));
        labelPanel.add(Box.createHorizontalGlue());		
        return labelPanel;
    }    

    /**
     * Creates the search button & inserts it in a panel.
     */
    private JPanel createSearchButtonPanel() {
        JPanel b = new JPanel(new FlowLayout(FlowLayout.LEFT, 0, 0));
        JButton searchButton = new JButton(
            GUIMediator.getStringResource("SEARCH_SEARCH_BUTTON_LABEL"));        
        searchButton.setToolTipText(
            GUIMediator.getStringResource("SEARCH_SEARCH_BUTTON_TIP"));

        searchButton.addActionListener(SEARCH_LISTENER);
      
        b.add(Box.createHorizontalGlue());
        b.add(searchButton);
        return b;
    }
    
    /**
     * Gets the visible component in META_PANEL.
     */
    private JComponent getVisibleComponent() {
        for(int i = 0; i < META_PANEL.getComponentCount(); i++) {
            Component current = META_PANEL.getComponent(i);
            if(current.isVisible())
                return (JComponent)current;
        }
        return null;
    }
    
    /**
     * Gets the visible scrollpane.
     */
    private JScrollPane getVisibleScrollPane() {
        JComponent parent = (JComponent)getVisibleComponent().getComponent(0);
        JScrollPane pane = null;
        for(int i = 0; i < parent.getComponentCount(); i++) {
            Component current = parent.getComponent(i);
            if(current.isVisible() && current instanceof JScrollPane) {
                return (JScrollPane)current;
            }
        }
        return null;
    }
    
    /**
     * Retrieves the InputPanel that is currently visible.
     */
    private InputPanel getInputPanel() {
        JScrollPane pane = getVisibleScrollPane();
        if(pane == null)
            return null;
        else
            return (InputPanel)pane.getViewport().getView();
    }
    
	/**
	 * Listener for selecting a new schema.
	 */
	private class SchemaListener implements ActionListener {
	    public void actionPerformed(ActionEvent event) {
	        if(SCHEMA_BOX.getSelectedSchema() != null) {
	            META_CARDS.show(META_PANEL, SCHEMA_BOX.getSelectedItem());
            } else {
                META_CARDS.show(META_PANEL, DEFAULT_PANEL_KEY);
            }
            WHATSNEW_SEARCH_LABEL.setText(SCHEMA_BOX.getSelectedItem());
            WHATSNEW_SEARCH_LABEL.setPreferredSize(
                new Dimension(GUIUtils.width(WHATSNEW_SEARCH_LABEL), 20));
            SEARCH_TYPE_LABEL.setText(SCHEMA_BOX.getSelectedItem());
            requestSearchFocus();
        }
    }
    
    /**
     * Listener for starting a new search.
     */
    private class SearchListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            String query = null;
            String xml = null;
            MediaType mt = SCHEMA_BOX.getSelectedMediaType();
            InputPanel panel = null;
            SearchInformation info = null;            
            
            if(isWhatIsNewSearchType()) {
                info = SearchInformation.createWhatsNewSearch(
                    SCHEMA_BOX.getSelectedItem(), mt);
            } else if(isNormalSearchType()) {
                if(SCHEMA_BOX.getSelectedSchema() != null) {
                    panel = getInputPanel();
                    query = panel.getStandardQuery();
                    xml = panel.getInput(true);
                } else {
                    query = SEARCH_FIELD.getText();
                }
                info = SearchInformation.createKeywordSearch(query, xml, mt);
            } else if(isBrowseHostSearchType()) {
                String user = BROWSE_HOST_FIELD.getText();
                info = SearchInformation.createBrowseHostSearch(user);
                mt = MediaType.getAnyTypeMediaType(); // always any
            }
            
            // If the search worked, store & clear it.
            if(SearchMediator.triggerSearch(info) != null) {
                if(info.isKeywordSearch()) {
                    // Add the necessary stuff for autocompletion.
                    if(panel != null) {
                        panel.storeInput();
                        panel.clear();
                    } else {
                        SEARCH_FIELD.addToDictionary();
                    }
    
                    // Clear the existing search.
                    SEARCH_FIELD.setText("");
                } else if(info.isBrowseHostSearch()) {
                    BROWSE_HOST_FIELD.addToDictionary();
                    BROWSE_HOST_FIELD.setText("");
                }
            }
        }
    }
    
    /**
     * Listener for 'more options'.
     */
    private class MoreOptionsListener implements ActionListener {
        public void actionPerformed(ActionEvent e) {
            JCheckBox box = (JCheckBox)e.getSource();
            JComponent c = getVisibleComponent();
            if(box.isSelected())
                c.setLayout(new BoxLayout(c, BoxLayout.Y_AXIS));
            else
                c.setLayout(new FlowLayout(FlowLayout.LEFT, 0, 0));

            invalidate();
            revalidate();
            repaint();
        }
    }
}    