package com.limegroup.gnutella.update;

import com.sun.java.util.collections.Hashtable;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.update.gui.UpdateCallback;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.ManagedThread;
import com.limegroup.gnutella.settings.ApplicationSettings;
import com.limegroup.gnutella.settings.SettingsHandler;
import java.io.*;
import java.net.*;
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.util.Properties;

/**
 * Handles updating of multiple files as well as updating of any other 
 * attributes necessary for the update, such as the classpath.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class MultipleFileUpdator extends AbstractUpdator implements Runnable {

	/**
	 * Constant for the name of the InstallAnywhere lax file.
	 */
	private final String LAX_FILE_NAME = "LimeWire.lax";

	/**
	 * Flag for whether or not the user has cancelled the update this 
	 * is volatile because multiple threads will likely modify it and 
	 * should not use their own copies.
	 */
	private volatile boolean _updateCancelled = false;
	
	/**
	 * Handle to a file downloader object for handling file downloads.
	 */
	private FileDownloader _fileDownloader;

	/**
	 * Constructor creates a new <tt>MultipleFileUpdator</tt> instance.
	 */
	MultipleFileUpdator() {
		_updateType = "MultipleFileUpdator";
		_fileDownloader = new FileDownloader();
	}

	/** 
	 * Implements the Updator interface.
	 *
	 * Cancels the update.
	 */
	public synchronized void cancelUpdate() {
		_updateCancelled = true;
		_fileDownloader.stopDownload();
		UpdateCallback.hideProgressWindow();

		// notify the waiting thread so that initialization 
		// can continue
		notifyAll();
	}

	/** 
	 * Implements the Updator interface.
	 * Prompts the user for whether or not they would like to update.  
	 * If they would like to update, it begins downloading the necessary
	 * files.
	 */
	public void doUpdate() {
		boolean update = UpdateCallback.showUpdatePrompt(_message);

		if(!update) return;

		// if the user has chosen to update, update the jar file.
		synchronized(this) {
			Thread jarUpdateThread = new ManagedThread(this, "FileUpdater");
			jarUpdateThread.setDaemon(true);
			jarUpdateThread.start();
			try {
				wait();
			} catch(InterruptedException ie) {
			}
		}
	}

	/**
	 * Implements the Runnable interface.  
	 * Cycles through the urls returned from the servlet, downloading each one.
	 */
	public void run() {
		StringTokenizer st = new StringTokenizer(_url, ";");
		String curUrl = "";
		String curFile = "";
		UpdateCallback.showProgressWindow();		
		while(st.hasMoreTokens() && !_updateCancelled) {
			curUrl = st.nextToken();
			curFile = getNewFileName(curUrl);
			try {
				// get the name of the new jar file from the url
				String newFileName = getNewFileName(curUrl);
				// get the full path for writing the new jar file
				String newFilePath = getNewFilePath(newFileName);
				_fileDownloader.downloadFile(curUrl, newFilePath);
				
			} catch(FileDownloadException fde) {
				displayErrorAndExit(fde.getMessage());
			}
		}		
		if(!_updateCancelled) {
			applyUpdateSettings();
			displayUpdateSucceededMessage();			
			GUIMediator.shutdown();
		}
	}

	/**
	 * Diplays a message to the user notifying them that the 
	 * update has succceeded and to restart the application.
	 */
	private void displayUpdateSucceededMessage() {
		// hide the progress window
		UpdateCallback.hideProgressWindow();
		// notify the user that the update has completed successfully
		String msg = "UPDATE_SUCCEEDED_MESSAGE";
		UpdateCallback.displayMessage(msg);						
	}

	/**
	 * Delegates the setting of the application properties to helper 
	 * methods depending on what operating system we're on (and  
	 * therefore which external text file we're using to get the 
	 * application properties).
	 */
	private void applyUpdateSettings() {
		if(CommonUtils.isMacClassic()) {
			try {
				applyLaxSettings();
			} catch(LaxFileUpdateException lfue) {
				displayErrorAndExit(lfue.getMessage());
			}
		}
		else applyStandardSettings();		
	}

	/**
	 * Applies any necessary settings for the next time LimeWire is started.
	 */
	private void applyStandardSettings() {
        ApplicationSettings.JAR_NAME.setValue(_classpath);
        SettingsHandler.save();
	}

	/**
	 * Updates the LimeWire.lax file that is used by InstallAnywhere 
	 * to set the classpath and the main class to load.
	 *
	 * @throws <tt>LaxFileUpdateException</tt> if the lax file could not 
	 *  be updated successfully
	 */
	private void applyLaxSettings() throws LaxFileUpdateException {
  		Properties props = new Properties();
  		FileInputStream fis = null;
		FileOutputStream fos = null;
  		try {
			File laxFile = new File(LAX_FILE_NAME);
  			fis = new FileInputStream(laxFile);
  			props.load(fis);
			props.put("lax.class.path", _classpath);
			props.put("lax.main.class", _mainClass);
			fis.close();
			fos = new FileOutputStream(laxFile);
			props.save(fos, "");
			fos.close();
  		} catch(FileNotFoundException fnfe) {
			String msg = "UPDATE_ERROR_LAX_FILE_NOT_FOUND";
			throw new LaxFileUpdateException(msg);
  		} catch(SecurityException se) {
			String msg = "UPDATE_ERROR_LAX_SECURITY";
			throw new LaxFileUpdateException(msg);
  		} catch(IOException ioe) {
			String msg = "UPDATE_ERROR_LAX_IO";
			throw new LaxFileUpdateException(msg);
  		} finally {
			try {
				if(fis != null) fis.close();
				if(fos != null) fos.close();
			} catch(IOException ioe) {
			}
		}
	}

	/**
	 * Displays an error message to the user and shuts down the application.
	 */
	private void displayErrorAndExit(String msg) {
		// notify the user of the error.
		UpdateCallback.displayErrorMessage(msg);

		// shutdown the application 
		GUIMediator.shutdown();
	}

	/**
	 * Parses the update url to obtain the name of the new jar file.
	 *
	 * @return The name of the new jar file
	 *
	 * @param url The url of the new jar file
	 */
	private String getNewFileName(String url) {
		// get the index in the url for extracting the jar name
		int index = url.lastIndexOf("/") + 1;

		// extract the name of the jar from the url
		return url.substring(index);
	}

	/**
	 * Determines what the full path of the new jar file should
	 * be on the local file system.
	 * 
	 * @param newJarName The name of the new jar file, such as
	 *                   "LimeWire15.jar".
	 *
	 * @return The full path where the new jar file should be stored.
	 */
	private String getNewFilePath(String newFileName) {
		return new File(newFileName).getAbsolutePath();
	}

//  	public static void main(String[] args) {
//    		SettingsManager.instance();
//  		MultipleFileUpdator updator = new MultipleFileUpdator();
//  		updator.setClassPath("totally random classpath");
//  		updator.setMainClass("totally random main class");
//  		try {
//  			updator.applyLaxSettings();
//  		} catch(LaxFileUpdateException lfue) {
//  			System.out.println("exception caught: "+lfue.getMessage());
//  		}
//  	}

	// tests the updating of the lax file
//    	public void testLaxFileUpdate(String url) {
//    		//StringTokenizer st = new StringTokenizer(url, ";");
//    		//String curUrl = "";
//    		//String curFile = "";
//    		//StringBuffer sb = new StringBuffer();
//    		//UpdateGUIHandler.instance().showProgressWindow();		
//    		//while(st.hasMoreTokens()) {
//  		//curUrl = st.nextToken();
//  		//curFile = getNewFileName(curUrl);
//  		//sb.append(curFile);
//    			// append the path separator only if this is not
//    			// the last file.
//  		//if(st.hasMoreTokens()) sb.append(File.pathSeparator); 
//    		//}		
//    		//_updateClassPath = sb.toString();
//    		//System.out.println("classpath: "+_updateClassPath);
//    		try {
//    			setLaxClassPath();
//    		} catch(LaxFileUpdateException lfue) {
//    			displayErrorAndExit(lfue.getMessage());
//    		}
//    	}

	// tests the multiple file updator without going through 
	// the servlet.
//  	public static void main(String[] args) {
//    		SettingsManager.instance();

//  		// this is necessary to enable cancel operations
//    		UpdateManager um = UpdateManager.instance();
//  		MultipleFileUpdator updator = new MultipleFileUpdator();

//  		// this is also necessary to enable cancel
//    		um.setUpdator(updator);
//  		String url1 = "http://www.limewire.com/updates/collections.jar";
//  		String url2 = "http://www.limewire.com/updates/LimeWire.jar";
//  		String url3 = "http://www.limewire.com/updates/LimeWireOSX.jar";		
//  		String url4 = "http://www.limewire.com/updates/LimeWireUpdateTest.jar";		
//  		String url = url1 + ";" + url2 + ";" + url3 + ";" + url4;
//  		updator.setUpdateURL(url);
//  		updator.setMessageToDisplay("THIS IS ONLY A TEST");
//  		updator.setClassPath("LimeWire.jar;collections.jar");
//  		updator.doUpdate();
//  	}
	
	// tests the updating of the lax file.
//    	public static void main(String[] args) {
//    		CommonUtils.initialize();
//    		MultipleFileUpdator updator = new MultipleFileUpdator();
//  		updator.setClassPath("lax.jar:collections.jar:LimeWire.jar:Newcookies.jar");
//  		updator.setMainClass("a.a.a.a.a.a.a");
//  		try {
//  			updator.applyLaxSettings();
//  		} catch(LaxFileUpdateException lfue) {
//  			System.out.println("lax exception caught");
//  		}
//  	}
	
	// tests the updating of the lax file.
//    	public static void main(String[] args) {
//    		CommonUtils.initialize();
//    		MultipleFileUpdator updator = new MultipleFileUpdator();
//    		String url1 = "http://www.limewire.com/updates/collections.jar";
//    		String url2 = "http://www.limewire.com/updates/LimeWire.jar";
//    		String url3 = "http://www.limewire.com/updates/LimeWireOSX.jar";		
//    		String url4 = "http://www.limewire.com/updates/LimeWireUpdateTest.jar";		
//    		String url = url1 + ";" + url2 + ";" + url3 + ";" + url4;
//    		updator.testLaxFileUpdate(url);
//    	}

//    	public static void main(String[] args) {
//    		CommonUtils.initialize();
//    		MultipleFileUpdator updator = new MultipleFileUpdator();
//    		boolean update = updator.updateJarNames("LimeWireUpdateTest.jar");
//    		System.out.println("jar renaming succeeded: "+update);
//    	}
	
//  	public boolean updateLAXFileTest(String name) {
//  		return updateLAXFile(name);
//  	}

//  	public static void main(String[] args) {
//  		CommonUtils.initialize();
//  		// test the lax file updating
//  		MultipleFileUpdator updator = new MultipleFileUpdator();
//  		System.out.println("lax update: "+updator.updateLAXFile(args[0]));
//  	}

//  	public static void main(String[] args) {
//  		String str = "http://www.limewire.com/updates/mac_classic/LimeWire14.jar";	
//  		// get the index in the url for extracting the jar name
//  		int index = str.lastIndexOf("/") + 1;

//  		// extract the name of the jar from the url
//  		String newJarName = str.substring(index);

//  		System.out.println("str test result: "+newJarName);
//  	}

//  	public static void main(String[] args) {
//  		System.out.println("MultipleFileUpdator::main");
//  		CommonUtils.initialize();
//  		MultipleFileUpdator updator = new MultipleFileUpdator();
//  		//updator.run();
		
//  	}
}
