package com.limegroup.gnutella.update.gui;

import com.limegroup.gnutella.gui.*;
import javax.swing.*;

/**
 * This class handles all gui callbacks from the udpate code,
 * displaying messages and such to the user.
 */
public final class UpdateCallback {
	
	/**
	 * Handle to the window displaying the progress of the update.
	 */
	private static ProgressWindow _progressWindow = 
		new ProgressWindow();

	/** 
	 * Shows the message window asking the user if they would like to 
	 * update to the newer version.  Returns true if the user would like 
	 * to upgrade.  If the user clicks "Don't check again", changes the 
	 * SettingsManager. 
	 *
	 * @param message the message to display to the user regarding
	 *                the update.
	 *
	 * @return <tt>true</tt> if the user selects the "yes" option in
	 *         response to the message, <tt>false</tt> otherwise
	 */
	public static final boolean showUpdatePrompt(String message) {
		UpdatePanel panel = 
			UpdatePanel.createFloatingUpdatePanel(message);
		int response = panel.showDialog();
		return (response == UpdatePanel.UPDATE_NOW);
	}

	
	public static final boolean showUpdatePromptWithParentFrame(String message) {
		UpdatePanel panel = 
			UpdatePanel.createUpdatePanelWithParentFrame(message);
		int response = panel.showDialog();
		return (response == UpdatePanel.UPDATE_NOW);		
	}

	/** 
	 * Shows the window to the user that shows the progress of the update.
	 */
	public static final void showProgressWindow() {
		_progressWindow.setVisible(true);
	}

	/**
	 * Sets the text displayed in the update progress window.
	 *
	 * @param text the text to display in the progress window
	 */
	public static final void setDownloadFile(String text) {
		_progressWindow.setDownloadFile(text);
	}

	/**
	 * Sets the size of the download for calculating the percent complete.
	 *
	 * @param size the size of the download in bytes
	 */
	public static final void setDownloadSize(int size) {
		_progressWindow.setDownloadSize(size);
	}
	
	/** 
	 * Hides the progress window and disposes of it. 
	 */
	public static final void hideProgressWindow() {
		_progressWindow.setVisible(false);
		_progressWindow.dispose();
	}

	/** 
	 * Updates the progress window with the new amount of the file that has
	 * been read.
	 *
	 * @param amountRead the amount of the file that has been read in bytes
	 */
	public static final void updateProgressWindow(int amountRead) {
		_progressWindow.setAmountRead(amountRead);
		SwingUtilities.invokeLater(_progressWindow);
	}

	/**
	 * Displays a message to the user indicating that the update has failed 
	 * for some reason. The error <tt>String</tt> parameter must be of 
	 * the form gerund (verb ending in 'ing') followed by the end of the 
	 * sentence (including a period), as in:
	 * 'connecting to the LimeWire server.'
	 * 
	 * @param error <tt>String</tt> beginning with a gerund and ending 
	 *              with a period that specifies the type of error for display 
	 *              to the user
	 */
	public static final void displayErrorMessage(String error) {
		String msgStart = "UPDATE_ERROR_MESSAGE_START";
		String msgEnd   = "UPDATE_ERROR_MESSAGE_END";
		GUIMediator.showError(msgStart, error, msgEnd);
	}

	/**
	 * Diplays a message to the user.
	 * 
	 * @param message the message to diplay
	 */
	public static final void displayMessage(String message) {
		GUIMediator.showMessage(message);
	}
}
