package com.limegroup.gnutella.xml.gui;

import java.awt.Component;
import java.awt.FlowLayout;

import javax.swing.*;

import com.limegroup.gnutella.gui.AutoCompleteTextField;
import com.sun.java.util.collections.Iterator;
import com.sun.java.util.collections.LinkedList;
import com.sun.java.util.collections.List;

/**
 * Specialized class encapsulating non-standard options in the XML search
 * user interface.  This class is necessary to explicitly handle theme and 
 * look and feel changes, as the extra options may not be visible when the
 * theme is changed, making it necessary to explicitly update their UI using
 * this class.
 */
final class MoreOptionsHandler {

    /**
     * Variable for the more options panel containing all of the text fields and
     * associated labels.
     */
    private Box _moreOptionsPanel;
    
    /**
     * Constant <tt>List</tt> of the more options text fields.
     */
    private final List FIELDS = new LinkedList();
    
    /**
     * Constant <tt>List</tt> of the more options string labels.
     */
    private final List LABELS = new LinkedList();

    /**
     * Creates a new <tt>MoreOptionsHandler</tt> instance.
     */
    public MoreOptionsHandler() {}

    /**
     * Adds the specified text field and label string to our stored data
     * of fields and string for this handler.
     * 
     * @param field the text field to add
     * @param label the label for the text field
     */
    void addTextField(AutoCompleteTextField field, String label)  {
        FIELDS.add(field);
        LABELS.add(label);
    }

    /**
     * Adds the specified combo box and label string to our stored data
     * of fields and string for this handler.
     * 
     * @param combo the combo box to add
     * @param label the label for the text field
     */
    void addComboBox(JComboBox combo, String label)  {
        FIELDS.add(combo);
        LABELS.add(label);
    }

    /**
     * Creates a new panel for the more options components.  This
     * is necessary because the look and feel/theme may have changes,
     * and we need to apply the new UI settings.
     * 
     * @return a new <tt>Component</tt> with all of the fields for this
     * set of additional options
     */
    Component createPanel() {
        _moreOptionsPanel = new Box(BoxLayout.Y_AXIS);
        Iterator fieldsIter = FIELDS.iterator();
        Iterator labelsIter = LABELS.iterator();
        while(fieldsIter.hasNext() && labelsIter.hasNext())  {
            JPanel outerPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 0, 0));
            outerPanel.setOpaque(false);
            Box curPanel = new Box(BoxLayout.Y_AXIS);
            String label = (String) labelsIter.next();
            JComponent comp = (JComponent) fieldsIter.next();
            comp.updateUI();
            
            curPanel.add(IndentingPanel.createLabel(label));
            curPanel.add(Box.createVerticalStrut(3));
            curPanel.add(comp);
            outerPanel.add(curPanel);
            _moreOptionsPanel.add(outerPanel);
        }
        return _moreOptionsPanel;
    }

    /**
     * Accessor for the more options panel.  Used particularly when removing
     * it from the UI when the user selects to no longer display the extra options.
     * 
     * @return the more options panel for display.
     */
    Box getPanel() {
        return _moreOptionsPanel;
    }
}

