/*
 * Copyright (c) 2002-2004 David Keiichi Watanabe
 * davew@xlife.org
 *
 * Modified by (c) 2004-2005 heavy_baby
 * heavy_baby@users.sourceforge.jp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.sourceforge.cabos;

import com.limegroup.gnutella.*;
import com.limegroup.gnutella.downloader.*;
import com.limegroup.gnutella.settings.*;
import com.limegroup.gnutella.util.*;
import com.limegroup.gnutella.gui.*;
import java.io.*;
import java.net.*;
import java.util.*;


public class AqMain {
    
    /* Instances */
    
    private static String javaBundlePath;
    protected static InputStreamReader reader = null;
    protected static OutputStreamWriter writer = null;
    
    /* State */
    
    protected static volatile boolean shouldShoutDown = false;
    
    /* Main */
    
    public static void main(String[] args) {
        
        javaBundlePath = args[0];
        if (javaBundlePath.startsWith("\""))
            javaBundlePath = javaBundlePath.substring(1,javaBundlePath.length() - 1);
        
        /* open streams */
        
        try {
            reader = new InputStreamReader(System.in, "UTF-8");
            writer = new OutputStreamWriter(System.out, "UTF-8");
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        /* make writable app support directories */
        
        if (CommonUtils.getUserHomeDir().exists())
            FileUtils.setWriteable(CommonUtils.getUserHomeDir());
        if (CommonUtils.getUserSettingsDir().exists())
            FileUtils.setWriteable(CommonUtils.getUserSettingsDir());
        
        /* set error callback */
        
        AqEventHandler handler = new AqEventHandler();
        ErrorService.setErrorCallback(handler);
        MessageService.setCallback(handler);

        /* start UPnP */
        
        if (CommonUtils.isJava14OrLater() && !ConnectionSettings.DISABLE_UPNP.getValue())
            UPnPManager.instance().start();
        
        /* start LW */
        
        RouterService router = new RouterService(handler);
        router.preGuiInit();
        router.start();
        
        /* start DAAP */
        
        if (CommonUtils.isJava14OrLater() && DaapSettings.DAAP_ENABLED.getValue() && 
            !DaapSettings.DAAP_SERVICE_NAME.getValue().equals("")) {
            try {
                DaapManager.instance().start();
                DaapManager.instance().init();
            } catch (IOException e) {
                DaapSettings.DAAP_ENABLED.setValue(false);
                DaapManager.instance().stop();
                e.printStackTrace();
            }
        }
        
        /* schedule threads */
        
        RouterService.schedule(new ConnectionUpdate(), 60 * 1000, 60 * 1000);
        RouterService.schedule(new TransferUpdate(), 1 * 1000, 1 * 1000);
                
        /* read loop */
        
        AqEvent.signalEvent(AqEvent.kLWEventCoreSettingRequest);
        readLoop();
        shutdown();
        
    }
    
    private static void readLoop() {
        
        int b;
        List args = new LinkedList();
        StringBuffer fragment = new StringBuffer();
        
        try {
            
            while (!shouldShoutDown && ((b = reader.read()) != -1)) {
                char c = (char)b;
                
                if (c == '|' || c == '\n') {
                    args.add(fragment.toString());
                    fragment.setLength(0);
                    
                    if (c == '\n') {
                        try {
                            AqDispatcher.dispatchCommand(args);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                        args.clear();
                    }
                    
                } else {
                    fragment.append(c);
                }
            }
            
        } catch (IOException e) {
            e.printStackTrace();
        }
        
    }
    
    public static void shutdown() {
        if (shouldShoutDown) return;
        shouldShoutDown = true;
        
        if (CommonUtils.isJava14OrLater() && !ConnectionSettings.DISABLE_UPNP.getValue())
            UPnPManager.instance().stop();
        if (CommonUtils.isJava14OrLater() && DaapSettings.DAAP_ENABLED.getValue())
            DaapManager.instance().stop();
        
        RouterService.shutdown();
        AqEvent.signalEvent(AqEvent.kLWEventCoreFinalized);
        
        /* close streams */
        
        try {
            if (reader != null) reader.close();
            if (writer != null) writer.close();
        } catch (IOException e) {
            e.printStackTrace();
        }

        System.exit(0);
    }
    
    public static String getJavaBundlePath() {
        return javaBundlePath;
    }
    
}


/* ConnectionUpdate */

class ConnectionUpdate implements Runnable {
    public void run() {
        AqEvent.signalEvent(AqEvent.kLWEventConnectionsUpdated);
    }
}


/* TransferUpdate */

class TransferUpdate implements Runnable {
    public void run() {
        
        /* Downloads */
        
        try {
            for (Iterator i = RouterService.getDownloadManager().getWaitingDownloads().iterator(); i.hasNext();
                 publishUpdate((ManagedDownloader)i.next()));
            for (Iterator i = RouterService.getDownloadManager().getActiveDownloads().iterator(); i.hasNext();
                 publishUpdate((ManagedDownloader)i.next()));
        } catch (Exception e) {}
        AqEvent.signalEvent(AqEvent.kLWEventDownloadsUpdated);
        
        /* Uploads */
        
        try {
            for (Iterator i = RouterService.getUploadManager().getActiveUploads().iterator(); i.hasNext();
                 publishUpdate((Uploader)i.next()));
        } catch (Exception e) {}
        AqEvent.signalEvent(AqEvent.kLWEventUploadsUpdated);
    
    }
    
    private void publishUpdate(ManagedDownloader d) {
        if (d != null)
            AqEvent.signalEvent(AqEvent.kLWEventUpdateDownloadStats, d);
    }
    
    private void publishUpdate(Uploader u) {
        if (u != null)
            AqEvent.signalEvent(AqEvent.kLWEventUpdateUploadStats, u);
    }
}
