/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitLogMessage.c,v 1.3 2003/09/13 05:28:03 tsutsumi Exp $
 */

/** @file
 * Log message output module.
 */

#include <ccunit/CCUnitLogMessage.h>
#include <stdio.h>
#include <stdarg.h>

/**
 * @addtogroup CCUnitLogMessage
 * Output messages.
 * @{
 */

/**
 * debug message flag.
 */
bool _ccunit_debug_message = false;

/**
 * verbose message flag.
 */
bool _ccunit_verbose_message = false;

/**
 * return stack growing direction.
 *
 * @param caller stack variable address of caller function.
 *
 * @return 1 when stack growing up, -1 when stack growing down.
 */
static int stackdir (const int* caller)
{
  auto const int stack = 0;
  return caller < &stack ? 1 : -1;
}

/**
 * count msg function call nestings
 *
 * @return 1 when function nested, -1 when function returned, 0 when
 * function is same level.
 */
static int nestings ()
{
  auto const int stack = 0;
  static const int* previous = NULL;
  static int dir = 0;
  int nest = 0;
  if (!dir)
    dir = stackdir (&stack);
  if (!previous)
    nest = 0;
  else if (previous < &stack)
    nest = dir;
  else if (previous > &stack)
    nest = -dir;
  else
    nest = 0;
  previous = &stack;
  return nest;
}

/**
 * print message.
 *
 * @param prompt message prompt string.
 * @param fmt message format string for printf.
 * @param args message arguments.
 */
static void msg (const char* prompt, const char* fmt, va_list args)
{
  static int leader = 1;
  const int inc = 2;
  leader += nestings () * inc;
  fprintf (stderr, "%s:%*c", prompt, leader, ' ');
  vfprintf (stderr, fmt, args);
  fputc ('\n', stderr);
}

inline void ccunit_dbg (const char* fmt, ...)
{
  if (_ccunit_debug_message)
    {
      va_list args;
      va_start (args, fmt);
      msg ("DBG", fmt, args);
      va_end (args);
    }
}

inline void ccunit_log (const char* fmt, ...)
{
  if (_ccunit_verbose_message || _ccunit_debug_message)
    {
      va_list args;
      va_start (args, fmt);
      msg ("LOG", fmt, args);
      va_end (args);
    }
}

inline void ccunit_err (const char* fmt, ...)
{
  va_list args;
  va_start (args, fmt);
  msg ("ERR", fmt, args);
  va_end (args);
}

inline void ccunit_msg (const char* fmt, ...)
{
  va_list args;
  va_start (args, fmt);
  msg ("CCUNIT", fmt, args);
  va_end (args);
}

/** @} */
