/* Copyright (C) 2003, 2010 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestFixture.c,v 1.12 2010/08/19 10:19:46 tsutsumi Exp $
 */

/**
 * @file
 * TestFixture module implementation.
 */

#include <ccunit/CCUnitTestFixture.h>
#include <ccunit/CCUnitTestResult.h>
#include <assert.h>
#include <setjmp.h>

/**
 * run tests exception environment.
 */
jmp_buf _ccunit_runTest_env;

/**
 * run tests exception.
 */
CCUnitTestFailure* _ccunit_testFailure;

extern void _ccunit_startTest (CCUnitTestResult* result,
			       struct CCUnitTestCase* test);
extern void _ccunit_endTest (CCUnitTestResult* result,
			     struct CCUnitTestCase* test);

inline void ccunit_addTestCase (CCUnitTestFixture* fixture,
				CCUnitTestCase* testCase)
{
  ccunit_addList (&fixture->testCases, testCase);
}

CCUnitTestCase* ccunit_addNewTestCase (CCUnitTestFixture* fixture,
				       const char* name,
				       const char* desc,
				       void (*runTest)())
{
  CCUnitTestCase* c;
  if (!fixture)
    return NULL;
  c = ccunit_newTestCase (name, desc, runTest);
  if (!c)
    return NULL;
  ccunit_addTestCase (fixture, c);
  return c;
}

/**
 * Runs the bare test sequence.
 *
 * @return failure
 */
static CCUnitTestFailure* runTest (CCUnitTestCase* testCase,
				   CCUnitTestFunc* setUp,
				   CCUnitTestFunc* tearDown,
				   CCUnitTestResult* result)
{
  CCUnitTestFailure* runFailure = NULL;
  _ccunit_testFailure = NULL;
  if (setUp && setUp->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	setUp->runTest ();
      else
	{
	  _ccunit_testFailure->testCase = setUp;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
  if (_ccunit_testFailure)
    ;
  else if (setjmp (_ccunit_runTest_env) == 0)
    {
      result->runCount ++;
      testCase->runTest ();
    }
  else
    {
      runFailure = _ccunit_testFailure;
      runFailure->testCase = testCase;
      _ccunit_testFailure = NULL;
    }
  if (tearDown && tearDown->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	tearDown->runTest ();
      else
	{
	  _ccunit_testFailure->testCase = tearDown;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
  return runFailure;
}

/**
 * Runs the test fixture and collects the results in CCUnitTestResult.
 * @param test A test to run.
 * @param result A result container.
 */
static void run (CCUnitTest* test, CCUnitTestResult* result)
{
  CCUnitTestFixture* fixture = (CCUnitTestFixture*)test;
  CCUnitListIterator itor;
  CCUnitTestCase* testCase;
  assert (test->type == ccunitTypeFixture);
  _ccunit_testFailure = NULL;
  if (fixture->setup_setUp && fixture->setup_setUp->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	fixture->setup_setUp->runTest ();
      else
	{
	  _ccunit_testFailure->testCase = fixture->setup_setUp;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
  if (!_ccunit_testFailure)
    {
      ccunit_initListIterator (&fixture->testCases, &itor);
      while ((testCase = ccunit_nextListIterator (&itor)) != NULL)
	{
	  CCUnitTestFailure* failure;
	  _ccunit_startTest (result, testCase);
	  failure = runTest (testCase, fixture->setUp, fixture->tearDown, result);
	  if (failure)
	    ccunit_addFailure (result, failure);
	  _ccunit_endTest (result, testCase);
	}
    }
  if (fixture->setup_tearDown && fixture->setup_tearDown->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	fixture->setup_tearDown->runTest ();
      else
	{
	  _ccunit_testFailure->testCase = fixture->setup_tearDown;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
}

/**
 * Destruct test fixture.
 * @param test destruct test.
 */
static void destroy (CCUnitTest* test)
{
  CCUnitTestFixture* fixture;
  assert (test->type == ccunitTypeFixture);
  fixture = (CCUnitTestFixture*)test;
  safe_free (fixture->name);
  ccunit_deleteTestFunc (fixture->setUp);
  ccunit_deleteTestFunc (fixture->tearDown);
  ccunit_deleteList (&fixture->testCases,
		     (void (*)(void*))ccunit_deleteTestCase);
}

CCUnitTestFixture* ccunit_newTestFixture (const char* name,
					  CCUnitTestFunc* setUp,
					  CCUnitTestFunc* tearDown)
{
  CCUnitTestFixture* fixture = calloc (1, sizeof (*fixture));
  ccunit_initTest (&fixture->test, ccunitTypeFixture, run, destroy);
  fixture->name = safe_strdup (name);
  ccunit_initList (&fixture->testCases);
  fixture->setUp = setUp;
  fixture->tearDown = tearDown;
  return fixture;
}

inline void ccunit_setTestFixtureSetup (CCUnitTestFixture* fixture,
					CCUnitTestFunc* setup_setUp,
					CCUnitTestFunc* setup_tearDown)
{
  if (!fixture)
    return;
  if (fixture->setup_setUp)
    ccunit_deleteTestCase (fixture->setup_setUp);
  fixture->setup_setUp = setup_setUp;
  if (fixture->setup_tearDown)
    ccunit_deleteTestCase (fixture->setup_tearDown);
  fixture->setup_tearDown = setup_tearDown;
}

inline struct CCUnitTestResult* ccunit_runTestFixture (CCUnitTestFixture* f)
{
  CCUnitTestResult* result;
  if (!f)
    return NULL;
  result = ccunit_newTestResult ();
  if (!result)
    return NULL;
  f->test.run (&f->test, result);
  return result;
}
