/* -*- mode: C; -*- */
/* Copyright (C) 2003, 2010, 2013 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id$
 */
/**@file
 * Assert functions and macros.
 */
#ifndef CCUNITASSERT_H
#define CCUNITASSERT_H

#include <ccunit/CCUnitConfig.h>
#if CCUNIT_HAVE_SYS_TYPES_H
#  include <sys/types.h>
#endif

/**
 * @ingroup ModuleHierarchy
 * @defgroup CCUnitAssert Assert
 *
 * A set of assert functions and macros.  Messages are only displayed
 * when an assert fails.
 *
 * @{
 */

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param file file name.
 * @param line line number.
 * @param condition assert condition.
 * @param condstr assert condition as string.
 *
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
extern void ccunit_assert (const char* file, unsigned int line,
			   bool condition, const char* condstr);

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param COND assert condition.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT(COND) ccunit_assert (__FILE__, __LINE__, COND, #COND)

/**
 * @name Assert Macros
 * The assert test macros.
 * @{
 */
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param TYP type of expect/actual value.
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST(TYP, EXPECT, OP, ACTUAL)			\
  do {									\
    const TYP ex = (EXPECT);						\
    const TYP ac = (ACTUAL);						\
    ccunit_assert_test_ ## TYP (__FILE__, __LINE__, (ex OP ac),		\
				(#EXPECT " " #OP " " #ACTUAL),		\
				ex, ac);				\
  } while (0)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_INT(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (int, EXPECT, OP, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_UINT(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (u_int, EXPECT, OP, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_LONG(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (long, EXPECT, OP, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_ULONG(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (u_long, EXPECT, OP, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_SHORT(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (short, EXPECT, OP, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_USHORT(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (u_short, EXPECT, OP, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_DOUBLE(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (double, EXPECT, OP, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_FLOAT(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (float, EXPECT, OP, ACTUAL)
#if CCUNIT_HAVE_QUAD_T
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_QUAD(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (quad_t, EXPECT, OP, ACTUAL)
#endif
#if CCUNIT_HAVE_U_QUAD_T
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_UQUAD(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (u_quad_t, EXPECT, OP, ACTUAL)
#endif
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_PTR(EXPECT, OP, ACTUAL)		\
  CCUNIT_ASSERT_TEST (_ccunit_ptr_t, EXPECT, OP, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_STR(EXPECT, OP, ACTUAL)			\
  do {									\
    const _ccunit_str_t ex = (EXPECT);					\
    const _ccunit_str_t ac = (ACTUAL);					\
    const char* condstr = "strcmp (" #EXPECT ", " #ACTUAL ") " #OP " 0"; \
    ccunit_assert_test__ccunit_str_t (__FILE__, __LINE__,		\
				      strcmp(ex, ac) OP 0, condstr, ex, ac); \
  } while (0)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @param N length to compare.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_NSTR(EXPECT, OP, ACTUAL, N)			\
  do {									\
    const _ccunit_str_t ex = (EXPECT);					\
    const _ccunit_str_t ac = (ACTUAL);					\
    const size_t n = (N);						\
    ccunit_assert_test__ccunit_str_t (__FILE__, __LINE__,		\
				      strncmp(ex, ac, n) OP 0,		\
				      "strncmp (" #EXPECT ", " #ACTUAL	\
				      ", " #N ") " #OP " 0",		\
				      ex, ac);				\
  } while (0)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_CASE_STR(EXPECT, OP, ACTUAL)			\
  do {									\
    const _ccunit_str_t ex = (EXPECT);					\
    const _ccunit_str_t ac = (ACTUAL);					\
    const size_t n = (N);						\
    ccunit_assert_test__ccunit_str_t (__FILE__, __LINE__,		\
				      strcasecmp(ex, ac) OP 0,		\
				      "strcasecmp (" #EXPECT ", " #ACTUAL \
				      ") " #OP " 0",			\
				      ex, ac);				\
  } while (0)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @param N length to compare.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_CASE_NSTR(EXPECT, OP, ACTUAL, N)		\
  do {									\
    const _ccunit_str_t ex = (EXPECT);					\
    const _ccunit_str_t ac = (ACTUAL);					\
    const size_t n = (N);						\
    ccunit_assert_test__ccunit_str_t (__FILE__, __LINE__,		\
				      strncasecmp(ex, ac, n) OP 0,	\
				      "strncasecmp (" #EXPECT ", " #ACTUAL \
				      ", " #N ") " #OP " 0",		\
				      ex, ac);				\
  } while (0)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param EXPECT expect value.
 * @param OP test operator.
 * @param ACTUAL actual value.
 * @param TOSTR pointer to function of converting to string.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define CCUNIT_ASSERT_TEST_OBJ(EXPECT, OP, ACTUAL, TOSTR)		\
  do {									\
    const _ccunit_obj_t ex = (EXPECT);					\
    const _ccunit_obj_t ac = (ACTUAL);					\
    const char* condstr = #OP " (" #EXPECT ", " #ACTUAL ")";		\
    ccunit_assert_test__ccunit_obj_t (__FILE__, __LINE__,		\
				      OP(ex, ac), condstr, ex, ac,	\
				      (ccunit_toString_t)TOSTR);	\
  } while (0)

/** @} */

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param TYP type of expect/actual value.
 * @param CONDITION test condition.
 * @param EXPECT expect value.
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 *
 * @note This macro has the possibility that a side effect happens in
 * the argument EXPECT and ACTUAL.
 */
#define CCUNIT_ASSERT_COND(TYP, CONDITION, EXPECT, ACTUAL)		\
  do {									\
    const TYP ex = (EXPECT);						\
    const TYP ac = (ACTUAL);						\
    ccunit_assert_test_ ## TYP (__FILE__, __LINE__, (CONDITION),	\
				(#CONDITION), ex, ac);			\
  } while (0)

/**
 * @name Assert Condition Macros
 * The assert condition macros.
 * @{
 */
/**
 * Asserts that a pointer is NULL. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual pointer value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NULL(ACTUAL)		\
  CCUNIT_ASSERT_TEST_PTR(NULL, ==, ACTUAL)

/**
 * Asserts that a pointer is not NULL. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NOT_NULL(ACTUAL)		\
  CCUNIT_ASSERT_TEST_PTR(NULL, !=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_INT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_INT (EXPECT, ==, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_INT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_INT (EXPECT, !=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_INT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_INT (EXPECT, <, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_INT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_INT (EXPECT, <=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_INT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_INT (EXPECT, >, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_INT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_INT (EXPECT, >=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_UINT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UINT (EXPECT, ==, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_UINT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UINT (EXPECT, !=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_UINT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UINT (EXPECT, <, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_UINT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UINT (EXPECT, <=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_UINT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UINT (EXPECT, >, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_UINT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UINT (EXPECT, >=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_DOUBLE(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_DOUBLE (EXPECT, ==, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_DOUBLE(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_DOUBLE (EXPECT, !=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_DOUBLE(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_DOUBLE (EXPECT, <, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_DOUBLE(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_DOUBLE (EXPECT, <=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_DOUBLE(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_DOUBLE (EXPECT, >, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_DOUBLE(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_DOUBLE (EXPECT, >=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_FLOAT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_FLOAT (EXPECT, ==, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_FLOAT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_FLOAT (EXPECT, !=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_FLOAT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_FLOAT (EXPECT, <, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_FLOAT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_FLOAT (EXPECT, <=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_FLOAT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_FLOAT (EXPECT, >, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_FLOAT(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_FLOAT (EXPECT, >=, ACTUAL)

#if CCUNIT_HAVE_QUAD_T
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_QUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_QUAD (EXPECT, ==, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_QUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_QUAD (EXPECT, !=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_QUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_QUAD (EXPECT, <, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_QUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_QUAD (EXPECT, <=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_QUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_QUAD (EXPECT, >, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_QUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_QUAD (EXPECT, >=, ACTUAL)
#endif

#if CCUNIT_HAVE_U_QUAD_T
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_UQUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UQUAD (EXPECT, ==, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_UQUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UQUAD (EXPECT, !=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_UQUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UQUAD (EXPECT, <, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_UQUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UQUAD (EXPECT, <=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_UQUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UQUAD (EXPECT, >, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_UQUAD(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_UQUAD (EXPECT, >=, ACTUAL)
#endif

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_STR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_STR (EXPECT, ==, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_STR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_STR (EXPECT, !=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_STR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_STR (EXPECT, <, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_STR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_STR (EXPECT, <=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_STR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_STR (EXPECT, >, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_STR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_STR (EXPECT, >=, ACTUAL)

/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_EQ_PTR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_PTR (EXPECT, ==, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_NE_PTR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_PTR (EXPECT, !=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LT_PTR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_PTR (EXPECT, <, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_LE_PTR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_PTR (EXPECT, <=, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GT_PTR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_PTR (EXPECT, >, ACTUAL)
/**
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param ACTUAL actual value.
 * @param EXPECT expect value.
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions
 */
#define CCUNIT_ASSERT_GE_PTR(EXPECT, ACTUAL)	\
  CCUNIT_ASSERT_TEST_PTR (EXPECT, >=, ACTUAL)

/** @} */

#ifndef CCUNIT_HAVE_U_CHAR
#define CCUNIT_HAVE_U_CHAR
typedef unsigned char u_char;			/**< u_char */
#endif
#ifndef CCUNIT_HAVE_U_INT
#define CCUNIT_HAVE_U_INT
typedef unsigned int u_int;			/**< u_int */
#endif
#ifndef CCUNIT_HAVE_U_SHORT
#define CCUNIT_HAVE_U_SHORT
typedef unsigned short u_short;			/**< u_short */
#endif
#ifndef CCUNIT_HAVE_U_LONG
#define CCUNIT_HAVE_U_LONG
typedef unsigned long u_long;			/**< u_long */
#endif
#ifndef CCUNIT_HAVE_QUAD_T
#ifdef CCUNIT_HAVE_INT64_T
#define CCUNIT_HAVE_QUAD_T 1
typedef int64_t quad_t;				/**< quad_t */
#else  /* !CCUNIT_HAVE_INT64_T */
#if CCUNIT_HAVE_LONG_LONG
#define CCUNIT_HAVE_QUAD_T 1
typedef long long quad_t;			/**< quad_t */
#endif	/* CCUNIT_HAVE_LONG_LONG */
#endif	/* CCUNIT_HAVE_INT64_T */
#endif	/* !CCUNIT_HAVE_QUAD_T */

#ifndef CCUNIT_HAVE_U_QUAD_T
#ifdef CCUNIT_HAVE_U_INT64_T
#define CCUNIT_HAVE_U_QUAD_T 1
typedef u_int64_t u_quad_t;			/**< u_quad_t */
#else  /* !CCUNIT_HAVE_U_INT64_T */
#ifdef CCUNIT_HAVE_LONG_LONG
#define CCUNIT_HAVE_U_QUAD_T 1
typedef unsigned long long u_quad_t;		/**< u_quad_t */
#endif	/* CCUNIT_HAVE_LONG_LONG */
#endif	/* CCUNIT_HAVE_U_INT64_T */
#endif	/* !CCUNIT_HAVE_U_QUAD_T */

typedef const char* _ccunit_str_t;		/**< string type */
typedef const void* _ccunit_ptr_t;		/**< pointer type */
typedef const void* _ccunit_obj_t;		/**< object type */

/*
 * Asserts that a condition is true. If it isn't it longjmp with
 * _ccunit_runTest_env.
 *
 * @param TYP test value type
 * @throws _ccunit_runTest_env If condition is false.
 * @ingroup Assertions 
 */
#define DCL_CCUNIT_ASSERT_TEST_TYPE(TYP)				\
  extern void ccunit_assert_test_ ## TYP (const char* file, unsigned int line, \
					  bool cond, const char* condstr, \
					  TYP expect, TYP acutal)

DCL_CCUNIT_ASSERT_TEST_TYPE(char);
DCL_CCUNIT_ASSERT_TEST_TYPE(u_char);
DCL_CCUNIT_ASSERT_TEST_TYPE(int);
DCL_CCUNIT_ASSERT_TEST_TYPE(u_int);
DCL_CCUNIT_ASSERT_TEST_TYPE(short);
DCL_CCUNIT_ASSERT_TEST_TYPE(u_short);
DCL_CCUNIT_ASSERT_TEST_TYPE(long);
DCL_CCUNIT_ASSERT_TEST_TYPE(u_long);
DCL_CCUNIT_ASSERT_TEST_TYPE(float);
DCL_CCUNIT_ASSERT_TEST_TYPE(double);
DCL_CCUNIT_ASSERT_TEST_TYPE(_ccunit_str_t);
DCL_CCUNIT_ASSERT_TEST_TYPE(_ccunit_ptr_t);
#if CCUNIT_HAVE_QUAD_T
DCL_CCUNIT_ASSERT_TEST_TYPE(quad_t);
#endif
#if CCUNIT_HAVE_U_QUAD_T
DCL_CCUNIT_ASSERT_TEST_TYPE(u_quad_t);
#endif

typedef char* (*ccunit_toString_t)(_ccunit_obj_t);

extern void ccunit_assert_test__ccunit_obj_t (const char* file, unsigned int line,
					      bool cond, const char* condstr,
					      _ccunit_obj_t expect,
					      _ccunit_obj_t acutal,
					      ccunit_toString_t);

/** @} */

#endif
