/* -*- mode: C; -*- */
/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id$
 */

/**
 * @file
 * Linked list module.
 */
#ifndef CCUNITLIST_H
#define CCUNITLIST_H

#include <ccunit/CCUnitConfig.h>

/**
 * @ingroup ModuleHierarchy
 * @defgroup CCUnitList List
 * Linked list container.
 * @{
 */

/**
 * list container.
 */
typedef struct CCUnitList
{
  struct CCUnitListCell* head;		/**< first cell of a linked list */
  struct CCUnitListCell** tailp;	/**< address of pointer to last cell*/
  size_t length;			/**< size of linked list */
  bool isAllocated;			/**< flag of allocated object */
} CCUnitList;

/**
 * Create new list object.
 * @return new list object.
 */
extern inline CCUnitList* ccunit_newList ();

/**
 * Add element to list object.
 * @param list List object.
 * @param contents A pointer to an object to register in the list.
 */
extern void ccunit_addList (CCUnitList* list, void* contents);

/**
 * Initialize list object.
 * @param list initializing list.
 * @return initialized list.
 */
extern CCUnitList* ccunit_initList (CCUnitList* list);

/**
 * Delete list object.
 *
 * @param list Deleting list.
 * @param deleteContents A pointer to the function which the object
 * registered in the list is deleted from.
 */
extern void ccunit_deleteList (CCUnitList* list, void (*deleteContents)(void*));

/**
 * @defgroup CCUnitListIterator ListIterator
 * List iterator
 * @{
 */

/**
 * list iterator.
 */
typedef struct CCUnitListIterator
{
  struct CCUnitListCell* current;		/**< current list cell  */
  bool isAllocated;				/**< flag of allocated object */
} CCUnitListIterator;

/**
 * create new list iterator.
 *
 * @param list An owner list of iterator.
 * @return New iterator.
 */
extern CCUnitListIterator* ccunit_newListIterator (const struct CCUnitList* list);

/**
 * initialize list iterator.
 * @param list An owner list of iterator.
 * @param it Iterator to initialize.
 * @return Initialized iterator.
 */
extern inline
CCUnitListIterator* ccunit_initListIterator (const struct CCUnitList* list,
					     struct CCUnitListIterator* it);

/**
 * delete list iterator.
 *
 * @param it iterator to delete.
 */
extern inline void ccunit_deleteListIterator (struct CCUnitListIterator* it);

/**
 * Get next element.
 * @param it An Iterator.
 * @return pointer to next element object.
 */
extern void* ccunit_nextListIterator (struct CCUnitListIterator* it);

/**
 * Check is there next element of iterator.
 * @param it An iterator.
 * @return true if there is next element, or false.
 */
extern bool ccunit_hasNextListIterator (struct CCUnitListIterator* it);

/** @}
 * end group CCUnitListIterator
 */

/** @}
 * end group CCUnitList
 */

#endif

