/* -*- mode: C; -*- */
/* Copyright (C) 2003, 2010, 2013 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id$
 */

/**
 * @file
 * TestCase module.
 */
#ifndef CCUNITTESTCASE_H
#define CCUNITTESTCASE_H

#include <ccunit/CCUnitConfig.h>
#include <ccunit/CCUnitList.h>
#include <ccunit/CCUnitTest.h>
#include <ccunit/CCUnitTestFunc.h>
#include <ccunit/CCUnitTestResult.h>

/**
 * @ingroup CCUnitTest
 * @defgroup CCUnitTestCase TestCase
 * A test case defines the case to run multiple tests.
 *
 * @{
 */

/**
 * Wraps a test case with setUp and tearDown function.
 *
 * A TestCase is used to provide a common environment for a set
 * of test cases.
 *
 * To define a test case, do the following:
 * - the case is defined by static variables
 * - initialize the case state by setUp function
 * - clean-up after a test by tearDown function
 *
 * Each test runs in its own case so there
 * can be no side effects among test runs.
 * Here is an example:
 * 
 * @code
 * static int value1, value2;
 *
 * void setUp_MathTest ()
 * {
 *   value1 = 2;
 *   value2 = 3;
 * }
 *
 * ...
 *
 * CCUnitTestCase* MathTest_newTestCase ()
 * {
 *   return ccunit_newTestCase ("MathTest", setUp_MathTest, NULL);
 * }
 * @endcode
 *
 * For each test implement a function which interacts with the
 * case. Verify the expected results with assertions specified by
 * calling CCUNIT_ASSERT on the expression you want to test:
 * 
 * @code
 * void testAdd ()
 * {
 *   int result = value1 + value2;
 *   CCUNIT_ASSERT (result == 5);
 * }
 *
 * ...
 *
 * void MathTest_newTestCase_testAdd ()
 * {
 *   return ccunit_newTestCase ("testAdd", "add test", testAdd);
 * }
 * @endcode
 * 
 * The tests to be run can be collected into a TestSuite. 
 * 
 * @code
 * CCUintTestSuite* MathTest_suite ()
 * {
 *   CCUnitTestSuite* suite = ccunit_newTestSuite ("MathTest");
 *   CCUnitTestCase* tcase = MathTest_newTestCase ();
 *   ccunit_addTestCase (suite, tcase);
 *   ccunit_addTestCase (tcase, MathTest_newTestCase_testAdd ());
 *   ccunit_addTestCase (tcase, MathTest_newTestCase_testDivZero ())
 *   return suite;
 * }
 * @endcode
 * 
 * Once the functions are defined you can run them. To do this, use a
 * TestRunner.
 *
 * @code
 *   CCUnitTestRunner *runner = ccunit_newTestRunner (stdout);
 *   CCUnitTestSuite *suite = MathTest_suite ();
 *   runner->run (runner, suite);
 * @endcode
 *
 * A command line tool have been created for convenience. It is located
 * in src/tools/ccunit_makeSuite.c.
 *
 * @see CCUnitTestResult, CCUnitTestCase, CCUnitTestSuite, CCUnitMakeSuite,
 *
 * @ingroup WritingTestCase
 */
typedef struct CCUnitTestCase
{
  CCUnitTest test;				/**< super class */
  const char* name;				/**< test case name */
  CCUnitTestFunc* setUpBeforeClass;		/**< setUp for a case */
  CCUnitTestFunc* tearDownAfterClass;		/**< tearDown for a case */
  CCUnitTestFunc* setUp;			/**< setUp for each tests */
  CCUnitTestFunc* tearDown;			/**< tearDown for each tests */
  CCUnitList testFuncs;				/**< test functions */
} CCUnitTestCase;

/**
 * TestCase definition structure
 */
typedef struct CCUnitTestCaseDfn
{
  CCUnitTestDfn test;				/**< super class */
  const char* name;				/**< test case name */
  CCUnitTestFunc* testFuncs;			/**< test functions */
} CCUnitTestCaseDfn;

/**
 * create new test case.
 *
 * @param name test case name.
 * @return new test case.
 * @ingroup WritingTestCase
 */
extern CCUnitTestCase* ccunit_newTestCase (const char* name);

/**
 * Destructs test case.
 * @param testCase deleting case.
 * @ingroup WritingTestCase
 */
extern inline void ccunit_deleteTestCase (CCUnitTestCase* testCase);

/**
 * add test function to test case.
 *
 * @param testCase test case.
 * @param f test function.
 * @ingroup WritingTestCase
 */
extern void ccunit_addTestFunc (CCUnitTestCase* testCase, CCUnitTestFunc* f);

/**
 * add new test func to test case.
 *
 * @param testCase test case.
 * @param name test case name.
 * @param desc test case description.
 * @param runTest run test function.
 * @return new test func
 * @ingroup WritingTestCase
 */
extern CCUnitTestFunc* ccunit_addNewTestFunc (CCUnitTestCase* testCase,
					      const char* name,
					      const char* desc,
					      void (*runTest)());
#define CCUNIT_ADDNEWTESTFUNC(TESTCASE, FN) ccunit_addNewTestFunc (TESTCASE, #FN, NULL, FN)

/**
 * run test cases and collect its results.
 * @param testCase test case.
 * @return test result.
 * @ingroup ExecutingTest
 */
extern inline struct CCUnitTestResult* ccunit_runTestCase (CCUnitTestCase* testCase);

/** @} */
#endif
