/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestRunner.c,v 1.9 2003/10/01 20:52:59 tsutsumi Exp $
 */
/**@file
 * TestRunner module implementation.
 */
#include <stdio.h>
#include <ccunit/CCUnitTestRunner.h>

#if CCUNIT_TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif

/** @addtogroup CCUnitTestRunner
 * @{
 */

/**
 * receive startTest notification.
 * @param listener notification listener.
 * @param test starting test case
 */
static void startTest (CCUnitTestListener* listener, CCUnitTestCase* test)
{
  CCUnitTestRunner* runner = (CCUnitTestRunner*)listener;
  fputc ('.', runner->ofp);
}

/**
 * receive endTest notification.
 * @param listener notification listener.
 * @param test ended test case
 */
static void endTest (CCUnitTestListener* listener, CCUnitTestCase* test)
{
}

/**
 * receive addFailure notification.
 * @param listener notification listener.
 * @param failure failure information.
 */
static void addFailure (CCUnitTestListener* listener, CCUnitTestFailure* failure)
{
  CCUnitTestRunner* runner = (CCUnitTestRunner*)listener;
  fputc ('F', runner->ofp);
}

/**
 * Prints the header of the report.
 * @param runner test runner.
 * @param result test result.
 */
static void printHeader (CCUnitTestRunner* runner, CCUnitTestResult* result)
{
  FILE* ofp;
  if (!runner)
    return;
  ofp = runner->ofp;
  if (ccunit_wasSuccessful (result))
    {
      fputc ('\n', ofp);
      fprintf (ofp, "OK (%lu tests)\n",
	       (unsigned long)ccunit_runCount (result));
    }
  else
    {
      fputc ('\n', ofp);
      fprintf (ofp,
	       "FAILURES!!!\n"
	       "Test Results: \n"
	       "Run %lu, Failures %lu\n",
	       (unsigned long)ccunit_runCount (result),
	       (unsigned long)ccunit_failureCount (result));
    }
}

/**
 * Prints failures.
 * @param runner test runner.
 * @param result test result.
 */
static void printFailures (CCUnitTestRunner* runner, CCUnitTestResult* result)
{
  CCUnitListIterator* itor;
  CCUnitTestFailure* f;
  unsigned long num;
  if (!runner || !result)
    return;
  num = ccunit_failureCount (result);
  if (!num)
    return;
  if (num == 1)
    fprintf (runner->ofp, "There was %lu failure:\n", num);
  else
    fprintf (runner->ofp, "There was %lu failures:\n", num);
  itor = ccunit_failures (result);
  while ((f = ccunit_nextListIterator (itor)) != NULL)
    {
      fprintf (runner->ofp, "%s:%u: %s:\n\t%s\n",
	       f->file, f->line,
	       f->testCase != NULL ? f->testCase->desc : "",
	       f->condstr);
      if (f->expect || f->actual)
	fprintf (runner->ofp, "\texpect: %s\n\tactual: %s\n", f->expect, f->actual);
    }
  ccunit_deleteListIterator (itor);
}

/**
 * Prints failures.
 * @param runner test runner
 * @param result test result
 */
static inline void print (CCUnitTestRunner* runner, CCUnitTestResult* result)
{
  printHeader (runner, result);
  printFailures (runner, result);
}

/**
 * Runs a single test and collects its results.
 * @param runner test runner
 * @param suite test suite
 * @return 0 if succeeded. -1 if error occurd.
 */
static int doRun (CCUnitTestRunner* runner, CCUnitTestSuite* suite)
{
  struct timeval startTime;
  struct timeval endTime;
  struct timeval runTime;
  if (!runner || !suite)
    return -1;
  runner->result = ccunit_newTestResult ();
  if (!runner->result)
    return -1;
  ccunit_addResultListener (runner->result, &runner->listener);
  timerclear (&startTime);
  timerclear (&endTime);
  timerclear (&runTime);
  gettimeofday (&startTime, NULL);
  ccunit_runTestSuite (suite, runner->result);
  gettimeofday (&endTime, NULL);
  timersub (&endTime, &startTime, &runTime);
  fprintf (runner->ofp,
	   "\n"
	   "Time: %lu.%06lu sec\n",
	   (unsigned long)runTime.tv_sec, (unsigned long)runTime.tv_usec);
  print (runner, runner->result);
  fputc ('\n', runner->ofp);
  return ccunit_wasSuccessful (runner->result) ? 0 : -1;
}

CCUnitTestRunner* ccunit_newTestRunner (FILE* output)
{
  CCUnitTestRunner* runner = calloc (1, sizeof (CCUnitTestRunner));
  if (!runner)
    return NULL;
  runner->listener.startTest = startTest;
  runner->listener.endTest = endTest;
  runner->listener.addFailure = addFailure;
  runner->ofp = !output ? stdout : output;
  runner->run = doRun;
  return runner;
}

void ccunit_deleteTestRunner (CCUnitTestRunner* runner)
{
  if (!runner)
    return;
  ccunit_deleteTestResult (runner->result);
  free (runner);
}

inline int ccunit_runTestRunner (CCUnitTestRunner* runner, CCUnitTestSuite* suite)
{
  return runner->run (runner, suite);
}

inline CCUnitTestResult* ccunit_getResult (CCUnitTestRunner* runner)
{
  return runner->result;
}

/** @} */
