#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <gsl/gsl_rng.h>
#include <cdescent.h>

double		thres = 0.5;
gsl_rng		*__rng__ = NULL;

static mm_sparse *
mm_real_create_sparse_symm (MMRealSymm symm, const int n)
{
	int			i, j, k;
	int			nnz = n * n;
	mm_sparse	*s = mm_real_new (MM_REAL_SPARSE, symm, n, n, nnz); 

	k = 0;
	for (j = 0; j < n; j++) {
		int	i0 = (mm_real_is_upper (s)) ? 0 : j;
		int	i1 = (mm_real_is_upper (s)) ? j : n - 1;
		for (i = i0; i <= i1; i++) {
			double	r = gsl_rng_uniform (__rng__);
			if (r < thres) continue;
			s->i[k] = i;
			s->data[k++] = 1. - 2. * gsl_rng_uniform (__rng__);
		}
		s->p[j + 1] = k;
	}
	if (nnz != k) mm_real_realloc (s, k);
	return s;
}

static mm_dense *
mm_real_create_dense_symm (MMRealSymm symm, const int n)
{
	int			j;
	int			nnz = n * n;
	mm_dense	*d = mm_real_new (MM_REAL_DENSE, symm, n, n, nnz); 

#pragma omp parallel for
	for (j = 0; j < n; j++) {
		int	i;
		int	i0 = (mm_real_is_upper (d)) ? 0 : j;
		int	i1 = (mm_real_is_upper (d)) ? j : n - 1;
		for (i = i0; i <= i1; i++) d->data[i + j * d->m] = 1. - 2. * gsl_rng_uniform (__rng__);
	}
	return d;
}

static mm_sparse *
mm_real_create_sparse (const int m, const int n)
{
	int			i, j, k;
	int			nnz = m * n;
	mm_sparse	*s = mm_real_new (MM_REAL_SPARSE, MM_REAL_GENERAL, m, n, nnz);

	k = 0;
	for (j = 0; j < n; j++) {
		for (i = 0; i < m; i++) {
			double	r = gsl_rng_uniform (__rng__);
			if (r < thres) continue;
			s->i[k] = i;
			s->data[k++] = 1. - 2. * gsl_rng_uniform (__rng__);
		}
		s->p[j + 1] = k;
	}
	if (nnz != k) mm_real_realloc (s, k);
	return s;
}

static mm_dense *
mm_real_create_dense (const int m, const int n)
{
	int			i;
	int			nnz = m * n;
	mm_dense	*d = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, m, n, nnz); 
#pragma omp parallel for
	for (i = 0; i < nnz; i++) d->data[i] = 1. - 2. * gsl_rng_uniform (__rng__);
	return d;
}

mm_real *
mm_real_create (gsl_rng *rng, MMRealFormat format, MMRealSymm symm, const int m, const int n)
{
	mm_real	*a;

	__rng__ = rng;

	if (format == MM_REAL_SPARSE) {
		a = (symm == MM_REAL_GENERAL) ? mm_real_create_sparse (m, n) : mm_real_create_sparse_symm (symm, m);
	} else {
		a = (symm == MM_REAL_GENERAL) ? mm_real_create_dense (m, n) : mm_real_create_dense_symm (symm, m);
	}
	return a;
}


static void
mm_real_fprintf_dense_0 (FILE *stream, const mm_dense *d, const char *format)
{
	int		i, j, k = 0;
	for (i = 0; i < d->m; i++) {
		for (j = 0; j < d->n; j++) {
			fprintf (stream, format, d->data[i + j * d->m]);
			if (j <= d->n - 1) fprintf (stream, " ");
		}
		fprintf (stream, "\n");
	}
	return;
}

static void
mm_real_fprintf_sparse (FILE *stream, const mm_sparse *x, const char *format)
{
	mm_dense	*d = mm_real_copy_sparse_to_dense (x);
	if (mm_real_is_symmetric (d)) mm_real_symmetric_to_general (d);
	mm_real_fprintf_dense_0 (stream, d, format);
	mm_real_free (d);
	return;
}

static void
mm_real_fprintf_dense (FILE *stream, const mm_dense *x, const char *format)
{
	mm_dense	*d = mm_real_copy (x);
	if (mm_real_is_symmetric (d)) mm_real_symmetric_to_general (d);
	mm_real_fprintf_dense_0 (stream, d, format);
	mm_real_free (d);
	return;
}

void
mm_real_fprintf (FILE *stream, const mm_real *x, const char *format)
{
	(mm_real_is_sparse (x)) ? mm_real_fprintf_sparse (stream, x, format) : mm_real_fprintf_dense (stream, x, format);
	return;
}

