#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <gsl/gsl_rng.h>
#include <gsl/gsl_randist.h>

#include <cdescent.h>

#include "mmreal_test.h"

const int	ione = 1;

static void
mm_real_copy_sparse_0 (const mm_sparse *src, mm_sparse *dest)
{
	int			k;
	int			n = src->n;
	int			nnz = src->nnz;

	int			*si = src->i;
	int			*sp = src->p;
	int			*di = dest->i;
	int			*dp = dest->p;

	for (k = 0; k < nnz; k++) di[k] = si[k];
	for (k = 0; k <= n; k++) dp[k] = sp[k];
	dcopy_ (&nnz, src->data, &ione, dest->data, &ione);

	return;
}

/*** set to general ***/
static void
mm_real_set_general (mm_real *x)
{
	if (!mm_real_is_symmetric (x)) return;
	mm_set_general (&(x->typecode));
	x->symm = MM_REAL_GENERAL;
	return;
}

/* binary search: search the value key from array *s of length n.
 * if found, return its index otherwise -1 */
static int
bin_search (const int key, const int *s, const int n)
{
	int	start = 0;
	int	end = n - 1;
	int	mid;
	while (start < end) {
		mid = (start + end) / 2;
		if (key <= s[mid]) end = mid;
		else start = mid + 1;
	}
	return (key == s[end]) ? end : -1;
}

/* find element that s->i[l] = j in the k-th column of s and return its index l */
static int
find_row_element (const int j, const mm_sparse *s, const int k)
{
	int		p = s->p[k];
	int		n = s->p[k + 1] - p;
	int		*si = s->i + p;
	int		res = bin_search (j, si, n);
	return (res < 0) ? -1 : res + p;
}

/* convert sparse symmetric -> sparse general */
static mm_sparse *
mm_real_symmetric_to_general_sparse (const mm_sparse *x)
{
	int			j, m;
	mm_sparse	*s;

	int			*si;
	int			*sp;
	double		*sd;
	int			xn = x->n;
	int			*xi = x->i;
	int			*xp = x->p;
	double		*xd = x->data;

	if (!mm_real_is_symmetric (x)) return mm_real_copy (x);
	s = mm_real_new (MM_REAL_SPARSE, MM_REAL_GENERAL, x->m, x->n, 2 * x->nnz);

	si = s->i;
	sp = s->p;
	sd = s->data;

	m = 0;
	for (j = 0; j < xn; j++) {
		int		k;
		int		pend = xp[j + 1];
		if (mm_real_is_upper (x)) {
			for (k = xp[j]; k < pend; k++) {
				si[m] = xi[k];
				sd[m++] = xd[k];
			}
			for (k = j + 1; k < xn; k++) {
				int		l = find_row_element (j, x, k);
				// if found
				if (l >= 0) {
					si[m] = k;
					sd[m++] = xd[l];
				}
			}
		} else if (mm_real_is_lower (x)) {
			for (k = 0; k < j; k++) {
				int		l = find_row_element (j, x, k);
				// if found
				if (l >= 0) {
					si[m] = k;
					sd[m++] = xd[l];
				}
			}
			for (k = xp[j]; k < pend; k++) {
				si[m] = xi[k];
				sd[m++] = xd[k];
			}
		}
		sp[j + 1] = m;
	}
	if (s->nnz != m) mm_real_realloc (s, m);
	return s;
}

/*#### from here ###*/

/* convert dense symmetric -> dense general */
static void
mm_real_symmetric_to_general_dense (mm_dense *d)
{
	int			j;
	if (mm_real_is_upper (d)) {
		for (j = 0; j < d->n; j++) {
			int		i0 = j + 1;
			int		n = d->m - i0;
			dcopy_ (&n, d->data + j + i0 * d->m, &d->m, d->data + i0 + j * d->m, &ione);
		}
	} else {
		for (j = 0; j < d->n; j++) dcopy_ (&j, d->data + j, &d->m, d->data + j * d->m, &ione);
	}
	mm_real_set_general (d);
	return;
}

/* convert symmetric -> general */
bool
mm_real_symmetric_to_general_2 (mm_real *x)
{
	if (!mm_real_is_symmetric (x)) return false;

	if (mm_real_is_sparse (x)) {
		mm_sparse	*s = mm_real_symmetric_to_general_sparse (x);
		if (s->nnz != x->nnz) mm_real_realloc (x, s->nnz);
		mm_real_copy_sparse_0 (s, x);
		mm_real_free (s);
	} else {
		mm_real_symmetric_to_general_dense (x);
	}

	return true;
}

double
mm_real_sub_nrm (mm_real *x1, mm_real *x2)
{
	int		k;
	double	nrm = 0.;
	for (k = 0; k < x1->nnz; k++) nrm += pow (x1->data[k] - x2->data[k], 2.);
	return nrm;
}

int
main (void)
{
	gsl_rng			*rng;

	int				n;
	MMRealFormat	format;
	MMRealSymm		symm;
	mm_real			*x;
	mm_real			*x1;

	double			nrm;

	rng = gsl_rng_alloc (gsl_rng_default);
	gsl_rng_set (rng, time (NULL));

	n = gsl_rng_uniform_int (rng, 1000) + 1;
	format = (gsl_rng_uniform (rng) < 0.5) ? MM_REAL_SPARSE : MM_REAL_DENSE;
	symm = (gsl_rng_uniform (rng) < 0.5) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
	x = mm_real_create (rng, format, symm, n, n);

	fprintf (stdout, "n = %d: symmetric", n);
	fprintf (stdout, (format == MM_REAL_SPARSE) ? " sparse" : " dense");
	fprintf (stdout, (symm == MM_REAL_SYMMETRIC_UPPER) ? " upper" : " lower");

	x1 = mm_real_symmetric_to_general (x);
	mm_real_symmetric_to_general_2 (x);
	nrm = mm_real_sub_nrm (x1, x);
	fprintf (stdout, " %.4e\n", nrm);

	if (fabs (nrm) > 1.e-5) fprintf (stderr, "nrm = %.4e: failed.\n", nrm);

	return EXIT_SUCCESS;
}

