module CGIKit

  # Utilitis is a module wihch collects utility methods based on cgi.rb.
  module Utilities
    CR            = "\015"
    LF            = "\012"
    EOL           = CR + LF
    RFC822_DAYS   = %w[ Sun Mon Tue Wed Thu Fri Sat ]
    RFC822_MONTHS = %w[ Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec ]
    MEM_CONTENT_LENGTH = 10240

    def query_from_headers( headers, input )
      if ("POST" == headers['REQUEST_METHOD']) and
          %r|\Amultipart/form-data.*boundary=\"?([^\";,]+)\"?|n.match(headers['CONTENT_TYPE'])
        boundary = $1.dup
        query_from_multipart(headers, input,
                             boundary, Integer(headers['CONTENT_LENGTH']))
      else
        case headers['REQUEST_METHOD']
        when "HEAD" then query = query_string_from_head(headers)
        when "GET"  then query = query_string_from_get(headers)
        when "POST" then query = query_string_from_post(headers, input)
        else             query = query_string_from_shell end
        Request.parse_query_string(query)
      end
    end

    def query_string_from_get( headers )
      headers['QUERY_STRING'] or ""
    end

    alias :query_string_from_head :query_string_from_get

    def query_string_from_post( headers, input )
      input.binmode
      input.read(Integer(headers['CONTENT_LENGTH'])) or ''
    end

    def query_string_from_shell
      require "shellwords"
      msg = %|(offline mode: enter name=value pairs on standard input)\n|
        string = unless ARGV.empty?
                   ARGV.join(' ')
                 else
                   if STDIN.tty?
                     STDERR.print(msg)
                   end
                   readlines.join(' ').gsub(/\n/n, '')
                 end.gsub(/\\=/n, '%3D').gsub(/\\&/n, '%26')
      words = Shellwords.shellwords(string)

      if words.find{|x| /=/n.match(x) }
        words.join('&')
      else
        words.join('+')
      end
    end

    def query_from_multipart( headers, input, boundary, content_length )
      params   = Hash.new([])
      boundary = "--" + boundary
      buf      = ""
      bufsize  = 10 * 1024

      # start multipart/form-data
      input.binmode
      boundary_size   = boundary.size + EOL.size
      content_length -= boundary_size
      status          = input.read boundary_size
      if status == nil then
        raise EOFError, "no content body"
      elsif boundary + EOL != status
        raise EOFError, "bad content body"
      end

      until content_length == -1
        head = nil
        tmp = nil
        if content_length > MEM_CONTENT_LENGTH
          require "tempfile"
          tmp = Tempfile.new("CGIKit")
          tmp.binmode
          data = TempfileByteData.new(tmp)
        else
          data = ByteData.new
        end

        until head and /#{boundary}(?:#{EOL}|--)/n.match(buf)
            if (not head) and /#{EOL}#{EOL}/n.match(buf)
                buf = buf.sub(/\A((?:.|\n)*?#{EOL})#{EOL}/n) do
              head = $1.dup
              ""
            end
              next
            end

          if head and ( (EOL + boundary + EOL).size < buf.size )
            data << buf[0 ... (buf.size - (EOL + boundary + EOL).size)]
            buf[0 ... (buf.size - (EOL + boundary + EOL).size)] = ""
          end

          if bufsize < content_length then
            c = input.read(bufsize) or ''
          else
            c = input.read(content_length) or ''
          end
          buf += c
          content_length -= c.size
        end

        buf = buf.sub(/\A((?:.|\n)*?)(?:#{EOL})?#{boundary}(#{EOL}|--)/n) do
          data << $1
          if "--" == $2
            content_length = -1
          end
          ""
        end

        /Content-Disposition:.* filename="?([^\";]*)"?/ni.match(head)
        filename = ($1 || "").dup
        if /Mac/ni.match(headers['HTTP_USER_AGENT']) and
            /Mozilla/ni.match(headers['HTTP_USER_AGENT']) and
            (not /MSIE/ni.match(headers['HTTP_USER_AGENT']))
          filename = Utilities.unescape_url filename
        end
        data.path = filename

        /Content-Type: ([^\r\n]*)/ni.match(head)
        if $1 then
          data.content_type = $1.dup
        else
          data = data.to_s
        end

        /Content-Disposition:.* name="?([^\";]*)"?/ni.match(head)
        name = $1.dup

        if params.has_key? name then
          params[name].push data
        else
          params[name] = [data]
        end
      end

      params
    end

    # Returns an encoded string for URL.
    def escape_url( string )
      string.gsub( /([^ a-zA-Z0-9_.-]+)/n ) do
        '%' + $1.unpack( 'H2' * $1.size ).join( '%' ).upcase
      end.tr( ' ', '+' )
    end

    # Returns a string decoded from URL.
    def unescape_url( string )
      string.tr( '+', ' ' ).gsub( /((?:%[0-9a-fA-F]{2})+)/n ) do
        [ $1.delete( '%' ) ].pack( 'H*' )
      end
    end

    # Escapes HTML control characters.
    def escape_html( string )
      string.gsub(/&/n, '&amp;').
        gsub(/\"/n, '&quot;').
        gsub(/>/n, '&gt;').
        gsub(/</n, '&lt;').
        gsub(/'/n, '&#39;')
    end

    # Unescapes HTML control characters.
    def unescape_html( string )
      string.gsub(/&(.*?);/n) do
        match = $1.dup
        case match
        when /\Aamp\z/ni           then '&'
        when /\Aquot\z/ni          then '"'
        when /\Agt\z/ni            then '>'
        when /\Alt\z/ni            then '<'
        when /\A#0*(\d+)\z/n       then
          if Integer($1) < 256
            Integer($1).chr
          else
            if Integer($1) < 65536 and \
              ($KCODE[0] == ?u or $KCODE[0] == ?U)
              [Integer($1)].pack("U")
            else
              "&##{$1};"
            end
          end
        when /\A#x([0-9a-f]+)\z/ni then
          if $1.hex < 256
            $1.hex.chr
          else
            if $1.hex < 65536 and \
              ($KCODE[0] == ?u or $KCODE[0] == ?U)
              [$1.hex].pack("U")
            else
              "&#x#{$1};"
            end
          end
        else
          "&#{match};"
        end
      end
    end

    # Formats Time object in RFC1123.
    # For example, "Sat, 1 Jan 2000 00:00:00 GMT".
    def date( time )
      t = time.clone.gmtime
      return format("%s, %.2d %s %.4d %.2d:%.2d:%.2d GMT",
                    RFC822_DAYS[t.wday], t.day, RFC822_MONTHS[t.month-1], t.year,
                    t.hour, t.min, t.sec)
    end

    def class_named_from( name, spaces = [CGIKit, Object] )
      spaces.each do |klass|
        base = klass
        name.split("::").each do |i|
          if i.empty? then
            i = :Object
          end
          if base.const_defined?(i) then
            base = base.const_get(i)
          else
            base = nil
            break
          end
        end
        return base if base
      end
      raise "No such '#{name}' class from #{spaces.inspect}."
    end

    module_function :query_from_headers
    module_function :query_string_from_get
    module_function :query_string_from_post
    module_function :query_string_from_head
    module_function :query_string_from_shell
    module_function :query_from_multipart
    module_function :escape_url
    module_function :unescape_url
    module_function :escape_html
    module_function :unescape_html
    module_function :date
    module_function :class_named_from

  end


  # FileLock is for locking files.
  class FileLock
    # Creates a shared file lock on a file.
    def self.shared_lock( filename, mode = 'r' )
      File.open( filename, mode ) do | io |
        io.flock File::LOCK_SH
        yield io
        io.flock File::LOCK_UN
      end
    end

    # Creates a exclusive file lock on a file.
    def self.exclusive_lock( filename, mode = 'w' )
      File.open( filename, mode ) do | io |
        io.flock File::LOCK_EX
        yield io
        io.flock File::LOCK_UN
      end
    end
  end


  class DummyLogger
    def fatal( msg = nil, &block ); end
    def warn( msg = nil, &block ); end
    def error( msg = nil, &block ); end
    def info( msg = nil, &block ); end
    def debug( msg = nil, &block ); end
    def method_missing( *args ); p args;end
  end


  module Logging

    def logger
      @application.logger
    end

    def fatal( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def error( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def warn( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def info( msg = nil, method = false, &block )
      _log(msg, method, :info, &block)
    end

    def debug( msg = nil, method = false, &block )
      _log(msg, method, :debug, &block)
    end

    def _log( msg = nil, method = false, level = :info, &block )
      msg = _log_message(msg) if method or msg.nil?
      msg = "<Thread:#{Thread.current.object_id}> #{msg}"
      if block_given? then
        logger().__send__(level, msg, block)
      else
        logger().__send__(level, msg)
      end
    end

    private

    def _log_message( msg )
      method = _method(caller[2])
      log = "#{self.class}##{method}"
      msg ? "#{log}: #{msg}" : log
    end

    def _method( at )
      if /^(.+?):(\d+)(?::in `(.*)')?/ =~ at then
        $3
      end
    end

  end

end
module CGIKit

  module Delegatable

    attr_accessor :delegate

    def delegate?( method )
      @delegate and @delegate.respond_to?(method)
    end

    def ask( method, context = nil )
      if delegate?(method)
        @delegate.context = context if context
        yield @delegate
      end
    end

  end


  class Delegate

    SUPPORT_CLASSES = []

    attr_accessor :element, :component, :context, :bindings

    class << self

      def support_classes
        SUPPORT_CLASSES
      end

      def support?( klass )
        klasses = support_classes()
        klasses.empty? or klasses.include?(klass)
      end

    end

    def value( name, does_resolve = true )
      value = @bindings[name]
      if does_resolve and Symbol === value then
        component.value_for_key(@bindings[name])
      else
        value
      end
    end

    def bool( name, does_resolve = true )
      Association.adapt_to_bool(value(name, does_resolve))
    end

    def set_value( name, value )
      component.take_value_for_key(@bindings[name], value)
    end

    def has_binding?( name )
      @bindings.key?(name)
    end

  end


  class HTMLTag

    class << self

      def a( attributes = {}, other = '', content = '' )
        new(:a, attributes, other, content)
      end

      def img( attributes = {}, other = '' )
        new(:img, attributes, other)
      end

      def form( attributes = {}, other = '' )
        new(:form, attributes, other)
      end

      def textarea( attributes = {}, other = '', content = '' )
        new(:textarea, attributes, other, content)
      end

      def input( attributes = {}, other = '' )
        new(:input, attributes, other)
      end

      def textfield( attributes = {}, other = '' )
        attributes[:type] ||= 'text'
        new(:input, attributes, other)
      end

      def radio( attributes = {}, other = '' )
        attributes[:type] = 'radio'
        new(:input, attributes, other)
      end

      def checkbox( attributes = {}, other = '' )
        attributes[:type] = 'checkbox'
        new(:input, attributes, other)
      end

      def submit( attributes = {}, other = '' )
        attributes[:type] = 'submit'
        new(:input, attributes, other)
      end

      def reset( attributes = {}, other = '' )
        attributes[:type] = 'reset'
        new(:input, attributes, other)
      end

      def upload( attributes = {}, other = '' )
        attributes[:type] = 'upload'
        new(:input, attributes, other)
      end

      def select( attributes = {}, other = '' )
        new(:select, attributes, other)
      end

      def option( attributes = {}, other = '', content = '' )
        new(:option, attributes, other, content)
      end

    end

    attr_accessor :name, :attributes, :other, :content

    def initialize( name = nil, attributes = {}, other = '', content = '' )
      @name = name
      @attributes = attributes
      @other = other
      @content = content
    end

    def []( key )
      @attributes[key]
    end

    def []=( key, value )
      @attributes[key] = value
    end

    def open_tag
      s = "<#@name #{attributes_string()}"
      unless @other.empty? then
        s << " #@other"
      end
      s << ">"
      s
    end

    def close_tag
      "</#@name>"
    end

    def empty_tag
      s = "<#@name #{attributes_string()}"
      unless @other.empty? then
        s << " #@other"
      end
      s << " />"
      s
    end

    def container_tag
      s = "<#@name #{attributes_string()}"
      unless @other.empty? then
        s << " #@other"
      end
      s << ">#@content</#@name>"
      s
    end

    def attributes_string
      s = ''
      @attributes.each do |key, value|
        if value then
          case key
          when :checked
            s << "checked=\"checked\" "
          when :selected
            s << "selected=\"selected\" "
          when :disabled
            s << "disabled=\"disabled\" "
          else
            s << "#{key}=\"#{value}\" "
          end
        end
      end
      s.chop!
      s
    end

  end

end
module CGIKit

  # ByteData objects manage bytes.
  class ByteData

    # Path of a file saving the bytes.
    attr_accessor :path

    # Content type of the bytes.
    attr_accessor :content_type

    class << self

      # Create an instance from specified IO object.
      # If you give this a File object, sets "path" attribute of the instance.
      def new_with_io( io, offset = nil, count = nil )
        io.pos = offset if offset
        bytes  = new io.read(count)
        if io.respond_to? 'path' then
          bytes.path = io.path
        end
        bytes
      end

      # Create an instance from specified file.
      def new_with_file( filename )
        bytes = nil
        open(filename) do |f|
          bytes      = new(f.read)
          bytes.path = f.path
          if ext = File.extname(filename) then
            ext.tr!('.', '')
            bytes.content_type = ResourceManager.mime(ext)
          end
        end
        bytes
      end

    end

    attr_accessor :tmp
    alias tmp? tmp

    def initialize( string = nil )
      @bytes = string || ''
    end

    def tempfile?
      false
    end

    # Returns bytes with spcified length or whole length if you omit it.
    def bytes( length = nil )
      if length then
        @bytes.slice(0, length)
      else
        @bytes.to_s
      end
    end

    # Executes the block for every byte.
    def each
      @bytes.each_byte do |byte|
        yield byte
      end
    end

    # Returns true if the bytes of each objects are equal.
    def ==( bytes )
      @bytes == bytes.bytes
    end

    # Returns length of the bytes.
    def length
      @bytes.size
    end
    alias size length

    # Appends bytes to the bytes.
    def <<( bytes )
      if bytes.is_a?(ByteData) then
        @bytes << bytes.bytes
      else
        @bytes << bytes
      end
      self
    end

    # Writes the bytes to a specified file.
    def write_to_file( filename, lock = true )
      if lock then
        FileLock.exclusive_lock(filename, 'w+b') do |file|
          file.write to_s
        end
      else
        File.open(filename, 'w+b') do |file|
          file.write to_s
        end
      end      
    end

    # Returns the object as a string.
    def to_s
      @bytes.to_s
    end

    def open; end
    def close; end

  end


  class TempfileByteData < ByteData

    attr_accessor :tempfile

    def initialize( tempfile )
      @tempfile = tempfile
      close
    end

    def tempfile?
      true
    end

    # Returns bytes with spcified length or whole length if you omit it.
    def bytes( length = nil )
      open do
        @tempfile.read(length)
      end
    end

    # Executes the block for every byte.
    def each
      open do
        @tempfile.each_byte do |byte|
          yield byte
        end
      end
    end

    def ==( bytes )
      @tempfile == bytes.tempfile
    end

    def length
      open do
        @tempfile.size
      end
    end

    def <<( bytes )
      open do
        @tempfile.seek(0, IO::SEEK_END)
        if ByteData === bytes then
          @tempfile << bytes.bytes
        else
          @tempfile << bytes
        end
        self
      end
    end

    def to_s
      bytes
    end

    def open( &block )
      @tempfile.open if @tempfile.closed?
      if block_given? then
        @tempfile.rewind
        value = block.call
        close
        return value
      end
    end

    def close
      @tempfile.close unless @tempfile.closed?
    end

    def _dump( limit )
      Marshal.dump(bytes(), limit)
    end

    def self._load( object )
      ByteData.new(Marshal.load(object))
    end

  end

end

module CGIKit

# KeyValueCoding provides methods to access object graph.
# The methods try accessing instance variables by using accessor method.
# If it is failure, its try accessing directly.
module KeyValueCoding

  # The method returns "true", the direct access is success. Or failure.
  # Default value is true.
  attr_accessor :access_attributes
  alias access_attributes? access_attributes

  @access_attributes = true

  # Retrieves value for the key.
  def value_for_key( __key )
    # escape sharing local variable scope of instance_eval by __key.
    __key = __key.to_s.gsub(/\A\^(\w+)/) { "binding_value(:#{$1})" }
    begin
      instance_eval(__key.to_s.untaint)
    rescue Exception => e
      if /\A[a-zA-Z0-9_]+\Z/ === __key.to_s then
        raise e, "#{e.message} - #{self.class}##{__key}"
      else
        raise e, "#{e.message} - \"#{__key}\" for #{self.class}"
      end
    end
  end

  def stored_value_for_key( key )
    instance_variable_get("@#{key}")
  end

  # Takes value for the key.
  def take_value_for_key( __key, __value )
    writer = "self.#{__key.to_s}=__value"
    instance_eval(writer.untaint)
  end

  def take_stored_value_for_key( key, value )
    instance_variable_set("@#{key}", value)
  end

  alias value_for_keypath value_for_key
  alias take_value_for_keypath take_value_for_key

  # used only accessing instance variables
  def method_missing( name, *args )
    if access_attributes? then
      if args.empty? then
        return stored_value_for_key(name.to_s)
      elsif (name.to_s =~ /([^=]+)=$/) and (args.size == 1) then
        return take_stored_value_for_key($1, args[0])
      end
    end
    super
  end

end

end
module CGIKit

  # Classes under Adapter are interface between CGIKit and web servers.
  # 
  # The role of adapter is to provide the consistent interface to 
  # Application. The difference of the relationship between 
  # CGIKit and a web server is aboserbed by the obejct. 
  module Adapter
    
    class CGI
      attr_accessor :headers, :params
      attr_reader :input, :output, :error

      CR  = "\r"
      LF  = "\n"
      EOL = CR + LF
      
      def initialize
        @headers = ENV
        @input   = $stdin
        @output  = $stdout
        @error   = $stderr
        @params = Utilities.query_from_headers(@headers, @input)
      end
      
      def create_request
        Request.new(headers, params)
      end
      
      def create_response
        Response.new(headers)
      end
      
      def run( request, response, &block )
        request ||= create_request
        
        if block_given?
          response = block.call(request)
        end

        print response.header
        print response.to_s
      end
      
    end
    
    class ModRuby < CGI
      
      def initialize
        @input   = $stdin
        @output  = $stdout
        @error   = $stderr
        
        ap_req = Apache.request
        
        @headers = {}
        ap_req.subprocess_env.each {|key,value|
          @headers[key] = value
        }
        
        @params = Utilities.query_from_headers(@headers, @input)
      end
      
      def run( request, response, &block )
        ck_req = CGIKit::Request.new(@headers, @params)

        if block_given?
          ck_res = block.call(ck_req)
        end
        
        # import from cgi.rb
        #
        # This implementation is awkward and slow.
        ap_req = Apache.request  
        table = ap_req.headers_out
        ck_res.header.scan(/([^:]+): (.+)#{EOL}/n){ |name, value|
          case name
          when 'Set-Cookie'
            table.add(name, value)
          when /^status$/ni
            ap_req.status_line = value
            ap_req.status = value.to_i
          when /^content-type$/ni
            ap_req.content_type = value
          when /^content-encoding$/ni
            ap_req.content_encoding = value
          when /^location$/ni
            if ap_req.status == 200
              ap_req.status = 302
            end
            ap_req.headers_out[name] = value
          else
            ap_req.headers_out[name] = value
          end
        }
        
        ap_req.send_http_header
        ap_req.write(ck_res.to_s)
      end
    end
    
    class Template < CGI
      def initialize
        require 'stringio'
        @headers = ENV
        @input   = StringIO.new
        @output  = StringIO.new
        @error   = StringIO.new
        @params  = Hash.new([])
      end
      
      def run( request, response, &block )
        request ||= create_request
        
        if block_given?
          response = block.call(request)
        end
        
        response
      end
    end
    
  end
  
end



require 'digest/md5'

unless Hash.method_defined?(:key)   # Ruby 1.8.x
  class Hash
    alias key index
  end
end

module CGIKit

  # A class for session management. Session objects have a hash of
  # arbitary objects and information about browser name, IP address, etc.
  # However, you can't set objects that can't be marshal ( IO, Proc, etc. )
  # to the session with default database manager FileSessionStore.
  class Session
    include Logging

    DEFAULT_SESSION_ID_FIGURES = 16
    DEFAULT_TIMEOUT = 60 * 60 * 24 * 7

    class << self
      def session_id?( session_id )
        session_id and (/\A([0-9A-Za-z]+)\Z/ === session_id)
      end

      def create_session_id
        md5 = Digest::MD5::new
        md5.update Time.now.to_s
        md5.update rand(0).to_s
        md5.update $$.to_s
        md5.hexdigest[0, session_id_figures]
      end

      def session_id_figures
        DEFAULT_SESSION_ID_FIGURES
      end
    end


    # A hash of arbitary objects.
    attr_accessor :values

    # Session ID.
    attr_accessor :session_id

    # Seconds until the session has timed out.
    attr_accessor :timeout

    # Name of browser.
    attr_accessor :user_agent

    # IP address.
    attr_accessor :remote_addr

    attr_accessor :context, :application, :session_store, :last_accessed_time, \
    :caches, :context_ids, :frame_components, :permanent_caches, :session_key, \
    :cookie_expires, :editing_context
    alias ec editing_context

    def initialize( session_id = nil )
      unless Session.session_id? session_id then
        session_id = Session.create_session_id
      end
      @session_id         = session_id
      @last_accessed_time = Time.new
      @context_ids        = {}
      @caches             = {}
      @permanent_caches   = {}
      @values             = {}
      @frame_components   = {}
      @user_agent         = nil
      @remote_addr        = nil
      @timeout            = DEFAULT_TIMEOUT
      @terminate          = false
      @last_component_id  = -1
      init
    end

    #
    # hook
    
    def init; end


    #
    # accessing
    #

    def []( key )
      @values[key]
    end

    def []=( key, value )
      @values[key] = value
    end

    def remove( key )
      @values.delete(key)
    end

    def terminate
      @terminate = true
    end

    def terminate?
      @terminate or timeout?
    end


    #
    # accessing cached components
    #

    def add_component( component, context = nil, permanently = false )
      unless component_id = component_id(component) then
        component_id = next_component_id()
        if context then
          context_id = context.context_id
        else
          context_id = "#{component_id}.0"
        end
        add_component_for_ids(component, component_id, \
                              context_id, permanently)
      end
      component_id
    end

    def next_component_id
      @last_component_id += 1
    end

    def add_component_for_ids( component,
                               component_id,
                               context_id,
                               permanently = false )
      if permanently then
        page_caches = @permanent_caches
      else
        page_caches = @caches
      end
      page_caches[component_id] = component
      @context_ids[context_id] = component_id
    end

    def component_id( component )
      if @caches.value?(component) then
        @caches.key(component)
      elsif @permanent_caches.value?(component) then
        @permanent_caches.index(component)
      else
        nil
      end
    end

    def component_for_component_id( component_id )
      @caches[component_id] || @permanent_caches[component_id]
    end

    def component_for_context_id( context_id )
      page = nil
      unless context_id.nil? then
        ids = context_id.split(Context::SEPARATOR)
        while ids.size > 1
          id = ids.join(Context::SEPARATOR)
          if page = _component(id) then break end
          ids.pop
          ids.push(0)
          id = ids.join(Context::SEPARATOR)
          if page = _component(id) then break end
          ids.pop
        end
      end
      page
    end

    alias component component_for_context_id

    private

    def _component( context_id )
      if id = @context_ids[context_id] then
        @caches[id] || @permanent_caches[id]
      end
    end

    public

    #
    # testing
    #

    # Enables or disables the use of URLs for storing session IDs.
    def store_in_url?
      @application.store_in_url
    end

    # Enables or disables the use of cookies for storing session IDs.
    def store_in_cookie?
      @application.store_in_cookie
    end

    # Enables or disables session authorization by browsers.
    def auth_by_user_agent?
      @application.auth_by_user_agent
    end

    # Enables or disables session authorization by IP addresses.
    def auth_by_remote_addr?
      @application.auth_by_remote_addr
    end

    # Returns true if the browser is equal to one when the session created.
    def user_agent?( user_agent )
      auth_by_user_agent? and (@user_agent == user_agent)
    end

    # Returns true if the IP address is equal to one when the session created.
    def remote_addr?( remote_addr )
      auth_by_remote_addr? and (@remote_addr == remote_addr)
    end

    # Returns true if the session isn't expired.
    def timeout?
      if @timeout then
        (@timeout != 0) and (timeout_expire <= Time.new)
      else
        false
      end
    end

    def timeout_expire
      @last_accessed_time + @timeout
    end


    #
    # Handling requests
    #

    def take_values_from_request( request, context )
      context.component.delegate_take_values_from_request(request, context)
    end

    def invoke_action( request, context )
      context.component.delegate_invoke_action(request, context)
    end

    def append_to_response( response, context )
      context.component.delegate_append_to_response(response, context)
    end


    #
    # Page management
    #

    def save_page( component, permanently = false )
      if permanently then
        caches = @permanent_caches
        size = @application.permanent_page_cache_size
      else
        caches = @caches
        size = @application.page_cache_size
      end
      component.add_all_components(self, permanently)
      if component.context then
        cid = component.context.context_id
        @context_ids[cid] = component_id(component)
      end
      _remove_old_cache(caches, size)
    end

    private

    def _remove_old_cache( caches, size )
      removes = caches.size - size
      if removes > 0 then
        if removed_keys = caches.keys.sort.slice!(0..removes-1) then
          caches.delete_if do |key, value|
            if removed_keys.include?(key) then
              true
            end
          end
        end
      end
    end

    public

    def restore_page( context_id )
      component(context_id)
    end


    #
    # marshaling
    #

    def marshal_dump
      dump                      = {}
      dump[:timeout]            = @timeout
      dump[:values]             = @values
      dump[:session_id]         = @session_id
      dump[:user_agent]         = @user_agent
      dump[:remote_addr]        = @remote_addr
      dump[:caches]             = @caches
      dump[:permanent_caches]   = @permanent_caches
      dump[:context_ids]        = @context_ids
      dump[:frame_components]   = @frame_components
      dump[:last_accessed_time] = @last_accessed_time
      dump[:last_component_id]  = @last_component_id
      dump[:cookie_expires]     = @cookie_expires
      dump
    end

    def marshal_load( object )
      @timeout            = object[:timeout]
      @values             = object[:values]
      @session_id         = object[:session_id]
      @user_agent         = object[:user_agent]
      @remote_addr        = object[:remote_addr]
      @context_ids        = object[:context_ids]
      @caches             = object[:caches]
      @permanent_caches   = object[:permanent_caches]
      @frame_components   = object[:frame_components]
      @last_accessed_time = object[:last_accessed_time]
      @last_component_id  = object[:last_component_id]
      @cookie_expires     = object[:cookie_expires]
    end

    def awake_from_restoration( application, request )
      @application         = application
      @timeout             = application.timeout
      @cookie_expires      = application.session_cookie_expires
      @session_key         = application.session_key
      @session_store       = application.session_store
      @last_accessed_time  = Time.new
      @user_agent          = request.user_agent
      @remote_addr         = request.remote_addr
      if defined?(TapKit::EditingContext) and @application.database then
        @editing_context = @application.database.create_editing_context
      end
    end


    #
    # validating
    #

    def validate
      validate_authorization
      validate_timeout
    end

    def validate_authorization
      if (auth_by_user_agent? and \
          !user_agent?(@context.request.user_agent)) or \
        (auth_by_remote_addr? and \
         !remote_addr?(@context.request.remote_addr)) then
        raise Application::SessionAuthorizationError, 'Your session is not authorized.'
      end
    end

    def validate_timeout
      if timeout? then
        flush
        raise Application::SessionTimeoutError, 'Your session has timed out.'
      end
    end

    def flush
      @session_store.remove(@session_id)
      if store_in_cookie? then
        remove_cookie(@context.response)
      end
    end


    #
    # managing cookie
    #

    def remove_cookie
      cookie         = Cookie.new(@session_key)
      cookie.expires = Time.new - 600
      @context.response.add_cookie(cookie)
    end

    def set_cookie
      cookie = Cookie.new(@session_key, @session_id)
      cookie.expires = Time.new + @cookie_expires
      @context.response.add_cookie(cookie)
    end

  end

end
module CGIKit

# SessionStore is a class for saving session into storage.
class SessionStore
  include Logging

  def initialize( application )
    @application = application
  end

  def checkin( context )
    debug("context ID = #{context.session.session_id}", true)
    save(context)
  end

  def checkout( session_id, request )
    if session_id.nil? then
      return nil
    end
    session = restore(session_id, request)
    debug("session ID = #{session_id}, restored = #{!session.nil?}", true)
    session
  end

  # abstract.
  def save( context ); end

  # abstract.
  def restore( session_id, request ); end

  # abstract.
  def remove( session_id ); end

  def sweep_sessions; end

end


# A FileSessionStore object saves a session to a file with marshaling.
class FileSessionStore < SessionStore
  TMPDIR = 'session'

  def initialize( application )
    super
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end

  def save( context )
    unless FileTest.directory? @tmpdir
      require 'ftools'
      File.makedirs @tmpdir
    end
    FileLock.exclusive_lock(tmpfile(context.session.session_id)) do |file|
      Marshal.dump(context.session, file)
    end
  end

  def restore( session_id, request )
    session = nil
    if exist?(session_id) then
      FileLock.shared_lock(tmpfile(session_id)) do | file |
        session = Marshal.load file
      end
      session.session_store = self
      session.application = @application
    end
    session
  end

  def tmpfile( session_id )
    File.join(@tmpdir, session_id).untaint
  end

  def exist?( session_id )
    FileTest.exist?(tmpfile(session_id))
  end

  def remove( session_id )
    if FileTest.exist?(tmpfile(session_id))
      File.delete(tmpfile(session_id))
    end
  end

  def sweep_sessions
    deleted = 0
    failed = 0
    Dir.foreach(@tmpdir) do |file|
      begin
        unless /\A\./ === file
          path = File.join(@tmpdir, file)
          session = nil
          FileLock.shared_lock(path) do |f|
            session = Marshal.load(f)
          end
          if session.timeout? then
            begin
              File.delete(path)
              deleted += 1
            rescue
              failed += 1
            end
          end
        end
      rescue
      end
    end
    info("deleted = #{deleted}, failed = #{failed}", true)
    [deleted, failed]
  end

end


class MemorySessionStore < SessionStore
  def initialize( application )
    super
    @caches = {}
  end

  def save( context )
    @caches[context.session.session_id] = context.session
  end

  def restore( session_id, request )
    if session = @caches[session_id] then
      session.session_store = self
      session.application = @application
    end
    session
  end

  def remove( session_id )
    @caches.delete(session_id)
  end

  def exist?( session_id )
    @caches.key?(session_id)
  end

  def sweep_sessions
    before = @caches.size
    @caches.delete_if do |id, session|
      session.timeout?
    end
    deleted = before - @caches.size
    failed = 0
    info("deleted = #{deleted}, failed = #{failed}", true)
    [deleted, failed]
  end

end

end
module CGIKit

class Template

  attr_accessor :template_store, :component, :template_node, :declaration_store, \
    :template_string, :declaration_hash, :template_mtime, :declaration_mtime

  def initialize
    @template_mtime = Time.now
    @declaration_mtime = Time.now
  end

  def terminate?
    @template_store.terminate?(@component)
  end

  def template_string=( string )
    @template_string = string
    @template_mtime = Time.now
  end

  def declaration_store=( store )
    @declaration_store = store
    @declaration_mtime = Time.now
  end

  def marshal_dump
    not_includes = ['@component', '@template_store']
    dump = {}
    instance_variables.each do |var|
      if not_includes.include?(var) == false then
        dump[var] = instance_variable_get(var)
      end
    end
    dump
  end

  def marshal_load( object )
    object.each do |key, value|
      instance_variable_set(key, value)
    end
  end

  def cache_copy
    copy = dup
    copy.template_node = @template_node.cache_copy
    copy
  end
end


class TemplateStore
  include Logging

  def initialize( application )
    @application = application
  end

  def cache?
    @application.cache_template
  end

  def checkin( template )
    debug(template.component.class, true)
    save(template)
  end

  def checkout( component )
    debug(component.class, true)
    template = restore(component)
    if template.nil? or (cache? and terminate?(template, component)) then
      template = create_template(component)
      isnew = true
      checkin(template) if cache?
      debug("create template #{component.class}")
    end
    template.component = component
    template.template_store = self
    template.template_node.component = component
    template
  end

  def terminate?( template, component )
    terminate_template?(template, component) or \
      terminate_declaration?(template, component)
  end

  def terminate_template?( template, component )
    (component.will_parse_template and \
     (component.will_parse_template != template.template_string)) or \
    (template.template_mtime and \
     (File.mtime(component.template_file) > template.template_mtime))
  end

  def terminate_declaration?( template, component )
    ((component.will_parse_declarations) and \
     (component.will_parse_declarations != template.declaration_hash)) or \
    (template.declaration_mtime and \
     (File.mtime(component.declaration_file) > template.declaration_mtime))
  end

  def create_template( component )
    template = Template.new
    template.component = component
    parser = load_template(component)
    template.template_node = parser.node
    template.template_string = parser.html_string
    template.declaration_store = load_declarations(component, parser.declarations)
    template
  end

  def load_template( component )
    klass = @application.htmlparser_class
    if string = component.will_parse_template then
      parser = klass.new
      parser.parse(string)
    elsif path = component.template_file then
      parser = klass.new(path)
    else
      raise "can't find a template file - #{component.class}"
    end
    parser
  end

  def load_declarations( component, source )
    store = nil
    if hash = component.will_parse_declarations then
      DeclarationStore.merge_source(source, hash)
      store = DeclarationStore.new_from_hash(hash)
    elsif path = component.declaration_file then
      begin
        store = DeclarationStore.new_with_file(path, source)
      rescue Exception => e
        raise "can't load declaration file (#{path}): #{e.message}"
      end
    else
      store = DeclarationStore.new_from_hash(source)
    end
    store
  end

  def save( template )
    raise StandardError, 'Not implemented'
  end
    
  def restore( component )
    raise StandardError, 'Not implemented'
  end
    
  def remove( component )
    raise StandardError, 'Not implemented'
  end
    
end


class FileTemplateStore < TemplateStore

    TMPDIR = 'template_cache'

  def initialize( application )
    super
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end
  
  def save( template )
    unless FileTest.directory?(@tmpdir) then
      require 'fileutils'
      FileUtils.makedirs(@tmpdir)
    end
    
    filename = cache_file_name(template.component)
    FileLock.exclusive_lock(filename) do |file|
      Marshal.dump(template, file)
    end
  end

  def restore( component )
    template = nil
    if cache? and exist?(component) then
      FileLock.shared_lock(cache_file_name(component)) do |file|
        template = Marshal.load(file)
      end
    end    
    template
  end
  
  def cache_file_name( component )
    name = component.template_file.gsub('::', '__')
    name.tr!('./', '_')
    File.join(@tmpdir, name)
  end

  def exist?( component )
    FileTest.exist?(cache_file_name(component))
  end
  
end


class MemoryTemplateStore < TemplateStore

  def initialize( application )
    super
    @caches = {}
  end

  def save( template )
    @caches[template.component.template_file] = template.cache_copy
  end

  def restore( component )
    if template = @caches[component.template_file] then
      template = template.cache_copy
    end
    template
  end

  def exist?( component )
    @caches.key?(component.class)
  end

end

end
module CGIKit

  class RequestHandler
    include Logging

    attr_reader :application

    class << self

      def register( app = nil )
        app ||= Application
        app.register_request_handler(request_handler_key(), self)
      end

      def request_handler_key; end

    end

    def initialize( application )
      @application = application
    end

    def request_handler_key
      @application.request_handler_key(self)
    end


    #
    # parsing session IDs and context IDs from URLs
    #

    def session_id( request )
      sid_url = session_id_from_url(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_url then
        sid_url
      else
        nil
      end
    end

    def session_id_from_cookie( request )
      sid = nil
      cookie = request.cookie(@application.session_key)
      if cookie then
        sid = cookie.value
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def session_id_from_url( request )
      sid = nil
      if path = request.request_handler_path then
        sid = _id_by_separating(path, false)
      end
      sid
    end

    private

    def _id_by_separating( path, context = true )
      path = path.reverse.chop.reverse
      separated = path.split('/')
      separated.delete_at(0)
      id = nil
      case separated.size
      when 1
        if context then
          if separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        else
          if !separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        end
      when 2
        if context then
          id = separated.last
        else
          id = separated.first
        end
      end
      if id and !context and !Session.session_id?(id) then
        id = nil
      end
      id
    end

    public

    def context_id( request )
      id = nil
      if path = request.request_handler_path then
        id = _id_by_separating(path, true)
      end
      id
    end


    #
    # request-response loop
    #

    # Abstract method. Returns a response object.
    def handle_request( request )
      request.session_id = session_id(request)
      request.context_id = context_id(request)
      info("session ID = #{request.session_id}, " <<
             "context ID = #{request.context_id}", true)
    end

    def transaction( element, context, &block )
      element.begin_context(context)
      block.call
      element.end_context(context)
    end

    def take_values_from_request( element, request, context )
      transaction(element, context) do
        element.delegate_take_values_from_request(request, context)
      end
    end

    def invoke_action( element, request, context )
      result = nil
      transaction(element, context) do
        result = element.delegate_invoke_action(request, context)
      end

      if (Component === result) or (Response === result) then
        result
      else
        nil
      end
    end

    def append_to_response( element, response, context )
      transaction(element, context) do
        element.delegate_append_to_response(response, context)
      end
    end


    #
    # generating URLs
    #

    def url( context, path, query, is_secure, port = nil, sid = true ); end

    def application_url( request, is_secure = false, port = nil )
      protocol = nil
      if is_secure == true then
        protocol = 'https://'
      else
        protocol = nil
      end

      domain = request.server_name || 'localhost'

      if port.nil? and request.server_port then
        port = request.server_port.to_i
      end
      if port == 80 then
        port = nil
      end

      script = (request.script_name || @application.path).dup
      script.sub!(/\A\//, '')

      if port then
        protocol ||= 'http://'
        path = "#{protocol}#{domain}:#{port}/#{script}"
        path.gsub!(/\/\Z/, '')
      elsif protocol then
        path = "#{protocol}#{domain}/#{script}"
      else
        path = "/#{script}"
      end
      path
    end

    def query_string( hash = {} )
      str = ''
      keys = hash.keys.sort do |a, b|
        a.to_s <=> b.to_s
      end
      keys.each do |key|
        value = hash[key]
        if Array === value then
          value.each do |item|
            str << query_association(key, item)
          end
        else
          str << query_association(key, value)
        end
      end
      str.chop!
      str
    end

    def query_association( key, value )
      unless value.nil? then
        "#{key}=#{Utilities.escape_url(value.to_s)};"
      else
        "#{key};"
      end
    end
  end


  class ComponentRequestHandler < RequestHandler
    def handle_request( request )
      super
      context = @application.create_context(request)
      context.request_handler_key = @application.component_request_handler_key

      @application.synchronize do
        transaction(context.component, context) do
          @application.take_values_from_request(request, context)
        end

        result = nil
        context.delete_all
        transaction(context.component, context) do
          result = @application.invoke_action(request, context)
        end

        if Response === result then
          context.response = result
        else
          if (Component === result) and (context.component != result) then
            result.awake_from_restoration(context)
            context.component = result
          else
            result = context.component.root
          end
          context.delete_all
          transaction(result, context) do
            @application.append_to_response(context.response, context)
          end
          context.response.component = result
        end
      end

      @application.save_session(context)
      context.response
    end

    def url( context, path = nil, query = {},
             is_secure = false, port = nil, sid = true )
      str = application_url(context.request, is_secure, port)
      str << "/#{@application.component_request_handler_key}"
      if sid and context.has_session? then
        str << "/#{context.session.session_id}"
      end
      str << "/#{context.context_id}"
      str << "/#{path}" if path
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

  end


  class ActionRequestHandler < RequestHandler

    class ActionResultError < StandardError #:nodoc:
    end

    def action_and_action_name( request )
      klass = request.action_class || default_action_class()
      action = klass.new(@application, request)
      name = request.action_name || action.default_action_name
      unless action.action?(name) then
        klass = default_action_class()
        action = klass.new(@application, request)
        name = action.default_action_name
      end
      [action, name]
    end

    def action_url( context, action_class, action_name, query, sid = true )
      action_class ||= default_action_class()
      path = action_path(action_class, action_name)
      url(context, path, query, false, nil, sid)
    end

    def action_path( action_class, action_name )
      if action_name == action_class.default_action_name then
        action_name = ''
      end
      if default_action_class() == action_class then
        action_name
      elsif action_name.empty?
        action_class
      else
        "#{action_class}/#{action_name}"
      end
    end

    def url( context, path = nil, query = {}, is_secure = false,
             port = 80, sid = true )
      str = application_url(context.request, is_secure, port)
      str << "/#{request_handler_key()}"
      str << "/#{path}" if path
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def default_action_class; end

  end


  class DirectActionRequestHandler < ActionRequestHandler

    def session_id( request )
      sid_query = session_id_from_query(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_query then
        sid_query
      else
        nil
      end
    end

    def session_id_from_query( request )
      if sid = request[@application.direct_action_session_key] then
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def handle_request( request )
      super
      direct_action, action_name = action_and_action_name(request)

      result = response = nil
      @application.synchronize do
        result = direct_action.perform_action(action_name)
        unless result.respond_to?(:generate_response) then
          raise ActionResultError, \
          "Direct action must return an object has generate_response()" +
            " and not nil. - #{direct_action.class}##{action_name}"
        end
        response = result.generate_response
      end

      if Component === result then
        response.component = result
        if result.context.has_session? then
          @application.save_session(result.context)
        end
      end
      response
    end

    def url( context, path = nil, query = {}, is_secure = false,
             port = 80, sid = true )
      str = application_url(context.request, is_secure, port)
      str << "/#{request_handler_key()}"
      str << "/#{path}" if path
      if sid and context.has_session? then
        query[@application.direct_action_session_key] = context.session.session_id
      end
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def default_action_class
      @application.direct_action_class
    end

  end


  class ResourceRequestHandler < RequestHandler

    RESOURCE_KEY = 'data'

    def handle_request( request )
      super
      response = Response.new
      key = request.form_value(RESOURCE_KEY)
      debug("request resource key: #{key}")
      if data = @application.resource_manager.bytedata(key) then
        response.headers['Content-Type'] = data.content_type
        response.content = data.to_s
        @application.resource_manager.remove_data(key) if data.tmp?
      else
        response.status = 404
      end
      response
    end

    def resource_url( name, request )
      str = application_url(request)
      str << "/#{@application.resource_request_handler_key}"
      str << "?#{RESOURCE_KEY}=#{name}"
      str
    end

  end

end
require 'monitor'

module CGIKit

  VERSION = '2.0.0'

  class CGIKitError < StandardError
  end


  class Application

    include KeyValueCoding, Logging

    class SessionCreationError < StandardError #:nodoc:
    end
    class SessionRestorationError < StandardError #:nodoc:
    end
    class SessionAuthorizationError < SessionRestorationError #:nodoc:
    end
    class SessionTimeoutError < SessionRestorationError #:nodoc:
    end
    class PageRestorationError < StandardError #:nodoc:
    end

    COMPONENT_REQUEST_HANDLER_KEY     = 'c'
    DIRECT_ACTION_REQUEST_HANDLER_KEY = 'd'
    RESOURCE_REQUEST_HANDLER_KEY      = 'r'

    # backward compatibility
    CGIKIT_LIB    = 'cgikit'
    COMPONENT_LIB = 'components'

    DATA_PATH    = 'data'
    CGIKIT_PATH  = 'cgikit'
    PACKAGE_PATH = 'packages'

    @@handlers = {
      COMPONENT_REQUEST_HANDLER_KEY => ComponentRequestHandler,
      DIRECT_ACTION_REQUEST_HANDLER_KEY => DirectActionRequestHandler,
      RESOURCE_REQUEST_HANDLER_KEY => ResourceRequestHandler
    }

    class << self

      def request_handlers
        @@handlers
      end

      def request_handler_key( klass )
        @@handlers.index(klass)
      end

      def request_handler_class( key )
        @@handlers[key]
      end

      def register_request_handler( key, klass )
        if @@handlers[key] then
          raise "request handler key '#{key}' is already registered."
        end
        @@handlers[key] = klass
      end

      def remove_request_handler( key )
        @handlers.delete(key)
      end

    end


    # Main component. If session ID or context ID aren't specified,
    # this component is shown. The default value is MainPage.
    attr_accessor :main

    # Document root directory.
    attr_accessor :document_root

    # The application URL based on SCRIPT_NAME.
    attr_accessor :baseurl

    # The file system path of the application.
    attr_accessor :path

    # The file system paths for components. Components are searched under it.
    attr_accessor :component_paths
    attr_accessor :component_path

    # Resource directory.
    # This directory includes files to be used by the application,
    attr_accessor :resource_path
    alias resources resource_path
    alias resources= resource_path=

    # Web server resources directory.
    # This directory includes files to be displayed to browser.
    # The files are used by Image element, etc.
    attr_accessor :web_server_resource_path
    alias web_server_resources web_server_resource_path
    alias web_server_resources= web_server_resource_path=

    # ResourceManager object.
    attr_accessor :resource_manager

    # Adapter object.
    attr_accessor :adapter

    # Adapter class. The default value is CGI class.
    attr_accessor :adapter_class

    # Name or class of an error page component to show caught errors.
    attr_accessor :error_page

    # Temporary directory to be used by the framework.
    # The framework uses this to store sessions and template caches.
    attr_accessor :tmpdir

    # Session key. This key is used in cookie.
    attr_accessor :session_key

    # Session key in direct action.
    # This key is used in hidden fields of form and URL when using direct action.
    attr_accessor :direct_action_session_key

    # Seconds until the session has timed out.
    attr_accessor :timeout

    # Expiry date of cookie for session. If you set the value to nil,
    # session cookies will be invalid when closing browser.
    attr_accessor :session_cookie_expires

    # Enables or disables the use of URLs for storing session IDs.
    attr_accessor :store_in_url

    # Enables or disables the use of cookies for storing session IDs.
    attr_accessor :store_in_cookie

    # Enables or disables session authorization by browsers.
    # If you set the value to true, the application raises error
    # when an user accesses it with browser that is different from
    # one registered session.
    attr_accessor :auth_by_user_agent

    # Enables or disables session authorization by IP addresses.
    # If you set the value to true, the application raises error
    # when an user accesses it with IP address that is different from
    # one registered session.
    attr_accessor :auth_by_remote_addr

    # Encoding to encode character code of form data.
    # The default implementation uses Kconv to encode Japanese character codes.
    # Then specify constant values of Kconv; Kconv::JIS, Kconv::SJIS, Kconv::EUC, etc.
    # If the value is nil, form data is not encoded. The default value is nil.
    attr_accessor :encoding

    # Request handler key to display components.
    attr_accessor :component_request_handler_key

    # Request handler key to invoke direct actions.
    attr_accessor :direct_action_request_handler_key

    # Request handler key to display resource files.
    attr_accessor :resource_request_handler_key

    # Default request handler key. Default key is component request handler key.
    attr_accessor :default_request_handler

    # Request handler for components.
    attr_accessor :component_request_handler

    # Request handler for direct actions.
    attr_accessor :direct_action_request_handler

    # Request handler for resources.
    attr_accessor :resource_request_handler

    # Session class. Default class is CGIKit::Session.
    attr_accessor :session_class

    # Whether or not validates setting of attributes for each elements.
    # If wrong attribute name or combination are found, raises error.
    attr_accessor :validate_api
    alias validate_api? validate_api

    # HTML parser class. Default class is CGIKit::HTMLParser::HTMLParser.
    attr_accessor :htmlparser_class

    # Size to cache components permanently in session.
    # Newly generated page is cached automatically.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :page_cache_size

    # Size to cache components permanently in session.
    # Permanent page cache is cached optionally, not automatically caching.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :permanent_page_cache_size

    # Direct action class. Default class is CGIKit::DirectAction.
    attr_accessor :direct_action_class

    # Session store class. Default class is CGIKit::FileSessionStore.
    attr_accessor :session_store_class

    # Context class. Default class is CGIKit::Context.
    attr_accessor :context_class

    # Whether or not caches templates to reduce parsing load.
    attr_accessor :cache_template

    attr_accessor :template_store
    attr_accessor :template_store_class
    attr_accessor :resource_store
    attr_accessor :resource_store_class
    attr_accessor :sweep_password
    attr_accessor :datadir
    attr_accessor :package_paths
    attr_accessor :required_packages
    attr_accessor :main_package_options
    attr_accessor :model_path
    attr_accessor :database
    attr_accessor :logger
    attr_accessor :log_options
    attr_accessor :concurrent_request_handling

    def initialize
      init_attributes
      init_request_handlers
      init_adapter
      init_component_paths
      init_name_spaces
      init
    end

    def init_attributes
      require 'rbconfig'
      @main                      = 'MainPage'
      @error_page                = 'ErrorPage'
      @tmpdir                    = './tmp' || ENV['TMP'] || ENV['TEMP']
      @datadir                   = Config::CONFIG['datadir']
      @package_paths             = [PACKAGE_PATH,
        File.join(@datadir, CGIKIT_PATH, PACKAGE_PATH),
        File.join(DATA_PATH, CGIKIT_PATH, PACKAGE_PATH)]
      @session_key               = '_session_id'
      @direct_action_session_key = '_sid'
      @manage_session            = false
      @timeout                   = 60 * 60 * 24 * 7
      @session_cookie_expires    = 60 * 60 * 24 * 7
      @store_in_url              = true
      @store_in_cookie           = false
      @auth_by_user_agent        = false
      @auth_by_remote_addr       = false
      @session_class             = Session
      @session_store             = nil
      @session_store_class       = FileSessionStore
      @template_store            = nil
      @template_store_class      = FileTemplateStore
      @resource_store            = nil
      @resource_store_class      = FileResourceStore
      @encoding                  = nil
      @resource_path             = Package::RESOURCE_PATH
      @web_server_resource_path  = Package::WEB_SERVER_RESOURCE_PATH
      @model_path                = Package::MODEL_PATH
      @validate_api              = true
      @cache_template            = true
      @htmlparser_class          = HTMLParser::HTMLParser
      @page_cache_size           = 30
      @permanent_page_cache_size = 30
      @direct_action_class       = DirectAction
      @context_class             = Context
      @baseurl                   = nil
      @required_packages         = []
      @main_package_options      = {}
      @request_handlers          = {}
      @concurrent_request_handling = true
      @lock                      = Monitor.new
      @log_options = {:level => nil, :name => 'CGIKit', :out => $stderr}
      if defined?(TapKit::Application) then
        @database = TapKit::Application.new
      end
    end

    def init_request_handlers
      self.class.request_handlers.each do |key, klass|
        register_request_handler(key, klass)
      end
      @component_request_handler_key     = COMPONENT_REQUEST_HANDLER_KEY
      @direct_action_request_handler_key = DIRECT_ACTION_REQUEST_HANDLER_KEY
      @resource_request_handler_key      = RESOURCE_REQUEST_HANDLER_KEY
      @component_request_handler =
        request_handler(@component_request_handler_key)
      @direct_action_request_handler =
        request_handler(@direct_action_request_handler_key)
      @resource_request_handler =
        request_handler(@resource_request_handler_key)
      @default_request_handler = @component_request_handler
    end

    def init_adapter
      # decides interface of adapter
      if defined?(MOD_RUBY) then
        @adapter_class = Adapter::ModRuby
        @path = Apache.request.filename
      else
        @adapter_class = Adapter::CGI
        @path = $0
      end
    end

    # backward compatibility
    def init_component_paths
      @component_paths = ['components', '.']
    end

    def init_name_spaces
      @name_spaces = [CGIKit, Object]
      klass = Object
      self.class.to_s.split('::').each do |class_name|
        klass = klass.const_get(class_name)
        @name_spaces << klass
      end
    end

    # Returns the name of the application without file extension.
    def name
      File.basename( @path, '.*' )
    end

    def take_values_from_hash( hash )
      hash.each do |key, value|
        self[key] = value
      end
    end

    def template_store
      @template_store ||= @template_store_class.new(self)
    end

    def resource_store
      @resource_store ||= @resource_store_class.new(self)
    end

    # backward compatibility
    def load_all_components( path, subdir = false )
      __each_component_file(path, subdir) do |file|
        require(file)
      end
    end

    # backward compatibility
    def autoload_all_components( path, mod = Object, subdir = false )
      __each_component_file(path, subdir) do |file|
        class_name = File.basename(file, '.rb')
        mod.autoload(class_name, file)
      end
    end

    private

    # backward compatibility
    def __each_component_file(path, subdir)
      if subdir
        pattern = File.join(path, '**', '*.rb')
      else
        pattern = File.join(path, '*.rb')
      end
      
      Dir.glob(pattern) do |file|
        if /\.rb$/ === file and FileTest::readable?(file)
          yield file
        end
      end
    end
    
    public

    def load_configuration( path )
      load(path)
      configure
    end

    def parse_request_handler_key( request )
      key = nil
      if info = request.request_handler_path then
        info = info.reverse.chop.reverse
        separated = info.split('/')
        key = separated.first
      end
      key
    end


    #
    # hook
    #

    def init; end

    def configure; end


    #
    # managing sessions
    #

    # Session database object (SessionStore).
    def session_store
      @session_store ||= @session_store_class.new(self)
    end

    # Creates a session.
    def create_session( request )
      begin
        klass = @session_class || Session
        session = klass.new(request.session_id)
        session.awake_from_restoration(self, request)
        return session
      rescue
        if sid = request.session_id then
          msg = "for #{sid}"
        else
          msg = "because session id is not specified"
        end 
        raise SessionCreationError, "Failed creating session #{msg}."
      end
    end

    # Returns a restored session objects with session ID.
    def restore_session( session_id, context )
      begin
        session = session_store.checkout(session_id, context.request)
      rescue Exception => e
        raise SessionRestorationError, "Failed restoring session for #{session_id}"
      end
      if session then
        context.session = session
        context.session.awake_from_restoration(self, context.request)
        context.session.validate
      elsif session_id and session.nil? then
        raise SessionTimeoutError, 'Your session has timed out.'
      end
      session
    end

    # Saves the session, and set a cookie if "store_in_cookie" attribute is
    # setted. If "clear" method of the session is called, the session is deleted.
    def save_session( context )
      unless context.has_session? then return end

      if context.session.terminate? then
        sid = context.session.session_id
        @session_store.remove(sid)
      else
        session_store.checkin(context)
        if context.session.store_in_cookie? then
          context.session.set_cookie
        end
      end
    end


    #
    # handling requests
    #

    # Runs the application.
    def run( command_or_request = nil, response = nil )
      if CGIKit.const_defined?(:Command) and \
        CGIKit.const_get(:Command) === command_or_request then
        request = command_or_request.request
      else
        request = command_or_request
      end

      set_adapter()
      @adapter.run(request, response) do |request|
        info('=== begin request response Loop', false)
        begin  
          set_attributes_from_request(request)
          request.request_handler_key = parse_request_handler_key(request)
          handler = request_handler(request.request_handler_key)
          request.context_id = handler.context_id(request)
          response = handler.handle_request(request)

        rescue Exception => e
          debug("exception: #{e.message} (#{e.backtrace.inspect})")

          # create context without session
          request.session_id = nil
          context = @context_class.new(request, self)
          begin
            response = handle_error(e, context)
          rescue Exception => e
            # trap error occured by customized handle_error 
            response = default_error_page(e, context)
          end
        end

        info('=== end request response loop', false)
        response
      end
      response
    end

    def request_handler_key( handler )
      @request_handlers.index(handler)
    end

    def request_handler( key )
      unless handler = @request_handlers[key] then
        handler = @default_request_handler
      end
      handler
    end

    def register_request_handler( key, klass )
      if @request_handlers[key] then
        raise "request handler key '#{key}' is already registered."
      end
      @request_handlers[key] = klass.new(self)
    end

    def remove_request_handler( key )
      @request_handlers.delete(key)
    end

    def set_adapter
      if @set_adapter_and_handler then return end
      @adapter = create_adapter
      @set_adapter_and_handler = true
      unless @logger then
        if level = @log_options[:level] then
          require 'logger'
          case level
          when :fatal
            level = Logger::FATAL
          when :error
            level = Logger::ERROR
          when :warn
            level = Logger::WARN
          when :info
            level = Logger::INFO
          else
            level = Logger::DEBUG
          end
          if file = @log_options[:file] then
            out = File.open(file, 'a+')
          else
            out = @log_options[:out]
          end
          @logger = Logger.new(out, @log_options[:shift_age] || 0,
                               @log_options[:shift_size] || 1048576)
          @logger.progname = @log_options[:name]
        else
          @logger = DummyLogger.new
        end
      end
    end

    def set_attributes_from_request( request )
      if @set_attributes_from_request then return end
      @baseurl       = request.script_name              unless @baseurl
      @document_root = request.headers['DOCUMENT_ROOT'] unless @document_root
      set_resource_manager()
      @set_attributes_from_request = true
    end

    def set_resource_manager
      # backward compatibility
      unless @component_path then
        @component_paths.each do |path|
          if FileTest.exist?(path) then
            @component_path = path
          end
        end
      end
      @resource_path = @resources if @resources
      @web_server_resource_path = @web_server_resources unless @web_server_resources

      @resource_manager = ResourceManager.new(self, @main_package_options)
      @required_packages.each do |name|
        @resource_manager.load_package(name)
      end
    end

    def take_values_from_request( request, context )
      context.session.take_values_from_request(request, context)
    end

    def invoke_action( request, context )
      context.session.invoke_action(request, context)
    end

    def append_to_response( response, context )
      context.session.append_to_response(response, context)
    end

    def create_context( request )
      handler = request_handler(request.request_handler_key)
      context = @context_class.new(request, self)
      
      if session = restore_session(request.session_id, context) then
        context.session = session
      else
        session = create_session(request)
        context.session = session
      end
      session.context = context
      if component = context.session.restore_page(context.sender_id) then
        root = component.root
        context.component = root
        root.awake_from_restoration(context)
      else
        context.component = page(@main, context)
      end
      context.delete_all
      context
    end

    # Creates an adapter object.
    def create_adapter
      @adapter_class.new
    end

    def synchronize( &block )
      unless @concurrent_request_handling then
        @lock.enter
      end
      begin
        block.call
      ensure
        unless @concurrent_request_handling then
          @lock.exit
        end
      end
    end


    #
    # handling errors
    #

    # Handles every errors and return an error page component.
    def handle_error( error, context )
      case error
      when SessionCreationError 
        response = handle_session_creation_error(error, context)
      when SessionRestorationError 
        response = handle_session_restoration_error(error, context)
      when PageRestorationError 
        response = handle_page_restoration_error(error, context)
      else
        page = default_error_page(error, context)
        response = page.generate_response
      end
      response
    end

    def handle_session_creation_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_session_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_page_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    # Return a default error page component.
    def default_error_page( error, context )
      error_page       = page(@error_page, context)
      error_page.error = error
      error_page
    end

    def url( key, context, path, query_string, is_secure, port = nil, sid = true )
      request_handler(key).url(context, path, query_string, is_secure, port, sid)
    end

    def direct_action_url( context, action_class, action_name, query, sid = true )
      handler = @direct_action_request_handler
      handler.action_url(context, action_class, action_name, query, sid)
    end


    #
    # Creating components
    #

    # Creates a specified page component.
    def page( name, request_or_context, *args )
      context = request_or_context
      if Request === request_or_context then
        context = @context_class.new(request_or_context, self)
      end
      if Class === name then
        klass = name
      else
        klass = class_named(name)
      end
      klass.new(context, *args)
    end

    def class_named( name )
      Utilities.class_named_from(name, @name_spaces)
    end

  end

end

module CGIKit

# An API object has information of dynamic element's or component's bindings.
# The API object can validate declaration of the element or component at runtime.
class API

  attr_accessor :element_type, :content, :bindings, :validations

  def initialize( element_type, content = false )
    @element_type = element_type
    @content      = content
    @bindings     = {}
    @validations  = []
  end


  #
  # accessing
  #

  def []( name )
    @bindings[name]
  end

  def <<( object )
    case object
    when Binding
      @bindings[object.name] = object
    when Validation
      @validations << object
    end
  end


  #
  # testing
  #

  def has_binding?( name )
    @bindings.key?(name)
  end

  def special_binding_key?( key )
    Declaration::SPECIAL_BINDING_KEYS.include?(key)
  end


  #
  # validating
  #

  # Returns nil or array contains errors.
  def validate( associations, component_name, element_name )
    errors = []
    bindings = @bindings.values
    target = bindings + @validations
    target.each do |t|
      begin
        t.validate(associations, component_name, element_name, @element_type)
      rescue ValidationError => e
        errors << e
      end
    end

    if errors.empty? then
      nil
    else
      errors
    end
  end


  #
  # generating string of other tags
  #

  # Return a string of values of optional attributes with
  # "other" attribute string excepting the element's own attributes.
  def other( component, associations )
    optional = ''
    associations.each do |key, as|
      if !has_binding?(key) and !special_binding_key?(key) then
        value = as.value(component).to_s
        optional << " #{key}=\"#{value}\""
      end
    end

    if as = associations[Declaration::OTHER_KEY] then
      other = ' ' + as.value(component).to_s
    else
      other = ''
    end      

    optional + other
  end

  def optional_attributes( component, associations, prefix = nil )
    attrs = {}
    associations.each do |key, as|
      if !has_binding?(key) and !special_binding_key?(key) then
        if prefix.nil? or (prefix and /\A#{prefix}/ === key.to_s) then
          attrs[key] = as.value(component)
        end
      end
    end
    attrs
  end

  def other_attribute( component, associations, prefix = nil )
    key = prefix.nil? ? Declaration::OTHER_KEY : \
      "#{prefix}_#{Declaration::OTHER_KEY}".intern
    if as = associations[key] then
      as.value(component)
    else
      ''
    end
  end

  def enabled( component, associations )
    enabled = true
    associations.each do |key, as|
      if key == Declaration::ENABLED_KEY then
        enabled = as.bool(component)
      end
    end
    if enabled then
      ''
    else
      ' disabled="disabled"'
    end
  end

end


class Binding
  BOOLEAN               = 'Boolean'
  PAGE_NAMES            = 'Page Names'
  MIME_TYPES            = 'MIME Types'
  RESOURCES             = 'Resources'
  DIRECT_ACTIONS        = 'Direct Actions'
  DIRECT_ACTION_CLASSES = 'Direct Action Classes'
  PACKAGES              = 'Packages'

  attr_accessor :name, :required, :settable

  # Value type to set.
  attr_accessor :value_set

  # Default value.
  attr_writer :default

  def initialize( name )
    @name     = name
    @required = false
    @settable = false
    @default  = nil
  end

  # Returns the default value.
  # If the value is Symbol, it is resolved with key-value coding.
  def default( component = nil )
    if (Symbol === @default) and component then
      component.value_for_keypath(@default)
    else
      @default
    end
  end

  def validate( associations, component_name, element_name, element_type )
    msg = nil
    isbind = false

    associations.each do |key, as|
      if (key == @name) 
        if (@settable == true) and (as.settable? == false) then
          msg = "'#@name' must be bound to a settable value."
        end
        isbind = true
        break
      end
    end

    if (isbind == false) and (@required == true) then
      msg = "'#@name' is a required binding."
    end

    if msg then
      raise ValidationError.new(msg, component_name, element_name, element_type)
    end
  end
end


class Validation
  attr_accessor :message, :condition

  def initialize( message, condition = nil )
    @message = message
    @condition = condition
  end

  def validate( associations, component_name, element_name, element_type )
    if @condition.eval?(associations) then
      raise ValidationError.new(@message, component_name, element_name, element_type)
    end
  end
end


class KeyErrorCondition
  BOUND      = 'bound'
  UNBOUND    = 'unbound'
  SETTABLE   = 'settable'
  UNSETTABLE = 'unsettable'

  attr_reader :key, :error

  def initialize( key, error )
    @key = key
    @error = error
  end

  def eval?( associations )
    as = associations[@key]
    if ((as and (@error == BOUND)) or
        (as.nil? and (@error == UNBOUND)) or
       (as and as.settable? and (@error == SETTABLE)) or
       (as and as.constant? and (@error == UNSETTABLE))) then
      true
    else
      false
    end
  end
end


class AndCondition
  attr_reader :conditions

  def initialize( conditions )
    @conditions = conditions
  end

  def eval?( associations )
    @conditions.each do |condition|
      unless condition.eval? associations then
        return false
      end
    end
    true
  end
end


class OrCondition
  attr_reader :conditions

  def initialize( conditions )
    @conditions = conditions
  end

  def eval?( associations )
    @conditions.each do |condition|
      if condition.eval? associations then
        return true
      end
    end
    false
  end
end


class NotCondition
  attr_reader :condition

  def initialize( condition )
    @condition = condition
  end

  def eval?( associations )
    unless @condition.eval? associations then
      true
    else
      false
    end
  end
end


class ValidationError < StandardError
  attr_accessor :errors, :element_name, :element_type, :component_name

  def initialize( message = '',
                  component_name = nil,
                  element_name = nil,
                  element_type = nil )
    super(message)
    @component_name = component_name
    @element_name = element_name
    @element_type = element_type
    @errors = []
  end

  def <<( error )
    if Array === error then
      @errors.concat(error)
    else
      @errors << error
    end
  end

end


module APIUtilities

  #
  # binding
  #

  def href_binding
    Binding.new(:href)
  end

  def name_binding( required = false )
    binding = Binding.new(:name)
    binding.required = required
    binding
  end

  def page_binding( required = false )
    binding = Binding.new(:page)
    binding.required = required
    binding.value_set = Binding::PAGE_NAMES
    binding
  end

  def src_binding( required = false )
    binding = Binding.new(:src)
    binding.required = required
    binding
  end

  def action_binding( required = false )
    binding = Binding.new(:action)
    binding.required = required
    binding
  end

  def list_binding( required = true )
    binding = Binding.new(:list)
    binding.required = required
    binding
  end

  def item_binding()
    binding = Binding.new(:item)
    binding.settable = true
    binding
  end

  def display_binding()
    binding = Binding.new(:display)
    binding
  end

  def value_binding( required = true, settable = false )
    binding = Binding.new(:value)
    binding.required = required
    binding.settable = settable
    binding
  end

  def enabled_binding
    binding = Binding.new(:enabled)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end

  def escape_binding
    binding = Binding.new(:escape)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end

  def checked_binding( default = false )
    binding = Binding.new(:checked)
    binding.default = default
    binding.value_set = Binding::BOOLEAN
    binding.settable = true
    binding
  end

  def selection_binding( required = false )
    binding = Binding.new(:selection)
    binding.required = required
    binding.settable = true
    binding
  end

  def selections_binding( required = false )
    binding = Binding.new(:selections)
    binding.required = required
    binding.settable = true
    binding
  end

  def selected_value_binding
    binding = Binding.new(:selected_value)
    binding.settable = true
    binding
  end

  def selected_values_binding
    binding = Binding.new(:selected_values)
    binding.settable = true
    binding
  end

  def string_binding
    binding = Binding.new(:string)
    binding
  end

  def validate_binding
    binding = Binding.new(:validate)
    binding
  end

  def pass_binding
    binding = Binding.new(:pass)
    binding.value_set = Binding::BOOLEAN
    binding.settable = true
    binding.default = false
    binding
  end

  def direct_action_binding
    binding = Binding.new(:direct_action)
    binding.value_set = Binding::DIRECT_ACTIONS
    binding
  end

  def action_class_binding
    binding = Binding.new(:action_class)
    binding.value_set = Binding::DIRECT_ACTION_CLASSES
    binding
  end

  def session_id_binding
    binding = Binding.new(:session_id)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end

  def query_binding
    binding = Binding.new(:query)
    binding.default = :'{}'
    binding
  end

  def frag_binding
    Binding.new(:frag)
  end

  def secure_binding
    binding = Binding.new(:secure)
    binding.value_set = Binding::BOOLEAN
    binding.default = false
    binding
  end

  def file_binding
    binding = Binding.new(:file)
    binding.value_set = Binding::RESOURCES
    binding
  end

  def mime_binding
    binding = Binding.new(:mime)
    binding.value_set = Binding::MIME_TYPES
    binding
  end

  def x_binding
    binding = Binding.new(:x)
    binding.settable = true
    binding
  end

  def y_binding
    binding = Binding.new(:y)
    binding.settable = true
    binding
  end

  def package_binding( default = nil )
    binding = Binding.new(:package)
    binding.value_set = Binding::PACKAGES
    binding.default = default
    binding
  end


  #
  # setting
  #

  def set_validation( api )
    api << validate_binding()
    api << pass_binding()
    api << universal_validation(:validate, :pass)
  end

  def set_direct_action( api )
    api << direct_action_binding()
    api << action_class_binding()
    api
  end


  #
  # condition
  #

  def existential_condition( *names )
    conditions = []
    names.each do |name|
      bound = KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
      other_bounds = []
      names.each do |other|
        if name != other then
          other_bounds << KeyErrorCondition.new(other, KeyErrorCondition::BOUND)
        end
      end
      existentials = OrCondition.new(other_bounds)
      conditions << AndCondition.new([bound, existentials])
    end
    OrCondition.new(conditions)
  end

  def universal_condition( *names )
    conditions = []
    names.each do |name|
      bound = KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
      other_bounds = []
      names.each do |other|
        if name != other then
          other_bounds << KeyErrorCondition.new(other, KeyErrorCondition::UNBOUND)
        end
      end
      universals = OrCondition.new(other_bounds)
      conditions << AndCondition.new([bound, universals])
    end
    OrCondition.new(conditions)
  end

  # * or(ex1:bound, ex2:bound) or(un1:bound, un2:bound) or(both1:bound, both2:bound)
  # * on(and(ex, un), and(ex, both), and(un, both))
  def required_condition( existential = [], universal = [], both = [] )
    all_unbound = unbound_condition(existential + universal + both)
    any = any_condition(existential, universal, both)
    OrCondition.new([all_unbound, any])
  end

  def any_condition( existential = [], universal = [], both = [] )
    ex_any = any_bound_condition(existential)
    un_any = any_bound_condition(universal)
    both_any = any_bound_condition(both)

    ex_un = AndCondition.new([ex_any, un_any])
    ex_both = AndCondition.new([ex_any, both_any])
    un_both = AndCondition.new([un_any, both_any])

    ex_cond = existential_condition(*existential)
    un_cond = universal_condition(*universal)

    OrCondition.new([ex_un, ex_both, un_both, ex_cond, un_cond])
  end

  def unbound_condition( names )
    and_condition(names, KeyErrorCondition::UNBOUND)
  end

  def and_condition( names, error )
    conditions = []
    names.each do |name|
      conditions << KeyErrorCondition.new(name, error)
    end
    AndCondition.new(conditions)
  end

  def any_bound_condition( names )
    conditions = []
    names.each do |name|
      conditions << KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
    end
    OrCondition.new(conditions)
  end


  #
  # validation
  #

  def existential_validation( *names )
    list = join_names(names, 'and')
    msg = "#{list} can't both be bound."
    Validation.new(msg, existential_condition(*names))
  end

  def universal_validation( *names )
    list = join_names(names, 'and')
    msg = "#{list} must be bound when one of the both is bound."
    Validation.new(msg, universal_condition(*names))
  end

  def required_validation( existential = [], universal = [], both = [] )
    composed_validation(true, existential, universal, both)
  end

  def any_validation( existential = [], universal = [], both = [] )
    composed_validation(false, existential, universal, both)
  end

  def composed_validation( is_required, existential, universal, both )
    list_existential = join_names(existential, 'or')
    list_universal = join_names(universal, 'and')
    list_both = join_names(both, 'and')

    if is_required and universal.empty? and both.empty? then
      prefix = 'exactly'
      aux = 'must'
    else
      prefix = 'either'
      aux = 'may'
    end
    msg = "#{prefix} "
    unless existential.empty? then
      msg << "one of #{list_existential} #{aux} be bound"
      if (universal.empty? == false) or (both.empty? == false) then
        msg << ', or '
      end
    end
    unless both.empty? then
      msg << "both of #{list_universal} #{aux} be bound"
      if (both.empty? == false) then
        msg << ', or '
      end
    end
    unless both.empty? then
      msg << "either or both of #{list_both} #{aux} be bound"
    end
    msg << '.'

    if is_required then
      Validation.new(msg, required_condition(existential, universal, both))
    else
      Validation.new(msg, any_condition(existential, universal, both))
    end
  end

  def join_names( names, keyword )
    list = ''
    names.each do |name|
      if names.first == name then
        list << "'#{name}'"
      elsif names.last == name then
        list << " #{keyword} '#{name}'"
      else
        list << ", '#{name}'"
      end
    end
    list
  end

  def item_display_value_validation
    msg = "'item' must be bound when 'display' or 'value' is bound."
    item_condition = KeyErrorCondition.new(:item, KeyErrorCondition::UNBOUND)
    display_condition = KeyErrorCondition.new(:display, KeyErrorCondition::BOUND)
    value_condition = KeyErrorCondition.new(:value, KeyErrorCondition::BOUND)
    display_value = OrCondition.new([display_condition, value_condition])
    condition = AndCondition.new([item_condition, display_value])
    Validation.new(msg, condition)
  end

end

end

module CGIKit

class Delegate

  module ElementDelegate

    def element_will_take_values_from_request( request ); end
    def element_did_take_values_from_request( request ); end
    def element_will_invoke_action( request ); end
    def element_did_invoke_action( request ); end
    def element_will_append_to_response( response ); end
    def element_did_append_to_response( response ); end

  end

end


class Element
  include Delegatable, Logging

  class UnknownElementError < StandardError; end #:nodoc:
  class AttributeError      < StandardError; end #:nodoc:

  @@apis = {}

  attr_accessor :name, :node, :associations
  attr_reader :api

  class << self
    include APIUtilities

    def api
      unless @@apis[self] then
        @@apis[self] = create_api
      end
      @@apis[self]
    end

    # Implemented by subclasses. Returns a new API object for the element.
    def create_api; end

    def subclasses
      klasses = []
      ObjectSpace.each_object(Class) do |klass|
        if CGIKit::Element > klass then
          klasses << klass
        end
      end
      klasses
    end

    def loop_classes
      subclasses() << CGIKit::Application << CGIKit::Session
    end

    def loop_regexp
      /\A(take_values_from_request|invoke_action|append_to_response)/
    end

  end

  def initialize
    @api ||= self.class.api
  end

  def begin_context( context )
    context.increment
  end

  def end_context( context ); end

  def take_values_from_request( request, context ); end
  def invoke_action( request, context ); end
  def append_to_response( response, context ); end

  def delegate_take_values_from_request( request, context )
    ask(:element_will_take_values_from_request, context) do |d|
      d.element_will_take_values_from_request(request)
    end
    take_values_from_request(request, context)
    ask(:element_did_take_values_from_request, context) do |d|
      d.element_did_take_values_from_request(request)
    end
  end

  def delegate_invoke_action( request, context )
    ask(:element_will_invoke_action, context) do |d|
      d.element_will_invoke_action(request)
    end
    result = invoke_action(request, context)
    ask(:element_did_invoke_action, context) do |d|
      d.element_did_invoke_action(request)
    end
    result
  end

  def delegate_append_to_response( response, context )
    ask(:element_will_append_to_response, context) do |d|
      d.element_will_append_to_response(response)
    end
    append_to_response(response, context)
    ask(:element_did_append_to_response, context) do |d|
      d.element_did_append_to_response(response)
    end
  end

  def empty?
    @node.nil? or @node.empty?
  end
end


# The super class of dynamic element classes.
# Dynamic elements convert themselves to HTML.
# These are very important in development with CGIKit.
#
# == Paths for searching elements and components
# Element objects, including components, are usually instantiated
# by Element.instance. This method loads elements/components and creates
# element objects. In this method, some paths are searched to require files.
#
# The searched paths:
# 1. CKApplicaton#component_path
# 2. ($LOAD_PATH)/cgikit/elements
# 3. ($LOAD_PATH)/cgikit/components
#
# The latter two paths are for extention elements or components.
# It is recommended that your own elements or components are installed
# in the 1st path.
class DynamicElement < Element

  DELETE_KEYS = []

  attr_accessor :name, :root, :values, :context_id, :application, :parent

  class << self

    def keys_to_delete_from_associations
      DELETE_KEYS
    end

    def inherited( subclass )
      var = subclass.api_var
      src = "@@#{var} = nil;"
      src << "def self.api;@@#{var} ||= create_api();end"
      subclass.class_eval(src)
    end

    def api_var
      "#{self.to_s.split('::').last}_api"
    end

  end

  def initialize( name, associations, root )
    super()
    @name = name
    @associations = associations
    @root = root
    @values = {}
    @once = {}
    @application = @root.application
    self.class.keys_to_delete_from_associations.each do |key|
      @associations.delete(key)
    end
    if bindings = value(:delegate) then
      if Class === bindings then
        klass = bindings
        options = {}
      else
        klass = bindings[:class]
      end
      if Delegate > klass then
        unless klass.support?(self.class) then
          raise "#{klass} doesn't support #{self.class}."
        end
      else
        raise "Delegate class #{klass} must inherit CGIKit::Delegate."
      end
      @delegate = klass.new
      @delegate.bindings = bindings
      @delegate.element = self
      @delegate.component = root
    end
    init
  end


  #
  # request-response loop
  #

  def take_values_from_request( request, context )
    @node.take_values_from_request(request, context) if @node
  end

  def invoke_action( request, context )
    @node.invoke_action(request, context) if @node
  end

  # Returns value from the request if the context is continued.
  # If not be continued, returns nil.
  def value_from_request( request, context )
    values_from_request(request, context)[0]
  end

  def values_from_request( request, context )
    values = nil
    if (values = request.form_values[context.context_id]) and \
      (values.empty? == false) then
    elsif @values[:name] and context.in_form? and \
      (values = request.form_values[@values[:name].to_s]) and \
      (values.empty? == false) then
    end
    values ||= []

    if !(ByteData === values) and (values.empty? == false) and \
      (encoding = application.encoding) then
      values = encode_strings(values, encoding)
    end
    values
  end

  def encode_strings( strings, encoding )
    encoded = []
    strings.each do |string|
      begin
        encoded << (encode_string(string, encoding) || string)
      rescue Exception => e
        encoded << (failed_encode_string(string, encoding, e) || string)
      end
    end
    encoded
  end


  #
  # take values for binding keys
  #

  def take_value_once( name, does_resolve = true )
    unless @once[name] then
      @values[name] = value(name, does_resolve)
      @once[name] = true
    end
  end

  def take_value( name, does_resolve = true )
    @values[name] = value(name, does_resolve)
  end

  def take_action_name( name )
    take_value(name, false)
  end

  def take_value_query( name )
    take_value(name)
    @values[name] = {} unless Hash === @values[name]
    @associations.reject! do |key, as|
      if (@api.has_binding?(key) == false) and (/\A\?/ === key.to_s) then
        query_value = value(key)
        query_key = key.to_s.sub(/\A\?/, '').intern
        @values[name].delete(query_key.to_s)
        @values[name][query_key] = query_value
        true
      else
        false
      end
    end
    @values[name]
  end

  def take_bool( name, does_resolve = true )
    @values[name] = bool(name, does_resolve)
  end

  def value( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.value(root)
      else
        as.value
      end
    else
      if @api and @api[name] then
        @api[name].default(root)
      else
        nil
      end
    end
  end

  def bool( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.bool(root)
      else
        as.bool
      end
    else
      if @api and @api[name] then
        Association.adapt_to_bool(@api[name].default(root))
      else
        nil
      end
    end
  end

  def set_value( name, value )
    if as = @associations[name] then
      as.set_value(value, root)
    end
  end

  def name_value( context )
    escaped_string(@values[:name] || context.context_id)
  end

  def has_binding?( name )
    @associations.key?(name)
  end
  alias declared? has_binding?

  def direct_action?
    declared?(:direct_action) or declared?(:action_class)
  end


  #
  # escaping
  #

  def escaped_string( string, escape = true )
    if escape then
      string = Utilities.escape_html(string.to_s)
    end
    string
  end


  #
  # creating attributes' string
  #

  def other
    @api.other(root, associations)
  end

  def enabled
    @api.enabled(root, associations)
  end

  def checked
    " checked=\"checked\""
  end

  def selected
    " selected=\"selected\""
  end

  def optional_attributes( key = nil )
    @api.optional_attributes(root, associations, key)
  end

  def other_attribute( key = nil )
    @api.other_attribute(root, associations, key)
  end


  #
  # validating
  #

  def validate( name )
    take_value(name)
    take_value(Declaration::VALIDATE_KEY, false)
    take_value(Declaration::PASS_KEY, false)
    method = @values[Declaration::VALIDATE_KEY]
    pass = @values[Declaration::PASS_KEY]
    if method and pass then
      result = root.__send__(method, @values[name])
      set_value(Declaration::PASS_KEY, result)
    end
  end


  #
  # hook
  #

  def init; end

  # Convert character code for the form values.
  def encode_string( string, encoding ); end

  # Invoked when encode_string() raised an error.
  # This method should be return string.
  def failed_encode_string( string, encoding, error ); end

end

end


class Array

  def take_values_from_request( request, context )
    each do |item|
      item.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    result = nil
    each do |item|
      if action_result = item.invoke_action(request, context) then
        result = action_result
      end
    end
    result
  end
  
  def append_to_response( response, context )
    each do |item|
      item.append_to_response(response, context)
    end
  end

  def cache_copy
    select {|item| item.cache_copy}
  end

  def reset
    each do |item|
      item.reset
    end
  end

end

module CGIKit

class Component < Element
  include KeyValueCoding

  CACHE_TEMPLATE_DIR = 'cache'

  # Parent component.
  attr_reader :parent

  attr_accessor :declaration_store, :context

  attr_accessor :context_id, :subcomponents, :application, :declaration_name

  def initialize( context, *args )
    @access_attributes = true
    @subcomponents = []
    @associations = nil
    load_files(context)
    __send__(:init, *args)
  end


  #
  # hook
  #

  # Invoked to initialize component.
  # Form data is not setted when the method is called.
  def init; end

  # Invoked when the component parses template.
  # A string returned by the method is used instead of template file.
  def will_parse_template; end

  # Invoked when the component parses declarations.
  # A hash returned by the method is used instead of declaration file.
  def will_parse_declarations; end

  # Invoked when saving the component.
  # If the component has data can't be saved or marshaled with FileSessionStore,
  # you must clear the data in this method.
  def will_save_page; end

  # Invoked after restoring the component.
  def did_restore_page; end


  #
  # accessing
  #

  def root
    component = self
    while (component.root? == false)
      component = component.parent
    end
    component
  end

  def []( key )
    value_for_key(key)
  end

  def []=( key, value )
    take_value_for_key(key, value)
  end

  # Returns base name of the component in name space package.
  def base_name
    self.class.to_s.split('::').last
  end

  # Creates a specified page component.
  def page( name, *args )
    @application.page(name, @context, *args)
  end

  def parent=( parent )
    @parent = parent
    @parent.subcomponents << self
  end

  # Path for the component.
  def path
    @path ||= package.path_for_component(base_name(), request.languages)
  end


  #
  # testing
  #

  def root?
    @parent.nil?
  end

  def subcomponent?
    @associations and @parent
  end

  # Return true if parent component syncronizes bound attribute to self.
  # Override this method returns false to create a non-synchronizing component.
  def sync?
    true
  end

  def stateless?
    false
  end

  def has_session?
    @context.has_session?
  end

  def loaded?
    @loaded
  end

  def has_binding?( name )
    @associations.key?(name)
  end

  def can_set_binding_value?( name )
    has_binding?(name) and Symbol === binding_value(name, false)
  end

  def can_get_binding_value?( name )
    has_binding?(name)
  end


  #
  # loading
  #

  # loads template, merges declaration hash, loads declaration file
  def load_files( context )
    @context = context
    @application = context.application

    template = @application.template_store.checkout(self)
    @template_node = template.template_node
    @declaration_store = template.declaration_store

    load_associations_from_parent_declaration
    validate_api
    @loaded = true
  end

  def load_associations_from_parent_declaration
    if parent and @declaration_name then
      @associations = parent.declaration_store[@declaration_name]
    end
  end

  def validate_api
    if application.validate_api? then
      @declaration_store.validate_api(self.class)
    end
  end


  #
  # request-response loop
  #

  def begin_context( context )
    increment_and_record_if_subcomponent(context)
    init_context_id(context)
  end

  def increment_and_record_if_subcomponent( context )
    if subcomponent? then
      context.increment
      @parent_context_id = context.context_id
    end
  end

  def init_context_id( context )
    context.delete_all
    if context.has_session? then
      context.init_component_id_with_session(self)
    else
      context.init_component_id_without_session(self)
    end
    context.append_zero
  end

  def end_context( context )
    if subcomponent? then
      context.component = parent
      context.context_id = @parent_context_id
    end
  end

  def sync( context, &block )
    unless loaded? then
      awake_from_restoration(context)
    end
    if subcomponent? and sync? then
      pull_values_from_parent
      result = block.call if block_given?
      push_values_to_parent
      result
    elsif block_given?
      block.call
    end
  end

  def take_values_from_request( request, context )
    sync(context) do
      @template_node.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    sync(context) do
      @template_node.invoke_action(request, context)
    end
  end

  def append_to_response( response, context )
    sync(context) do
      if !stateless? and context.has_session? then
        context.session.save_page(self) 
      end
      @template_node.append_to_response(response, context)
    end
  end

  def generate_response
    info()
    response = Response.new_with_response(response())
    handler = application.component_request_handler
    handler.append_to_response(self, response, context)
    response.component = self
    response
  end

  def to_html
    generate_response.content
  end


  #
  # marshaling
  #

  def dup
    component = super
    component.context = @context
    component.declaration_store = @declaration_store
    component.node = @template_node
    component
  end

  def marshal_dump
    will_save_page
    not_includes = ['@context', '@declaration_store', '@template_node',
      '@node', '@application', '@loaded']
    dump = {}
    instance_variables.each do |var|
      if not_includes.include?(var) == false then
        dump[var] = instance_variable_get(var)
      end
    end
    dump
  end

  def marshal_load( object )
    object.each do |key, value|
      instance_variable_set(key, value)
    end
  end

  # Invoked after the component is restored to initialize.
  # This method invokes deletage method did_restore_page() finally.
  def awake_from_restoration( context )
    @context = context
    @application = context.application
    debug(object_id(), true)
    load_files(context)
    @subcomponents.each do |sub|
      sub.awake_from_restoration(context)
    end
    did_restore_page
  end

  def add_all_components( session, permanently = false )
    unless stateless? then
      session.add_component(self, @context, permanently)
      @subcomponents.each do |sub|
        sub.add_all_components(session, permanently)
      end
    end
  end


  #
  # aliases
  #

  # Returns a request object of the context.
  def request; @context.request; end

  # Returns a response object of the context.
  def response; @context.response; end

  # Returns the session object.
  # If the session isn't existed, returns a new session.
  def session; @context.session; end

  # Returns a resource manager object.
  def resource_manager; @application.resource_manager; end


  #
  # template paths
  #

  def package
    @package ||= resource_manager().package(self.class)
  end

  # Returns name of the template file.
  def template_file
    package().path_for_component("#{base_name()}.html", request().languages)
  end

  # Returns name of the declaration( binding ) file.
  def declaration_file
    package().path_for_component("#{base_name()}.ckd", request().languages)
  end


  #
  # Synchronizing components
  #

  def next_subcomponent( context_id )
    if @context_id == context_id then
      return @subcomponents.first
    end
    @subcomponents.each_with_index do |component, index|
      if context_id == component.context_id then
        return @subcomponents[index]
      end
    end
    nil
  end

  # Allows a subcomponent to invoke an action method of its parent component
  # bound to the child component.
  def perform_parent_action( name )
    @parent.value_for_key(name) if sync?
  end

  def pull_values_from_parent
    @associations.each do |key, as|
      if @api and @api.has_binding?(key) and as.settable_in_component?(self) then
        take_value_for_key(key, as.value)
      else
        take_value_for_key(key, as.value(@parent))
      end
    end
  end

  def push_values_to_parent
    @associations.each do |key, as|
      if (key != Declaration::ELEMENT_KEY) and @api.nil? or \
        (@api and @api.has_binding?(key) and \
         (as.settable_in_component?(self) == false)) then
        @parent.take_value_for_key(key, value_for_key(key))
      end
    end
  end

  def binding_value( name, does_resolve = true )
    if as = @associations[name] then
      if does_resolve then
        as.value(@parent)
      else
        as.value
      end
    else
      nil
    end
  end

  def set_binding_value( name, value )
    if as = @associations[name] then
      as.set_value(value, @parent)
    end
  end


  #
  # message accessing
  #

  def message( key, name = nil, package_name = nil, languages = nil )
    package_name ||= package().name
    languages ||= request().languages
    resource_manager().message(key, name, package_name, languages)
  end
  alias _ message

  def nmessage( key, plural_key, n, name = nil, package_name = nil, languages = nil )
    package_name ||= package().name
    languages ||= request().languages
    resource_manager().nmessage(key, plural_key, n, name, package_name, languages)
  end
  alias n_ nmessage

end


class StatelessComponent < Component

  def sync?
    false
  end

  def stateless?
    true
  end

end


  # backward compatibility
  class ErrorPage < Component; end
  CKErrorPage = ErrorPage

end
if $0 == __FILE__
  require 'cgikit'
  require 'cgikit/lang/ja'  
end

module CGIKit::HTMLParser
  
  def self.attribute_string(attrs)
    str = ''
    attrs.each do |key, value|
      str << " #{key}=\"#{value}\""
    end
    str
  end
  
  # CGIKit::HTMLParser::Node stands for nodes of Template's tree structure.
  # Node is base class. Actually, its subclasses are used in tree structure.
  class Node
    
    attr_accessor :node, :name, :content, :parent, :attributes
    
    def initialize( name = nil )
      @node = []
      @name = name || ''
      @content = ''
    end
    
    # returns the `index`th child node.  
    def []( index )
      @node[index]
    end
    
    # returns the number of children.
    def size
      @node.size
    end
    
    # returns the last child node.
    def last
      @node.last
    end
    
    # returns the root node of Template's tree structure. This must be the CGIKit::HTMLParser::RootNode.
    def root
      cur = self
      while cur.parent
        cur = cur.parent        
      end
      cur
    end
    
    # returns true if `other` is the same. 
    def ==( other )
      (@name == other.name) and (@node == other.node)
    end
    
    # adds node as child. 
    def <<( node )
      node.parent = self
      @node << node
    end
    
    # returns true if `self` has no children.
    def empty?
      @node.nil? or @node.empty?
    end
    
    # sets values of request object to node if node is CGIKit element.
    def take_values_from_request( request, context )
      @node.each do |n|
        n.take_values_from_request( request, context )
      end
    end
    
    # invokes action method of node if node is CGIKit element.
    def invoke_action( request, context )
      result = nil
      @node.each do |node|
        if node_result = node.invoke_action(request, context) then
          result = node_result
        end
      end
      result
    end
    
    # adds HTML to response object
    def append_to_response( response, context )
      @node.each do |node|
        node.append_to_response(response, context)
      end
    end
        
    # converts `self` to HTML 
    def to_s
      string = ''
      @node.each do |node|
        string << node.to_s
      end
      string
    end
   
    def cache_copy
      copy = self.class.new(@name)
      copy.attributes = @attributes
      copy.content = @content
      @node.each do |node|
        copy << node.cache_copy
      end
      copy
    end

    def reset
      if @node then
        @node.each do |node|
          node.reset
        end
      end
    end
    
  end
  
  # This class is root node of Template's tree structure.
  # Of all nodes, this node only has reference to CGIKit::Component object.
  class RootNode < Node
    attr_accessor :component
    
    def marshal_dump
      dump = {}
      (instance_variables - ['@component']).each do |var|
        dump[var] = instance_variable_get(var)
      end
      dump
    end

    def marshal_load( object )
      object.each do |key, value|
        instance_variable_set(key, value)
      end
    end
    
    def parent
      nil
    end
    
  end
  
  # This class has represents text. The role of this class is that
  # when CGIKit expands component's template, TextNode appends its `content` to `response` object.
  class TextNode < Node
    
    def initialize(name = nil)
      @content = ''
    end
    
    def take_values_from_request( request, context )
    end
    
    def invoke_action( request, context )
    end
    
    def append_to_response( response, context )
      response.content << @content
    end
    
    def to_s
      @content
    end
    
    def cache_copy
      copy = self.class.new(@name)
      copy.content = @content
      copy
    end
    
  end
  
  
  class CGIKitNode < Node
    
    attr_accessor :element, :ckid
    
    def initialize( name = nil )
      if ::String === name then
        name = name.intern
      end
      super
    end
    
    def take_values_from_request( request, context )
      element = element_for_context(context)
      handler(context).take_values_from_request(element, request, context)
    end
    
    def invoke_action( request, context )
      element = element_for_context(context)
      if result = handler(context).invoke_action(element, request, context) then
        set_result(result)
      end
      result
    end
    
    def set_result( result )
      @element = result
    end
    
    def append_to_response( response, context )
      element = element_for_context(context)
      unless element then return end
      handler(context).append_to_response(element, response, context)
    end

    def element_for_context( context )
      unless @element then
        @element = create_element(context)
        @element.node = @node
      end
      @element
    end
    
    def create_element( context )
      klass = element_class(context)
      unless klass then
        raise "Element class for #{context.parent.class}:#{content} is not specified."
      end

      if CGIKit::DynamicElement > klass then
        element = create_dynamic_element(klass, context)
      else
        element = create_subcomponent(klass, context)
      end

      init_element_for_form(element)
      element
    end
    
    def create_dynamic_element( klass, context )
      klass.new(@ckid, associations(context), root.component)
    end
    
    def create_subcomponent( klass, context )
      element = context.component.next_subcomponent(context.context_id)
      unless element then
        element = klass.new(context)
        element.parent = root.component
        element.context_id = context.context_id
      end
      element.declaration_name = @ckid
      element.associations = associations(context)
      element
    end
    
    def init_element_for_form( element )
      if (CGIKitNode === @parent) and (CGIKit::DynamicElement === @parent.element) then
        if CGIKit::Component === element             
          element.parent = root.component    
        else
          element.parent = @parent.element
        end
      end
    end
    
    def element_class( context )
      type = declaration_store.element_type
      if Class === type then
        klass = type
      else
        evaluated = root.component.value_for_keypath(type)
        if Class === evaluated then
          klass = evaluated
        else
          klass = context.application.class_named(evaluated)
        end
      end
      klass
    end
    
    def declaration_store
      unless dec = root.component.declaration_store[@ckid] then
        raise "Element '#@ckid' is not declared in '#{root.component.class}'."
      end
      dec
    end
    
    def associations( context )
      declaration_store.association_hash
    end
    
    def handler( context )
      context.application.request_handler(context.request.request_handler_key)
    end
    
    def to_s
      content = super
      if content.empty? then
        "<#@name#{CGIKit::HTMLParser.attribute_string(@attributes)} />"
      else
        "<#@name#{CGIKit::HTMLParser.attribute_string(@attributes)}>#{super}</#@name>"
      end
    end
    
    def cache_copy
      copy = super
      copy.ckid = @ckid
      copy
    end

    def reset
      @element = nil
      @node.each do |node|
        node.reset
      end
    end

  end
  
  
  
  class HTMLParser
    
    class ParseError < CGIKit::CGIKitError; end #:nodoc:
    
    def self.cgikit_attribute
      :ckid      
    end
    
    def self.cgikit_regexp
      /\Ack:/u
    end
    
    def self.parse_comment_reg
      /\A\s*ck\s/u
    end
    
    def self.element_attribute; CGIKit::Declaration::ELEMENT_KEY; end
    
    attr_accessor :html_string, :declarations, :node, :encoding, \
    :last_token, :next_token, :inner_nodes
    
    def initialize( filename = nil )
      @filename = filename
      
      @ckid = self.class.cgikit_attribute.to_s
      @id_reg = self.class.cgikit_regexp
      @parse_comment_reg = self.class.parse_comment_reg
      
      if filename then
        string = nil
        #CGIKit::FileLock.shared_lock(filename) do |f|
        File.open(filename) do |f|
          string = f.read
        end
        parse(string)
      end
    end
    
    def init_listener
      @buf = []
      @declarations = {}
      
      @tag_level = 1
      @node2tag_level = Hash.new(0)
      @in_doctype = false
      @html_string = ''
      
      @doctype_buf = []
    end
    
    def parse(string)
      unless Object.const_defined?('REXML')
        require 'rexml/document'
        require 'rexml/streamlistener'
        if CGIKit::Application.respond_to?(:precede_iconv_as_rexml_encoding_module) and not(CGIKit::Application.precede_iconv_as_rexml_encoding_module)
          require 'cgikit/lang/encoding-patch.rb'
        end
      end
      
      @html_string = string
      init_listener
      
      @node = CGIKit::HTMLParser::RootNode.new
      @cur = @node
      
      @parser = REXML::Parsers::BaseParser.new(string)

      @encoding = Thread.current[:ck_root_component_encoding]
      @root_component_parsed = Thread.current[:ck_root_component_parsed]

      begin
        __parse
      rescue REXML::ParseException
        raise CGIKit::HTMLParser::HTMLParser::ParseError, "REXML raises Error when parsing #{@filename}.\nREXML error message: #{$!.to_s.gsub(/\r?\n/, ' ')}\n"
      end
      
      unless Thread.current[:ck_root_component_parsed]
        Thread.current[:ck_root_component_encoding] = @encoding
        Thread.current[:ck_root_component_parsed] = true
      end

      if @buf.size > 0
        @cur << buffer2textnode 
      end
      
      self.node
    end
    
    #
    # use REXML::Parsers::BaseParser API
    # copied from REXML::Parsers::TreeParse and StreamParser
    
    # This is dirty but may be a bit faster.
    def __parse
      name = attrs = nil
      # entity string
      while true
        event = @parser.pull
        case event[0]
        when :end_document
          # end of XML
          return
        when :start_element
          # not normalize
          @tag_level += 1
          name = event[1]
          attrs = event[2]
          
          if match_pattern = self.cgikit_element?(name, attrs)
            if tn = buffer2textnode
              @cur << tn
            end
            
            new_node = CGIKit::HTMLParser::CGIKitNode.new(name)
            
            ck_attrs = {}
            attrs.each do |k,v|
              ck_attrs[k.intern] = self.value_for_string(v)
            end
            case match_pattern
            when :id
              ck_attrs[@ckid.intern] = attrs['id'].sub(@id_reg, '')
            when :ns_id
              ck_attrs[@ckid.intern] = attrs['ck:id']
              ck_attrs.delete(:'ck:id')
            end
            
            new_node.attributes = ck_attrs
            add_declaration(new_node.name, new_node.attributes)
            new_node.ckid = new_node.attributes[@ckid.intern].intern
            
            @node2tag_level[@cur] = @tag_level - 1
            @cur << new_node
            @cur = new_node
            @node2tag_level[new_node] = 1
            @tag_level = 1
          else
            @buf << "<#{name}#{CGIKit::HTMLParser.attribute_string(attrs)}>"
          end
        when :end_element
          @tag_level -= 1
          name = event[1]
          if @tag_level == 0
            if node = buffer2textnode
              @cur << node
            end
            
            unless RootNode === @cur
              @cur = @cur.parent
            end
            @tag_level = @node2tag_level[@cur]
          else
            if (@buf.size != 0) and (@buf.last.index("<#{name}") == 0)
              s = @buf.pop
              ss = s.sub(/>\z/um, ' />')
              @buf << ss
            else
              @buf << "</#{name}>"
            end
          end
        when :text
          unless @in_doctype
            # not normalize
            @buf << event[1]
          end
        when :end_doctype
          @in_doctype = false
          end_doctype
        when :start_doctype
          @in_doctype = true                    
          start_doctype( *event[1..-1] )
        when :processing_instruction
          instruction( event[1], event[2] )
        when :externalentity
          externalentity( event[1] )
        when :elementdecl
          elementdecl(event[1])
        when :entitydecl
          entitydecl(event)
        when :comment, :attlistdecl, :cdata, :xmldecl, :notationdecl, :entitydecl
          #__send__( event[0].to_s, *event[1..-1] )
          __send__( event[0], *event[1..-1] )
        else
          raise CGIKit::HTMLParser::HTMLParser::ParseError, "#{@filename}: CGIKit doesn't recognize the event(#{event.inspect})"
        end
      end
    end
  
    #def tag_start(name, attrs)
    #end
    #def tag_end(name)     
    #end
    #def text(text)
    #end
    
    def instruction(name, instruction)
      @buf << %Q|<?#{name}#{instruction}?>|
    end
    
    def comment(comment)
      # comment has already been converted to UTF-8.
      @buf <<  '<!--'
      
      if @parse_comment_reg =~  comment
        @cur << buffer2textnode
        
        s = comment.sub(@parse_comment_reg, '')
        # don't need to set encoding. 
        parser = REXML::Parsers::BaseParser.new(s)
        org = @parser
        @parser = parser 
        __parse
        @parser = org
      else
        @buf << comment
      end
      
      @buf << '-->'
    end
    
    def start_doctype(name, pub_sys, long_name, uri)
      if tn = buffer2textnode
        @cur << tn
      end
      
      s = ''
      s  << "<!DOCTYPE #{name} #{pub_sys}"
      if long_name
        s << ' '
        s << long_name
      end
      if uri
        s << ' '
        s << uri
      end
      
      # for the time being, "[" is used.
      s << '['
      
      @buf << s
    end
    
    def end_doctype
      if REXML::Parsers::BaseParser::DOCTYPE_START =~ @buf.last
        # There is no `markupdecl`
        s = @buf.pop
        ss = s.sub(/\[\z/um, '>')
        @buf << ss
      else
        @buf << ']>' 
      end
    end
    
    def externalentity(content)
      @buf << (content + "\n")
    end    
    
    def elementdecl(content)
      @buf << (content + '>')
    end
    
    def attlistdecl(element_name, attributes, raw_content)
      @buf << raw_content
    end
    
    def entitydecl(event)
      s = ''
      REXML::Entity.new(event).write(s)
      @buf << s
    end
    
    def notationdecl(name, middle, rest)
      @buf << "<!NOTATION #{name} '#{middle} #{rest}'>"
    end
    
    def entity(content)
      @buf << %Q|%#{content};|
    end
    
    def cdata(content)
      @buf << %Q|<![CDATA[#{content}]]>|
    end
    
    def xmldecl(version, encoding, standalone)
      s = '<?xml '

      if version
        s << %Q|version="#{version}"|
      end
      
      if encoding
        if @encoding and @encoding != encoding
          raise ParseError, "#{@filename}: charcter encoding does not match that of root component.\nRoot component is #{@encoding}.\n#{@filename} is #{encoding}\n"
        end
        @encoding = encoding
        s  << %Q| encoding="#{encoding}"|
      end
      
      if standalone
        s << %Q| standalone="#{standalone}"|
      end
      
      s <<  %Q|?>|
      @buf << s
    end
    
    #
    # end of BaseParser API
    #
    
    
    def cgikit_element?(tag, attrs)
      if attrs.size == 0
        false
      else	
        if attrs.key?(@ckid)
          :ckid
        elsif attrs.key?('ck:id')
          :ns_id
        elsif @id_reg =~ attrs['id'] 
          :id
        else
          false
        end
      end
    end
    
    def buffer2textnode
      if @buf.size > 0
        tn = CGIKit::HTMLParser::TextNode.new(nil)
        o = REXML::Output.new(tn.content, @encoding)
        
        # REXML::Output encodes `@buf` and add its result to `tn.content`
        o << @buf.join('')
        
        @buf.clear
        tn
      else
        nil
      end
    end
    
    def add_declaration( tag, ck_attrs )
      dec = {}
      name = ck_attrs[@ckid.intern].intern
      
      #if klass = class_for_element(tag, ck_attrs) 
      #	 dec[self.class.element_attribute] = klass
      #end			
      
      keys = ck_attrs.keys - [@ckid.intern]
      keys.each do |key|
        value = ck_attrs[key]
        if key == self.class.element_attribute then
          dec[key] = class_for_name(value)
        else
          dec[key] = value_for_string(value)
        end
      end
      
      @declarations[name] = dec
    end
    
    def class_named( name )
      CGIKit::Utilities.class_named_from(name)
    end
    alias class_for_name class_named
    
    def class_for_element( tag_type, attributes )
      case tag_type
      when :a        then Link
      when :form     then Form
      when :img      then Image
      when :textarea then Text
      when :select then
        if attributes[:multiple] or attributes.key?(:size) then
          Browser
        else
          Popup
        end
      when :input then
        unless attributes[:type] then
          return TextField
        end
        case attributes[:type].downcase
        when 'text'     then TextField
        when 'password' then TextField
        when 'hidden'   then TextField
        when 'checkbox' then Checkbox
        when 'radio'    then Radio
        when 'submit'   then Submit
        when 'reset'    then Reset
        when 'file'     then Upload
        else
          TextField
        end
      end
    end
    
    def value_for_string( value )
      case value
      when /\A:/ then
        value.sub(/\A:/, '').intern
      when /\A\s*ruby\s*:/ # it should be add `i` option
        value.sub(/\A\s*ruby\s*:/, '').intern
      when 'true' then
        true
      when 'false' then
        false
      when 'nil' then
        nil
      else
        value
      end
    end		
    
  end
  
end


if $0 == __FILE__
  require 'pp'
  
  parser = CGIKit::HTMLParser::HTMLParser.new(ARGV[0])
  #pp parser.node
  pp parser.declarations
end
module CGIKit

class DeclarationStore
  class DeclarationError < StandardError #:nodoc:
  end

  class << self
    def new_with_file( filename, source = nil )
      hash = nil
      str = nil
      open(filename) { |f| str = f.read }
      Thread.start(str) do |s|
        $SAFE = 4
        hash = eval(s)
      end.join
      hash ||= {}
      merge_source(source, hash) if source
      new_from_hash(hash)
    end

    def merge_source( source, destination )
      source.each do |decname, dechash|
        if destination.key?(decname) then
          dechash.each do |attrname, attrvalue|
            unless destination[decname].key?(attrname) then
              destination[decname][attrname] = attrvalue
            end
          end
        else
          destination[decname] = dechash
        end
      end
    end

    def new_from_hash( hash )
      decs = new
      hash.each do |name, values|
        decs[name] = Declaration.new_from_hash(name, values)
      end
      decs.original_hash = hash
      decs
    end
  end

  attr_accessor :declarations, :original_hash

  def initialize
    @declarations = {}
  end

  def []( key )
    @declarations[key]
  end

  def []=( key, declaration )
    @declarations[key] = declaration
  end

  def each
    @declarations.each do |key, declaration|
      yield key, declaration
    end
  end

  def keys
    @declarations.keys
  end

  def validate_api( component_class = nil )
    msg = "Found validation errors in \"#{component_class}\"."
    error = ValidationError.new(msg)
    keys = @declarations.keys
    keys.map! {|key| key.to_s}
    keys.sort.each do |key|
      dec = @declarations[key.intern]
      if api = dec.element_type.api then
        errors = api.validate(dec.association_hash, component_class, key)
        if errors then
          error << errors
        end
      end
    end

    unless error.errors.empty? then
      raise error
    end
  end

end


# A Declaration object has a declaration set.
#
#  ex)
#  :BindingName => {
#    :element => String,        # element type (class)
#    :value => :'method.chain', # symbol value is dealt as method chain
#    :empty => false            # other value is dealt as constant value
#  }
#    
class Declaration
  attr_accessor :element_name, :element_type, :association_hash

  ELEMENT_KEY  = :element
  OTHER_KEY    = :other
  VALIDATE_KEY = :validate
  PASS_KEY     = :pass
  ENABLED_KEY  = :enabled
  DELEGATE_KEY = :delegate
  SPECIAL_BINDING_KEYS = [ELEMENT_KEY, OTHER_KEY, DELEGATE_KEY]

  class << self
    def new_from_hash( name, hash )
      unless hash.key?(ELEMENT_KEY) then
        raise "'#{name}' don't define #{ELEMENT_KEY.inspect}."
      end
      unless Class === hash[ELEMENT_KEY] then
        raise "value of '#{ELEMENT_KEY}' must be class."
      end
      dec = Declaration.new(name, hash[ELEMENT_KEY])
      ass = nil
      hash.each do |key, value|
        if Symbol === value then
          ass = Association.new_with_keypath value
        else
          ass = Association.new_with_value value
        end
        dec[key] = ass
      end
      dec
    end
  end

  def initialize( name, type )
    @element_name = name
    @element_type = type
    @association_hash = {}
  end

  def []( name )
    @association_hash[name]
  end

  def []=( name, value )
    @association_hash[name] = value
  end

  def each
    @association_hash.each do |key, as|
      yield key, as
    end
  end
end

end
module CGIKit

  # ResourceManager class manages resources of an application.
  class ResourceManager
    include Logging

    MIME = {
      'ez'      => 'application/andrew-inset',
      'hqx'     => 'application/mac-binhex40',
      'cpt'     => 'application/mac-compactpro',
      'doc'     => 'application/msword',
      'bin'     => 'application/octet-stream',
      'dms'     => 'application/octet-stream',
      'lha'     => 'application/octet-stream',
      'lzh'     => 'application/octet-stream',
      'exe'     => 'application/octet-stream',
      'class'   => 'application/octet-stream',
      'so'      => 'application/octet-stream',
      'dll'     => 'application/octet-stream',
      'oda'     => 'application/oda',
      'pdf'     => 'application/pdf',
      'ai'      => 'application/postscript',
      'eps'     => 'application/postscript',
      'ps'      => 'application/postscript',
      'smi'     => 'application/smil',
      'smil'    => 'application/smil',
      'mif'     => 'application/vnd.mif',
      'xls'     => 'application/vnd.ms-excel',
      'ppt'     => 'application/vnd.ms-powerpoint',
      'wbxml'   => 'application/vnd.wap.wbxml',
      'wmlc'    => 'application/vnd.wap.wmlc',
      'wmlsc'   => 'application/vnd.wap.wmlscriptc',
      'bcpio'   => 'application/x-bcpio',
      'vcd'     => 'application/x-cdlink',
      'pgn'     => 'application/x-chess-pgn',
      'cpio'    => 'application/x-cpio',
      'csh'     => 'application/x-csh',
      'dcr'     => 'application/x-director',
      'dir'     => 'application/x-director',
      'dxr'     => 'application/x-director',
      'dvi'     => 'application/x-dvi',
      'spl'     => 'application/x-futuresplash',
      'gtar'    => 'application/x-gtar',
      'hdf'     => 'application/x-hdf',
      'js'      => 'application/x-javascript',
      'skp'     => 'application/x-koan',
      'skd'     => 'application/x-koan',
      'skt'     => 'application/x-koan',
      'skm'     => 'application/x-koan',
      'latex'   => 'application/x-latex',
      'nc'      => 'application/x-netcdf',
      'cdf'     => 'application/x-netcdf',
      'sh'      => 'application/x-sh',
      'shar'    => 'application/x-shar',
      'swf'     => 'application/x-shockwave-flash',
      'sit'     => 'application/x-stuffit',
      'sv4cpio' => 'application/x-sv4cpio',
      'sv4crc'  => 'application/x-sv4crc',
      'tar'     => 'application/x-tar',
      'tcl'     => 'application/x-tcl',
      'tex'     => 'application/x-tex',
      'texinfo' => 'application/x-texinfo',
      'texi'    => 'application/x-texinfo',
      't'       => 'application/x-troff',
      'tr'      => 'application/x-troff',
      'roff'    => 'application/x-troff',
      'man'     => 'application/x-troff-man',
      'me'      => 'application/x-troff-me',
      'ms'      => 'application/x-troff-ms',
      'ustar'   => 'application/x-ustar',
      'src'     => 'application/x-wais-source',
      'xhtml'   => 'application/xhtml+xml',
      'xht'     => 'application/xhtml+xml',
      'zip'     => 'application/zip',
      'au'      => 'audio/basic',
      'snd'     => 'audio/basic',
      'mid'     => 'audio/midi',
      'midi'    => 'audio/midi',
      'kar'     => 'audio/midi',
      'mpga'    => 'audio/mpeg',
      'mp2'     => 'audio/mpeg',
      'mp3'     => 'audio/mpeg',
      'aif'     => 'audio/x-aiff',
      'aiff'    => 'audio/x-aiff',
      'aifc'    => 'audio/x-aiff',
      'm3u'     => 'audio/x-mpegurl',
      'ram'     => 'audio/x-pn-realaudio',
      'rm'      => 'audio/x-pn-realaudio',
      'rpm'     => 'audio/x-pn-realaudio-plugin',
      'ra'      => 'audio/x-realaudio',
      'wav'     => 'audio/x-wav',
      'pdb'     => 'chemical/x-pdb',
      'xyz'     => 'chemical/x-xyz',
      'bmp'     => 'image/bmp',
      'gif'     => 'image/gif',
      'ief'     => 'image/ief',
      'jpeg'    => 'image/jpeg',
      'jpg'     => 'image/jpeg',
      'jpe'     => 'image/jpeg',
      'png'     => 'image/png',
      'tiff'    => 'image/tiff',
      'tif'     => 'image/tiff',
      'djvu'    => 'image/vnd.djvu',
      'djv'     => 'image/vnd.djvu',
      'wbmp'    => 'image/vnd.wap.wbmp',
      'ras'     => 'image/x-cmu-raster',
      'pnm'     => 'image/x-portable-anymap',
      'pbm'     => 'image/x-portable-bitmap',
      'pgm'     => 'image/x-portable-graymap',
      'ppm'     => 'image/x-portable-pixmap',
      'rgb'     => 'image/x-rgb',
      'xbm'     => 'image/x-xbitmap',
      'xpm'     => 'image/x-xpixmap',
      'xwd'     => 'image/x-xwindowdump',
      'igs'     => 'model/iges',
      'iges'    => 'model/iges',
      'msh'     => 'model/mesh',
      'mesh'    => 'model/mesh',
      'silo'    => 'model/mesh',
      'wrl'     => 'model/vrml',
      'vrml'    => 'model/vrml',
      'css'     => 'text/css',
      'html'    => 'text/html',
      'htm'     => 'text/html',
      'asc'     => 'text/plain',
      'txt'     => 'text/plain',
      'rtx'     => 'text/richtext',
      'rtf'     => 'text/rtf',
      'sgml'    => 'text/sgml',
      'sgm'     => 'text/sgml',
      'tsv'     => 'text/tab-separated-values',
      'wml'     => 'text/vnd.wap.wml',
      'wmls'    => 'text/vnd.wap.wmlscript',
      'etx'     => 'text/x-setext',
      'xml'     => 'text/xml',
      'xsl'     => 'text/xml',
      'mpeg'    => 'video/mpeg',
      'mpg'     => 'video/mpeg',
      'mpe'     => 'video/mpeg',
      'qt'      => 'video/quicktime',
      'mov'     => 'video/quicktime',
      'mxu'     => 'video/vnd.mpegurl',
      'avi'     => 'video/x-msvideo',
      'movie'   => 'video/x-sgi-movie',
      'ice'     => 'x-conference/x-cooltalk'
    }

    DEFAULT_TMP_DATA_KEY_FIGURES = 16
    RESOURCE_PATH                = 'resources'
    WEB_SERVER_RESOURCE_PATH     = 'www'
    CGIKIT_PACKAGE               = 'CGIKit'

    class << self
      def create_tmp_data_key
        md5 = Digest::MD5::new
        md5.update Time.now.to_s
        md5.update rand(0).to_s
        md5.update $$.to_s
        md5.hexdigest[0, tmp_data_key_figures]
      end

      def tmp_data_key_figures
        DEFAULT_TMP_DATA_KEY_FIGURES
      end

      def mime( extension )
        MIME[extension]
      end

    end

    attr_reader :packages, :main_package

    def initialize( application, main_package_options = {} )
      @application          = application
      @resource_store       = @application.resource_store
      @resources            = @application.resources
      @web_server_resources = @application.web_server_resources
      @document_root        = @application.document_root || '.'
      @keys                 = {}
      @packages             = {}
      load_package(CGIKIT_PACKAGE)
      @main_package = create_main_package(main_package_options)
      load_packages()
      @packages[@main_package.name] = @main_package
    end

    def create_main_package( options = {} )
      main = Package.new(File.dirname(@application.path), options)
      main.lib_path = nil
      if @application.component_path then
        main.component_path = @application.component_path
      end
      if @application.resource_path then
        main.resource_path = @application.resource_path
      end
      if @application.web_server_resource_path then
        main.web_server_resource_path = @application.web_server_resource_path
      end
      main.load
      info("load main package: #{main.name}")
      main
    end

    def load_package( name )
      info("load package: #{name}")
      @application.package_paths.each do |path|
        fullpath = File.join(path, name)
        if FileTest.exist?(fullpath) then
          package = Package.new(fullpath)
          @packages[package.name] = package
          if @application.database then
            @application.database.add_models(package.paths_for_model)
          end
          return
        end
      end
      raise "#{name}: No such package"
    end

    def load_packages
      package_path = Application::PACKAGE_PATH
      if FileTest.exist?(package_path) then
        Dir.foreach(package_path) do |name|
          path = File.join(Application::PACKAGE_PATH, name)
          if /\A[^.]/ === name and FileTest.directory?(path) then
            load_package(name)
          end
        end
      end
    end

    def package( klass )
      name = klass.to_s.split('::').first
      @packages[name] || @main_package
    end


    #
    # path accessing
    #

    # Returns the public URL for the specified resource
    # when it is under the web server resources directory.
    # Otherwise returns nil.
    def url( name, package_name = nil, languages = [], request = nil )
      url = @resource_store.url(name, request)
      unless url then
        package_name ||= @main_package.name
        package = @packages[package_name]
        path = package.path_for_web_server_resource(@document_root, name, languages)
        if /\A#{File.expand_path(@document_root)}/ === path then
          url = path.sub(File.expand_path(@document_root), '')
        end
      end
      url
    end

    private

    def _resource_url( base, file, request )
      key = file.sub(base, '')
      key = Utilities.escape_url(key)
      @application.resource_request_handler.resource_url(key, request)
    end

    public

    # Returns the file path of the specified resource.
    def path( name, package_name = nil, languages = [] )
      package_name ||= @main_package.name
      package = @packages[package_name]
      package.path_for_resource(name, languages) ||
        package.path_for_web_server_resource(@document_root, name, languages)
    end

    # Returns a ByteData object for the specified resource.
    def bytedata( name, package_name = nil, languages = [] )
      data = @resource_store.bytedata(name)
      if data.nil? and filepath = path(name) then
        data = ByteData.new_with_file(filepath)
        data.content_type = content_type(filepath)
      end
      data
    end

    # Finds the content type for extension of the specified path.
    # If the path don't have extension, returns nil.
    def content_type( path )
      # for ruby 1.6
      base = File.basename path
      type = nil
      if base =~ /\.([^\.]+)$/ then
        type = MIME[$1]
      end
      type
    end

    def set_data( data, key, mime )
      data.tmp = key.nil?
      key ||= ResourceManager.create_tmp_data_key
      @resource_store.set_data(data, key, mime)
    end

    def key( data )
      @resource_store.key(data)
    end

    def remove_data( key )
      @resource_store.remove_data( key )
    end

    def flush
      @resource_store.flush
    end

    def message( key, name, package_name, languages )
      package_name ||= @main_package.name
      package = @packages[package_name]
      package.message(key, name, languages)
    end

   def nmessage( key, plural_key, n, name, package_name, languages )
      package_name ||= @main_package.name
      package = @packages[package_name]
      package.nmessage(key, plural_key, n, name, languages)
   end

  end


  class MemoryResourceStore

    def initialize( application )
      @application = application
      @keys = {}
    end

    def key( data )
      @keys.index(data)
    end

    def url( name, request )
      if @keys.key?(name) then
        @application.resource_request_handler.resource_url(name, request)
      else
        nil
      end
    end

    def bytedata( name )
      @keys[name]
    end

    def set_data( keys, key, mime )
      cache = ByteData.new(keys.to_s)
      cache.content_type = mime
      @keys[key] = cache
    end

    def remove_keys( key )
      @keys.delete(key)
    end

    def flush
      @keys.clear
    end

  end


  class FileResourceStore < MemoryResourceStore

    TMPDIR = 'resource'

    def initialize( application )
      super
      @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
    end

    def url( name, request )
      if FileTest.exist?(tmpfile(name)) then
        @application.resource_request_handler.resource_url(name, request)
      else
        nil
      end
    end

    def bytedata( name )
      unless data = super then
        path = File.join(@tmpdir, name)
        if FileTest.exist?(path) then
          cache = nil
          FileLock.shared_lock(path) do |file|
            cache = Marshal.load(file)
          end
          data = ByteData.new(cache[:data])
          data.path = path
          data.content_type = cache[:mime]
        end
      end
      data
    end

    def set_data( data, key, mime )
      super
      unless FileTest.directory? @tmpdir
        require 'ftools'
        File.makedirs @tmpdir
      end
      cache = {}
      cache[:data] = data
      cache[:key]  = key
      cache[:mime] = mime
      FileLock.exclusive_lock(tmpfile(key)) do |file|
        Marshal.dump(cache, file)
      end
    end

    def remove_data( key )
      super
      path = tmpfile(key)
      if FileTest.exist?(path)
        File.delete(path)
      end
    end

    def tmpfile( filename )
      File.join(@tmpdir, filename).untaint
    end

    def flush
      super
      Dir.foreach(@tmpdir) do |file|
        unless /\A\./ === file
          path = File.join(@tmpdir, file)
          File.delete(path)
        end
      end
    end

  end


  module ResourceLoadable

    def caching_url( file, package, data, key, mime, request )
      rm = @application.resource_manager
      url = nil
      if file then
        url = rm.url(file, package, request.languages, request)
        unless url then
          data = rm.bytedata(file)
        end
      end

      if data and url.nil? then
        if String === data then
          data = ByteData.new(data)
        end
        mime = data.content_type || mime
        rm.set_data(data, key, mime)
        key ||= rm.key(data)
        url = rm.url(key, package, request.languages, request)
      end
      url
    end

    def resource_urls( files, package, request )
      urls = []
      unless Array === files then
        files = [files]
      end
      rm = @application.resource_manager
      files.each do |file|
        if url = rm.url(file, package, request.languages, request) then
          urls << url
        end
      end
      urls
    end

    def cgikit_resource_urls( files )
      urls = []
      files.each do |file|
        if url = cgikit_resource_url(file) then
          urls << url
        end
      end
      urls
    end

    def cgikit_resource_url( file )
      @application.resource_manager.url(file, ResourceManager::CGIKIT_PACKAGE)
    end

  end

end

module CGIKit

# The super class of HTTP Request-Response classes.
class Message
  # HTML content.
  attr_accessor :content

  # Hash of HTTP headers.
  attr_accessor :headers

  # HTTP version. The default value is "1.1".
  attr_accessor :http_version

  # The encoding used for the content.
  attr_accessor :encoding

  # Array of Cookie objects.
  attr_accessor :cookies

  EOL = "\r\n"

  def initialize( headers = nil )
    @headers      = headers || {}
    @http_version = "1.1"
    @cookies      = []
    @content      = ''
  end

  # Adds a cookie object.
  def add_cookie( cookie )
    @cookies << cookie
  end

  # Removes the specified cookie in the cookies.
  def remove_cookie( cookie )
    @cookies.delete cookie
  end

  # Returns HTTP version.
  def http_version_line
    "HTTP/#@http_version"
  end
end

end
module CGIKit

class Request < Message

  attr_accessor :form_values, :session_id, :context_id, \
  :request_handler_key, :request_handler_path

  class << self
    # Parse query string and return a hash of parameters.
    def parse_query_string( query )
      params = Hash.new([])
      query.split(/[&;]/n).each do |pairs|
        key, value = pairs.split('=',2).collect{|v| Utilities.unescape_url(v) }
        if params.has_key?(key)
          params[key].push(value)
        else
          params[key] = [value]
        end
      end
      params
    end
  end

  def initialize( headers = nil, form_values = nil )
    super headers
    @cookies     = Cookie.parse_raw_cookie(@headers['HTTP_COOKIE'])
    @form_values = Hash.new([])
    @form_values.update(form_values) if form_values
    @request_handler_path = @headers['PATH_INFO']
    @parsed_action = false
  end

  def form_value( key )
    if @form_values.key?(key)
      @form_values[key].first
    else
      ''
    end
  end
  alias [] form_value

  def cookie( key )
    @cookies.each { | cookie |
      if cookie.name == key
        return cookie
      end
    }
    nil
  end

  def cookie_value( key )
    @cookies.each { | cookie |
      if cookie.name == key
        return cookie.value
      end
    }
    nil
  end

  def cookie_values( key = nil )
    if key then
      _cookie_values_for_key( @cookies, key )
    else
      _cookie_values @cookies
    end
  end

  private

  def _cookie_values( cookies )
    values = {}
    cookies.each do |cookie|
      values[cookie.name] = cookie.value
    end
    values
  end

  def _cookie_values_for_key( cookies, key )
    values = []
    cookies.each do |cookie|
      if cookie.name == key then
        values << cookie.value
      end
    end
    values
  end

  public

  def languages
    unless @languages then
      @languages = []
      if accept_language then
        @languages = accept_language.split(',').collect { |entry|
          lang, quality = entry.split(';')
          lang.strip!
          lang = lang.split('-')[0]
          if /^q=(.+)/ =~ quality
            quality = $1.to_f
          else
            quality = 1.0
          end
          [lang, quality]
        }.sort { |a, b| b[1] <=> a[1] }.collect { |i| i[0] }
      end
    end
    @languages
  end

  def action_class
    _parse_action_class_and_name() unless @parsed_action
    @action_class
  end

  def action_name
    _parse_action_class_and_name() unless @parsed_action
    @action_name
  end

  private

  def _parse_action_class_and_name
    klass = nil
    action_name = nil
    if path = @request_handler_path then
      path = path.dup
      path.gsub!(/\A\//, '')
      path.gsub!(/\?(.*)/, '')
      key, class_name, action_name = path.split('/')
      begin
        klass = Object
        class_name.split('::').each do |name|
          klass = klass.const_get(name)
        end
      rescue Exception => e
        klass = nil
        unless action_name then
          action_name = class_name
        end
      end
    end
    if klass and !(klass <= Action) then
      klass = default
      action_name = nil
    end
    @parsed_action = true
    @action_class = klass
    @action_name = action_name
  end

  public

  def https?
    /\Ahttps/ === self.referer or /on/i === @headers['HTTPS']
  end


  #
  # HTTP request headers
  #

  def accept;            @headers['HTTP_ACCEPT']          end
  def accept_charset;    @headers['HTTP_ACCEPT_CHARSET']  end
  def accept_language;   @headers['HTTP_ACCEPT_LANGUAGE'] end
  def auth_type;         @headers['AUTH_TYPE']            end
  def content_length;    @headers['CONTENT_LENGTH']       end
  def content_type;      @headers['CONTENT_TYPE']         end
  def from;              @headers['HTTP_FROM']            end
  def gateway_interface; @headers['GATEWAY_INTERFACE']    end
  def path_info;         @headers['PATH_INFO']            end
  def path_translated;   @headers['PATH_TRANSLATED']      end
  def query_string;      @headers['QUERY_STRING']         end
  def raw_cookie;        @headers['HTTP_COOKIE']          end
  def referer;           @headers['HTTP_REFERER']         end
  def remote_addr;       @headers['REMOTE_ADDR']          end
  def remote_host;       @headers['HTTP_HOST']            end
  def remote_ident;      @headers['REMOTE_IDENT']         end
  def remote_user;       @headers['REMOTE_USER']          end
  def request_method;    @headers['REQUEST_METHOD']       end
  def script_name;       @headers['SCRIPT_NAME']          end
  def server_name;       @headers['SERVER_NAME']          end
  def server_port;       @headers['SERVER_PORT']          end
  def server_protocol;   @headers['SERVER_PROTOCOL']      end
  def server_software;   @headers['SERVER_SOFTWARE']      end
  def uri;               @headers['REQUEST_URI']          end
  def user_agent;        @headers['HTTP_USER_AGENT']      end

  def accept=(value);            @headers['HTTP_ACCEPT']=value          end
  def accept_charset=(value);    @headers['HTTP_ACCEPT_CHARSET']=value  end
  def accept_language=(value);   @headers['HTTP_ACCEPT_LANGUAGE']=value end
  def auth_type=(value);         @headers['AUTH_TYPE']=value            end
  def content_length=(value);    @headers['CONTENT_LENGTH']=value       end
  def content_type=(value);      @headers['CONTENT_TYPE']=value         end
  def from=(value);              @headers['HTTP_FROM']=value            end
  def gateway_interface=(value); @headers['GATEWAY_INTERFACE']=value    end
  def path_info=(value);         @headers['PATH_INFO']=value            end
  def path_translated=(value);   @headers['PATH_TRANSLATED']=value      end
  def query_string=(value);      @headers['QUERY_STRING']=value         end
  def raw_cookie=(value);        @headers['HTTP_COOKIE']=value          end
  def referer=(value);           @headers['HTTP_REFERER']=value         end
  def remote_addr=(value);       @headers['REMOTE_ADDR']=value          end
  def remote_host=(value);       @headers['HTTP_HOST']=value            end
  def remote_ident=(value);      @headers['REMOTE_IDENT']=value         end
  def remote_user=(value);       @headers['REMOTE_USER']=value          end
  def request_method=(value);    @headers['REQUEST_METHOD']=value       end
  def script_name=(value);       @headers['SCRIPT_NAME']=value          end
  def server_name=(value);       @headers['SERVER_NAME']=value          end
  def server_port=(value);       @headers['SERVER_PORT']=value          end
  def server_protocol=(value);   @headers['SERVER_PROTOCOL']=value      end
  def server_software=(value);   @headers['SERVER_SOFTWARE']=value      end
  def uri=(value);               @headers['REQUEST_URI']=value          end
  def user_agent=(value);        @headers['HTTP_USER_AGENT']=value      end

  alias url uri
  alias url= uri=

end

end
module CGIKit

# A response object that is sent to a browser by a Adapter object.
#
# == Getting a response object
# You can get a response object by Application#response or Component#response.
#
# == Setting headers for a response object
# To send HTTP response headers, append a pair of key and value to headers.
# For example,
#
#  application.response.headers['Content-Type'] = 'text/html'
#
class Response < Message
  # Status code in HTTP. Default status is 200 ( OK ).
  attr_accessor :status

  attr_accessor :component

  STATUS = { 
    100 => 'Continue',
    101 => 'Switching Protocols',
    200 => 'OK',
    201 => 'Created',
    202 => 'Accepted',
    203 => 'Non-Authoritative Information',
    204 => 'No Content',
    205 => 'Reset Content',
    206 => 'Partial Content',
    300 => 'Multiple Choices',
    301 => 'Moved Permanently',
    302 => 'Found',
    303 => 'See Other',
    304 => 'Not Modified',
    305 => 'Use Proxy',
    307 => 'Temporary Redirect',
    400 => 'Bad Request',
    401 => 'Unauthorized',
    402 => 'Payment Required',
    403 => 'Forbidden',
    404 => 'Not Found',
    405 => 'Method Not Allowed',
    406 => 'Not Acceptable',
    407 => 'Proxy Authentication Required',
    408 => 'Request Timeout',
    409 => 'Conflict',
    410 => 'Gone',
    411 => 'Length Required',
    412 => 'Precondition Failed',
    413 => 'Request Entity Too Large',
    414 => 'Request-URI Too Long',
    415 => 'Unsupported Media Type',
    416 => 'Requested Range Not Satisfiable',
    417 => 'Expectation Failed',
    500 => 'Internal Server Error',
    501 => 'Not Implemented',
    502 => 'Bad Gateway',
    503 => 'Service Unavailable',
    504 => 'Gateway Timeout',
    505 => 'HTTP Version Not Supported'
  }

  class << self

    def new_with_response( response )
      res = new
      res.headers = response.headers
      res.status = response.status
      res.http_version = response.http_version
      res.encoding = response.encoding
      res.cookies = response.cookies
      res
    end

  end

  def initialize( headers = nil )
    super
    self.status = 200
    @headers['Content-Type'] = 'text/html'
    @headers['Cache-Control'] = \
      'private, no-cache, no-store, must-revalidate, max-age=0'
    @headers['Expires'] = Utilities.date(Time.now - 60)
    @headers['Date'] = Utilities.date(Time.now)
    @headers['Pragma'] = 'no-cache'
  end

  def header
    response = cookie_header()
    @headers.each do |key, value|
      if key != 'Content-Type' then
        response << header_line(key, value)
      end
    end
    response << content_type_header()
    response << EOL
    response
  end

  def to_s
    response = ''
    response << (self.content || '') unless redirect?
    response
  end

  def header_line( key, value )
    "#{key}: #{value}#{EOL}"
  end

  def content_type_header
    header = "Content-Type: #{self.headers['Content-Type']}"
    header << "; charset=#{self.encoding}" if self.encoding
    header << EOL
  end

  def cookie_header
    return '' if @cookies.empty?
    header = ''
    @cookies.each do |cookie|
      header << "Set-Cookie: #{cookie.to_s}"
      header << EOL
    end
    header
  end

  def status=( value )
    @status = value
    @headers['Status'] = "#@status #{STATUS[@status]}"
  end

  # Sends a temporary redirect response to the client using the specified URL.
  def set_redirect( url )
    @status = 302
    @headers['Location'] = url
  end
  alias redirect= set_redirect

  # Returns true if the response is setted redirect.
  def redirect?
    (@status == 302) or (@status == 307)
  end

  # Returns self. This method is invoked to display results in direct action.
  def generate_response
    self
  end
end

end
module CGIKit

# Cookie is a class for cookie.
# To send cookies to a browser needs to create cookie objects 
# and set them to a response object. Instead of creating cookie objects,
# you can also get cookie objects from a request object.
#
# Cookie objects have a pair of a cookie name and value.
# If you make the objects have multiple values for one name, 
# you must write code by yourself. 
# 
# == Controlling cookie objects
#
# === Creating cookies
# Give arguments of initialize() a name or a pair of name/value.
# The value of cookie is omittable.
#
#  cookie = Cookie.new( name, value )
#
# === Getting cookies from a request object
# Request has some methods for getting cookies.
# The methods are cookie(key), cookies, cookie_value(key), cookie_values(key).
# See also Request.
#
# === Setting cookies to a response object
# Response has methods for setting cookies. These methods are 
# defined in Message, the superclass of Response.
# Use add_cookie(cookie) and remove_cookie(cookie).
class Cookie
  # Name of the cookie.
  attr_accessor :name

  # Value of the cookie.
  attr_accessor :value

  # Restricts the cookie in the site.
  attr_accessor :path

  # Domain that can receive the cookie.
  attr_accessor :domain

  # Expiry date. You set Time object to the cookie object.
  # The value is formatted when the cookie is returned.
  attr_accessor :expires

  # Decides whether the cookie is encrypted or not.
  attr_accessor :secure

  class << self
    # Parse raw cookie string and return an array of cookies.
    def parse_raw_cookie( raw_cookie )
      cookies = []
      return cookies unless raw_cookie

      raw_cookie.split('; ').each do |pairs|
        name, value = pairs.split('=',2)
        name  = Utilities.unescape_url name
        value = Utilities.unescape_url value

        cookies << Cookie.new( name, value )
      end

      cookies
    end
  end

  def initialize( name, value = nil, domain = nil, path = nil, secure = false )
    @name   = name
    @value  = value
    @domain = domain
    @path   = path
    @secure = secure
  end

  def to_s
    buf = "#@name="
    if @value   then buf << Utilities.escape_url(@value.to_s) end
    if @domain  then buf << "; domain=#@domain" end
    if @path    then buf << "; path=#@path" end
    if @expires then buf << "; expires=#{Utilities.date(@expires)}" end
    if @secure == true then buf << '; secure' end
    buf
  end
end

end
module CGIKit

# == Context ID Format
# * When generating a component, 0 appends to context ID.
# * When generating a dynamic element, last element ID is incremented.
#   If in form, "name" attribute will be setted value or the last element ID.
# * In form, 0 appends to context ID.
#   The 0 last element ID is deleted at end of the form.
class Context

  SEPARATOR = '.'

  attr_accessor :request, :response, :application, :sender_id, :in_form
  attr_accessor :component, :request_handler_key

  attr_writer :session

  alias in_form? in_form

  def initialize( request = nil, application = nil )
    @request = request
    @response = Response.new
    @application = application
    @contexts = []
    @counts = []
    @session = nil
    @sender_id = request.context_id if request
  end

  def context_id( digit = false )
    if @contexts.empty? then
      ''
    elsif digit == true
      copied = @contexts.dup
      copied.pop
      copied.push(@counts.last)
      copied.join(SEPARATOR)
    else
      @contexts.join(SEPARATOR)
    end
  end

  def context_id=( id )
    delete_all
    id.to_s.split(SEPARATOR).each do |item|
      if item =~ /\A\d+\Z/ then
        @contexts << item.to_i
        @counts << item.to_i
      else
        @contexts << item
      end
    end
  end

  def element_id
    @contexts.last.to_s
  end

  def component_id
    @contexts.first
  end

  def component_id=( cid )
    @contexts[0] = cid
  end 

  def has_session?
    !@session.nil?
  end

  # Returns the session. If the receiver doesn't have a session,
  # it creates and returns a new session.
  def session
    unless has_session? then
      session = @application.restore_session(@request.session_id, self)
      unless session then
        session = @application.create_session(@request)
      end
      @session = session
      init_component_id_with_session
    end
    @session
  end

  def init_component_id_with_session( component = @component )
    unless component_id = session.component_id(component) then
      # register current component with ID for creating new session
      component_id = component_id() || @session.next_component_id
      @session.add_component_for_ids(component, component_id, context_id())
      @session.save_page(component)
      component_id = session.component_id(component)
    end
    init_context_id_and_component_id(component, component_id)
  end

  def init_component_id_without_session( component )
    if @component_id then
      component_id = @component_id + 1
    else
      component_id = 0
    end
    init_context_id_and_component_id(component, component_id)
  end

  def init_context_id_and_component_id( component, component_id )
    @component = component
    self.component_id = component_id
  end

  def action?( request, context_id = nil )
    context_id ||= self.context_id
    context_id == request.context_id
  end

  def in_form=( flag )
    @in_form = flag
    if @in_form == true then
      @form_id = context_id
    else
      @form_id = nil
    end
  end

  def current_form?( request )
    @form_id and (@form_id == request.context_id)
  end


  #
  # manipulating element ID
  #

  def append( string )
    @contexts << string
    _append_zero_to_count
  end

  alias << append

  def append_zero
    @contexts << 0
    _append_zero_to_count
  end

  def delete_all
    @contexts.clear
    _delete_all_of_count
  end

  def delete
    @contexts.pop
    _delete_last_of_count
  end

  def increment( string = nil )
    if string then
      @contexts.pop
      @contexts << string
      _increment_count
    else
      _increment_count
      @contexts.pop
      @contexts << @counts.last
    end
  end

  private

  def _append_zero_to_count
    @counts << 0
  end

  def _delete_all_of_count
    @counts.clear
  end

  def _delete_last_of_count
    @counts.pop
  end

  def _increment_count
    index = @contexts.size - 1
    @counts[index] ||= 0
    @counts[index] += 1
  end

  public

  #
  # generating URLs
  #

  def direct_action_url( action_class, action_name, query = {}, sid = true )
    action_class ||= @application.direct_action_class || CGIKit::DirectAction
    app = @session || @application
    url = @application.direct_action_url(self, action_class, action_name, query, sid)
    url
  end

  def component_action_url( query = {}, is_secure = false )
    complete_url(@application.component_request_handler_key, nil, query, is_secure)
  end

  def complete_url( key, path, query, is_secure, port = nil )
    @application.url(key, self, path, query, is_secure, port)
  end

  def url( key, path = nil, query = {} )
    complete_url(key, path, query, false)
  end

end

end
module CGIKit

# new_with_keypath, new_with_value
# constant?(component=nil), settable?(component=nil)
# set_value(value, component), value(component)

class Association
  class InternalInconsistencyError < StandardError #:nodoc:
  end

  attr_accessor :keypath
  attr_accessor :_value, :_constant #:nodoc:

  class << self
    def new_with_keypath( keypath )
      as = Association.new
      as.keypath = keypath
      as._constant = false
      as
    end

    def new_with_value( value )
      as = Association.new
      as._value = value
      as._constant = true
      as
    end

    def adapt_to_bool(value)
      if (value == false) or value.nil? or \
        (value == 0) or (value == 'false') then
        false
      else
        true
      end
    end
  end

  def constant?
    _constant == true
  end

  def constant_in_component?( component )
    if _constant == true then
      return false
    else
      return !component.can_set_binding_value?(@keypath)
    end
  end

  def settable?
    _constant != true
  end

  def settable_in_component?( component )
    if _constant == true then
      return false
    else
      return component.can_set_binding_value?(@keypath)
    end
  end

  def set_value( value, component )
    if settable? then
      component.take_value_for_keypath(keypath, value)
    else
      raise InternalInconsistencyError, "The receiver's value is not settable."
    end
  end

  def value( component = nil )
    if constant? then
      _value
    elsif component.nil? then
      @keypath
    else
      component.value_for_keypath(@keypath)
    end
  end

  # Returns false if - null, false, 0, "false"
  def bool( component = nil )
    value = value(component)
    Association.adapt_to_bool(value)
  end

end

end
module CGIKit

  module SessionSweepable

    SWEEP_ACTION_NAME = '__sweep_sessions'
    SWEEP_PASS_KEY = 'pass'
    SWEEP_PROOF = '<!-- CGIKit::DirectAction#__sweep_sessions -->'
    SWEEP_DELETED_START = '<!-- SWEEP_DELETED_START -->'
    SWEEP_DELETED_END   = '<!-- SWEEP_DELETED_END -->'
    SWEEP_FAILED_START = '<!-- SWEEP_FAILED_START -->'
    SWEEP_FAILED_END   = '<!-- SWEEP_FAILED_END -->'

    def __sweep_sessions_action
      response = Response.new
      if @application.sweep_password then
        response.content = <<EOF
<html><head><title>Sweep Sessions</title></head><body>
#{SWEEP_PROOF}<form method="POST">
<input type="password" name="#{SWEEP_PASS_KEY}"/><input type="submit" value="Sweep"/>
</form>
EOF
        pass = @request.form_value('pass')
        if pass == @application.sweep_password then
          successed, failed = @application.session_store.sweep_sessions
          response.content << <<EOF
<p><strong>Deleted: #{SWEEP_DELETED_START}#{successed}#{SWEEP_DELETED_END}, 
Failed: #{SWEEP_FAILED_START}#{failed}#{SWEEP_FAILED_END}</p>
EOF
        end
        response.content << "</body></html>"
      end
      response
    end

    def self.sweep_info( content )
      /#{SWEEP_DELETED_START}([0-9]+)#{SWEEP_DELETED_END}/ === content
      deleted = $1
      /#{SWEEP_FAILED_START}([0-9]+)#{SWEEP_FAILED_END}/ === content
      failed = $1
      [deleted, failed]
    end

  end


  class Action

    DEFAULT_ACTION_NAME = 'default'
    ACTION_TEXT = '_action'

    class << self

      def default_action_name
        DEFAULT_ACTION_NAME
      end

    end

    def initialize( application, request )
      @application = application
      @request = request
    end

    def key_for_action( action )
      action + ACTION_TEXT
    end

    def default_action_name
      DEFAULT_ACTION_NAME
    end

    def perform_action( action )
      __send__(key_for_action(action))
    end

    def default_action; end

    def action?( action )
      respond_to?(key_for_action(action))
    end

  end


  # == URLs to invoke methods
  # ../App.cgi/d/::            default_action on class "DirectAction"
  # ../App.cgi/d/search::      search_action on class "DirectAction" or
  #                            default_action on class "search"
  # ../App.cgi/d/Data/search:: search_action on class "Data"
  class DirectAction < Action
    include KeyValueCoding, Logging, SessionSweepable

    attr_reader :context, :request, :application

    def initialize( application, request )
      super
      @context = @application.context_class.new(@request, @application)
      @handler = @application.direct_action_request_handler
    end


    #
    # accessing
    #

    def session
      unless @session then
        unless @session = existing_session then
          @session = @application.session_class.new
        end
      end
      @session
    end

    def existing_session
      unless session_id = session_id(@request) then
        return nil
      end
      session = @application.restore_session(session_id, @context)
      if session.nil? or session.terminate? then
        nil
      else
        session
      end
    end

    def session_id( request )
      @handler.session_id(request)
    end

    def page( name, *args )
      @application.page(name, @context, *args)
    end


    #
    # performing
    #

    def default_action
      page(@application.main)
    end


    #
    # testing
    #

    def self.sweep_page?( content )
      /#{SWEEP_PROOF}/ === content
    end


    #
    # taking form values
    #

    def take_form_values( keys )
      keys.each do |key|
        value = @request.form_value(key)
        take_value_for_key(key, value)
      end
    end

    def take_form_value_arrays( keys )
      keys.each do |key|
        value = @request.form_values[key]
        take_value_for_key(key, value)
      end
    end

  end

end
module CGIKit

  class Package

    BIN_PATH                     = 'bin'
    LIB_PATH                     = 'lib'
    COMPONENT_PATH               = 'components'
    RESOURCE_PATH                = 'resources'
    WEB_SERVER_RESOURCE_PATH     = 'www'
    MODEL_PATH                   = 'models'
    LOCALE_PATH                  = 'locale'
    PACKAGE_PATH                 = 'packages'
    MESSAGE_PATH                 = 'messages'
    AUTOLOAD_COMPONENT_LOAD_TYPE = 'autoload'
    REQUIRE_COMPONENT_LOAD_TYPE  = 'require'
    CONFIG_FILE                  = 'config.rb'
    PRE_LOAD_FILE                = 'pre-load.rb'
    POST_LOAD_FILE               = 'post-load.rb'
    DEFAULT_MESSAGE_FILE         = 'default.mo'

    attr_reader :path, :name
    attr_accessor :lib_path, :component_path, :resource_path, \
    :web_server_resource_path, :model_path, :message_path, \
    :component_load_type, :package_class_name, :message_encoding

    class << self

      def all_resource_paths
        [BIN_PATH, LIB_PATH, COMPONENT_PATH, RESOURCE_PATH, 
          WEB_SERVER_RESOURCE_PATH, MODEL_PATH, LOCALE_PATH, 
          PACKAGE_PATH, MESSAGE_PATH]
      end

    end

    def initialize( path = nil, options = {} )
      @bin_path       = options[:bin_path] || BIN_PATH
      @lib_path       = options[:lib_path] || LIB_PATH
      @component_path = options[:component_path] || COMPONENT_PATH
      @resource_path  = options[:resource_path] || RESOURCE_PATH
      @web_server_resource_path = options[:web_server_resource_path] || \
        WEB_SERVER_RESOURCE_PATH
      @model_path     = options[:model_path] || MODEL_PATH
      @locale_path    = options[:locale_path] || LOCALE_PATH
      @package_path   = options[:package_path] || PACKAGE_PATH
      @message_path   = options[:message_path] || MESSAGE_PATH

      @component_load_type = options[:component_load_type] || \
        REQUIRE_COMPONENT_LOAD_TYPE
      @package_class_name  = options[:package_class_name]
      @config_file         = options[:config_file] || CONFIG_FILE
      @pre_load_file       = options[:pre_lod_file] || PRE_LOAD_FILE
      @post_load_file      = options[:post_load_file] || POST_LOAD_FILE
      @message_encoding    = options[:message_encoding]
      @messages = {}
      if path then
        self.path = path
        load
      end
    end


    #
    # accessing
    #

    def path=( path )
      @path = path
      @name = File.basename(path)
    end

    def all_resource_paths
      [@bin_path, @lib_path, @component_path, @resource_path,
        @web_server_resource_path, @model_path, @locale_path,
        @package_path, @message_path]
    end


    #
    # loading
    #

    def load
      load_config
      exec_load_hook(path)
      load_lib if @lib_path
      load_components if @component_path
      load_resources if @resource_path
      load_web_server_resources if @web_server_resource_path
    end

    def load_config
      path = File.join(@path, CONFIG_FILE)
      return unless FileTest.exist?(path)
      config = nil
      str = nil
      open(path) { |f| str = f.read }
      Thread.start(str) do
        $SAFE = 4
        config = eval(str)
      end.join
      config.each do |key, value|
        self.__send__("#{key}=", value)
      end
    end

    def load_lib
      lib = load_path(@lib_path)
      Dir.glob(lib) do |path|
        $LOAD_PATH << path
      end
    end

    def load_components
      comp = load_path(@component_path, '**', '*.rb')
      Dir.glob(comp) do |path|
        next unless /\A[A-Z]/ === File.basename(path)
        name = File.basename(path, '.*')
        if autoload_component_load_type? then
          mod = package_module()
          mod.autoload(name, path)
        elsif require_component_load_type? then
          require(path)
        end
      end
    end

    def load_resources
      load_path(@resource_path)
    end

    def load_web_server_resources
      load_path(@web_server_resource_path)
    end

    def load_messages( name, lang )
      @messages[lang] ||= {}
      path = locale_path(lang, @message_path, name)
      return unless FileTest.exist?(path)
      catalog = MessageCatalog.new(path, @message_encoding)
      @messages[lang][name] = catalog
    end


    #
    # testing
    #

    def autoload_component_load_type?
      @component_load_type == AUTOLOAD_COMPONENT_LOAD_TYPE
    end

    def require_component_load_type?
      @component_load_type == REQUIRE_COMPONENT_LOAD_TYPE
    end


    #
    # executing
    #

    def exec_hook( path, pre, post, &block )
      pre = File.join(path, pre)
      post = File.join(path, post)
      if FileTest.exist?(pre) then
        require(pre) 
      end
      block.call if block_given?
      if FileTest.exist?(post) then
        require(post) 
      end
    end

    def exec_load_hook( path, &block )
      exec_hook(path, PRE_LOAD_FILE, POST_LOAD_FILE, &block)
    end


    #
    # path accessing
    #

    def package_module
      modname = (@package_class_name || @name).to_s.intern
      unless Object.const_defined?(modname) then
        obj = Module.new
        Object.const_set(modname, obj)
      end
      Object.const_get(modname)
    end

    def load_path( *paths )
      File.join(@path, '**', *paths)
    end

    def locale_path( lang, *paths )
      File.join(@path, @locale_path, lang, *paths)
    end

    def path_for_component( name, languages = [] )
      unless dir = name.include?('.') then
        name = "#{name}.*"
      end
      file = File.join(@component_path, '**', name)
      path = file_path(file, languages)
      unless dir then
        path = File.dirname(path)
      end
      path
    end

    def file_path( file, languages = [] )
      path = nil
      languages.each do |lang|
        pattern = locale_path(lang, file)
        paths = Dir.glob(pattern)
        unless paths.empty? then
          path = paths[0]
          break
        end
      end
      unless path then
        path = Dir.glob(File.join(@path, file))[0]
      end
      path
    end

    def path_for_resource( name, languages = [] )
      file = File.join(@resource_path, name)
      file_path(file, languages)
    end

    # DOCROOT/cgikit/packages/(packages)
    def path_for_web_server_resource( docroot, name, languages = [] )
      www = File.join(docroot, Application::CGIKIT_PATH,
                      Application::PACKAGE_PATH, @name)
      paths = []
      languages.each do |lang|
        paths << File.join(www, @locale_path, lang, @web_server_resource_path, name)
      end
      paths << File.join(www, @web_server_resource_path, name)
      paths.each do |path|
        if FileTest.exist?(path) then
          return path
        end
      end
      file_path(File.join(@web_server_resource_path, name), languages)
    end

    def paths_for_model
      paths = []
      Dir.foreach(File.join(@path, @model_path)) do |path|
        unless /\A\./ === path then
          paths << File.join(@path, @model_path, path)
        end
      end
      paths
    end


    #
    # message accessing
    #

    def message( key, name, languages )
      if catalog = catalog(name, languages) then
        catalog.gettext(key)
      else
        key
      end
    end

    def nmessage( key, plural_key, n, name, languages )
      if catalog = catalog(name, languages) then
        catalog.ngettext(key, plural_key, n)
      else
        plural_key
      end
    end

    def catalog( name, languages )
      name ||= DEFAULT_MESSAGE_FILE
      unless Array === languages then
        languages = [languages]
      end
      languages.each do |lang|
        if !@messages[lang] or !@messages[lang][name] then
          load_messages(name, lang)
        end
        if catalog = @messages[lang][name] then
          return catalog
        end
      end
      nil
    end

  end


  class MessageCatalog

    def initialize( path, encoding )
      require 'cgikit/lang/mo.rb' unless defined?(MOFile)
      @path = path
      @mo = MOFile.open(path, encoding)
    end

    def gettext(msgid)
      if @mo
	if @mo[msgid]
	  result = @mo[msgid].size > 0 ? @mo[msgid] : msgid
	else
	  result = msgid
	end
      else
        result = msgid
      end
      result
    end

    def ngettext(msgid, msgid_plural, n)
      msg = gettext(msgid + "\000" + msgid_plural)
      if msg.include?("\000")
        ary = msg.split("\000")
        if @mo
          plural = eval(@mo.plural)
          if plural.kind_of?(Numeric)
            msg = ary[plural]
          else
            msg = plural ? ary[1] : ary[0]
          end
        else
          msg = n == 1 ? ary[0] : ary[1]
        end
      end
      msg
    end

  end

end

=begin
    mo.rb - A simple class for operating GNU MO file.

    Copyright (C) 2003,2004  Masao Mutoh
    Copyright (C) 2002  Masahiro Sakai, Masao Mutoh
    Copyright (C) 2001  Masahiro Sakai

        Masahiro Sakai                  <s01397ms@sfc.keio.ac.jp>
        Masao Mutoh                     <mutoh@highway.ne.jp>

    You can redistribute this file and/or modify it under the same term
    of Ruby.  License of Ruby is included with Ruby distribution in
    the file "README".

    $Id: mo.rb,v 1.1 2005/07/06 17:57:45 reno Exp $
=end

class MOFile < Hash
  class InvalidFormat < RuntimeError; end;

  Header = Struct.new(:magic,
                      :revision,
                      :nstrings,
                      :orig_table_offset,
                      :translated_table_offset,
                      :hash_table_size,
                      :hash_table_offset)

  MAGIC_BIG_ENDIAN    = "\x95\x04\x12\xde"
  MAGIC_LITTLE_ENDIAN = "\xde\x12\x04\x95"

  def self.open(arg = nil, output_charset = nil)
    result = self.new(output_charset)
    case arg
    when String
      result.load_from_file(arg)
    when IO
      result.load_from_stream(arg)
    end
    result
  end

  def initialize(output_charset = nil)
    @little_endian = true
    @output_charset = output_charset
    super()
  end

  def load_from_stream(io)
    magic = io.read(4)
    case magic
    when MAGIC_BIG_ENDIAN
      @little_endian = false
    when MAGIC_LITTLE_ENDIAN
      @little_endian = true
    else
      raise InvalidFormat.new(sprintf("Unknown signature %s", magic.dump))
    end

    header = Header.new(magic, *(io.read(4 * 6).unpack(@little_endian ? 'V6' : 'N6')))
    raise InvalidFormat.new(sprintf("file format revision %d isn't supported", header.revision)) if header.revision > 0

    io.pos = header.orig_table_offset
    orig_table_data = io.read((4 * 2) * header.nstrings).unpack(@little_endian ? 'V*' : 'N*')

    io.pos = header.translated_table_offset
    trans_table_data = io.read((4 * 2) * header.nstrings).unpack(@little_endian ? 'V*' : 'N*')

    original_strings = Array.new(header.nstrings)
    for i in 0...header.nstrings
      io.pos = orig_table_data[i * 2 + 1]
      original_strings[i] = io.read(orig_table_data[i * 2 + 0])
    end

    clear
    for i in 0...header.nstrings
      io.pos = trans_table_data[i * 2 + 1]
      str = io.read(trans_table_data[i * 2 + 0])

      if original_strings[i] == ""
        if str
          @charset = nil
          @nplurals = nil
          @plural = nil
          str.each_line{|line|
            if /^Content-Type:/i =~ line and /charset=((?:\w|-)+)/i =~ line
              @charset = $1
            elsif /^Plural-Forms:\s*nplurals\s*\=\s*(\d*);\s*plural\s*\=\s*([^;]*)\n?/ =~ line
              @nplurals = $1
              @plural = $2
            end
            break if @charset and @nplurals
          }
          @nplurals = "1" unless @nplurals
          @plural = "0" unless @plural
        end
      else
        if @output_charset and @charset
          begin
            require 'iconv'
            str = Iconv.iconv(@output_charset, @charset, str).join
          rescue Iconv::Failure
            if $DEBUG
              print "@charset = ", @charset, "\n"
              print "@output_charset = ", @output_charset, "\n"
              print "msgid = ", original_strings[i], "\n"
              print "msgstr = ", str, "\n"
            end
          end
        end
      end
      self[original_strings[i]] = str
    end
    self
  end

  # From gettext-0.12.1/gettext-tools/lib/hash.c
  def prime?(candidate)
    divn = 3
    sq = divn * divn

    while (sq < candidate && candidate % divn != 0)
      divn += 1
      sq += 4 * divn
      divn += 1
    end
    candidate % divn != 0
  end

  # From gettext-0.12.1/gettext-tools/lib/hash.c
  def next_prime(seed)
    seed |= 1
    while (! prime?(seed))
      seed += 2
    end
    seed
  end

  # From gettext-0.12.1/gettext-runtime/intl/hash-string.h
  # Defines the so called `hashpjw' function by P.J. Weinberger
  # [see Aho/Sethi/Ullman, COMPILERS: Principles, Techniques and Tools,
  # 1986, 1987 Bell Telephone Laboratories, Inc.] 
  HASHWORDBITS = 32
  def hash_string(str)
    hval = 0
    i = 0
    str.each_byte do |b|
      break if b == '\0'
      hval <<= 4
      hval += b.to_i
      g = hval & (0xf << (HASHWORDBITS - 4))
      if (g != 0)
        hval ^= g >> (HASHWORDBITS - 8)
        hval ^= g
      end
    end
    hval
  end

  def save_to_stream(io)
    #Save data as little endian format.
    header_size = 4 * 7
    table_size  = 4 * 2 * size

    hash_table_size = next_prime((size * 4) / 3) 
    hash_table_size = 3 if hash_table_size <= 2
    header = Header.new(
      MAGIC_LITTLE_ENDIAN,          # magic
      0,                            # revision
      size,                         # nstrings
      header_size,                  # orig_table_offset
      header_size + table_size,     # translated_table_offset
      hash_table_size,              # hash_table_size
      header_size + table_size * 2  # hash_table_offset
    )
    io.write(header.to_a.pack('a4V*'))

    ary = to_a
    ary.sort!{|a, b| a[0] <=> b[0]} # sort by original string

    pos = header.hash_table_size * 4 + header.hash_table_offset

    orig_table_data = Array.new()
    ary.each{|item, _|
      orig_table_data.push(item.size)
      orig_table_data.push(pos)
      pos += item.size + 1 # +1 is <NUL>
    }
    io.write(orig_table_data.pack('V*'))

    trans_table_data = Array.new()
    ary.each{|_, item|
      trans_table_data.push(item.size)
      trans_table_data.push(pos)
      pos += item.size + 1 # +1 is <NUL>
    }
    io.write(trans_table_data.pack('V*'))

    hash_tab = Array.new(hash_table_size)
    j = 0
    ary[0...size].each {|key, _|
      hash_val = hash_string(key)
      idx = hash_val % hash_table_size
      if hash_tab[idx] != nil
        incr = 1 + (hash_val % (hash_table_size - 2))
        begin
          if (idx >= hash_table_size - incr)
            idx -= hash_table_size - incr
          else
            idx += incr
          end
        end until (hash_tab[idx] == nil)
      end
      hash_tab[idx] = j + 1
      j += 1
    }
    hash_tab.collect!{|i| i ? i : 0}

    io.write(hash_tab.pack('V*'))

    ary.each{|item, _| io.write(item); io.write("\0") }
    ary.each{|_, item| io.write(item); io.write("\0") }

    self
  end

  def load_from_file(filename)
    File.open(filename, 'rb'){|f| load_from_stream(f)}
  end

  def save_to_file(filename)
    File.open(filename, 'wb'){|f| save_to_stream(f)}
  end

  attr_accessor :little_endian
  attr_reader :charset, :nplurals, :plural
end


# Test

if $0 == __FILE__
  if (ARGV.include? "-h") or (ARGV.include? "--help")
    STDERR.puts("mo.rb [filename.mo ...]")
    exit
  end

  ARGV.each{ |item|
    mo = MOFile.open(item)
    puts "------------------------------------------------------------------"
    puts "charset  = \"#{mo.charset}\""
    puts "nplurals = \"#{mo.nplurals}\""
    puts "plural   = \"#{mo.plural}\""
    puts "------------------------------------------------------------------"
    mo.each do |key, value|
      puts "original message = \"#{key}\""
      puts "translated message = \"#{value}\""
      puts "--------------------------------------------------------------------"
    end
  }
end
module CGIKit

class Reset < DynamicElement

  class << self
    def create_api
      api = API.new(:Reset)
      value = Binding.new(:value)
      value.default = 'Reset'
      api << value
      api << enabled_binding()
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def append_to_response( response, context )
    take_value(:value)
    html  = "<input type=\"reset\" name=\"#{name_value(context)}\" value=\"#{escaped_string(@values[:value])}\""
    html << other()
    html << enabled()
    html << " />"
    response.content << html
  end

end

end
module CGIKit

  class Delegate

    module BrowserDelegate

      def browser_will_generate_tags( select, options ); end

    end

  end


  class Browser < DynamicElement

    class << self
      def create_api
        api = API.new(:Browser)
        multiple = Binding.new(:multiple)
        multiple.value_set = Binding::BOOLEAN
        multiple.default = true
        api << multiple
        api << name_binding()
        api << display_binding()
        api << list_binding()
        api << item_binding()
        api << escape_binding()
        api << value_binding(false, false)
        api << selections_binding()
        api << selected_values_binding()
        api << enabled_binding()
        api << item_display_value_validation()
        api << existential_validation(:selected_values, :selections)
        api
      end

      def keys_to_delete_from_associations
        [:type]
      end
    end

    def begin_context( context )
      take_value(:name)
      context.increment(@values[:name])
    end

    def take_values_from_request( request, context )
      take_value(:list)
      take_value(:selections)
      take_value(:selected_values)
      names = [:selections, :selected_values]
      names.each do |name|
        if @values[name].nil? then
          @values[name] = []
          set_value(name, @values[name])
        end
      end

      options = values_from_request(request, context)
      if !options.empty? and context.current_form?(request) then
        if declared?(:selections) then
          @values[:selections].clear
          if declared?(:value) then
            @values[:list].each do |item|
              set_value(:item, item)
              take_value(:value)
              if options.include?(@values[:value].to_s) then
                @values[:selections] << item
              end
            end
          else
            options.each do |option|
              @values[:selections] << @values[:list][option.to_i]
            end
          end
        elsif declared?(:selected_values) then
          @values[:selected_values].replace(options)
        end
      end
    end

    def append_to_response( response, context )
      take_value(:list)
      take_value(:selections)
      take_value(:selected_values)
      take_bool(:escape)
      take_bool(:multiple)
      take_bool(:enabled)
      selattrs = { :name => name_value(context),
        :multiple => @values[:multiple], :disabled => !@values[:enabled] }
      selattrs.update(optional_attributes())
      seltag = HTMLTag.select(selattrs, other_attribute())
      optags = []
      options = values_from_request(context.request, context)

      if @values[:list] then
        @values[:list].each_with_index do |item, index|
          set_value(:item, item)
          take_value(:value)
          take_value(:display)
          attrs = {}
          if (@values[:selections] and @values[:selections].include?(item)) or \
            (options and 
               (options.include?(@values[:value].to_s) or \
                options.include?(index.to_s))) then
            attrs[:selected] = true
          end
          if @values[:value] then
            attrs[:value] = escaped_string(@values[:value], @values[:escape])
          else
            attrs[:value] = index
          end
          display = @values[:display] || item.to_s
          content = escaped_string(display, @values[:escape])
          optags << HTMLTag.option(attrs, '', content)
        end
      end

      ask(:browser_will_generate_tags) do |d|
        seltag, optags = d.browser_will_generate_tags(seltag, optags) \
          || seltag, optags
      end

      response.content << seltag.open_tag
      response.content << "\n"
      optags.each do |tag|
        response.content << tag.container_tag
        response.content << "\n"
      end
      response.content << seltag.close_tag
    end
  end

end
module CGIKit

  class Delegate

    module CheckboxDelegate

      def checkbox_will_generate_tag( tag ); end

    end

  end


  class Checkbox < DynamicElement

    DELETE_KEYS = [:type]
    DEFAULT_VALUE = '1'

    class << self

      def create_api
        api = API.new(:Checkbox)
        api << name_binding()
        api << checked_binding()
        api << selection_binding()
        api << value_binding(false, false)
        api << enabled_binding()
        api << required_validation([:checked, :value])
        api << universal_validation(:value, :selection)
        api
      end

      def keys_to_delete_from_associations
        DELETE_KEYS
      end

    end

    def begin_context( context )
      take_value(:name)
      context.increment(@values[:name])
    end

    def take_values_from_request( request, context )
      take_value(:value)
      take_value(:selection)
      take_bool(:checked)

      if value = value_from_request(request, context) then
        if declared?(:checked) then
          set_value(:checked, true)
        else
          set_value(:selection, @values[:value])
        end
      elsif context.current_form?(request) then
        if declared?(:checked) then
          set_value(:checked, false)
        else
          set_value(:selection, nil)
        end
      end
    end

    def append_to_response( response, context )
      take_bool(:enabled)
      take_value(:value)
      take_value(:selection)
      take_bool(:checked)
      attrs = { :name => name_value(context), :disabled => !@values[:enabled],
        :value => @values[:value] }
      attrs.update(optional_attributes())

      tag = HTMLTag.checkbox(attrs, other_attribute())
      if declared?(:checked) then
        attrs[:checked] = @values[:checked]
      else
        attrs[:value] = @values[:value]
        attrs[:checked] = \
          (@values[:selection] and (@values[:value] == @values[:selection]))
      end
      ask(:checkbox_will_generate_tag) do |d|
        tag = d.checkbox_will_generate_tag(tag) || tag
      end
      if tag[:value] then
        tag[:value] = escaped_string(tag[:value])
      else
        tag[:value] ||= DEFAULT_VALUE
      end
      response.content << tag.empty_tag
    end
  end

end
module CGIKit

  class Delegate

    module LinkDelegate

      def link_should_invoke_action; end
      def link_generate_url( url ); end
      def link_will_generate_tag( tag ); end

    end

  end


  class Link < DynamicElement

    module Utilities

      def action_url( context )
        take_value(:frag)
        take_value(:page)
        take_value(:action, false)
        take_bool(:secure)
        take_value_query(:query)
        take_value(:session_id)
        take_value(:direct_action, false)
        take_value(:action_class)

        if @values[:page] or @values[:action] then
          # create session if it doesn't have session
          context.session
          url = context.component_action_url(@values[:query], @values[:secure])
        elsif direct_action? then
          url = context.direct_action_url(@values[:action_class],
                                          @values[:direct_action],
                                          @values[:query],
                                          @values[:session_id])
        end
        append_frag(url, @values[:frag]) if @values[:frag]
        url
      end

      def append_frag( url, frag )
        frag = "##{frag}"
        unless url.sub!(/\?/, "#{frag}?") then
          url << frag
        end
        url
      end

      def set_href( tag, context )
        take_value(:href)
        tag[:href] = @values[:href] || action_url(context)
        ask(:link_generate_url) do |d|
          tag[:href] = d.link_generate_url(tag[:href])
        end
        ask(:link_will_generate_tag) do |d|
          tag = d.link_will_generate_tag(tag) || tag
        end
        tag
      end

      module_function :append_frag

    end

    include Utilities

    class << self
      def create_api
        api = API.new(:Link)
        string = Binding.new(:string)

        api << string
        api << action_binding()
        api << enabled_binding()
        api << href_binding()
        api << escape_binding()
        api << page_binding()
        api << secure_binding()
        api << query_binding()
        api << frag_binding()
        api << session_id_binding()
        set_direct_action(api)
        api << required_validation([:action, :href, :page], [], \
                                   [:direct_action, :action_class])
        api
      end
    end

    def begin_context( context )
      context.increment
      context.append_zero
    end

    def end_context( context )
      context.delete
    end

    def invoke_action( request, context )
      ask(:link_should_invoke_action) do |d|
        return unless d.link_should_invoke_action
      end

      take_value(:page)
      take_value(:action, false)
      result = nil
      if context.action?(request) then
        if @values[:page] then
          result = @application.page(@values[:page], context)
        else
          result = @root[@values[:action]]
        end
      end
      result
    end

    def append_to_response( response, context )
      take_value(:name)
      take_bool(:enabled)
      take_bool(:escape)
      take_value(:string)

      if @values[:string] then
        @values[:string] = escaped_string(@values[:string], @values[:escape])
      end

      unless @values[:enabled] then
        if @values[:string] then
          response.content << @values[:string]
        elsif empty? == false then
          @node.append_to_response(response, context)
        end
        return
      end

      attrs = {}
      attrs.update(optional_attributes())
      tag = HTMLTag.a(attrs, other_attribute())
      set_href(tag, context)
      response.content << tag.open_tag
      if empty? == false then
        @node.append_to_response(response, context)
      elsif @values[:string] then
        response.content << @values[:string]
      end
      response.content << tag.close_tag
    end

  end

end
module CGIKit

  class Delegate

    module StringDelegate

      def string_will_generate_content( value ); end

    end

  end


  class String < DynamicElement

    class << self
      def create_api
        api = API.new(:String)
        empty = Binding.new(:empty)
        br = Binding.new(:br)
        br.value_set = Binding::BOOLEAN
        api << value_binding()
        api << escape_binding()
        api << empty
        api << br
        api
      end
    end

    def append_to_response( response, context )
      take_value(:value)
      take_value(:empty)
      take_bool(:escape)
      take_bool(:br)

      ask(:string_will_generate_content, context) do |d|
        alt = d.string_will_generate_content(@values[:value])
        @values[:value] = alt if alt
      end

      str = ''
      if @values[:value] then
        str = @values[:value].to_s
      elsif @values[:empty] then
        str = @values[:empty].to_s
      end
      str = escaped_string(str, @values[:escape])
      if @values[:br] then
        str.gsub!(/(\r\n|\r|\n)/, "<br />")
      end
      response.content << str
    end
  end

end
module CGIKit

  class CheckboxGroup < DynamicElement

    class << self

      def create_api
        api = API.new(:CheckboxGroup)
        index = Binding.new(:index)
        prefix = Binding.new(:prefix)
        suffix = Binding.new(:suffix)
        api << index
        api << prefix
        api << suffix
        api << list_binding()
        api << item_binding()
        api << selections_binding(true)
        api << name_binding()
        api << display_binding()
        api << escape_binding()
        api << enabled_binding()
        api
      end

    end

    DEFAULT_VALUE = '1'

    def begin_context( context )
      context.increment
      context.append_zero
    end

    def end_context( context )
      context.delete
    end

    def take_values_from_request( request, context )
      take_value(:list)
      take_value(:selections)
      unless @values[:selections] then
        @values[:selections] = []
        set_value(:selections, @values[:selections])
      end
      @values[:selections].clear
      @values[:list].each_with_index do |item, index|
        if value_from_request(request, context) then
          @values[:selections] << item
        end
        context.increment
      end
    end

    def append_to_response( response, context )
      take_value(:list)
      take_value(:index, false)
      take_value(:selections)
      take_bool(:escape)

      @values[:list].each_with_index do |item, index|
        set_value(:item, item)
        set_value(:index, index) if @values[:index]
        take_value(:name)
        take_value(:display)
        take_value(:prefix)
        take_value(:suffix)
        take_bool(:enabled)
        attrs = { :name => name_value(context),
          :disabled => !@values[:enabled],
          :value => DEFAULT_VALUE,
          :checked => @values[:selections].include?(item) }
        attrs.update(optional_attributes())

        tag = HTMLTag.checkbox(attrs, other_attribute())
        display = @values[:display] || item.to_s
        display = escaped_string(display) if @values[:escape]
        response.content << \
          "#{@values[:prefix]}#{tag.empty_tag} #{display}#{@values[:suffix]}\n"
        context.increment
      end
    end
  end

end

module CGIKit

  class Outline < DynamicElement

    class << self
      def create_api
        api = API.new(:Outline)
        default = Binding.new(:default)
        default.value_set = Binding::BOOLEAN
        default.required = false
        default.default = false
        expanded = Binding.new(:expanded)
        expanded.required = true
        expanded.value_set = Binding::BOOLEAN
        open_label = Binding.new(:open_label)
        open_label.default = 'open'
        close_label = Binding.new(:close_label)
        close_label.default = 'close'
        open_image = Binding.new(:open_image)
        open_image.value_set = Binding::RESOURCES
        open_image.default = 'outline_open_arrow.png'
        close_image = Binding.new(:close_image)
        close_image.value_set = Binding::RESOURCES
        close_image.default = 'outline_close_arrow.png'
        label_tag = Binding.new(:label_tag)
        label_tag.default = 'div'
        label_class = Binding.new(:label_class)
        expand_tag = Binding.new(:expand_tag)
        expand_tag.default = 'div'
        expand_class = Binding.new(:expand_class)
        api << default
        api << expanded
        api << open_label
        api << close_label
        api << open_image
        api << close_image
        api << label_tag
        api << label_class
        api << expand_tag
        api << expand_class
        api << package_binding('CGIKit')
        api << action_binding()
        api
      end
    end

    def begin_context( context )
      context.increment
      context.append_zero
    end

    def end_context( context )
      context.delete
    end

    def invoke_action( request, context )
      take_value(:action, false)
      take_bool(:expanded)

      result = nil
      if context.action?(request) then
        set_value(:expanded, !@values[:expanded])
        if @values[:action] then
          result = @root[@values[:action]]
        else
          result = @node.invoke_action(request, context)
        end
      end
      result
    end

    def append_to_response( response, context )
      take_bool(:default)
      take_bool(:expanded)
      take_value(:open_label)
      take_value(:close_label)
      take_value(:open_image)
      take_value(:close_image)
      take_value(:label_tag)
      take_value(:label_class)
      take_value(:expand_tag)
      take_value(:expand_class)
      take_value(:package)

      label_open_tag = label_close_tag = nil
      label_close_tag = "</#{@values[:label_tag]}>"
      expand_close_tag = "</#{@values[:expand_tag]}>"
      if klass = @values[:label_class] then
        label_open_tag = "<#{@values[:label_tag]} class=\"#{klass}\">"
      else
        label_open_tag = "<#{@values[:label_tag]}>"
      end
      if klass = @values[:expand_class] then
        expand_open_tag = "<#{@values[:expand_tag]} class=\"#{klass}\">"
      else
        expand_open_tag = "<#{@values[:expand_tag]}>"
      end

      expanded = @values[:expanded] || @values[:default]
      context.session
      url = context.component_action_url
      if expanded then
        response.content << label_open_tag
        if image = image_url(@values[:open_image], context.request) then
          response.content << "<a href=\"#{url}\"><image src=\"#{image}\"></a> "
        end
        response.content << "<a href=\"#{url}\">#{@values[:open_label]}</a>"
        response.content << label_close_tag
        response.content << expand_open_tag
        @node.append_to_response(response, context)
        response.content << expand_close_tag
      else
        response.content << label_open_tag
        if image = image_url(@values[:close_image], context.request) then
          response.content << "<a href=\"#{url}\"><image src=\"#{image}\"></a> "
        end
        response.content << "<a href=\"#{url}\">#{@values[:close_label]}</a>"
        response.content << label_close_tag
      end
    end

    def image_url( name, request )
      @application.resource_manager.url(name,
                                        @values[:package],
                                        request.languages,
                                        request)
    end

  end

end
module CGIKit

  class Delegate

    module SubmitDelegate

      def submit_should_invoke_action; end
      def submit_will_generate_tag( tag ); end

    end

  end


  class Submit < DynamicElement
    class << self
      def create_api
        api = API.new(:Sumbit)
        action = Binding.new(:action)
        value = Binding.new(:value)
        value.default = 'Submit'
        api << action
        api << value
        api << enabled_binding()
        api << name_binding()
        set_direct_action(api)
        api << any_validation([:action], [], [:direct_action, :action_class])
        api
      end

      def keys_to_delete_from_associations
        [:type]
      end
    end

    def begin_context( context )
      take_value(:name)
      context.increment(@values[:name])
    end

    def invoke_action( request, context )
      ask(:submit_should_invoke_action) do |d|
        return unless d.submit_should_invoke_action
      end

      take_value(:action, false)
      take_value(:direct_action, false)
      take_value(:action_class)

      result = nil
      if value_from_request(request, context) then
        if @values[:action] then
          result = @root[@values[:action]]
          notify_existing_action_to_form
        elsif @values[:direct_action] or @values[:action_class] then
          result = perform_direct_action(request, @values[:action_class],
                                         @values[:direct_action])
          notify_existing_action_to_form
        end
      end
      result
    end

    def notify_existing_action_to_form
      ancestor = @parent
      while ancestor do
        if Form === ancestor then
          ancestor.notify_existing_action_in_container
          break
        end
        ancestor = ancestor.parent
      end
    end

    def perform_direct_action( request, klass, action = DirectAction::DEFAULT_ACTION )
      unless klass then
        klass = @application.direct_action_class
      end
      direct_action = klass.new(@application, request)
      direct_action.perform_action(action)
    end

    def append_to_response( response, context )
      take_value(:value)
      take_bool(:enabled)
      attrs = {}
      if @values[:value] then
        @values[:value] = escaped_string(@values[:value])
      end

      attrs[:type] = 'submit'
      attrs[:name] = name_value(context)
      attrs[:value] = @values[:value]
      attrs[:disabled] = !@values[:enabled]
      attrs.update(optional_attributes())
      other_s = other_attribute()

      tag = HTMLTag.submit(attrs, other_s)
      ask(:submit_will_generate_tag) do |d|
        tag = d.submit_will_generate_tag(tag) || tag
      end
      response.content << tag.empty_tag
    end

  end

end
module CGIKit

  class Delegate

    module ConditionalDelegate

      def conditional_will_display; end
      def conditional_will_hide; end

    end

  end


  class Conditional < DynamicElement

    class << self
      def create_api
        api = API.new(:Conditional)
        condition = Binding.new(:condition)
        condition.required = true
        condition.value_set = Binding::BOOLEAN
        negate = Binding.new(:negate)
        negate.value_set = Binding::BOOLEAN
        negate.default = false
        api << condition
        api << negate
        api
      end
    end

    def begin_context( context )
      context.increment
      context.append_zero
    end

    def end_context( context )
      context.delete
    end

    def append_to_response( response, context )
      take_value(:condition)
      take_value(:negate)

      if (((not @values[:negate]) and @values[:condition]) or \
          ((not @values[:condition]) and @values[:negate])) then
        ask(:conditional_will_display, context) do |d|
          d.conditional_will_display
        end
        @node.append_to_response(response, context)
      else
        ask(:conditional_will_hide, context) do |d|
          d.conditional_will_hide
        end
      end
    end

  end

end
module CGIKit

  class Delegate

    module PopupDelegate

      def popup_will_generate_tags( select, options ); end

    end

  end


  class Popup < DynamicElement

    class << self
      def create_api
        api = API.new(:Popup)
        default = Binding.new(:default)
        selected_value = Binding.new(:selected_value)
        api << default
        api << selected_value
        api << name_binding()
        api << display_binding()
        api << list_binding()
        api << item_binding()
        api << escape_binding()
        api << value_binding(false, false)
        api << selection_binding()
        api << enabled_binding()
        api << item_display_value_validation()
        api << existential_validation(:selected_value, :selection)
        api
      end
    end

    def begin_context( context )
      take_value(:name)
      context.increment(@values[:name])
    end

    def take_values_from_request( request, context )
      take_value(:list)
      if (option = value_from_request(request, context)) and \
        context.current_form?(request) then
        if declared?(:selection) then
          if option.empty? then
            selection = nil
          elsif declared?(:value)
            @values[:list].each do |item|
              set_value(:item, item)
              take_value(:value)
              if option == @values[:value].to_s then
                selection = item
                break
              end
            end
          else
            selection = @values[:list][option.to_i]
          end
          set_value(:selection, selection)
        elsif declared?(:selected_value) then
          set_value(:selected_value, option)
        end
      end
    end

    def append_to_response( response, context )
      take_value(:list)
      take_bool(:escape)
      take_value(:default)
      take_value(:selection)
      take_value(:selected_value)
      take_bool(:enabled)
      selattrs = { :name => name_value(context), :disabled => !@values[:enabled] }
      selattrs.update(optional_attributes())
      option = value_from_request(context.request, context)
      seltag = HTMLTag.select(selattrs, other_attribute())
      optags = []

      if @values[:default] then
        attrs = { :selected => @values[:selection].nil?, :value => '' }
        optags << HTMLTag.option(attrs, '', @values[:default])
      end

      if @values[:list] then
        @values[:list].each_with_index do |item, index|
          set_value(:item, item)
          take_value(:value)
          take_value(:display)
          attrs = {}
          if (@values[:selection] == item) or \
            (option and !option.empty? and
               ((option == @values[:value].to_s) or \
                (option == index.to_s))) then
            attrs[:selected] = true
          end
          if @values[:value] then
            attrs[:value] = escaped_string(@values[:value], @values[:escape])
          else
            attrs[:value] = index
          end
          display = @values[:display] || item.to_s
          content = escaped_string(display, @values[:escape])
          optags << HTMLTag.option(attrs, '', content)
        end
      end

      ask(:popup_will_generate_tags) do |d|
        seltag, optags = d.popup_will_generate_tags(seltag, optags) || seltag, optags
      end

      response.content << seltag.open_tag
      response.content << "\n"
      optags.each do |tag|
        response.content << tag.container_tag
        response.content << "\n"
      end
      response.content << seltag.close_tag
    end
  end

end
module CGIKit

class Switcher < DynamicElement
  class << self
    def create_api
      api = API.new(:Switcher)
      component = Binding.new(:component)
      component.value_set = Binding::PAGE_NAMES
      component.required = true
      api << component
      api
    end
  end

  def begin_context( context )
    take_value_once(:component)
    context.increment
    unless @switch then
      page = context.session.restore_page(context.context_id)
      if page == root then
        @switch = application.page(@values[:component], context)
        @switch.parent = root
      else
        @switch = page
      end
      @switch.node = @node
      @switch.declaration_name = @name
      @switch.awake_from_restroration(context)
    end
    @switch.begin_context(context)
  end

  def end_context( context )
    @switch.end_context(context)
  end

  def take_values_from_request( request, context )
    @switch.take_values_from_request(request, context)
  end

  def invoke_action( request, context )
    @switch.invoke_action(request, context)
  end

  def append_to_response( response, context )
    @switch.append_to_response(response, context)
  end
end

end
module CGIKit

class Content < DynamicElement

  def take_values_from_request( request, context )
    preserve_context_id(context) do
      @root.node.take_values_from_request(request, context)
    end
  end

  def invoke_action( request, context )
    result = nil
    preserve_context_id(context) do
      result = @root.node.invoke_action(request, context)
    end
    result
  end

  def append_to_response( response, context )
    preserve_context_id(context) do
      @root.node.append_to_response(response, context)
    end
  end

  def preserve_context_id( context, &block )
    before = context.context_id
    context.context_id = @root.parent_context_id
    block.call
    context.context_id = before
  end

end

end
module CGIKit

  class Delegate

    module RadioDelegate

      def radio_will_generate_tag( tag ); end

    end

  end


  class Radio < DynamicElement
    class << self
      def create_api
        api = API.new(:Radio)
        api << name_binding()
        api << checked_binding()
        api << selection_binding()
        api << value_binding(false, false)
        api << enabled_binding()
        api << required_validation([:checked, :value])
        api << universal_validation(:value, :selection)
        api
      end

      def keys_to_delete_from_associations
        [:type]
      end
    end

    DEFAULT_VALUE = '1'

    def begin_context( context )
      take_value(:name)
      context.increment
      context.append_zero
      context.increment(@values[:name])
    end

    def end_context( context )
      context.delete
    end

    def take_values_from_request( request, context )
      take_bool(:checked)
      take_value(:value)
      take_value(:selection)

      if value = value_from_request(request, context) then
        if declared?(:checked) then
          set_value(:checked, true)
        elsif value == @values[:value] then
          set_value(:selection, @values[:value])
        end
      elsif context.current_form?(request) then
        if declared?(:checked) then
          set_value(:checked, false)
        else
          set_value(:selection, nil)
        end
      end
    end

    def append_to_response( response, context )
      take_bool(:enabled)
      take_bool(:checked)
      take_value(:value)
      take_value(:selection)
      attrs = { :name => name_value(context), :disabled => !@values[:enabled],
        :value => @values[:value] }
      attrs.update(optional_attributes())

      tag = HTMLTag.radio(attrs, other_attribute())
      if declared?(:checked) then
        attrs[:checked] = @values[:checked]
      else
        attrs[:value] = @values[:value]
        attrs[:checked] = \
          (@values[:selection] and (@values[:value] == @values[:selection]))
      end
      ask(:radio_will_generate_tag) do |d|
        tag = d.radio_will_generate_tag(tag) || tag
      end
      if tag[:value] then
        tag[:value] = escaped_string(tag[:value])
      else
        tag[:value] ||= DEFAULT_VALUE
      end
      response.content << tag.empty_tag
    end
  end

end

module CGIKit

  class Delegate

    module TextDelegate

      def text_set_value( value ); end
      def text_validate_value( value ); end
      def text_will_generate_tag( tag ); end

    end

  end


  class Text < DynamicElement

    class << self
      def create_api
        api = API.new(:Text)
        api << name_binding()
        api << value_binding(true, true)
        api << enabled_binding()
        set_validation(api)
        api
      end
    end

    def begin_context( context )
      take_value(:name)
      context.increment(@values[:name])
    end

    def take_values_from_request( request, context )
      if value = value_from_request(request, context) then
        ask(:text_validate_value) do |d|
          value = nil unless d.text_validate_value(value)
        end
        ask(:text_set_value) do |d|
          value = d.text_set_value(value)
        end
        set_value(:value, value)
        validate(:value)
      end
    end

    def append_to_response( response, context )
      take_value(:value)
      take_bool(:enabled)
      attrs = { :name => name_value(context),
        :disabled => !@values[:enabled] }
      attrs.update(optional_attributes())

      tag = HTMLTag.textarea(attrs, other_attribute(), @values[:value])
      ask(:text_will_generate_tag) do |d|
        tag = d.text_will_generate_tag(tag) || tag
      end
      if tag.content then
        tag.content = escaped_string(tag.content)
      end
      response.content << tag.container_tag
    end
  end

end
module CGIKit

  class Delegate

    module FormDelegete

      def form_should_invoke_action; end
      def form_take_values( values ); end
      def form_will_generate_tag( tag, query ); end

    end

  end


  class Form < DynamicElement

    MULTIPART_FORM_DATA = "multipart/form-data"

    class << self
      def create_api
        api = API.new(:Form)
        method = Binding.new(:method)
        method.default = 'POST'
        enctype = Binding.new(:enctype)
        query = Binding.new(:query)
        upload = Binding.new(:upload)
        upload.value_set = Binding::BOOLEAN
        upload.default = false
        action = Binding.new(:action)

        api << method
        api << enctype
        api << href_binding()
        api << query
        api << upload
        api << action
        api << name_binding()
        api << secure_binding()
        api << session_id_binding()
        api << existential_validation(:href, :action)
        api << existential_validation(:enctype, :upload)
        set_direct_action(api)
        api
      end
    end

    attr_accessor :has_action_in_container

    def init
      @has_action_in_container = false
    end

    def notify_existing_action_in_container
      @has_action_in_container = true
    end

    def begin_context( context )
      take_value(:name)
      context.increment
      context.append_zero
      context.in_form = true
      unless direct_action? then
        context.session.save_page(@root)
        context.component_id = context.session.component_id(@root)
      end
    end

    def end_context( context )
      context.delete
      context.in_form = false
    end

    def take_values_from_request( request, context )
      ask(:form_take_values) do |d|
        if values = d.form_take_values(request.form_values) then
          request.form_values = values
        end
      end
      @node.take_values_from_request(request, context)
    end

    def invoke_action( request, context )
      ask(:form_should_invoke_action) do |d|
        return unless d.form_should_invoke_action
      end

      take_value(:action, false)
      before = context.context_id
      result = @node.invoke_action(request, context)
      if !@has_action_in_container and \
        context.action?(request, before) and @values[:action] then
        result = @root[@values[:action]]
      end
      result
    end

    def append_to_response( response, context )
      take_value(:method)
      take_value(:enctype)
      take_value(:href)
      take_value(:query)
      take_bool(:secure)
      take_bool(:upload)
      take_value(:session_id)
      take_value(:direct_action, false)
      take_value(:action_class)

      attrs = {}
      unless (@values[:method].downcase == 'post') or \
        (@values[:method].downcase == 'get') then
        @values[:method] = 'post'
      end
      attrs[:method] = @values[:method]
      attrs[:enctype] = @values[:upload] ? MULTIPART_FORM_DATA : @values[:enctype]

      attrs[:name] = name_value(context)
      if @values[:href] then
        url = @values[:href]
      elsif direct_action? then
        url = context.direct_action_url(@values[:action_class],
                                        @values[:direct_action],
                                        {},
                                        @values[:session_id])
      else
        secure = has_binding?(:secure) ? @values[:secure] : context.request.https?
        url = context.component_action_url({}, secure)
      end
      attrs[:action] = url

      attrs.update(optional_attributes())
      other_s = other_attribute()

      tag = HTMLTag.form(attrs, other_s)
      query = @values[:query] || {}
      ask(:form_will_generate_tag) do |d|
        tag = d.form_will_generate_tag(tag, query) || tag
      end

      response.content << tag.open_tag
      response.content << "\n"
      response.content << hidden_fields(query)
      @node.append_to_response(response, context)
      response.content << tag.close_tag
    end


    def hidden_fields( query )
      fields = ''
      query.each do | key, value |
        value = Utilities.escape_html(value.to_s)
        fields << \
        "<input type=\"hidden\" name=\"#{key}\" value=\"#{value}\"/>\n"
      end
      fields
    end
  end

end
module CGIKit

  class RadioGroup < DynamicElement

    class << self

      def create_api
        api = API.new(:RadioGroup)
        index = Binding.new(:index)
        prefix = Binding.new(:prefix)
        suffix = Binding.new(:suffix)
        api << index
        api << prefix
        api << suffix
        api << list_binding()
        api << item_binding()
        api << selection_binding(true)
        api << name_binding()
        api << value_binding(false, false)
        api << display_binding()
        api << escape_binding()
        api << enabled_binding()
        api
      end

    end

    def begin_context( context )
      context.increment
      context.append_zero
    end

    def end_context( context )
      context.delete
    end

    def take_values_from_request( request, context )
      if value = value_from_request(request, context) then
        take_value(:list)
        set_value(:selection, @values[:list][value.to_i])
      end
    end

    def append_to_response( response, context )
      take_value(:list)
      take_value(:index, false)
      take_value(:name)
      take_value(:selection)
      take_bool(:escape)
      @values[:name] ||= name_value(context)

      @values[:list].each_with_index do |item, index|
        set_value(:item, item)
        set_value(:index, index) if @values[:index]
        take_value(:value)
        take_value(:display)
        take_value(:prefix)
        take_value(:suffix)
        take_bool(:enabled)
        attrs = { :name => @values[:name],
          :disabled => !@values[:enabled],
          :value => (@values[:value] || index.to_s),
          :checked => (@values[:selection] == item) }
        attrs.update(optional_attributes())

        tag = HTMLTag.radio(attrs, other_attribute())
        display = @values[:display] || item.to_s
        display = escaped_string(display) if @values[:escape]
        response.content << \
          "#{@values[:prefix]}#{tag.empty_tag} #{display}#{@values[:suffix]}\n"
        context.increment
      end
    end
  end

end

module CGIKit

  class Delegate

    module TextFieldDelegate

      def textfield_set_value( value ); end
      def textfield_validate_value( value ); end
      def textfield_will_generate_tag( tag ); end

    end

  end


  class TextField < DynamicElement
    class << self
      def create_api
        api = API.new(:TextField)
        type = Binding.new(:type)
        type.default = 'text'
        size = Binding.new(:size)
        maxlength = Binding.new(:maxlength)
        api << type
        api << size
        api << maxlength
        api << name_binding()
        api << value_binding(true, true)
        api << enabled_binding()
        set_validation(api)
        api
      end
    end

    def begin_context( context )
      take_value(:name)
      context.increment(@values[:name])
    end

    def take_values_from_request( request, context )
      if value = value_from_request(request, context) then
        ask(:textfield_validate_value) do |d|
          unless d.textfield_validate_value(value) then
            value = nil
          end
        end
        ask(:textfield_set_value) do |d|
          value = d.textfield_set_value(value)
        end
        set_value(:value, value)
        validate(:value)
      end
    end

    def append_to_response( response, context )
      take_bool(:enabled)
      take_value(:value)
      take_value(:type)
      take_value(:size)
      take_value(:maxlength)
      attrs = { :type => @values[:type], :value => @values[:value],
        :size => @values[:size], :maxlength => @values[:maxlength],
        :name => name_value(context), :disabled => !@values[:enabled] }
      attrs.update(optional_attributes())

      tag = HTMLTag.textfield(attrs, other_attribute())
      ask(:textfield_will_generate_tag) do |d|
        tag = d.textfield_will_generate_tag(tag) || tag
      end
      if tag[:value] and (tag[:type] != 'password') then
        tag[:value] = escaped_string(tag[:value])
      end
      response.content << tag.empty_tag
    end
  end

end
module CGIKit

class Frame < DynamicElement

  class << self
    def create_api
      api = API.new(:Frame)
      api << page_binding()
      api << src_binding()
      api << value_binding(false)
      api << required_validation([:page, :src, :value])
      api
    end
  end

  def begin_context( context )
    context.increment
  end

  def invoke_action( request, context )
    result = nil
    if context.action?(request) then
      result = context.session.component(context.context_id)
    end
    result
  end

  def append_to_response( response, context )
    take_value(:src)

    src = nil
    if @values[:src] then
      src = values[:src]
    else
      src = context.url(context.request.request_handler_key)
    end
    html =  "<frame src=\"#{src}\""
    html << other()
    html << " />"
    response.content << html

    # the reason is if a browser accesses each frames concurrently,
    # context IDs may be repeated.
    generate_and_register_components(context)
  end

  def generate_and_register_components( context )
    if declared?(:page) or declared?(:value) then
      if declared?(:page) then
        take_value(:page)
        page = application.page(@values[:page], context)
      elsif declared?(:value) then
        page = value(:value)
      end
      session = context.session
      session.save_page(page)
      component_id = session.component_id(page)
      session.frame_components[context.context_id] = component_id
    end
  end

end

end
module CGIKit

  class Redirect < StatelessComponent

    class << self
      def create_api
        api = API.new(:Redirct)
        url = Binding.new(:url)
        api << url
        api
      end
    end

    attr_reader :url

    def init( url = nil )
      self.url = url if url
    end

    def url=( url )
      response.set_redirect(url)
    end

    def load_files( context )
      @context = context
      @application = context.application
      load_associations_from_parent_declaration
      @loaded = true
    end

    def take_values_from_request( request, context )
    end

    def invoke_action( request, context )
    end

    def append_to_response( response, context )
      sync(context)
    end

  end

end
module CGIKit

class Upload < DynamicElement

  class << self
    def create_api
      api = API.new(:Upload)
      api << value_binding()
      api << enabled_binding()
      set_validation(api)
      api
    end

    def keys_to_delete_from_associations
      [:type]
    end
  end

  def begin_context( context )
    take_value(:name)
    context.increment(@values[:name])
  end

  def take_values_from_request( request, context )
    if (value = value_from_request(request, context)) and (ByteData === value) then
      set_value(:value, value)
      validate(:value)
    end
  end

  def append_to_response( response, context )
    html  = "<input type=\"file\" name=\"#{name_value(context)}\""
    html << other()
    html << enabled()
    html << " />"
    response.content << html
  end
end

end
module CGIKit

class GenericElement < DynamicElement

  NO_CLOSE_TAGS = [ 'area', 'base', 'basefont', 'br', 'col', 'frame', 'hr',
                    'img', 'input', 'link', 'map', 'meta', 'param' ]
  NAME_TAGS = ['frame', 'form', 'input', 'select', 'textarea', 'button',
               'a', 'img', 'map', 'applet', 'iframe', 'meta', 'object', 'param']
  VALUE_TAGS = ['param', 'li', 'input', 'option', 'button']
  HREF_TAGS = ['a', 'area']

  class << self
    def create_api
      api = API.new(:GenericElement)
      tag = Binding.new(:tag)
      tag.required = true
      displayed = Binding.new(:displayed)
      displayed.value_set = Binding::BOOLEAN
      displayed.default = true
      form_value = Binding.new(:form_value)
      form_values = Binding.new(:form_values)
      invoke_action = Binding.new(:invoke_action)
      api << tag
      api << displayed
      api << form_value
      api << form_values
      api << invoke_action
      api << name_binding()
      api
    end
  end


  #
  # testing
  #

  def form?
    @values[:tag].downcase == 'form'
  end

  def link?
    @values[:tag].downcase == 'a'
  end

  def no_close_tag?( tag )
    NO_CLOSE_TAGS.include?(tag)
  end

  def has_name?( tag )
    NAME_TAGS.include?(tag)
  end

  def has_value?( tag )
    VALUE_TAGS.include?(tag)
  end

  def has_href?( tag )
    HREF_TAGS.include?(tag)
  end


  #
  # request-response loop
  #

  def begin_context( context )
    take_value(:name)
    take_value(:tag)
    context.increment(@values[:name])
    if form? then
      context.append_zero
      context.in_form = true
    elsif link? then
      context.append_zero
    end
  end

  def end_context( context )
    if form? then
      context.delete
      context.in_form = false
    elsif link? then
      context.delete
    end
  end

  def take_values_from_request( request, context )
    if form_values = values_from_request(request, context) then
      if declared?(:form_value) then
         set_value(:form_value, form_values[0])
      elsif declared?(:form_values) then
         set_value(:form_values, form_values)
      end
    end
    @node.take_values_from_request(request, context)
  end

  def invoke_action( request, context )
    take_value(:invoke_action, false)
    result = nil
    if declared?(:invoke_action) then
      if (link? and context.action?(request)) or \
        value_from_request(request, context) then
        result = @root[@values[:invoke_action]]
      end
    else
      result = @node.invoke_action(request, context)
    end
    result
  end

  def append_to_response( response, context )
    take_bool(:displayed)
    take_value(:form_value)
    take_value(:form_values)
    unless @values[:displayed] then return end

    if @values[:form_value] then
      value = @values[:form_value]
    elsif @values[:form_values] then
      value = @values[:form_values]
    end

    html =  "<#{@values[:tag]}"
    if has_name?(@values[:tag]) then
      html << " name=\"#{name_value(context)}\""
    end
    if has_value?(@values[:tag]) and \
      (declared?(:form_value) or declared?(:form_values)) then
      html << " value=\"#{value}\""
    end
    if has_href?(@values[:tag]) and declared?(:invoke_action) then
      url = context.url(context.request.request_handler_key)
      html << " href=\"#{url}\""
    end
    html << other()
    if no_close_tag?(@values[:tag]) then
      html << ' />'
    else
      html << '>'
    end
    response.content << html
    unless empty? then
      @node.append_to_response(response, context)
    end

    unless no_close_tag?(@values[:tag]) then
      response.content << "</#{@values[:tag]}>"
    end
  end

end

end
module CGIKit

  class Refresh < Link

    class << self
      def create_api
        api = API.new(:Refresh)
        api << Binding.new(:seconds)
        api << Binding.new(:url)
        api << action_binding()
        api << page_binding()
        api << secure_binding()
        api << query_binding()
        api << frag_binding()
        api << session_id_binding()
        set_direct_action(api)
        api << required_validation([:action, :url, :page], [], \
                                   [:direct_action, :action_class])
        api
      end
    end

    def invoke_action( request, context )
      take_value(:page)
      take_value(:action, false)
      result = nil
      if context.action?(request) then
        if @values[:page] then
          result = @application.page(@values[:page], context)
        else
          result = @root[@values[:action]]
        end
      end
      result
    end

    def append_to_response( response, context )
      take_value(:seconds)
      take_value(:url)

      unless url = @values[:url] then
        url = action_url(context)
      end
      html =  "<meta http-equiv=\"refresh\" "
      html << "content=\"#{@values[:seconds]}; URL=#{url}\""
      html << other()
      html << ' />'
      response.content << html
    end

  end

end
module CGIKit

  class Delegate

    module ImageDelegate

      def image_generate_url( url ); end
      def image_will_generate_tag( tag ); end

    end

  end


  class Image < DynamicElement
    include ResourceLoadable

    class << self
      def create_api
        api = API.new(:Image)
        data = Binding.new(:data)
        file = Binding.new(:file)
        file.value_set = Binding::RESOURCES
        mime = Binding.new(:mime)
        mime.value_set = Binding::MIME_TYPES
        key = Binding.new(:key)
        api << data
        api << file
        api << mime
        api << key
        api << src_binding()
        api << package_binding()
        api << required_validation([:file, :src, :data])
        api
      end
    end

    def append_to_response( response, context )
      take_value(:file)
      take_value(:package)
      take_value(:data)
      take_value(:key)
      take_value(:mime)
      take_value(:src)

      unless @values[:src] then
        @values[:src] = caching_url(@values[:file], @values[:package],
                                    @values[:data], @values[:key],
                                    @values[:mime], context.request)
      end
      ask(:image_generate_url) do |d|
        @values[:src] = d.image_generate_url(@values[:src])
      end

      attrs = { :src => @values[:src] }
      attrs.update(optional_attributes())
      tag = HTMLTag.img(attrs, other_attribute())
      ask(:image_will_generate_tag) do |d|
        tag = d.image_will_generate_tag(tag) || tag
      end
      response.content << tag.empty_tag
    end
  end

end
module CGIKit

  class Delegate

    module RepetitionDelegate

      def repetition_should_repeat( list ); end
      def repetition_should_each_repeat( list ); end
      def repetition_will_each_repeat( item, index ); end
      def repetition_did_each_repeat( item, index ); end

    end

  end


  module ElementEnumerable

    def each( context )
      ask(:repetition_should_repeat, context) do |d|
        unless d = d.repetition_should_repeat(@values[:list]) then
          return
        end
      end
      flag = true
      each_node(context) do |node|
        flag = true
        ask(:repetition_should_each_repeat, context) do |d|
          flag = d.repetition_should_each_repeat(@values[:list])
        end
        yield node if flag
        increment_context_for_next_loop(context)
      end
    end

    def each_node( context )
      if list = @values[:list] then
        if Hash === list then
          list.each_with_index do |item, index|
            set_value(:key, item[0])
            set_value(:item, item[1])
            set_value(:index, index) if @values[:index]
            @node.reset
            delegate_phase(context, item, index) do
              yield @node
            end
          end
        else
          index = 0
          list.each do |item|
            set_value(:item, item)
            set_value(:index, index) if @values[:index]
            @node.reset
            delegate_phase(context, item, index) do
              yield @node
            end
            index += 1
          end
        end
      elsif @values[:count] then
        @values[:count].times do |index|
          set_value(:index, index) if @values[:index]
          @node.reset
          delegate_phase(context, nil, index) do
            yield @node
          end
        end
      end
    end

    def increment_context_for_next_loop( context )
      context.delete
      context.increment
      context.append_zero
    end

    def delegate_phase( context, item, index, &block )
      ask(:repetition_will_each_repeat, context) do |d|
        d.repetition_will_each_repeat(item, index)
      end
      block.call
      ask(:repetition_did_each_repeat, context) do |d|
        d.repetition_did_each_repeat(item, index)
      end
    end
  end


  class Repetition < DynamicElement
    include ElementEnumerable

    class << self
      def create_api
        api = API.new(:Repetition)
        index = Binding.new(:index)
        count = Binding.new(:count)
        key = Binding.new(:key)
        key.settable = true
        api << list_binding(false)
        api << item_binding()
        api << index
        api << count
        api << key
        api << required_validation([:count, :list])
        api
      end
    end

    def begin_context( context )
      context.increment
      context.append_zero
    end

    def end_context( context )
      context.delete
    end

    def take_values_from_request( request, context )
      take_value(:index, false)
      take_value(:count)
      take_value(:list)

      each(context) do |node|
        node.take_values_from_request(request, context)
      end
    end

    def invoke_action( request, context )
      result = nil
      each(context) do |node|
        if node_result = node.invoke_action(request, context) then
          result = node_result
        end
      end
      result
    end

    def append_to_response( response, context )
      take_value(:list)
      take_value(:index, false)
      take_value(:count)

      each(context) do |node|
        node.append_to_response(response, context)
      end
    end

  end

end
module CGIKit

  class ActionURL < Link
    include Link::Utilities

    class << self
      def create_api
        api = API.new(:ActionURL)
        api << action_binding()
        api << page_binding()
        api << secure_binding()
        api << query_binding()
        api << frag_binding()
        api << session_id_binding()
        set_direct_action(api)
        api << required_validation([:action, :page], [], \
                                   [:direct_action, :action_class])
        api
      end
    end

    def begin_context( context )
      context.increment
    end

    def end_context( context ); end

    def append_to_response( response, context )
      response.content << action_url(context)
    end

  end

end
module CGIKit

  class ResourceURL < DynamicElement
    include ResourceLoadable

    class << self
      def create_api
        api = API.new(:Image)
        data = Binding.new(:data)
        file = Binding.new(:file)
        file.value_set = Binding::RESOURCES
        mime = Binding.new(:mime)
        mime.value_set = Binding::MIME_TYPES
        key = Binding.new(:key)
        api << data
        api << file
        api << package_binding()
        api << mime
        api << key
        api << required_validation([:file, :data])
        api
      end
    end

    def append_to_response( response, context )
      take_value(:file)
      take_value(:package)
      take_value(:data)
      take_value(:key)
      take_value(:mime)
      url = caching_url(@values[:file], @values[:package],
                        @values[:data], @values[:key],
                        @values[:mime], context.request)
      response.content << url
    end
  end

end
module CGIKit

  class ImageLink < Link
    include ResourceLoadable
    include Link::Utilities

    class << self
      def create_api
        api = API.new(:ImageLink)
        data = Binding.new(:data)
        key = Binding.new(:key)
        api << data
        api << key
        api << file_binding()
        api << package_binding()
        api << mime_binding()
        api << src_binding()
        api << action_binding()
        api << enabled_binding()
        api << href_binding()
        api << page_binding()
        api << secure_binding()
        api << query_binding()
        api << frag_binding()
        api << session_id_binding()
        set_direct_action(api)
        api << required_validation([:action, :href, :page], [], \
                                   [:direct_action, :action_class])
        api
      end
    end

    def begin_context( context )
      context.increment
    end

    def end_context( context ); end

    def append_to_response( response, context )
      take_value(:file)
      take_value(:package)
      take_value(:data)
      take_value(:key)
      take_value(:mime)
      take_value(:src)
      take_bool(:enabled)
      aattrs = { :name => name_value(context), :disabled => !@values[:enabled] }
      aattrs.update(optional_attributes(:a))
      imgattrs = { :src => @values[:src] }
      imgattrs.update(optional_attributes(:img))
      unless imgattrs[:src] then
        imgattrs[:src] = caching_url(@values[:file], @values[:package],
                                     @values[:data], @values[:key],
                                     @values[:mime], context.request)
      end
      ask(:image_generate_url) do |d|
        imgattrs[:src] = d.image_generate_url(imgattrs[:src])
      end
      imgtag = HTMLTag.img(imgattrs, other_attribute(:img))
      ask(:image_will_generate_tag) do |d|
        imgtag = d.image_will_generate_tag(imgtag) || imgtag
      end
      atag = HTMLTag.a(aattrs, other_attribute(:a), imgtag.empty_tag)
      set_href(atag, context)
      response.content << atag.container_tag
    end

  end

end
module CGIKit

  class ImageSubmit < Submit

    module Utilities

      def x_y_values_from_request( request, context )
        xnames = ["#{context.context_id}.x"]
        ynames = ["#{context.context_id}.y"]
        if @values[:name] then
          xnames << "#{@values[:name].to_s}.x"
          ynames << "#{@values[:name].to_s}.y"
        end
        x = y = nil
        xnames.each do |name|
          if x = request.form_values[name] then
            x = x[0].to_i
            break
          end
        end
        ynames.each do |name|
          if y = request.form_values[name] then
            y = y[0].to_i
            break
          end
        end
        [x, y]
      end

    end

    include ResourceLoadable
    include ImageSubmit::Utilities

    DELETE_KEYS = [:type]

    class << self

      def create_api
        api = API.new(:Sumbit)
        action = Binding.new(:action)
        value = Binding.new(:value)
        value.default = 'Submit'
        data = Binding.new(:data)
        key = Binding.new(:key)
        api << data
        api << key
        api << action
        api << value
        api << enabled_binding()
        api << name_binding()
        api << file_binding()
        api << package_binding()
        api << mime_binding()
        api << src_binding()
        api << x_binding()
        api << y_binding()
        set_direct_action(api)
        api << any_validation([:action], [], [:direct_action, :action_class])
        api
      end

      def keys_to_delete_from_associations
        DELETE_KEYS
      end

    end

    def take_values_from_request( request, context )
      x, y = x_y_values_from_request(request, context)
      if x and y then
        set_value(:x, x)
        set_value(:y, y)
      end
    end

    def append_to_response( response, context )
      take_value(:file)
      take_value(:package)
      take_value(:data)
      take_value(:key)
      take_value(:mime)
      take_value(:value)
      take_value(:src)
      take_bool(:enabled)
      attrs = { :type => 'image', :name => name_value(context),
        :value => @values[:value], :disabled => !@values[:enabled],
        :src => @values[:src] }
      attrs.update(optional_attributes())
      unless attrs[:src] then
        attrs[:src] = caching_url(@values[:file], @values[:package],
                                  @values[:data], @values[:key],
                                  @values[:mime], context.request)
      end
      ask(:image_generate_url) do |d|
        attrs[:src] = d.image_generate_url(attrs[:src])
      end
      attrs[:value] = escaped_string(attrs[:value]) if attrs[:value]
      tag = HTMLTag.input(attrs, other_attribute())
      ask(:submit_will_generate_tag) do |d|
        tag = d.submit_will_generate_tag(tag) || tag
      end
      response.content << tag.empty_tag
    end

  end

end
