require 'monitor'

module CGIKit

  VERSION = '2.0.0'

  class CGIKitError < StandardError
  end


  class Application

    include KeyValueCoding, Logging

    class SessionCreationError < StandardError #:nodoc:
    end
    class SessionRestorationError < StandardError #:nodoc:
    end
    class SessionAuthorizationError < SessionRestorationError #:nodoc:
    end
    class SessionTimeoutError < SessionRestorationError #:nodoc:
    end
    class PageRestorationError < StandardError #:nodoc:
    end

    COMPONENT_REQUEST_HANDLER_KEY     = 'c'
    DIRECT_ACTION_REQUEST_HANDLER_KEY = 'd'
    RESOURCE_REQUEST_HANDLER_KEY      = 'r'

    # backward compatibility
    CGIKIT_LIB    = 'cgikit'
    COMPONENT_LIB = 'components'

    DATA_PATH    = 'data'
    CGIKIT_PATH  = 'cgikit'
    PACKAGE_PATH = 'packages'

    @@handlers = {
      COMPONENT_REQUEST_HANDLER_KEY => ComponentRequestHandler,
      DIRECT_ACTION_REQUEST_HANDLER_KEY => DirectActionRequestHandler,
      RESOURCE_REQUEST_HANDLER_KEY => ResourceRequestHandler
    }

    class << self

      def request_handlers
        @@handlers
      end

      def request_handler_key( klass )
        @@handlers.index(klass)
      end

      def request_handler_class( key )
        @@handlers[key]
      end

      def register_request_handler( key, klass )
        if @@handlers[key] then
          raise "request handler key '#{key}' is already registered."
        end
        @@handlers[key] = klass
      end

      def remove_request_handler( key )
        @handlers.delete(key)
      end

    end


    # Main component. If session ID or context ID aren't specified,
    # this component is shown. The default value is MainPage.
    attr_accessor :main

    # Document root directory.
    attr_accessor :document_root

    # The application URL based on SCRIPT_NAME.
    attr_accessor :baseurl

    # The file system path of the application.
    attr_accessor :path

    # The file system paths for components. Components are searched under it.
    attr_accessor :component_paths
    attr_accessor :component_path

    # Resource directory.
    # This directory includes files to be used by the application,
    attr_accessor :resource_path
    alias resources resource_path
    alias resources= resource_path=

    # Web server resources directory.
    # This directory includes files to be displayed to browser.
    # The files are used by Image element, etc.
    attr_accessor :web_server_resource_path
    alias web_server_resources web_server_resource_path
    alias web_server_resources= web_server_resource_path=

    # ResourceManager object.
    attr_accessor :resource_manager

    # Adapter object.
    attr_accessor :adapter

    # Adapter class. The default value is CGI class.
    attr_accessor :adapter_class

    # Name or class of an error page component to show caught errors.
    attr_accessor :error_page

    # Temporary directory to be used by the framework.
    # The framework uses this to store sessions and template caches.
    attr_accessor :tmpdir

    # Session key. This key is used in cookie.
    attr_accessor :session_key

    # Session key in direct action.
    # This key is used in hidden fields of form and URL when using direct action.
    attr_accessor :direct_action_session_key

    # Seconds until the session has timed out.
    attr_accessor :timeout

    # Expiry date of cookie for session. If you set the value to nil,
    # session cookies will be invalid when closing browser.
    attr_accessor :session_cookie_expires

    # Enables or disables the use of URLs for storing session IDs.
    attr_accessor :store_in_url

    # Enables or disables the use of cookies for storing session IDs.
    attr_accessor :store_in_cookie

    # Enables or disables session authorization by browsers.
    # If you set the value to true, the application raises error
    # when an user accesses it with browser that is different from
    # one registered session.
    attr_accessor :auth_by_user_agent

    # Enables or disables session authorization by IP addresses.
    # If you set the value to true, the application raises error
    # when an user accesses it with IP address that is different from
    # one registered session.
    attr_accessor :auth_by_remote_addr

    # Encoding to encode character code of form data.
    # The default implementation uses Kconv to encode Japanese character codes.
    # Then specify constant values of Kconv; Kconv::JIS, Kconv::SJIS, Kconv::EUC, etc.
    # If the value is nil, form data is not encoded. The default value is nil.
    attr_accessor :encoding

    # Request handler key to display components.
    attr_accessor :component_request_handler_key

    # Request handler key to invoke direct actions.
    attr_accessor :direct_action_request_handler_key

    # Request handler key to display resource files.
    attr_accessor :resource_request_handler_key

    # Default request handler key. Default key is component request handler key.
    attr_accessor :default_request_handler

    # Request handler for components.
    attr_accessor :component_request_handler

    # Request handler for direct actions.
    attr_accessor :direct_action_request_handler

    # Request handler for resources.
    attr_accessor :resource_request_handler

    # Session class. Default class is CGIKit::Session.
    attr_accessor :session_class

    # Whether or not validates setting of attributes for each elements.
    # If wrong attribute name or combination are found, raises error.
    attr_accessor :validate_api
    alias validate_api? validate_api

    # HTML parser class. Default class is CGIKit::HTMLParser::HTMLParser.
    attr_accessor :htmlparser_class

    # Size to cache components permanently in session.
    # Newly generated page is cached automatically.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :page_cache_size

    # Size to cache components permanently in session.
    # Permanent page cache is cached optionally, not automatically caching.
    # If holded page size is over the value, oldest pages are deleted.
    attr_accessor :permanent_page_cache_size

    # Direct action class. Default class is CGIKit::DirectAction.
    attr_accessor :direct_action_class

    # Session store class. Default class is CGIKit::FileSessionStore.
    attr_accessor :session_store_class

    # Context class. Default class is CGIKit::Context.
    attr_accessor :context_class

    # Whether or not caches templates to reduce parsing load.
    attr_accessor :cache_template

    attr_accessor :template_store
    attr_accessor :template_store_class
    attr_accessor :resource_store
    attr_accessor :resource_store_class
    attr_accessor :sweep_password
    attr_accessor :datadir
    attr_accessor :package_paths
    attr_accessor :required_packages
    attr_accessor :main_package_options
    attr_accessor :model_path
    attr_accessor :database
    attr_accessor :logger
    attr_accessor :log_options
    attr_accessor :concurrent_request_handling

    def initialize
      init_attributes
      init_request_handlers
      init_adapter
      init_component_paths
      init_name_spaces
      init
    end

    def init_attributes
      require 'rbconfig'
      @main                      = 'MainPage'
      @error_page                = 'ErrorPage'
      @tmpdir                    = './tmp' || ENV['TMP'] || ENV['TEMP']
      @datadir                   = Config::CONFIG['datadir']
      @package_paths             = [PACKAGE_PATH,
        File.join(@datadir, CGIKIT_PATH, PACKAGE_PATH),
        File.join(DATA_PATH, CGIKIT_PATH, PACKAGE_PATH)]
      @session_key               = '_session_id'
      @direct_action_session_key = '_sid'
      @manage_session            = false
      @timeout                   = 60 * 60 * 24 * 7
      @session_cookie_expires    = 60 * 60 * 24 * 7
      @store_in_url              = true
      @store_in_cookie           = false
      @auth_by_user_agent        = false
      @auth_by_remote_addr       = false
      @session_class             = Session
      @session_store             = nil
      @session_store_class       = FileSessionStore
      @template_store            = nil
      @template_store_class      = FileTemplateStore
      @resource_store            = nil
      @resource_store_class      = FileResourceStore
      @encoding                  = nil
      @resource_path             = Package::RESOURCE_PATH
      @web_server_resource_path  = Package::WEB_SERVER_RESOURCE_PATH
      @model_path                = Package::MODEL_PATH
      @validate_api              = true
      @cache_template            = true
      @htmlparser_class          = HTMLParser::HTMLParser
      @page_cache_size           = 30
      @permanent_page_cache_size = 30
      @direct_action_class       = DirectAction
      @context_class             = Context
      @baseurl                   = nil
      @required_packages         = []
      @main_package_options      = {}
      @request_handlers          = {}
      @concurrent_request_handling = true
      @lock                      = Monitor.new
      @log_options = {:level => nil, :name => 'CGIKit', :out => $stderr}
      if defined?(TapKit::Application) then
        @database = TapKit::Application.new
      end
    end

    def init_request_handlers
      self.class.request_handlers.each do |key, klass|
        register_request_handler(key, klass)
      end
      @component_request_handler_key     = COMPONENT_REQUEST_HANDLER_KEY
      @direct_action_request_handler_key = DIRECT_ACTION_REQUEST_HANDLER_KEY
      @resource_request_handler_key      = RESOURCE_REQUEST_HANDLER_KEY
      @component_request_handler =
        request_handler(@component_request_handler_key)
      @direct_action_request_handler =
        request_handler(@direct_action_request_handler_key)
      @resource_request_handler =
        request_handler(@resource_request_handler_key)
      @default_request_handler = @component_request_handler
    end

    def init_adapter
      # decides interface of adapter
      if defined?(MOD_RUBY) then
        @adapter_class = Adapter::ModRuby
        @path = Apache.request.filename
      else
        @adapter_class = Adapter::CGI
        @path = $0
      end
    end

    # backward compatibility
    def init_component_paths
      @component_paths = ['components', '.']
    end

    def init_name_spaces
      @name_spaces = [CGIKit, Object]
      klass = Object
      self.class.to_s.split('::').each do |class_name|
        klass = klass.const_get(class_name)
        @name_spaces << klass
      end
    end

    # Returns the name of the application without file extension.
    def name
      File.basename( @path, '.*' )
    end

    def take_values_from_hash( hash )
      hash.each do |key, value|
        self[key] = value
      end
    end

    def template_store
      @template_store ||= @template_store_class.new(self)
    end

    def resource_store
      @resource_store ||= @resource_store_class.new(self)
    end

    # backward compatibility
    def load_all_components( path, subdir = false )
      __each_component_file(path, subdir) do |file|
        require(file)
      end
    end

    # backward compatibility
    def autoload_all_components( path, mod = Object, subdir = false )
      __each_component_file(path, subdir) do |file|
        class_name = File.basename(file, '.rb')
        mod.autoload(class_name, file)
      end
    end

    private

    # backward compatibility
    def __each_component_file(path, subdir)
      if subdir
        pattern = File.join(path, '**', '*.rb')
      else
        pattern = File.join(path, '*.rb')
      end
      
      Dir.glob(pattern) do |file|
        if /\.rb$/ === file and FileTest::readable?(file)
          yield file
        end
      end
    end
    
    public

    def load_configuration( path )
      load(path)
      configure
    end

    def parse_request_handler_key( request )
      key = nil
      if info = request.request_handler_path then
        info = info.reverse.chop.reverse
        separated = info.split('/')
        key = separated.first
      end
      key
    end


    #
    # hook
    #

    def init; end

    def configure; end


    #
    # managing sessions
    #

    # Session database object (SessionStore).
    def session_store
      @session_store ||= @session_store_class.new(self)
    end

    # Creates a session.
    def create_session( request )
      begin
        klass = @session_class || Session
        session = klass.new(request.session_id)
        session.awake_from_restoration(self, request)
        return session
      rescue
        if sid = request.session_id then
          msg = "for #{sid}"
        else
          msg = "because session id is not specified"
        end 
        raise SessionCreationError, "Failed creating session #{msg}."
      end
    end

    # Returns a restored session objects with session ID.
    def restore_session( session_id, context )
      begin
        session = session_store.checkout(session_id, context.request)
      rescue Exception => e
        raise SessionRestorationError, "Failed restoring session for #{session_id}"
      end
      if session then
        context.session = session
        context.session.awake_from_restoration(self, context.request)
        context.session.validate
      elsif session_id and session.nil? then
        raise SessionTimeoutError, 'Your session has timed out.'
      end
      session
    end

    # Saves the session, and set a cookie if "store_in_cookie" attribute is
    # setted. If "clear" method of the session is called, the session is deleted.
    def save_session( context )
      unless context.has_session? then return end

      if context.session.terminate? then
        sid = context.session.session_id
        @session_store.remove(sid)
      else
        session_store.checkin(context)
        if context.session.store_in_cookie? then
          context.session.set_cookie
        end
      end
    end


    #
    # handling requests
    #

    # Runs the application.
    def run( command_or_request = nil, response = nil )
      if CGIKit.const_defined?(:Command) and \
        CGIKit.const_get(:Command) === command_or_request then
        request = command_or_request.request
      else
        request = command_or_request
      end

      set_adapter()
      @adapter.run(request, response) do |request|
        info('=== begin request response Loop', false)
        begin  
          set_attributes_from_request(request)
          request.request_handler_key = parse_request_handler_key(request)
          handler = request_handler(request.request_handler_key)
          request.context_id = handler.context_id(request)
          response = handler.handle_request(request)

        rescue Exception => e
          debug("exception: #{e.message} (#{e.backtrace.inspect})")

          # create context without session
          request.session_id = nil
          context = @context_class.new(request, self)
          begin
            response = handle_error(e, context)
          rescue Exception => e
            # trap error occured by customized handle_error 
            response = default_error_page(e, context)
          end
        end

        info('=== end request response loop', false)
        response
      end
      response
    end

    def request_handler_key( handler )
      @request_handlers.index(handler)
    end

    def request_handler( key )
      unless handler = @request_handlers[key] then
        handler = @default_request_handler
      end
      handler
    end

    def register_request_handler( key, klass )
      if @request_handlers[key] then
        raise "request handler key '#{key}' is already registered."
      end
      @request_handlers[key] = klass.new(self)
    end

    def remove_request_handler( key )
      @request_handlers.delete(key)
    end

    def set_adapter
      if @set_adapter_and_handler then return end
      @adapter = create_adapter
      @set_adapter_and_handler = true
      unless @logger then
        if level = @log_options[:level] then
          require 'logger'
          case level
          when :fatal
            level = Logger::FATAL
          when :error
            level = Logger::ERROR
          when :warn
            level = Logger::WARN
          when :info
            level = Logger::INFO
          else
            level = Logger::DEBUG
          end
          if file = @log_options[:file] then
            out = File.open(file, 'a+')
          else
            out = @log_options[:out]
          end
          @logger = Logger.new(out, @log_options[:shift_age] || 0,
                               @log_options[:shift_size] || 1048576)
          @logger.progname = @log_options[:name]
        else
          @logger = DummyLogger.new
        end
      end
    end

    def set_attributes_from_request( request )
      if @set_attributes_from_request then return end
      @baseurl       = request.script_name              unless @baseurl
      @document_root = request.headers['DOCUMENT_ROOT'] unless @document_root
      set_resource_manager()
      @set_attributes_from_request = true
    end

    def set_resource_manager
      # backward compatibility
      unless @component_path then
        @component_paths.each do |path|
          if FileTest.exist?(path) then
            @component_path = path
          end
        end
      end
      @resource_path = @resources if @resources
      @web_server_resource_path = @web_server_resources unless @web_server_resources

      @resource_manager = ResourceManager.new(self, @main_package_options)
      @required_packages.each do |name|
        @resource_manager.load_package(name)
      end
    end

    def take_values_from_request( request, context )
      context.session.take_values_from_request(request, context)
    end

    def invoke_action( request, context )
      context.session.invoke_action(request, context)
    end

    def append_to_response( response, context )
      context.session.append_to_response(response, context)
    end

    def create_context( request )
      handler = request_handler(request.request_handler_key)
      context = @context_class.new(request, self)
      
      if session = restore_session(request.session_id, context) then
        context.session = session
      else
        session = create_session(request)
        context.session = session
      end
      session.context = context
      if component = context.session.restore_page(context.sender_id) then
        root = component.root
        context.component = root
        root.awake_from_restoration(context)
      else
        context.component = page(@main, context)
      end
      context.delete_all
      context
    end

    # Creates an adapter object.
    def create_adapter
      @adapter_class.new
    end

    def synchronize( &block )
      unless @concurrent_request_handling then
        @lock.enter
      end
      begin
        block.call
      ensure
        unless @concurrent_request_handling then
          @lock.exit
        end
      end
    end


    #
    # handling errors
    #

    # Handles every errors and return an error page component.
    def handle_error( error, context )
      case error
      when SessionCreationError 
        response = handle_session_creation_error(error, context)
      when SessionRestorationError 
        response = handle_session_restoration_error(error, context)
      when PageRestorationError 
        response = handle_page_restoration_error(error, context)
      else
        page = default_error_page(error, context)
        response = page.generate_response
      end
      response
    end

    def handle_session_creation_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_session_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    def handle_page_restoration_error( error, context )
      default_error_page(error, context).generate_response
    end

    # Return a default error page component.
    def default_error_page( error, context )
      error_page       = page(@error_page, context)
      error_page.error = error
      error_page
    end

    def url( key, context, path, query_string, is_secure, port = nil, sid = true )
      request_handler(key).url(context, path, query_string, is_secure, port, sid)
    end

    def direct_action_url( context, action_class, action_name, query, sid = true )
      handler = @direct_action_request_handler
      handler.action_url(context, action_class, action_name, query, sid)
    end


    #
    # Creating components
    #

    # Creates a specified page component.
    def page( name, request_or_context, *args )
      context = request_or_context
      if Request === request_or_context then
        context = @context_class.new(request_or_context, self)
      end
      if Class === name then
        klass = name
      else
        klass = class_named(name)
      end
      klass.new(context, *args)
    end

    def class_named( name )
      Utilities.class_named_from(name, @name_spaces)
    end

  end

end

