require 'strscan'

module CGIKit

class Command

  attr_reader :controls, :session_id, :session, :form_values, :content
  attr_accessor :accept, :accept_charset, :accept_language, :auth_type, \
    :content_length, :content_type, :from, :gateway_interface, :path_info, \
    :path_translated, :query_string, :raw_cookie, :referer, :remote_addr, \
    :remote_host, :remote_ident, :remote_user, :request_method, :script_name, \
    :server_name, :server_port, :server_protocol, :server_software, \
    :uri, :user_agent

  alias url uri
  alias url= uri=

  @@default_key = :name

  def self.default_key
    @@default_key
  end

  def initialize( response = nil )
    @elements = []
    @form_values = Hash.new([])
    @accept = nil
    @accept_charset = nil
    @accept_language = nil
    @auth_type = nil
    @content_length = nil
    @content_type = nil
    @from = nil
    @gateway_interface = nil
    @path_info = nil
    @path_translated = nil
    @query_string = nil
    @raw_cookie = nil
    @referer = nil
    @remote_addr = nil
    @remote_host = nil
    @remote_ident = nil
    @remote_user = nil
    @request_method = nil
    @script_name = nil
    @server_name = nil
    @server_port = nil
    @server_protocol = nil
    @server_software = nil
    @uri = nil
    @user_agent = nil
    if response then
      parse(response)
      @content = response.content
    end
  end

  def parse( response )
    doc = REXML::Document.new(response.content)
    parse_element(doc)
  end

  def parse_element( element )
    if REXML::Text === element then return end
    attrs = attributes_from_element(element)
    distinct_element(element, attrs)
    element.each do |child|
      parse_element(child)
    end
  end

  def attributes_from_element( element )
    attrs = {}
    element.attributes.each do |key, value|
      attrs[key.intern] = value
    end
    attrs
  end

  def distinct_element( element, attrs )
    case element.name
    when 'a'
      @elements << HTMLLink.new(self, attrs, element.text)
    when 'img'
      @elements << HTMLImage.new(self, attrs)
    when 'frame'
      @elements << HTMLFrame.new(self, attrs)
    when 'form'
      @form = HTMLForm.new(self, attrs)
      @elements << @form
    when 'input'
      @elements << HTMLControl.control(self, attrs, @form)
    when 'textarea'
      textarea = HTMLTextArea.new(self, attrs, element.text)
      textarea.form = @form
      @form.controls << textarea
      @elements << textarea
    when 'select'
      @select = HTMLSelect.new(self, attrs)
      @select.form = @form
      @form.controls << @select
      @elements << @select
    when 'option'
      option = HTMLOption.new(self, attrs, element.texts[0].to_s)
      @select.options << option
      @elements << option
    end
  end

  def self.element( elements, value, klass, key )
    elements.each do |element|
      if (element[key] == value) and (klass === element) then
        return element
      end
    end
    nil
  end

  def self.element_at( elements, index, klass = HTMLElement )
    count = 0
    elements.each do |element|
      if klass === element then
        if index == count then
          return element
        end
        count += 1
      end
    end
    nil
  end

  def element( value, klass = HTMLElement, key = @@default_key )
    Command.element(@elements, value, klass, key)
  end

  def element_at( index, klass = HTMLElement )
    Command.element_at(@elements, index, klass)
  end

  def []( name )
    if Integer === name then
      element_at(name)
    else
      element(name)
    end
  end

  def image( name );       element(name, HTMLImage) end
  def image_at( index );   element_at(index, HTMLImage) end
  def control( name );     element(name, HTMLControl) end
  def control_at( index ); element_at(index, HTMLControl) end
  def link( name );        element(name, HTMLLink) end
  def link_at( index );    element_at(index, HTMLLink) end
  def form( name );        element(name, HTMLForm) end
  def form_at( index );    element_at(index, HTMLForm) end
  def submit( name );      element(name, HTMLSubmit) end
  def submit_at( index );  element_at(index, HTMLSubmit) end

  def request
    Request.new(headers(), @form_values)
  end

  def headers
    headers = {}
    headers['HTTP_ACCEPT']          = @accept
    headers['HTTP_ACCEPT_CHARSET']  = @accept_charset
    headers['HTTP_ACCEPT_LANGUAGE'] = @accept_language
    headers['AUTH_TYPE']            = @auth_type
    headers['CONTENT_LENGTH']       = @content_length
    headers['CONTENT_TYPE']         = @content_type
    headers['HTTP_FROM']            = @from
    headers['GATEWAY_INTERFACE']    = @gateway_interface
    headers['PATH_INFO']            = @path_info
    headers['PATH_TRANSLATED']      = @path_translated
    headers['QUERY_STRING']         = @query_string
    headers['HTTP_COOKIE']          = @raw_cookie
    headers['HTTP_REFERER']         = @referer
    headers['REMOTE_ADDR']          = @remote_addr
    headers['HTTP_HOST']            = @remote_host
    headers['REMOTE_IDENT']         = @remote_ident
    headers['REMOTE_USER']          = @remote_user
    headers['REQUEST_METHOD']       = @request_method
    headers['SCRIPT_NAME']          = @script_name
    headers['SERVER_NAME']          = @server_name
    headers['SERVER_PORT']          = @server_port
    headers['SERVER_PROTOCOL']      = @server_protocol
    headers['SERVER_SOFTWARE']      = @server_software
    headers['REQUEST_URI']          = @uri
    headers['HTTP_USER_AGENT']      = @user_agent
    headers
  end


class HTMLElement

  attr_accessor :attributes, :string, :command, :content

  def self.path_info( url )
    paths = url.split('/')
    ok = false
    infos = []
    paths.each do |path|
      if /#$0/ === path then
        ok = true
      elsif ok then
        infos << path
      end
    end
    path_info = "/#{infos.join('/')}"
    path_info
  end

  def initialize( command, attributes = {}, content = nil )
    @attributes = attributes
    @command = command
    @content = content
  end

  def []( name )
    @attributes[name]
  end

  def []=( name, value )
    @attributes[name] = value
  end

end


class HTMLLink < HTMLElement
  def submit
    path, query = @attributes[:href].split('?')
    @command.path_info = HTMLElement.path_info(path)
    @command.query_string = query
  end
end


class HTMLImage < HTMLElement
  def web_server_resource_name
    @attributes[:src].split('/').last
  end

  def resource_name
    query =@attributes[:src].split('/').last.split('?').last
    params = Request.parse_query_string(query)
    params[ResourceRequestHandler::RESOURCE_KEY][0]
  end
end


class HTMLFrame < HTMLElement
  def command_from_src
    command = Command.new
    command.path_info = HTMLElement.path_info(@attributes[:src])
    command
  end
end


class HTMLForm < HTMLElement

  attr_accessor :controls

  def initialize( command, attributes = {}, content = nil )
    super
    @controls = []
  end

  def set_all_controls( submit )
    @command.path_info = HTMLElement.path_info(@attributes[:action])
    @controls.each do |control|
      if (HTMLSubmit === control) and (control != submit) then
        next
      elsif control.set_value? then
        unless @command.form_values.has_key?(control[:name]) then
          @command.form_values[control[:name]] = []
        end
        control.set_value_to_form_values(@command.form_values)
      end
    end
  end

  def control( value, klass = HTMLControl, key = Command.default_key )
    Command.element(@controls, value, klass, key)
  end

  def control_at( index, klass = HTMLControl )
    Command.element_at(@controls, index, klass)
  end

  def textfield( name );     control(name,     HTMLTextField) end
  def textfield_at( index ); control_at(index, HTMLTextField) end
  def hidden( name );        control(name,     HTMLHidden) end
  def hidden_at( index );    control_at(index, HTMLHidden) end
  def password( name );      control(name,     HTMLPassword) end
  def password_at( index );  control_at(index, HTMLPassword) end
  def radio( name );         control(name,     HTMLRadio) end
  def radio_at( index );     control_at(index, HTMLHTMLRadio) end
  def checkbox( name );      control(name,     HTMLCheckbox) end
  def checkbox_at( index );  control_at(index, HTMLCheckbox) end
  def submit( name );        control(name,     HTMLSubmit) end
  def submit_at( index );    control_at(index, HTMLSubmit) end
  def reset( name );         control(name,     HTMLReset) end
  def reset_at( index );     control_at(index, HTMLReset) end
  def textarea( name );      control(name,     HTMLTextArea) end
  def textarea_at( index );  control_at(index, HTMLTextArea) end

  def submit_form
    submit_at(0).submit
  end

end


class HTMLControl < HTMLElement
  attr_accessor :form

  def self.control( command, attrs, form )
    case attrs[:type]
    when 'radio' then klass = HTMLRadio
    when 'checkbox' then klass = HTMLCheckbox
    when 'submit' then klass = HTMLSubmit
    when 'reset' then klass = HTMLReset
    when 'text' then klass = HTMLTextField
    when 'hidden' then klass = HTMLHidden
    when 'password' then klass = HTMLPassword
    when 'file' then klass = HTMLUpload
    end

    element = klass.new(command, attrs)
    element.form = form
    form.controls << element
    element
  end

  def enabled?
    !@attributes.key?(:disabled)
  end

  def set_value?
    @attributes.key?(:value)
  end

  def set_value_to_form_values( form_values )
    form_values[@attributes[:name]] << @attributes[:value]
  end

end


class HTMLRadio < HTMLControl

  def checked?
    @attributes.key?(:checked)
  end

  def checked
    !@attributes[:value].nil?
  end

  def checked=( flag )
    if flag == true then
      @attributes[:value] = 'checked'
    else
      @attributes[:value] = nil
    end
  end

end


class HTMLCheckbox < HTMLRadio
end


class HTMLTextField < HTMLControl
end

class HTMLPassword < HTMLControl
end

class HTMLHidden < HTMLControl
end


class HTMLTextArea < HTMLControl
  attr_accessor :value

  def initialize( command, attributes = {}, content = nil )
    super
    @value = nil
  end

  def set_value?
    @value
  end

  def set_value_to_form_values( form_values )
    form_values[@attributes[:name]] << @value
  end
end


class HTMLSelect < HTMLControl
  attr_accessor :options

  def initialize( command, attributes = {}, content = nil )
    super
    @options = []
  end

  def set_value?
    @options.each do |option|
      if option.selected? then
        return true
      end
    end
    false
  end

  def set_value_to_form_values( form_values )
    @options.each do |option|
      if option.selected? then
        form_values[@attributes[:name]] << option[:value]
      end
    end
  end

  def multiple?
    @attributes.key?(:multiple)
  end
end


class HTMLOption < HTMLControl
  attr_accessor :selected

  def initialize( command, attributes = {}, content = nil )
    super
    @selected = false
  end

  def selected?
    @attributes.key?(:selected) or @selected
  end
end


class HTMLUpload < HTMLControl
end


class HTMLSubmit < HTMLControl
  attr_accessor :submitted

  def set_value?
    true
  end

  def set_value_to_form_values( form_values )
    form_values[@attributes[:name]] << (@attributes[:value] || 'Submit')
  end

  def submit
    @form.set_all_controls(self)
  end
end

class HTMLReset < HTMLControl
end

end # Command

end # CGIKit
