module CGIKit

# Cookie is a class for cookie.
# To send cookies to a browser needs to create cookie objects 
# and set them to a response object. Instead of creating cookie objects,
# you can also get cookie objects from a request object.
#
# Cookie objects have a pair of a cookie name and value.
# If you make the objects have multiple values for one name, 
# you must write code by yourself. 
# 
# == Controlling cookie objects
#
# === Creating cookies
# Give arguments of initialize() a name or a pair of name/value.
# The value of cookie is omittable.
#
#  cookie = Cookie.new( name, value )
#
# === Getting cookies from a request object
# Request has some methods for getting cookies.
# The methods are cookie(key), cookies, cookie_value(key), cookie_values(key).
# See also Request.
#
# === Setting cookies to a response object
# Response has methods for setting cookies. These methods are 
# defined in Message, the superclass of Response.
# Use add_cookie(cookie) and remove_cookie(cookie).
class Cookie
  # Name of the cookie.
  attr_accessor :name

  # Value of the cookie.
  attr_accessor :value

  # Restricts the cookie in the site.
  attr_accessor :path

  # Domain that can receive the cookie.
  attr_accessor :domain

  # Expiry date. You set Time object to the cookie object.
  # The value is formatted when the cookie is returned.
  attr_accessor :expires

  # Decides whether the cookie is encrypted or not.
  attr_accessor :secure

  class << self
    # Parse raw cookie string and return an array of cookies.
    def parse_raw_cookie( raw_cookie )
      cookies = []
      return cookies unless raw_cookie

      raw_cookie.split('; ').each do |pairs|
        name, value = pairs.split('=',2)
        name  = Utilities.unescape_url name
        value = Utilities.unescape_url value

        cookies << Cookie.new( name, value )
      end

      cookies
    end
  end

  def initialize( name, value = nil, domain = nil, path = nil, secure = false )
    @name   = name
    @value  = value
    @domain = domain
    @path   = path
    @secure = secure
  end

  def to_s
    buf = "#@name="
    if @value   then buf << Utilities.escape_url(@value.to_s) end
    if @domain  then buf << "; domain=#@domain" end
    if @path    then buf << "; path=#@path" end
    if @expires then buf << "; expires=#{Utilities.date(@expires)}" end
    if @secure == true then buf << '; secure' end
    buf
  end
end

end
