module CGIKit

  class DisplayGroup

    attr_accessor :data_source, :objects, :default_values, :data_class, \
    :query_max, :query_min, :query_match, :query_operator, \
    :query_mode, :qualifier, :sort_keys, :sorter
    attr_reader :displays, :selections, :page_index, :per_page, \
    :selection_indexes, :first_display_index, :last_display_index

    def initialize( data_source = nil )
      @data_source = data_source
      @objects = []
      @selections = []
      @selection_indexes = []
      @displays = []
      @first_display_index = 0
      @last_display_index = 0
      @per_page = 10
      @page_index = 1
      @default_values = {}
      @data_class = Hash
      @query_max = {}
      @query_min = {}
      @query_match = {}
      @query_operator = {}
      @query_mode = false
      @sort_keys = []
    end


    #
    # accessing
    #

    def selection
      @selections[0]
    end

    def selection=( object )
      if index = @objects.index(object) then
        @selection_indexes << index
        @selections.clear
        @selections << object
      end
    end

    def selections=( objects )
      @selections.clear
      objects.each do |object|
        if index = @objects.index(object) then
          @selection_indexes << index
          @selections << object
        end
      end
    end

    def selection_indexes=( indexes )
      @selection_indexes.replace(indexes)
      objects = []
      indexes.each do |index|
        objects << @objects[index]
      end
      self.selections = objects
    end

    def page_index=( index )
      index = 1 if index <= 0
      @page_index = index
      update
    end

    def per_page=( num )
      @per_page = num
      update
    end

    def first_display_index=( index )
      @first_display_index = index
      if @first_display_index <= 0 then
        @first_display_index = 1
      end
      if @first_display_index > @objects.size then
        @first_display_index = @objects.size
      end
    end

    def last_display_index=( index )
      @last_display_index = index
      if @last_display_index < 0 then
        @last_display_index = 1
      end
      if @last_display_index > @objects.size then
        @last_display_index = @objects.size
      end
    end

    def page_count
      count = @objects.size / @per_page
      if count <= 0 then
        count = 1
      elsif (@objects.size % @per_page) > 0 then
        count += 1
      end
      count
    end

    def qualifier_from_query_values
      proc do |obj|
        break false if obj.nil?
        target = nil
        hash = Hash === obj

        flag = false
        @query_max.each do |key, value|
          if hash then
            target = obj[key]
          else
            target = obj.__send__(key)
          end
          if target.nil? or !(target <= value) then
            flag = true
            break
          end
        end
        break false if flag

        flag = false
        @query_min.each do |key, value|
          if hash then
            target = obj[key]
          else
            target = obj.__send__(key)
          end
          if target.nil? or !(target >= value) then
            flag = true
            break
          end
        end
        break false if flag

        flag = false
        @query_match.each do |key, value|
          if op = @query_operator[key] then
            if hash then
              target = obj[key]
            else
              target = obj.__send__(key)
            end
            begin
              unless target.__send__(op, value) then
                flag = true
                break
              end
            rescue
              flag = true
              break
            end
          end
        end
        break false if flag

        true
      end
    end

    def add_sort_key( key, method )
      @sort_keys << SortKey.new(key, method)
    end

    def remove_sort_key( key )
      @sort_keys.delete_if do |sort|
        sort.key == key
      end
    end

    def sorter_from_sort_keys
      proc do |a, b|
        if a.nil? and b.nil? then
          break 0
        elsif a.nil? then
          break -1
        elsif b.nil? then
          break 1
        end

        hash1 = Hash === a
        hash2 = Hash === b

        @sort_keys.each do |sort|
          if hash1 then
            a = a[sort.key]
          else
            a = a.__send__(sort.key)
          end
          if hash2 then
            b = b[sort.key]
          else
            b = b.__send__(sort.key)
          end

          if a.nil? and b.nil? then
            break 0
          elsif a.nil? then
            break -1
          elsif b.nil? then
            break 1
          else
            if a.__send__(sort.method, b) then
              break 1
            else
              break -1
            end
          end
        end
      end
    end


    #
    # testing
    #

    def multiple_pages?
      page_count > 1
    end

    def first_page_index?
      @page_index <= 1
    end

    def last_page_index?
      @page_index >= page_count()
    end


    #
    # displaying
    #

    def update
      _filter
      _sort
      display_current
      @displays.clear
      @displays.replace(objects_to_display)
    end

    private

    def _filter
      if @qualifier then
        @objects.delete_if do |obj|
          !@qualifier.call(obj)
        end
      end
    end

    def _sort
      sorter = (@sort_keys.empty? ? @sorter : sorter_from_sort_keys)
      if sorter then
        @objects.sort! do |a, b|
          sorter.call(a, b)
        end
      end
    end

    public

    def display_current
      if @page_index > (count = page_count) then
        @page_index = count
      end
      self.first_display_index = (@page_index - 1 ) * @per_page + 1
      self.last_display_index = @first_display_index + @per_page - 1
    end

    def display_next
      return if (@page_index + 1) > page_count()
      @page_index += 1
      @first_display_index += @per_page
      @last_display_index += @per_page
      @displays.replace(objects_to_display)
      clear_selection
    end

    def display_pre
      return if (@page_index - 1) <= 0
      @page_index -= 1
      self.first_display_index -= @per_page
      self.last_display_index -= @per_page
      @displays.replace(objects_to_display)
      clear_selection
    end

    def objects_to_display
      @objects[@first_display_index-1..@last_display_index-1] || []
    end

    def qualify
      @qualifier = qualifier_from_query_values
      update
      @query_mode = false
    end

    def qualify_data_source
      @data_source.qualifier = qualifier_from_query_values if @data_source
      fetch
      update
      @query_mode = false
    end

    def clear_queries
      @query_max.clear
      @query_min.clear
      @query_match.clear
      @query_operator.clear
    end


    #
    # operating
    #

    def fetch
      @objects.replace(@data_source.fetch) if @data_source
    end

    def insert( index = nil, object = nil )
      object ||= create
      index ||= @first_display_index - 1
      if sindex = @selection_indexes[0] then
        index = sindex + 1
      end
      if @data_source then
        @data_source.insert(index, object)
      else
        @objects.insert(index, object)
      end
      update
    end

    def create
      object = nil
      if @data_source then
        object = @data_source.create || @data_class.new
      else
        object = @data_class.new
      end
      @default_values.each do |key, value|
        writer = "#{key}=".untaint
        if object.respond_to?(writer) then
          object.__send__(writer, value)
        elsif object.respond_to?(:[]=) then
          object[key] = value
        end
      end
      object
    end

    def delete_at( index )
      object = @objects[index]
      if @data_source then
        @data_source.delete(object)
      else
        @objects.delete(object)
      end
    end

    def delete
      @selection_indexes.each do |index|
        delete_at(index)
      end
    end

    def clear_selection
      @selection_indexes.clear
      @selections.clear
    end


    #
    # selecting
    #

    def select_next
      if @objects.empty? then
        # do nothing
      elsif @selections.empty? then
        obj = @objects[0]
        self.selection = obj
      elsif @selections.first == @displays.last then
        obj = @displays.first
        self.selection = obj
      else
        index = @objects.index(@selections.first) + 1
        obj = @objects[index]
        self.selection = obj
      end
    end

    def select_pre
      if @objects.empty? then
        # do nothing
      elsif @selections.empty? then
        obj = @objects[0]
        self.selection = obj
      elsif @selections.first == @displays.last then
        obj = @displays.last
        self.selection = obj
      else
        index = @objects.index(@selections.first) - 1
        obj = @objects[index]
        self.selection = obj
      end
    end

  end


  class DataSource

    def fetch; end
    def insert(object); end
    def create; end
    def delete(object); end

  end


  class SortKey

    attr_accessor :key, :method

    def initialize( key, method )
      @key = key
      @method = method
    end

  end

end

