module CGIKit

  class RequestHandler
    include Logging

    attr_reader :application

    class << self

      def register( app = nil )
        app ||= Application
        app.register_request_handler(request_handler_key(), self)
      end

      def request_handler_key; end

    end

    def initialize( application )
      @application = application
    end

    def request_handler_key
      @application.request_handler_key(self)
    end


    #
    # parsing session IDs and context IDs from URLs
    #

    def session_id( request )
      sid_url = session_id_from_url(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_url then
        sid_url
      else
        nil
      end
    end

    def session_id_from_cookie( request )
      sid = nil
      cookie = request.cookie(@application.session_key)
      if cookie then
        sid = cookie.value
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def session_id_from_url( request )
      sid = nil
      if path = request.request_handler_path then
        sid = _id_by_separating(path, false)
      end
      sid
    end

    private

    def _id_by_separating( path, context = true )
      path = path.reverse.chop.reverse
      separated = path.split('/')
      separated.delete_at(0)
      id = nil
      case separated.size
      when 1
        if context then
          if separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        else
          if !separated.last.include?(Context::SEPARATOR) then
            id = separated.last
          end
        end
      when 2
        if context then
          id = separated.last
        else
          id = separated.first
        end
      end
      if id and !context and !Session.session_id?(id) then
        id = nil
      end
      id
    end

    public

    def context_id( request )
      id = nil
      if path = request.request_handler_path then
        id = _id_by_separating(path, true)
      end
      id
    end


    #
    # request-response loop
    #

    # Abstract method. Returns a response object.
    def handle_request( request )
      request.session_id = session_id(request)
      request.context_id = context_id(request)
      info("session ID = #{request.session_id}, " <<
             "context ID = #{request.context_id}", true)
    end

    def transaction( element, context, &block )
      element.begin_context(context)
      block.call
      element.end_context(context)
    end

    def take_values_from_request( element, request, context )
      transaction(element, context) do
        element.delegate_take_values_from_request(request, context)
      end
    end

    def invoke_action( element, request, context )
      result = nil
      transaction(element, context) do
        result = element.delegate_invoke_action(request, context)
      end

      if (Component === result) or (Response === result) then
        result
      else
        nil
      end
    end

    def append_to_response( element, response, context )
      transaction(element, context) do
        element.delegate_append_to_response(response, context)
      end
    end


    #
    # generating URLs
    #

    def url( context, path, query, is_secure, port = nil, sid = true ); end

    def application_url( request, is_secure = false, port = nil )
      protocol = nil
      if is_secure == true then
        protocol = 'https://'
      else
        protocol = nil
      end

      domain = request.server_name || 'localhost'

      if port.nil? and request.server_port then
        port = request.server_port.to_i
      end
      if port == 80 then
        port = nil
      end

      script = (request.script_name || @application.path).dup
      script.sub!(/\A\//, '')

      if port then
        protocol ||= 'http://'
        path = "#{protocol}#{domain}:#{port}/#{script}"
        path.gsub!(/\/\Z/, '')
      elsif protocol then
        path = "#{protocol}#{domain}/#{script}"
      else
        path = "/#{script}"
      end
      path
    end

    def query_string( hash = {} )
      str = ''
      keys = hash.keys.sort do |a, b|
        a.to_s <=> b.to_s
      end
      keys.each do |key|
        value = hash[key]
        if Array === value then
          value.each do |item|
            str << query_association(key, item)
          end
        else
          str << query_association(key, value)
        end
      end
      str.chop!
      str
    end

    def query_association( key, value )
      unless value.nil? then
        "#{key}=#{Utilities.escape_url(value.to_s)};"
      else
        "#{key};"
      end
    end
  end


  class ComponentRequestHandler < RequestHandler
    def handle_request( request )
      super
      context = @application.create_context(request)
      context.request_handler_key = @application.component_request_handler_key

      @application.synchronize do
        transaction(context.component, context) do
          @application.take_values_from_request(request, context)
        end

        result = nil
        context.delete_all
        transaction(context.component, context) do
          result = @application.invoke_action(request, context)
        end

        if Response === result then
          context.response = result
        else
          if (Component === result) and (context.component != result) then
            result.awake_from_restoration(context)
            context.component = result
          else
            result = context.component.root
          end
          context.delete_all
          transaction(result, context) do
            @application.append_to_response(context.response, context)
          end
          context.response.component = result
        end
      end

      @application.save_session(context)
      context.response
    end

    def url( context, path = nil, query = {},
             is_secure = false, port = nil, sid = true )
      str = application_url(context.request, is_secure, port)
      str << "/#{@application.component_request_handler_key}"
      if sid and context.has_session? then
        str << "/#{context.session.session_id}"
      end
      str << "/#{context.context_id}"
      str << "/#{path}" if path
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

  end


  class ActionRequestHandler < RequestHandler

    class ActionResultError < StandardError #:nodoc:
    end

    def action_and_action_name( request )
      klass = request.action_class || default_action_class()
      action = klass.new(@application, request)
      name = request.action_name || action.default_action_name
      unless action.action?(name) then
        klass = default_action_class()
        action = klass.new(@application, request)
        name = action.default_action_name
      end
      [action, name]
    end

    def action_url( context, action_class, action_name, query, sid = true )
      action_class ||= default_action_class()
      path = action_path(action_class, action_name)
      url(context, path, query, false, nil, sid)
    end

    def action_path( action_class, action_name )
      if action_name == action_class.default_action_name then
        action_name = ''
      end
      if default_action_class() == action_class then
        action_name
      elsif action_name.empty?
        action_class
      else
        "#{action_class}/#{action_name}"
      end
    end

    def url( context, path = nil, query = {}, is_secure = false,
             port = 80, sid = true )
      str = application_url(context.request, is_secure, port)
      str << "/#{request_handler_key()}"
      str << "/#{path}" if path
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def default_action_class; end

  end


  class DirectActionRequestHandler < ActionRequestHandler

    def session_id( request )
      sid_query = session_id_from_query(request)
      sid_cookie = session_id_from_cookie(request)
      if @application.store_in_cookie and sid_cookie then
        sid_cookie
      elsif sid_query then
        sid_query
      else
        nil
      end
    end

    def session_id_from_query( request )
      if sid = request[@application.direct_action_session_key] then
        unless Session.session_id?(sid) then
          sid = nil
        end
      end
      sid
    end

    def handle_request( request )
      super
      direct_action, action_name = action_and_action_name(request)

      result = response = nil
      @application.synchronize do
        result = direct_action.perform_action(action_name)
        unless result.respond_to?(:generate_response) then
          raise ActionResultError, \
          "Direct action must return an object has generate_response()" +
            " and not nil. - #{direct_action.class}##{action_name}"
        end
        response = result.generate_response
      end

      if Component === result then
        response.component = result
        if result.context.has_session? then
          @application.save_session(result.context)
        end
      end
      response
    end

    def url( context, path = nil, query = {}, is_secure = false,
             port = 80, sid = true )
      str = application_url(context.request, is_secure, port)
      str << "/#{request_handler_key()}"
      str << "/#{path}" if path
      if sid and context.has_session? then
        query[@application.direct_action_session_key] = context.session.session_id
      end
      qstr = query_string(query)
      unless qstr.empty? then
        str << "?#{qstr}"
      end
      str
    end

    def default_action_class
      @application.direct_action_class
    end

  end


  class ResourceRequestHandler < RequestHandler

    RESOURCE_KEY = 'data'

    def handle_request( request )
      super
      response = Response.new
      key = request.form_value(RESOURCE_KEY)
      debug("request resource key: #{key}")
      if data = @application.resource_manager.bytedata(key) then
        response.headers['Content-Type'] = data.content_type
        response.content = data.to_s
        @application.resource_manager.remove_data(key) if data.tmp?
      else
        response.status = 404
      end
      response
    end

    def resource_url( name, request )
      str = application_url(request)
      str << "/#{@application.resource_request_handler_key}"
      str << "?#{RESOURCE_KEY}=#{name}"
      str
    end

  end

end
