module CGIKit

  class Package

    BIN_PATH                     = 'bin'
    LIB_PATH                     = 'lib'
    COMPONENT_PATH               = 'components'
    RESOURCE_PATH                = 'resources'
    WEB_SERVER_RESOURCE_PATH     = 'www'
    MODEL_PATH                   = 'models'
    LOCALE_PATH                  = 'locale'
    PACKAGE_PATH                 = 'packages'
    MESSAGE_PATH                 = 'messages'
    AUTOLOAD_COMPONENT_LOAD_TYPE = 'autoload'
    REQUIRE_COMPONENT_LOAD_TYPE  = 'require'
    CONFIG_FILE                  = 'config.rb'
    PRE_LOAD_FILE                = 'pre-load.rb'
    POST_LOAD_FILE               = 'post-load.rb'
    DEFAULT_MESSAGE_FILE         = 'default.mo'

    attr_reader :path, :name
    attr_accessor :lib_path, :component_path, :resource_path, \
    :web_server_resource_path, :model_path, :message_path, \
    :component_load_type, :package_class_name, :message_encoding

    class << self

      def all_resource_paths
        [BIN_PATH, LIB_PATH, COMPONENT_PATH, RESOURCE_PATH, 
          WEB_SERVER_RESOURCE_PATH, MODEL_PATH, LOCALE_PATH, 
          PACKAGE_PATH, MESSAGE_PATH]
      end

    end

    def initialize( path = nil, options = {} )
      @bin_path       = options[:bin_path] || BIN_PATH
      @lib_path       = options[:lib_path] || LIB_PATH
      @component_path = options[:component_path] || COMPONENT_PATH
      @resource_path  = options[:resource_path] || RESOURCE_PATH
      @web_server_resource_path = options[:web_server_resource_path] || \
        WEB_SERVER_RESOURCE_PATH
      @model_path     = options[:model_path] || MODEL_PATH
      @locale_path    = options[:locale_path] || LOCALE_PATH
      @package_path   = options[:package_path] || PACKAGE_PATH
      @message_path   = options[:message_path] || MESSAGE_PATH

      @component_load_type = options[:component_load_type] || \
        REQUIRE_COMPONENT_LOAD_TYPE
      @package_class_name  = options[:package_class_name]
      @config_file         = options[:config_file] || CONFIG_FILE
      @pre_load_file       = options[:pre_lod_file] || PRE_LOAD_FILE
      @post_load_file      = options[:post_load_file] || POST_LOAD_FILE
      @message_encoding    = options[:message_encoding]
      @messages = {}
      if path then
        self.path = path
        load
      end
    end


    #
    # accessing
    #

    def path=( path )
      @path = path
      @name = File.basename(path)
    end

    def all_resource_paths
      [@bin_path, @lib_path, @component_path, @resource_path,
        @web_server_resource_path, @model_path, @locale_path,
        @package_path, @message_path]
    end


    #
    # loading
    #

    def load
      load_config
      exec_load_hook(path)
      load_lib if @lib_path
      load_components if @component_path
      load_resources if @resource_path
      load_web_server_resources if @web_server_resource_path
    end

    def load_config
      path = File.join(@path, CONFIG_FILE)
      return unless FileTest.exist?(path)
      config = nil
      str = nil
      open(path) { |f| str = f.read }
      Thread.start(str) do
        $SAFE = 4
        config = eval(str)
      end.join
      config.each do |key, value|
        self.__send__("#{key}=", value)
      end
    end

    def load_lib
      lib = load_path(@lib_path)
      Dir.glob(lib) do |path|
        $LOAD_PATH << path
      end
    end

    def load_components
      comp = load_path(@component_path, '**', '*.rb')
      Dir.glob(comp) do |path|
        next unless /\A[A-Z]/ === File.basename(path)
        name = File.basename(path, '.*')
        if autoload_component_load_type? then
          mod = package_module()
          mod.autoload(name, path)
        elsif require_component_load_type? then
          require(path)
        end
      end
    end

    def load_resources
      load_path(@resource_path)
    end

    def load_web_server_resources
      load_path(@web_server_resource_path)
    end

    def load_messages( name, lang )
      @messages[lang] ||= {}
      path = locale_path(lang, @message_path, name)
      return unless FileTest.exist?(path)
      catalog = MessageCatalog.new(path, @message_encoding)
      @messages[lang][name] = catalog
    end


    #
    # testing
    #

    def autoload_component_load_type?
      @component_load_type == AUTOLOAD_COMPONENT_LOAD_TYPE
    end

    def require_component_load_type?
      @component_load_type == REQUIRE_COMPONENT_LOAD_TYPE
    end


    #
    # executing
    #

    def exec_hook( path, pre, post, &block )
      pre = File.join(path, pre)
      post = File.join(path, post)
      if FileTest.exist?(pre) then
        require(pre) 
      end
      block.call if block_given?
      if FileTest.exist?(post) then
        require(post) 
      end
    end

    def exec_load_hook( path, &block )
      exec_hook(path, PRE_LOAD_FILE, POST_LOAD_FILE, &block)
    end


    #
    # path accessing
    #

    def package_module
      modname = (@package_class_name || @name).to_s.intern
      unless Object.const_defined?(modname) then
        obj = Module.new
        Object.const_set(modname, obj)
      end
      Object.const_get(modname)
    end

    def load_path( *paths )
      File.join(@path, '**', *paths)
    end

    def locale_path( lang, *paths )
      File.join(@path, @locale_path, lang, *paths)
    end

    def path_for_component( name, languages = [] )
      unless dir = name.include?('.') then
        name = "#{name}.*"
      end
      file = File.join(@component_path, '**', name)
      path = file_path(file, languages)
      unless dir then
        path = File.dirname(path)
      end
      path
    end

    def file_path( file, languages = [] )
      path = nil
      languages.each do |lang|
        pattern = locale_path(lang, file)
        paths = Dir.glob(pattern)
        unless paths.empty? then
          path = paths[0]
          break
        end
      end
      unless path then
        path = Dir.glob(File.join(@path, file))[0]
      end
      path
    end

    def path_for_resource( name, languages = [] )
      file = File.join(@resource_path, name)
      file_path(file, languages)
    end

    # DOCROOT/cgikit/packages/(packages)
    def path_for_web_server_resource( docroot, name, languages = [] )
      www = File.join(docroot, Application::CGIKIT_PATH,
                      Application::PACKAGE_PATH, @name)
      paths = []
      languages.each do |lang|
        paths << File.join(www, @locale_path, lang, @web_server_resource_path, name)
      end
      paths << File.join(www, @web_server_resource_path, name)
      paths.each do |path|
        if FileTest.exist?(path) then
          return path
        end
      end
      file_path(File.join(@web_server_resource_path, name), languages)
    end

    def paths_for_model
      paths = []
      Dir.foreach(File.join(@path, @model_path)) do |path|
        unless /\A\./ === path then
          paths << File.join(@path, @model_path, path)
        end
      end
      paths
    end


    #
    # message accessing
    #

    def message( key, name, languages )
      if catalog = catalog(name, languages) then
        catalog.gettext(key)
      else
        key
      end
    end

    def nmessage( key, plural_key, n, name, languages )
      if catalog = catalog(name, languages) then
        catalog.ngettext(key, plural_key, n)
      else
        plural_key
      end
    end

    def catalog( name, languages )
      name ||= DEFAULT_MESSAGE_FILE
      unless Array === languages then
        languages = [languages]
      end
      languages.each do |lang|
        if !@messages[lang] or !@messages[lang][name] then
          load_messages(name, lang)
        end
        if catalog = @messages[lang][name] then
          return catalog
        end
      end
      nil
    end

  end


  class MessageCatalog

    def initialize( path, encoding )
      require 'cgikit/lang/mo.rb' unless defined?(MOFile)
      @path = path
      @mo = MOFile.open(path, encoding)
    end

    def gettext(msgid)
      if @mo
	if @mo[msgid]
	  result = @mo[msgid].size > 0 ? @mo[msgid] : msgid
	else
	  result = msgid
	end
      else
        result = msgid
      end
      result
    end

    def ngettext(msgid, msgid_plural, n)
      msg = gettext(msgid + "\000" + msgid_plural)
      if msg.include?("\000")
        ary = msg.split("\000")
        if @mo
          plural = eval(@mo.plural)
          if plural.kind_of?(Numeric)
            msg = ary[plural]
          else
            msg = plural ? ary[1] : ary[0]
          end
        else
          msg = n == 1 ? ary[0] : ary[1]
        end
      end
      msg
    end

  end

end

