module CGIKit

# A response object that is sent to a browser by a Adapter object.
#
# == Getting a response object
# You can get a response object by Application#response or Component#response.
#
# == Setting headers for a response object
# To send HTTP response headers, append a pair of key and value to headers.
# For example,
#
#  application.response.headers['Content-Type'] = 'text/html'
#
class Response < Message
  # Status code in HTTP. Default status is 200 ( OK ).
  attr_accessor :status

  attr_accessor :component

  STATUS = { 
    100 => 'Continue',
    101 => 'Switching Protocols',
    200 => 'OK',
    201 => 'Created',
    202 => 'Accepted',
    203 => 'Non-Authoritative Information',
    204 => 'No Content',
    205 => 'Reset Content',
    206 => 'Partial Content',
    300 => 'Multiple Choices',
    301 => 'Moved Permanently',
    302 => 'Found',
    303 => 'See Other',
    304 => 'Not Modified',
    305 => 'Use Proxy',
    307 => 'Temporary Redirect',
    400 => 'Bad Request',
    401 => 'Unauthorized',
    402 => 'Payment Required',
    403 => 'Forbidden',
    404 => 'Not Found',
    405 => 'Method Not Allowed',
    406 => 'Not Acceptable',
    407 => 'Proxy Authentication Required',
    408 => 'Request Timeout',
    409 => 'Conflict',
    410 => 'Gone',
    411 => 'Length Required',
    412 => 'Precondition Failed',
    413 => 'Request Entity Too Large',
    414 => 'Request-URI Too Long',
    415 => 'Unsupported Media Type',
    416 => 'Requested Range Not Satisfiable',
    417 => 'Expectation Failed',
    500 => 'Internal Server Error',
    501 => 'Not Implemented',
    502 => 'Bad Gateway',
    503 => 'Service Unavailable',
    504 => 'Gateway Timeout',
    505 => 'HTTP Version Not Supported'
  }

  class << self

    def new_with_response( response )
      res = new
      res.headers = response.headers
      res.status = response.status
      res.http_version = response.http_version
      res.encoding = response.encoding
      res.cookies = response.cookies
      res
    end

  end

  def initialize( headers = nil )
    super
    self.status = 200
    @headers['Content-Type'] = 'text/html'
    @headers['Cache-Control'] = \
      'private, no-cache, no-store, must-revalidate, max-age=0'
    @headers['Expires'] = Utilities.date(Time.now - 60)
    @headers['Date'] = Utilities.date(Time.now)
    @headers['Pragma'] = 'no-cache'
  end

  def header
    response = cookie_header()
    @headers.each do |key, value|
      if key != 'Content-Type' then
        response << header_line(key, value)
      end
    end
    response << content_type_header()
    response << EOL
    response
  end

  def to_s
    response = ''
    response << (self.content || '') unless redirect?
    response
  end

  def header_line( key, value )
    "#{key}: #{value}#{EOL}"
  end

  def content_type_header
    header = "Content-Type: #{self.headers['Content-Type']}"
    header << "; charset=#{self.encoding}" if self.encoding
    header << EOL
  end

  def cookie_header
    return '' if @cookies.empty?
    header = ''
    @cookies.each do |cookie|
      header << "Set-Cookie: #{cookie.to_s}"
      header << EOL
    end
    header
  end

  def status=( value )
    @status = value
    @headers['Status'] = "#@status #{STATUS[@status]}"
  end

  # Sends a temporary redirect response to the client using the specified URL.
  def set_redirect( url )
    @status = 302
    @headers['Location'] = url
  end
  alias redirect= set_redirect

  # Returns true if the response is setted redirect.
  def redirect?
    (@status == 302) or (@status == 307)
  end

  # Returns self. This method is invoked to display results in direct action.
  def generate_response
    self
  end
end

end
