module CGIKit

class Template

  attr_accessor :template_store, :component, :template_node, :declaration_store, \
    :template_string, :declaration_hash, :template_mtime, :declaration_mtime

  def initialize
    @template_mtime = Time.now
    @declaration_mtime = Time.now
  end

  def terminate?
    @template_store.terminate?(@component)
  end

  def template_string=( string )
    @template_string = string
    @template_mtime = Time.now
  end

  def declaration_store=( store )
    @declaration_store = store
    @declaration_mtime = Time.now
  end

  def marshal_dump
    not_includes = ['@component', '@template_store']
    dump = {}
    instance_variables.each do |var|
      if not_includes.include?(var) == false then
        dump[var] = instance_variable_get(var)
      end
    end
    dump
  end

  def marshal_load( object )
    object.each do |key, value|
      instance_variable_set(key, value)
    end
  end

  def cache_copy
    copy = dup
    copy.template_node = @template_node.cache_copy
    copy
  end
end


class TemplateStore
  include Logging

  def initialize( application )
    @application = application
  end

  def cache?
    @application.cache_template
  end

  def checkin( template )
    debug(template.component.class, true)
    save(template)
  end

  def checkout( component )
    debug(component.class, true)
    template = restore(component)
    if template.nil? or (cache? and terminate?(template, component)) then
      template = create_template(component)
      isnew = true
      checkin(template) if cache?
      debug("create template #{component.class}")
    end
    template.component = component
    template.template_store = self
    template.template_node.component = component
    template
  end

  def terminate?( template, component )
    terminate_template?(template, component) or \
      terminate_declaration?(template, component)
  end

  def terminate_template?( template, component )
    (component.will_parse_template and \
     (component.will_parse_template != template.template_string)) or \
    (template.template_mtime and \
     (File.mtime(component.template_file) > template.template_mtime))
  end

  def terminate_declaration?( template, component )
    ((component.will_parse_declarations) and \
     (component.will_parse_declarations != template.declaration_hash)) or \
    (template.declaration_mtime and \
     (File.mtime(component.declaration_file) > template.declaration_mtime))
  end

  def create_template( component )
    template = Template.new
    template.component = component
    parser = load_template(component)
    template.template_node = parser.node
    template.template_string = parser.html_string
    template.declaration_store = load_declarations(component, parser.declarations)
    template
  end

  def load_template( component )
    klass = @application.htmlparser_class
    if string = component.will_parse_template then
      parser = klass.new
      parser.parse(string)
    elsif path = component.template_file then
      parser = klass.new(path)
    else
      raise "can't find a template file - #{component.class}"
    end
    parser
  end

  def load_declarations( component, source )
    store = nil
    if hash = component.will_parse_declarations then
      DeclarationStore.merge_source(source, hash)
      store = DeclarationStore.new_from_hash(hash)
    elsif path = component.declaration_file then
      begin
        store = DeclarationStore.new_with_file(path, source)
      rescue Exception => e
        raise "can't load declaration file (#{path}): #{e.message}"
      end
    else
      store = DeclarationStore.new_from_hash(source)
    end
    store
  end

  def save( template )
    raise StandardError, 'Not implemented'
  end
    
  def restore( component )
    raise StandardError, 'Not implemented'
  end
    
  def remove( component )
    raise StandardError, 'Not implemented'
  end
    
end


class FileTemplateStore < TemplateStore

    TMPDIR = 'template_cache'

  def initialize( application )
    super
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end
  
  def save( template )
    unless FileTest.directory?(@tmpdir) then
      require 'fileutils'
      FileUtils.makedirs(@tmpdir)
    end
    
    filename = cache_file_name(template.component)
    FileLock.exclusive_lock(filename) do |file|
      Marshal.dump(template, file)
    end
  end

  def restore( component )
    template = nil
    if cache? and exist?(component) then
      FileLock.shared_lock(cache_file_name(component)) do |file|
        template = Marshal.load(file)
      end
    end    
    template
  end
  
  def cache_file_name( component )
    name = component.template_file.gsub('::', '__')
    name.tr!('./', '_')
    File.join(@tmpdir, name)
  end

  def exist?( component )
    FileTest.exist?(cache_file_name(component))
  end
  
end


class MemoryTemplateStore < TemplateStore

  def initialize( application )
    super
    @caches = {}
  end

  def save( template )
    @caches[template.component.template_file] = template.cache_copy
  end

  def restore( component )
    if template = @caches[component.template_file] then
      template = template.cache_copy
    end
    template
  end

  def exist?( component )
    @caches.key?(component.class)
  end

end

end
