package charactermanaj.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.logging.ConsoleHandler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;


public final class ApplicationLogger {

	private static final String LOGGING_PROPERTIES = "logging.properties";
	
	private static final Logger logger;
	
	static {
		// ログマネージャ.
		// 初期時にJRE等にある初期設定がなされている.
		LogManager logManager = LogManager.getLogManager();

		// 設定を一旦クリアする.
		// ルートロガーがInfoになり、ハンドラはすべてリセット。ルートロガー以外の設定は空にされる.
		logManager.reset();

		Exception configurationError = null;
		try {
			// ユーザーごとのアプリケーション設定ディレクトリ上の設定ファイルを取得する.
			File appDataDir = ConfigurationDirUtilities.getUserDataDir();
			File logConfig = new File(appDataDir, LOGGING_PROPERTIES);

			InputStream is = null;
			if (logConfig.exists()) {
				// ユーザー指定のロギングプロパティがある場合
				is = new FileInputStream(logConfig);
			} else {
				// リソース上のロギングプロパティ
				is = ApplicationLogger.class.getResourceAsStream("/" + LOGGING_PROPERTIES);
			}
			if (is != null) {
				try {
					// ログを再設定する.
					logManager.readConfiguration(is);

				} finally {
					is.close();
				}
			}
		} catch (Exception ex) {
			// 初期化に失敗した場合はログに記録するために例外を保存するが、
			// 処理は継続する.
			configurationError = ex;
		}
		
		// ロガーを取得
		logger = Logger.getAnonymousLogger();
		
		// 初期化時に失敗した場合、デフォルトのコンソールハンドラを設定し、ログに出力する.
		if (configurationError != null) {
			logger.addHandler(new ConsoleHandler());
			logger.addHandler(new ApplicationLogHandler());
			logger.log(Level.WARNING, "LogConfigurationFailed", configurationError);
		}
		
		// 初期化時のログ
		logger.info("open logger.");
		logger.info("application configuration: baseDir="
				+ ConfigurationDirUtilities.getApplicationBaseDir() + "  appData="
				+ ConfigurationDirUtilities.getUserDataDir());
	}
	
	private ApplicationLogger() {
		super();
	}

	public static Logger getLogger() {
		return logger;
	}
	
}
