/*
 * chasen.c - ChaSen main program
 *
 * Copyright (C) 1996,1997 Nara Institute of Science and Technology
 *
 * Author: A.Kitauchi <akira-k@is.aist-nara.ac.jp>, Apr. 1997
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "chalib.h"

#ifndef CHASEN_PORT
#define CHASEN_PORT   31000  /* 㥻, äȶ줷(^^; */
#endif

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

static char chasen_host[MAXHOSTNAMELEN];
static unsigned short chasen_port = CHASEN_PORT;

#ifdef KOCHA
#define CHA_NAME       "KoCha"
#define CHA_PROG       "kocha"
#else
#define CHA_NAME       "ChaSen"
#define CHA_PROG       "chasen"
#endif

/*
 * opt_form_usage()
 */
static void opt_form_usage(fp)
    FILE *fp;
{
    static char *message[] = {
	"Conversion characters of -F option:\n",
	"  %m    surface form (inflected form)\n",
	"  %M    surface form (base form)\n",
	"  %y    pronounciation (inflected form)\n",
	"  %Y    pronounciation (base form)\n",
	"  %i    semantic information\n",
	"  %Ic   semantic information (if NIL, print character 'c'.)\n",
#ifdef VGRAM
	"  %Pc   part of speech separated by character 'c'\n",
	"  %Pnc  part of speech separated by character 'c'\n",
#endif
	"  %h    part of speech (code)\n",
	"  %H    part of speech (name)\n",
	"  %b    sub-part of speech (code)\n",
	"  %BB   sub-part of speech (name)(if not, print part of speech)\n",
	"  %BM   sub-part of speech (name)(if not, print part of speech)\n",
	"  %Bc   sub-part of speech (name)(if not, print character 'c')\n",
	"  %t    inflection type (code)\n",
	"  %Tc   inflection type (name)(if not, print character 'c')\n",
	"  %f    inflected form (code)\n",
	"  %Fc   inflected form (name)(if not,  print character 'c')\n",
	"  %c    cost value of the morpheme\n",
	"  %%    '%'\n",
	"  .     specify the field width\n",
	"  -     specify the field width\n",
	"  1-9   specify the field width\n",
	"  \\n    carriage return\n",
	"  \\t    tab\n",
	"  \\\\    back slash\n",
	"  \\'    single quotation mark\n",
	"  \\\"    double quotation mark\n",
	"\n",
	"Examples:\n",
	"  \"\%M \"         split words by space (wakachi-gaki)\n",
	"  \"\%y\"          Kana to Kanji conversion\n",
	"  \"\%r ()\"       print with ruby\n",
#ifdef VGRAM
	"  \"\%m\\t\%y\\t\%M\\t\%U(\%P-)\\t\%T \\t\%F \\n\"           same as -f option (default)\n",
	"  \"\%m\\t\%U(\%y)\\t\%M\\t\%P- \%h \%T* \%t \%F* \%f\\n\"    same as -e option\n",
#else
	"  \"\%-11m \%-11y \%-11M \%-14U(\%BB) \%-14T  \%F \\n\"    same as -f option (default)\n",
	"  \"\%m \%U(\%y) \%M \%H \%h \%B* \%b \%T* \%t \%F* \%f\\n\"    same as -e option\n",
#endif
	"\n",
	"Note:\n",
	"  If the format ends with `\\n' then outputs `EOS',\n",
	"  otherwise outputs newline every sentence.\n",
	NULL
    };
    char **mes;

    if (fp)
      for (mes = message; *mes; mes++)
	fputs(*mes, fp);
}

/*
 *  usage()
 */
static void usage(fp)
    FILE *fp;
{
    static char *message[] = {
	"Usage: ", CHA_PROG, " [options] [file...]\n",
	"  -s             start ", CHA_NAME, " server\n",
	"  -P port        specify ", CHA_NAME, " server's port number\n",
	"                 (use with -s, the default is 31000)\n",
	"  -D host[:port] connect to ", CHA_NAME, " server\n",
	"  -R             does not read ", CHA_PROG, "rc file (use with -D)\n",
	"  -a             run standalone even if environment variable CHASENSERVER\n",
        "                   is set\n",
	"\n",
	"  -b             show the best path (default)\n",
	"  -m             show all morphemes\n",
	"  -p             show all paths\n",
	"\n",
	"  -f             show formatted morpheme data (default)\n",
	"  -e             show entire morpheme data\n",
	"  -c             show coded morpheme data\n",
	"  -d             show detailed morpheme data for Prolog\n",
	"  -v             show detailed morpheme data for ViCha\n",
	"  -F format      show morpheme with formatted output\n",
	"  -Fh            print help of -F option\n",
	"\n",
	"  -j             Japanese sentence mode\n",
	"  -w width       specify the cost width\n",
	"  -C             use command mode\n",
	"  -r rc_file     use rc_file as a ", CHA_PROG, "rc file other than the default\n",
	"  -L lang        specify languages\n",
	"  -h             print this help\n",
	"  -V             print ", CHA_NAME, " version number\n",
	NULL
    };
    char **mes;

    cha_version(fp);
    if (fp)
      for (mes = message; *mes; mes++)
	fputs(*mes, fp);
}

/*
 * set_chasen_port()
 */
static void set_chasen_port(port)
    char *port;
{
    if ((chasen_port = atoi(port)) <= 0)
      cha_exit(1, "Illegal port No: %s", port);
}

/*
 * set_chasen_server()
 */
static void set_chasen_server(server)
    char *server;
{
    char *colon;
    int len;

    if ((colon = strchr(server, ':')) == NULL) {
	len = strlen(server);
    } else {
	set_chasen_port(colon + 1);
	len = (int)(colon - server);
    }

    if (len >= MAXHOSTNAMELEN)
      len = MAXHOSTNAMELEN - 1;

    memcpy(chasen_host, server, len);
    chasen_host[len] = '\0';
}

/*
 * getopt_argv()
 */
static void getopt_argv(argv)
    char **argv;
{
    int c;

    Cha_optind = 0;
    while ((c = cha_getopt_chasen(argv, stderr)) != EOF) {
	switch (c) {
	  case 'a': /* run standalone */
	    chasen_host[0] = '\0';
	    break;
	  case 's': /* start server */
	    Cha_server_mode = 1;
	    break;
	  case 'D': /* connect to server */
	    set_chasen_server(Cha_optarg);
	    break;
	  case 'r': /* chasenrc file */
	    set_chasenrc_path(Cha_optarg);
	    break;
	  case 'R': /* don't read chasenrc file */
	    set_chasenrc_path("*");
	    break;
	  case 'P': /* port no */
	    set_chasen_port(Cha_optarg);
	    break;
#if 0
	  case 'M':
	    Cha_onmem_dic = 1;
	    break;
#endif
	  case 'F':
	    /* -Fh: print help of -F */
	    if (Cha_optarg[0] == 'h' && Cha_optarg[1] == '\0') {
		opt_form_usage(stderr);
		exit(0);
	    }
	    break;
	  case 'V': cha_version(stderr); exit(0);
	  case 'h': usage(stderr); exit(0);
	  case '?': usage(stderr); exit(1);
	}
    }
}

/*
 * do_chasen_standalone()
 */
static void do_chasen_standalone(fp)
    FILE *fp;
{
    int istty;

    /* ϤɸϤɤ */
    istty = isatty(fileno(stdout));

#if 1
    while (!chasen_fparse(fp, stdout))
      if (!istty)
	fflush(stdout);
#else
    {
	char *out;
	while ((out = chasen_fparse_tostr(fp)) != NULL) {
	    fputs(out, stdout);
	    if (!istty)
	      fflush(stdout);
	}
    }
#endif

}

/*
 * chasen_standalone()
 *
 * return: exit code
 */
static int chasen_standalone(argv)
    char **argv;
{
    /* standalone */
    if (chasen_getopt_argv(argv, stderr))
      return 1;
    argv += Cha_optind;

    if (*argv == NULL)
      do_chasen_standalone(stdin);
    else
      for (; *argv; argv++)
	do_chasen_standalone(cha_fopen(*argv, "r", 1));

    return 0;
}

/*
 * main()
 */
int main(argc, argv)
    int argc;
    char **argv;
{
    char *serv_env;
    extern char *getenv();
    extern int chasen_client(), chasen_server();

    set_progpath(argv[0]);

    /* Ķѿ CHASENSERVER */
    if ((serv_env = getenv("CHASENSERVER")) != NULL)
      set_chasen_server(serv_env);

    getopt_argv(argv);

#if 0
    { char **av;
      for (av = argv; *av; av++)
	printf("## %d: %s\n", argv-av, *av); }
#endif

    if (Cha_server_mode)
      return chasen_server(argv, chasen_port);
    else if (chasen_host[0])
      return chasen_client(argv, chasen_host, chasen_port);
    else
      return chasen_standalone(argv);
}

