/*
    ChibiOS - Copyright (C) 2006,2007,2008,2009,2010,2011,2012,2013,2014,
              2015,2016,2017,2018,2019,2020,2021 Giovanni Di Sirio.

    This file is part of ChibiOS.

    ChibiOS is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation version 3 of the License.

    ChibiOS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * @file    rt/src/chstats.c
 * @brief   Statistics module code.
 *
 * @addtogroup statistics
 * @details Statistics services.
 * @{
 */

#include "ch.h"

#if (CH_DBG_STATISTICS == TRUE) || defined(__DOXYGEN__)

/*===========================================================================*/
/* Module local definitions.                                                 */
/*===========================================================================*/

/*===========================================================================*/
/* Module exported variables.                                                */
/*===========================================================================*/

/*===========================================================================*/
/* Module local types.                                                       */
/*===========================================================================*/

/*===========================================================================*/
/* Module local variables.                                                   */
/*===========================================================================*/

/*===========================================================================*/
/* Module local functions.                                                   */
/*===========================================================================*/

/*===========================================================================*/
/* Module exported functions.                                                */
/*===========================================================================*/

/**
 * @brief   Initializes the statistics module.
 *
 * @init
 */
void _stats_init(void) {

  ch.kernel_stats.n_irq = (ucnt_t)0;
  ch.kernel_stats.n_ctxswc = (ucnt_t)0;
  chTMObjectInit(&ch.kernel_stats.m_crit_thd);
  chTMObjectInit(&ch.kernel_stats.m_crit_isr);
}

/**
 * @brief   Increases the IRQ counter.
 */
void _stats_increase_irq(void) {

  port_lock_from_isr();
  ch.kernel_stats.n_irq++;
  port_unlock_from_isr();
}

/**
 * @brief   Updates context switch related statistics.
 *
 * @param[in] ntp       the thread to be switched in
 * @param[in] otp       the thread to be switched out
 */
void _stats_ctxswc(thread_t *ntp, thread_t *otp) {

  ch.kernel_stats.n_ctxswc++;
  chTMChainMeasurementToX(&otp->stats, &ntp->stats);
}

/**
 * @brief   Starts the measurement of a thread critical zone.
 */
void _stats_start_measure_crit_thd(void) {

  chTMStartMeasurementX(&ch.kernel_stats.m_crit_thd);
}

/**
 * @brief   Stops the measurement of a thread critical zone.
 */
void _stats_stop_measure_crit_thd(void) {

  chTMStopMeasurementX(&ch.kernel_stats.m_crit_thd);
}

/**
 * @brief   Starts the measurement of an ISR critical zone.
 */
void _stats_start_measure_crit_isr(void) {

  chTMStartMeasurementX(&ch.kernel_stats.m_crit_isr);
}

/**
 * @brief   Stops the measurement of an ISR critical zone.
 */
void _stats_stop_measure_crit_isr(void) {

  chTMStopMeasurementX(&ch.kernel_stats.m_crit_isr);
}

#endif /* CH_DBG_STATISTICS == TRUE */

/** @} */
