/**
  * <h2><center>&copy; COPYRIGHT(c) 2017 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#ifndef __GD32VF103_H
#define __GD32VF103_H

#ifdef __cplusplus
 extern "C" {
#endif 

#include <stdint.h>

/** 
  * @brief Analog to Digital Converter  
  */

typedef struct
{
  __IO uint32_t STAT;
  __IO uint32_t CTL0;
  __IO uint32_t CTL1;
  __IO uint32_t SAMPT0;
  __IO uint32_t SAMPT1;
  __IO uint32_t IOFF0;
  __IO uint32_t IOFF1;
  __IO uint32_t IOFF2;
  __IO uint32_t IOFF3;
  __IO uint32_t WDHT;
  __IO uint32_t WDLT;
  __IO uint32_t RSQ0;
  __IO uint32_t RSQ1;
  __IO uint32_t RSQ2;
  __IO uint32_t ISQ;
  __IO uint32_t IDATA0;
  __IO uint32_t IDATA1;
  __IO uint32_t IDATA2;
  __IO uint32_t IDATA3;
  __IO uint32_t RDATA;
  uint32_t  RESERVED[12];
  __IO uint32_t OVSAMPCTL;
} ADC_TypeDef;

typedef struct
{
  __IO uint32_t STAT;               /*!< ADC status register,    used for ADC multimode (bits common to several ADC instances). Address offset: ADC0 base address         */
  __IO uint32_t CTL0;              /*!< ADC control register 1, used for ADC multimode (bits common to several ADC instances). Address offset: ADC0 base address + 0x04  */
  __IO uint32_t CTL1;              /*!< ADC control register 2, used for ADC multimode (bits common to several ADC instances). Address offset: ADC0 base address + 0x08  */
  uint32_t  RESERVED[16];
  __IO uint32_t RDATA;               /*!< ADC data register,      used for ADC multimode (bits common to several ADC instances). Address offset: ADC0 base address + 0x4C  */
} ADC_Common_TypeDef;

/** 
  * @brief Backup Registers  
  */

typedef struct
{
  uint32_t  RESERVED0;
  __IO uint32_t DATA1;
  __IO uint32_t DATA2;
  __IO uint32_t DATA3;
  __IO uint32_t DATA4;
  __IO uint32_t DATA5;
  __IO uint32_t DATA6;
  __IO uint32_t DATA7;
  __IO uint32_t DATA8;
  __IO uint32_t DATA9;
  __IO uint32_t DATA10;
  __IO uint32_t OCTL;
  __IO uint32_t TPCTL;
  __IO uint32_t TPCS;
  uint32_t  RESERVED13[2];
  __IO uint32_t DATA11;
  __IO uint32_t DATA12;
  __IO uint32_t DATA13;
  __IO uint32_t DATA14;
  __IO uint32_t DATA15;
  __IO uint32_t DATA16;
  __IO uint32_t DATA17;
  __IO uint32_t DATA18;
  __IO uint32_t DATA19;
  __IO uint32_t DATA20;
  __IO uint32_t DATA21;
  __IO uint32_t DATA22;
  __IO uint32_t DATA23;
  __IO uint32_t DATA24;
  __IO uint32_t DATA25;
  __IO uint32_t DATA26;
  __IO uint32_t DATA27;
  __IO uint32_t DATA28;
  __IO uint32_t DATA29;
  __IO uint32_t DATA30;
  __IO uint32_t DATA31;
  __IO uint32_t DATA32;
  __IO uint32_t DATA33;
  __IO uint32_t DATA34;
  __IO uint32_t DATA35;
  __IO uint32_t DATA36;
  __IO uint32_t DATA37;
  __IO uint32_t DATA38;
  __IO uint32_t DATA39;
  __IO uint32_t DATA40;
  __IO uint32_t DATA41;
  __IO uint32_t DATA42;
} BKP_TypeDef;
  
/** 
  * @brief Controller Area Network TxMailBox 
  */

typedef struct
{
  __IO uint32_t TMI;
  __IO uint32_t TMP;
  __IO uint32_t TMDATA0;
  __IO uint32_t TMDATA1;
} CAN_TxMailBox_TypeDef;

/** 
  * @brief Controller Area Network FIFOMailBox 
  */
  
typedef struct
{
  __IO uint32_t RFIFOMI;
  __IO uint32_t RFIFOMP;
  __IO uint32_t RFIFOMDATA0;
  __IO uint32_t RFIFOMDATA1;
} CAN_FIFOMailBox_TypeDef;

/** 
  * @brief Controller Area Network FilterRegister 
  */
  
typedef struct
{
  __IO uint32_t FR1;
  __IO uint32_t FR2;
} CAN_FilterRegister_TypeDef;

/** 
  * @brief Controller Area Network 
  */
  
typedef struct
{
  __IO uint32_t CTL;
  __IO uint32_t STAT;
  __IO uint32_t TSTAT;
  __IO uint32_t RFIFO0;
  __IO uint32_t RFIFO1;
  __IO uint32_t INTEN;
  __IO uint32_t ERR;
  __IO uint32_t BT;
  uint32_t  RESERVED0[88];
  CAN_TxMailBox_TypeDef sTxMailBox[3];
  CAN_FIFOMailBox_TypeDef sFIFOMailBox[2];
  uint32_t  RESERVED1[12];
  __IO uint32_t FCTL;
  __IO uint32_t FMCFG;
  uint32_t  RESERVED2;
  __IO uint32_t FSCFG;
  uint32_t  RESERVED3;
  __IO uint32_t FAFIFO;
  uint32_t  RESERVED4;
  __IO uint32_t FW;
  uint32_t  RESERVED5[8];
  CAN_FilterRegister_TypeDef sFilterRegister[28];
} CAN_TypeDef;

/** 
  * @brief CRC calculation unit 
  */

typedef struct
{
  __IO uint32_t DATA;           /*!< CRC Data register,                           Address offset: 0x00 */
  __IO uint8_t  FDATA;          /*!< CRC Independent data register,               Address offset: 0x04 */
  uint8_t       RESERVED0;    /*!< Reserved,                                    Address offset: 0x05 */
  uint16_t      RESERVED1;    /*!< Reserved,                                    Address offset: 0x06 */  
  __IO uint32_t CTL;           /*!< CRC Control register,                        Address offset: 0x08 */ 
} CRC_TypeDef;

/** 
  * @brief Digital to Analog Converter
  */

typedef struct
{
  __IO uint32_t CTL;
  __IO uint32_t SWT;
  __IO uint32_t R12DH0;
  __IO uint32_t L12DH0;
  __IO uint32_t R8DH0;
  __IO uint32_t R12DH1;
  __IO uint32_t L12DH1;
  __IO uint32_t R8DH1;
  __IO uint32_t R12DH;
  __IO uint32_t L12DH;
  __IO uint32_t R8DH;
  __IO uint32_t DO0;
  __IO uint32_t DO1;
} DAC_TypeDef;

/** 
  * @brief Debug MCU
  */

typedef struct
{
  __IO uint32_t IDCODE;
  __IO uint32_t CR;
} DBGMCU_TypeDef;

/** 
  * @brief DMA Controller
  */

typedef struct
{
  __IO uint32_t CTL;
  __IO uint32_t CNT;
  __IO uint32_t PADDR;
  __IO uint32_t MADDR;
} DMA_Channel_TypeDef;

typedef struct
{
  __IO uint32_t INTF;
  __IO uint32_t INTC ;
} DMA_TypeDef;



/** 
  * @brief External Interrupt/Event Controller
  */

typedef struct
{
  __IO uint32_t INTEN;
  __IO uint32_t EVEN;
  __IO uint32_t RTEN;
  __IO uint32_t FTEN;
  __IO uint32_t SWIEV;
  __IO uint32_t PD;
} EXTI_TypeDef;

/** 
  * @brief FLASH Registers
  */

typedef struct
{
  __IO uint32_t WS;
  __IO uint32_t KEY;
  __IO uint32_t OBKEY;
  __IO uint32_t STAT;
  __IO uint32_t CTL;
  __IO uint32_t ADDR;
  __IO uint32_t RESERVED0;
  __IO uint32_t OBSTAT;
  __IO uint32_t WP;
  uint32_t RESERVED1[55];
  __IO  uint32_t PID;
} FLASH_TypeDef;

/** 
  * @brief Option Bytes Registers
  */
  
typedef struct
{
  __IO uint16_t RDP;
  __IO uint16_t USER;
  __IO uint16_t Data0;
  __IO uint16_t Data1;
  __IO uint16_t WRP0;
  __IO uint16_t WRP1;
  __IO uint16_t WRP2;
  __IO uint16_t WRP3;
} OB_TypeDef;

/** 
  * @brief General Purpose I/O
  */

typedef struct
{
  __IO uint32_t CTL0;
  __IO uint32_t CTL1;
  __IO uint32_t ISTAT;
  __IO uint32_t OCTL;
  __IO uint32_t BOP;
  __IO uint32_t BC;
  __IO uint32_t LOCK;
} GPIO_TypeDef;

/** 
  * @brief Alternate Function I/O
  */

typedef struct
{
  __IO uint32_t EVCR;
  __IO uint32_t MAPR;
  __IO uint32_t EXTICR[4];
  uint32_t RESERVED0;
  __IO uint32_t MAPR2;  
} AFIO_TypeDef;
/** 
  * @brief Inter Integrated Circuit Interface
  */

typedef struct
{
  __IO uint32_t CTL0;
  __IO uint32_t CTL1;
  __IO uint32_t SADDR0;
  __IO uint32_t SADDR1;
  __IO uint32_t DATA;
  __IO uint32_t STAT0;
  __IO uint32_t STAT1;
  __IO uint32_t CKCFG;
  __IO uint32_t RT;
  uint32_t RESERVED[27];
  __IO uint32_t FMPCFG;
} I2C_TypeDef;

/** 
  * @brief Independent WATCHDOG
  */

typedef struct
{
  __IO uint32_t CTL;           /*!< Key register,                                Address offset: 0x00 */
  __IO uint32_t PSC;           /*!< Prescaler register,                          Address offset: 0x04 */
  __IO uint32_t RLD;          /*!< Reload register,                             Address offset: 0x08 */
  __IO uint32_t STAT;           /*!< Status register,                             Address offset: 0x0C */
} FWDGT_TypeDef;

/** 
  * @brief Power Control
  */

typedef struct
{
  __IO uint32_t CTL;
  __IO uint32_t CS;
} PMU_TypeDef;

/** 
  * @brief Reset and Clock Control
  */

typedef struct
{
  __IO uint32_t CTL;
  __IO uint32_t CFG0;
  __IO uint32_t INT;
  __IO uint32_t APB2RST;
  __IO uint32_t APB1RST;
  __IO uint32_t AHBEN;
  __IO uint32_t APB2EN;
  __IO uint32_t APB1EN;
  __IO uint32_t BDCTL;
  __IO uint32_t RSTSCK;
  __IO uint32_t AHBRST;
  __IO uint32_t CFG1;
  uint32_t RESERVED;
  __IO uint32_t DSV;
} RCU_TypeDef;

/** 
  * @brief Real-Time Clock
  */

typedef struct
{
  __IO uint32_t INTEN;
  __IO uint32_t CTL;
  __IO uint32_t PSCH;
  __IO uint32_t PSCL;
  __IO uint32_t DIVH;
  __IO uint32_t DIVL;
  __IO uint32_t CNTH;
  __IO uint32_t CNTL;
  __IO uint32_t ALRMH;
  __IO uint32_t ALRML;
} RTC_TypeDef;

/** 
  * @brief Serial Peripheral Interface
  */

typedef struct
{
  __IO uint32_t CTL0;
  __IO uint32_t CTL1;
  __IO uint32_t STAT;
  __IO uint32_t DATA;
  __IO uint32_t CRCPOLY;
  __IO uint32_t RCRC;
  __IO uint32_t TCRC;
  __IO uint32_t I2SCTL;
  __IO uint32_t I2SPSC;
} SPI_TypeDef;


/** 
  * @brief Universal Synchronous Asynchronous Receiver Transmitter
  */
 
typedef struct
{
  __IO uint32_t STAT;         /*!< USART Status register,                   Address offset: 0x00 */
  __IO uint32_t DATA;         /*!< USART Data register,                     Address offset: 0x04 */
  __IO uint32_t BAUD;        /*!< USART Baud rate register,                Address offset: 0x08 */
  __IO uint32_t CTL0;        /*!< USART Control register 1,                Address offset: 0x0C */
  __IO uint32_t CTL1;        /*!< USART Control register 2,                Address offset: 0x10 */
  __IO uint32_t CTL2;        /*!< USART Control register 3,                Address offset: 0x14 */
  __IO uint32_t GP;       /*!< USART Guard time and prescaler register, Address offset: 0x18 */
} USART_TypeDef;

/** 
  * @brief Window WATCHDOG
  */

typedef struct
{
  __IO uint32_t CR;   /*!< WWDG Control register,       Address offset: 0x00 */
  __IO uint32_t CFR;  /*!< WWDG Configuration register, Address offset: 0x04 */
  __IO uint32_t SR;   /*!< WWDG Status register,        Address offset: 0x08 */
} WWDG_TypeDef;

/**
  * @}
  */
  
/** @addtogroup Peripheral_memory_map
  * @{
  */


#define FLASH_BANK1_END       0x0803FFFFU /*!< FLASH END address of bank1 */
#define PERIPH_BASE           0x40000000U /*!< Peripheral base address in the alias region */

#define SRAM_BB_BASE          0x22000000U /*!< SRAM base address in the bit-band region */
#define PERIPH_BB_BASE        0x42000000U /*!< Peripheral base address in the bit-band region */


/*!< Peripheral memory map */
#define APB1PERIPH_BASE       PERIPH_BASE
#define APB2PERIPH_BASE       (PERIPH_BASE + 0x00010000U)
#define AHBPERIPH_BASE        (PERIPH_BASE + 0x00020000U)

#define TIM1_BASE             (APB1PERIPH_BASE + 0x00000000U)
#define TIM2_BASE             (APB1PERIPH_BASE + 0x00000400U)
#define TIM3_BASE             (APB1PERIPH_BASE + 0x00000800U)
#define TIM4_BASE             (APB1PERIPH_BASE + 0x00000C00U)
#define TIM5_BASE             (APB1PERIPH_BASE + 0x00001000U)
#define TIM6_BASE             (APB1PERIPH_BASE + 0x00001400U)
#define WWDG_BASE             (APB1PERIPH_BASE + 0x00002C00U)
#define SPI1_BASE             (APB1PERIPH_BASE + 0x00003800U)
#define SPI2_BASE             (APB1PERIPH_BASE + 0x00003C00U)
#define USART1_BASE           (APB1PERIPH_BASE + 0x00004400U)
#define USART2_BASE           (APB1PERIPH_BASE + 0x00004800U)
#define UART3_BASE            (APB1PERIPH_BASE + 0x00004C00U)
#define UART4_BASE            (APB1PERIPH_BASE + 0x00005000U)
#define I2C0_BASE             (APB1PERIPH_BASE + 0x00005400U)
#define I2C1_BASE             (APB1PERIPH_BASE + 0x5800)
#define CAN0_BASE             (APB1PERIPH_BASE + 0x00006400U)
#define CAN1_BASE             (APB1PERIPH_BASE + 0x00006800U)
#define GPIOA_BASE            (APB2PERIPH_BASE + 0x00000800U)
#define GPIOB_BASE            (APB2PERIPH_BASE + 0x00000C00U)
#define GPIOC_BASE            (APB2PERIPH_BASE + 0x00001000U)
#define GPIOD_BASE            (APB2PERIPH_BASE + 0x00001400U)
#define GPIOE_BASE            (APB2PERIPH_BASE + 0x00001800U)
#define ADC0_BASE             (APB2PERIPH_BASE + 0x00002400U)
#define ADC1_BASE             (APB2PERIPH_BASE + 0x00002800U)
#define TIM0_BASE             (APB2PERIPH_BASE + 0x00002C00U)
#define SPI0_BASE             (APB2PERIPH_BASE + 0x00003000U)
#define USART0_BASE           (APB2PERIPH_BASE + 0x00003800U)

#define SDIO_BASE             (PERIPH_BASE + 0x00018000U)

#define DMA0_BASE             (AHBPERIPH_BASE + 0x00000000U)
#define DMA0_Channel0_BASE    (AHBPERIPH_BASE + 0x00000008U)
#define DMA0_Channel1_BASE    (AHBPERIPH_BASE + 0x0000001CU)
#define DMA0_Channel2_BASE    (AHBPERIPH_BASE + 0x00000030U)
#define DMA0_Channel3_BASE    (AHBPERIPH_BASE + 0x00000044U)
#define DMA0_Channel4_BASE    (AHBPERIPH_BASE + 0x00000058U)
#define DMA0_Channel5_BASE    (AHBPERIPH_BASE + 0x0000006CU)
#define DMA0_Channel6_BASE    (AHBPERIPH_BASE + 0x00000080U)
#define DMA1_BASE             (AHBPERIPH_BASE + 0x00000400U)
#define DMA1_Channel0_BASE    (AHBPERIPH_BASE + 0x00000408U)
#define DMA1_Channel1_BASE    (AHBPERIPH_BASE + 0x0000041CU)
#define DMA1_Channel2_BASE    (AHBPERIPH_BASE + 0x00000430U)
#define DMA1_Channel3_BASE    (AHBPERIPH_BASE + 0x00000444U)
#define DMA1_Channel4_BASE    (AHBPERIPH_BASE + 0x00000458U)

#define FLASH_R_BASE          (AHBPERIPH_BASE + 0x00002000U) /*!< Flash registers base address */
#define FLASHSIZE_BASE        0x1FFFF7E0U    /*!< FLASH Size register base address */
#define UID_BASE              0x1FFFF7E8U    /*!< Unique device ID register base address */



#define DBGMCU_BASE          0xE0042000U /*!< Debug MCU registers base address */


/*!< USB registers base address */
#define USB_USBFS_PERIPH_BASE               0x50000000U

#define USB_OTG_GLOBAL_BASE                  0x00000000U
#define USB_OTG_DEVICE_BASE                  0x00000800U
#define USB_OTG_IN_ENDPOINT_BASE             0x00000900U
#define USB_OTG_OUT_ENDPOINT_BASE            0x00000B00U
#define USB_OTG_EP_REG_SIZE                  0x00000020U
#define USB_OTG_HOST_BASE                    0x00000400U
#define USB_OTG_HOST_PORT_BASE               0x00000440U
#define USB_OTG_HOST_CHANNEL_BASE            0x00000500U
#define USB_OTG_HOST_CHANNEL_SIZE            0x00000020U
#define USB_OTG_PMUCLKCTL_BASE                 0x00000E00U
#define USB_OTG_FIFO_BASE                    0x00001000U
#define USB_OTG_FIFO_SIZE                    0x00001000U

/**
  * @}
  */
  
/** @addtogroup Peripheral_declaration
  * @{
  */  

#define TIM1                ((TIM_TypeDef *)TIM1_BASE)
#define TIM2                ((TIM_TypeDef *)TIM2_BASE)
#define TIM3                ((TIM_TypeDef *)TIM3_BASE)
#define TIM4                ((TIM_TypeDef *)TIM4_BASE)
#define TIM5                ((TIM_TypeDef *)TIM5_BASE)
#define TIM6                ((TIM_TypeDef *)TIM6_BASE)
#define RTC                 ((RTC_TypeDef *)RTC_BASE)
#define WWDG                ((WWDG_TypeDef *)WWDG_BASE)
#define FWDGT                ((FWDGT_TypeDef *)FWDGT_BASE)
#define SPI1                ((SPI_TypeDef *)SPI1_BASE)
#define SPI2                ((SPI_TypeDef *)SPI2_BASE)
#define USART1              ((USART_TypeDef *)USART1_BASE)
#define USART2              ((USART_TypeDef *)USART2_BASE)
#define UART3               ((USART_TypeDef *)UART3_BASE)
#define UART4               ((USART_TypeDef *)UART4_BASE)
#define I2C0                ((I2C_TypeDef *)I2C0_BASE)
#define I2C1                ((I2C_TypeDef *)I2C1_BASE)
#define CAN0                ((CAN_TypeDef *)CAN0_BASE)
#define CAN1                ((CAN_TypeDef *)CAN1_BASE)
#define BKP                 ((BKP_TypeDef *)BKP_BASE)
#define PMU                 ((PMU_TypeDef *)PMU_BASE)
#define DAC                ((DAC_TypeDef *)DAC_BASE)
#define AFIO                ((AFIO_TypeDef *)AFIO_BASE)
#define EXTI                ((EXTI_TypeDef *)EXTI_BASE)
#define GPIOA               ((GPIO_TypeDef *)GPIOA_BASE)
#define GPIOB               ((GPIO_TypeDef *)GPIOB_BASE)
#define GPIOC               ((GPIO_TypeDef *)GPIOC_BASE)
#define GPIOD               ((GPIO_TypeDef *)GPIOD_BASE)
#define GPIOE               ((GPIO_TypeDef *)GPIOE_BASE)
#define ADC0                ((ADC_TypeDef *)ADC0_BASE)
#define ADC1                ((ADC_TypeDef *)ADC1_BASE)
#define ADC02_COMMON        ((ADC_Common_TypeDef *)ADC0_BASE)
#define TIM0                ((TIM_TypeDef *)TIM0_BASE)
#define SPI0                ((SPI_TypeDef *)SPI0_BASE)
#define USART0              ((USART_TypeDef *)USART0_BASE)
#define SDIO                ((SDIO_TypeDef *)SDIO_BASE)
#define DMA0                ((DMA_TypeDef *)DMA0_BASE)
#define DMA1                ((DMA_TypeDef *)DMA1_BASE)
#define DMA0_Channel0       ((DMA_Channel_TypeDef *)DMA0_Channel0_BASE)
#define DMA0_Channel1       ((DMA_Channel_TypeDef *)DMA0_Channel1_BASE)
#define DMA0_Channel2       ((DMA_Channel_TypeDef *)DMA0_Channel2_BASE)
#define DMA0_Channel3       ((DMA_Channel_TypeDef *)DMA0_Channel3_BASE)
#define DMA0_Channel4       ((DMA_Channel_TypeDef *)DMA0_Channel4_BASE)
#define DMA0_Channel5       ((DMA_Channel_TypeDef *)DMA0_Channel5_BASE)
#define DMA0_Channel6       ((DMA_Channel_TypeDef *)DMA0_Channel6_BASE)
#define DMA1_Channel0       ((DMA_Channel_TypeDef *)DMA1_Channel0_BASE)
#define DMA1_Channel1       ((DMA_Channel_TypeDef *)DMA1_Channel1_BASE)
#define DMA1_Channel2       ((DMA_Channel_TypeDef *)DMA1_Channel2_BASE)
#define DMA1_Channel3       ((DMA_Channel_TypeDef *)DMA1_Channel3_BASE)
#define DMA1_Channel4       ((DMA_Channel_TypeDef *)DMA1_Channel4_BASE)
#define RCU                 ((RCU_TypeDef *)RCU_BASE)
#define CRC                 ((CRC_TypeDef *)CRC_BASE)
#define FLASH               ((FLASH_TypeDef *)FLASH_R_BASE)
#define OB                  ((OB_TypeDef *)OB_BASE)
#define DBGMCU              ((DBGMCU_TypeDef *)DBGMCU_BASE)

#define USB_USBFS          ((USB_OTG_GlobalTypeDef *)USB_USBFS_PERIPH_BASE)

/**
  * @}
  */

/** @addtogroup Exported_constants
  * @{
  */
  
  /** @addtogroup Peripheral_Registers_Bits_Definition
  * @{
  */
    
/******************************************************************************/
/*                         Peripheral Registers_Bits_Definition               */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/*                       CRC calculation unit (CRC)                           */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for CRC_DATA register  *********************/
#define CRC_DATA_DATA_Pos                       (0U)                               
#define CRC_DATA_DATA_Msk                       (0xFFFFFFFFU << CRC_DATA_DATA_Pos)     /*!< 0xFFFFFFFF */
#define CRC_DATA_DATA                           CRC_DATA_DATA_Msk                      /*!< Data register bits */

/*******************  Bit definition for CRC_FDATA register  ********************/
#define CRC_FDATA_FDATA_Pos                     (0U)                               
#define CRC_FDATA_FDATA_Msk                     (0xFFU << CRC_FDATA_FDATA_Pos)         /*!< 0x000000FF */
#define CRC_FDATA_FDATA                         CRC_FDATA_FDATA_Msk                    /*!< General-purpose 8-bit data register bits */

/********************  Bit definition for CRC_CTL register  ********************/
#define CRC_CTL_RST_Pos                          (0U)                               
#define CRC_CTL_RST_Msk                          (0x1U << CRC_CTL_RST_Pos)         /*!< 0x00000001 */
#define CRC_CTL_RST                              CRC_CTL_RST_Msk                   /*!< RESET bit */

/******************************************************************************/
/*                                                                            */
/*                             Power Control                                  */
/*                                                                            */
/******************************************************************************/

/********************  Bit definition for PMU_CTL register  ********************/
#define PMU_CTL_LDOLP_Pos                     (0U)                               
#define PMU_CTL_LDOLP_Msk                     (0x1U << PMU_CTL_LDOLP_Pos)          /*!< 0x00000001 */
#define PMU_CTL_LDOLP                         PMU_CTL_LDOLP_Msk                    /*!< Low-Power Deepsleep */
#define PMU_CTL_STBMOD_Pos                     (1U)                               
#define PMU_CTL_STBMOD_Msk                     (0x1U << PMU_CTL_STBMOD_Pos)          /*!< 0x00000002 */
#define PMU_CTL_STBMOD                         PMU_CTL_STBMOD_Msk                    /*!< Power Down Deepsleep */
#define PMU_CTL_WURST_Pos                     (2U)                               
#define PMU_CTL_WURST_Msk                     (0x1U << PMU_CTL_WURST_Pos)          /*!< 0x00000004 */
#define PMU_CTL_WURST                         PMU_CTL_WURST_Msk                    /*!< Clear Wakeup Flag */
#define PMU_CTL_STBRST_Pos                     (3U)                               
#define PMU_CTL_STBRST_Msk                     (0x1U << PMU_CTL_STBRST_Pos)          /*!< 0x00000008 */
#define PMU_CTL_STBRST                         PMU_CTL_STBRST_Msk                    /*!< Clear Standby Flag */
#define PMU_CTL_LVDEN_Pos                     (4U)                               
#define PMU_CTL_LVDEN_Msk                     (0x1U << PMU_CTL_LVDEN_Pos)          /*!< 0x00000010 */
#define PMU_CTL_LVDEN                         PMU_CTL_LVDEN_Msk                    /*!< Power Voltage Detector Enable */

#define PMU_CTL_LVDT_Pos                      (5U)                               
#define PMU_CTL_LVDT_Msk                      (0x7U << PMU_CTL_LVDT_Pos)           /*!< 0x000000E0 */
#define PMU_CTL_LVDT                          PMU_CTL_LVDT_Msk                     /*!< PLS[2:0] bits (PVD Level Selection) */
#define PMU_CTL_LVDT_0                        (0x1U << PMU_CTL_LVDT_Pos)           /*!< 0x00000020 */
#define PMU_CTL_LVDT_1                        (0x2U << PMU_CTL_LVDT_Pos)           /*!< 0x00000040 */
#define PMU_CTL_LVDT_2                        (0x4U << PMU_CTL_LVDT_Pos)           /*!< 0x00000080 */

/*!< PVD level configuration */
#define PMU_CTL_LVDT_LEV0                      0x00000000U                           /*!< PVD level 2.2V */
#define PMU_CTL_LVDT_LEV1                      0x00000020U                           /*!< PVD level 2.3V */
#define PMU_CTL_LVDT_LEV2                      0x00000040U                           /*!< PVD level 2.4V */
#define PMU_CTL_LVDT_LEV3                      0x00000060U                           /*!< PVD level 2.5V */
#define PMU_CTL_LVDT_LEV4                      0x00000080U                           /*!< PVD level 2.6V */
#define PMU_CTL_LVDT_LEV5                      0x000000A0U                           /*!< PVD level 2.7V */
#define PMU_CTL_LVDT_LEV6                      0x000000C0U                           /*!< PVD level 2.8V */
#define PMU_CTL_LVDT_LEV7                      0x000000E0U                           /*!< PVD level 2.9V */

/* Legacy defines */
#define PMU_CTL_LVDT_2V2                       PMU_CTL_LVDT_LEV0
#define PMU_CTL_LVDT_2V3                       PMU_CTL_LVDT_LEV1
#define PMU_CTL_LVDT_2V4                       PMU_CTL_LVDT_LEV2
#define PMU_CTL_LVDT_2V5                       PMU_CTL_LVDT_LEV3
#define PMU_CTL_LVDT_2V6                       PMU_CTL_LVDT_LEV4
#define PMU_CTL_LVDT_2V7                       PMU_CTL_LVDT_LEV5
#define PMU_CTL_LVDT_2V8                       PMU_CTL_LVDT_LEV6
#define PMU_CTL_LVDT_2V9                       PMU_CTL_LVDT_LEV7

#define PMU_CTL_BKPWEN_Pos                      (8U)                               
#define PMU_CTL_BKPWEN_Msk                      (0x1U << PMU_CTL_BKPWEN_Pos)           /*!< 0x00000100 */
#define PMU_CTL_BKPWEN                          PMU_CTL_BKPWEN_Msk                     /*!< Disable Backup Domain write protection */


/*******************  Bit definition for PMU_CS register  ********************/
#define PMU_CS_WUF_Pos                     (0U)                               
#define PMU_CS_WUF_Msk                     (0x1U << PMU_CS_WUF_Pos)          /*!< 0x00000001 */
#define PMU_CS_WUF                         PMU_CS_WUF_Msk                    /*!< Wakeup Flag */
#define PMU_CS_STBF_Pos                     (1U)                               
#define PMU_CS_STBF_Msk                     (0x1U << PMU_CS_STBF_Pos)          /*!< 0x00000002 */
#define PMU_CS_STBF                         PMU_CS_STBF_Msk                    /*!< Standby Flag */
#define PMU_CS_LVDF_Pos                    (2U)                               
#define PMU_CS_LVDF_Msk                    (0x1U << PMU_CS_LVDF_Pos)         /*!< 0x00000004 */
#define PMU_CS_LVDF                        PMU_CS_LVDF_Msk                   /*!< PVD Output */
#define PMU_CS_WUPEN_Pos                    (8U)                               
#define PMU_CS_WUPEN_Msk                    (0x1U << PMU_CS_WUPEN_Pos)         /*!< 0x00000100 */
#define PMU_CS_WUPEN                        PMU_CS_WUPEN_Msk                   /*!< Enable WKUP pin */

/******************************************************************************/
/*                                                                            */
/*                            Backup registers                                */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for BKP_DR1 register  ********************/
#define BKP_DATA1_DATA_Pos                       (0U)                               
#define BKP_DATA1_DATA_Msk                       (0xFFFFU << BKP_DATA1_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA1_DATA                           BKP_DATA1_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR2 register  ********************/
#define BKP_DATA2_DATA_Pos                       (0U)                               
#define BKP_DATA2_DATA_Msk                       (0xFFFFU << BKP_DATA2_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA2_DATA                           BKP_DATA2_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR3 register  ********************/
#define BKP_DATA3_DATA_Pos                       (0U)                               
#define BKP_DATA3_DATA_Msk                       (0xFFFFU << BKP_DATA3_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA3_DATA                           BKP_DATA3_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR4 register  ********************/
#define BKP_DATA4_DATA_Pos                       (0U)                               
#define BKP_DATA4_DATA_Msk                       (0xFFFFU << BKP_DATA4_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA4_DATA                           BKP_DATA4_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR5 register  ********************/
#define BKP_DATA5_DATA_Pos                       (0U)                               
#define BKP_DATA5_DATA_Msk                       (0xFFFFU << BKP_DATA5_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA5_DATA                           BKP_DATA5_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR6 register  ********************/
#define BKP_DATA6_DATA_Pos                       (0U)                               
#define BKP_DATA6_DATA_Msk                       (0xFFFFU << BKP_DATA6_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA6_DATA                           BKP_DATA6_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR7 register  ********************/
#define BKP_DATA7_DATA_Pos                       (0U)                               
#define BKP_DATA7_DATA_Msk                       (0xFFFFU << BKP_DATA7_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA7_DATA                           BKP_DATA7_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR8 register  ********************/
#define BKP_DATA8_DATA_Pos                       (0U)                               
#define BKP_DATA8_DATA_Msk                       (0xFFFFU << BKP_DATA8_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA8_DATA                           BKP_DATA8_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR9 register  ********************/
#define BKP_DATA9_DATA_Pos                       (0U)                               
#define BKP_DATA9_DATA_Msk                       (0xFFFFU << BKP_DATA9_DATA_Pos)         /*!< 0x0000FFFF */
#define BKP_DATA9_DATA                           BKP_DATA9_DATA_Msk                      /*!< Backup data */

/*******************  Bit definition for BKP_DR10 register  *******************/
#define BKP_DATA10_DATA_Pos                      (0U)                               
#define BKP_DATA10_DATA_Msk                      (0xFFFFU << BKP_DATA10_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA10_DATA                          BKP_DATA10_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR11 register  *******************/
#define BKP_DATA11_DATA_Pos                      (0U)                               
#define BKP_DATA11_DATA_Msk                      (0xFFFFU << BKP_DATA11_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA11_DATA                          BKP_DATA11_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR12 register  *******************/
#define BKP_DATA12_DATA_Pos                      (0U)                               
#define BKP_DATA12_DATA_Msk                      (0xFFFFU << BKP_DATA12_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA12_DATA                          BKP_DATA12_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR13 register  *******************/
#define BKP_DATA13_DATA_Pos                      (0U)                               
#define BKP_DATA13_DATA_Msk                      (0xFFFFU << BKP_DATA13_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA13_DATA                          BKP_DATA13_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR14 register  *******************/
#define BKP_DATA14_DATA_Pos                      (0U)                               
#define BKP_DATA14_DATA_Msk                      (0xFFFFU << BKP_DATA14_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA14_DATA                          BKP_DATA14_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR15 register  *******************/
#define BKP_DATA15_DATA_Pos                      (0U)                               
#define BKP_DATA15_DATA_Msk                      (0xFFFFU << BKP_DATA15_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA15_DATA                          BKP_DATA15_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR16 register  *******************/
#define BKP_DATA16_DATA_Pos                      (0U)                               
#define BKP_DATA16_DATA_Msk                      (0xFFFFU << BKP_DATA16_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA16_DATA                          BKP_DATA16_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR17 register  *******************/
#define BKP_DATA17_DATA_Pos                      (0U)                               
#define BKP_DATA17_DATA_Msk                      (0xFFFFU << BKP_DATA17_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA17_DATA                          BKP_DATA17_DATA_Msk                     /*!< Backup data */

/******************  Bit definition for BKP_DR18 register  ********************/
#define BKP_DATA18_DATA_Pos                      (0U)                               
#define BKP_DATA18_DATA_Msk                      (0xFFFFU << BKP_DATA18_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA18_DATA                          BKP_DATA18_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR19 register  *******************/
#define BKP_DATA19_DATA_Pos                      (0U)                               
#define BKP_DATA19_DATA_Msk                      (0xFFFFU << BKP_DATA19_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA19_DATA                          BKP_DATA19_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR20 register  *******************/
#define BKP_DATA20_DATA_Pos                      (0U)                               
#define BKP_DATA20_DATA_Msk                      (0xFFFFU << BKP_DATA20_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA20_DATA                          BKP_DATA20_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR21 register  *******************/
#define BKP_DATA21_DATA_Pos                      (0U)                               
#define BKP_DATA21_DATA_Msk                      (0xFFFFU << BKP_DATA21_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA21_DATA                          BKP_DATA21_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR22 register  *******************/
#define BKP_DATA22_DATA_Pos                      (0U)                               
#define BKP_DATA22_DATA_Msk                      (0xFFFFU << BKP_DATA22_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA22_DATA                          BKP_DATA22_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR23 register  *******************/
#define BKP_DATA23_DATA_Pos                      (0U)                               
#define BKP_DATA23_DATA_Msk                      (0xFFFFU << BKP_DATA23_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA23_DATA                          BKP_DATA23_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR24 register  *******************/
#define BKP_DATA24_DATA_Pos                      (0U)                               
#define BKP_DATA24_DATA_Msk                      (0xFFFFU << BKP_DATA24_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA24_DATA                          BKP_DATA24_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR25 register  *******************/
#define BKP_DATA25_DATA_Pos                      (0U)                               
#define BKP_DATA25_DATA_Msk                      (0xFFFFU << BKP_DATA25_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA25_DATA                          BKP_DATA25_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR26 register  *******************/
#define BKP_DATA26_DATA_Pos                      (0U)                               
#define BKP_DATA26_DATA_Msk                      (0xFFFFU << BKP_DATA26_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA26_DATA                          BKP_DATA26_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR27 register  *******************/
#define BKP_DATA27_DATA_Pos                      (0U)                               
#define BKP_DATA27_DATA_Msk                      (0xFFFFU << BKP_DATA27_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA27_DATA                          BKP_DATA27_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR28 register  *******************/
#define BKP_DATA28_DATA_Pos                      (0U)                               
#define BKP_DATA28_DATA_Msk                      (0xFFFFU << BKP_DATA28_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA28_DATA                          BKP_DATA28_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR29 register  *******************/
#define BKP_DATA29_DATA_Pos                      (0U)                               
#define BKP_DATA29_DATA_Msk                      (0xFFFFU << BKP_DATA29_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA29_DATA                          BKP_DATA29_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR30 register  *******************/
#define BKP_DATA30_DATA_Pos                      (0U)                               
#define BKP_DATA30_DATA_Msk                      (0xFFFFU << BKP_DATA30_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA30_DATA                          BKP_DATA30_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR31 register  *******************/
#define BKP_DATA31_DATA_Pos                      (0U)                               
#define BKP_DATA31_DATA_Msk                      (0xFFFFU << BKP_DATA31_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA31_DATA                          BKP_DATA31_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR32 register  *******************/
#define BKP_DATA32_DATA_Pos                      (0U)                               
#define BKP_DATA32_DATA_Msk                      (0xFFFFU << BKP_DATA32_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA32_DATA                          BKP_DATA32_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR33 register  *******************/
#define BKP_DATA33_DATA_Pos                      (0U)                               
#define BKP_DATA33_DATA_Msk                      (0xFFFFU << BKP_DATA33_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA33_DATA                          BKP_DATA33_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR34 register  *******************/
#define BKP_DATA34_DATA_Pos                      (0U)                               
#define BKP_DATA34_DATA_Msk                      (0xFFFFU << BKP_DATA34_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA34_DATA                          BKP_DATA34_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR35 register  *******************/
#define BKP_DATA35_DATA_Pos                      (0U)                               
#define BKP_DATA35_DATA_Msk                      (0xFFFFU << BKP_DATA35_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA35_DATA                          BKP_DATA35_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR36 register  *******************/
#define BKP_DATA36_DATA_Pos                      (0U)                               
#define BKP_DATA36_DATA_Msk                      (0xFFFFU << BKP_DATA36_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA36_DATA                          BKP_DATA36_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR37 register  *******************/
#define BKP_DATA37_DATA_Pos                      (0U)                               
#define BKP_DATA37_DATA_Msk                      (0xFFFFU << BKP_DATA37_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA37_DATA                          BKP_DATA37_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR38 register  *******************/
#define BKP_DATA38_DATA_Pos                      (0U)                               
#define BKP_DATA38_DATA_Msk                      (0xFFFFU << BKP_DATA38_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA38_DATA                          BKP_DATA38_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR39 register  *******************/
#define BKP_DATA39_DATA_Pos                      (0U)                               
#define BKP_DATA39_DATA_Msk                      (0xFFFFU << BKP_DATA39_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA39_DATA                          BKP_DATA39_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR40 register  *******************/
#define BKP_DATA40_DATA_Pos                      (0U)                               
#define BKP_DATA40_DATA_Msk                      (0xFFFFU << BKP_DATA40_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA40_DATA                          BKP_DATA40_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR41 register  *******************/
#define BKP_DATA41_DATA_Pos                      (0U)                               
#define BKP_DATA41_DATA_Msk                      (0xFFFFU << BKP_DATA41_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA41_DATA                          BKP_DATA41_DATA_Msk                     /*!< Backup data */

/*******************  Bit definition for BKP_DR42 register  *******************/
#define BKP_DATA42_DATA_Pos                      (0U)                               
#define BKP_DATA42_DATA_Msk                      (0xFFFFU << BKP_DATA42_DATA_Pos)        /*!< 0x0000FFFF */
#define BKP_DATA42_DATA                          BKP_DATA42_DATA_Msk                     /*!< Backup data */

#define RTC_BKP_NUMBER 42

/******************  Bit definition for BKP_OCTL register  *******************/
#define BKP_OCTL_RCCV_Pos                   (0U)                               
#define BKP_OCTL_RCCV_Msk                   (0x7FU << BKP_OCTL_RCCV_Pos)       /*!< 0x0000007F */
#define BKP_OCTL_RCCV                       BKP_OCTL_RCCV_Msk                  /*!< Calibration value */
#define BKP_OCTL_COEN_Pos                   (7U)                               
#define BKP_OCTL_COEN_Msk                   (0x1U << BKP_OCTL_COEN_Pos)        /*!< 0x00000080 */
#define BKP_OCTL_COEN                       BKP_OCTL_COEN_Msk                  /*!< Calibration Clock Output */
#define BKP_OCTL_ASOEN_Pos                  (8U)                               
#define BKP_OCTL_ASOEN_Msk                  (0x1U << BKP_OCTL_ASOEN_Pos)       /*!< 0x00000100 */
#define BKP_OCTL_ASOEN                      BKP_OCTL_ASOEN_Msk                 /*!< Alarm or Second Output Enable */
#define BKP_OCTL_ROSEL_Pos                  (9U)                               
#define BKP_OCTL_ROSEL_Msk                  (0x1U << BKP_OCTL_ROSEL_Pos)       /*!< 0x00000200 */
#define BKP_OCTL_ROSEL                      BKP_OCTL_ROSEL_Msk                 /*!< Alarm or Second Output Selection */

/********************  Bit definition for BKP_TPCTL register  ********************/
#define BKP_TPCTL_TPEN_Pos                      (0U)                               
#define BKP_TPCTL_TPEN_Msk                      (0x1U << BKP_TPCTL_TPEN_Pos)           /*!< 0x00000001 */
#define BKP_TPCTL_TPEN                          BKP_TPCTL_TPEN_Msk                     /*!< TAMPER pin enable */
#define BKP_TPCTL_TPAL_Pos                     (1U)                               
#define BKP_TPCTL_TPAL_Msk                     (0x1U << BKP_TPCTL_TPAL_Pos)          /*!< 0x00000002 */
#define BKP_TPCTL_TPAL                         BKP_TPCTL_TPAL_Msk                    /*!< TAMPER pin active level */

/*******************  Bit definition for BKP_TPCS register  ********************/
#define BKP_TPCS_TER_Pos                     (0U)                               
#define BKP_TPCS_TER_Msk                     (0x1U << BKP_TPCS_TER_Pos)          /*!< 0x00000001 */
#define BKP_TPCS_TER                         BKP_TPCS_TER_Msk                    /*!< Clear Tamper event */
#define BKP_TPCS_TIR_Pos                     (1U)                               
#define BKP_TPCS_TIR_Msk                     (0x1U << BKP_TPCS_TIR_Pos)          /*!< 0x00000002 */
#define BKP_TPCS_TIR                         BKP_TPCS_TIR_Msk                    /*!< Clear Tamper Interrupt */
#define BKP_TPCS_TPIE_Pos                    (2U)                               
#define BKP_TPCS_TPIE_Msk                    (0x1U << BKP_TPCS_TPIE_Pos)         /*!< 0x00000004 */
#define BKP_TPCS_TPIE                        BKP_TPCS_TPIE_Msk                   /*!< TAMPER Pin interrupt enable */
#define BKP_TPCS_TEF_Pos                     (8U)                               
#define BKP_TPCS_TEF_Msk                     (0x1U << BKP_TPCS_TEF_Pos)          /*!< 0x00000100 */
#define BKP_TPCS_TEF                         BKP_TPCS_TEF_Msk                    /*!< Tamper Event Flag */
#define BKP_TPCS_TIF_Pos                     (9U)                               
#define BKP_TPCS_TIF_Msk                     (0x1U << BKP_TPCS_TIF_Pos)          /*!< 0x00000200 */
#define BKP_TPCS_TIF                         BKP_TPCS_TIF_Msk                    /*!< Tamper Interrupt Flag */

/******************************************************************************/
/*                                                                            */
/*                         Reset and Clock Control                            */
/*                                                                            */
/******************************************************************************/

/********************  Bit definition for RCU_CTL register  ********************/
#define RCU_CTL_IRC8MEN_Pos                     (0U)                              
#define RCU_CTL_IRC8MEN_Msk                     (0x1U << RCU_CTL_IRC8MEN_Pos)        /*!< 0x00000001 */
#define RCU_CTL_IRC8MEN                         RCU_CTL_IRC8MEN_Msk                  /*!< Internal High Speed clock enable */
#define RCU_CTL_IRC8MSTB_Pos                    (1U)                              
#define RCU_CTL_IRC8MSTB_Msk                    (0x1U << RCU_CTL_IRC8MSTB_Pos)       /*!< 0x00000002 */
#define RCU_CTL_IRC8MSTB                        RCU_CTL_IRC8MSTB_Msk                 /*!< Internal High Speed clock ready flag */
#define RCU_CTL_IRC8MADJ_Pos                   (3U)                              
#define RCU_CTL_IRC8MADJ_Msk                   (0x1FU << RCU_CTL_IRC8MADJ_Pos)     /*!< 0x000000F8 */
#define RCU_CTL_IRC8MADJ                       RCU_CTL_IRC8MADJ_Msk                /*!< Internal High Speed clock trimming */
#define RCU_CTL_IRC8CALIB_Pos                    (8U)                              
#define RCU_CTL_IRC8CALIB_Msk                    (0xFFU << RCU_CTL_IRC8CALIB_Pos)      /*!< 0x0000FF00 */
#define RCU_CTL_IRC8CALIB                        RCU_CTL_IRC8CALIB_Msk                 /*!< Internal High Speed clock Calibration */
#define RCU_CTL_HXTALEN_Pos                     (16U)                             
#define RCU_CTL_HXTALEN_Msk                     (0x1U << RCU_CTL_HXTALEN_Pos)        /*!< 0x00010000 */
#define RCU_CTL_HXTALEN                         RCU_CTL_HXTALEN_Msk                  /*!< External High Speed clock enable */
#define RCU_CTL_HXTALSTB_Pos                    (17U)                             
#define RCU_CTL_HXTALSTB_Msk                    (0x1U << RCU_CTL_HXTALSTB_Pos)       /*!< 0x00020000 */
#define RCU_CTL_HXTALSTB                        RCU_CTL_HXTALSTB_Msk                 /*!< External High Speed clock ready flag */
#define RCU_CTL_HXTALBPS_Pos                    (18U)                             
#define RCU_CTL_HXTALBPS_Msk                    (0x1U << RCU_CTL_HXTALBPS_Pos)       /*!< 0x00040000 */
#define RCU_CTL_HXTALBPS                        RCU_CTL_HXTALBPS_Msk                 /*!< External High Speed clock Bypass */
#define RCU_CTL_CKMEN_Pos                     (19U)                             
#define RCU_CTL_CKMEN_Msk                     (0x1U << RCU_CTL_CKMEN_Pos)        /*!< 0x00080000 */
#define RCU_CTL_CKMEN                         RCU_CTL_CKMEN_Msk                  /*!< Clock Security System enable */
#define RCU_CTL_PLLEN_Pos                     (24U)                             
#define RCU_CTL_PLLEN_Msk                     (0x1U << RCU_CTL_PLLEN_Pos)        /*!< 0x01000000 */
#define RCU_CTL_PLLEN                         RCU_CTL_PLLEN_Msk                  /*!< PLL enable */
#define RCU_CTL_PLLSTB_Pos                    (25U)                             
#define RCU_CTL_PLLSTB_Msk                    (0x1U << RCU_CTL_PLLSTB_Pos)       /*!< 0x02000000 */
#define RCU_CTL_PLLSTB                        RCU_CTL_PLLSTB_Msk                 /*!< PLL clock ready flag */
#define RCU_CTL_PLL1EN_Pos                    (26U)                             
#define RCU_CTL_PLL1EN_Msk                    (0x1U << RCU_CTL_PLL1EN_Pos)       /*!< 0x04000000 */
#define RCU_CTL_PLL1EN                        RCU_CTL_PLL1EN_Msk                 /*!< PLL2 enable */
#define RCU_CTL_PLL1STB_Pos                   (27U)                             
#define RCU_CTL_PLL1STB_Msk                   (0x1U << RCU_CTL_PLL1STB_Pos)      /*!< 0x08000000 */
#define RCU_CTL_PLL1STB                       RCU_CTL_PLL1STB_Msk                /*!< PLL2 clock ready flag */
#define RCU_CTL_PLL2EN_Pos                    (28U)                             
#define RCU_CTL_PLL2EN_Msk                    (0x1U << RCU_CTL_PLL2EN_Pos)       /*!< 0x10000000 */
#define RCU_CTL_PLL2EN                        RCU_CTL_PLL2EN_Msk                 /*!< PLL3 enable */
#define RCU_CTL_PLL2STB_Pos                   (29U)                             
#define RCU_CTL_PLL2STB_Msk                   (0x1U << RCU_CTL_PLL2STB_Pos)      /*!< 0x20000000 */
#define RCU_CTL_PLL2STB                       RCU_CTL_PLL2STB_Msk                /*!< PLL3 clock ready flag */

/*******************  Bit definition for RCU_CFG0 register  *******************/
/*!< SW configuration */
#define RCU_CFG0_SCS_Pos                      (0U)                              
#define RCU_CFG0_SCS_Msk                      (0x3U << RCU_CFG0_SCS_Pos)         /*!< 0x00000003 */
#define RCU_CFG0_SCS                          RCU_CFG0_SCS_Msk                   /*!< SW[1:0] bits (System clock Switch) */
#define RCU_CFG0_SCS_0                        (0x1U << RCU_CFG0_SCS_Pos)         /*!< 0x00000001 */
#define RCU_CFG0_SCS_1                        (0x2U << RCU_CFG0_SCS_Pos)         /*!< 0x00000002 */

#define RCU_CFG0_SCS_IRC8M                      0x00000000U                       /*!< IRC8M selected as system clock */
#define RCU_CFG0_SCS_HXTAL                      0x00000001U                       /*!< HXTAL selected as system clock */
#define RCU_CFG0_SCS_PLL                      0x00000002U                       /*!< PLL selected as system clock */

/*!< SWS configuration */
#define RCU_CFG0_SCSS_Pos                     (2U)                              
#define RCU_CFG0_SCSS_Msk                     (0x3U << RCU_CFG0_SCSS_Pos)        /*!< 0x0000000C */
#define RCU_CFG0_SCSS                         RCU_CFG0_SCSS_Msk                  /*!< SWS[1:0] bits (System Clock Switch Status) */
#define RCU_CFG0_SCSS_0                       (0x1U << RCU_CFG0_SCSS_Pos)        /*!< 0x00000004 */
#define RCU_CFG0_SCSS_1                       (0x2U << RCU_CFG0_SCSS_Pos)        /*!< 0x00000008 */

#define RCU_CFG0_SCSS_IRC8M                     0x00000000U                       /*!< IRC8M oscillator used as system clock */
#define RCU_CFG0_SCSS_HXTAL                     0x00000004U                       /*!< HXTAL oscillator used as system clock */
#define RCU_CFG0_SCSS_PLL                     0x00000008U                       /*!< PLL used as system clock */

/*!< HPRE configuration */
#define RCU_CFG0_AHBPSC_Pos                    (4U)                              
#define RCU_CFG0_AHBPSC_Msk                    (0xFU << RCU_CFG0_AHBPSC_Pos)       /*!< 0x000000F0 */
#define RCU_CFG0_AHBPSC                        RCU_CFG0_AHBPSC_Msk                 /*!< HPRE[3:0] bits (AHB prescaler) */
#define RCU_CFG0_AHBPSC_0                      (0x1U << RCU_CFG0_AHBPSC_Pos)       /*!< 0x00000010 */
#define RCU_CFG0_AHBPSC_1                      (0x2U << RCU_CFG0_AHBPSC_Pos)       /*!< 0x00000020 */
#define RCU_CFG0_AHBPSC_2                      (0x4U << RCU_CFG0_AHBPSC_Pos)       /*!< 0x00000040 */
#define RCU_CFG0_AHBPSC_3                      (0x8U << RCU_CFG0_AHBPSC_Pos)       /*!< 0x00000080 */

#define RCU_CFG0_AHBPSC_DIV1                   0x00000000U                       /*!< SYSCLK not divided */
#define RCU_CFG0_AHBPSC_DIV2                   0x00000080U                       /*!< SYSCLK divided by 2 */
#define RCU_CFG0_AHBPSC_DIV4                   0x00000090U                       /*!< SYSCLK divided by 4 */
#define RCU_CFG0_AHBPSC_DIV8                   0x000000A0U                       /*!< SYSCLK divided by 8 */
#define RCU_CFG0_AHBPSC_DIV16                  0x000000B0U                       /*!< SYSCLK divided by 16 */
#define RCU_CFG0_AHBPSC_DIV64                  0x000000C0U                       /*!< SYSCLK divided by 64 */
#define RCU_CFG0_AHBPSC_DIV128                 0x000000D0U                       /*!< SYSCLK divided by 128 */
#define RCU_CFG0_AHBPSC_DIV256                 0x000000E0U                       /*!< SYSCLK divided by 256 */
#define RCU_CFG0_AHBPSC_DIV512                 0x000000F0U                       /*!< SYSCLK divided by 512 */

/*!< PPRE1 configuration */
#define RCU_CFG0_APB1PSC_Pos                   (8U)                              
#define RCU_CFG0_APB1PSC_Msk                   (0x7U << RCU_CFG0_APB1PSC_Pos)      /*!< 0x00000700 */
#define RCU_CFG0_APB1PSC                       RCU_CFG0_APB1PSC_Msk                /*!< PRE1[2:0] bits (APB1 prescaler) */
#define RCU_CFG0_APB1PSC_0                     (0x1U << RCU_CFG0_APB1PSC_Pos)      /*!< 0x00000100 */
#define RCU_CFG0_APB1PSC_1                     (0x2U << RCU_CFG0_APB1PSC_Pos)      /*!< 0x00000200 */
#define RCU_CFG0_APB1PSC_2                     (0x4U << RCU_CFG0_APB1PSC_Pos)      /*!< 0x00000400 */

#define RCU_CFG0_APB1PSC_DIV1                  0x00000000U                       /*!< HCLK not divided */
#define RCU_CFG0_APB1PSC_DIV2                  0x00000400U                       /*!< HCLK divided by 2 */
#define RCU_CFG0_APB1PSC_DIV4                  0x00000500U                       /*!< HCLK divided by 4 */
#define RCU_CFG0_APB1PSC_DIV8                  0x00000600U                       /*!< HCLK divided by 8 */
#define RCU_CFG0_APB1PSC_DIV16                 0x00000700U                       /*!< HCLK divided by 16 */

/*!< PPRE2 configuration */
#define RCU_CFG0_APB2PSC_Pos                   (11U)                             
#define RCU_CFG0_APB2PSC_Msk                   (0x7U << RCU_CFG0_APB2PSC_Pos)      /*!< 0x00003800 */
#define RCU_CFG0_APB2PSC                       RCU_CFG0_APB2PSC_Msk                /*!< PRE2[2:0] bits (APB2 prescaler) */
#define RCU_CFG0_APB2PSC_0                     (0x1U << RCU_CFG0_APB2PSC_Pos)      /*!< 0x00000800 */
#define RCU_CFG0_APB2PSC_1                     (0x2U << RCU_CFG0_APB2PSC_Pos)      /*!< 0x00001000 */
#define RCU_CFG0_APB2PSC_2                     (0x4U << RCU_CFG0_APB2PSC_Pos)      /*!< 0x00002000 */

#define RCU_CFG0_APB2PSC_DIV1                  0x00000000U                       /*!< HCLK not divided */
#define RCU_CFG0_APB2PSC_DIV2                  0x00002000U                       /*!< HCLK divided by 2 */
#define RCU_CFG0_APB2PSC_DIV4                  0x00002800U                       /*!< HCLK divided by 4 */
#define RCU_CFG0_APB2PSC_DIV8                  0x00003000U                       /*!< HCLK divided by 8 */
#define RCU_CFG0_APB2PSC_DIV16                 0x00003800U                       /*!< HCLK divided by 16 */

/*!< ADCPPRE configuration */
#define RCU_CFG0_ADCPSC_Pos                  (14U)                             
#define RCU_CFG0_ADCPSC_Msk                  (0x3U << RCU_CFG0_ADCPSC_Pos)     /*!< 0x0000C000 */
#define RCU_CFG0_ADCPSC                      RCU_CFG0_ADCPSC_Msk               /*!< ADCPRE[1:0] bits (ADC prescaler) */
#define RCU_CFG0_ADCPSC_0                    (0x1U << RCU_CFG0_ADCPSC_Pos)     /*!< 0x00004000 */
#define RCU_CFG0_ADCPSC_1                    (0x2U << RCU_CFG0_ADCPSC_Pos)     /*!< 0x00008000 */

#define RCU_CFG0_ADCPSC_DIV2                 0x00000000U                       /*!< PCLK2 divided by 2 */
#define RCU_CFG0_ADCPSC_DIV4                 0x00004000U                       /*!< PCLK2 divided by 4 */
#define RCU_CFG0_ADCPSC_DIV6                 0x00008000U                       /*!< PCLK2 divided by 6 */
#define RCU_CFG0_ADCPSC_DIV8                 0x0000C000U                       /*!< PCLK2 divided by 8 */

#define RCU_CFG0_PLLSEL_Pos                  (16U)                             
#define RCU_CFG0_PLLSEL_Msk                  (0x1U << RCU_CFG0_PLLSEL_Pos)     /*!< 0x00010000 */
#define RCU_CFG0_PLLSEL                      RCU_CFG0_PLLSEL_Msk               /*!< PLL entry clock source */

#define RCU_CFG0_PREDV0_LSB_Pos                (17U)                             
#define RCU_CFG0_PREDV0_LSB_Msk                (0x1U << RCU_CFG0_PREDV0_LSB_Pos)   /*!< 0x00020000 */
#define RCU_CFG0_PREDV0_LSB                    RCU_CFG0_PREDV0_LSB_Msk             /*!< HXTAL divider for PLL entry */

/*!< PLLMUL configuration */
#define RCU_CFG0_PLLMF_Pos                 (18U)                             
#define RCU_CFG0_PLLMF_Msk                 (0xFU << RCU_CFG0_PLLMF_Pos)    /*!< 0x003C0000 */
#define RCU_CFG0_PLLMF                     RCU_CFG0_PLLMF_Msk              /*!< PLLMUL[3:0] bits (PLL multiplication factor) */
#define RCU_CFG0_PLLMF_0                   (0x1U << RCU_CFG0_PLLMF_Pos)    /*!< 0x00040000 */
#define RCU_CFG0_PLLMF_1                   (0x2U << RCU_CFG0_PLLMF_Pos)    /*!< 0x00080000 */
#define RCU_CFG0_PLLMF_2                   (0x4U << RCU_CFG0_PLLMF_Pos)    /*!< 0x00100000 */
#define RCU_CFG0_PLLMF_3                   (0x8U << RCU_CFG0_PLLMF_Pos)    /*!< 0x00200000 */

#define RCU_CFG0_PREDV0_LSB_PREDV0            0x00000000U                       /*!< PREDV0 clock not divided for PLL entry */
#define RCU_CFG0_PREDV0_LSB_PREDV0_DIV2       0x00020000U                       /*!< PREDV0 clock divided by 2 for PLL entry */

#define RCU_CFG0_PLLMF4_Pos                (19U)                             
#define RCU_CFG0_PLLMF4_Msk                (0x1U << RCU_CFG0_PLLMF4_Pos)   /*!< 0x00080000 */
#define RCU_CFG0_PLLMF4                    RCU_CFG0_PLLMF4_Msk             /*!< PLL input clock * 4 */
#define RCU_CFG0_PLLMF5_Pos                (18U)                             
#define RCU_CFG0_PLLMF5_Msk                (0x3U << RCU_CFG0_PLLMF5_Pos)   /*!< 0x000C0000 */
#define RCU_CFG0_PLLMF5                    RCU_CFG0_PLLMF5_Msk             /*!< PLL input clock * 5 */
#define RCU_CFG0_PLLMF6_Pos                (20U)                             
#define RCU_CFG0_PLLMF6_Msk                (0x1U << RCU_CFG0_PLLMF6_Pos)   /*!< 0x00100000 */
#define RCU_CFG0_PLLMF6                    RCU_CFG0_PLLMF6_Msk             /*!< PLL input clock * 6 */
#define RCU_CFG0_PLLMF7_Pos                (18U)                             
#define RCU_CFG0_PLLMF7_Msk                (0x5U << RCU_CFG0_PLLMF7_Pos)   /*!< 0x00140000 */
#define RCU_CFG0_PLLMF7                    RCU_CFG0_PLLMF7_Msk             /*!< PLL input clock * 7 */
#define RCU_CFG0_PLLMF8_Pos                (19U)                             
#define RCU_CFG0_PLLMF8_Msk                (0x3U << RCU_CFG0_PLLMF8_Pos)   /*!< 0x00180000 */
#define RCU_CFG0_PLLMF8                    RCU_CFG0_PLLMF8_Msk             /*!< PLL input clock * 8 */
#define RCU_CFG0_PLLMF9_Pos                (18U)                             
#define RCU_CFG0_PLLMF9_Msk                (0x7U << RCU_CFG0_PLLMF9_Pos)   /*!< 0x001C0000 */
#define RCU_CFG0_PLLMF9                    RCU_CFG0_PLLMF9_Msk             /*!< PLL input clock * 9 */
#define RCU_CFG0_PLLMF6_5                  0x00340000U                       /*!< PLL input clock * 6.5 */
 
#define RCU_CFG0_USBFSPSC_Pos                (22U)                             
#define RCU_CFG0_USBFSPSC_Msk                (0x1U << RCU_CFG0_USBFSPSC_Pos)   /*!< 0x00400000 */
#define RCU_CFG0_USBFSPSC                    RCU_CFG0_USBFSPSC_Msk             /*!< USB OTG FS prescaler */
 
/*!< MCO configuration */
#define RCU_CFG0_CKOUT0SEL_Pos                     (24U)                             
#define RCU_CFG0_CKOUT0SEL_Msk                     (0xFU << RCU_CFG0_CKOUT0SEL_Pos)        /*!< 0x0F000000 */
#define RCU_CFG0_CKOUT0SEL                         RCU_CFG0_CKOUT0SEL_Msk                  /*!< MCO[3:0] bits (Microcontroller Clock Output) */
#define RCU_CFG0_CKOUT0SEL_0                       (0x1U << RCU_CFG0_CKOUT0SEL_Pos)        /*!< 0x01000000 */
#define RCU_CFG0_CKOUT0SEL_1                       (0x2U << RCU_CFG0_CKOUT0SEL_Pos)        /*!< 0x02000000 */
#define RCU_CFG0_CKOUT0SEL_2                       (0x4U << RCU_CFG0_CKOUT0SEL_Pos)        /*!< 0x04000000 */
#define RCU_CFG0_CKOUT0SEL_3                       (0x8U << RCU_CFG0_CKOUT0SEL_Pos)        /*!< 0x08000000 */

#define RCU_CFG0_CKOUT0SEL_NOCLOCK                 0x00000000U                       /*!< No clock */
#define RCU_CFG0_CKOUT0SEL_SYSCLK                  0x04000000U                       /*!< System clock selected as MCO source */
#define RCU_CFG0_CKOUT0SEL_IRC8M                     0x05000000U                       /*!< IRC8M clock selected as MCO source */
#define RCU_CFG0_CKOUT0SEL_HXTAL                     0x06000000U                       /*!< HXTAL clock selected as MCO source */
#define RCU_CFG0_CKOUT0SEL_CK_PLL_DIV2             0x07000000U                       /*!< PLL clock divided by 2 selected as MCO source */
#define RCU_CFG0_CKOUT0SEL_CK_PLL1                 0x08000000U                       /*!< PLL2 clock selected as MCO source*/
#define RCU_CFG0_CKOUT0SEL_CK_PLL2_DIV2            0x09000000U                       /*!< PLL3 clock divided by 2 selected as MCO source*/
#define RCU_CFG0_CKOUT0SEL_EXT_HXTAL                 0x0A000000U                       /*!< XT1 external 3-25 MHz oscillator clock selected as MCO source */
#define RCU_CFG0_CKOUT0SEL_CK_PLL2                 0x0B000000U                       /*!< PLL3 clock selected as MCO source */

/*!<******************  Bit definition for RCU_INT register  ********************/
#define RCU_INT_IRC40KSTBIF_Pos                  (0U)                              
#define RCU_INT_IRC40KSTBIF_Msk                  (0x1U << RCU_INT_IRC40KSTBIF_Pos)     /*!< 0x00000001 */
#define RCU_INT_IRC40KSTBIF                      RCU_INT_IRC40KSTBIF_Msk               /*!< IRC40K Ready Interrupt flag */
#define RCU_INT_LXTALSTBIF_Pos                  (1U)                              
#define RCU_INT_LXTALSTBIF_Msk                  (0x1U << RCU_INT_LXTALSTBIF_Pos)     /*!< 0x00000002 */
#define RCU_INT_LXTALSTBIF                      RCU_INT_LXTALSTBIF_Msk               /*!< LXTAL Ready Interrupt flag */
#define RCU_INT_IRC8MSTBIF_Pos                  (2U)                              
#define RCU_INT_IRC8MSTBIF_Msk                  (0x1U << RCU_INT_IRC8MSTBIF_Pos)     /*!< 0x00000004 */
#define RCU_INT_IRC8MSTBIF                      RCU_INT_IRC8MSTBIF_Msk               /*!< IRC8M Ready Interrupt flag */
#define RCU_INT_HXTALSTBIF_Pos                  (3U)                              
#define RCU_INT_HXTALSTBIF_Msk                  (0x1U << RCU_INT_HXTALSTBIF_Pos)     /*!< 0x00000008 */
#define RCU_INT_HXTALSTBIF                      RCU_INT_HXTALSTBIF_Msk               /*!< HXTAL Ready Interrupt flag */
#define RCU_INT_PLLSTBIF_Pos                  (4U)                              
#define RCU_INT_PLLSTBIF_Msk                  (0x1U << RCU_INT_PLLSTBIF_Pos)     /*!< 0x00000010 */
#define RCU_INT_PLLSTBIF                      RCU_INT_PLLSTBIF_Msk               /*!< PLL Ready Interrupt flag */
#define RCU_INT_CKMIF_Pos                     (7U)                              
#define RCU_INT_CKMIF_Msk                     (0x1U << RCU_INT_CKMIF_Pos)        /*!< 0x00000080 */
#define RCU_INT_CKMIF                         RCU_INT_CKMIF_Msk                  /*!< Clock Security System Interrupt flag */
#define RCU_INT_IRC40KSTBIE_Pos                 (8U)                              
#define RCU_INT_IRC40KSTBIE_Msk                 (0x1U << RCU_INT_IRC40KSTBIE_Pos)    /*!< 0x00000100 */
#define RCU_INT_IRC40KSTBIE                     RCU_INT_IRC40KSTBIE_Msk              /*!< IRC40K Ready Interrupt Enable */
#define RCU_INT_LXTALSTBIE_Pos                 (9U)                              
#define RCU_INT_LXTALSTBIE_Msk                 (0x1U << RCU_INT_LXTALSTBIE_Pos)    /*!< 0x00000200 */
#define RCU_INT_LXTALSTBIE                     RCU_INT_LXTALSTBIE_Msk              /*!< LXTAL Ready Interrupt Enable */
#define RCU_INT_IRC8MSTBIE_Pos                 (10U)                             
#define RCU_INT_IRC8MSTBIE_Msk                 (0x1U << RCU_INT_IRC8MSTBIE_Pos)    /*!< 0x00000400 */
#define RCU_INT_IRC8MSTBIE                     RCU_INT_IRC8MSTBIE_Msk              /*!< IRC8M Ready Interrupt Enable */
#define RCU_INT_HXTALSTBIE_Pos                 (11U)                             
#define RCU_INT_HXTALSTBIE_Msk                 (0x1U << RCU_INT_HXTALSTBIE_Pos)    /*!< 0x00000800 */
#define RCU_INT_HXTALSTBIE                     RCU_INT_HXTALSTBIE_Msk              /*!< HXTAL Ready Interrupt Enable */
#define RCU_INT_PLLSTBIE_Pos                 (12U)                             
#define RCU_INT_PLLSTBIE_Msk                 (0x1U << RCU_INT_PLLSTBIE_Pos)    /*!< 0x00001000 */
#define RCU_INT_PLLSTBIE                     RCU_INT_PLLSTBIE_Msk              /*!< PLL Ready Interrupt Enable */
#define RCU_INT_IRC40KSTBIC_Pos                  (16U)                             
#define RCU_INT_IRC40KSTBIC_Msk                  (0x1U << RCU_INT_IRC40KSTBIC_Pos)     /*!< 0x00010000 */
#define RCU_INT_IRC40KSTBIC                      RCU_INT_IRC40KSTBIC_Msk               /*!< IRC40K Ready Interrupt Clear */
#define RCU_INT_LXTALSTBIC_Pos                  (17U)                             
#define RCU_INT_LXTALSTBIC_Msk                  (0x1U << RCU_INT_LXTALSTBIC_Pos)     /*!< 0x00020000 */
#define RCU_INT_LXTALSTBIC                      RCU_INT_LXTALSTBIC_Msk               /*!< LXTAL Ready Interrupt Clear */
#define RCU_INT_IRC8MSTBIC_Pos                  (18U)                             
#define RCU_INT_IRC8MSTBIC_Msk                  (0x1U << RCU_INT_IRC8MSTBIC_Pos)     /*!< 0x00040000 */
#define RCU_INT_IRC8MSTBIC                      RCU_INT_IRC8MSTBIC_Msk               /*!< IRC8M Ready Interrupt Clear */
#define RCU_INT_HXTALSTBIC_Pos                  (19U)                             
#define RCU_INT_HXTALSTBIC_Msk                  (0x1U << RCU_INT_HXTALSTBIC_Pos)     /*!< 0x00080000 */
#define RCU_INT_HXTALSTBIC                      RCU_INT_HXTALSTBIC_Msk               /*!< HXTAL Ready Interrupt Clear */
#define RCU_INT_PLLSTBIC_Pos                  (20U)                             
#define RCU_INT_PLLSTBIC_Msk                  (0x1U << RCU_INT_PLLSTBIC_Pos)     /*!< 0x00100000 */
#define RCU_INT_PLLSTBIC                      RCU_INT_PLLSTBIC_Msk               /*!< PLL Ready Interrupt Clear */
#define RCU_INT_CKMIC_Pos                     (23U)                             
#define RCU_INT_CKMIC_Msk                     (0x1U << RCU_INT_CKMIC_Pos)        /*!< 0x00800000 */
#define RCU_INT_CKMIC                         RCU_INT_CKMIC_Msk                  /*!< Clock Security System Interrupt Clear */

#define RCU_INT_PLL1STBIF_Pos                 (5U)                              
#define RCU_INT_PLL1STBIF_Msk                 (0x1U << RCU_INT_PLL1STBIF_Pos)    /*!< 0x00000020 */
#define RCU_INT_PLL1STBIF                     RCU_INT_PLL1STBIF_Msk              /*!< PLL2 Ready Interrupt flag */
#define RCU_INT_PLL2STBIF_Pos                 (6U)                              
#define RCU_INT_PLL2STBIF_Msk                 (0x1U << RCU_INT_PLL2STBIF_Pos)    /*!< 0x00000040 */
#define RCU_INT_PLL2STBIF                     RCU_INT_PLL2STBIF_Msk              /*!< PLL3 Ready Interrupt flag */
#define RCU_INT_PLL1STBIE_Pos                (13U)                             
#define RCU_INT_PLL1STBIE_Msk                (0x1U << RCU_INT_PLL1STBIE_Pos)   /*!< 0x00002000 */
#define RCU_INT_PLL1STBIE                    RCU_INT_PLL1STBIE_Msk             /*!< PLL2 Ready Interrupt Enable */
#define RCU_INT_PLL2STBIE_Pos                (14U)                             
#define RCU_INT_PLL2STBIE_Msk                (0x1U << RCU_INT_PLL2STBIE_Pos)   /*!< 0x00004000 */
#define RCU_INT_PLL2STBIE                    RCU_INT_PLL2STBIE_Msk             /*!< PLL3 Ready Interrupt Enable */
#define RCU_INT_PLL1STBIC_Pos                 (21U)                             
#define RCU_INT_PLL1STBIC_Msk                 (0x1U << RCU_INT_PLL1STBIC_Pos)    /*!< 0x00200000 */
#define RCU_INT_PLL1STBIC                     RCU_INT_PLL1STBIC_Msk              /*!< PLL2 Ready Interrupt Clear */
#define RCU_INT_PLL2STBIC_Pos                 (22U)                             
#define RCU_INT_PLL2STBIC_Msk                 (0x1U << RCU_INT_PLL2STBIC_Pos)    /*!< 0x00400000 */
#define RCU_INT_PLL2STBIC                     RCU_INT_PLL2STBIC_Msk              /*!< PLL3 Ready Interrupt Clear */

/*****************  Bit definition for RCU_APB2RST register  *****************/
#define RCU_APB2RST_AFRST_Pos             (0U)                              
#define RCU_APB2RST_AFRST_Msk             (0x1U << RCU_APB2RST_AFRST_Pos) /*!< 0x00000001 */
#define RCU_APB2RST_AFRST                 RCU_APB2RST_AFRST_Msk          /*!< Alternate Function I/O reset */
#define RCU_APB2RST_PARST_Pos             (2U)                              
#define RCU_APB2RST_PARST_Msk             (0x1U << RCU_APB2RST_PARST_Pos) /*!< 0x00000004 */
#define RCU_APB2RST_PARST                 RCU_APB2RST_PARST_Msk          /*!< I/O port A reset */
#define RCU_APB2RST_PBRST_Pos             (3U)                              
#define RCU_APB2RST_PBRST_Msk             (0x1U << RCU_APB2RST_PBRST_Pos) /*!< 0x00000008 */
#define RCU_APB2RST_PBRST                 RCU_APB2RST_PBRST_Msk          /*!< I/O port B reset */
#define RCU_APB2RST_PCRST_Pos             (4U)                              
#define RCU_APB2RST_PCRST_Msk             (0x1U << RCU_APB2RST_PCRST_Pos) /*!< 0x00000010 */
#define RCU_APB2RST_PCRST                 RCU_APB2RST_PCRST_Msk          /*!< I/O port C reset */
#define RCU_APB2RST_PDRST_Pos             (5U)                              
#define RCU_APB2RST_PDRST_Msk             (0x1U << RCU_APB2RST_PDRST_Pos) /*!< 0x00000020 */
#define RCU_APB2RST_PDRST                 RCU_APB2RST_PDRST_Msk          /*!< I/O port D reset */
#define RCU_APB2RST_ADC0RST_Pos             (9U)                              
#define RCU_APB2RST_ADC0RST_Msk             (0x1U << RCU_APB2RST_ADC0RST_Pos) /*!< 0x00000200 */
#define RCU_APB2RST_ADC0RST                 RCU_APB2RST_ADC0RST_Msk          /*!< ADC 1 interface reset */

#define RCU_APB2RST_ADC1RST_Pos             (10U)                             
#define RCU_APB2RST_ADC1RST_Msk             (0x1U << RCU_APB2RST_ADC1RST_Pos) /*!< 0x00000400 */
#define RCU_APB2RST_ADC1RST                 RCU_APB2RST_ADC1RST_Msk          /*!< ADC 2 interface reset */

#define RCU_APB2RST_TIMER0RST_Pos             (11U)                             
#define RCU_APB2RST_TIMER0RST_Msk             (0x1U << RCU_APB2RST_TIMER0RST_Pos) /*!< 0x00000800 */
#define RCU_APB2RST_TIMER0RST                 RCU_APB2RST_TIMER0RST_Msk          /*!< TIM0 Timer reset */
#define RCU_APB2RST_SPI0RST_Pos             (12U)                             
#define RCU_APB2RST_SPI0RST_Msk             (0x1U << RCU_APB2RST_SPI0RST_Pos) /*!< 0x00001000 */
#define RCU_APB2RST_SPI0RST                 RCU_APB2RST_SPI0RST_Msk          /*!< SPI 1 reset */
#define RCU_APB2RST_USART0RST_Pos           (14U)                             
#define RCU_APB2RST_USART0RST_Msk           (0x1U << RCU_APB2RST_USART0RST_Pos) /*!< 0x00004000 */
#define RCU_APB2RST_USART0RST               RCU_APB2RST_USART0RST_Msk        /*!< USART0 reset */


#define RCU_APB2RST_PERST_Pos             (6U)                              
#define RCU_APB2RST_PERST_Msk             (0x1U << RCU_APB2RST_PERST_Pos) /*!< 0x00000040 */
#define RCU_APB2RST_PERST                 RCU_APB2RST_PERST_Msk          /*!< I/O port E reset */


/*****************  Bit definition for RCU_APB1RST register  *****************/
#define RCU_APB1RST_TIMER1RST_Pos             (0U)                              
#define RCU_APB1RST_TIMER1RST_Msk             (0x1U << RCU_APB1RST_TIMER1RST_Pos) /*!< 0x00000001 */
#define RCU_APB1RST_TIMER1RST                 RCU_APB1RST_TIMER1RST_Msk          /*!< Timer 2 reset */
#define RCU_APB1RST_TIMER2RST_Pos             (1U)                              
#define RCU_APB1RST_TIMER2RST_Msk             (0x1U << RCU_APB1RST_TIMER2RST_Pos) /*!< 0x00000002 */
#define RCU_APB1RST_TIMER2RST                 RCU_APB1RST_TIMER2RST_Msk          /*!< Timer 3 reset */
#define RCU_APB1RST_WWDGTRST_Pos             (11U)                             
#define RCU_APB1RST_WWDGTRST_Msk             (0x1U << RCU_APB1RST_WWDGTRST_Pos) /*!< 0x00000800 */
#define RCU_APB1RST_WWDGTRST                 RCU_APB1RST_WWDGTRST_Msk          /*!< Window Watchdog reset */
#define RCU_APB1RST_USART1RST_Pos           (17U)                             
#define RCU_APB1RST_USART1RST_Msk           (0x1U << RCU_APB1RST_USART1RST_Pos) /*!< 0x00020000 */
#define RCU_APB1RST_USART1RST               RCU_APB1RST_USART1RST_Msk        /*!< USART 2 reset */
#define RCU_APB1RST_I2C0RST_Pos             (21U)                             
#define RCU_APB1RST_I2C0RST_Msk             (0x1U << RCU_APB1RST_I2C0RST_Pos) /*!< 0x00200000 */
#define RCU_APB1RST_I2C0RST                 RCU_APB1RST_I2C0RST_Msk          /*!< I2C 1 reset */

#define RCU_APB1RST_CAN0RST_Pos             (25U)                             
#define RCU_APB1RST_CAN0RST_Msk             (0x1U << RCU_APB1RST_CAN0RST_Pos) /*!< 0x02000000 */
#define RCU_APB1RST_CAN0RST                 RCU_APB1RST_CAN0RST_Msk          /*!< CAN0 reset */

#define RCU_APB1RST_BKPIRST_Pos              (27U)                             
#define RCU_APB1RST_BKPIRST_Msk              (0x1U << RCU_APB1RST_BKPIRST_Pos) /*!< 0x08000000 */
#define RCU_APB1RST_BKPIRST                  RCU_APB1RST_BKPIRST_Msk           /*!< Backup interface reset */
#define RCU_APB1RST_PMURST_Pos              (28U)                             
#define RCU_APB1RST_PMURST_Msk              (0x1U << RCU_APB1RST_PMURST_Pos) /*!< 0x10000000 */
#define RCU_APB1RST_PMURST                  RCU_APB1RST_PMURST_Msk           /*!< Power interface reset */

#define RCU_APB1RST_TIMER3RST_Pos             (2U)                              
#define RCU_APB1RST_TIMER3RST_Msk             (0x1U << RCU_APB1RST_TIMER3RST_Pos) /*!< 0x00000004 */
#define RCU_APB1RST_TIMER3RST                 RCU_APB1RST_TIMER3RST_Msk          /*!< Timer 4 reset */
#define RCU_APB1RST_SPI1RST_Pos             (14U)                             
#define RCU_APB1RST_SPI1RST_Msk             (0x1U << RCU_APB1RST_SPI1RST_Pos) /*!< 0x00004000 */
#define RCU_APB1RST_SPI1RST                 RCU_APB1RST_SPI1RST_Msk          /*!< SPI 2 reset */
#define RCU_APB1RST_USART2RST_Pos           (18U)                             
#define RCU_APB1RST_USART2RST_Msk           (0x1U << RCU_APB1RST_USART2RST_Pos) /*!< 0x00040000 */
#define RCU_APB1RST_USART2RST               RCU_APB1RST_USART2RST_Msk        /*!< USART 3 reset */
#define RCU_APB1RST_I2C1RST_Pos             (22U)                             
#define RCU_APB1RST_I2C1RST_Msk             (0x1U << RCU_APB1RST_I2C1RST_Pos) /*!< 0x00400000 */
#define RCU_APB1RST_I2C1RST                 RCU_APB1RST_I2C1RST_Msk          /*!< I2C 2 reset */


#define RCU_APB1RST_TIMER4RST_Pos             (3U)                              
#define RCU_APB1RST_TIMER4RST_Msk             (0x1U << RCU_APB1RST_TIMER4RST_Pos) /*!< 0x00000008 */
#define RCU_APB1RST_TIMER4RST                 RCU_APB1RST_TIMER4RST_Msk          /*!< Timer 5 reset */
#define RCU_APB1RST_TIMER5RST_Pos             (4U)                              
#define RCU_APB1RST_TIMER5RST_Msk             (0x1U << RCU_APB1RST_TIMER5RST_Pos) /*!< 0x00000010 */
#define RCU_APB1RST_TIMER5RST                 RCU_APB1RST_TIMER5RST_Msk          /*!< Timer 6 reset */
#define RCU_APB1RST_TIMER6RST_Pos             (5U)                              
#define RCU_APB1RST_TIMER6RST_Msk             (0x1U << RCU_APB1RST_TIMER6RST_Pos) /*!< 0x00000020 */
#define RCU_APB1RST_TIMER6RST                 RCU_APB1RST_TIMER6RST_Msk          /*!< Timer 7 reset */
#define RCU_APB1RST_SPI2RST_Pos             (15U)                             
#define RCU_APB1RST_SPI2RST_Msk             (0x1U << RCU_APB1RST_SPI2RST_Pos) /*!< 0x00008000 */
#define RCU_APB1RST_SPI2RST                 RCU_APB1RST_SPI2RST_Msk          /*!< SPI 3 reset */
#define RCU_APB1RST_UART3RST_Pos            (19U)                             
#define RCU_APB1RST_UART3RST_Msk            (0x1U << RCU_APB1RST_UART3RST_Pos) /*!< 0x00080000 */
#define RCU_APB1RST_UART3RST                RCU_APB1RST_UART3RST_Msk         /*!< UART 4 reset */
#define RCU_APB1RST_UART4RST_Pos            (20U)                             
#define RCU_APB1RST_UART4RST_Msk            (0x1U << RCU_APB1RST_UART4RST_Pos) /*!< 0x00100000 */
#define RCU_APB1RST_UART4RST                RCU_APB1RST_UART4RST_Msk         /*!< UART 5 reset */

#define RCU_APB1RST_CAN1RST_Pos             (26U)                             
#define RCU_APB1RST_CAN1RST_Msk             (0x1U << RCU_APB1RST_CAN1RST_Pos) /*!< 0x04000000 */
#define RCU_APB1RST_CAN1RST                 RCU_APB1RST_CAN1RST_Msk          /*!< CAN1 reset */

#define RCU_APB1RST_DACRST_Pos              (29U)                             
#define RCU_APB1RST_DACRST_Msk              (0x1U << RCU_APB1RST_DACRST_Pos) /*!< 0x20000000 */
#define RCU_APB1RST_DACRST                  RCU_APB1RST_DACRST_Msk           /*!< DAC interface reset */

/******************  Bit definition for RCU_AHBEN register  ******************/
#define RCU_AHBEN_DMA0EN_Pos                (0U)                              
#define RCU_AHBEN_DMA0EN_Msk                (0x1U << RCU_AHBEN_DMA0EN_Pos)   /*!< 0x00000001 */
#define RCU_AHBEN_DMA0EN                    RCU_AHBEN_DMA0EN_Msk             /*!< DMA0 clock enable */
#define RCU_AHBEN_SRAMSPEN_Pos                (2U)                              
#define RCU_AHBEN_SRAMSPEN_Msk                (0x1U << RCU_AHBEN_SRAMSPEN_Pos)   /*!< 0x00000004 */
#define RCU_AHBEN_SRAMSPEN                    RCU_AHBEN_SRAMSPEN_Msk             /*!< SRAM interface clock enable */
#define RCU_AHBEN_FMCSPEN_Pos               (4U)                              
#define RCU_AHBEN_FMCSPEN_Msk               (0x1U << RCU_AHBEN_FMCSPEN_Pos)  /*!< 0x00000010 */
#define RCU_AHBEN_FMCSPEN                   RCU_AHBEN_FMCSPEN_Msk            /*!< FLITF clock enable */
#define RCU_AHBEN_CRCEN_Pos                 (6U)                              
#define RCU_AHBEN_CRCEN_Msk                 (0x1U << RCU_AHBEN_CRCEN_Pos)    /*!< 0x00000040 */
#define RCU_AHBEN_CRCEN                     RCU_AHBEN_CRCEN_Msk              /*!< CRC clock enable */

#define RCU_AHBEN_DMA1EN_Pos                (1U)                              
#define RCU_AHBEN_DMA1EN_Msk                (0x1U << RCU_AHBEN_DMA1EN_Pos)   /*!< 0x00000002 */
#define RCU_AHBEN_DMA1EN                    RCU_AHBEN_DMA1EN_Msk             /*!< DMA1 clock enable */

#define RCU_AHBEN_EXMCEN_Pos               (8U)                             
#define RCU_AHBEN_EXMCEN_Msk               (0x1U << RCU_AHBEN_USBFSEN_Pos)  /*!< 0x00001000 */
#define RCU_AHBEN_EXMCEN                   RCU_AHBEN_USBFSEN_Msk            /*!< USB OTG FS clock enable */

#define RCU_AHBEN_USBFSEN_Pos               (12U)                             
#define RCU_AHBEN_USBFSEN_Msk               (0x1U << RCU_AHBEN_USBFSEN_Pos)  /*!< 0x00001000 */
#define RCU_AHBEN_USBFSEN                   RCU_AHBEN_USBFSEN_Msk            /*!< USB OTG FS clock enable */

/******************  Bit definition for RCU_APB2EN register  *****************/
#define RCU_APB2EN_AFEN_Pos               (0U)                              
#define RCU_APB2EN_AFEN_Msk               (0x1U << RCU_APB2EN_AFEN_Pos)  /*!< 0x00000001 */
#define RCU_APB2EN_AFEN                   RCU_APB2EN_AFEN_Msk            /*!< Alternate Function I/O clock enable */
#define RCU_APB2EN_PAEN_Pos               (2U)                              
#define RCU_APB2EN_PAEN_Msk               (0x1U << RCU_APB2EN_PAEN_Pos)  /*!< 0x00000004 */
#define RCU_APB2EN_PAEN                   RCU_APB2EN_PAEN_Msk            /*!< I/O port A clock enable */
#define RCU_APB2EN_PBEN_Pos               (3U)                              
#define RCU_APB2EN_PBEN_Msk               (0x1U << RCU_APB2EN_PBEN_Pos)  /*!< 0x00000008 */
#define RCU_APB2EN_PBEN                   RCU_APB2EN_PBEN_Msk            /*!< I/O port B clock enable */
#define RCU_APB2EN_PCEN_Pos               (4U)                              
#define RCU_APB2EN_PCEN_Msk               (0x1U << RCU_APB2EN_PCEN_Pos)  /*!< 0x00000010 */
#define RCU_APB2EN_PCEN                   RCU_APB2EN_PCEN_Msk            /*!< I/O port C clock enable */
#define RCU_APB2EN_PDEN_Pos               (5U)                              
#define RCU_APB2EN_PDEN_Msk               (0x1U << RCU_APB2EN_PDEN_Pos)  /*!< 0x00000020 */
#define RCU_APB2EN_PDEN                   RCU_APB2EN_PDEN_Msk            /*!< I/O port D clock enable */
#define RCU_APB2EN_ADC0EN_Pos               (9U)                              
#define RCU_APB2EN_ADC0EN_Msk               (0x1U << RCU_APB2EN_ADC0EN_Pos)  /*!< 0x00000200 */
#define RCU_APB2EN_ADC0EN                   RCU_APB2EN_ADC0EN_Msk            /*!< ADC 1 interface clock enable */

#define RCU_APB2EN_ADC1EN_Pos               (10U)                             
#define RCU_APB2EN_ADC1EN_Msk               (0x1U << RCU_APB2EN_ADC1EN_Pos)  /*!< 0x00000400 */
#define RCU_APB2EN_ADC1EN                   RCU_APB2EN_ADC1EN_Msk            /*!< ADC 2 interface clock enable */

#define RCU_APB2EN_TIMER0EN_Pos               (11U)                             
#define RCU_APB2EN_TIMER0EN_Msk               (0x1U << RCU_APB2EN_TIMER0EN_Pos)  /*!< 0x00000800 */
#define RCU_APB2EN_TIMER0EN                   RCU_APB2EN_TIMER0EN_Msk            /*!< TIM0 Timer clock enable */
#define RCU_APB2EN_SPI0EN_Pos               (12U)                             
#define RCU_APB2EN_SPI0EN_Msk               (0x1U << RCU_APB2EN_SPI0EN_Pos)  /*!< 0x00001000 */
#define RCU_APB2EN_SPI0EN                   RCU_APB2EN_SPI0EN_Msk            /*!< SPI 1 clock enable */
#define RCU_APB2EN_USART0EN_Pos             (14U)                             
#define RCU_APB2EN_USART0EN_Msk             (0x1U << RCU_APB2EN_USART0EN_Pos) /*!< 0x00004000 */
#define RCU_APB2EN_USART0EN                 RCU_APB2EN_USART0EN_Msk          /*!< USART0 clock enable */

#define RCU_APB2EN_PEEN_Pos               (6U)                              
#define RCU_APB2EN_PEEN_Msk               (0x1U << RCU_APB2EN_PEEN_Pos)  /*!< 0x00000040 */
#define RCU_APB2EN_PEEN                   RCU_APB2EN_PEEN_Msk            /*!< I/O port E clock enable */

/*****************  Bit definition for RCU_APB1EN register  ******************/
#define RCU_APB1EN_TIMER1EN_Pos               (0U)                              
#define RCU_APB1EN_TIMER1EN_Msk               (0x1U << RCU_APB1EN_TIMER1EN_Pos)  /*!< 0x00000001 */
#define RCU_APB1EN_TIMER1EN                   RCU_APB1EN_TIMER1EN_Msk            /*!< Timer 2 clock enabled*/
#define RCU_APB1EN_TIMER2EN_Pos               (1U)                              
#define RCU_APB1EN_TIMER2EN_Msk               (0x1U << RCU_APB1EN_TIMER2EN_Pos)  /*!< 0x00000002 */
#define RCU_APB1EN_TIMER2EN                   RCU_APB1EN_TIMER2EN_Msk            /*!< Timer 3 clock enable */
#define RCU_APB1EN_WWDGTEN_Pos               (11U)                             
#define RCU_APB1EN_WWDGTEN_Msk               (0x1U << RCU_APB1EN_WWDGTEN_Pos)  /*!< 0x00000800 */
#define RCU_APB1EN_WWDGTEN                   RCU_APB1EN_WWDGTEN_Msk            /*!< Window Watchdog clock enable */
#define RCU_APB1EN_USART1EN_Pos             (17U)                             
#define RCU_APB1EN_USART1EN_Msk             (0x1U << RCU_APB1EN_USART1EN_Pos) /*!< 0x00020000 */
#define RCU_APB1EN_USART1EN                 RCU_APB1EN_USART1EN_Msk          /*!< USART 2 clock enable */
#define RCU_APB1EN_I2C0EN_Pos               (21U)                             
#define RCU_APB1EN_I2C0EN_Msk               (0x1U << RCU_APB1EN_I2C0EN_Pos)  /*!< 0x00200000 */
#define RCU_APB1EN_I2C0EN                   RCU_APB1EN_I2C0EN_Msk            /*!< I2C 1 clock enable */

#define RCU_APB1EN_CAN0EN_Pos               (25U)                             
#define RCU_APB1EN_CAN0EN_Msk               (0x1U << RCU_APB1EN_CAN0EN_Pos)  /*!< 0x02000000 */
#define RCU_APB1EN_CAN0EN                   RCU_APB1EN_CAN0EN_Msk            /*!< CAN0 clock enable */

#define RCU_APB1EN_BKPIEN_Pos                (27U)                             
#define RCU_APB1EN_BKPIEN_Msk                (0x1U << RCU_APB1EN_BKPIEN_Pos)   /*!< 0x08000000 */
#define RCU_APB1EN_BKPIEN                    RCU_APB1EN_BKPIEN_Msk             /*!< Backup interface clock enable */
#define RCU_APB1EN_PMUEN_Pos                (28U)                             
#define RCU_APB1EN_PMUEN_Msk                (0x1U << RCU_APB1EN_PMUEN_Pos)   /*!< 0x10000000 */
#define RCU_APB1EN_PMUEN                    RCU_APB1EN_PMUEN_Msk             /*!< Power interface clock enable */

#define RCU_APB1EN_TIMER3EN_Pos               (2U)                              
#define RCU_APB1EN_TIMER3EN_Msk               (0x1U << RCU_APB1EN_TIMER3EN_Pos)  /*!< 0x00000004 */
#define RCU_APB1EN_TIMER3EN                   RCU_APB1EN_TIMER3EN_Msk            /*!< Timer 4 clock enable */
#define RCU_APB1EN_SPI1EN_Pos               (14U)                             
#define RCU_APB1EN_SPI1EN_Msk               (0x1U << RCU_APB1EN_SPI1EN_Pos)  /*!< 0x00004000 */
#define RCU_APB1EN_SPI1EN                   RCU_APB1EN_SPI1EN_Msk            /*!< SPI 2 clock enable */
#define RCU_APB1EN_USART2EN_Pos             (18U)                             
#define RCU_APB1EN_USART2EN_Msk             (0x1U << RCU_APB1EN_USART2EN_Pos) /*!< 0x00040000 */
#define RCU_APB1EN_USART2EN                 RCU_APB1EN_USART2EN_Msk          /*!< USART 3 clock enable */
#define RCU_APB1EN_I2C1EN_Pos               (22U)                             
#define RCU_APB1EN_I2C1EN_Msk               (0x1U << RCU_APB1EN_I2C1EN_Pos)  /*!< 0x00400000 */
#define RCU_APB1EN_I2C1EN                   RCU_APB1EN_I2C1EN_Msk            /*!< I2C 2 clock enable */


#define RCU_APB1EN_TIMER4EN_Pos               (3U)                              
#define RCU_APB1EN_TIMER4EN_Msk               (0x1U << RCU_APB1EN_TIMER4EN_Pos)  /*!< 0x00000008 */
#define RCU_APB1EN_TIMER4EN                   RCU_APB1EN_TIMER4EN_Msk            /*!< Timer 5 clock enable */
#define RCU_APB1EN_TIMER5EN_Pos               (4U)                              
#define RCU_APB1EN_TIMER5EN_Msk               (0x1U << RCU_APB1EN_TIMER5EN_Pos)  /*!< 0x00000010 */
#define RCU_APB1EN_TIMER5EN                   RCU_APB1EN_TIMER5EN_Msk            /*!< Timer 6 clock enable */
#define RCU_APB1EN_TIMER6EN_Pos               (5U)                              
#define RCU_APB1EN_TIMER6EN_Msk               (0x1U << RCU_APB1EN_TIMER6EN_Pos)  /*!< 0x00000020 */
#define RCU_APB1EN_TIMER6EN                   RCU_APB1EN_TIMER6EN_Msk            /*!< Timer 7 clock enable */
#define RCU_APB1EN_SPI2EN_Pos               (15U)                             
#define RCU_APB1EN_SPI2EN_Msk               (0x1U << RCU_APB1EN_SPI2EN_Pos)  /*!< 0x00008000 */
#define RCU_APB1EN_SPI2EN                   RCU_APB1EN_SPI2EN_Msk            /*!< SPI 3 clock enable */
#define RCU_APB1EN_UART3EN_Pos              (19U)                             
#define RCU_APB1EN_UART3EN_Msk              (0x1U << RCU_APB1EN_UART3EN_Pos) /*!< 0x00080000 */
#define RCU_APB1EN_UART3EN                  RCU_APB1EN_UART3EN_Msk           /*!< UART 4 clock enable */
#define RCU_APB1EN_UART4EN_Pos              (20U)                             
#define RCU_APB1EN_UART4EN_Msk              (0x1U << RCU_APB1EN_UART4EN_Pos) /*!< 0x00100000 */
#define RCU_APB1EN_UART4EN                  RCU_APB1EN_UART4EN_Msk           /*!< UART 5 clock enable */



#define RCU_APB1EN_CAN1EN_Pos               (26U)                             
#define RCU_APB1EN_CAN1EN_Msk               (0x1U << RCU_APB1EN_CAN1EN_Pos)  /*!< 0x04000000 */
#define RCU_APB1EN_CAN1EN                   RCU_APB1EN_CAN1EN_Msk            /*!< CAN1 clock enable */

#define RCU_APB1EN_DACEN_Pos                (29U)                             
#define RCU_APB1EN_DACEN_Msk                (0x1U << RCU_APB1EN_DACEN_Pos)   /*!< 0x20000000 */
#define RCU_APB1EN_DACEN                    RCU_APB1EN_DACEN_Msk             /*!< DAC interface clock enable */

/*******************  Bit definition for RCU_BDCTL register  *******************/
#define RCU_BDCTL_LXTALEN_Pos                   (0U)                              
#define RCU_BDCTL_LXTALEN_Msk                   (0x1U << RCU_BDCTL_LXTALEN_Pos)      /*!< 0x00000001 */
#define RCU_BDCTL_LXTALEN                       RCU_BDCTL_LXTALEN_Msk                /*!< External Low Speed oscillator enable */
#define RCU_BDCTL_LXTALSTB_Pos                  (1U)                              
#define RCU_BDCTL_LXTALSTB_Msk                  (0x1U << RCU_BDCTL_LXTALSTB_Pos)     /*!< 0x00000002 */
#define RCU_BDCTL_LXTALSTB                      RCU_BDCTL_LXTALSTB_Msk               /*!< External Low Speed oscillator Ready */
#define RCU_BDCTL_LXTALBPS_Pos                  (2U)                              
#define RCU_BDCTL_LXTALBPS_Msk                  (0x1U << RCU_BDCTL_LXTALBPS_Pos)     /*!< 0x00000004 */
#define RCU_BDCTL_LXTALBPS                      RCU_BDCTL_LXTALBPS_Msk               /*!< External Low Speed oscillator Bypass */

#define RCU_BDCTL_RTCSRC_Pos                  (8U)                              
#define RCU_BDCTL_RTCSRC_Msk                  (0x3U << RCU_BDCTL_RTCSRC_Pos)     /*!< 0x00000300 */
#define RCU_BDCTL_RTCSRC                      RCU_BDCTL_RTCSRC_Msk               /*!< RTCSEL[1:0] bits (RTC clock source selection) */
#define RCU_BDCTL_RTCSRC_0                    (0x1U << RCU_BDCTL_RTCSRC_Pos)     /*!< 0x00000100 */
#define RCU_BDCTL_RTCSRC_1                    (0x2U << RCU_BDCTL_RTCSRC_Pos)     /*!< 0x00000200 */

/*!< RTC congiguration */
#define RCU_BDCTL_RTCSRC_NOCLOCK              0x00000000U                       /*!< No clock */
#define RCU_BDCTL_RTCSRC_LXTAL                  0x00000100U                       /*!< LXTAL oscillator clock used as RTC clock */
#define RCU_BDCTL_RTCSRC_IRC40K                  0x00000200U                       /*!< IRC40K oscillator clock used as RTC clock */
#define RCU_BDCTL_RTCSRC_HXTAL                  0x00000300U                       /*!< HXTAL oscillator clock divided by 128 used as RTC clock */

#define RCU_BDCTL_RTCEN_Pos                   (15U)                             
#define RCU_BDCTL_RTCEN_Msk                   (0x1U << RCU_BDCTL_RTCEN_Pos)      /*!< 0x00008000 */
#define RCU_BDCTL_RTCEN                       RCU_BDCTL_RTCEN_Msk                /*!< RTC clock enable */
#define RCU_BDCTL_BKPRST_Pos                   (16U)                             
#define RCU_BDCTL_BKPRST_Msk                   (0x1U << RCU_BDCTL_BKPRST_Pos)      /*!< 0x00010000 */
#define RCU_BDCTL_BKPRST                       RCU_BDCTL_BKPRST_Msk                /*!< Backup domain software reset  */

/*******************  Bit definition for RCU_RSTSCK register  ********************/  
#define RCU_RSTSCK_IRC40KEN_Pos                    (0U)                              
#define RCU_RSTSCK_IRC40KEN_Msk                    (0x1U << RCU_RSTSCK_IRC40KEN_Pos)       /*!< 0x00000001 */
#define RCU_RSTSCK_IRC40KEN                        RCU_RSTSCK_IRC40KEN_Msk                 /*!< Internal Low Speed oscillator enable */
#define RCU_RSTSCK_IRC40KSTB_Pos                   (1U)                              
#define RCU_RSTSCK_IRC40KSTB_Msk                   (0x1U << RCU_RSTSCK_IRC40KSTB_Pos)      /*!< 0x00000002 */
#define RCU_RSTSCK_IRC40KSTB                       RCU_RSTSCK_IRC40KSTB_Msk                /*!< Internal Low Speed oscillator Ready */
#define RCU_RSTSCK_RSTCF_Pos                     (24U)                             
#define RCU_RSTSCK_RSTCF_Msk                     (0x1U << RCU_RSTSCK_RSTCF_Pos)        /*!< 0x01000000 */
#define RCU_RSTSCK_RSTCF                         RCU_RSTSCK_RSTCF_Msk                  /*!< Remove reset flag */
#define RCU_RSTSCK_EPRSTF_Pos                  (26U)                             
#define RCU_RSTSCK_EPRSTF_Msk                  (0x1U << RCU_RSTSCK_EPRSTF_Pos)     /*!< 0x04000000 */
#define RCU_RSTSCK_EPRSTF                      RCU_RSTSCK_EPRSTF_Msk               /*!< PIN reset flag */
#define RCU_RSTSCK_PORRSTF_Pos                  (27U)                             
#define RCU_RSTSCK_PORRSTF_Msk                  (0x1U << RCU_RSTSCK_PORRSTF_Pos)     /*!< 0x08000000 */
#define RCU_RSTSCK_PORRSTF                      RCU_RSTSCK_PORRSTF_Msk               /*!< POR/PDR reset flag */
#define RCU_RSTSCK_SWRSTF_Pos                  (28U)                             
#define RCU_RSTSCK_SWRSTF_Msk                  (0x1U << RCU_RSTSCK_SWRSTF_Pos)     /*!< 0x10000000 */
#define RCU_RSTSCK_SWRSTF                      RCU_RSTSCK_SWRSTF_Msk               /*!< Software Reset flag */
#define RCU_RSTSCK_FWDGTRSTF_Pos                 (29U)                             
#define RCU_RSTSCK_FWDGTRSTF_Msk                 (0x1U << RCU_RSTSCK_FWDGTRSTF_Pos)    /*!< 0x20000000 */
#define RCU_RSTSCK_FWDGTRSTF                     RCU_RSTSCK_FWDGTRSTF_Msk              /*!< Independent Watchdog reset flag */
#define RCU_RSTSCK_WWDGTRSTF_Pos                 (30U)                             
#define RCU_RSTSCK_WWDGTRSTF_Msk                 (0x1U << RCU_RSTSCK_WWDGTRSTF_Pos)    /*!< 0x40000000 */
#define RCU_RSTSCK_WWDGTRSTF                     RCU_RSTSCK_WWDGTRSTF_Msk              /*!< Window watchdog reset flag */
#define RCU_RSTSCK_LPRSTF_Pos                 (31U)                             
#define RCU_RSTSCK_LPRSTF_Msk                 (0x1U << RCU_RSTSCK_LPRSTF_Pos)    /*!< 0x80000000 */
#define RCU_RSTSCK_LPRSTF                     RCU_RSTSCK_LPRSTF_Msk              /*!< Low-Power reset flag */

/*******************  Bit definition for RCU_AHBRST register  ****************/
#define RCU_AHBRST_USBFSRST_Pos             (12U)                             
#define RCU_AHBRST_USBFSRST_Msk             (0x1U << RCU_AHBRST_USBFSRST_Pos) /*!< 0x00001000 */
#define RCU_AHBRST_USBFSRST                 RCU_AHBRST_USBFSRST_Msk          /*!< USB OTG FS reset */

/******************************************************************************/
/*                                                                            */
/*                General Purpose and Alternate Function I/O                  */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for GPIO_CTL0 register  *******************/
#define GPIO_CTL0_MODE_Pos                    (0U)                              
#define GPIO_CTL0_MODE_Msk                    (0x33333333U << GPIO_CTL0_MODE_Pos) /*!< 0x33333333 */
#define GPIO_CTL0_MODE                        GPIO_CTL0_MODE_Msk                 /*!< Port x mode bits */

#define GPIO_CTL0_MD0_Pos                   (0U)                              
#define GPIO_CTL0_MD0_Msk                   (0x3U << GPIO_CTL0_MD0_Pos)      /*!< 0x00000003 */
#define GPIO_CTL0_MD0                       GPIO_CTL0_MD0_Msk                /*!< MODE0[1:0] bits (Port x mode bits, pin 0) */
#define GPIO_CTL0_MD0_0                     (0x1U << GPIO_CTL0_MD0_Pos)      /*!< 0x00000001 */
#define GPIO_CTL0_MD0_1                     (0x2U << GPIO_CTL0_MD0_Pos)      /*!< 0x00000002 */

#define GPIO_CTL0_MD1_Pos                   (4U)                              
#define GPIO_CTL0_MD1_Msk                   (0x3U << GPIO_CTL0_MD1_Pos)      /*!< 0x00000030 */
#define GPIO_CTL0_MD1                       GPIO_CTL0_MD1_Msk                /*!< MODE1[1:0] bits (Port x mode bits, pin 1) */
#define GPIO_CTL0_MD1_0                     (0x1U << GPIO_CTL0_MD1_Pos)      /*!< 0x00000010 */
#define GPIO_CTL0_MD1_1                     (0x2U << GPIO_CTL0_MD1_Pos)      /*!< 0x00000020 */

#define GPIO_CTL0_MD2_Pos                   (8U)                              
#define GPIO_CTL0_MD2_Msk                   (0x3U << GPIO_CTL0_MD2_Pos)      /*!< 0x00000300 */
#define GPIO_CTL0_MD2                       GPIO_CTL0_MD2_Msk                /*!< MODE2[1:0] bits (Port x mode bits, pin 2) */
#define GPIO_CTL0_MD2_0                     (0x1U << GPIO_CTL0_MD2_Pos)      /*!< 0x00000100 */
#define GPIO_CTL0_MD2_1                     (0x2U << GPIO_CTL0_MD2_Pos)      /*!< 0x00000200 */

#define GPIO_CTL0_MD3_Pos                   (12U)                             
#define GPIO_CTL0_MD3_Msk                   (0x3U << GPIO_CTL0_MD3_Pos)      /*!< 0x00003000 */
#define GPIO_CTL0_MD3                       GPIO_CTL0_MD3_Msk                /*!< MODE3[1:0] bits (Port x mode bits, pin 3) */
#define GPIO_CTL0_MD3_0                     (0x1U << GPIO_CTL0_MD3_Pos)      /*!< 0x00001000 */
#define GPIO_CTL0_MD3_1                     (0x2U << GPIO_CTL0_MD3_Pos)      /*!< 0x00002000 */

#define GPIO_CTL0_MD4_Pos                   (16U)                             
#define GPIO_CTL0_MD4_Msk                   (0x3U << GPIO_CTL0_MD4_Pos)      /*!< 0x00030000 */
#define GPIO_CTL0_MD4                       GPIO_CTL0_MD4_Msk                /*!< MODE4[1:0] bits (Port x mode bits, pin 4) */
#define GPIO_CTL0_MD4_0                     (0x1U << GPIO_CTL0_MD4_Pos)      /*!< 0x00010000 */
#define GPIO_CTL0_MD4_1                     (0x2U << GPIO_CTL0_MD4_Pos)      /*!< 0x00020000 */

#define GPIO_CTL0_MD5_Pos                   (20U)                             
#define GPIO_CTL0_MD5_Msk                   (0x3U << GPIO_CTL0_MD5_Pos)      /*!< 0x00300000 */
#define GPIO_CTL0_MD5                       GPIO_CTL0_MD5_Msk                /*!< MODE5[1:0] bits (Port x mode bits, pin 5) */
#define GPIO_CTL0_MD5_0                     (0x1U << GPIO_CTL0_MD5_Pos)      /*!< 0x00100000 */
#define GPIO_CTL0_MD5_1                     (0x2U << GPIO_CTL0_MD5_Pos)      /*!< 0x00200000 */

#define GPIO_CTL0_MD6_Pos                   (24U)                             
#define GPIO_CTL0_MD6_Msk                   (0x3U << GPIO_CTL0_MD6_Pos)      /*!< 0x03000000 */
#define GPIO_CTL0_MD6                       GPIO_CTL0_MD6_Msk                /*!< MODE6[1:0] bits (Port x mode bits, pin 6) */
#define GPIO_CTL0_MD6_0                     (0x1U << GPIO_CTL0_MD6_Pos)      /*!< 0x01000000 */
#define GPIO_CTL0_MD6_1                     (0x2U << GPIO_CTL0_MD6_Pos)      /*!< 0x02000000 */

#define GPIO_CTL0_MD7_Pos                   (28U)                             
#define GPIO_CTL0_MD7_Msk                   (0x3U << GPIO_CTL0_MD7_Pos)      /*!< 0x30000000 */
#define GPIO_CTL0_MD7                       GPIO_CTL0_MD7_Msk                /*!< MODE7[1:0] bits (Port x mode bits, pin 7) */
#define GPIO_CTL0_MD7_0                     (0x1U << GPIO_CTL0_MD7_Pos)      /*!< 0x10000000 */
#define GPIO_CTL0_MD7_1                     (0x2U << GPIO_CTL0_MD7_Pos)      /*!< 0x20000000 */

#define GPIO_CTL0_CTL_Pos                     (2U)                              
#define GPIO_CTL0_CTL_Msk                     (0x33333333U << GPIO_CTL0_CTL_Pos) /*!< 0xCCCCCCCC */
#define GPIO_CTL0_CTL                         GPIO_CTL0_CTL_Msk                  /*!< Port x configuration bits */

#define GPIO_CTL0_CTL0_Pos                    (2U)                              
#define GPIO_CTL0_CTL0_Msk                    (0x3U << GPIO_CTL0_CTL0_Pos)       /*!< 0x0000000C */
#define GPIO_CTL0_CTL0                        GPIO_CTL0_CTL0_Msk                 /*!< CNF0[1:0] bits (Port x configuration bits, pin 0) */
#define GPIO_CTL0_CTL0_0                      (0x1U << GPIO_CTL0_CTL0_Pos)       /*!< 0x00000004 */
#define GPIO_CTL0_CTL0_1                      (0x2U << GPIO_CTL0_CTL0_Pos)       /*!< 0x00000008 */

#define GPIO_CTL0_CTL1_Pos                    (6U)                              
#define GPIO_CTL0_CTL1_Msk                    (0x3U << GPIO_CTL0_CTL1_Pos)       /*!< 0x000000C0 */
#define GPIO_CTL0_CTL1                        GPIO_CTL0_CTL1_Msk                 /*!< CNF1[1:0] bits (Port x configuration bits, pin 1) */
#define GPIO_CTL0_CTL1_0                      (0x1U << GPIO_CTL0_CTL1_Pos)       /*!< 0x00000040 */
#define GPIO_CTL0_CTL1_1                      (0x2U << GPIO_CTL0_CTL1_Pos)       /*!< 0x00000080 */

#define GPIO_CTL0_CTL2_Pos                    (10U)                             
#define GPIO_CTL0_CTL2_Msk                    (0x3U << GPIO_CTL0_CTL2_Pos)       /*!< 0x00000C00 */
#define GPIO_CTL0_CTL2                        GPIO_CTL0_CTL2_Msk                 /*!< CNF2[1:0] bits (Port x configuration bits, pin 2) */
#define GPIO_CTL0_CTL2_0                      (0x1U << GPIO_CTL0_CTL2_Pos)       /*!< 0x00000400 */
#define GPIO_CTL0_CTL2_1                      (0x2U << GPIO_CTL0_CTL2_Pos)       /*!< 0x00000800 */

#define GPIO_CTL0_CTL3_Pos                    (14U)                             
#define GPIO_CTL0_CTL3_Msk                    (0x3U << GPIO_CTL0_CTL3_Pos)       /*!< 0x0000C000 */
#define GPIO_CTL0_CTL3                        GPIO_CTL0_CTL3_Msk                 /*!< CNF3[1:0] bits (Port x configuration bits, pin 3) */
#define GPIO_CTL0_CTL3_0                      (0x1U << GPIO_CTL0_CTL3_Pos)       /*!< 0x00004000 */
#define GPIO_CTL0_CTL3_1                      (0x2U << GPIO_CTL0_CTL3_Pos)       /*!< 0x00008000 */

#define GPIO_CTL0_CTL4_Pos                    (18U)                             
#define GPIO_CTL0_CTL4_Msk                    (0x3U << GPIO_CTL0_CTL4_Pos)       /*!< 0x000C0000 */
#define GPIO_CTL0_CTL4                        GPIO_CTL0_CTL4_Msk                 /*!< CNF4[1:0] bits (Port x configuration bits, pin 4) */
#define GPIO_CTL0_CTL4_0                      (0x1U << GPIO_CTL0_CTL4_Pos)       /*!< 0x00040000 */
#define GPIO_CTL0_CTL4_1                      (0x2U << GPIO_CTL0_CTL4_Pos)       /*!< 0x00080000 */

#define GPIO_CTL0_CTL5_Pos                    (22U)                             
#define GPIO_CTL0_CTL5_Msk                    (0x3U << GPIO_CTL0_CTL5_Pos)       /*!< 0x00C00000 */
#define GPIO_CTL0_CTL5                        GPIO_CTL0_CTL5_Msk                 /*!< CNF5[1:0] bits (Port x configuration bits, pin 5) */
#define GPIO_CTL0_CTL5_0                      (0x1U << GPIO_CTL0_CTL5_Pos)       /*!< 0x00400000 */
#define GPIO_CTL0_CTL5_1                      (0x2U << GPIO_CTL0_CTL5_Pos)       /*!< 0x00800000 */

#define GPIO_CTL0_CTL6_Pos                    (26U)                             
#define GPIO_CTL0_CTL6_Msk                    (0x3U << GPIO_CTL0_CTL6_Pos)       /*!< 0x0C000000 */
#define GPIO_CTL0_CTL6                        GPIO_CTL0_CTL6_Msk                 /*!< CNF6[1:0] bits (Port x configuration bits, pin 6) */
#define GPIO_CTL0_CTL6_0                      (0x1U << GPIO_CTL0_CTL6_Pos)       /*!< 0x04000000 */
#define GPIO_CTL0_CTL6_1                      (0x2U << GPIO_CTL0_CTL6_Pos)       /*!< 0x08000000 */

#define GPIO_CTL0_CTL7_Pos                    (30U)                             
#define GPIO_CTL0_CTL7_Msk                    (0x3U << GPIO_CTL0_CTL7_Pos)       /*!< 0xC0000000 */
#define GPIO_CTL0_CTL7                        GPIO_CTL0_CTL7_Msk                 /*!< CNF7[1:0] bits (Port x configuration bits, pin 7) */
#define GPIO_CTL0_CTL7_0                      (0x1U << GPIO_CTL0_CTL7_Pos)       /*!< 0x40000000 */
#define GPIO_CTL0_CTL7_1                      (0x2U << GPIO_CTL0_CTL7_Pos)       /*!< 0x80000000 */

/*******************  Bit definition for GPIO_CTL1 register  *******************/
#define GPIO_CTL1_MD_Pos                    (0U)                              
#define GPIO_CTL1_MD_Msk                    (0x33333333U << GPIO_CTL1_MD_Pos) /*!< 0x33333333 */
#define GPIO_CTL1_MD                        GPIO_CTL1_MD_Msk                 /*!< Port x mode bits */

#define GPIO_CTL1_MD8_Pos                   (0U)                              
#define GPIO_CTL1_MD8_Msk                   (0x3U << GPIO_CTL1_MD8_Pos)      /*!< 0x00000003 */
#define GPIO_CTL1_MD8                       GPIO_CTL1_MD8_Msk                /*!< MODE8[1:0] bits (Port x mode bits, pin 8) */
#define GPIO_CTL1_MD8_0                     (0x1U << GPIO_CTL1_MD8_Pos)      /*!< 0x00000001 */
#define GPIO_CTL1_MD8_1                     (0x2U << GPIO_CTL1_MD8_Pos)      /*!< 0x00000002 */

#define GPIO_CTL1_MD9_Pos                   (4U)                              
#define GPIO_CTL1_MD9_Msk                   (0x3U << GPIO_CTL1_MD9_Pos)      /*!< 0x00000030 */
#define GPIO_CTL1_MD9                       GPIO_CTL1_MD9_Msk                /*!< MODE9[1:0] bits (Port x mode bits, pin 9) */
#define GPIO_CTL1_MD9_0                     (0x1U << GPIO_CTL1_MD9_Pos)      /*!< 0x00000010 */
#define GPIO_CTL1_MD9_1                     (0x2U << GPIO_CTL1_MD9_Pos)      /*!< 0x00000020 */

#define GPIO_CTL1_MD10_Pos                  (8U)                              
#define GPIO_CTL1_MD10_Msk                  (0x3U << GPIO_CTL1_MD10_Pos)     /*!< 0x00000300 */
#define GPIO_CTL1_MD10                      GPIO_CTL1_MD10_Msk               /*!< MODE10[1:0] bits (Port x mode bits, pin 10) */
#define GPIO_CTL1_MD10_0                    (0x1U << GPIO_CTL1_MD10_Pos)     /*!< 0x00000100 */
#define GPIO_CTL1_MD10_1                    (0x2U << GPIO_CTL1_MD10_Pos)     /*!< 0x00000200 */

#define GPIO_CTL1_MD11_Pos                  (12U)                             
#define GPIO_CTL1_MD11_Msk                  (0x3U << GPIO_CTL1_MD11_Pos)     /*!< 0x00003000 */
#define GPIO_CTL1_MD11                      GPIO_CTL1_MD11_Msk               /*!< MODE11[1:0] bits (Port x mode bits, pin 11) */
#define GPIO_CTL1_MD11_0                    (0x1U << GPIO_CTL1_MD11_Pos)     /*!< 0x00001000 */
#define GPIO_CTL1_MD11_1                    (0x2U << GPIO_CTL1_MD11_Pos)     /*!< 0x00002000 */

#define GPIO_CTL1_MD12_Pos                  (16U)                             
#define GPIO_CTL1_MD12_Msk                  (0x3U << GPIO_CTL1_MD12_Pos)     /*!< 0x00030000 */
#define GPIO_CTL1_MD12                      GPIO_CTL1_MD12_Msk               /*!< MODE12[1:0] bits (Port x mode bits, pin 12) */
#define GPIO_CTL1_MD12_0                    (0x1U << GPIO_CTL1_MD12_Pos)     /*!< 0x00010000 */
#define GPIO_CTL1_MD12_1                    (0x2U << GPIO_CTL1_MD12_Pos)     /*!< 0x00020000 */

#define GPIO_CTL1_MD13_Pos                  (20U)                             
#define GPIO_CTL1_MD13_Msk                  (0x3U << GPIO_CTL1_MD13_Pos)     /*!< 0x00300000 */
#define GPIO_CTL1_MD13                      GPIO_CTL1_MD13_Msk               /*!< MODE13[1:0] bits (Port x mode bits, pin 13) */
#define GPIO_CTL1_MD13_0                    (0x1U << GPIO_CTL1_MD13_Pos)     /*!< 0x00100000 */
#define GPIO_CTL1_MD13_1                    (0x2U << GPIO_CTL1_MD13_Pos)     /*!< 0x00200000 */

#define GPIO_CTL1_MD14_Pos                  (24U)                             
#define GPIO_CTL1_MD14_Msk                  (0x3U << GPIO_CTL1_MD14_Pos)     /*!< 0x03000000 */
#define GPIO_CTL1_MD14                      GPIO_CTL1_MD14_Msk               /*!< MODE14[1:0] bits (Port x mode bits, pin 14) */
#define GPIO_CTL1_MD14_0                    (0x1U << GPIO_CTL1_MD14_Pos)     /*!< 0x01000000 */
#define GPIO_CTL1_MD14_1                    (0x2U << GPIO_CTL1_MD14_Pos)     /*!< 0x02000000 */

#define GPIO_CTL1_MD15_Pos                  (28U)                             
#define GPIO_CTL1_MD15_Msk                  (0x3U << GPIO_CTL1_MD15_Pos)     /*!< 0x30000000 */
#define GPIO_CTL1_MD15                      GPIO_CTL1_MD15_Msk               /*!< MODE15[1:0] bits (Port x mode bits, pin 15) */
#define GPIO_CTL1_MD15_0                    (0x1U << GPIO_CTL1_MD15_Pos)     /*!< 0x10000000 */
#define GPIO_CTL1_MD15_1                    (0x2U << GPIO_CTL1_MD15_Pos)     /*!< 0x20000000 */

#define GPIO_CTL1_CNF_Pos                     (2U)                              
#define GPIO_CTL1_CNF_Msk                     (0x33333333U << GPIO_CTL1_CNF_Pos) /*!< 0xCCCCCCCC */
#define GPIO_CTL1_CNF                         GPIO_CTL1_CNF_Msk                  /*!< Port x configuration bits */

#define GPIO_CTL1_CTL8_Pos                    (2U)                              
#define GPIO_CTL1_CTL8_Msk                    (0x3U << GPIO_CTL1_CTL8_Pos)       /*!< 0x0000000C */
#define GPIO_CTL1_CTL8                        GPIO_CTL1_CTL8_Msk                 /*!< CNF8[1:0] bits (Port x configuration bits, pin 8) */
#define GPIO_CTL1_CTL8_0                      (0x1U << GPIO_CTL1_CTL8_Pos)       /*!< 0x00000004 */
#define GPIO_CTL1_CTL8_1                      (0x2U << GPIO_CTL1_CTL8_Pos)       /*!< 0x00000008 */

#define GPIO_CTL1_CTL9_Pos                    (6U)                              
#define GPIO_CTL1_CTL9_Msk                    (0x3U << GPIO_CTL1_CTL9_Pos)       /*!< 0x000000C0 */
#define GPIO_CTL1_CTL9                        GPIO_CTL1_CTL9_Msk                 /*!< CNF9[1:0] bits (Port x configuration bits, pin 9) */
#define GPIO_CTL1_CTL9_0                      (0x1U << GPIO_CTL1_CTL9_Pos)       /*!< 0x00000040 */
#define GPIO_CTL1_CTL9_1                      (0x2U << GPIO_CTL1_CTL9_Pos)       /*!< 0x00000080 */

#define GPIO_CTL1_CTL10_Pos                   (10U)                             
#define GPIO_CTL1_CTL10_Msk                   (0x3U << GPIO_CTL1_CTL10_Pos)      /*!< 0x00000C00 */
#define GPIO_CTL1_CTL10                       GPIO_CTL1_CTL10_Msk                /*!< CNF10[1:0] bits (Port x configuration bits, pin 10) */
#define GPIO_CTL1_CTL10_0                     (0x1U << GPIO_CTL1_CTL10_Pos)      /*!< 0x00000400 */
#define GPIO_CTL1_CTL10_1                     (0x2U << GPIO_CTL1_CTL10_Pos)      /*!< 0x00000800 */

#define GPIO_CTL1_CTL11_Pos                   (14U)                             
#define GPIO_CTL1_CTL11_Msk                   (0x3U << GPIO_CTL1_CTL11_Pos)      /*!< 0x0000C000 */
#define GPIO_CTL1_CTL11                       GPIO_CTL1_CTL11_Msk                /*!< CNF11[1:0] bits (Port x configuration bits, pin 11) */
#define GPIO_CTL1_CTL11_0                     (0x1U << GPIO_CTL1_CTL11_Pos)      /*!< 0x00004000 */
#define GPIO_CTL1_CTL11_1                     (0x2U << GPIO_CTL1_CTL11_Pos)      /*!< 0x00008000 */

#define GPIO_CTL1_CTL12_Pos                   (18U)                             
#define GPIO_CTL1_CTL12_Msk                   (0x3U << GPIO_CTL1_CTL12_Pos)      /*!< 0x000C0000 */
#define GPIO_CTL1_CTL12                       GPIO_CTL1_CTL12_Msk                /*!< CNF12[1:0] bits (Port x configuration bits, pin 12) */
#define GPIO_CTL1_CTL12_0                     (0x1U << GPIO_CTL1_CTL12_Pos)      /*!< 0x00040000 */
#define GPIO_CTL1_CTL12_1                     (0x2U << GPIO_CTL1_CTL12_Pos)      /*!< 0x00080000 */

#define GPIO_CTL1_CTL13_Pos                   (22U)                             
#define GPIO_CTL1_CTL13_Msk                   (0x3U << GPIO_CTL1_CTL13_Pos)      /*!< 0x00C00000 */
#define GPIO_CTL1_CTL13                       GPIO_CTL1_CTL13_Msk                /*!< CNF13[1:0] bits (Port x configuration bits, pin 13) */
#define GPIO_CTL1_CTL13_0                     (0x1U << GPIO_CTL1_CTL13_Pos)      /*!< 0x00400000 */
#define GPIO_CTL1_CTL13_1                     (0x2U << GPIO_CTL1_CTL13_Pos)      /*!< 0x00800000 */

#define GPIO_CTL1_CTL14_Pos                   (26U)                             
#define GPIO_CTL1_CTL14_Msk                   (0x3U << GPIO_CTL1_CTL14_Pos)      /*!< 0x0C000000 */
#define GPIO_CTL1_CTL14                       GPIO_CTL1_CTL14_Msk                /*!< CNF14[1:0] bits (Port x configuration bits, pin 14) */
#define GPIO_CTL1_CTL14_0                     (0x1U << GPIO_CTL1_CTL14_Pos)      /*!< 0x04000000 */
#define GPIO_CTL1_CTL14_1                     (0x2U << GPIO_CTL1_CTL14_Pos)      /*!< 0x08000000 */

#define GPIO_CTL1_CTL15_Pos                   (30U)                             
#define GPIO_CTL1_CTL15_Msk                   (0x3U << GPIO_CTL1_CTL15_Pos)      /*!< 0xC0000000 */
#define GPIO_CTL1_CTL15                       GPIO_CTL1_CTL15_Msk                /*!< CNF15[1:0] bits (Port x configuration bits, pin 15) */
#define GPIO_CTL1_CTL15_0                     (0x1U << GPIO_CTL1_CTL15_Pos)      /*!< 0x40000000 */
#define GPIO_CTL1_CTL15_1                     (0x2U << GPIO_CTL1_CTL15_Pos)      /*!< 0x80000000 */

/*!<******************  Bit definition for GPIO_IDR register  *******************/
#define GPIO_ISTAT_ISTAT0_Pos                    (0U)                              
#define GPIO_ISTAT_ISTAT0_Msk                    (0x1U << GPIO_ISTAT_ISTAT0_Pos)       /*!< 0x00000001 */
#define GPIO_ISTAT_ISTAT0                        GPIO_ISTAT_ISTAT0_Msk                 /*!< Port input data, bit 0 */
#define GPIO_ISTAT_ISTAT1_Pos                    (1U)                              
#define GPIO_ISTAT_ISTAT1_Msk                    (0x1U << GPIO_ISTAT_ISTAT1_Pos)       /*!< 0x00000002 */
#define GPIO_ISTAT_ISTAT1                        GPIO_ISTAT_ISTAT1_Msk                 /*!< Port input data, bit 1 */
#define GPIO_ISTAT_ISTAT2_Pos                    (2U)                              
#define GPIO_ISTAT_ISTAT2_Msk                    (0x1U << GPIO_ISTAT_ISTAT2_Pos)       /*!< 0x00000004 */
#define GPIO_ISTAT_ISTAT2                        GPIO_ISTAT_ISTAT2_Msk                 /*!< Port input data, bit 2 */
#define GPIO_ISTAT_ISTAT3_Pos                    (3U)                              
#define GPIO_ISTAT_ISTAT3_Msk                    (0x1U << GPIO_ISTAT_ISTAT3_Pos)       /*!< 0x00000008 */
#define GPIO_ISTAT_ISTAT3                        GPIO_ISTAT_ISTAT3_Msk                 /*!< Port input data, bit 3 */
#define GPIO_ISTAT_ISTAT4_Pos                    (4U)                              
#define GPIO_ISTAT_ISTAT4_Msk                    (0x1U << GPIO_ISTAT_ISTAT4_Pos)       /*!< 0x00000010 */
#define GPIO_ISTAT_ISTAT4                        GPIO_ISTAT_ISTAT4_Msk                 /*!< Port input data, bit 4 */
#define GPIO_ISTAT_ISTAT5_Pos                    (5U)                              
#define GPIO_ISTAT_ISTAT5_Msk                    (0x1U << GPIO_ISTAT_ISTAT5_Pos)       /*!< 0x00000020 */
#define GPIO_ISTAT_ISTAT5                        GPIO_ISTAT_ISTAT5_Msk                 /*!< Port input data, bit 5 */
#define GPIO_ISTAT_ISTAT6_Pos                    (6U)                              
#define GPIO_ISTAT_ISTAT6_Msk                    (0x1U << GPIO_ISTAT_ISTAT6_Pos)       /*!< 0x00000040 */
#define GPIO_ISTAT_ISTAT6                        GPIO_ISTAT_ISTAT6_Msk                 /*!< Port input data, bit 6 */
#define GPIO_ISTAT_ISTAT7_Pos                    (7U)                              
#define GPIO_ISTAT_ISTAT7_Msk                    (0x1U << GPIO_ISTAT_ISTAT7_Pos)       /*!< 0x00000080 */
#define GPIO_ISTAT_ISTAT7                        GPIO_ISTAT_ISTAT7_Msk                 /*!< Port input data, bit 7 */
#define GPIO_ISTAT_ISTAT8_Pos                    (8U)                              
#define GPIO_ISTAT_ISTAT8_Msk                    (0x1U << GPIO_ISTAT_ISTAT8_Pos)       /*!< 0x00000100 */
#define GPIO_ISTAT_ISTAT8                        GPIO_ISTAT_ISTAT8_Msk                 /*!< Port input data, bit 8 */
#define GPIO_ISTAT_ISTAT9_Pos                    (9U)                              
#define GPIO_ISTAT_ISTAT9_Msk                    (0x1U << GPIO_ISTAT_ISTAT9_Pos)       /*!< 0x00000200 */
#define GPIO_ISTAT_ISTAT9                        GPIO_ISTAT_ISTAT9_Msk                 /*!< Port input data, bit 9 */
#define GPIO_ISTAT_ISTAT10_Pos                   (10U)                             
#define GPIO_ISTAT_ISTAT10_Msk                   (0x1U << GPIO_ISTAT_ISTAT10_Pos)      /*!< 0x00000400 */
#define GPIO_ISTAT_ISTAT10                       GPIO_ISTAT_ISTAT10_Msk                /*!< Port input data, bit 10 */
#define GPIO_ISTAT_ISTAT11_Pos                   (11U)                             
#define GPIO_ISTAT_ISTAT11_Msk                   (0x1U << GPIO_ISTAT_ISTAT11_Pos)      /*!< 0x00000800 */
#define GPIO_ISTAT_ISTAT11                       GPIO_ISTAT_ISTAT11_Msk                /*!< Port input data, bit 11 */
#define GPIO_ISTAT_ISTAT12_Pos                   (12U)                             
#define GPIO_ISTAT_ISTAT12_Msk                   (0x1U << GPIO_ISTAT_ISTAT12_Pos)      /*!< 0x00001000 */
#define GPIO_ISTAT_ISTAT12                       GPIO_ISTAT_ISTAT12_Msk                /*!< Port input data, bit 12 */
#define GPIO_ISTAT_ISTAT13_Pos                   (13U)                             
#define GPIO_ISTAT_ISTAT13_Msk                   (0x1U << GPIO_ISTAT_ISTAT13_Pos)      /*!< 0x00002000 */
#define GPIO_ISTAT_ISTAT13                       GPIO_ISTAT_ISTAT13_Msk                /*!< Port input data, bit 13 */
#define GPIO_ISTAT_ISTAT14_Pos                   (14U)                             
#define GPIO_ISTAT_ISTAT14_Msk                   (0x1U << GPIO_ISTAT_ISTAT14_Pos)      /*!< 0x00004000 */
#define GPIO_ISTAT_ISTAT14                       GPIO_ISTAT_ISTAT14_Msk                /*!< Port input data, bit 14 */
#define GPIO_ISTAT_ISTAT15_Pos                   (15U)                             
#define GPIO_ISTAT_ISTAT15_Msk                   (0x1U << GPIO_ISTAT_ISTAT15_Pos)      /*!< 0x00008000 */
#define GPIO_ISTAT_ISTAT15                       GPIO_ISTAT_ISTAT15_Msk                /*!< Port input data, bit 15 */

/*******************  Bit definition for GPIO_ODR register  *******************/
#define GPIO_OCTL_OCTL0_Pos                    (0U)                              
#define GPIO_OCTL_OCTL0_Msk                    (0x1U << GPIO_OCTL_OCTL0_Pos)       /*!< 0x00000001 */
#define GPIO_OCTL_OCTL0                        GPIO_OCTL_OCTL0_Msk                 /*!< Port output data, bit 0 */
#define GPIO_OCTL_OCTL1_Pos                    (1U)                              
#define GPIO_OCTL_OCTL1_Msk                    (0x1U << GPIO_OCTL_OCTL1_Pos)       /*!< 0x00000002 */
#define GPIO_OCTL_OCTL1                        GPIO_OCTL_OCTL1_Msk                 /*!< Port output data, bit 1 */
#define GPIO_OCTL_OCTL2_Pos                    (2U)                              
#define GPIO_OCTL_OCTL2_Msk                    (0x1U << GPIO_OCTL_OCTL2_Pos)       /*!< 0x00000004 */
#define GPIO_OCTL_OCTL2                        GPIO_OCTL_OCTL2_Msk                 /*!< Port output data, bit 2 */
#define GPIO_OCTL_OCTL3_Pos                    (3U)                              
#define GPIO_OCTL_OCTL3_Msk                    (0x1U << GPIO_OCTL_OCTL3_Pos)       /*!< 0x00000008 */
#define GPIO_OCTL_OCTL3                        GPIO_OCTL_OCTL3_Msk                 /*!< Port output data, bit 3 */
#define GPIO_OCTL_OCTL4_Pos                    (4U)                              
#define GPIO_OCTL_OCTL4_Msk                    (0x1U << GPIO_OCTL_OCTL4_Pos)       /*!< 0x00000010 */
#define GPIO_OCTL_OCTL4                        GPIO_OCTL_OCTL4_Msk                 /*!< Port output data, bit 4 */
#define GPIO_OCTL_OCTL5_Pos                    (5U)                              
#define GPIO_OCTL_OCTL5_Msk                    (0x1U << GPIO_OCTL_OCTL5_Pos)       /*!< 0x00000020 */
#define GPIO_OCTL_OCTL5                        GPIO_OCTL_OCTL5_Msk                 /*!< Port output data, bit 5 */
#define GPIO_OCTL_OCTL6_Pos                    (6U)                              
#define GPIO_OCTL_OCTL6_Msk                    (0x1U << GPIO_OCTL_OCTL6_Pos)       /*!< 0x00000040 */
#define GPIO_OCTL_OCTL6                        GPIO_OCTL_OCTL6_Msk                 /*!< Port output data, bit 6 */
#define GPIO_OCTL_OCTL7_Pos                    (7U)                              
#define GPIO_OCTL_OCTL7_Msk                    (0x1U << GPIO_OCTL_OCTL7_Pos)       /*!< 0x00000080 */
#define GPIO_OCTL_OCTL7                        GPIO_OCTL_OCTL7_Msk                 /*!< Port output data, bit 7 */
#define GPIO_OCTL_OCTL8_Pos                    (8U)                              
#define GPIO_OCTL_OCTL8_Msk                    (0x1U << GPIO_OCTL_OCTL8_Pos)       /*!< 0x00000100 */
#define GPIO_OCTL_OCTL8                        GPIO_OCTL_OCTL8_Msk                 /*!< Port output data, bit 8 */
#define GPIO_OCTL_OCTL9_Pos                    (9U)                              
#define GPIO_OCTL_OCTL9_Msk                    (0x1U << GPIO_OCTL_OCTL9_Pos)       /*!< 0x00000200 */
#define GPIO_OCTL_OCTL9                        GPIO_OCTL_OCTL9_Msk                 /*!< Port output data, bit 9 */
#define GPIO_OCTL_OCTL10_Pos                   (10U)                             
#define GPIO_OCTL_OCTL10_Msk                   (0x1U << GPIO_OCTL_OCTL10_Pos)      /*!< 0x00000400 */
#define GPIO_OCTL_OCTL10                       GPIO_OCTL_OCTL10_Msk                /*!< Port output data, bit 10 */
#define GPIO_OCTL_OCTL11_Pos                   (11U)                             
#define GPIO_OCTL_OCTL11_Msk                   (0x1U << GPIO_OCTL_OCTL11_Pos)      /*!< 0x00000800 */
#define GPIO_OCTL_OCTL11                       GPIO_OCTL_OCTL11_Msk                /*!< Port output data, bit 11 */
#define GPIO_OCTL_OCTL12_Pos                   (12U)                             
#define GPIO_OCTL_OCTL12_Msk                   (0x1U << GPIO_OCTL_OCTL12_Pos)      /*!< 0x00001000 */
#define GPIO_OCTL_OCTL12                       GPIO_OCTL_OCTL12_Msk                /*!< Port output data, bit 12 */
#define GPIO_OCTL_OCTL13_Pos                   (13U)                             
#define GPIO_OCTL_OCTL13_Msk                   (0x1U << GPIO_OCTL_OCTL13_Pos)      /*!< 0x00002000 */
#define GPIO_OCTL_OCTL13                       GPIO_OCTL_OCTL13_Msk                /*!< Port output data, bit 13 */
#define GPIO_OCTL_OCTL14_Pos                   (14U)                             
#define GPIO_OCTL_OCTL14_Msk                   (0x1U << GPIO_OCTL_OCTL14_Pos)      /*!< 0x00004000 */
#define GPIO_OCTL_OCTL14                       GPIO_OCTL_OCTL14_Msk                /*!< Port output data, bit 14 */
#define GPIO_OCTL_OCTL15_Pos                   (15U)                             
#define GPIO_OCTL_OCTL15_Msk                   (0x1U << GPIO_OCTL_OCTL15_Pos)      /*!< 0x00008000 */
#define GPIO_OCTL_OCTL15                       GPIO_OCTL_OCTL15_Msk                /*!< Port output data, bit 15 */

/******************  Bit definition for GPIO_BOP register  *******************/
#define GPIO_BOP_BOP0_Pos                    (0U)                              
#define GPIO_BOP_BOP0_Msk                    (0x1U << GPIO_BOP_BOP0_Pos)       /*!< 0x00000001 */
#define GPIO_BOP_BOP0                        GPIO_BOP_BOP0_Msk                 /*!< Port x Set bit 0 */
#define GPIO_BOP_BOP1_Pos                    (1U)                              
#define GPIO_BOP_BOP1_Msk                    (0x1U << GPIO_BOP_BOP1_Pos)       /*!< 0x00000002 */
#define GPIO_BOP_BOP1                        GPIO_BOP_BOP1_Msk                 /*!< Port x Set bit 1 */
#define GPIO_BOP_BOP2_Pos                    (2U)                              
#define GPIO_BOP_BOP2_Msk                    (0x1U << GPIO_BOP_BOP2_Pos)       /*!< 0x00000004 */
#define GPIO_BOP_BOP2                        GPIO_BOP_BOP2_Msk                 /*!< Port x Set bit 2 */
#define GPIO_BOP_BOP3_Pos                    (3U)                              
#define GPIO_BOP_BOP3_Msk                    (0x1U << GPIO_BOP_BOP3_Pos)       /*!< 0x00000008 */
#define GPIO_BOP_BOP3                        GPIO_BOP_BOP3_Msk                 /*!< Port x Set bit 3 */
#define GPIO_BOP_BOP4_Pos                    (4U)                              
#define GPIO_BOP_BOP4_Msk                    (0x1U << GPIO_BOP_BOP4_Pos)       /*!< 0x00000010 */
#define GPIO_BOP_BOP4                        GPIO_BOP_BOP4_Msk                 /*!< Port x Set bit 4 */
#define GPIO_BOP_BOP5_Pos                    (5U)                              
#define GPIO_BOP_BOP5_Msk                    (0x1U << GPIO_BOP_BOP5_Pos)       /*!< 0x00000020 */
#define GPIO_BOP_BOP5                        GPIO_BOP_BOP5_Msk                 /*!< Port x Set bit 5 */
#define GPIO_BOP_BOP6_Pos                    (6U)                              
#define GPIO_BOP_BOP6_Msk                    (0x1U << GPIO_BOP_BOP6_Pos)       /*!< 0x00000040 */
#define GPIO_BOP_BOP6                        GPIO_BOP_BOP6_Msk                 /*!< Port x Set bit 6 */
#define GPIO_BOP_BOP7_Pos                    (7U)                              
#define GPIO_BOP_BOP7_Msk                    (0x1U << GPIO_BOP_BOP7_Pos)       /*!< 0x00000080 */
#define GPIO_BOP_BOP7                        GPIO_BOP_BOP7_Msk                 /*!< Port x Set bit 7 */
#define GPIO_BOP_BOP8_Pos                    (8U)                              
#define GPIO_BOP_BOP8_Msk                    (0x1U << GPIO_BOP_BOP8_Pos)       /*!< 0x00000100 */
#define GPIO_BOP_BOP8                        GPIO_BOP_BOP8_Msk                 /*!< Port x Set bit 8 */
#define GPIO_BOP_BOP9_Pos                    (9U)                              
#define GPIO_BOP_BOP9_Msk                    (0x1U << GPIO_BOP_BOP9_Pos)       /*!< 0x00000200 */
#define GPIO_BOP_BOP9                        GPIO_BOP_BOP9_Msk                 /*!< Port x Set bit 9 */
#define GPIO_BOP_BOP10_Pos                   (10U)                             
#define GPIO_BOP_BOP10_Msk                   (0x1U << GPIO_BOP_BOP10_Pos)      /*!< 0x00000400 */
#define GPIO_BOP_BOP10                       GPIO_BOP_BOP10_Msk                /*!< Port x Set bit 10 */
#define GPIO_BOP_BOP11_Pos                   (11U)                             
#define GPIO_BOP_BOP11_Msk                   (0x1U << GPIO_BOP_BOP11_Pos)      /*!< 0x00000800 */
#define GPIO_BOP_BOP11                       GPIO_BOP_BOP11_Msk                /*!< Port x Set bit 11 */
#define GPIO_BOP_BOP12_Pos                   (12U)                             
#define GPIO_BOP_BOP12_Msk                   (0x1U << GPIO_BOP_BOP12_Pos)      /*!< 0x00001000 */
#define GPIO_BOP_BOP12                       GPIO_BOP_BOP12_Msk                /*!< Port x Set bit 12 */
#define GPIO_BOP_BOP13_Pos                   (13U)                             
#define GPIO_BOP_BOP13_Msk                   (0x1U << GPIO_BOP_BOP13_Pos)      /*!< 0x00002000 */
#define GPIO_BOP_BOP13                       GPIO_BOP_BOP13_Msk                /*!< Port x Set bit 13 */
#define GPIO_BOP_BOP14_Pos                   (14U)                             
#define GPIO_BOP_BOP14_Msk                   (0x1U << GPIO_BOP_BOP14_Pos)      /*!< 0x00004000 */
#define GPIO_BOP_BOP14                       GPIO_BOP_BOP14_Msk                /*!< Port x Set bit 14 */
#define GPIO_BOP_BOP15_Pos                   (15U)                             
#define GPIO_BOP_BOP15_Msk                   (0x1U << GPIO_BOP_BOP15_Pos)      /*!< 0x00008000 */
#define GPIO_BOP_BOP15                       GPIO_BOP_BOP15_Msk                /*!< Port x Set bit 15 */

#define GPIO_BOP_CR0_Pos                    (16U)                             
#define GPIO_BOP_CR0_Msk                    (0x1U << GPIO_BOP_CR0_Pos)       /*!< 0x00010000 */
#define GPIO_BOP_CR0                        GPIO_BOP_CR0_Msk                 /*!< Port x Reset bit 0 */
#define GPIO_BOP_CR1_Pos                    (17U)                             
#define GPIO_BOP_CR1_Msk                    (0x1U << GPIO_BOP_CR1_Pos)       /*!< 0x00020000 */
#define GPIO_BOP_CR1                        GPIO_BOP_CR1_Msk                 /*!< Port x Reset bit 1 */
#define GPIO_BOP_CR2_Pos                    (18U)                             
#define GPIO_BOP_CR2_Msk                    (0x1U << GPIO_BOP_CR2_Pos)       /*!< 0x00040000 */
#define GPIO_BOP_CR2                        GPIO_BOP_CR2_Msk                 /*!< Port x Reset bit 2 */
#define GPIO_BOP_CR3_Pos                    (19U)                             
#define GPIO_BOP_CR3_Msk                    (0x1U << GPIO_BOP_CR3_Pos)       /*!< 0x00080000 */
#define GPIO_BOP_CR3                        GPIO_BOP_CR3_Msk                 /*!< Port x Reset bit 3 */
#define GPIO_BOP_CR4_Pos                    (20U)                             
#define GPIO_BOP_CR4_Msk                    (0x1U << GPIO_BOP_CR4_Pos)       /*!< 0x00100000 */
#define GPIO_BOP_CR4                        GPIO_BOP_CR4_Msk                 /*!< Port x Reset bit 4 */
#define GPIO_BOP_CR5_Pos                    (21U)                             
#define GPIO_BOP_CR5_Msk                    (0x1U << GPIO_BOP_CR5_Pos)       /*!< 0x00200000 */
#define GPIO_BOP_CR5                        GPIO_BOP_CR5_Msk                 /*!< Port x Reset bit 5 */
#define GPIO_BOP_CR6_Pos                    (22U)                             
#define GPIO_BOP_CR6_Msk                    (0x1U << GPIO_BOP_CR6_Pos)       /*!< 0x00400000 */
#define GPIO_BOP_CR6                        GPIO_BOP_CR6_Msk                 /*!< Port x Reset bit 6 */
#define GPIO_BOP_CR7_Pos                    (23U)                             
#define GPIO_BOP_CR7_Msk                    (0x1U << GPIO_BOP_CR7_Pos)       /*!< 0x00800000 */
#define GPIO_BOP_CR7                        GPIO_BOP_CR7_Msk                 /*!< Port x Reset bit 7 */
#define GPIO_BOP_CR8_Pos                    (24U)                             
#define GPIO_BOP_CR8_Msk                    (0x1U << GPIO_BOP_CR8_Pos)       /*!< 0x01000000 */
#define GPIO_BOP_CR8                        GPIO_BOP_CR8_Msk                 /*!< Port x Reset bit 8 */
#define GPIO_BOP_CR9_Pos                    (25U)                             
#define GPIO_BOP_CR9_Msk                    (0x1U << GPIO_BOP_CR9_Pos)       /*!< 0x02000000 */
#define GPIO_BOP_CR9                        GPIO_BOP_CR9_Msk                 /*!< Port x Reset bit 9 */
#define GPIO_BOP_CR10_Pos                   (26U)                             
#define GPIO_BOP_CR10_Msk                   (0x1U << GPIO_BOP_CR10_Pos)      /*!< 0x04000000 */
#define GPIO_BOP_CR10                       GPIO_BOP_CR10_Msk                /*!< Port x Reset bit 10 */
#define GPIO_BOP_CR11_Pos                   (27U)                             
#define GPIO_BOP_CR11_Msk                   (0x1U << GPIO_BOP_CR11_Pos)      /*!< 0x08000000 */
#define GPIO_BOP_CR11                       GPIO_BOP_CR11_Msk                /*!< Port x Reset bit 11 */
#define GPIO_BOP_CR12_Pos                   (28U)                             
#define GPIO_BOP_CR12_Msk                   (0x1U << GPIO_BOP_CR12_Pos)      /*!< 0x10000000 */
#define GPIO_BOP_CR12                       GPIO_BOP_CR12_Msk                /*!< Port x Reset bit 12 */
#define GPIO_BOP_CR13_Pos                   (29U)                             
#define GPIO_BOP_CR13_Msk                   (0x1U << GPIO_BOP_CR13_Pos)      /*!< 0x20000000 */
#define GPIO_BOP_CR13                       GPIO_BOP_CR13_Msk                /*!< Port x Reset bit 13 */
#define GPIO_BOP_CR14_Pos                   (30U)                             
#define GPIO_BOP_CR14_Msk                   (0x1U << GPIO_BOP_CR14_Pos)      /*!< 0x40000000 */
#define GPIO_BOP_CR14                       GPIO_BOP_CR14_Msk                /*!< Port x Reset bit 14 */
#define GPIO_BOP_CR15_Pos                   (31U)                             
#define GPIO_BOP_CR15_Msk                   (0x1U << GPIO_BOP_CR15_Pos)      /*!< 0x80000000 */
#define GPIO_BOP_CR15                       GPIO_BOP_CR15_Msk                /*!< Port x Reset bit 15 */

/*******************  Bit definition for GPIO_BC register  *******************/
#define GPIO_BC_CR0_Pos                     (0U)                              
#define GPIO_BC_CR0_Msk                     (0x1U << GPIO_BC_CR0_Pos)        /*!< 0x00000001 */
#define GPIO_BC_CR0                         GPIO_BC_CR0_Msk                  /*!< Port x Reset bit 0 */
#define GPIO_BC_CR1_Pos                     (1U)                              
#define GPIO_BC_CR1_Msk                     (0x1U << GPIO_BC_CR1_Pos)        /*!< 0x00000002 */
#define GPIO_BC_CR1                         GPIO_BC_CR1_Msk                  /*!< Port x Reset bit 1 */
#define GPIO_BC_CR2_Pos                     (2U)                              
#define GPIO_BC_CR2_Msk                     (0x1U << GPIO_BC_CR2_Pos)        /*!< 0x00000004 */
#define GPIO_BC_CR2                         GPIO_BC_CR2_Msk                  /*!< Port x Reset bit 2 */
#define GPIO_BC_CR3_Pos                     (3U)                              
#define GPIO_BC_CR3_Msk                     (0x1U << GPIO_BC_CR3_Pos)        /*!< 0x00000008 */
#define GPIO_BC_CR3                         GPIO_BC_CR3_Msk                  /*!< Port x Reset bit 3 */
#define GPIO_BC_CR4_Pos                     (4U)                              
#define GPIO_BC_CR4_Msk                     (0x1U << GPIO_BC_CR4_Pos)        /*!< 0x00000010 */
#define GPIO_BC_CR4                         GPIO_BC_CR4_Msk                  /*!< Port x Reset bit 4 */
#define GPIO_BC_CR5_Pos                     (5U)                              
#define GPIO_BC_CR5_Msk                     (0x1U << GPIO_BC_CR5_Pos)        /*!< 0x00000020 */
#define GPIO_BC_CR5                         GPIO_BC_CR5_Msk                  /*!< Port x Reset bit 5 */
#define GPIO_BC_CR6_Pos                     (6U)                              
#define GPIO_BC_CR6_Msk                     (0x1U << GPIO_BC_CR6_Pos)        /*!< 0x00000040 */
#define GPIO_BC_CR6                         GPIO_BC_CR6_Msk                  /*!< Port x Reset bit 6 */
#define GPIO_BC_CR7_Pos                     (7U)                              
#define GPIO_BC_CR7_Msk                     (0x1U << GPIO_BC_CR7_Pos)        /*!< 0x00000080 */
#define GPIO_BC_CR7                         GPIO_BC_CR7_Msk                  /*!< Port x Reset bit 7 */
#define GPIO_BC_CR8_Pos                     (8U)                              
#define GPIO_BC_CR8_Msk                     (0x1U << GPIO_BC_CR8_Pos)        /*!< 0x00000100 */
#define GPIO_BC_CR8                         GPIO_BC_CR8_Msk                  /*!< Port x Reset bit 8 */
#define GPIO_BC_CR9_Pos                     (9U)                              
#define GPIO_BC_CR9_Msk                     (0x1U << GPIO_BC_CR9_Pos)        /*!< 0x00000200 */
#define GPIO_BC_CR9                         GPIO_BC_CR9_Msk                  /*!< Port x Reset bit 9 */
#define GPIO_BC_CR10_Pos                    (10U)                             
#define GPIO_BC_CR10_Msk                    (0x1U << GPIO_BC_CR10_Pos)       /*!< 0x00000400 */
#define GPIO_BC_CR10                        GPIO_BC_CR10_Msk                 /*!< Port x Reset bit 10 */
#define GPIO_BC_CR11_Pos                    (11U)                             
#define GPIO_BC_CR11_Msk                    (0x1U << GPIO_BC_CR11_Pos)       /*!< 0x00000800 */
#define GPIO_BC_CR11                        GPIO_BC_CR11_Msk                 /*!< Port x Reset bit 11 */
#define GPIO_BC_CR12_Pos                    (12U)                             
#define GPIO_BC_CR12_Msk                    (0x1U << GPIO_BC_CR12_Pos)       /*!< 0x00001000 */
#define GPIO_BC_CR12                        GPIO_BC_CR12_Msk                 /*!< Port x Reset bit 12 */
#define GPIO_BC_CR13_Pos                    (13U)                             
#define GPIO_BC_CR13_Msk                    (0x1U << GPIO_BC_CR13_Pos)       /*!< 0x00002000 */
#define GPIO_BC_CR13                        GPIO_BC_CR13_Msk                 /*!< Port x Reset bit 13 */
#define GPIO_BC_CR14_Pos                    (14U)                             
#define GPIO_BC_CR14_Msk                    (0x1U << GPIO_BC_CR14_Pos)       /*!< 0x00004000 */
#define GPIO_BC_CR14                        GPIO_BC_CR14_Msk                 /*!< Port x Reset bit 14 */
#define GPIO_BC_CR15_Pos                    (15U)                             
#define GPIO_BC_CR15_Msk                    (0x1U << GPIO_BC_CR15_Pos)       /*!< 0x00008000 */
#define GPIO_BC_CR15                        GPIO_BC_CR15_Msk                 /*!< Port x Reset bit 15 */

/******************  Bit definition for GPIO_LOCK register  *******************/
#define GPIO_LOCK_LK0_Pos                   (0U)                              
#define GPIO_LOCK_LK0_Msk                   (0x1U << GPIO_LOCK_LK0_Pos)      /*!< 0x00000001 */
#define GPIO_LOCK_LK0                       GPIO_LOCK_LK0_Msk                /*!< Port x Lock bit 0 */
#define GPIO_LOCK_LK1_Pos                   (1U)                              
#define GPIO_LOCK_LK1_Msk                   (0x1U << GPIO_LOCK_LK1_Pos)      /*!< 0x00000002 */
#define GPIO_LOCK_LK1                       GPIO_LOCK_LK1_Msk                /*!< Port x Lock bit 1 */
#define GPIO_LOCK_LK2_Pos                   (2U)                              
#define GPIO_LOCK_LK2_Msk                   (0x1U << GPIO_LOCK_LK2_Pos)      /*!< 0x00000004 */
#define GPIO_LOCK_LK2                       GPIO_LOCK_LK2_Msk                /*!< Port x Lock bit 2 */
#define GPIO_LOCK_LK3_Pos                   (3U)                              
#define GPIO_LOCK_LK3_Msk                   (0x1U << GPIO_LOCK_LK3_Pos)      /*!< 0x00000008 */
#define GPIO_LOCK_LK3                       GPIO_LOCK_LK3_Msk                /*!< Port x Lock bit 3 */
#define GPIO_LOCK_LK4_Pos                   (4U)                              
#define GPIO_LOCK_LK4_Msk                   (0x1U << GPIO_LOCK_LK4_Pos)      /*!< 0x00000010 */
#define GPIO_LOCK_LK4                       GPIO_LOCK_LK4_Msk                /*!< Port x Lock bit 4 */
#define GPIO_LOCK_LK5_Pos                   (5U)                              
#define GPIO_LOCK_LK5_Msk                   (0x1U << GPIO_LOCK_LK5_Pos)      /*!< 0x00000020 */
#define GPIO_LOCK_LK5                       GPIO_LOCK_LK5_Msk                /*!< Port x Lock bit 5 */
#define GPIO_LOCK_LK6_Pos                   (6U)                              
#define GPIO_LOCK_LK6_Msk                   (0x1U << GPIO_LOCK_LK6_Pos)      /*!< 0x00000040 */
#define GPIO_LOCK_LK6                       GPIO_LOCK_LK6_Msk                /*!< Port x Lock bit 6 */
#define GPIO_LOCK_LK7_Pos                   (7U)                              
#define GPIO_LOCK_LK7_Msk                   (0x1U << GPIO_LOCK_LK7_Pos)      /*!< 0x00000080 */
#define GPIO_LOCK_LK7                       GPIO_LOCK_LK7_Msk                /*!< Port x Lock bit 7 */
#define GPIO_LOCK_LK8_Pos                   (8U)                              
#define GPIO_LOCK_LK8_Msk                   (0x1U << GPIO_LOCK_LK8_Pos)      /*!< 0x00000100 */
#define GPIO_LOCK_LK8                       GPIO_LOCK_LK8_Msk                /*!< Port x Lock bit 8 */
#define GPIO_LOCK_LK9_Pos                   (9U)                              
#define GPIO_LOCK_LK9_Msk                   (0x1U << GPIO_LOCK_LK9_Pos)      /*!< 0x00000200 */
#define GPIO_LOCK_LK9                       GPIO_LOCK_LK9_Msk                /*!< Port x Lock bit 9 */
#define GPIO_LOCK_LK10_Pos                  (10U)                             
#define GPIO_LOCK_LK10_Msk                  (0x1U << GPIO_LOCK_LK10_Pos)     /*!< 0x00000400 */
#define GPIO_LOCK_LK10                      GPIO_LOCK_LK10_Msk               /*!< Port x Lock bit 10 */
#define GPIO_LOCK_LK11_Pos                  (11U)                             
#define GPIO_LOCK_LK11_Msk                  (0x1U << GPIO_LOCK_LK11_Pos)     /*!< 0x00000800 */
#define GPIO_LOCK_LK11                      GPIO_LOCK_LK11_Msk               /*!< Port x Lock bit 11 */
#define GPIO_LOCK_LK12_Pos                  (12U)                             
#define GPIO_LOCK_LK12_Msk                  (0x1U << GPIO_LOCK_LK12_Pos)     /*!< 0x00001000 */
#define GPIO_LOCK_LK12                      GPIO_LOCK_LK12_Msk               /*!< Port x Lock bit 12 */
#define GPIO_LOCK_LK13_Pos                  (13U)                             
#define GPIO_LOCK_LK13_Msk                  (0x1U << GPIO_LOCK_LK13_Pos)     /*!< 0x00002000 */
#define GPIO_LOCK_LK13                      GPIO_LOCK_LK13_Msk               /*!< Port x Lock bit 13 */
#define GPIO_LOCK_LK14_Pos                  (14U)                             
#define GPIO_LOCK_LK14_Msk                  (0x1U << GPIO_LOCK_LK14_Pos)     /*!< 0x00004000 */
#define GPIO_LOCK_LK14                      GPIO_LOCK_LK14_Msk               /*!< Port x Lock bit 14 */
#define GPIO_LOCK_LK15_Pos                  (15U)                             
#define GPIO_LOCK_LK15_Msk                  (0x1U << GPIO_LOCK_LK15_Pos)     /*!< 0x00008000 */
#define GPIO_LOCK_LK15                      GPIO_LOCK_LK15_Msk               /*!< Port x Lock bit 15 */
#define GPIO_LOCK_LKK_Pos                   (16U)                             
#define GPIO_LOCK_LKK_Msk                   (0x1U << GPIO_LOCK_LKK_Pos)      /*!< 0x00010000 */
#define GPIO_LOCK_LKK                       GPIO_LOCK_LKK_Msk                /*!< Lock key */

/*----------------------------------------------------------------------------*/

/******************  Bit definition for AFIO_EC register  *******************/
#define AFIO_EC_PIN_Pos                    (0U)                              
#define AFIO_EC_PIN_Msk                    (0xFU << AFIO_EC_PIN_Pos)       /*!< 0x0000000F */
#define AFIO_EC_PIN                        AFIO_EC_PIN_Msk                 /*!< PIN[3:0] bits (Pin selection) */
#define AFIO_EC_PIN_0                      (0x1U << AFIO_EC_PIN_Pos)       /*!< 0x00000001 */
#define AFIO_EC_PIN_1                      (0x2U << AFIO_EC_PIN_Pos)       /*!< 0x00000002 */
#define AFIO_EC_PIN_2                      (0x4U << AFIO_EC_PIN_Pos)       /*!< 0x00000004 */
#define AFIO_EC_PIN_3                      (0x8U << AFIO_EC_PIN_Pos)       /*!< 0x00000008 */

/*!< PIN configuration */
#define AFIO_EC_PIN_PX0                    0x00000000U                       /*!< Pin 0 selected */
#define AFIO_EC_PIN_PX1_Pos                (0U)                              
#define AFIO_EC_PIN_PX1_Msk                (0x1U << AFIO_EC_PIN_PX1_Pos)   /*!< 0x00000001 */
#define AFIO_EC_PIN_PX1                    AFIO_EC_PIN_PX1_Msk             /*!< Pin 1 selected */
#define AFIO_EC_PIN_PX2_Pos                (1U)                              
#define AFIO_EC_PIN_PX2_Msk                (0x1U << AFIO_EC_PIN_PX2_Pos)   /*!< 0x00000002 */
#define AFIO_EC_PIN_PX2                    AFIO_EC_PIN_PX2_Msk             /*!< Pin 2 selected */
#define AFIO_EC_PIN_PX3_Pos                (0U)                              
#define AFIO_EC_PIN_PX3_Msk                (0x3U << AFIO_EC_PIN_PX3_Pos)   /*!< 0x00000003 */
#define AFIO_EC_PIN_PX3                    AFIO_EC_PIN_PX3_Msk             /*!< Pin 3 selected */
#define AFIO_EC_PIN_PX4_Pos                (2U)                              
#define AFIO_EC_PIN_PX4_Msk                (0x1U << AFIO_EC_PIN_PX4_Pos)   /*!< 0x00000004 */
#define AFIO_EC_PIN_PX4                    AFIO_EC_PIN_PX4_Msk             /*!< Pin 4 selected */
#define AFIO_EC_PIN_PX5_Pos                (0U)                              
#define AFIO_EC_PIN_PX5_Msk                (0x5U << AFIO_EC_PIN_PX5_Pos)   /*!< 0x00000005 */
#define AFIO_EC_PIN_PX5                    AFIO_EC_PIN_PX5_Msk             /*!< Pin 5 selected */
#define AFIO_EC_PIN_PX6_Pos                (1U)                              
#define AFIO_EC_PIN_PX6_Msk                (0x3U << AFIO_EC_PIN_PX6_Pos)   /*!< 0x00000006 */
#define AFIO_EC_PIN_PX6                    AFIO_EC_PIN_PX6_Msk             /*!< Pin 6 selected */
#define AFIO_EC_PIN_PX7_Pos                (0U)                              
#define AFIO_EC_PIN_PX7_Msk                (0x7U << AFIO_EC_PIN_PX7_Pos)   /*!< 0x00000007 */
#define AFIO_EC_PIN_PX7                    AFIO_EC_PIN_PX7_Msk             /*!< Pin 7 selected */
#define AFIO_EC_PIN_PX8_Pos                (3U)                              
#define AFIO_EC_PIN_PX8_Msk                (0x1U << AFIO_EC_PIN_PX8_Pos)   /*!< 0x00000008 */
#define AFIO_EC_PIN_PX8                    AFIO_EC_PIN_PX8_Msk             /*!< Pin 8 selected */
#define AFIO_EC_PIN_PX9_Pos                (0U)                              
#define AFIO_EC_PIN_PX9_Msk                (0x9U << AFIO_EC_PIN_PX9_Pos)   /*!< 0x00000009 */
#define AFIO_EC_PIN_PX9                    AFIO_EC_PIN_PX9_Msk             /*!< Pin 9 selected */
#define AFIO_EC_PIN_PX10_Pos               (1U)                              
#define AFIO_EC_PIN_PX10_Msk               (0x5U << AFIO_EC_PIN_PX10_Pos)  /*!< 0x0000000A */
#define AFIO_EC_PIN_PX10                   AFIO_EC_PIN_PX10_Msk            /*!< Pin 10 selected */
#define AFIO_EC_PIN_PX11_Pos               (0U)                              
#define AFIO_EC_PIN_PX11_Msk               (0xBU << AFIO_EC_PIN_PX11_Pos)  /*!< 0x0000000B */
#define AFIO_EC_PIN_PX11                   AFIO_EC_PIN_PX11_Msk            /*!< Pin 11 selected */
#define AFIO_EC_PIN_PX12_Pos               (2U)                              
#define AFIO_EC_PIN_PX12_Msk               (0x3U << AFIO_EC_PIN_PX12_Pos)  /*!< 0x0000000C */
#define AFIO_EC_PIN_PX12                   AFIO_EC_PIN_PX12_Msk            /*!< Pin 12 selected */
#define AFIO_EC_PIN_PX13_Pos               (0U)                              
#define AFIO_EC_PIN_PX13_Msk               (0xDU << AFIO_EC_PIN_PX13_Pos)  /*!< 0x0000000D */
#define AFIO_EC_PIN_PX13                   AFIO_EC_PIN_PX13_Msk            /*!< Pin 13 selected */
#define AFIO_EC_PIN_PX14_Pos               (1U)                              
#define AFIO_EC_PIN_PX14_Msk               (0x7U << AFIO_EC_PIN_PX14_Pos)  /*!< 0x0000000E */
#define AFIO_EC_PIN_PX14                   AFIO_EC_PIN_PX14_Msk            /*!< Pin 14 selected */
#define AFIO_EC_PIN_PX15_Pos               (0U)                              
#define AFIO_EC_PIN_PX15_Msk               (0xFU << AFIO_EC_PIN_PX15_Pos)  /*!< 0x0000000F */
#define AFIO_EC_PIN_PX15                   AFIO_EC_PIN_PX15_Msk            /*!< Pin 15 selected */

#define AFIO_EC_PORT_Pos                   (4U)                              
#define AFIO_EC_PORT_Msk                   (0x7U << AFIO_EC_PORT_Pos)      /*!< 0x00000070 */
#define AFIO_EC_PORT                       AFIO_EC_PORT_Msk                /*!< PORT[2:0] bits (Port selection) */
#define AFIO_EC_PORT_0                     (0x1U << AFIO_EC_PORT_Pos)      /*!< 0x00000010 */
#define AFIO_EC_PORT_1                     (0x2U << AFIO_EC_PORT_Pos)      /*!< 0x00000020 */
#define AFIO_EC_PORT_2                     (0x4U << AFIO_EC_PORT_Pos)      /*!< 0x00000040 */

/*!< PORT configuration */
#define AFIO_EC_PORT_PA                    0x00000000                        /*!< Port A selected */
#define AFIO_EC_PORT_PB_Pos                (4U)                              
#define AFIO_EC_PORT_PB_Msk                (0x1U << AFIO_EC_PORT_PB_Pos)   /*!< 0x00000010 */
#define AFIO_EC_PORT_PB                    AFIO_EC_PORT_PB_Msk             /*!< Port B selected */
#define AFIO_EC_PORT_PC_Pos                (5U)                              
#define AFIO_EC_PORT_PC_Msk                (0x1U << AFIO_EC_PORT_PC_Pos)   /*!< 0x00000020 */
#define AFIO_EC_PORT_PC                    AFIO_EC_PORT_PC_Msk             /*!< Port C selected */
#define AFIO_EC_PORT_PD_Pos                (4U)                              
#define AFIO_EC_PORT_PD_Msk                (0x3U << AFIO_EC_PORT_PD_Pos)   /*!< 0x00000030 */
#define AFIO_EC_PORT_PD                    AFIO_EC_PORT_PD_Msk             /*!< Port D selected */
#define AFIO_EC_PORT_PE_Pos                (6U)                              
#define AFIO_EC_PORT_PE_Msk                (0x1U << AFIO_EC_PORT_PE_Pos)   /*!< 0x00000040 */
#define AFIO_EC_PORT_PE                    AFIO_EC_PORT_PE_Msk             /*!< Port E selected */

#define AFIO_EC_EOE_Pos                   (7U)                              
#define AFIO_EC_EOE_Msk                   (0x1U << AFIO_EC_EOE_Pos)      /*!< 0x00000080 */
#define AFIO_EC_EOE                       AFIO_EC_EOE_Msk                /*!< Event Output Enable */

/******************  Bit definition for AFIO_PCF0 register  *******************/
#define AFIO_PCF0_SPI0_REMAP_Pos             (0U)                              
#define AFIO_PCF0_SPI0_REMAP_Msk             (0x1U << AFIO_PCF0_SPI0_REMAP_Pos) /*!< 0x00000001 */
#define AFIO_PCF0_SPI0_REMAP                 AFIO_PCF0_SPI0_REMAP_Msk          /*!< SPI0 remapping */
#define AFIO_PCF0_I2C0_REMAP_Pos             (1U)                              
#define AFIO_PCF0_I2C0_REMAP_Msk             (0x1U << AFIO_PCF0_I2C0_REMAP_Pos) /*!< 0x00000002 */
#define AFIO_PCF0_I2C0_REMAP                 AFIO_PCF0_I2C0_REMAP_Msk          /*!< I2C0 remapping */
#define AFIO_PCF0_USART0_REMAP_Pos           (2U)                              
#define AFIO_PCF0_USART0_REMAP_Msk           (0x1U << AFIO_PCF0_USART0_REMAP_Pos) /*!< 0x00000004 */
#define AFIO_PCF0_USART0_REMAP               AFIO_PCF0_USART0_REMAP_Msk        /*!< USART1 remapping */
#define AFIO_PCF0_USART1_REMAP_Pos           (3U)                              
#define AFIO_PCF0_USART1_REMAP_Msk           (0x1U << AFIO_PCF0_USART1_REMAP_Pos) /*!< 0x00000008 */
#define AFIO_PCF0_USART1_REMAP               AFIO_PCF0_USART1_REMAP_Msk        /*!< USART2 remapping */

#define AFIO_PCF0_USART2_REMAP_Pos           (4U)                              
#define AFIO_PCF0_USART2_REMAP_Msk           (0x3U << AFIO_PCF0_USART2_REMAP_Pos) /*!< 0x00000030 */
#define AFIO_PCF0_USART2_REMAP               AFIO_PCF0_USART2_REMAP_Msk        /*!< USART3_REMAP[1:0] bits (USART3 remapping) */
#define AFIO_PCF0_USART2_REMAP_0             (0x1U << AFIO_PCF0_USART2_REMAP_Pos) /*!< 0x00000010 */
#define AFIO_PCF0_USART2_REMAP_1             (0x2U << AFIO_PCF0_USART2_REMAP_Pos) /*!< 0x00000020 */

/* USART3_REMAP configuration */
#define AFIO_PCF0_USART2_REMAP_NOREMAP       0x00000000U                          /*!< No remap (TX/PB10, RX/PB11, CK/PB12, CTS/PB13, RTS/PB14) */
#define AFIO_PCF0_USART2_REMAP_PARTIALREMAP_Pos (4U)                           
#define AFIO_PCF0_USART2_REMAP_PARTIALREMAP_Msk (0x1U << AFIO_PCF0_USART2_REMAP_PARTIALREMAP_Pos) /*!< 0x00000010 */
#define AFIO_PCF0_USART2_REMAP_PARTIALREMAP  AFIO_PCF0_USART2_REMAP_PARTIALREMAP_Msk /*!< Partial remap (TX/PC10, RX/PC11, CK/PC12, CTS/PB13, RTS/PB14) */
#define AFIO_PCF0_USART2_REMAP_FULLREMAP_Pos (4U)                              
#define AFIO_PCF0_USART2_REMAP_FULLREMAP_Msk (0x3U << AFIO_PCF0_USART2_REMAP_FULLREMAP_Pos) /*!< 0x00000030 */
#define AFIO_PCF0_USART2_REMAP_FULLREMAP     AFIO_PCF0_USART2_REMAP_FULLREMAP_Msk /*!< Full remap (TX/PD8, RX/PD9, CK/PD10, CTS/PD11, RTS/PD12) */

#define AFIO_PCF0_TIMER0_REMAP_Pos             (6U)                              
#define AFIO_PCF0_TIMER0_REMAP_Msk             (0x3U << AFIO_PCF0_TIMER0_REMAP_Pos) /*!< 0x000000C0 */
#define AFIO_PCF0_TIMER0_REMAP                 AFIO_PCF0_TIMER0_REMAP_Msk          /*!< TIM1_REMAP[1:0] bits (TIM1 remapping) */
#define AFIO_PCF0_TIMER0_REMAP_0               (0x1U << AFIO_PCF0_TIMER0_REMAP_Pos) /*!< 0x00000040 */
#define AFIO_PCF0_TIMER0_REMAP_1               (0x2U << AFIO_PCF0_TIMER0_REMAP_Pos) /*!< 0x00000080 */

/*!< TIM1_REMAP configuration */
#define AFIO_PCF0_TIMER0_REMAP_NOREMAP         0x00000000U                          /*!< No remap (ETR/PA12, CH1/PA8, CH2/PA9, CH3/PA10, CH4/PA11, BKIN/PB12, CH1N/PB13, CH2N/PB14, CH3N/PB15) */
#define AFIO_PCF0_TIMER0_REMAP_PARTIALREMAP_Pos (6U)                             
#define AFIO_PCF0_TIMER0_REMAP_PARTIALREMAP_Msk (0x1U << AFIO_PCF0_TIMER0_REMAP_PARTIALREMAP_Pos) /*!< 0x00000040 */
#define AFIO_PCF0_TIMER0_REMAP_PARTIALREMAP    AFIO_PCF0_TIMER0_REMAP_PARTIALREMAP_Msk /*!< Partial remap (ETR/PA12, CH1/PA8, CH2/PA9, CH3/PA10, CH4/PA11, BKIN/PA6, CH1N/PA7, CH2N/PB0, CH3N/PB1) */
#define AFIO_PCF0_TIMER0_REMAP_FULLREMAP_Pos   (6U)                              
#define AFIO_PCF0_TIMER0_REMAP_FULLREMAP_Msk   (0x3U << AFIO_PCF0_TIMER0_REMAP_FULLREMAP_Pos) /*!< 0x000000C0 */
#define AFIO_PCF0_TIMER0_REMAP_FULLREMAP       AFIO_PCF0_TIMER0_REMAP_FULLREMAP_Msk /*!< Full remap (ETR/PE7, CH1/PE9, CH2/PE11, CH3/PE13, CH4/PE14, BKIN/PE15, CH1N/PE8, CH2N/PE10, CH3N/PE12) */

#define AFIO_PCF0_TIMER1_REMAP_Pos             (8U)                              
#define AFIO_PCF0_TIMER1_REMAP_Msk             (0x3U << AFIO_PCF0_TIMER1_REMAP_Pos) /*!< 0x00000300 */
#define AFIO_PCF0_TIMER1_REMAP                 AFIO_PCF0_TIMER1_REMAP_Msk          /*!< TIM2_REMAP[1:0] bits (TIM2 remapping) */
#define AFIO_PCF0_TIMER1_REMAP_0               (0x1U << AFIO_PCF0_TIMER1_REMAP_Pos) /*!< 0x00000100 */
#define AFIO_PCF0_TIMER1_REMAP_1               (0x2U << AFIO_PCF0_TIMER1_REMAP_Pos) /*!< 0x00000200 */

/*!< TIM2_REMAP configuration */
#define AFIO_PCF0_TIMER1_REMAP_NOREMAP         0x00000000U                          /*!< No remap (CH1/ETR/PA0, CH2/PA1, CH3/PA2, CH4/PA3) */
#define AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP1_Pos (8U)                            
#define AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP1_Msk (0x1U << AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP1_Pos) /*!< 0x00000100 */
#define AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP1   AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP1_Msk /*!< Partial remap (CH1/ETR/PA15, CH2/PB3, CH3/PA2, CH4/PA3) */
#define AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP2_Pos (9U)                            
#define AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP2_Msk (0x1U << AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP2_Pos) /*!< 0x00000200 */
#define AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP2   AFIO_PCF0_TIMER1_REMAP_PARTIALREMAP2_Msk /*!< Partial remap (CH1/ETR/PA0, CH2/PA1, CH3/PB10, CH4/PB11) */
#define AFIO_PCF0_TIMER1_REMAP_FULLREMAP_Pos   (8U)                              
#define AFIO_PCF0_TIMER1_REMAP_FULLREMAP_Msk   (0x3U << AFIO_PCF0_TIMER1_REMAP_FULLREMAP_Pos) /*!< 0x00000300 */
#define AFIO_PCF0_TIMER1_REMAP_FULLREMAP       AFIO_PCF0_TIMER1_REMAP_FULLREMAP_Msk /*!< Full remap (CH1/ETR/PA15, CH2/PB3, CH3/PB10, CH4/PB11) */

#define AFIO_PCF0_TIMER2_REMAP_Pos             (10U)                             
#define AFIO_PCF0_TIMER2_REMAP_Msk             (0x3U << AFIO_PCF0_TIMER2_REMAP_Pos) /*!< 0x00000C00 */
#define AFIO_PCF0_TIMER2_REMAP                 AFIO_PCF0_TIMER2_REMAP_Msk          /*!< TIM3_REMAP[1:0] bits (TIM3 remapping) */
#define AFIO_PCF0_TIMER2_REMAP_0               (0x1U << AFIO_PCF0_TIMER2_REMAP_Pos) /*!< 0x00000400 */
#define AFIO_PCF0_TIMER2_REMAP_1               (0x2U << AFIO_PCF0_TIMER2_REMAP_Pos) /*!< 0x00000800 */

/*!< TIM3_REMAP configuration */
#define AFIO_PCF0_TIMER2_REMAP_NOREMAP         0x00000000U                          /*!< No remap (CH1/PA6, CH2/PA7, CH3/PB0, CH4/PB1) */
#define AFIO_PCF0_TIMER2_REMAP_PARTIALREMAP_Pos (11U)                            
#define AFIO_PCF0_TIMER2_REMAP_PARTIALREMAP_Msk (0x1U << AFIO_PCF0_TIMER2_REMAP_PARTIALREMAP_Pos) /*!< 0x00000800 */
#define AFIO_PCF0_TIMER2_REMAP_PARTIALREMAP    AFIO_PCF0_TIMER2_REMAP_PARTIALREMAP_Msk /*!< Partial remap (CH1/PB4, CH2/PB5, CH3/PB0, CH4/PB1) */
#define AFIO_PCF0_TIMER2_REMAP_FULLREMAP_Pos   (10U)                             
#define AFIO_PCF0_TIMER2_REMAP_FULLREMAP_Msk   (0x3U << AFIO_PCF0_TIMER2_REMAP_FULLREMAP_Pos) /*!< 0x00000C00 */
#define AFIO_PCF0_TIMER2_REMAP_FULLREMAP       AFIO_PCF0_TIMER2_REMAP_FULLREMAP_Msk /*!< Full remap (CH1/PC6, CH2/PC7, CH3/PC8, CH4/PC9) */

#define AFIO_PCF0_TIMER3_REMAP_Pos             (12U)                             
#define AFIO_PCF0_TIMER3_REMAP_Msk             (0x1U << AFIO_PCF0_TIMER3_REMAP_Pos) /*!< 0x00001000 */
#define AFIO_PCF0_TIMER3_REMAP                 AFIO_PCF0_TIMER3_REMAP_Msk          /*!< TIM4_REMAP bit (TIM4 remapping) */

#define AFIO_PCF0_CAN0_REMAP_Pos              (13U)                             
#define AFIO_PCF0_CAN0_REMAP_Msk              (0x3U << AFIO_PCF0_CAN0_REMAP_Pos) /*!< 0x00006000 */
#define AFIO_PCF0_CAN0_REMAP                  AFIO_PCF0_CAN0_REMAP_Msk           /*!< CAN_REMAP[1:0] bits (CAN Alternate function remapping) */
#define AFIO_PCF0_CAN0_REMAP_0                (0x1U << AFIO_PCF0_CAN0_REMAP_Pos) /*!< 0x00002000 */
#define AFIO_PCF0_CAN0_REMAP_1                (0x2U << AFIO_PCF0_CAN0_REMAP_Pos) /*!< 0x00004000 */

/*!< CAN_REMAP configuration */
#define AFIO_PCF0_CAN0_REMAP_REMAP1           0x00000000U                          /*!< CANRX mapped to PA11, CANTX mapped to PA12 */
#define AFIO_PCF0_CAN0_REMAP_REMAP2_Pos       (14U)                             
#define AFIO_PCF0_CAN0_REMAP_REMAP2_Msk       (0x1U << AFIO_PCF0_CAN0_REMAP_REMAP2_Pos) /*!< 0x00004000 */
#define AFIO_PCF0_CAN0_REMAP_REMAP2           AFIO_PCF0_CAN0_REMAP_REMAP2_Msk    /*!< CANRX mapped to PB8, CANTX mapped to PB9 */
#define AFIO_PCF0_CAN0_REMAP_REMAP3_Pos       (13U)                             
#define AFIO_PCF0_CAN0_REMAP_REMAP3_Msk       (0x3U << AFIO_PCF0_CAN0_REMAP_REMAP3_Pos) /*!< 0x00006000 */
#define AFIO_PCF0_CAN0_REMAP_REMAP3           AFIO_PCF0_CAN0_REMAP_REMAP3_Msk    /*!< CANRX mapped to PD0, CANTX mapped to PD1 */

#define AFIO_PCF0_PD01_REMAP_Pos             (15U)                             
#define AFIO_PCF0_PD01_REMAP_Msk             (0x1U << AFIO_PCF0_PD01_REMAP_Pos) /*!< 0x00008000 */
#define AFIO_PCF0_PD01_REMAP                 AFIO_PCF0_PD01_REMAP_Msk          /*!< Port D0/Port D1 mapping on OSC_IN/OSC_OUT */

#define AFIO_PCF0_TIM4CH3_IREMAP_Pos         (16U)                             
#define AFIO_PCF0_TIM4CH3_IREMAP_Msk         (0x1U << AFIO_PCF0_TIM4CH3_IREMAP_Pos) /*!< 0x00010000 */
#define AFIO_PCF0_TIM4CH3_IREMAP             AFIO_PCF0_TIM4CH3_IREMAP_Msk      /*!< TIM5 Channel4 Internal Remap */

/*!< SWJ_CFG configuration */
#define AFIO_PCF0_SWJ_CFG_Pos                (24U)                             
#define AFIO_PCF0_SWJ_CFG_Msk                (0x7U << AFIO_PCF0_SWJ_CFG_Pos)   /*!< 0x07000000 */
#define AFIO_PCF0_SWJ_CFG                    AFIO_PCF0_SWJ_CFG_Msk             /*!< SWJ_CFG[2:0] bits (Serial Wire JTAG configuration) */
#define AFIO_PCF0_SWJ_CFG_0                  (0x1U << AFIO_PCF0_SWJ_CFG_Pos)   /*!< 0x01000000 */
#define AFIO_PCF0_SWJ_CFG_1                  (0x2U << AFIO_PCF0_SWJ_CFG_Pos)   /*!< 0x02000000 */
#define AFIO_PCF0_SWJ_CFG_2                  (0x4U << AFIO_PCF0_SWJ_CFG_Pos)   /*!< 0x04000000 */

#define AFIO_PCF0_SWJ_CFG_RESET              0x00000000U                          /*!< Full SWJ (JTAG-DP + SW-DP) : Reset State */
#define AFIO_PCF0_SWJ_CFG_NOJNTRST_Pos       (24U)                             
#define AFIO_PCF0_SWJ_CFG_NOJNTRST_Msk       (0x1U << AFIO_PCF0_SWJ_CFG_NOJNTRST_Pos) /*!< 0x01000000 */
#define AFIO_PCF0_SWJ_CFG_NOJNTRST           AFIO_PCF0_SWJ_CFG_NOJNTRST_Msk    /*!< Full SWJ (JTAG-DP + SW-DP) but without JNTRST */
#define AFIO_PCF0_SWJ_CFG_JTAGDISABLE_Pos    (25U)                             
#define AFIO_PCF0_SWJ_CFG_JTAGDISABLE_Msk    (0x1U << AFIO_PCF0_SWJ_CFG_JTAGDISABLE_Pos) /*!< 0x02000000 */
#define AFIO_PCF0_SWJ_CFG_JTAGDISABLE        AFIO_PCF0_SWJ_CFG_JTAGDISABLE_Msk /*!< JTAG-DP Disabled and SW-DP Enabled */
#define AFIO_PCF0_SWJ_CFG_DISABLE_Pos        (26U)                             
#define AFIO_PCF0_SWJ_CFG_DISABLE_Msk        (0x1U << AFIO_PCF0_SWJ_CFG_DISABLE_Pos) /*!< 0x04000000 */
#define AFIO_PCF0_SWJ_CFG_DISABLE            AFIO_PCF0_SWJ_CFG_DISABLE_Msk     /*!< JTAG-DP Disabled and SW-DP Disabled */

/*!< CAN2_REMAP configuration */
#define AFIO_PCF0_CAN1_REMAP_Pos             (22U)                             
#define AFIO_PCF0_CAN1_REMAP_Msk             (0x1U << AFIO_PCF0_CAN1_REMAP_Pos) /*!< 0x00400000 */
#define AFIO_PCF0_CAN1_REMAP                 AFIO_PCF0_CAN1_REMAP_Msk          /*!< CAN2_REMAP bit (CAN2 I/O remapping) */

/*!< SPI2_REMAP configuration */
#define AFIO_PCF0_SPI2_REMAP_Pos             (28U)                             
#define AFIO_PCF0_SPI2_REMAP_Msk             (0x1U << AFIO_PCF0_SPI2_REMAP_Pos) /*!< 0x10000000 */
#define AFIO_PCF0_SPI2_REMAP                 AFIO_PCF0_SPI2_REMAP_Msk          /*!< SPI2_REMAP bit (SPI2 remapping) */

/*!< TIM2ITR1_IREMAP configuration */
#define AFIO_PCF0_TIM1ITR1_IREMAP_Pos        (29U)                             
#define AFIO_PCF0_TIM1ITR1_IREMAP_Msk        (0x1U << AFIO_PCF0_TIM1ITR1_IREMAP_Pos) /*!< 0x20000000 */
#define AFIO_PCF0_TIM1ITR1_IREMAP            AFIO_PCF0_TIM1ITR1_IREMAP_Msk     /*!< TIM2ITR1_IREMAP bit (TIM2 internal trigger 1 remapping) */

/*****************  Bit definition for AFIO_EXTISS0 register  *****************/
#define AFIO_EXTISS0_EXTI0_SS_Pos               (0U)                              
#define AFIO_EXTISS0_EXTI0_SS_Msk               (0xFU << AFIO_EXTISS0_EXTI0_SS_Pos)  /*!< 0x0000000F */
#define AFIO_EXTISS0_EXTI0_SS                   AFIO_EXTISS0_EXTI0_SS_Msk            /*!< EXTI 0 configuration */
#define AFIO_EXTISS0_EXTI1_SS_Pos               (4U)                              
#define AFIO_EXTISS0_EXTI1_SS_Msk               (0xFU << AFIO_EXTISS0_EXTI1_SS_Pos)  /*!< 0x000000F0 */
#define AFIO_EXTISS0_EXTI1_SS                   AFIO_EXTISS0_EXTI1_SS_Msk            /*!< EXTI 1 configuration */
#define AFIO_EXTISS0_EXTI2_SS_Pos               (8U)                              
#define AFIO_EXTISS0_EXTI2_SS_Msk               (0xFU << AFIO_EXTISS0_EXTI2_SS_Pos)  /*!< 0x00000F00 */
#define AFIO_EXTISS0_EXTI2_SS                   AFIO_EXTISS0_EXTI2_SS_Msk            /*!< EXTI 2 configuration */
#define AFIO_EXTISS0_EXTI3_SS_Pos               (12U)                             
#define AFIO_EXTISS0_EXTI3_SS_Msk               (0xFU << AFIO_EXTISS0_EXTI3_SS_Pos)  /*!< 0x0000F000 */
#define AFIO_EXTISS0_EXTI3_SS                   AFIO_EXTISS0_EXTI3_SS_Msk            /*!< EXTI 3 configuration */

/*!< EXTI0 configuration */
#define AFIO_EXTISS0_EXTI0_SS_PA                0x00000000U                          /*!< PA[0] pin */
#define AFIO_EXTISS0_EXTI0_SS_PB_Pos            (0U)                              
#define AFIO_EXTISS0_EXTI0_SS_PB_Msk            (0x1U << AFIO_EXTISS0_EXTI0_SS_PB_Pos) /*!< 0x00000001 */
#define AFIO_EXTISS0_EXTI0_SS_PB                AFIO_EXTISS0_EXTI0_SS_PB_Msk         /*!< PB[0] pin */
#define AFIO_EXTISS0_EXTI0_SS_PC_Pos            (1U)                              
#define AFIO_EXTISS0_EXTI0_SS_PC_Msk            (0x1U << AFIO_EXTISS0_EXTI0_SS_PC_Pos) /*!< 0x00000002 */
#define AFIO_EXTISS0_EXTI0_SS_PC                AFIO_EXTISS0_EXTI0_SS_PC_Msk         /*!< PC[0] pin */
#define AFIO_EXTISS0_EXTI0_SS_PD_Pos            (0U)                              
#define AFIO_EXTISS0_EXTI0_SS_PD_Msk            (0x3U << AFIO_EXTISS0_EXTI0_SS_PD_Pos) /*!< 0x00000003 */
#define AFIO_EXTISS0_EXTI0_SS_PD                AFIO_EXTISS0_EXTI0_SS_PD_Msk         /*!< PD[0] pin */
#define AFIO_EXTISS0_EXTI0_SS_PE_Pos            (2U)                              
#define AFIO_EXTISS0_EXTI0_SS_PE_Msk            (0x1U << AFIO_EXTISS0_EXTI0_SS_PE_Pos) /*!< 0x00000004 */
#define AFIO_EXTISS0_EXTI0_SS_PE                AFIO_EXTISS0_EXTI0_SS_PE_Msk         /*!< PE[0] pin */
#define AFIO_EXTISS0_EXTI0_SS_PF_Pos            (0U)                              
#define AFIO_EXTISS0_EXTI0_SS_PF_Msk            (0x5U << AFIO_EXTISS0_EXTI0_SS_PF_Pos) /*!< 0x00000005 */
#define AFIO_EXTISS0_EXTI0_SS_PF                AFIO_EXTISS0_EXTI0_SS_PF_Msk         /*!< PF[0] pin */
#define AFIO_EXTISS0_EXTI0_SS_PG_Pos            (1U)                              
#define AFIO_EXTISS0_EXTI0_SS_PG_Msk            (0x3U << AFIO_EXTISS0_EXTI0_SS_PG_Pos) /*!< 0x00000006 */
#define AFIO_EXTISS0_EXTI0_SS_PG                AFIO_EXTISS0_EXTI0_SS_PG_Msk         /*!< PG[0] pin */

/*!< EXTI1 configuration */
#define AFIO_EXTISS0_EXTI1_SS_PA                0x00000000U                          /*!< PA[1] pin */
#define AFIO_EXTISS0_EXTI1_SS_PB_Pos            (4U)                              
#define AFIO_EXTISS0_EXTI1_SS_PB_Msk            (0x1U << AFIO_EXTISS0_EXTI1_SS_PB_Pos) /*!< 0x00000010 */
#define AFIO_EXTISS0_EXTI1_SS_PB                AFIO_EXTISS0_EXTI1_SS_PB_Msk         /*!< PB[1] pin */
#define AFIO_EXTISS0_EXTI1_SS_PC_Pos            (5U)                              
#define AFIO_EXTISS0_EXTI1_SS_PC_Msk            (0x1U << AFIO_EXTISS0_EXTI1_SS_PC_Pos) /*!< 0x00000020 */
#define AFIO_EXTISS0_EXTI1_SS_PC                AFIO_EXTISS0_EXTI1_SS_PC_Msk         /*!< PC[1] pin */
#define AFIO_EXTISS0_EXTI1_SS_PD_Pos            (4U)                              
#define AFIO_EXTISS0_EXTI1_SS_PD_Msk            (0x3U << AFIO_EXTISS0_EXTI1_SS_PD_Pos) /*!< 0x00000030 */
#define AFIO_EXTISS0_EXTI1_SS_PD                AFIO_EXTISS0_EXTI1_SS_PD_Msk         /*!< PD[1] pin */
#define AFIO_EXTISS0_EXTI1_SS_PE_Pos            (6U)                              
#define AFIO_EXTISS0_EXTI1_SS_PE_Msk            (0x1U << AFIO_EXTISS0_EXTI1_SS_PE_Pos) /*!< 0x00000040 */
#define AFIO_EXTISS0_EXTI1_SS_PE                AFIO_EXTISS0_EXTI1_SS_PE_Msk         /*!< PE[1] pin */
#define AFIO_EXTISS0_EXTI1_SS_PF_Pos            (4U)                              
#define AFIO_EXTISS0_EXTI1_SS_PF_Msk            (0x5U << AFIO_EXTISS0_EXTI1_SS_PF_Pos) /*!< 0x00000050 */
#define AFIO_EXTISS0_EXTI1_SS_PF                AFIO_EXTISS0_EXTI1_SS_PF_Msk         /*!< PF[1] pin */
#define AFIO_EXTISS0_EXTI1_SS_PG_Pos            (5U)                              
#define AFIO_EXTISS0_EXTI1_SS_PG_Msk            (0x3U << AFIO_EXTISS0_EXTI1_SS_PG_Pos) /*!< 0x00000060 */
#define AFIO_EXTISS0_EXTI1_SS_PG                AFIO_EXTISS0_EXTI1_SS_PG_Msk         /*!< PG[1] pin */

/*!< EXTI2 configuration */  
#define AFIO_EXTISS0_EXTI2_SS_PA                0x00000000U                          /*!< PA[2] pin */
#define AFIO_EXTISS0_EXTI2_SS_PB_Pos            (8U)                              
#define AFIO_EXTISS0_EXTI2_SS_PB_Msk            (0x1U << AFIO_EXTISS0_EXTI2_SS_PB_Pos) /*!< 0x00000100 */
#define AFIO_EXTISS0_EXTI2_SS_PB                AFIO_EXTISS0_EXTI2_SS_PB_Msk         /*!< PB[2] pin */
#define AFIO_EXTISS0_EXTI2_SS_PC_Pos            (9U)                              
#define AFIO_EXTISS0_EXTI2_SS_PC_Msk            (0x1U << AFIO_EXTISS0_EXTI2_SS_PC_Pos) /*!< 0x00000200 */
#define AFIO_EXTISS0_EXTI2_SS_PC                AFIO_EXTISS0_EXTI2_SS_PC_Msk         /*!< PC[2] pin */
#define AFIO_EXTISS0_EXTI2_SS_PD_Pos            (8U)                              
#define AFIO_EXTISS0_EXTI2_SS_PD_Msk            (0x3U << AFIO_EXTISS0_EXTI2_SS_PD_Pos) /*!< 0x00000300 */
#define AFIO_EXTISS0_EXTI2_SS_PD                AFIO_EXTISS0_EXTI2_SS_PD_Msk         /*!< PD[2] pin */
#define AFIO_EXTISS0_EXTI2_SS_PE_Pos            (10U)                             
#define AFIO_EXTISS0_EXTI2_SS_PE_Msk            (0x1U << AFIO_EXTISS0_EXTI2_SS_PE_Pos) /*!< 0x00000400 */
#define AFIO_EXTISS0_EXTI2_SS_PE                AFIO_EXTISS0_EXTI2_SS_PE_Msk         /*!< PE[2] pin */
#define AFIO_EXTISS0_EXTI2_SS_PF_Pos            (8U)                              
#define AFIO_EXTISS0_EXTI2_SS_PF_Msk            (0x5U << AFIO_EXTISS0_EXTI2_SS_PF_Pos) /*!< 0x00000500 */
#define AFIO_EXTISS0_EXTI2_SS_PF                AFIO_EXTISS0_EXTI2_SS_PF_Msk         /*!< PF[2] pin */
#define AFIO_EXTISS0_EXTI2_SS_PG_Pos            (9U)                              
#define AFIO_EXTISS0_EXTI2_SS_PG_Msk            (0x3U << AFIO_EXTISS0_EXTI2_SS_PG_Pos) /*!< 0x00000600 */
#define AFIO_EXTISS0_EXTI2_SS_PG                AFIO_EXTISS0_EXTI2_SS_PG_Msk         /*!< PG[2] pin */

/*!< EXTI3 configuration */
#define AFIO_EXTISS0_EXTI3_SS_PA                0x00000000U                          /*!< PA[3] pin */
#define AFIO_EXTISS0_EXTI3_SS_PB_Pos            (12U)                             
#define AFIO_EXTISS0_EXTI3_SS_PB_Msk            (0x1U << AFIO_EXTISS0_EXTI3_SS_PB_Pos) /*!< 0x00001000 */
#define AFIO_EXTISS0_EXTI3_SS_PB                AFIO_EXTISS0_EXTI3_SS_PB_Msk         /*!< PB[3] pin */
#define AFIO_EXTISS0_EXTI3_SS_PC_Pos            (13U)                             
#define AFIO_EXTISS0_EXTI3_SS_PC_Msk            (0x1U << AFIO_EXTISS0_EXTI3_SS_PC_Pos) /*!< 0x00002000 */
#define AFIO_EXTISS0_EXTI3_SS_PC                AFIO_EXTISS0_EXTI3_SS_PC_Msk         /*!< PC[3] pin */
#define AFIO_EXTISS0_EXTI3_SS_PD_Pos            (12U)                             
#define AFIO_EXTISS0_EXTI3_SS_PD_Msk            (0x3U << AFIO_EXTISS0_EXTI3_SS_PD_Pos) /*!< 0x00003000 */
#define AFIO_EXTISS0_EXTI3_SS_PD                AFIO_EXTISS0_EXTI3_SS_PD_Msk         /*!< PD[3] pin */
#define AFIO_EXTISS0_EXTI3_SS_PE_Pos            (14U)                             
#define AFIO_EXTISS0_EXTI3_SS_PE_Msk            (0x1U << AFIO_EXTISS0_EXTI3_SS_PE_Pos) /*!< 0x00004000 */
#define AFIO_EXTISS0_EXTI3_SS_PE                AFIO_EXTISS0_EXTI3_SS_PE_Msk         /*!< PE[3] pin */
#define AFIO_EXTISS0_EXTI3_SS_PF_Pos            (12U)                             
#define AFIO_EXTISS0_EXTI3_SS_PF_Msk            (0x5U << AFIO_EXTISS0_EXTI3_SS_PF_Pos) /*!< 0x00005000 */
#define AFIO_EXTISS0_EXTI3_SS_PF                AFIO_EXTISS0_EXTI3_SS_PF_Msk         /*!< PF[3] pin */
#define AFIO_EXTISS0_EXTI3_SS_PG_Pos            (13U)                             
#define AFIO_EXTISS0_EXTI3_SS_PG_Msk            (0x3U << AFIO_EXTISS0_EXTI3_SS_PG_Pos) /*!< 0x00006000 */
#define AFIO_EXTISS0_EXTI3_SS_PG                AFIO_EXTISS0_EXTI3_SS_PG_Msk         /*!< PG[3] pin */

/*****************  Bit definition for AFIO_EXTISS1 register  *****************/
#define AFIO_EXTISS1_EXTI4_SS_Pos               (0U)                              
#define AFIO_EXTISS1_EXTI4_SS_Msk               (0xFU << AFIO_EXTISS1_EXTI4_SS_Pos)  /*!< 0x0000000F */
#define AFIO_EXTISS1_EXTI4_SS                   AFIO_EXTISS1_EXTI4_SS_Msk            /*!< EXTI 4 configuration */
#define AFIO_EXTISS1_EXTI5_SS_Pos               (4U)                              
#define AFIO_EXTISS1_EXTI5_SS_Msk               (0xFU << AFIO_EXTISS1_EXTI5_SS_Pos)  /*!< 0x000000F0 */
#define AFIO_EXTISS1_EXTI5_SS                   AFIO_EXTISS1_EXTI5_SS_Msk            /*!< EXTI 5 configuration */
#define AFIO_EXTISS1_EXTI6_SS_Pos               (8U)                              
#define AFIO_EXTISS1_EXTI6_SS_Msk               (0xFU << AFIO_EXTISS1_EXTI6_SS_Pos)  /*!< 0x00000F00 */
#define AFIO_EXTISS1_EXTI6_SS                   AFIO_EXTISS1_EXTI6_SS_Msk            /*!< EXTI 6 configuration */
#define AFIO_EXTISS1_EXTI7_SS_Pos               (12U)                             
#define AFIO_EXTISS1_EXTI7_SS_Msk               (0xFU << AFIO_EXTISS1_EXTI7_SS_Pos)  /*!< 0x0000F000 */
#define AFIO_EXTISS1_EXTI7_SS                   AFIO_EXTISS1_EXTI7_SS_Msk            /*!< EXTI 7 configuration */

/*!< EXTI4 configuration */
#define AFIO_EXTISS1_EXTI4_SS_PA                0x00000000U                          /*!< PA[4] pin */
#define AFIO_EXTISS1_EXTI4_SS_PB_Pos            (0U)                              
#define AFIO_EXTISS1_EXTI4_SS_PB_Msk            (0x1U << AFIO_EXTISS1_EXTI4_SS_PB_Pos) /*!< 0x00000001 */
#define AFIO_EXTISS1_EXTI4_SS_PB                AFIO_EXTISS1_EXTI4_SS_PB_Msk         /*!< PB[4] pin */
#define AFIO_EXTISS1_EXTI4_SS_PC_Pos            (1U)                              
#define AFIO_EXTISS1_EXTI4_SS_PC_Msk            (0x1U << AFIO_EXTISS1_EXTI4_SS_PC_Pos) /*!< 0x00000002 */
#define AFIO_EXTISS1_EXTI4_SS_PC                AFIO_EXTISS1_EXTI4_SS_PC_Msk         /*!< PC[4] pin */
#define AFIO_EXTISS1_EXTI4_SS_PD_Pos            (0U)                              
#define AFIO_EXTISS1_EXTI4_SS_PD_Msk            (0x3U << AFIO_EXTISS1_EXTI4_SS_PD_Pos) /*!< 0x00000003 */
#define AFIO_EXTISS1_EXTI4_SS_PD                AFIO_EXTISS1_EXTI4_SS_PD_Msk         /*!< PD[4] pin */
#define AFIO_EXTISS1_EXTI4_SS_PE_Pos            (2U)                              
#define AFIO_EXTISS1_EXTI4_SS_PE_Msk            (0x1U << AFIO_EXTISS1_EXTI4_SS_PE_Pos) /*!< 0x00000004 */
#define AFIO_EXTISS1_EXTI4_SS_PE                AFIO_EXTISS1_EXTI4_SS_PE_Msk         /*!< PE[4] pin */
#define AFIO_EXTISS1_EXTI4_SS_PF_Pos            (0U)                              
#define AFIO_EXTISS1_EXTI4_SS_PF_Msk            (0x5U << AFIO_EXTISS1_EXTI4_SS_PF_Pos) /*!< 0x00000005 */
#define AFIO_EXTISS1_EXTI4_SS_PF                AFIO_EXTISS1_EXTI4_SS_PF_Msk         /*!< PF[4] pin */
#define AFIO_EXTISS1_EXTI4_SS_PG_Pos            (1U)                              
#define AFIO_EXTISS1_EXTI4_SS_PG_Msk            (0x3U << AFIO_EXTISS1_EXTI4_SS_PG_Pos) /*!< 0x00000006 */
#define AFIO_EXTISS1_EXTI4_SS_PG                AFIO_EXTISS1_EXTI4_SS_PG_Msk         /*!< PG[4] pin */

/* EXTI5 configuration */
#define AFIO_EXTISS1_EXTI5_SS_PA                0x00000000U                          /*!< PA[5] pin */
#define AFIO_EXTISS1_EXTI5_SS_PB_Pos            (4U)                              
#define AFIO_EXTISS1_EXTI5_SS_PB_Msk            (0x1U << AFIO_EXTISS1_EXTI5_SS_PB_Pos) /*!< 0x00000010 */
#define AFIO_EXTISS1_EXTI5_SS_PB                AFIO_EXTISS1_EXTI5_SS_PB_Msk         /*!< PB[5] pin */
#define AFIO_EXTISS1_EXTI5_SS_PC_Pos            (5U)                              
#define AFIO_EXTISS1_EXTI5_SS_PC_Msk            (0x1U << AFIO_EXTISS1_EXTI5_SS_PC_Pos) /*!< 0x00000020 */
#define AFIO_EXTISS1_EXTI5_SS_PC                AFIO_EXTISS1_EXTI5_SS_PC_Msk         /*!< PC[5] pin */
#define AFIO_EXTISS1_EXTI5_SS_PD_Pos            (4U)                              
#define AFIO_EXTISS1_EXTI5_SS_PD_Msk            (0x3U << AFIO_EXTISS1_EXTI5_SS_PD_Pos) /*!< 0x00000030 */
#define AFIO_EXTISS1_EXTI5_SS_PD                AFIO_EXTISS1_EXTI5_SS_PD_Msk         /*!< PD[5] pin */
#define AFIO_EXTISS1_EXTI5_SS_PE_Pos            (6U)                              
#define AFIO_EXTISS1_EXTI5_SS_PE_Msk            (0x1U << AFIO_EXTISS1_EXTI5_SS_PE_Pos) /*!< 0x00000040 */
#define AFIO_EXTISS1_EXTI5_SS_PE                AFIO_EXTISS1_EXTI5_SS_PE_Msk         /*!< PE[5] pin */
#define AFIO_EXTISS1_EXTI5_SS_PF_Pos            (4U)                              
#define AFIO_EXTISS1_EXTI5_SS_PF_Msk            (0x5U << AFIO_EXTISS1_EXTI5_SS_PF_Pos) /*!< 0x00000050 */
#define AFIO_EXTISS1_EXTI5_SS_PF                AFIO_EXTISS1_EXTI5_SS_PF_Msk         /*!< PF[5] pin */
#define AFIO_EXTISS1_EXTI5_SS_PG_Pos            (5U)                              
#define AFIO_EXTISS1_EXTI5_SS_PG_Msk            (0x3U << AFIO_EXTISS1_EXTI5_SS_PG_Pos) /*!< 0x00000060 */
#define AFIO_EXTISS1_EXTI5_SS_PG                AFIO_EXTISS1_EXTI5_SS_PG_Msk         /*!< PG[5] pin */

/*!< EXTI6 configuration */  
#define AFIO_EXTISS1_EXTI6_SS_PA                0x00000000U                          /*!< PA[6] pin */
#define AFIO_EXTISS1_EXTI6_SS_PB_Pos            (8U)                              
#define AFIO_EXTISS1_EXTI6_SS_PB_Msk            (0x1U << AFIO_EXTISS1_EXTI6_SS_PB_Pos) /*!< 0x00000100 */
#define AFIO_EXTISS1_EXTI6_SS_PB                AFIO_EXTISS1_EXTI6_SS_PB_Msk         /*!< PB[6] pin */
#define AFIO_EXTISS1_EXTI6_SS_PC_Pos            (9U)                              
#define AFIO_EXTISS1_EXTI6_SS_PC_Msk            (0x1U << AFIO_EXTISS1_EXTI6_SS_PC_Pos) /*!< 0x00000200 */
#define AFIO_EXTISS1_EXTI6_SS_PC                AFIO_EXTISS1_EXTI6_SS_PC_Msk         /*!< PC[6] pin */
#define AFIO_EXTISS1_EXTI6_SS_PD_Pos            (8U)                              
#define AFIO_EXTISS1_EXTI6_SS_PD_Msk            (0x3U << AFIO_EXTISS1_EXTI6_SS_PD_Pos) /*!< 0x00000300 */
#define AFIO_EXTISS1_EXTI6_SS_PD                AFIO_EXTISS1_EXTI6_SS_PD_Msk         /*!< PD[6] pin */
#define AFIO_EXTISS1_EXTI6_SS_PE_Pos            (10U)                             
#define AFIO_EXTISS1_EXTI6_SS_PE_Msk            (0x1U << AFIO_EXTISS1_EXTI6_SS_PE_Pos) /*!< 0x00000400 */
#define AFIO_EXTISS1_EXTI6_SS_PE                AFIO_EXTISS1_EXTI6_SS_PE_Msk         /*!< PE[6] pin */
#define AFIO_EXTISS1_EXTI6_SS_PF_Pos            (8U)                              
#define AFIO_EXTISS1_EXTI6_SS_PF_Msk            (0x5U << AFIO_EXTISS1_EXTI6_SS_PF_Pos) /*!< 0x00000500 */
#define AFIO_EXTISS1_EXTI6_SS_PF                AFIO_EXTISS1_EXTI6_SS_PF_Msk         /*!< PF[6] pin */
#define AFIO_EXTISS1_EXTI6_SS_PG_Pos            (9U)                              
#define AFIO_EXTISS1_EXTI6_SS_PG_Msk            (0x3U << AFIO_EXTISS1_EXTI6_SS_PG_Pos) /*!< 0x00000600 */
#define AFIO_EXTISS1_EXTI6_SS_PG                AFIO_EXTISS1_EXTI6_SS_PG_Msk         /*!< PG[6] pin */

/*!< EXTI7 configuration */
#define AFIO_EXTISS1_EXTI7_SS_PA                0x00000000U                          /*!< PA[7] pin */
#define AFIO_EXTISS1_EXTI7_SS_PB_Pos            (12U)                             
#define AFIO_EXTISS1_EXTI7_SS_PB_Msk            (0x1U << AFIO_EXTISS1_EXTI7_SS_PB_Pos) /*!< 0x00001000 */
#define AFIO_EXTISS1_EXTI7_SS_PB                AFIO_EXTISS1_EXTI7_SS_PB_Msk         /*!< PB[7] pin */
#define AFIO_EXTISS1_EXTI7_SS_PC_Pos            (13U)                             
#define AFIO_EXTISS1_EXTI7_SS_PC_Msk            (0x1U << AFIO_EXTISS1_EXTI7_SS_PC_Pos) /*!< 0x00002000 */
#define AFIO_EXTISS1_EXTI7_SS_PC                AFIO_EXTISS1_EXTI7_SS_PC_Msk         /*!< PC[7] pin */
#define AFIO_EXTISS1_EXTI7_SS_PD_Pos            (12U)                             
#define AFIO_EXTISS1_EXTI7_SS_PD_Msk            (0x3U << AFIO_EXTISS1_EXTI7_SS_PD_Pos) /*!< 0x00003000 */
#define AFIO_EXTISS1_EXTI7_SS_PD                AFIO_EXTISS1_EXTI7_SS_PD_Msk         /*!< PD[7] pin */
#define AFIO_EXTISS1_EXTI7_SS_PE_Pos            (14U)                             
#define AFIO_EXTISS1_EXTI7_SS_PE_Msk            (0x1U << AFIO_EXTISS1_EXTI7_SS_PE_Pos) /*!< 0x00004000 */
#define AFIO_EXTISS1_EXTI7_SS_PE                AFIO_EXTISS1_EXTI7_SS_PE_Msk         /*!< PE[7] pin */
#define AFIO_EXTISS1_EXTI7_SS_PF_Pos            (12U)                             
#define AFIO_EXTISS1_EXTI7_SS_PF_Msk            (0x5U << AFIO_EXTISS1_EXTI7_SS_PF_Pos) /*!< 0x00005000 */
#define AFIO_EXTISS1_EXTI7_SS_PF                AFIO_EXTISS1_EXTI7_SS_PF_Msk         /*!< PF[7] pin */
#define AFIO_EXTISS1_EXTI7_SS_PG_Pos            (13U)                             
#define AFIO_EXTISS1_EXTI7_SS_PG_Msk            (0x3U << AFIO_EXTISS1_EXTI7_SS_PG_Pos) /*!< 0x00006000 */
#define AFIO_EXTISS1_EXTI7_SS_PG                AFIO_EXTISS1_EXTI7_SS_PG_Msk         /*!< PG[7] pin */

/*****************  Bit definition for AFIO_EXTISS2 register  *****************/
#define AFIO_EXTISS2_EXTI8_SS_Pos               (0U)                              
#define AFIO_EXTISS2_EXTI8_SS_Msk               (0xFU << AFIO_EXTISS2_EXTI8_SS_Pos)  /*!< 0x0000000F */
#define AFIO_EXTISS2_EXTI8_SS                   AFIO_EXTISS2_EXTI8_SS_Msk            /*!< EXTI 8 configuration */
#define AFIO_EXTISS2_EXTI9_SS_Pos               (4U)                              
#define AFIO_EXTISS2_EXTI9_SS_Msk               (0xFU << AFIO_EXTISS2_EXTI9_SS_Pos)  /*!< 0x000000F0 */
#define AFIO_EXTISS2_EXTI9_SS                   AFIO_EXTISS2_EXTI9_SS_Msk            /*!< EXTI 9 configuration */
#define AFIO_EXTISS2_EXTI10_SS_Pos              (8U)                              
#define AFIO_EXTISS2_EXTI10_SS_Msk              (0xFU << AFIO_EXTISS2_EXTI10_SS_Pos) /*!< 0x00000F00 */
#define AFIO_EXTISS2_EXTI10_SS                  AFIO_EXTISS2_EXTI10_SS_Msk           /*!< EXTI 10 configuration */
#define AFIO_EXTISS2_EXTI11_SS_Pos              (12U)                             
#define AFIO_EXTISS2_EXTI11_SS_Msk              (0xFU << AFIO_EXTISS2_EXTI11_SS_Pos) /*!< 0x0000F000 */
#define AFIO_EXTISS2_EXTI11_SS                  AFIO_EXTISS2_EXTI11_SS_Msk           /*!< EXTI 11 configuration */

/*!< EXTI8 configuration */
#define AFIO_EXTISS2_EXTI8_SS_PA                0x00000000U                          /*!< PA[8] pin */
#define AFIO_EXTISS2_EXTI8_SS_PB_Pos            (0U)                              
#define AFIO_EXTISS2_EXTI8_SS_PB_Msk            (0x1U << AFIO_EXTISS2_EXTI8_SS_PB_Pos) /*!< 0x00000001 */
#define AFIO_EXTISS2_EXTI8_SS_PB                AFIO_EXTISS2_EXTI8_SS_PB_Msk         /*!< PB[8] pin */
#define AFIO_EXTISS2_EXTI8_SS_PC_Pos            (1U)                              
#define AFIO_EXTISS2_EXTI8_SS_PC_Msk            (0x1U << AFIO_EXTISS2_EXTI8_SS_PC_Pos) /*!< 0x00000002 */
#define AFIO_EXTISS2_EXTI8_SS_PC                AFIO_EXTISS2_EXTI8_SS_PC_Msk         /*!< PC[8] pin */
#define AFIO_EXTISS2_EXTI8_SS_PD_Pos            (0U)                              
#define AFIO_EXTISS2_EXTI8_SS_PD_Msk            (0x3U << AFIO_EXTISS2_EXTI8_SS_PD_Pos) /*!< 0x00000003 */
#define AFIO_EXTISS2_EXTI8_SS_PD                AFIO_EXTISS2_EXTI8_SS_PD_Msk         /*!< PD[8] pin */
#define AFIO_EXTISS2_EXTI8_SS_PE_Pos            (2U)                              
#define AFIO_EXTISS2_EXTI8_SS_PE_Msk            (0x1U << AFIO_EXTISS2_EXTI8_SS_PE_Pos) /*!< 0x00000004 */
#define AFIO_EXTISS2_EXTI8_SS_PE                AFIO_EXTISS2_EXTI8_SS_PE_Msk         /*!< PE[8] pin */
#define AFIO_EXTISS2_EXTI8_SS_PF_Pos            (0U)                              
#define AFIO_EXTISS2_EXTI8_SS_PF_Msk            (0x5U << AFIO_EXTISS2_EXTI8_SS_PF_Pos) /*!< 0x00000005 */
#define AFIO_EXTISS2_EXTI8_SS_PF                AFIO_EXTISS2_EXTI8_SS_PF_Msk         /*!< PF[8] pin */
#define AFIO_EXTISS2_EXTI8_SS_PG_Pos            (1U)                              
#define AFIO_EXTISS2_EXTI8_SS_PG_Msk            (0x3U << AFIO_EXTISS2_EXTI8_SS_PG_Pos) /*!< 0x00000006 */
#define AFIO_EXTISS2_EXTI8_SS_PG                AFIO_EXTISS2_EXTI8_SS_PG_Msk         /*!< PG[8] pin */

/*!< EXTI9 configuration */
#define AFIO_EXTISS2_EXTI9_SS_PA                0x00000000U                          /*!< PA[9] pin */
#define AFIO_EXTISS2_EXTI9_SS_PB_Pos            (4U)                              
#define AFIO_EXTISS2_EXTI9_SS_PB_Msk            (0x1U << AFIO_EXTISS2_EXTI9_SS_PB_Pos) /*!< 0x00000010 */
#define AFIO_EXTISS2_EXTI9_SS_PB                AFIO_EXTISS2_EXTI9_SS_PB_Msk         /*!< PB[9] pin */
#define AFIO_EXTISS2_EXTI9_SS_PC_Pos            (5U)                              
#define AFIO_EXTISS2_EXTI9_SS_PC_Msk            (0x1U << AFIO_EXTISS2_EXTI9_SS_PC_Pos) /*!< 0x00000020 */
#define AFIO_EXTISS2_EXTI9_SS_PC                AFIO_EXTISS2_EXTI9_SS_PC_Msk         /*!< PC[9] pin */
#define AFIO_EXTISS2_EXTI9_SS_PD_Pos            (4U)                              
#define AFIO_EXTISS2_EXTI9_SS_PD_Msk            (0x3U << AFIO_EXTISS2_EXTI9_SS_PD_Pos) /*!< 0x00000030 */
#define AFIO_EXTISS2_EXTI9_SS_PD                AFIO_EXTISS2_EXTI9_SS_PD_Msk         /*!< PD[9] pin */
#define AFIO_EXTISS2_EXTI9_SS_PE_Pos            (6U)                              
#define AFIO_EXTISS2_EXTI9_SS_PE_Msk            (0x1U << AFIO_EXTISS2_EXTI9_SS_PE_Pos) /*!< 0x00000040 */
#define AFIO_EXTISS2_EXTI9_SS_PE                AFIO_EXTISS2_EXTI9_SS_PE_Msk         /*!< PE[9] pin */
#define AFIO_EXTISS2_EXTI9_SS_PF_Pos            (4U)                              
#define AFIO_EXTISS2_EXTI9_SS_PF_Msk            (0x5U << AFIO_EXTISS2_EXTI9_SS_PF_Pos) /*!< 0x00000050 */
#define AFIO_EXTISS2_EXTI9_SS_PF                AFIO_EXTISS2_EXTI9_SS_PF_Msk         /*!< PF[9] pin */
#define AFIO_EXTISS2_EXTI9_SS_PG_Pos            (5U)                              
#define AFIO_EXTISS2_EXTI9_SS_PG_Msk            (0x3U << AFIO_EXTISS2_EXTI9_SS_PG_Pos) /*!< 0x00000060 */
#define AFIO_EXTISS2_EXTI9_SS_PG                AFIO_EXTISS2_EXTI9_SS_PG_Msk         /*!< PG[9] pin */

/*!< EXTI10 configuration */  
#define AFIO_EXTISS2_EXTI10_SS_PA               0x00000000U                          /*!< PA[10] pin */
#define AFIO_EXTISS2_EXTI10_SS_PB_Pos           (8U)                              
#define AFIO_EXTISS2_EXTI10_SS_PB_Msk           (0x1U << AFIO_EXTISS2_EXTI10_SS_PB_Pos) /*!< 0x00000100 */
#define AFIO_EXTISS2_EXTI10_SS_PB               AFIO_EXTISS2_EXTI10_SS_PB_Msk        /*!< PB[10] pin */
#define AFIO_EXTISS2_EXTI10_SS_PC_Pos           (9U)                              
#define AFIO_EXTISS2_EXTI10_SS_PC_Msk           (0x1U << AFIO_EXTISS2_EXTI10_SS_PC_Pos) /*!< 0x00000200 */
#define AFIO_EXTISS2_EXTI10_SS_PC               AFIO_EXTISS2_EXTI10_SS_PC_Msk        /*!< PC[10] pin */
#define AFIO_EXTISS2_EXTI10_SS_PD_Pos           (8U)                              
#define AFIO_EXTISS2_EXTI10_SS_PD_Msk           (0x3U << AFIO_EXTISS2_EXTI10_SS_PD_Pos) /*!< 0x00000300 */
#define AFIO_EXTISS2_EXTI10_SS_PD               AFIO_EXTISS2_EXTI10_SS_PD_Msk        /*!< PD[10] pin */
#define AFIO_EXTISS2_EXTI10_SS_PE_Pos           (10U)                             
#define AFIO_EXTISS2_EXTI10_SS_PE_Msk           (0x1U << AFIO_EXTISS2_EXTI10_SS_PE_Pos) /*!< 0x00000400 */
#define AFIO_EXTISS2_EXTI10_SS_PE               AFIO_EXTISS2_EXTI10_SS_PE_Msk        /*!< PE[10] pin */
#define AFIO_EXTISS2_EXTI10_SS_PF_Pos           (8U)                              
#define AFIO_EXTISS2_EXTI10_SS_PF_Msk           (0x5U << AFIO_EXTISS2_EXTI10_SS_PF_Pos) /*!< 0x00000500 */
#define AFIO_EXTISS2_EXTI10_SS_PF               AFIO_EXTISS2_EXTI10_SS_PF_Msk        /*!< PF[10] pin */
#define AFIO_EXTISS2_EXTI10_SS_PG_Pos           (9U)                              
#define AFIO_EXTISS2_EXTI10_SS_PG_Msk           (0x3U << AFIO_EXTISS2_EXTI10_SS_PG_Pos) /*!< 0x00000600 */
#define AFIO_EXTISS2_EXTI10_SS_PG               AFIO_EXTISS2_EXTI10_SS_PG_Msk        /*!< PG[10] pin */

/*!< EXTI11 configuration */
#define AFIO_EXTISS2_EXTI11_SS_PA               0x00000000U                          /*!< PA[11] pin */
#define AFIO_EXTISS2_EXTI11_SS_PB_Pos           (12U)                             
#define AFIO_EXTISS2_EXTI11_SS_PB_Msk           (0x1U << AFIO_EXTISS2_EXTI11_SS_PB_Pos) /*!< 0x00001000 */
#define AFIO_EXTISS2_EXTI11_SS_PB               AFIO_EXTISS2_EXTI11_SS_PB_Msk        /*!< PB[11] pin */
#define AFIO_EXTISS2_EXTI11_SS_PC_Pos           (13U)                             
#define AFIO_EXTISS2_EXTI11_SS_PC_Msk           (0x1U << AFIO_EXTISS2_EXTI11_SS_PC_Pos) /*!< 0x00002000 */
#define AFIO_EXTISS2_EXTI11_SS_PC               AFIO_EXTISS2_EXTI11_SS_PC_Msk        /*!< PC[11] pin */
#define AFIO_EXTISS2_EXTI11_SS_PD_Pos           (12U)                             
#define AFIO_EXTISS2_EXTI11_SS_PD_Msk           (0x3U << AFIO_EXTISS2_EXTI11_SS_PD_Pos) /*!< 0x00003000 */
#define AFIO_EXTISS2_EXTI11_SS_PD               AFIO_EXTISS2_EXTI11_SS_PD_Msk        /*!< PD[11] pin */
#define AFIO_EXTISS2_EXTI11_SS_PE_Pos           (14U)                             
#define AFIO_EXTISS2_EXTI11_SS_PE_Msk           (0x1U << AFIO_EXTISS2_EXTI11_SS_PE_Pos) /*!< 0x00004000 */
#define AFIO_EXTISS2_EXTI11_SS_PE               AFIO_EXTISS2_EXTI11_SS_PE_Msk        /*!< PE[11] pin */
#define AFIO_EXTISS2_EXTI11_SS_PF_Pos           (12U)                             
#define AFIO_EXTISS2_EXTI11_SS_PF_Msk           (0x5U << AFIO_EXTISS2_EXTI11_SS_PF_Pos) /*!< 0x00005000 */
#define AFIO_EXTISS2_EXTI11_SS_PF               AFIO_EXTISS2_EXTI11_SS_PF_Msk        /*!< PF[11] pin */
#define AFIO_EXTISS2_EXTI11_SS_PG_Pos           (13U)                             
#define AFIO_EXTISS2_EXTI11_SS_PG_Msk           (0x3U << AFIO_EXTISS2_EXTI11_SS_PG_Pos) /*!< 0x00006000 */
#define AFIO_EXTISS2_EXTI11_SS_PG               AFIO_EXTISS2_EXTI11_SS_PG_Msk        /*!< PG[11] pin */

/*****************  Bit definition for AFIO_EXTISS3 register  *****************/
#define AFIO_EXTISS3_EXTI12_SS_Pos              (0U)                              
#define AFIO_EXTISS3_EXTI12_SS_Msk              (0xFU << AFIO_EXTISS3_EXTI12_SS_Pos) /*!< 0x0000000F */
#define AFIO_EXTISS3_EXTI12_SS                  AFIO_EXTISS3_EXTI12_SS_Msk           /*!< EXTI 12 configuration */
#define AFIO_EXTISS3_EXTI13_SS_Pos              (4U)                              
#define AFIO_EXTISS3_EXTI13_SS_Msk              (0xFU << AFIO_EXTISS3_EXTI13_SS_Pos) /*!< 0x000000F0 */
#define AFIO_EXTISS3_EXTI13_SS                  AFIO_EXTISS3_EXTI13_SS_Msk           /*!< EXTI 13 configuration */
#define AFIO_EXTISS3_EXTI14_SS_Pos              (8U)                              
#define AFIO_EXTISS3_EXTI14_SS_Msk              (0xFU << AFIO_EXTISS3_EXTI14_SS_Pos) /*!< 0x00000F00 */
#define AFIO_EXTISS3_EXTI14_SS                  AFIO_EXTISS3_EXTI14_SS_Msk           /*!< EXTI 14 configuration */
#define AFIO_EXTISS3_EXTI15_SS_Pos              (12U)                             
#define AFIO_EXTISS3_EXTI15_SS_Msk              (0xFU << AFIO_EXTISS3_EXTI15_SS_Pos) /*!< 0x0000F000 */
#define AFIO_EXTISS3_EXTI15_SS                  AFIO_EXTISS3_EXTI15_SS_Msk           /*!< EXTI 15 configuration */

/* EXTI12 configuration */
#define AFIO_EXTISS3_EXTI12_SS_PA               0x00000000U                          /*!< PA[12] pin */
#define AFIO_EXTISS3_EXTI12_SS_PB_Pos           (0U)                              
#define AFIO_EXTISS3_EXTI12_SS_PB_Msk           (0x1U << AFIO_EXTISS3_EXTI12_SS_PB_Pos) /*!< 0x00000001 */
#define AFIO_EXTISS3_EXTI12_SS_PB               AFIO_EXTISS3_EXTI12_SS_PB_Msk        /*!< PB[12] pin */
#define AFIO_EXTISS3_EXTI12_SS_PC_Pos           (1U)                              
#define AFIO_EXTISS3_EXTI12_SS_PC_Msk           (0x1U << AFIO_EXTISS3_EXTI12_SS_PC_Pos) /*!< 0x00000002 */
#define AFIO_EXTISS3_EXTI12_SS_PC               AFIO_EXTISS3_EXTI12_SS_PC_Msk        /*!< PC[12] pin */
#define AFIO_EXTISS3_EXTI12_SS_PD_Pos           (0U)                              
#define AFIO_EXTISS3_EXTI12_SS_PD_Msk           (0x3U << AFIO_EXTISS3_EXTI12_SS_PD_Pos) /*!< 0x00000003 */
#define AFIO_EXTISS3_EXTI12_SS_PD               AFIO_EXTISS3_EXTI12_SS_PD_Msk        /*!< PD[12] pin */
#define AFIO_EXTISS3_EXTI12_SS_PE_Pos           (2U)                              
#define AFIO_EXTISS3_EXTI12_SS_PE_Msk           (0x1U << AFIO_EXTISS3_EXTI12_SS_PE_Pos) /*!< 0x00000004 */
#define AFIO_EXTISS3_EXTI12_SS_PE               AFIO_EXTISS3_EXTI12_SS_PE_Msk        /*!< PE[12] pin */
#define AFIO_EXTISS3_EXTI12_SS_PF_Pos           (0U)                              
#define AFIO_EXTISS3_EXTI12_SS_PF_Msk           (0x5U << AFIO_EXTISS3_EXTI12_SS_PF_Pos) /*!< 0x00000005 */
#define AFIO_EXTISS3_EXTI12_SS_PF               AFIO_EXTISS3_EXTI12_SS_PF_Msk        /*!< PF[12] pin */
#define AFIO_EXTISS3_EXTI12_SS_PG_Pos           (1U)                              
#define AFIO_EXTISS3_EXTI12_SS_PG_Msk           (0x3U << AFIO_EXTISS3_EXTI12_SS_PG_Pos) /*!< 0x00000006 */
#define AFIO_EXTISS3_EXTI12_SS_PG               AFIO_EXTISS3_EXTI12_SS_PG_Msk        /*!< PG[12] pin */

/* EXTI13 configuration */
#define AFIO_EXTISS3_EXTI13_SS_PA               0x00000000U                          /*!< PA[13] pin */
#define AFIO_EXTISS3_EXTI13_SS_PB_Pos           (4U)                              
#define AFIO_EXTISS3_EXTI13_SS_PB_Msk           (0x1U << AFIO_EXTISS3_EXTI13_SS_PB_Pos) /*!< 0x00000010 */
#define AFIO_EXTISS3_EXTI13_SS_PB               AFIO_EXTISS3_EXTI13_SS_PB_Msk        /*!< PB[13] pin */
#define AFIO_EXTISS3_EXTI13_SS_PC_Pos           (5U)                              
#define AFIO_EXTISS3_EXTI13_SS_PC_Msk           (0x1U << AFIO_EXTISS3_EXTI13_SS_PC_Pos) /*!< 0x00000020 */
#define AFIO_EXTISS3_EXTI13_SS_PC               AFIO_EXTISS3_EXTI13_SS_PC_Msk        /*!< PC[13] pin */
#define AFIO_EXTISS3_EXTI13_SS_PD_Pos           (4U)                              
#define AFIO_EXTISS3_EXTI13_SS_PD_Msk           (0x3U << AFIO_EXTISS3_EXTI13_SS_PD_Pos) /*!< 0x00000030 */
#define AFIO_EXTISS3_EXTI13_SS_PD               AFIO_EXTISS3_EXTI13_SS_PD_Msk        /*!< PD[13] pin */
#define AFIO_EXTISS3_EXTI13_SS_PE_Pos           (6U)                              
#define AFIO_EXTISS3_EXTI13_SS_PE_Msk           (0x1U << AFIO_EXTISS3_EXTI13_SS_PE_Pos) /*!< 0x00000040 */
#define AFIO_EXTISS3_EXTI13_SS_PE               AFIO_EXTISS3_EXTI13_SS_PE_Msk        /*!< PE[13] pin */
#define AFIO_EXTISS3_EXTI13_SS_PF_Pos           (4U)                              
#define AFIO_EXTISS3_EXTI13_SS_PF_Msk           (0x5U << AFIO_EXTISS3_EXTI13_SS_PF_Pos) /*!< 0x00000050 */
#define AFIO_EXTISS3_EXTI13_SS_PF               AFIO_EXTISS3_EXTI13_SS_PF_Msk        /*!< PF[13] pin */
#define AFIO_EXTISS3_EXTI13_SS_PG_Pos           (5U)                              
#define AFIO_EXTISS3_EXTI13_SS_PG_Msk           (0x3U << AFIO_EXTISS3_EXTI13_SS_PG_Pos) /*!< 0x00000060 */
#define AFIO_EXTISS3_EXTI13_SS_PG               AFIO_EXTISS3_EXTI13_SS_PG_Msk        /*!< PG[13] pin */

/*!< EXTI14 configuration */  
#define AFIO_EXTISS3_EXTI14_SS_PA               0x00000000U                          /*!< PA[14] pin */
#define AFIO_EXTISS3_EXTI14_SS_PB_Pos           (8U)                              
#define AFIO_EXTISS3_EXTI14_SS_PB_Msk           (0x1U << AFIO_EXTISS3_EXTI14_SS_PB_Pos) /*!< 0x00000100 */
#define AFIO_EXTISS3_EXTI14_SS_PB               AFIO_EXTISS3_EXTI14_SS_PB_Msk        /*!< PB[14] pin */
#define AFIO_EXTISS3_EXTI14_SS_PC_Pos           (9U)                              
#define AFIO_EXTISS3_EXTI14_SS_PC_Msk           (0x1U << AFIO_EXTISS3_EXTI14_SS_PC_Pos) /*!< 0x00000200 */
#define AFIO_EXTISS3_EXTI14_SS_PC               AFIO_EXTISS3_EXTI14_SS_PC_Msk        /*!< PC[14] pin */
#define AFIO_EXTISS3_EXTI14_SS_PD_Pos           (8U)                              
#define AFIO_EXTISS3_EXTI14_SS_PD_Msk           (0x3U << AFIO_EXTISS3_EXTI14_SS_PD_Pos) /*!< 0x00000300 */
#define AFIO_EXTISS3_EXTI14_SS_PD               AFIO_EXTISS3_EXTI14_SS_PD_Msk        /*!< PD[14] pin */
#define AFIO_EXTISS3_EXTI14_SS_PE_Pos           (10U)                             
#define AFIO_EXTISS3_EXTI14_SS_PE_Msk           (0x1U << AFIO_EXTISS3_EXTI14_SS_PE_Pos) /*!< 0x00000400 */
#define AFIO_EXTISS3_EXTI14_SS_PE               AFIO_EXTISS3_EXTI14_SS_PE_Msk        /*!< PE[14] pin */
#define AFIO_EXTISS3_EXTI14_SS_PF_Pos           (8U)                              
#define AFIO_EXTISS3_EXTI14_SS_PF_Msk           (0x5U << AFIO_EXTISS3_EXTI14_SS_PF_Pos) /*!< 0x00000500 */
#define AFIO_EXTISS3_EXTI14_SS_PF               AFIO_EXTISS3_EXTI14_SS_PF_Msk        /*!< PF[14] pin */
#define AFIO_EXTISS3_EXTI14_SS_PG_Pos           (9U)                              
#define AFIO_EXTISS3_EXTI14_SS_PG_Msk           (0x3U << AFIO_EXTISS3_EXTI14_SS_PG_Pos) /*!< 0x00000600 */
#define AFIO_EXTISS3_EXTI14_SS_PG               AFIO_EXTISS3_EXTI14_SS_PG_Msk        /*!< PG[14] pin */

/*!< EXTI15 configuration */
#define AFIO_EXTISS3_EXTI15_SS_PA               0x00000000U                          /*!< PA[15] pin */
#define AFIO_EXTISS3_EXTI15_SS_PB_Pos           (12U)                             
#define AFIO_EXTISS3_EXTI15_SS_PB_Msk           (0x1U << AFIO_EXTISS3_EXTI15_SS_PB_Pos) /*!< 0x00001000 */
#define AFIO_EXTISS3_EXTI15_SS_PB               AFIO_EXTISS3_EXTI15_SS_PB_Msk        /*!< PB[15] pin */
#define AFIO_EXTISS3_EXTI15_SS_PC_Pos           (13U)                             
#define AFIO_EXTISS3_EXTI15_SS_PC_Msk           (0x1U << AFIO_EXTISS3_EXTI15_SS_PC_Pos) /*!< 0x00002000 */
#define AFIO_EXTISS3_EXTI15_SS_PC               AFIO_EXTISS3_EXTI15_SS_PC_Msk        /*!< PC[15] pin */
#define AFIO_EXTISS3_EXTI15_SS_PD_Pos           (12U)                             
#define AFIO_EXTISS3_EXTI15_SS_PD_Msk           (0x3U << AFIO_EXTISS3_EXTI15_SS_PD_Pos) /*!< 0x00003000 */
#define AFIO_EXTISS3_EXTI15_SS_PD               AFIO_EXTISS3_EXTI15_SS_PD_Msk        /*!< PD[15] pin */
#define AFIO_EXTISS3_EXTI15_SS_PE_Pos           (14U)                             
#define AFIO_EXTISS3_EXTI15_SS_PE_Msk           (0x1U << AFIO_EXTISS3_EXTI15_SS_PE_Pos) /*!< 0x00004000 */
#define AFIO_EXTISS3_EXTI15_SS_PE               AFIO_EXTISS3_EXTI15_SS_PE_Msk        /*!< PE[15] pin */
#define AFIO_EXTISS3_EXTI15_SS_PF_Pos           (12U)                             
#define AFIO_EXTISS3_EXTI15_SS_PF_Msk           (0x5U << AFIO_EXTISS3_EXTI15_SS_PF_Pos) /*!< 0x00005000 */
#define AFIO_EXTISS3_EXTI15_SS_PF               AFIO_EXTISS3_EXTI15_SS_PF_Msk        /*!< PF[15] pin */
#define AFIO_EXTISS3_EXTI15_SS_PG_Pos           (13U)                             
#define AFIO_EXTISS3_EXTI15_SS_PG_Msk           (0x3U << AFIO_EXTISS3_EXTI15_SS_PG_Pos) /*!< 0x00006000 */
#define AFIO_EXTISS3_EXTI15_SS_PG               AFIO_EXTISS3_EXTI15_SS_PG_Msk        /*!< PG[15] pin */

/******************  Bit definition for AFIO_PCF1 register  ******************/
#define AFIO_PCF1_EXMC_NADV_REMAP_Pos       (10U)                             
#define AFIO_PCF1_EXMC_NADV_REMAP_Msk       (0x1U << AFIO_PCF1_EXMC_NADV_REMAP_Pos) /*!< 0x00000400 */
#define AFIO_PCF1_EXMC_NADV_REMAP           AFIO_PCF1_EXMC_NADV_REMAP_Msk    /*!< EXMC NADV remapping */

/******************************************************************************/
/*                                                                            */
/*                             DMA Controller                                 */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for DMA_INTF register  ********************/
#define DMA_INTF_GIF0_Pos                    (0U)                               
#define DMA_INTF_GIF0_Msk                    (0x1U << DMA_INTF_GIF0_Pos)         /*!< 0x00000001 */
#define DMA_INTF_GIF0                        DMA_INTF_GIF0_Msk                   /*!< Channel 1 Global interrupt flag */
#define DMA_INTF_FTFIF0_Pos                   (1U)                               
#define DMA_INTF_FTFIF0_Msk                   (0x1U << DMA_INTF_FTFIF0_Pos)        /*!< 0x00000002 */
#define DMA_INTF_FTFIF0                       DMA_INTF_FTFIF0_Msk                  /*!< Channel 1 Full Transfer finish flag */
#define DMA_INTF_HTFIF0_Pos                   (2U)                               
#define DMA_INTF_HTFIF0_Msk                   (0x1U << DMA_INTF_HTFIF0_Pos)        /*!< 0x00000004 */
#define DMA_INTF_HTFIF0                       DMA_INTF_HTFIF0_Msk                  /*!< Channel 1 Half Transfer flag */
#define DMA_INTF_ERRIF0_Pos                   (3U)                               
#define DMA_INTF_ERRIF0_Msk                   (0x1U << DMA_INTF_ERRIF0_Pos)        /*!< 0x00000008 */
#define DMA_INTF_ERRIF0                       DMA_INTF_ERRIF0_Msk                  /*!< Channel 1 Transfer Error flag */
#define DMA_INTF_GIF1_Pos                    (4U)                               
#define DMA_INTF_GIF1_Msk                    (0x1U << DMA_INTF_GIF1_Pos)         /*!< 0x00000010 */
#define DMA_INTF_GIF1                        DMA_INTF_GIF1_Msk                   /*!< Channel 2 Global interrupt flag */
#define DMA_INTF_FTFIF1_Pos                   (5U)                               
#define DMA_INTF_FTFIF1_Msk                   (0x1U << DMA_INTF_FTFIF1_Pos)        /*!< 0x00000020 */
#define DMA_INTF_FTFIF1                       DMA_INTF_FTFIF1_Msk                  /*!< Channel 2 Full Transfer finish flag */
#define DMA_INTF_HTFIF1_Pos                   (6U)                               
#define DMA_INTF_HTFIF1_Msk                   (0x1U << DMA_INTF_HTFIF1_Pos)        /*!< 0x00000040 */
#define DMA_INTF_HTFIF1                       DMA_INTF_HTFIF1_Msk                  /*!< Channel 2 Half Transfer flag */
#define DMA_INTF_ERRIF1_Pos                   (7U)                               
#define DMA_INTF_ERRIF1_Msk                   (0x1U << DMA_INTF_ERRIF1_Pos)        /*!< 0x00000080 */
#define DMA_INTF_ERRIF1                       DMA_INTF_ERRIF1_Msk                  /*!< Channel 2 Transfer Error flag */
#define DMA_INTF_GIF2_Pos                    (8U)                               
#define DMA_INTF_GIF2_Msk                    (0x1U << DMA_INTF_GIF2_Pos)         /*!< 0x00000100 */
#define DMA_INTF_GIF2                        DMA_INTF_GIF2_Msk                   /*!< Channel 3 Global interrupt flag */
#define DMA_INTF_FTFIF2_Pos                   (9U)                               
#define DMA_INTF_FTFIF2_Msk                   (0x1U << DMA_INTF_FTFIF2_Pos)        /*!< 0x00000200 */
#define DMA_INTF_FTFIF2                       DMA_INTF_FTFIF2_Msk                  /*!< Channel 3 Full Transfer finish flag */
#define DMA_INTF_HTFIF2_Pos                   (10U)                              
#define DMA_INTF_HTFIF2_Msk                   (0x1U << DMA_INTF_HTFIF2_Pos)        /*!< 0x00000400 */
#define DMA_INTF_HTFIF2                       DMA_INTF_HTFIF2_Msk                  /*!< Channel 3 Half Transfer flag */
#define DMA_INTF_ERRIF2_Pos                   (11U)                              
#define DMA_INTF_ERRIF2_Msk                   (0x1U << DMA_INTF_ERRIF2_Pos)        /*!< 0x00000800 */
#define DMA_INTF_ERRIF2                       DMA_INTF_ERRIF2_Msk                  /*!< Channel 3 Transfer Error flag */
#define DMA_INTF_GIF3_Pos                    (12U)                              
#define DMA_INTF_GIF3_Msk                    (0x1U << DMA_INTF_GIF3_Pos)         /*!< 0x00001000 */
#define DMA_INTF_GIF3                        DMA_INTF_GIF3_Msk                   /*!< Channel 4 Global interrupt flag */
#define DMA_INTF_FTFIF3_Pos                   (13U)                              
#define DMA_INTF_FTFIF3_Msk                   (0x1U << DMA_INTF_FTFIF3_Pos)        /*!< 0x00002000 */
#define DMA_INTF_FTFIF3                       DMA_INTF_FTFIF3_Msk                  /*!< Channel 4 Full Transfer finish flag */
#define DMA_INTF_HTFIF3_Pos                   (14U)                              
#define DMA_INTF_HTFIF3_Msk                   (0x1U << DMA_INTF_HTFIF3_Pos)        /*!< 0x00004000 */
#define DMA_INTF_HTFIF3                       DMA_INTF_HTFIF3_Msk                  /*!< Channel 4 Half Transfer flag */
#define DMA_INTF_ERRIF3_Pos                   (15U)                              
#define DMA_INTF_ERRIF3_Msk                   (0x1U << DMA_INTF_ERRIF3_Pos)        /*!< 0x00008000 */
#define DMA_INTF_ERRIF3                       DMA_INTF_ERRIF3_Msk                  /*!< Channel 4 Transfer Error flag */
#define DMA_INTF_GIF4_Pos                    (16U)                              
#define DMA_INTF_GIF4_Msk                    (0x1U << DMA_INTF_GIF4_Pos)         /*!< 0x00010000 */
#define DMA_INTF_GIF4                        DMA_INTF_GIF4_Msk                   /*!< Channel 5 Global interrupt flag */
#define DMA_INTF_FTFIF4_Pos                   (17U)                              
#define DMA_INTF_FTFIF4_Msk                   (0x1U << DMA_INTF_FTFIF4_Pos)        /*!< 0x00020000 */
#define DMA_INTF_FTFIF4                       DMA_INTF_FTFIF4_Msk                  /*!< Channel 5 Full Transfer finish flag */
#define DMA_INTF_HTFIF4_Pos                   (18U)                              
#define DMA_INTF_HTFIF4_Msk                   (0x1U << DMA_INTF_HTFIF4_Pos)        /*!< 0x00040000 */
#define DMA_INTF_HTFIF4                       DMA_INTF_HTFIF4_Msk                  /*!< Channel 5 Half Transfer flag */
#define DMA_INTF_ERRIF4_Pos                   (19U)                              
#define DMA_INTF_ERRIF4_Msk                   (0x1U << DMA_INTF_ERRIF4_Pos)        /*!< 0x00080000 */
#define DMA_INTF_ERRIF4                       DMA_INTF_ERRIF4_Msk                  /*!< Channel 5 Transfer Error flag */
#define DMA_INTF_GIF5_Pos                    (20U)                              
#define DMA_INTF_GIF5_Msk                    (0x1U << DMA_INTF_GIF5_Pos)         /*!< 0x00100000 */
#define DMA_INTF_GIF5                        DMA_INTF_GIF5_Msk                   /*!< Channel 6 Global interrupt flag */
#define DMA_INTF_FTFIF5_Pos                   (21U)                              
#define DMA_INTF_FTFIF5_Msk                   (0x1U << DMA_INTF_FTFIF5_Pos)        /*!< 0x00200000 */
#define DMA_INTF_FTFIF5                       DMA_INTF_FTFIF5_Msk                  /*!< Channel 6 Full Transfer finish flag */
#define DMA_INTF_HTFIF5_Pos                   (22U)                              
#define DMA_INTF_HTFIF5_Msk                   (0x1U << DMA_INTF_HTFIF5_Pos)        /*!< 0x00400000 */
#define DMA_INTF_HTFIF5                       DMA_INTF_HTFIF5_Msk                  /*!< Channel 6 Half Transfer flag */
#define DMA_INTF_ERRIF5_Pos                   (23U)                              
#define DMA_INTF_ERRIF5_Msk                   (0x1U << DMA_INTF_ERRIF5_Pos)        /*!< 0x00800000 */
#define DMA_INTF_ERRIF5                       DMA_INTF_ERRIF5_Msk                  /*!< Channel 6 Transfer Error flag */
#define DMA_INTF_GIF6_Pos                    (24U)                              
#define DMA_INTF_GIF6_Msk                    (0x1U << DMA_INTF_GIF6_Pos)         /*!< 0x01000000 */
#define DMA_INTF_GIF6                        DMA_INTF_GIF6_Msk                   /*!< Channel 7 Global interrupt flag */
#define DMA_INTF_FTFIF6_Pos                   (25U)                              
#define DMA_INTF_FTFIF6_Msk                   (0x1U << DMA_INTF_FTFIF6_Pos)        /*!< 0x02000000 */
#define DMA_INTF_FTFIF6                       DMA_INTF_FTFIF6_Msk                  /*!< Channel 7 Full Transfer finish flag */
#define DMA_INTF_HTFIF6_Pos                   (26U)                              
#define DMA_INTF_HTFIF6_Msk                   (0x1U << DMA_INTF_HTFIF6_Pos)        /*!< 0x04000000 */
#define DMA_INTF_HTFIF6                       DMA_INTF_HTFIF6_Msk                  /*!< Channel 7 Half Transfer flag */
#define DMA_INTF_ERRIF6_Pos                   (27U)                              
#define DMA_INTF_ERRIF6_Msk                   (0x1U << DMA_INTF_ERRIF6_Pos)        /*!< 0x08000000 */
#define DMA_INTF_ERRIF6                       DMA_INTF_ERRIF6_Msk                  /*!< Channel 7 Transfer Error flag */

/*******************  Bit definition for DMA_INTC register  *******************/
#define DMA_INTC_GIFC0_Pos                  (0U)                               
#define DMA_INTC_GIFC0_Msk                  (0x1U << DMA_INTC_GIFC0_Pos)       /*!< 0x00000001 */
#define DMA_INTC_GIFC0                      DMA_INTC_GIFC0_Msk                 /*!< Channel 1 Global interrupt clear */
#define DMA_INTC_FTFIFC0_Pos                 (1U)                               
#define DMA_INTC_FTFIFC0_Msk                 (0x1U << DMA_INTC_FTFIFC0_Pos)      /*!< 0x00000002 */
#define DMA_INTC_FTFIFC0                     DMA_INTC_FTFIFC0_Msk                /*!< Channel 1 Transfer Complete clear */
#define DMA_INTC_HTFIFC0_Pos                 (2U)                               
#define DMA_INTC_HTFIFC0_Msk                 (0x1U << DMA_INTC_HTFIFC0_Pos)      /*!< 0x00000004 */
#define DMA_INTC_HTFIFC0                     DMA_INTC_HTFIFC0_Msk                /*!< Channel 1 Half Transfer clear */
#define DMA_INTC_ERRIFC0_Pos                 (3U)                               
#define DMA_INTC_ERRIFC0_Msk                 (0x1U << DMA_INTC_ERRIFC0_Pos)      /*!< 0x00000008 */
#define DMA_INTC_ERRIFC0                     DMA_INTC_ERRIFC0_Msk                /*!< Channel 1 Transfer Error clear */
#define DMA_INTC_GIFC1_Pos                  (4U)                               
#define DMA_INTC_GIFC1_Msk                  (0x1U << DMA_INTC_GIFC1_Pos)       /*!< 0x00000010 */
#define DMA_INTC_GIFC1                      DMA_INTC_GIFC1_Msk                 /*!< Channel 2 Global interrupt clear */
#define DMA_INTC_FTFIFC1_Pos                 (5U)                               
#define DMA_INTC_FTFIFC1_Msk                 (0x1U << DMA_INTC_FTFIFC1_Pos)      /*!< 0x00000020 */
#define DMA_INTC_FTFIFC1                     DMA_INTC_FTFIFC1_Msk                /*!< Channel 2 Transfer Complete clear */
#define DMA_INTC_HTFIFC1_Pos                 (6U)                               
#define DMA_INTC_HTFIFC1_Msk                 (0x1U << DMA_INTC_HTFIFC1_Pos)      /*!< 0x00000040 */
#define DMA_INTC_HTFIFC1                     DMA_INTC_HTFIFC1_Msk                /*!< Channel 2 Half Transfer clear */
#define DMA_INTC_ERRIFC1_Pos                 (7U)                               
#define DMA_INTC_ERRIFC1_Msk                 (0x1U << DMA_INTC_ERRIFC1_Pos)      /*!< 0x00000080 */
#define DMA_INTC_ERRIFC1                     DMA_INTC_ERRIFC1_Msk                /*!< Channel 2 Transfer Error clear */
#define DMA_INTC_GIFC2_Pos                  (8U)                               
#define DMA_INTC_GIFC2_Msk                  (0x1U << DMA_INTC_GIFC2_Pos)       /*!< 0x00000100 */
#define DMA_INTC_GIFC2                      DMA_INTC_GIFC2_Msk                 /*!< Channel 3 Global interrupt clear */
#define DMA_INTC_FTFIFC2_Pos                 (9U)                               
#define DMA_INTC_FTFIFC2_Msk                 (0x1U << DMA_INTC_FTFIFC2_Pos)      /*!< 0x00000200 */
#define DMA_INTC_FTFIFC2                     DMA_INTC_FTFIFC2_Msk                /*!< Channel 3 Transfer Complete clear */
#define DMA_INTC_HTFIFC2_Pos                 (10U)                              
#define DMA_INTC_HTFIFC2_Msk                 (0x1U << DMA_INTC_HTFIFC2_Pos)      /*!< 0x00000400 */
#define DMA_INTC_HTFIFC2                     DMA_INTC_HTFIFC2_Msk                /*!< Channel 3 Half Transfer clear */
#define DMA_INTC_ERRIFC2_Pos                 (11U)                              
#define DMA_INTC_ERRIFC2_Msk                 (0x1U << DMA_INTC_ERRIFC2_Pos)      /*!< 0x00000800 */
#define DMA_INTC_ERRIFC2                     DMA_INTC_ERRIFC2_Msk                /*!< Channel 3 Transfer Error clear */
#define DMA_INTC_GIFC3_Pos                  (12U)                              
#define DMA_INTC_GIFC3_Msk                  (0x1U << DMA_INTC_GIFC3_Pos)       /*!< 0x00001000 */
#define DMA_INTC_GIFC3                      DMA_INTC_GIFC3_Msk                 /*!< Channel 4 Global interrupt clear */
#define DMA_INTC_FTFIFC3_Pos                 (13U)                              
#define DMA_INTC_FTFIFC3_Msk                 (0x1U << DMA_INTC_FTFIFC3_Pos)      /*!< 0x00002000 */
#define DMA_INTC_FTFIFC3                     DMA_INTC_FTFIFC3_Msk                /*!< Channel 4 Transfer Complete clear */
#define DMA_INTC_HTFIFC3_Pos                 (14U)                              
#define DMA_INTC_HTFIFC3_Msk                 (0x1U << DMA_INTC_HTFIFC3_Pos)      /*!< 0x00004000 */
#define DMA_INTC_HTFIFC3                     DMA_INTC_HTFIFC3_Msk                /*!< Channel 4 Half Transfer clear */
#define DMA_INTC_ERRIFC3_Pos                 (15U)                              
#define DMA_INTC_ERRIFC3_Msk                 (0x1U << DMA_INTC_ERRIFC3_Pos)      /*!< 0x00008000 */
#define DMA_INTC_ERRIFC3                     DMA_INTC_ERRIFC3_Msk                /*!< Channel 4 Transfer Error clear */
#define DMA_INTC_GIFC4_Pos                  (16U)                              
#define DMA_INTC_GIFC4_Msk                  (0x1U << DMA_INTC_GIFC4_Pos)       /*!< 0x00010000 */
#define DMA_INTC_GIFC4                      DMA_INTC_GIFC4_Msk                 /*!< Channel 5 Global interrupt clear */
#define DMA_INTC_FTFIFC4_Pos                 (17U)                              
#define DMA_INTC_FTFIFC4_Msk                 (0x1U << DMA_INTC_FTFIFC4_Pos)      /*!< 0x00020000 */
#define DMA_INTC_FTFIFC4                     DMA_INTC_FTFIFC4_Msk                /*!< Channel 5 Transfer Complete clear */
#define DMA_INTC_HTFIFC4_Pos                 (18U)                              
#define DMA_INTC_HTFIFC4_Msk                 (0x1U << DMA_INTC_HTFIFC4_Pos)      /*!< 0x00040000 */
#define DMA_INTC_HTFIFC4                     DMA_INTC_HTFIFC4_Msk                /*!< Channel 5 Half Transfer clear */
#define DMA_INTC_ERRIFC4_Pos                 (19U)                              
#define DMA_INTC_ERRIFC4_Msk                 (0x1U << DMA_INTC_ERRIFC4_Pos)      /*!< 0x00080000 */
#define DMA_INTC_ERRIFC4                     DMA_INTC_ERRIFC4_Msk                /*!< Channel 5 Transfer Error clear */
#define DMA_INTC_GIFC5_Pos                  (20U)                              
#define DMA_INTC_GIFC5_Msk                  (0x1U << DMA_INTC_GIFC5_Pos)       /*!< 0x00100000 */
#define DMA_INTC_GIFC5                      DMA_INTC_GIFC5_Msk                 /*!< Channel 6 Global interrupt clear */
#define DMA_INTC_FTFIFC5_Pos                 (21U)                              
#define DMA_INTC_FTFIFC5_Msk                 (0x1U << DMA_INTC_FTFIFC5_Pos)      /*!< 0x00200000 */
#define DMA_INTC_FTFIFC5                     DMA_INTC_FTFIFC5_Msk                /*!< Channel 6 Transfer Complete clear */
#define DMA_INTC_HTFIFC5_Pos                 (22U)                              
#define DMA_INTC_HTFIFC5_Msk                 (0x1U << DMA_INTC_HTFIFC5_Pos)      /*!< 0x00400000 */
#define DMA_INTC_HTFIFC5                     DMA_INTC_HTFIFC5_Msk                /*!< Channel 6 Half Transfer clear */
#define DMA_INTC_ERRIFC5_Pos                 (23U)                              
#define DMA_INTC_ERRIFC5_Msk                 (0x1U << DMA_INTC_ERRIFC5_Pos)      /*!< 0x00800000 */
#define DMA_INTC_ERRIFC5                     DMA_INTC_ERRIFC5_Msk                /*!< Channel 6 Transfer Error clear */
#define DMA_INTC_GIFC6_Pos                  (24U)                              
#define DMA_INTC_GIFC6_Msk                  (0x1U << DMA_INTC_GIFC6_Pos)       /*!< 0x01000000 */
#define DMA_INTC_GIFC6                      DMA_INTC_GIFC6_Msk                 /*!< Channel 7 Global interrupt clear */
#define DMA_INTC_FTFIFC6_Pos                 (25U)                              
#define DMA_INTC_FTFIFC6_Msk                 (0x1U << DMA_INTC_FTFIFC6_Pos)      /*!< 0x02000000 */
#define DMA_INTC_FTFIFC6                     DMA_INTC_FTFIFC6_Msk                /*!< Channel 7 Transfer Complete clear */
#define DMA_INTC_HTFIFC6_Pos                 (26U)                              
#define DMA_INTC_HTFIFC6_Msk                 (0x1U << DMA_INTC_HTFIFC6_Pos)      /*!< 0x04000000 */
#define DMA_INTC_HTFIFC6                     DMA_INTC_HTFIFC6_Msk                /*!< Channel 7 Half Transfer clear */
#define DMA_INTC_ERRIFC6_Pos                 (27U)                              
#define DMA_INTC_ERRIFC6_Msk                 (0x1U << DMA_INTC_ERRIFC6_Pos)      /*!< 0x08000000 */
#define DMA_INTC_ERRIFC6                     DMA_INTC_ERRIFC6_Msk                /*!< Channel 7 Transfer Error clear */

/*******************  Bit definition for DMA_CTL register   *******************/
#define DMA_CTL_CHEN_Pos                      (0U)                               
#define DMA_CTL_CHEN_Msk                      (0x1U << DMA_CTL_CHEN_Pos)           /*!< 0x00000001 */
#define DMA_CTL_CHEN                          DMA_CTL_CHEN_Msk                     /*!< Channel enable */
#define DMA_CTL_FTFIE_Pos                    (1U)                               
#define DMA_CTL_FTFIE_Msk                    (0x1U << DMA_CTL_FTFIE_Pos)         /*!< 0x00000002 */
#define DMA_CTL_FTFIE                        DMA_CTL_FTFIE_Msk                   /*!< Transfer complete interrupt enable */
#define DMA_CTL_HTFIE_Pos                    (2U)                               
#define DMA_CTL_HTFIE_Msk                    (0x1U << DMA_CTL_HTFIE_Pos)         /*!< 0x00000004 */
#define DMA_CTL_HTFIE                        DMA_CTL_HTFIE_Msk                   /*!< Half Transfer interrupt enable */
#define DMA_CTL_ERRIE_Pos                    (3U)                               
#define DMA_CTL_ERRIE_Msk                    (0x1U << DMA_CTL_ERRIE_Pos)         /*!< 0x00000008 */
#define DMA_CTL_ERRIE                        DMA_CTL_ERRIE_Msk                   /*!< Transfer error interrupt enable */
#define DMA_CTL_DIR_Pos                     (4U)                               
#define DMA_CTL_DIR_Msk                     (0x1U << DMA_CTL_DIR_Pos)          /*!< 0x00000010 */
#define DMA_CTL_DIR                         DMA_CTL_DIR_Msk                    /*!< Data transfer direction */
#define DMA_CTL_CMEN_Pos                    (5U)                               
#define DMA_CTL_CMEN_Msk                    (0x1U << DMA_CTL_CMEN_Pos)         /*!< 0x00000020 */
#define DMA_CTL_CMEN                        DMA_CTL_CMEN_Msk                   /*!< Circular mode */
#define DMA_CTL_PNAGA_Pos                    (6U)                               
#define DMA_CTL_PNAGA_Msk                    (0x1U << DMA_CTL_PNAGA_Pos)         /*!< 0x00000040 */
#define DMA_CTL_PNAGA                        DMA_CTL_PNAGA_Msk                   /*!< Peripheral increment mode */
#define DMA_CTL_MNAGA_Pos                    (7U)                               
#define DMA_CTL_MNAGA_Msk                    (0x1U << DMA_CTL_MNAGA_Pos)         /*!< 0x00000080 */
#define DMA_CTL_MNAGA                        DMA_CTL_MNAGA_Msk                   /*!< Memory increment mode */

#define DMA_CTL_PWIDTH_Pos                   (8U)                               
#define DMA_CTL_PWIDTH_Msk                   (0x3U << DMA_CTL_PWIDTH_Pos)        /*!< 0x00000300 */
#define DMA_CTL_PWIDTH                       DMA_CTL_PWIDTH_Msk                  /*!< PSIZE[1:0] bits (Peripheral size) */
#define DMA_CTL_PWIDTH_0                     (0x1U << DMA_CTL_PWIDTH_Pos)        /*!< 0x00000100 */
#define DMA_CTL_PWIDTH_1                     (0x2U << DMA_CTL_PWIDTH_Pos)        /*!< 0x00000200 */

#define DMA_CTL_MWIDTH_Pos                   (10U)                              
#define DMA_CTL_MWIDTH_Msk                   (0x3U << DMA_CTL_MWIDTH_Pos)        /*!< 0x00000C00 */
#define DMA_CTL_MWIDTH                       DMA_CTL_MWIDTH_Msk                  /*!< MSIZE[1:0] bits (Memory size) */
#define DMA_CTL_MWIDTH_0                     (0x1U << DMA_CTL_MWIDTH_Pos)        /*!< 0x00000400 */
#define DMA_CTL_MWIDTH_1                     (0x2U << DMA_CTL_MWIDTH_Pos)        /*!< 0x00000800 */

#define DMA_CTL_PRIO_Pos                      (12U)                              
#define DMA_CTL_PRIO_Msk                      (0x3U << DMA_CTL_PRIO_Pos)           /*!< 0x00003000 */
#define DMA_CTL_PRIO                          DMA_CTL_PRIO_Msk                     /*!< PL[1:0] bits(Channel Priority level) */
#define DMA_CTL_PRIO_0                        (0x1U << DMA_CTL_PRIO_Pos)           /*!< 0x00001000 */
#define DMA_CTL_PRIO_1                        (0x2U << DMA_CTL_PRIO_Pos)           /*!< 0x00002000 */

#define DMA_CTL_M2M_Pos                 (14U)                              
#define DMA_CTL_M2M_Msk                 (0x1U << DMA_CTL_M2M_Pos)      /*!< 0x00004000 */
#define DMA_CTL_M2M                     DMA_CTL_M2M_Msk                /*!< Memory to memory mode */

/******************  Bit definition for DMA_CNT  register  ******************/
#define DMA_CNT_CNT_Pos                   (0U)                               
#define DMA_CNT_CNT_Msk                   (0xFFFFU << DMA_CNT_CNT_Pos)     /*!< 0x0000FFFF */
#define DMA_CNT_CNT                       DMA_CNT_CNT_Msk                  /*!< Number of data to Transfer */

/******************  Bit definition for DMA_PADDR  register  *******************/
#define DMA_PADDR_PADDR_Pos                     (0U)                               
#define DMA_PADDR_PADDR_Msk                     (0xFFFFFFFFU << DMA_PADDR_PADDR_Pos)   /*!< 0xFFFFFFFF */
#define DMA_PADDR_PADDR                         DMA_PADDR_PADDR_Msk                    /*!< Peripheral Address */

/******************  Bit definition for DMA_MADDR  register  *******************/
#define DMA_MADDR_MADDR_Pos                     (0U)                               
#define DMA_MADDR_MADDR_Msk                     (0xFFFFFFFFU << DMA_MADDR_MADDR_Pos)   /*!< 0xFFFFFFFF */
#define DMA_MADDR_MADDR                         DMA_MADDR_MADDR_Msk                    /*!< Memory Address */

/******************************************************************************/
/*                                                                            */
/*                      Analog to Digital Converter (ADC)                     */
/*                                                                            */
/******************************************************************************/

/*
 * @brief Specific device feature definitions (not present on all devices in the GD32VF103 family)
 */
#define ADC_MULTIMODE_SUPPORT                          /*!< ADC feature available only on specific devices: multimode available on devices with several ADC instances */

/********************  Bit definition for ADC_STAT register  ********************/
#define ADC_STAT_WDE_Pos                      (0U)                               
#define ADC_STAT_WDE_Msk                      (0x1U << ADC_STAT_WDE_Pos)           /*!< 0x00000001 */
#define ADC_STAT_WDE                          ADC_STAT_WDE_Msk                     /*!< ADC analog watchdog 1 flag */
#define ADC_STAT_EOC_Pos                      (1U)                               
#define ADC_STAT_EOC_Msk                      (0x1U << ADC_STAT_EOC_Pos)           /*!< 0x00000002 */
#define ADC_STAT_EOC                          ADC_STAT_EOC_Msk                     /*!< ADC group regular end of sequence conversions flag */
#define ADC_STAT_EOIC_Pos                     (2U)                               
#define ADC_STAT_EOIC_Msk                     (0x1U << ADC_STAT_EOIC_Pos)          /*!< 0x00000004 */
#define ADC_STAT_EOIC                         ADC_STAT_EOIC_Msk                    /*!< ADC group injected end of sequence conversions flag */
#define ADC_STAT_STIC_Pos                    (3U)                               
#define ADC_STAT_STIC_Msk                    (0x1U << ADC_STAT_STIC_Pos)         /*!< 0x00000008 */
#define ADC_STAT_STIC                        ADC_STAT_STIC_Msk                   /*!< ADC group injected conversion start flag */
#define ADC_STAT_STRC_Pos                     (4U)                               
#define ADC_STAT_STRC_Msk                     (0x1U << ADC_STAT_STRC_Pos)          /*!< 0x00000010 */
#define ADC_STAT_STRC                         ADC_STAT_STRC_Msk                    /*!< ADC group regular conversion start flag */

/*******************  Bit definition for ADC_CTL0 register  ********************/
#define ADC_CTL0_WDCHSEL_Pos                   (0U)                               
#define ADC_CTL0_WDCHSEL_Msk                   (0x1FU << ADC_CTL0_WDCHSEL_Pos)       /*!< 0x0000001F */
#define ADC_CTL0_WDCHSEL                       ADC_CTL0_WDCHSEL_Msk                  /*!< ADC analog watchdog 1 monitored channel selection */
#define ADC_CTL0_WDCHSEL_0                     (0x01U << ADC_CTL0_WDCHSEL_Pos)       /*!< 0x00000001 */
#define ADC_CTL0_WDCHSEL_1                     (0x02U << ADC_CTL0_WDCHSEL_Pos)       /*!< 0x00000002 */
#define ADC_CTL0_WDCHSEL_2                     (0x04U << ADC_CTL0_WDCHSEL_Pos)       /*!< 0x00000004 */
#define ADC_CTL0_WDCHSEL_3                     (0x08U << ADC_CTL0_WDCHSEL_Pos)       /*!< 0x00000008 */
#define ADC_CTL0_WDCHSEL_4                     (0x10U << ADC_CTL0_WDCHSEL_Pos)       /*!< 0x00000010 */

#define ADC_CTL0_EOCIE_Pos                   (5U)                               
#define ADC_CTL0_EOCIE_Msk                   (0x1U << ADC_CTL0_EOCIE_Pos)        /*!< 0x00000020 */
#define ADC_CTL0_EOCIE                       ADC_CTL0_EOCIE_Msk                  /*!< ADC group regular end of sequence conversions interrupt */
#define ADC_CTL0_WDEIE_Pos                   (6U)                               
#define ADC_CTL0_WDEIE_Msk                   (0x1U << ADC_CTL0_WDEIE_Pos)        /*!< 0x00000040 */
#define ADC_CTL0_WDEIE                       ADC_CTL0_WDEIE_Msk                  /*!< ADC analog watchdog 1 interrupt */
#define ADC_CTL0_EOICIE_Pos                  (7U)                               
#define ADC_CTL0_EOICIE_Msk                  (0x1U << ADC_CTL0_EOICIE_Pos)       /*!< 0x00000080 */
#define ADC_CTL0_EOICIE                      ADC_CTL0_EOICIE_Msk                 /*!< ADC group injected end of sequence conversions interrupt */
#define ADC_CTL0_SM_Pos                    (8U)                               
#define ADC_CTL0_SM_Msk                    (0x1U << ADC_CTL0_SM_Pos)         /*!< 0x00000100 */
#define ADC_CTL0_SM                        ADC_CTL0_SM_Msk                   /*!< ADC scan mode */
#define ADC_CTL0_WDSC_Pos                  (9U)                               
#define ADC_CTL0_WDSC_Msk                  (0x1U << ADC_CTL0_WDSC_Pos)       /*!< 0x00000200 */
#define ADC_CTL0_WDSC                      ADC_CTL0_WDSC_Msk                 /*!< ADC analog watchdog 1 monitoring a single channel or all channels */
#define ADC_CTL0_ICA_Pos                   (10U)                              
#define ADC_CTL0_ICA_Msk                   (0x1U << ADC_CTL0_ICA_Pos)        /*!< 0x00000400 */
#define ADC_CTL0_ICA                       ADC_CTL0_ICA_Msk                  /*!< ADC group injected automatic trigger mode */
#define ADC_CTL0_DISRC_Pos                  (11U)                              
#define ADC_CTL0_DISRC_Msk                  (0x1U << ADC_CTL0_DISRC_Pos)       /*!< 0x00000800 */
#define ADC_CTL0_DISRC                      ADC_CTL0_DISRC_Msk                 /*!< ADC group regular sequencer discontinuous mode */
#define ADC_CTL0_DISIC_Pos                 (12U)                              
#define ADC_CTL0_DISIC_Msk                 (0x1U << ADC_CTL0_DISIC_Pos)      /*!< 0x00001000 */
#define ADC_CTL0_DISIC                     ADC_CTL0_DISIC_Msk                /*!< ADC group injected sequencer discontinuous mode */

#define ADC_CTL0_DISNUM_Pos                 (13U)                              
#define ADC_CTL0_DISNUM_Msk                 (0x7U << ADC_CTL0_DISNUM_Pos)      /*!< 0x0000E000 */
#define ADC_CTL0_DISNUM                     ADC_CTL0_DISNUM_Msk                /*!< ADC group regular sequencer discontinuous number of ranks */
#define ADC_CTL0_DISNUM_0                   (0x1U << ADC_CTL0_DISNUM_Pos)      /*!< 0x00002000 */
#define ADC_CTL0_DISNUM_1                   (0x2U << ADC_CTL0_DISNUM_Pos)      /*!< 0x00004000 */
#define ADC_CTL0_DISNUM_2                   (0x4U << ADC_CTL0_DISNUM_Pos)      /*!< 0x00008000 */

#define ADC_CTL0_SYNCM_Pos                 (16U)                              
#define ADC_CTL0_SYNCM_Msk                 (0xFU << ADC_CTL0_SYNCM_Pos)      /*!< 0x000F0000 */
#define ADC_CTL0_SYNCM                     ADC_CTL0_SYNCM_Msk                /*!< ADC multimode mode selection */
#define ADC_CTL0_SYNCM_0                   (0x1U << ADC_CTL0_SYNCM_Pos)      /*!< 0x00010000 */
#define ADC_CTL0_SYNCM_1                   (0x2U << ADC_CTL0_SYNCM_Pos)      /*!< 0x00020000 */
#define ADC_CTL0_SYNCM_2                   (0x4U << ADC_CTL0_SYNCM_Pos)      /*!< 0x00040000 */
#define ADC_CTL0_SYNCM_3                   (0x8U << ADC_CTL0_SYNCM_Pos)      /*!< 0x00080000 */

#define ADC_CTL0_IWDEN_Pos                  (22U)                              
#define ADC_CTL0_IWDEN_Msk                  (0x1U << ADC_CTL0_IWDEN_Pos)       /*!< 0x00400000 */
#define ADC_CTL0_IWDEN                      ADC_CTL0_IWDEN_Msk                 /*!< ADC analog watchdog 1 enable on scope ADC group injected */
#define ADC_CTL0_RWDEN_Pos                   (23U)                              
#define ADC_CTL0_RWDEN_Msk                   (0x1U << ADC_CTL0_RWDEN_Pos)        /*!< 0x00800000 */
#define ADC_CTL0_RWDEN                       ADC_CTL0_RWDEN_Msk                  /*!< ADC analog watchdog 1 enable on scope ADC group regular */

/*******************  Bit definition for ADC_CTL1 register  ********************/
#define ADC_CTL1_ADCON_Pos                    (0U)                               
#define ADC_CTL1_ADCON_Msk                    (0x1U << ADC_CTL1_ADCON_Pos)         /*!< 0x00000001 */
#define ADC_CTL1_ADCON                        ADC_CTL1_ADCON_Msk                   /*!< ADC enable */
#define ADC_CTL1_CTN_Pos                    (1U)                               
#define ADC_CTL1_CTN_Msk                    (0x1U << ADC_CTL1_CTN_Pos)         /*!< 0x00000002 */
#define ADC_CTL1_CTN                        ADC_CTL1_CTN_Msk                   /*!< ADC group regular continuous conversion mode */
#define ADC_CTL1_CLB_Pos                     (2U)                               
#define ADC_CTL1_CLB_Msk                     (0x1U << ADC_CTL1_CLB_Pos)          /*!< 0x00000004 */
#define ADC_CTL1_CLB                         ADC_CTL1_CLB_Msk                    /*!< ADC calibration start */
#define ADC_CTL1_RSTCLB_Pos                  (3U)                               
#define ADC_CTL1_RSTCLB_Msk                  (0x1U << ADC_CTL1_RSTCLB_Pos)       /*!< 0x00000008 */
#define ADC_CTL1_RSTCLB                      ADC_CTL1_RSTCLB_Msk                 /*!< ADC calibration reset */
#define ADC_CTL1_DMA_Pos                     (8U)                               
#define ADC_CTL1_DMA_Msk                     (0x1U << ADC_CTL1_DMA_Pos)          /*!< 0x00000100 */
#define ADC_CTL1_DMA                         ADC_CTL1_DMA_Msk                    /*!< ADC DMA transfer enable */
#define ADC_CTL1_DAL_Pos                   (11U)                              
#define ADC_CTL1_DAL_Msk                   (0x1U << ADC_CTL1_DAL_Pos)        /*!< 0x00000800 */
#define ADC_CTL1_DAL                       ADC_CTL1_DAL_Msk                  /*!< ADC data alignement */

#define ADC_CTL1_ETSIC_Pos                 (12U)                              
#define ADC_CTL1_ETSIC_Msk                 (0x7U << ADC_CTL1_ETSIC_Pos)      /*!< 0x00007000 */
#define ADC_CTL1_ETSIC                     ADC_CTL1_ETSIC_Msk                /*!< ADC group injected external trigger source */
#define ADC_CTL1_ETSIC_0                   (0x1U << ADC_CTL1_ETSIC_Pos)      /*!< 0x00001000 */
#define ADC_CTL1_ETSIC_1                   (0x2U << ADC_CTL1_ETSIC_Pos)      /*!< 0x00002000 */
#define ADC_CTL1_ETSIC_2                   (0x4U << ADC_CTL1_ETSIC_Pos)      /*!< 0x00004000 */

#define ADC_CTL1_ETEIC_Pos                (15U)                              
#define ADC_CTL1_ETEIC_Msk                (0x1U << ADC_CTL1_ETEIC_Pos)     /*!< 0x00008000 */
#define ADC_CTL1_ETEIC                    ADC_CTL1_ETEIC_Msk               /*!< ADC group injected external trigger enable */

#define ADC_CTL1_ETSRC_Pos                  (17U)                              
#define ADC_CTL1_ETSRC_Msk                  (0x7U << ADC_CTL1_ETSRC_Pos)       /*!< 0x000E0000 */
#define ADC_CTL1_ETSRC                      ADC_CTL1_ETSRC_Msk                 /*!< ADC group regular external trigger source */
#define ADC_CTL1_ETSRC_0                    (0x1U << ADC_CTL1_ETSRC_Pos)       /*!< 0x00020000 */
#define ADC_CTL1_ETSRC_1                    (0x2U << ADC_CTL1_ETSRC_Pos)       /*!< 0x00040000 */
#define ADC_CTL1_ETSRC_2                    (0x4U << ADC_CTL1_ETSRC_Pos)       /*!< 0x00080000 */

#define ADC_CTL1_ETERC_Pos                 (20U)                              
#define ADC_CTL1_ETERC_Msk                 (0x1U << ADC_CTL1_ETERC_Pos)      /*!< 0x00100000 */
#define ADC_CTL1_ETERC                     ADC_CTL1_ETERC_Msk                /*!< ADC group regular external trigger enable */
#define ADC_CTL1_SWICST_Pos                (21U)                              
#define ADC_CTL1_SWICST_Msk                (0x1U << ADC_CTL1_SWICST_Pos)     /*!< 0x00200000 */
#define ADC_CTL1_SWICST                    ADC_CTL1_SWICST_Msk               /*!< ADC group injected conversion start */
#define ADC_CTL1_SWRCST_Pos                 (22U)                              
#define ADC_CTL1_SWRCST_Msk                 (0x1U << ADC_CTL1_SWRCST_Pos)      /*!< 0x00400000 */
#define ADC_CTL1_SWRCST                     ADC_CTL1_SWRCST_Msk                /*!< ADC group regular conversion start */
#define ADC_CTL1_TSVREN_Pos                 (23U)                              
#define ADC_CTL1_TSVREN_Msk                 (0x1U << ADC_CTL1_TSVREN_Pos)      /*!< 0x00800000 */
#define ADC_CTL1_TSVREN                     ADC_CTL1_TSVREN_Msk                /*!< ADC internal path to VrefInt and temperature sensor enable */

/******************  Bit definition for ADC_SAMPT0 register  *******************/
#define ADC_SAMPT0_SPT10_Pos                 (0U)                               
#define ADC_SAMPT0_SPT10_Msk                 (0x7U << ADC_SAMPT0_SPT10_Pos)      /*!< 0x00000007 */
#define ADC_SAMPT0_SPT10                     ADC_SAMPT0_SPT10_Msk                /*!< ADC channel 10 sampling time selection  */
#define ADC_SAMPT0_SPT10_0                   (0x1U << ADC_SAMPT0_SPT10_Pos)      /*!< 0x00000001 */
#define ADC_SAMPT0_SPT10_1                   (0x2U << ADC_SAMPT0_SPT10_Pos)      /*!< 0x00000002 */
#define ADC_SAMPT0_SPT10_2                   (0x4U << ADC_SAMPT0_SPT10_Pos)      /*!< 0x00000004 */

#define ADC_SAMPT0_SPT11_Pos                 (3U)                               
#define ADC_SAMPT0_SPT11_Msk                 (0x7U << ADC_SAMPT0_SPT11_Pos)      /*!< 0x00000038 */
#define ADC_SAMPT0_SPT11                     ADC_SAMPT0_SPT11_Msk                /*!< ADC channel 11 sampling time selection  */
#define ADC_SAMPT0_SPT11_0                   (0x1U << ADC_SAMPT0_SPT11_Pos)      /*!< 0x00000008 */
#define ADC_SAMPT0_SPT11_1                   (0x2U << ADC_SAMPT0_SPT11_Pos)      /*!< 0x00000010 */
#define ADC_SAMPT0_SPT11_2                   (0x4U << ADC_SAMPT0_SPT11_Pos)      /*!< 0x00000020 */

#define ADC_SAMPT0_SPT12_Pos                 (6U)                               
#define ADC_SAMPT0_SPT12_Msk                 (0x7U << ADC_SAMPT0_SPT12_Pos)      /*!< 0x000001C0 */
#define ADC_SAMPT0_SPT12                     ADC_SAMPT0_SPT12_Msk                /*!< ADC channel 12 sampling time selection  */
#define ADC_SAMPT0_SPT12_0                   (0x1U << ADC_SAMPT0_SPT12_Pos)      /*!< 0x00000040 */
#define ADC_SAMPT0_SPT12_1                   (0x2U << ADC_SAMPT0_SPT12_Pos)      /*!< 0x00000080 */
#define ADC_SAMPT0_SPT12_2                   (0x4U << ADC_SAMPT0_SPT12_Pos)      /*!< 0x00000100 */

#define ADC_SAMPT0_SPT13_Pos                 (9U)                               
#define ADC_SAMPT0_SPT13_Msk                 (0x7U << ADC_SAMPT0_SPT13_Pos)      /*!< 0x00000E00 */
#define ADC_SAMPT0_SPT13                     ADC_SAMPT0_SPT13_Msk                /*!< ADC channel 13 sampling time selection  */
#define ADC_SAMPT0_SPT13_0                   (0x1U << ADC_SAMPT0_SPT13_Pos)      /*!< 0x00000200 */
#define ADC_SAMPT0_SPT13_1                   (0x2U << ADC_SAMPT0_SPT13_Pos)      /*!< 0x00000400 */
#define ADC_SAMPT0_SPT13_2                   (0x4U << ADC_SAMPT0_SPT13_Pos)      /*!< 0x00000800 */

#define ADC_SAMPT0_SPT14_Pos                 (12U)                              
#define ADC_SAMPT0_SPT14_Msk                 (0x7U << ADC_SAMPT0_SPT14_Pos)      /*!< 0x00007000 */
#define ADC_SAMPT0_SPT14                     ADC_SAMPT0_SPT14_Msk                /*!< ADC channel 14 sampling time selection  */
#define ADC_SAMPT0_SPT14_0                   (0x1U << ADC_SAMPT0_SPT14_Pos)      /*!< 0x00001000 */
#define ADC_SAMPT0_SPT14_1                   (0x2U << ADC_SAMPT0_SPT14_Pos)      /*!< 0x00002000 */
#define ADC_SAMPT0_SPT14_2                   (0x4U << ADC_SAMPT0_SPT14_Pos)      /*!< 0x00004000 */

#define ADC_SAMPT0_SPT15_Pos                 (15U)                              
#define ADC_SAMPT0_SPT15_Msk                 (0x7U << ADC_SAMPT0_SPT15_Pos)      /*!< 0x00038000 */
#define ADC_SAMPT0_SPT15                     ADC_SAMPT0_SPT15_Msk                /*!< ADC channel 15 sampling time selection  */
#define ADC_SAMPT0_SPT15_0                   (0x1U << ADC_SAMPT0_SPT15_Pos)      /*!< 0x00008000 */
#define ADC_SAMPT0_SPT15_1                   (0x2U << ADC_SAMPT0_SPT15_Pos)      /*!< 0x00010000 */
#define ADC_SAMPT0_SPT15_2                   (0x4U << ADC_SAMPT0_SPT15_Pos)      /*!< 0x00020000 */

#define ADC_SAMPT0_SPT16_Pos                 (18U)                              
#define ADC_SAMPT0_SPT16_Msk                 (0x7U << ADC_SAMPT0_SPT16_Pos)      /*!< 0x001C0000 */
#define ADC_SAMPT0_SPT16                     ADC_SAMPT0_SPT16_Msk                /*!< ADC channel 16 sampling time selection  */
#define ADC_SAMPT0_SPT16_0                   (0x1U << ADC_SAMPT0_SPT16_Pos)      /*!< 0x00040000 */
#define ADC_SAMPT0_SPT16_1                   (0x2U << ADC_SAMPT0_SPT16_Pos)      /*!< 0x00080000 */
#define ADC_SAMPT0_SPT16_2                   (0x4U << ADC_SAMPT0_SPT16_Pos)      /*!< 0x00100000 */

#define ADC_SAMPT0_SPT17_Pos                 (21U)                              
#define ADC_SAMPT0_SPT17_Msk                 (0x7U << ADC_SAMPT0_SPT17_Pos)      /*!< 0x00E00000 */
#define ADC_SAMPT0_SPT17                     ADC_SAMPT0_SPT17_Msk                /*!< ADC channel 17 sampling time selection  */
#define ADC_SAMPT0_SPT17_0                   (0x1U << ADC_SAMPT0_SPT17_Pos)      /*!< 0x00200000 */
#define ADC_SAMPT0_SPT17_1                   (0x2U << ADC_SAMPT0_SPT17_Pos)      /*!< 0x00400000 */
#define ADC_SAMPT0_SPT17_2                   (0x4U << ADC_SAMPT0_SPT17_Pos)      /*!< 0x00800000 */

/******************  Bit definition for ADC_SAMPT1 register  *******************/
#define ADC_SAMPT1_SPT0_Pos                  (0U)                               
#define ADC_SAMPT1_SPT0_Msk                  (0x7U << ADC_SAMPT1_SPT0_Pos)       /*!< 0x00000007 */
#define ADC_SAMPT1_SPT0                      ADC_SAMPT1_SPT0_Msk                 /*!< ADC channel 0 sampling time selection  */
#define ADC_SAMPT1_SPT0_0                    (0x1U << ADC_SAMPT1_SPT0_Pos)       /*!< 0x00000001 */
#define ADC_SAMPT1_SPT0_1                    (0x2U << ADC_SAMPT1_SPT0_Pos)       /*!< 0x00000002 */
#define ADC_SAMPT1_SPT0_2                    (0x4U << ADC_SAMPT1_SPT0_Pos)       /*!< 0x00000004 */

#define ADC_SAMPT1_SPT1_Pos                  (3U)                               
#define ADC_SAMPT1_SPT1_Msk                  (0x7U << ADC_SAMPT1_SPT1_Pos)       /*!< 0x00000038 */
#define ADC_SAMPT1_SPT1                      ADC_SAMPT1_SPT1_Msk                 /*!< ADC channel 1 sampling time selection  */
#define ADC_SAMPT1_SPT1_0                    (0x1U << ADC_SAMPT1_SPT1_Pos)       /*!< 0x00000008 */
#define ADC_SAMPT1_SPT1_1                    (0x2U << ADC_SAMPT1_SPT1_Pos)       /*!< 0x00000010 */
#define ADC_SAMPT1_SPT1_2                    (0x4U << ADC_SAMPT1_SPT1_Pos)       /*!< 0x00000020 */

#define ADC_SAMPT1_SPT2_Pos                  (6U)                               
#define ADC_SAMPT1_SPT2_Msk                  (0x7U << ADC_SAMPT1_SPT2_Pos)       /*!< 0x000001C0 */
#define ADC_SAMPT1_SPT2                      ADC_SAMPT1_SPT2_Msk                 /*!< ADC channel 2 sampling time selection  */
#define ADC_SAMPT1_SPT2_0                    (0x1U << ADC_SAMPT1_SPT2_Pos)       /*!< 0x00000040 */
#define ADC_SAMPT1_SPT2_1                    (0x2U << ADC_SAMPT1_SPT2_Pos)       /*!< 0x00000080 */
#define ADC_SAMPT1_SPT2_2                    (0x4U << ADC_SAMPT1_SPT2_Pos)       /*!< 0x00000100 */

#define ADC_SAMPT1_SPT3_Pos                  (9U)                               
#define ADC_SAMPT1_SPT3_Msk                  (0x7U << ADC_SAMPT1_SPT3_Pos)       /*!< 0x00000E00 */
#define ADC_SAMPT1_SPT3                      ADC_SAMPT1_SPT3_Msk                 /*!< ADC channel 3 sampling time selection  */
#define ADC_SAMPT1_SPT3_0                    (0x1U << ADC_SAMPT1_SPT3_Pos)       /*!< 0x00000200 */
#define ADC_SAMPT1_SPT3_1                    (0x2U << ADC_SAMPT1_SPT3_Pos)       /*!< 0x00000400 */
#define ADC_SAMPT1_SPT3_2                    (0x4U << ADC_SAMPT1_SPT3_Pos)       /*!< 0x00000800 */

#define ADC_SAMPT1_SPT4_Pos                  (12U)                              
#define ADC_SAMPT1_SPT4_Msk                  (0x7U << ADC_SAMPT1_SPT4_Pos)       /*!< 0x00007000 */
#define ADC_SAMPT1_SPT4                      ADC_SAMPT1_SPT4_Msk                 /*!< ADC channel 4 sampling time selection  */
#define ADC_SAMPT1_SPT4_0                    (0x1U << ADC_SAMPT1_SPT4_Pos)       /*!< 0x00001000 */
#define ADC_SAMPT1_SPT4_1                    (0x2U << ADC_SAMPT1_SPT4_Pos)       /*!< 0x00002000 */
#define ADC_SAMPT1_SPT4_2                    (0x4U << ADC_SAMPT1_SPT4_Pos)       /*!< 0x00004000 */

#define ADC_SAMPT1_SPT5_Pos                  (15U)                              
#define ADC_SAMPT1_SPT5_Msk                  (0x7U << ADC_SAMPT1_SPT5_Pos)       /*!< 0x00038000 */
#define ADC_SAMPT1_SPT5                      ADC_SAMPT1_SPT5_Msk                 /*!< ADC channel 5 sampling time selection  */
#define ADC_SAMPT1_SPT5_0                    (0x1U << ADC_SAMPT1_SPT5_Pos)       /*!< 0x00008000 */
#define ADC_SAMPT1_SPT5_1                    (0x2U << ADC_SAMPT1_SPT5_Pos)       /*!< 0x00010000 */
#define ADC_SAMPT1_SPT5_2                    (0x4U << ADC_SAMPT1_SPT5_Pos)       /*!< 0x00020000 */

#define ADC_SAMPT1_SPT6_Pos                  (18U)                              
#define ADC_SAMPT1_SPT6_Msk                  (0x7U << ADC_SAMPT1_SPT6_Pos)       /*!< 0x001C0000 */
#define ADC_SAMPT1_SPT6                      ADC_SAMPT1_SPT6_Msk                 /*!< ADC channel 6 sampling time selection  */
#define ADC_SAMPT1_SPT6_0                    (0x1U << ADC_SAMPT1_SPT6_Pos)       /*!< 0x00040000 */
#define ADC_SAMPT1_SPT6_1                    (0x2U << ADC_SAMPT1_SPT6_Pos)       /*!< 0x00080000 */
#define ADC_SAMPT1_SPT6_2                    (0x4U << ADC_SAMPT1_SPT6_Pos)       /*!< 0x00100000 */

#define ADC_SAMPT1_SPT7_Pos                  (21U)                              
#define ADC_SAMPT1_SPT7_Msk                  (0x7U << ADC_SAMPT1_SPT7_Pos)       /*!< 0x00E00000 */
#define ADC_SAMPT1_SPT7                      ADC_SAMPT1_SPT7_Msk                 /*!< ADC channel 7 sampling time selection  */
#define ADC_SAMPT1_SPT7_0                    (0x1U << ADC_SAMPT1_SPT7_Pos)       /*!< 0x00200000 */
#define ADC_SAMPT1_SPT7_1                    (0x2U << ADC_SAMPT1_SPT7_Pos)       /*!< 0x00400000 */
#define ADC_SAMPT1_SPT7_2                    (0x4U << ADC_SAMPT1_SPT7_Pos)       /*!< 0x00800000 */

#define ADC_SAMPT1_SPT8_Pos                  (24U)                              
#define ADC_SAMPT1_SPT8_Msk                  (0x7U << ADC_SAMPT1_SPT8_Pos)       /*!< 0x07000000 */
#define ADC_SAMPT1_SPT8                      ADC_SAMPT1_SPT8_Msk                 /*!< ADC channel 8 sampling time selection  */
#define ADC_SAMPT1_SPT8_0                    (0x1U << ADC_SAMPT1_SPT8_Pos)       /*!< 0x01000000 */
#define ADC_SAMPT1_SPT8_1                    (0x2U << ADC_SAMPT1_SPT8_Pos)       /*!< 0x02000000 */
#define ADC_SAMPT1_SPT8_2                    (0x4U << ADC_SAMPT1_SPT8_Pos)       /*!< 0x04000000 */

#define ADC_SAMPT1_SPT9_Pos                  (27U)                              
#define ADC_SAMPT1_SPT9_Msk                  (0x7U << ADC_SAMPT1_SPT9_Pos)       /*!< 0x38000000 */
#define ADC_SAMPT1_SPT9                      ADC_SAMPT1_SPT9_Msk                 /*!< ADC channel 9 sampling time selection  */
#define ADC_SAMPT1_SPT9_0                    (0x1U << ADC_SAMPT1_SPT9_Pos)       /*!< 0x08000000 */
#define ADC_SAMPT1_SPT9_1                    (0x2U << ADC_SAMPT1_SPT9_Pos)       /*!< 0x10000000 */
#define ADC_SAMPT1_SPT9_2                    (0x4U << ADC_SAMPT1_SPT9_Pos)       /*!< 0x20000000 */

/******************  Bit definition for ADC_IOFF0 register  *******************/
#define ADC_IOFF0_IOFF_Pos              (0U)                               
#define ADC_IOFF0_IOFF_Msk              (0xFFFU << ADC_IOFF0_IOFF_Pos) /*!< 0x00000FFF */
#define ADC_IOFF0_IOFF                  ADC_IOFF0_IOFF_Msk             /*!< ADC group injected sequencer rank 1 offset value */

/******************  Bit definition for ADC_IOFF1 register  *******************/
#define ADC_IOFF1_IOFF_Pos              (0U)                               
#define ADC_IOFF1_IOFF_Msk              (0xFFFU << ADC_IOFF1_IOFF_Pos) /*!< 0x00000FFF */
#define ADC_IOFF1_IOFF                  ADC_IOFF1_IOFF_Msk             /*!< ADC group injected sequencer rank 2 offset value */

/******************  Bit definition for ADC_IOFF2 register  *******************/
#define ADC_IOFF2_IOFF_Pos              (0U)                               
#define ADC_IOFF2_IOFF_Msk              (0xFFFU << ADC_IOFF2_IOFF_Pos) /*!< 0x00000FFF */
#define ADC_IOFF2_IOFF                  ADC_IOFF2_IOFF_Msk             /*!< ADC group injected sequencer rank 3 offset value */

/******************  Bit definition for ADC_IOFF3 register  *******************/
#define ADC_IOFF3_IOFF_Pos              (0U)                               
#define ADC_IOFF3_IOFF_Msk              (0xFFFU << ADC_IOFF3_IOFF_Pos) /*!< 0x00000FFF */
#define ADC_IOFF3_IOFF                  ADC_IOFF3_IOFF_Msk             /*!< ADC group injected sequencer rank 4 offset value */

/*******************  Bit definition for ADC_WDHT register  ********************/
#define ADC_WDHT_WDHT_Pos                      (0U)                               
#define ADC_WDHT_WDHT_Msk                      (0xFFFU << ADC_WDHT_WDHT_Pos)         /*!< 0x00000FFF */
#define ADC_WDHT_WDHT                          ADC_WDHT_WDHT_Msk                     /*!< ADC analog watchdog 1 threshold high */

/*******************  Bit definition for ADC_WDLT register  ********************/
#define ADC_WDLT_WDLT_Pos                      (0U)                               
#define ADC_WDLT_WDLT_Msk                      (0xFFFU << ADC_WDLT_WDLT_Pos)         /*!< 0x00000FFF */
#define ADC_WDLT_WDLT                          ADC_WDLT_WDLT_Msk                     /*!< ADC analog watchdog 1 threshold low */

/*******************  Bit definition for ADC_RSQ0 register  *******************/
#define ADC_RSQ0_RSQ13_Pos                   (0U)                               
#define ADC_RSQ0_RSQ13_Msk                   (0x1FU << ADC_RSQ0_RSQ13_Pos)       /*!< 0x0000001F */
#define ADC_RSQ0_RSQ13                       ADC_RSQ0_RSQ13_Msk                  /*!< ADC group regular sequencer rank 13 */
#define ADC_RSQ0_RSQ13_0                     (0x01U << ADC_RSQ0_RSQ13_Pos)       /*!< 0x00000001 */
#define ADC_RSQ0_RSQ13_1                     (0x02U << ADC_RSQ0_RSQ13_Pos)       /*!< 0x00000002 */
#define ADC_RSQ0_RSQ13_2                     (0x04U << ADC_RSQ0_RSQ13_Pos)       /*!< 0x00000004 */
#define ADC_RSQ0_RSQ13_3                     (0x08U << ADC_RSQ0_RSQ13_Pos)       /*!< 0x00000008 */
#define ADC_RSQ0_RSQ13_4                     (0x10U << ADC_RSQ0_RSQ13_Pos)       /*!< 0x00000010 */

#define ADC_RSQ0_RSQ14_Pos                   (5U)                               
#define ADC_RSQ0_RSQ14_Msk                   (0x1FU << ADC_RSQ0_RSQ14_Pos)       /*!< 0x000003E0 */
#define ADC_RSQ0_RSQ14                       ADC_RSQ0_RSQ14_Msk                  /*!< ADC group regular sequencer rank 14 */
#define ADC_RSQ0_RSQ14_0                     (0x01U << ADC_RSQ0_RSQ14_Pos)       /*!< 0x00000020 */
#define ADC_RSQ0_RSQ14_1                     (0x02U << ADC_RSQ0_RSQ14_Pos)       /*!< 0x00000040 */
#define ADC_RSQ0_RSQ14_2                     (0x04U << ADC_RSQ0_RSQ14_Pos)       /*!< 0x00000080 */
#define ADC_RSQ0_RSQ14_3                     (0x08U << ADC_RSQ0_RSQ14_Pos)       /*!< 0x00000100 */
#define ADC_RSQ0_RSQ14_4                     (0x10U << ADC_RSQ0_RSQ14_Pos)       /*!< 0x00000200 */

#define ADC_RSQ0_RSQ15_Pos                   (10U)                              
#define ADC_RSQ0_RSQ15_Msk                   (0x1FU << ADC_RSQ0_RSQ15_Pos)       /*!< 0x00007C00 */
#define ADC_RSQ0_RSQ15                       ADC_RSQ0_RSQ15_Msk                  /*!< ADC group regular sequencer rank 15 */
#define ADC_RSQ0_RSQ15_0                     (0x01U << ADC_RSQ0_RSQ15_Pos)       /*!< 0x00000400 */
#define ADC_RSQ0_RSQ15_1                     (0x02U << ADC_RSQ0_RSQ15_Pos)       /*!< 0x00000800 */
#define ADC_RSQ0_RSQ15_2                     (0x04U << ADC_RSQ0_RSQ15_Pos)       /*!< 0x00001000 */
#define ADC_RSQ0_RSQ15_3                     (0x08U << ADC_RSQ0_RSQ15_Pos)       /*!< 0x00002000 */
#define ADC_RSQ0_RSQ15_4                     (0x10U << ADC_RSQ0_RSQ15_Pos)       /*!< 0x00004000 */

#define ADC_RSQ0_RSQ16_Pos                   (15U)                              
#define ADC_RSQ0_RSQ16_Msk                   (0x1FU << ADC_RSQ0_RSQ16_Pos)       /*!< 0x000F8000 */
#define ADC_RSQ0_RSQ16                       ADC_RSQ0_RSQ16_Msk                  /*!< ADC group regular sequencer rank 16 */
#define ADC_RSQ0_RSQ16_0                     (0x01U << ADC_RSQ0_RSQ16_Pos)       /*!< 0x00008000 */
#define ADC_RSQ0_RSQ16_1                     (0x02U << ADC_RSQ0_RSQ16_Pos)       /*!< 0x00010000 */
#define ADC_RSQ0_RSQ16_2                     (0x04U << ADC_RSQ0_RSQ16_Pos)       /*!< 0x00020000 */
#define ADC_RSQ0_RSQ16_3                     (0x08U << ADC_RSQ0_RSQ16_Pos)       /*!< 0x00040000 */
#define ADC_RSQ0_RSQ16_4                     (0x10U << ADC_RSQ0_RSQ16_Pos)       /*!< 0x00080000 */

#define ADC_RSQ0_RL_Pos                      (20U)                              
#define ADC_RSQ0_RL_Msk                      (0xFU << ADC_RSQ0_RL_Pos)           /*!< 0x00F00000 */
#define ADC_RSQ0_RL                          ADC_RSQ0_RL_Msk                     /*!< ADC group regular sequencer scan length */
#define ADC_RSQ0_RL_0                        (0x1U << ADC_RSQ0_RL_Pos)           /*!< 0x00100000 */
#define ADC_RSQ0_RL_1                        (0x2U << ADC_RSQ0_RL_Pos)           /*!< 0x00200000 */
#define ADC_RSQ0_RL_2                        (0x4U << ADC_RSQ0_RL_Pos)           /*!< 0x00400000 */
#define ADC_RSQ0_RL_3                        (0x8U << ADC_RSQ0_RL_Pos)           /*!< 0x00800000 */

/*******************  Bit definition for ADC_RSQ1 register  *******************/
#define ADC_RSQ1_RSQ7_Pos                    (0U)                               
#define ADC_RSQ1_RSQ7_Msk                    (0x1FU << ADC_RSQ1_RSQ7_Pos)        /*!< 0x0000001F */
#define ADC_RSQ1_RSQ7                        ADC_RSQ1_RSQ7_Msk                   /*!< ADC group regular sequencer rank 7 */
#define ADC_RSQ1_RSQ7_0                      (0x01U << ADC_RSQ1_RSQ7_Pos)        /*!< 0x00000001 */
#define ADC_RSQ1_RSQ7_1                      (0x02U << ADC_RSQ1_RSQ7_Pos)        /*!< 0x00000002 */
#define ADC_RSQ1_RSQ7_2                      (0x04U << ADC_RSQ1_RSQ7_Pos)        /*!< 0x00000004 */
#define ADC_RSQ1_RSQ7_3                      (0x08U << ADC_RSQ1_RSQ7_Pos)        /*!< 0x00000008 */
#define ADC_RSQ1_RSQ7_4                      (0x10U << ADC_RSQ1_RSQ7_Pos)        /*!< 0x00000010 */

#define ADC_RSQ1_RSQ8_Pos                    (5U)                               
#define ADC_RSQ1_RSQ8_Msk                    (0x1FU << ADC_RSQ1_RSQ8_Pos)        /*!< 0x000003E0 */
#define ADC_RSQ1_RSQ8                        ADC_RSQ1_RSQ8_Msk                   /*!< ADC group regular sequencer rank 8 */
#define ADC_RSQ1_RSQ8_0                      (0x01U << ADC_RSQ1_RSQ8_Pos)        /*!< 0x00000020 */
#define ADC_RSQ1_RSQ8_1                      (0x02U << ADC_RSQ1_RSQ8_Pos)        /*!< 0x00000040 */
#define ADC_RSQ1_RSQ8_2                      (0x04U << ADC_RSQ1_RSQ8_Pos)        /*!< 0x00000080 */
#define ADC_RSQ1_RSQ8_3                      (0x08U << ADC_RSQ1_RSQ8_Pos)        /*!< 0x00000100 */
#define ADC_RSQ1_RSQ8_4                      (0x10U << ADC_RSQ1_RSQ8_Pos)        /*!< 0x00000200 */

#define ADC_RSQ1_RSQ9_Pos                    (10U)                              
#define ADC_RSQ1_RSQ9_Msk                    (0x1FU << ADC_RSQ1_RSQ9_Pos)        /*!< 0x00007C00 */
#define ADC_RSQ1_RSQ9                        ADC_RSQ1_RSQ9_Msk                   /*!< ADC group regular sequencer rank 9 */
#define ADC_RSQ1_RSQ9_0                      (0x01U << ADC_RSQ1_RSQ9_Pos)        /*!< 0x00000400 */
#define ADC_RSQ1_RSQ9_1                      (0x02U << ADC_RSQ1_RSQ9_Pos)        /*!< 0x00000800 */
#define ADC_RSQ1_RSQ9_2                      (0x04U << ADC_RSQ1_RSQ9_Pos)        /*!< 0x00001000 */
#define ADC_RSQ1_RSQ9_3                      (0x08U << ADC_RSQ1_RSQ9_Pos)        /*!< 0x00002000 */
#define ADC_RSQ1_RSQ9_4                      (0x10U << ADC_RSQ1_RSQ9_Pos)        /*!< 0x00004000 */

#define ADC_RSQ1_RSQ10_Pos                   (15U)                              
#define ADC_RSQ1_RSQ10_Msk                   (0x1FU << ADC_RSQ1_RSQ10_Pos)       /*!< 0x000F8000 */
#define ADC_RSQ1_RSQ10                       ADC_RSQ1_RSQ10_Msk                  /*!< ADC group regular sequencer rank 10 */
#define ADC_RSQ1_RSQ10_0                     (0x01U << ADC_RSQ1_RSQ10_Pos)       /*!< 0x00008000 */
#define ADC_RSQ1_RSQ10_1                     (0x02U << ADC_RSQ1_RSQ10_Pos)       /*!< 0x00010000 */
#define ADC_RSQ1_RSQ10_2                     (0x04U << ADC_RSQ1_RSQ10_Pos)       /*!< 0x00020000 */
#define ADC_RSQ1_RSQ10_3                     (0x08U << ADC_RSQ1_RSQ10_Pos)       /*!< 0x00040000 */
#define ADC_RSQ1_RSQ10_4                     (0x10U << ADC_RSQ1_RSQ10_Pos)       /*!< 0x00080000 */

#define ADC_RSQ1_RSQ11_Pos                   (20U)                              
#define ADC_RSQ1_RSQ11_Msk                   (0x1FU << ADC_RSQ1_RSQ11_Pos)       /*!< 0x01F00000 */
#define ADC_RSQ1_RSQ11                       ADC_RSQ1_RSQ11_Msk                  /*!< ADC group regular sequencer rank 1 */
#define ADC_RSQ1_RSQ11_0                     (0x01U << ADC_RSQ1_RSQ11_Pos)       /*!< 0x00100000 */
#define ADC_RSQ1_RSQ11_1                     (0x02U << ADC_RSQ1_RSQ11_Pos)       /*!< 0x00200000 */
#define ADC_RSQ1_RSQ11_2                     (0x04U << ADC_RSQ1_RSQ11_Pos)       /*!< 0x00400000 */
#define ADC_RSQ1_RSQ11_3                     (0x08U << ADC_RSQ1_RSQ11_Pos)       /*!< 0x00800000 */
#define ADC_RSQ1_RSQ11_4                     (0x10U << ADC_RSQ1_RSQ11_Pos)       /*!< 0x01000000 */

#define ADC_RSQ1_RSQ12_Pos                   (25U)                              
#define ADC_RSQ1_RSQ12_Msk                   (0x1FU << ADC_RSQ1_RSQ12_Pos)       /*!< 0x3E000000 */
#define ADC_RSQ1_RSQ12                       ADC_RSQ1_RSQ12_Msk                  /*!< ADC group regular sequencer rank 12 */
#define ADC_RSQ1_RSQ12_0                     (0x01U << ADC_RSQ1_RSQ12_Pos)       /*!< 0x02000000 */
#define ADC_RSQ1_RSQ12_1                     (0x02U << ADC_RSQ1_RSQ12_Pos)       /*!< 0x04000000 */
#define ADC_RSQ1_RSQ12_2                     (0x04U << ADC_RSQ1_RSQ12_Pos)       /*!< 0x08000000 */
#define ADC_RSQ1_RSQ12_3                     (0x08U << ADC_RSQ1_RSQ12_Pos)       /*!< 0x10000000 */
#define ADC_RSQ1_RSQ12_4                     (0x10U << ADC_RSQ1_RSQ12_Pos)       /*!< 0x20000000 */

/*******************  Bit definition for ADC_RSQ2 register  *******************/
#define ADC_RSQ2_RSQ1_Pos                    (0U)                               
#define ADC_RSQ2_RSQ1_Msk                    (0x1FU << ADC_RSQ2_RSQ1_Pos)        /*!< 0x0000001F */
#define ADC_RSQ2_RSQ1                        ADC_RSQ2_RSQ1_Msk                   /*!< ADC group regular sequencer rank 1 */
#define ADC_RSQ2_RSQ1_0                      (0x01U << ADC_RSQ2_RSQ1_Pos)        /*!< 0x00000001 */
#define ADC_RSQ2_RSQ1_1                      (0x02U << ADC_RSQ2_RSQ1_Pos)        /*!< 0x00000002 */
#define ADC_RSQ2_RSQ1_2                      (0x04U << ADC_RSQ2_RSQ1_Pos)        /*!< 0x00000004 */
#define ADC_RSQ2_RSQ1_3                      (0x08U << ADC_RSQ2_RSQ1_Pos)        /*!< 0x00000008 */
#define ADC_RSQ2_RSQ1_4                      (0x10U << ADC_RSQ2_RSQ1_Pos)        /*!< 0x00000010 */

#define ADC_RSQ2_RSQ2_Pos                    (5U)                               
#define ADC_RSQ2_RSQ2_Msk                    (0x1FU << ADC_RSQ2_RSQ2_Pos)        /*!< 0x000003E0 */
#define ADC_RSQ2_RSQ2                        ADC_RSQ2_RSQ2_Msk                   /*!< ADC group regular sequencer rank 2 */
#define ADC_RSQ2_RSQ2_0                      (0x01U << ADC_RSQ2_RSQ2_Pos)        /*!< 0x00000020 */
#define ADC_RSQ2_RSQ2_1                      (0x02U << ADC_RSQ2_RSQ2_Pos)        /*!< 0x00000040 */
#define ADC_RSQ2_RSQ2_2                      (0x04U << ADC_RSQ2_RSQ2_Pos)        /*!< 0x00000080 */
#define ADC_RSQ2_RSQ2_3                      (0x08U << ADC_RSQ2_RSQ2_Pos)        /*!< 0x00000100 */
#define ADC_RSQ2_RSQ2_4                      (0x10U << ADC_RSQ2_RSQ2_Pos)        /*!< 0x00000200 */

#define ADC_RSQ2_RSQ3_Pos                    (10U)                              
#define ADC_RSQ2_RSQ3_Msk                    (0x1FU << ADC_RSQ2_RSQ3_Pos)        /*!< 0x00007C00 */
#define ADC_RSQ2_RSQ3                        ADC_RSQ2_RSQ3_Msk                   /*!< ADC group regular sequencer rank 3 */
#define ADC_RSQ2_RSQ3_0                      (0x01U << ADC_RSQ2_RSQ3_Pos)        /*!< 0x00000400 */
#define ADC_RSQ2_RSQ3_1                      (0x02U << ADC_RSQ2_RSQ3_Pos)        /*!< 0x00000800 */
#define ADC_RSQ2_RSQ3_2                      (0x04U << ADC_RSQ2_RSQ3_Pos)        /*!< 0x00001000 */
#define ADC_RSQ2_RSQ3_3                      (0x08U << ADC_RSQ2_RSQ3_Pos)        /*!< 0x00002000 */
#define ADC_RSQ2_RSQ3_4                      (0x10U << ADC_RSQ2_RSQ3_Pos)        /*!< 0x00004000 */

#define ADC_RSQ2_RSQ4_Pos                    (15U)                              
#define ADC_RSQ2_RSQ4_Msk                    (0x1FU << ADC_RSQ2_RSQ4_Pos)        /*!< 0x000F8000 */
#define ADC_RSQ2_RSQ4                        ADC_RSQ2_RSQ4_Msk                   /*!< ADC group regular sequencer rank 4 */
#define ADC_RSQ2_RSQ4_0                      (0x01U << ADC_RSQ2_RSQ4_Pos)        /*!< 0x00008000 */
#define ADC_RSQ2_RSQ4_1                      (0x02U << ADC_RSQ2_RSQ4_Pos)        /*!< 0x00010000 */
#define ADC_RSQ2_RSQ4_2                      (0x04U << ADC_RSQ2_RSQ4_Pos)        /*!< 0x00020000 */
#define ADC_RSQ2_RSQ4_3                      (0x08U << ADC_RSQ2_RSQ4_Pos)        /*!< 0x00040000 */
#define ADC_RSQ2_RSQ4_4                      (0x10U << ADC_RSQ2_RSQ4_Pos)        /*!< 0x00080000 */

#define ADC_RSQ2_RSQ5_Pos                    (20U)                              
#define ADC_RSQ2_RSQ5_Msk                    (0x1FU << ADC_RSQ2_RSQ5_Pos)        /*!< 0x01F00000 */
#define ADC_RSQ2_RSQ5                        ADC_RSQ2_RSQ5_Msk                   /*!< ADC group regular sequencer rank 5 */
#define ADC_RSQ2_RSQ5_0                      (0x01U << ADC_RSQ2_RSQ5_Pos)        /*!< 0x00100000 */
#define ADC_RSQ2_RSQ5_1                      (0x02U << ADC_RSQ2_RSQ5_Pos)        /*!< 0x00200000 */
#define ADC_RSQ2_RSQ5_2                      (0x04U << ADC_RSQ2_RSQ5_Pos)        /*!< 0x00400000 */
#define ADC_RSQ2_RSQ5_3                      (0x08U << ADC_RSQ2_RSQ5_Pos)        /*!< 0x00800000 */
#define ADC_RSQ2_RSQ5_4                      (0x10U << ADC_RSQ2_RSQ5_Pos)        /*!< 0x01000000 */

#define ADC_RSQ2_RSQ6_Pos                    (25U)                              
#define ADC_RSQ2_RSQ6_Msk                    (0x1FU << ADC_RSQ2_RSQ6_Pos)        /*!< 0x3E000000 */
#define ADC_RSQ2_RSQ6                        ADC_RSQ2_RSQ6_Msk                   /*!< ADC group regular sequencer rank 6 */
#define ADC_RSQ2_RSQ6_0                      (0x01U << ADC_RSQ2_RSQ6_Pos)        /*!< 0x02000000 */
#define ADC_RSQ2_RSQ6_1                      (0x02U << ADC_RSQ2_RSQ6_Pos)        /*!< 0x04000000 */
#define ADC_RSQ2_RSQ6_2                      (0x04U << ADC_RSQ2_RSQ6_Pos)        /*!< 0x08000000 */
#define ADC_RSQ2_RSQ6_3                      (0x08U << ADC_RSQ2_RSQ6_Pos)        /*!< 0x10000000 */
#define ADC_RSQ2_RSQ6_4                      (0x10U << ADC_RSQ2_RSQ6_Pos)        /*!< 0x20000000 */

/*******************  Bit definition for ADC_ISQ register  *******************/
#define ADC_ISQ_ISQ1_Pos                   (0U)                               
#define ADC_ISQ_ISQ1_Msk                   (0x1FU << ADC_ISQ_ISQ1_Pos)       /*!< 0x0000001F */
#define ADC_ISQ_ISQ1                       ADC_ISQ_ISQ1_Msk                  /*!< ADC group injected sequencer rank 1 */
#define ADC_ISQ_ISQ1_0                     (0x01U << ADC_ISQ_ISQ1_Pos)       /*!< 0x00000001 */
#define ADC_ISQ_ISQ1_1                     (0x02U << ADC_ISQ_ISQ1_Pos)       /*!< 0x00000002 */
#define ADC_ISQ_ISQ1_2                     (0x04U << ADC_ISQ_ISQ1_Pos)       /*!< 0x00000004 */
#define ADC_ISQ_ISQ1_3                     (0x08U << ADC_ISQ_ISQ1_Pos)       /*!< 0x00000008 */
#define ADC_ISQ_ISQ1_4                     (0x10U << ADC_ISQ_ISQ1_Pos)       /*!< 0x00000010 */

#define ADC_ISQ_ISQ2_Pos                   (5U)                               
#define ADC_ISQ_ISQ2_Msk                   (0x1FU << ADC_ISQ_ISQ2_Pos)       /*!< 0x000003E0 */
#define ADC_ISQ_ISQ2                       ADC_ISQ_ISQ2_Msk                  /*!< ADC group injected sequencer rank 2 */
#define ADC_ISQ_ISQ2_0                     (0x01U << ADC_ISQ_ISQ2_Pos)       /*!< 0x00000020 */
#define ADC_ISQ_ISQ2_1                     (0x02U << ADC_ISQ_ISQ2_Pos)       /*!< 0x00000040 */
#define ADC_ISQ_ISQ2_2                     (0x04U << ADC_ISQ_ISQ2_Pos)       /*!< 0x00000080 */
#define ADC_ISQ_ISQ2_3                     (0x08U << ADC_ISQ_ISQ2_Pos)       /*!< 0x00000100 */
#define ADC_ISQ_ISQ2_4                     (0x10U << ADC_ISQ_ISQ2_Pos)       /*!< 0x00000200 */

#define ADC_ISQ_ISQ3_Pos                   (10U)                              
#define ADC_ISQ_ISQ3_Msk                   (0x1FU << ADC_ISQ_ISQ3_Pos)       /*!< 0x00007C00 */
#define ADC_ISQ_ISQ3                       ADC_ISQ_ISQ3_Msk                  /*!< ADC group injected sequencer rank 3 */
#define ADC_ISQ_ISQ3_0                     (0x01U << ADC_ISQ_ISQ3_Pos)       /*!< 0x00000400 */
#define ADC_ISQ_ISQ3_1                     (0x02U << ADC_ISQ_ISQ3_Pos)       /*!< 0x00000800 */
#define ADC_ISQ_ISQ3_2                     (0x04U << ADC_ISQ_ISQ3_Pos)       /*!< 0x00001000 */
#define ADC_ISQ_ISQ3_3                     (0x08U << ADC_ISQ_ISQ3_Pos)       /*!< 0x00002000 */
#define ADC_ISQ_ISQ3_4                     (0x10U << ADC_ISQ_ISQ3_Pos)       /*!< 0x00004000 */

#define ADC_ISQ_ISQ4_Pos                   (15U)                              
#define ADC_ISQ_ISQ4_Msk                   (0x1FU << ADC_ISQ_ISQ4_Pos)       /*!< 0x000F8000 */
#define ADC_ISQ_ISQ4                       ADC_ISQ_ISQ4_Msk                  /*!< ADC group injected sequencer rank 4 */
#define ADC_ISQ_ISQ4_0                     (0x01U << ADC_ISQ_ISQ4_Pos)       /*!< 0x00008000 */
#define ADC_ISQ_ISQ4_1                     (0x02U << ADC_ISQ_ISQ4_Pos)       /*!< 0x00010000 */
#define ADC_ISQ_ISQ4_2                     (0x04U << ADC_ISQ_ISQ4_Pos)       /*!< 0x00020000 */
#define ADC_ISQ_ISQ4_3                     (0x08U << ADC_ISQ_ISQ4_Pos)       /*!< 0x00040000 */
#define ADC_ISQ_ISQ4_4                     (0x10U << ADC_ISQ_ISQ4_Pos)       /*!< 0x00080000 */

#define ADC_ISQ_IL_Pos                     (20U)                              
#define ADC_ISQ_IL_Msk                     (0x3U << ADC_ISQ_IL_Pos)          /*!< 0x00300000 */
#define ADC_ISQ_IL                         ADC_ISQ_IL_Msk                    /*!< ADC group injected sequencer scan length */
#define ADC_ISQ_IL_0                       (0x1U << ADC_ISQ_IL_Pos)          /*!< 0x00100000 */
#define ADC_ISQ_IL_1                       (0x2U << ADC_ISQ_IL_Pos)          /*!< 0x00200000 */

/*******************  Bit definition for ADC_IDATA0 register  *******************/
#define ADC_IDATA0_IDATA_Pos                  (0U)                               
#define ADC_IDATA0_IDATA_Msk                  (0xFFFFU << ADC_IDATA0_IDATA_Pos)    /*!< 0x0000FFFF */
#define ADC_IDATA0_IDATA                      ADC_IDATA0_IDATA_Msk                 /*!< ADC group injected sequencer rank 1 conversion data */

/*******************  Bit definition for ADC_IDATA1 register  *******************/
#define ADC_IDATA1_IDATA_Pos                  (0U)                               
#define ADC_IDATA1_IDATA_Msk                  (0xFFFFU << ADC_IDATA1_IDATA_Pos)    /*!< 0x0000FFFF */
#define ADC_IDATA1_IDATA                      ADC_IDATA1_IDATA_Msk                 /*!< ADC group injected sequencer rank 2 conversion data */

/*******************  Bit definition for ADC_IDATA2 register  *******************/
#define ADC_IDATA2_IDATA_Pos                  (0U)                               
#define ADC_IDATA2_IDATA_Msk                  (0xFFFFU << ADC_IDATA2_IDATA_Pos)    /*!< 0x0000FFFF */
#define ADC_IDATA2_IDATA                      ADC_IDATA2_IDATA_Msk                 /*!< ADC group injected sequencer rank 3 conversion data */

/*******************  Bit definition for ADC_IDATA3 register  *******************/
#define ADC_IDATA3_IDATA_Pos                  (0U)                               
#define ADC_IDATA3_IDATA_Msk                  (0xFFFFU << ADC_IDATA3_IDATA_Pos)    /*!< 0x0000FFFF */
#define ADC_IDATA3_IDATA                      ADC_IDATA3_IDATA_Msk                 /*!< ADC group injected sequencer rank 4 conversion data */

/********************  Bit definition for ADC_RDATA register  ********************/
#define ADC_RDATA_RDATA_Pos                     (0U)                               
#define ADC_RDATA_RDATA_Msk                     (0xFFFFU << ADC_RDATA_RDATA_Pos)       /*!< 0x0000FFFF */
#define ADC_RDATA_RDATA                         ADC_RDATA_RDATA_Msk                    /*!< ADC group regular conversion data */
#define ADC_RDATA_ADC1RDTR_Pos                 (16U)                              
#define ADC_RDATA_ADC1RDTR_Msk                 (0xFFFFU << ADC_RDATA_ADC1RDTR_Pos)   /*!< 0xFFFF0000 */
#define ADC_RDATA_ADC1RDTR                     ADC_RDATA_ADC1RDTR_Msk                /*!< ADC group regular conversion data for ADC slave, in multimode */
/******************************************************************************/
/*                                                                            */
/*                      Digital to Analog Converter                           */
/*                                                                            */
/******************************************************************************/

/********************  Bit definition for DAC_CTL register  ********************/
#define DAC_CTL_DEN0_Pos                      (0U)                               
#define DAC_CTL_DEN0_Msk                      (0x1U << DAC_CTL_DEN0_Pos)           /*!< 0x00000001 */
#define DAC_CTL_DEN0                          DAC_CTL_DEN0_Msk                     /*!< DAC channel1 enable */
#define DAC_CTL_DBOFF0_Pos                    (1U)                               
#define DAC_CTL_DBOFF0_Msk                    (0x1U << DAC_CTL_DBOFF0_Pos)         /*!< 0x00000002 */
#define DAC_CTL_DBOFF0                        DAC_CTL_DBOFF0_Msk                   /*!< DAC channel1 output buffer disable */
#define DAC_CTL_DTEN0_Pos                     (2U)                               
#define DAC_CTL_DTEN0_Msk                     (0x1U << DAC_CTL_DTEN0_Pos)          /*!< 0x00000004 */
#define DAC_CTL_DTEN0                         DAC_CTL_DTEN0_Msk                    /*!< DAC channel1 Trigger enable */

#define DAC_CTL_DTSEL0_Pos                    (3U)                               
#define DAC_CTL_DTSEL0_Msk                    (0x7U << DAC_CTL_DTSEL0_Pos)         /*!< 0x00000038 */
#define DAC_CTL_DTSEL0                        DAC_CTL_DTSEL0_Msk                   /*!< TSEL1[2:0] (DAC channel1 Trigger selection) */
#define DAC_CTL_DTSEL0_0                      (0x1U << DAC_CTL_DTSEL0_Pos)         /*!< 0x00000008 */
#define DAC_CTL_DTSEL0_1                      (0x2U << DAC_CTL_DTSEL0_Pos)         /*!< 0x00000010 */
#define DAC_CTL_DTSEL0_2                      (0x4U << DAC_CTL_DTSEL0_Pos)         /*!< 0x00000020 */

#define DAC_CTL_DWM0_Pos                    (6U)                               
#define DAC_CTL_DWM0_Msk                    (0x3U << DAC_CTL_DWM0_Pos)         /*!< 0x000000C0 */
#define DAC_CTL_DWM0                        DAC_CTL_DWM0_Msk                   /*!< WAVE1[1:0] (DAC channel1 noise/triangle wave generation enable) */
#define DAC_CTL_DWM0_0                      (0x1U << DAC_CTL_DWM0_Pos)         /*!< 0x00000040 */
#define DAC_CTL_DWM0_1                      (0x2U << DAC_CTL_DWM0_Pos)         /*!< 0x00000080 */

#define DAC_CTL_DWBW0_Pos                    (8U)                               
#define DAC_CTL_DWBW0_Msk                    (0xFU << DAC_CTL_DWBW0_Pos)         /*!< 0x00000F00 */
#define DAC_CTL_DWBW0                        DAC_CTL_DWBW0_Msk                   /*!< MAMP1[3:0] (DAC channel1 Mask/Amplitude selector) */
#define DAC_CTL_DWBW0_0                      (0x1U << DAC_CTL_DWBW0_Pos)         /*!< 0x00000100 */
#define DAC_CTL_DWBW0_1                      (0x2U << DAC_CTL_DWBW0_Pos)         /*!< 0x00000200 */
#define DAC_CTL_DWBW0_2                      (0x4U << DAC_CTL_DWBW0_Pos)         /*!< 0x00000400 */
#define DAC_CTL_DWBW0_3                      (0x8U << DAC_CTL_DWBW0_Pos)         /*!< 0x00000800 */

#define DAC_CTL_DDMAEN0_Pos                   (12U)                              
#define DAC_CTL_DDMAEN0_Msk                   (0x1U << DAC_CTL_DDMAEN0_Pos)        /*!< 0x00001000 */
#define DAC_CTL_DDMAEN0                       DAC_CTL_DDMAEN0_Msk                  /*!< DAC channel1 DMA enable */
#define DAC_CTL_DEN1_Pos                      (16U)                              
#define DAC_CTL_DEN1_Msk                      (0x1U << DAC_CTL_DEN1_Pos)           /*!< 0x00010000 */
#define DAC_CTL_DEN1                          DAC_CTL_DEN1_Msk                     /*!< DAC channel2 enable */
#define DAC_CTL_DBOFF1_Pos                    (17U)                              
#define DAC_CTL_DBOFF1_Msk                    (0x1U << DAC_CTL_DBOFF1_Pos)         /*!< 0x00020000 */
#define DAC_CTL_DBOFF1                        DAC_CTL_DBOFF1_Msk                   /*!< DAC channel2 output buffer disable */
#define DAC_CTL_DTEN1_Pos                     (18U)                              
#define DAC_CTL_DTEN1_Msk                     (0x1U << DAC_CTL_DTEN1_Pos)          /*!< 0x00040000 */
#define DAC_CTL_DTEN1                         DAC_CTL_DTEN1_Msk                    /*!< DAC channel2 Trigger enable */

#define DAC_CTL_DTSEL1_Pos                    (19U)                              
#define DAC_CTL_DTSEL1_Msk                    (0x7U << DAC_CTL_DTSEL1_Pos)         /*!< 0x00380000 */
#define DAC_CTL_DTSEL1                        DAC_CTL_DTSEL1_Msk                   /*!< TSEL2[2:0] (DAC channel2 Trigger selection) */
#define DAC_CTL_DTSEL1_0                      (0x1U << DAC_CTL_DTSEL1_Pos)         /*!< 0x00080000 */
#define DAC_CTL_DTSEL1_1                      (0x2U << DAC_CTL_DTSEL1_Pos)         /*!< 0x00100000 */
#define DAC_CTL_DTSEL1_2                      (0x4U << DAC_CTL_DTSEL1_Pos)         /*!< 0x00200000 */

#define DAC_CTL_DWM1_Pos                    (22U)                              
#define DAC_CTL_DWM1_Msk                    (0x3U << DAC_CTL_DWM1_Pos)         /*!< 0x00C00000 */
#define DAC_CTL_DWM1                        DAC_CTL_DWM1_Msk                   /*!< WAVE2[1:0] (DAC channel2 noise/triangle wave generation enable) */
#define DAC_CTL_DWM1_0                      (0x1U << DAC_CTL_DWM1_Pos)         /*!< 0x00400000 */
#define DAC_CTL_DWM1_1                      (0x2U << DAC_CTL_DWM1_Pos)         /*!< 0x00800000 */

#define DAC_CTL_DWBW1_Pos                    (24U)                              
#define DAC_CTL_DWBW1_Msk                    (0xFU << DAC_CTL_DWBW1_Pos)         /*!< 0x0F000000 */
#define DAC_CTL_DWBW1                        DAC_CTL_DWBW1_Msk                   /*!< MAMP2[3:0] (DAC channel2 Mask/Amplitude selector) */
#define DAC_CTL_DWBW1_0                      (0x1U << DAC_CTL_DWBW1_Pos)         /*!< 0x01000000 */
#define DAC_CTL_DWBW1_1                      (0x2U << DAC_CTL_DWBW1_Pos)         /*!< 0x02000000 */
#define DAC_CTL_DWBW1_2                      (0x4U << DAC_CTL_DWBW1_Pos)         /*!< 0x04000000 */
#define DAC_CTL_DWBW1_3                      (0x8U << DAC_CTL_DWBW1_Pos)         /*!< 0x08000000 */

#define DAC_CTL_DDMAEN1_Pos                   (28U)                              
#define DAC_CTL_DDMAEN1_Msk                   (0x1U << DAC_CTL_DDMAEN1_Pos)        /*!< 0x10000000 */
#define DAC_CTL_DDMAEN1                       DAC_CTL_DDMAEN1_Msk                  /*!< DAC channel2 DMA enabled */


/*****************  Bit definition for DAC_SWT register  ******************/
#define DAC_SWT_SWTR0_Pos             (0U)                               
#define DAC_SWT_SWTR0_Msk             (0x1U << DAC_SWT_SWTR0_Pos)  /*!< 0x00000001 */
#define DAC_SWT_SWTR0                 DAC_SWT_SWTR0_Msk            /*!< DAC channel1 software trigger */
#define DAC_SWT_SWTR1_Pos             (1U)                               
#define DAC_SWT_SWTR1_Msk             (0x1U << DAC_SWT_SWTR1_Pos)  /*!< 0x00000002 */
#define DAC_SWT_SWTR1                 DAC_SWT_SWTR1_Msk            /*!< DAC channel2 software trigger */

/*****************  Bit definition for DAC_DAC0_R12DH register  ******************/
#define DAC_DAC0_R12DH_DAC0_DH_Pos            (0U)                               
#define DAC_DAC0_R12DH_DAC0_DH_Msk            (0xFFFU << DAC_DAC0_R12DH_DAC0_DH_Pos) /*!< 0x00000FFF */
#define DAC_DAC0_R12DH_DAC0_DH                DAC_DAC0_R12DH_DAC0_DH_Msk           /*!< DAC channel1 12-bit Right aligned data */

/*****************  Bit definition for DAC_DAC0_L12DH register  ******************/
#define DAC_DAC0_L12DH_DAC0_DH_Pos            (4U)                               
#define DAC_DAC0_L12DH_DAC0_DH_Msk            (0xFFFU << DAC_DAC0_L12DH_DAC0_DH_Pos) /*!< 0x0000FFF0 */
#define DAC_DAC0_L12DH_DAC0_DH                DAC_DAC0_L12DH_DAC0_DH_Msk           /*!< DAC channel1 12-bit Left aligned data */

/******************  Bit definition for DAC_DAC0_R8DH register  ******************/
#define DAC_DAC0_R8DH_DAC0_DH_Pos             (0U)                               
#define DAC_DAC0_R8DH_DAC0_DH_Msk             (0xFFU << DAC_DAC0_R8DH_DAC0_DH_Pos) /*!< 0x000000FF */
#define DAC_DAC0_R8DH_DAC0_DH                 DAC_DAC0_R8DH_DAC0_DH_Msk            /*!< DAC channel1 8-bit Right aligned data */

/*****************  Bit definition for DAC_DAC1_R12DH register  ******************/
#define DAC_DAC1_R12DH_DAC1_DH_Pos            (0U)                               
#define DAC_DAC1_R12DH_DAC1_DH_Msk            (0xFFFU << DAC_DAC1_R12DH_DAC1_DH_Pos) /*!< 0x00000FFF */
#define DAC_DAC1_R12DH_DAC1_DH                DAC_DAC1_R12DH_DAC1_DH_Msk           /*!< DAC channel2 12-bit Right aligned data */

/*****************  Bit definition for DAC_DAC1_L12DH register  ******************/
#define DAC_DAC1_L12DH_DAC1_DH_Pos            (4U)                               
#define DAC_DAC1_L12DH_DAC1_DH_Msk            (0xFFFU << DAC_DAC1_L12DH_DAC1_DH_Pos) /*!< 0x0000FFF0 */
#define DAC_DAC1_L12DH_DAC1_DH                DAC_DAC1_L12DH_DAC1_DH_Msk           /*!< DAC channel2 12-bit Left aligned data */

/******************  Bit definition for DAC_DAC1_R8DH register  ******************/
#define DAC_DAC1_R8DH_DAC1_DH_Pos             (0U)                               
#define DAC_DAC1_R8DH_DAC1_DH_Msk             (0xFFU << DAC_DAC1_R8DH_DAC1_DH_Pos) /*!< 0x000000FF */
#define DAC_DAC1_R8DH_DAC1_DH                 DAC_DAC1_R8DH_DAC1_DH_Msk            /*!< DAC channel2 8-bit Right aligned data */

/*****************  Bit definition for DAC_DACC_R12DH register  ******************/
#define DAC_DACC_R12DH_DAC0_DH_Pos            (0U)                               
#define DAC_DACC_R12DH_DAC0_DH_Msk            (0xFFFU << DAC_DACC_R12DH_DAC0_DH_Pos) /*!< 0x00000FFF */
#define DAC_DACC_R12DH_DAC0_DH                DAC_DACC_R12DH_DAC0_DH_Msk           /*!< DAC channel1 12-bit Right aligned data */
#define DAC_DACC_R12DH_DAC1_DH_Pos            (16U)                              
#define DAC_DACC_R12DH_DAC1_DH_Msk            (0xFFFU << DAC_DACC_R12DH_DAC1_DH_Pos) /*!< 0x0FFF0000 */
#define DAC_DACC_R12DH_DAC1_DH                DAC_DACC_R12DH_DAC1_DH_Msk           /*!< DAC channel2 12-bit Right aligned data */

/*****************  Bit definition for DAC_DACC_L12DH register  ******************/
#define DAC_DACC_L12DH_DAC0_DH_Pos            (4U)                               
#define DAC_DACC_L12DH_DAC0_DH_Msk            (0xFFFU << DAC_DACC_L12DH_DAC0_DH_Pos) /*!< 0x0000FFF0 */
#define DAC_DACC_L12DH_DAC0_DH                DAC_DACC_L12DH_DAC0_DH_Msk           /*!< DAC channel1 12-bit Left aligned data */
#define DAC_DACC_L12DH_DAC1_DH_Pos            (20U)                              
#define DAC_DACC_L12DH_DAC1_DH_Msk            (0xFFFU << DAC_DACC_L12DH_DAC1_DH_Pos) /*!< 0xFFF00000 */
#define DAC_DACC_L12DH_DAC1_DH                DAC_DACC_L12DH_DAC1_DH_Msk           /*!< DAC channel2 12-bit Left aligned data */

/******************  Bit definition for DAC_DACC_R8DH register  ******************/
#define DAC_DACC_R8DH_DAC0_DH_Pos             (0U)                               
#define DAC_DACC_R8DH_DAC0_DH_Msk             (0xFFU << DAC_DACC_R8DH_DAC0_DH_Pos) /*!< 0x000000FF */
#define DAC_DACC_R8DH_DAC0_DH                 DAC_DACC_R8DH_DAC0_DH_Msk            /*!< DAC channel1 8-bit Right aligned data */
#define DAC_DACC_R8DH_DAC1_DH_Pos             (8U)                               
#define DAC_DACC_R8DH_DAC1_DH_Msk             (0xFFU << DAC_DACC_R8DH_DAC1_DH_Pos) /*!< 0x0000FF00 */
#define DAC_DACC_R8DH_DAC1_DH                 DAC_DACC_R8DH_DAC1_DH_Msk            /*!< DAC channel2 8-bit Right aligned data */

/*******************  Bit definition for DAC_DAC0_DO register  *******************/
#define DAC_DAC0_DO_DAC0_DO_Pos               (0U)                               
#define DAC_DAC0_DO_DAC0_DO_Msk               (0xFFFU << DAC_DAC0_DO_DAC0_DO_Pos)  /*!< 0x00000FFF */
#define DAC_DAC0_DO_DAC0_DO                   DAC_DAC0_DO_DAC0_DO_Msk              /*!< DAC channel1 data output */

/*******************  Bit definition for DAC_DAC1_DO register  *******************/
#define DAC_DAC1_DO_DAC1_DO_Pos               (0U)                               
#define DAC_DAC1_DO_DAC1_DO_Msk               (0xFFFU << DAC_DAC1_DO_DAC1_DO_Pos)  /*!< 0x00000FFF */
#define DAC_DAC1_DO_DAC1_DO                   DAC_DAC1_DO_DAC1_DO_Msk              /*!< DAC channel2 data output */

/*****************************************************************************/
/*                                                                           */
/*                               Timers (TIM)                                */
/*                                                                           */
/*****************************************************************************/
/*******************  Bit definition for TIM_CR1 register  *******************/
#define TIM_CR1_CEN_Pos                     (0U)                               
#define TIM_CR1_CEN_Msk                     (0x1U << TIM_CR1_CEN_Pos)          /*!< 0x00000001 */
#define TIM_CR1_CEN                         TIM_CR1_CEN_Msk                    /*!<Counter enable */
#define TIM_CR1_UDIS_Pos                    (1U)                               
#define TIM_CR1_UDIS_Msk                    (0x1U << TIM_CR1_UDIS_Pos)         /*!< 0x00000002 */
#define TIM_CR1_UDIS                        TIM_CR1_UDIS_Msk                   /*!<Update disable */
#define TIM_CR1_URS_Pos                     (2U)                               
#define TIM_CR1_URS_Msk                     (0x1U << TIM_CR1_URS_Pos)          /*!< 0x00000004 */
#define TIM_CR1_URS                         TIM_CR1_URS_Msk                    /*!<Update request source */
#define TIM_CR1_OPM_Pos                     (3U)                               
#define TIM_CR1_OPM_Msk                     (0x1U << TIM_CR1_OPM_Pos)          /*!< 0x00000008 */
#define TIM_CR1_OPM                         TIM_CR1_OPM_Msk                    /*!<One pulse mode */
#define TIM_CR1_DIR_Pos                     (4U)                               
#define TIM_CR1_DIR_Msk                     (0x1U << TIM_CR1_DIR_Pos)          /*!< 0x00000010 */
#define TIM_CR1_DIR                         TIM_CR1_DIR_Msk                    /*!<Direction */

#define TIM_CR1_CMS_Pos                     (5U)                               
#define TIM_CR1_CMS_Msk                     (0x3U << TIM_CR1_CMS_Pos)          /*!< 0x00000060 */
#define TIM_CR1_CMS                         TIM_CR1_CMS_Msk                    /*!<CMS[1:0] bits (Center-aligned mode selection) */
#define TIM_CR1_CMS_0                       (0x1U << TIM_CR1_CMS_Pos)          /*!< 0x00000020 */
#define TIM_CR1_CMS_1                       (0x2U << TIM_CR1_CMS_Pos)          /*!< 0x00000040 */

#define TIM_CR1_ARPE_Pos                    (7U)                               
#define TIM_CR1_ARPE_Msk                    (0x1U << TIM_CR1_ARPE_Pos)         /*!< 0x00000080 */
#define TIM_CR1_ARPE                        TIM_CR1_ARPE_Msk                   /*!<Auto-reload preload enable */

#define TIM_CR1_CKD_Pos                     (8U)                               
#define TIM_CR1_CKD_Msk                     (0x3U << TIM_CR1_CKD_Pos)          /*!< 0x00000300 */
#define TIM_CR1_CKD                         TIM_CR1_CKD_Msk                    /*!<CKD[1:0] bits (clock division) */
#define TIM_CR1_CKD_0                       (0x1U << TIM_CR1_CKD_Pos)          /*!< 0x00000100 */
#define TIM_CR1_CKD_1                       (0x2U << TIM_CR1_CKD_Pos)          /*!< 0x00000200 */

/*******************  Bit definition for TIM_CR2 register  *******************/
#define TIM_CR2_CCPC_Pos                    (0U)                               
#define TIM_CR2_CCPC_Msk                    (0x1U << TIM_CR2_CCPC_Pos)         /*!< 0x00000001 */
#define TIM_CR2_CCPC                        TIM_CR2_CCPC_Msk                   /*!<Capture/Compare Preloaded Control */
#define TIM_CR2_CCUS_Pos                    (2U)                               
#define TIM_CR2_CCUS_Msk                    (0x1U << TIM_CR2_CCUS_Pos)         /*!< 0x00000004 */
#define TIM_CR2_CCUS                        TIM_CR2_CCUS_Msk                   /*!<Capture/Compare Control Update Selection */
#define TIM_CR2_CCDS_Pos                    (3U)                               
#define TIM_CR2_CCDS_Msk                    (0x1U << TIM_CR2_CCDS_Pos)         /*!< 0x00000008 */
#define TIM_CR2_CCDS                        TIM_CR2_CCDS_Msk                   /*!<Capture/Compare DMA Selection */

#define TIM_CR2_MMS_Pos                     (4U)                               
#define TIM_CR2_MMS_Msk                     (0x7U << TIM_CR2_MMS_Pos)          /*!< 0x00000070 */
#define TIM_CR2_MMS                         TIM_CR2_MMS_Msk                    /*!<MMS[2:0] bits (Master Mode Selection) */
#define TIM_CR2_MMS_0                       (0x1U << TIM_CR2_MMS_Pos)          /*!< 0x00000010 */
#define TIM_CR2_MMS_1                       (0x2U << TIM_CR2_MMS_Pos)          /*!< 0x00000020 */
#define TIM_CR2_MMS_2                       (0x4U << TIM_CR2_MMS_Pos)          /*!< 0x00000040 */

#define TIM_CR2_TI1S_Pos                    (7U)                               
#define TIM_CR2_TI1S_Msk                    (0x1U << TIM_CR2_TI1S_Pos)         /*!< 0x00000080 */
#define TIM_CR2_TI1S                        TIM_CR2_TI1S_Msk                   /*!<TI1 Selection */
#define TIM_CR2_OIS1_Pos                    (8U)                               
#define TIM_CR2_OIS1_Msk                    (0x1U << TIM_CR2_OIS1_Pos)         /*!< 0x00000100 */
#define TIM_CR2_OIS1                        TIM_CR2_OIS1_Msk                   /*!<Output Idle state 1 (OC1 output) */
#define TIM_CR2_OIS1N_Pos                   (9U)                               
#define TIM_CR2_OIS1N_Msk                   (0x1U << TIM_CR2_OIS1N_Pos)        /*!< 0x00000200 */
#define TIM_CR2_OIS1N                       TIM_CR2_OIS1N_Msk                  /*!<Output Idle state 1 (OC1N output) */
#define TIM_CR2_OIS2_Pos                    (10U)                              
#define TIM_CR2_OIS2_Msk                    (0x1U << TIM_CR2_OIS2_Pos)         /*!< 0x00000400 */
#define TIM_CR2_OIS2                        TIM_CR2_OIS2_Msk                   /*!<Output Idle state 2 (OC2 output) */
#define TIM_CR2_OIS2N_Pos                   (11U)                              
#define TIM_CR2_OIS2N_Msk                   (0x1U << TIM_CR2_OIS2N_Pos)        /*!< 0x00000800 */
#define TIM_CR2_OIS2N                       TIM_CR2_OIS2N_Msk                  /*!<Output Idle state 2 (OC2N output) */
#define TIM_CR2_OIS3_Pos                    (12U)                              
#define TIM_CR2_OIS3_Msk                    (0x1U << TIM_CR2_OIS3_Pos)         /*!< 0x00001000 */
#define TIM_CR2_OIS3                        TIM_CR2_OIS3_Msk                   /*!<Output Idle state 3 (OC3 output) */
#define TIM_CR2_OIS3N_Pos                   (13U)                              
#define TIM_CR2_OIS3N_Msk                   (0x1U << TIM_CR2_OIS3N_Pos)        /*!< 0x00002000 */
#define TIM_CR2_OIS3N                       TIM_CR2_OIS3N_Msk                  /*!<Output Idle state 3 (OC3N output) */
#define TIM_CR2_OIS4_Pos                    (14U)                              
#define TIM_CR2_OIS4_Msk                    (0x1U << TIM_CR2_OIS4_Pos)         /*!< 0x00004000 */
#define TIM_CR2_OIS4                        TIM_CR2_OIS4_Msk                   /*!<Output Idle state 4 (OC4 output) */

/*******************  Bit definition for TIM_SMCR register  ******************/
#define TIM_SMCR_SMS_Pos                    (0U)                               
#define TIM_SMCR_SMS_Msk                    (0x7U << TIM_SMCR_SMS_Pos)         /*!< 0x00000007 */
#define TIM_SMCR_SMS                        TIM_SMCR_SMS_Msk                   /*!<SMS[2:0] bits (Slave mode selection) */
#define TIM_SMCR_SMS_0                      (0x1U << TIM_SMCR_SMS_Pos)         /*!< 0x00000001 */
#define TIM_SMCR_SMS_1                      (0x2U << TIM_SMCR_SMS_Pos)         /*!< 0x00000002 */
#define TIM_SMCR_SMS_2                      (0x4U << TIM_SMCR_SMS_Pos)         /*!< 0x00000004 */

#define TIM_SMCR_TS_Pos                     (4U)                               
#define TIM_SMCR_TS_Msk                     (0x7U << TIM_SMCR_TS_Pos)          /*!< 0x00000070 */
#define TIM_SMCR_TS                         TIM_SMCR_TS_Msk                    /*!<TS[2:0] bits (Trigger selection) */
#define TIM_SMCR_TS_0                       (0x1U << TIM_SMCR_TS_Pos)          /*!< 0x00000010 */
#define TIM_SMCR_TS_1                       (0x2U << TIM_SMCR_TS_Pos)          /*!< 0x00000020 */
#define TIM_SMCR_TS_2                       (0x4U << TIM_SMCR_TS_Pos)          /*!< 0x00000040 */

#define TIM_SMCR_MSM_Pos                    (7U)                               
#define TIM_SMCR_MSM_Msk                    (0x1U << TIM_SMCR_MSM_Pos)         /*!< 0x00000080 */
#define TIM_SMCR_MSM                        TIM_SMCR_MSM_Msk                   /*!<Master/slave mode */

#define TIM_SMCR_ETF_Pos                    (8U)                               
#define TIM_SMCR_ETF_Msk                    (0xFU << TIM_SMCR_ETF_Pos)         /*!< 0x00000F00 */
#define TIM_SMCR_ETF                        TIM_SMCR_ETF_Msk                   /*!<ETF[3:0] bits (External trigger filter) */
#define TIM_SMCR_ETF_0                      (0x1U << TIM_SMCR_ETF_Pos)         /*!< 0x00000100 */
#define TIM_SMCR_ETF_1                      (0x2U << TIM_SMCR_ETF_Pos)         /*!< 0x00000200 */
#define TIM_SMCR_ETF_2                      (0x4U << TIM_SMCR_ETF_Pos)         /*!< 0x00000400 */
#define TIM_SMCR_ETF_3                      (0x8U << TIM_SMCR_ETF_Pos)         /*!< 0x00000800 */

#define TIM_SMCR_ETPS_Pos                   (12U)                              
#define TIM_SMCR_ETPS_Msk                   (0x3U << TIM_SMCR_ETPS_Pos)        /*!< 0x00003000 */
#define TIM_SMCR_ETPS                       TIM_SMCR_ETPS_Msk                  /*!<ETPS[1:0] bits (External trigger prescaler) */
#define TIM_SMCR_ETPS_0                     (0x1U << TIM_SMCR_ETPS_Pos)        /*!< 0x00001000 */
#define TIM_SMCR_ETPS_1                     (0x2U << TIM_SMCR_ETPS_Pos)        /*!< 0x00002000 */

#define TIM_SMCR_ECE_Pos                    (14U)                              
#define TIM_SMCR_ECE_Msk                    (0x1U << TIM_SMCR_ECE_Pos)         /*!< 0x00004000 */
#define TIM_SMCR_ECE                        TIM_SMCR_ECE_Msk                   /*!<External clock enable */
#define TIM_SMCR_ETP_Pos                    (15U)                              
#define TIM_SMCR_ETP_Msk                    (0x1U << TIM_SMCR_ETP_Pos)         /*!< 0x00008000 */
#define TIM_SMCR_ETP                        TIM_SMCR_ETP_Msk                   /*!<External trigger polarity */

/*******************  Bit definition for TIM_DIER register  ******************/
#define TIM_DIER_UIE_Pos                    (0U)                               
#define TIM_DIER_UIE_Msk                    (0x1U << TIM_DIER_UIE_Pos)         /*!< 0x00000001 */
#define TIM_DIER_UIE                        TIM_DIER_UIE_Msk                   /*!<Update interrupt enable */
#define TIM_DIER_CC1IE_Pos                  (1U)                               
#define TIM_DIER_CC1IE_Msk                  (0x1U << TIM_DIER_CC1IE_Pos)       /*!< 0x00000002 */
#define TIM_DIER_CC1IE                      TIM_DIER_CC1IE_Msk                 /*!<Capture/Compare 1 interrupt enable */
#define TIM_DIER_CC2IE_Pos                  (2U)                               
#define TIM_DIER_CC2IE_Msk                  (0x1U << TIM_DIER_CC2IE_Pos)       /*!< 0x00000004 */
#define TIM_DIER_CC2IE                      TIM_DIER_CC2IE_Msk                 /*!<Capture/Compare 2 interrupt enable */
#define TIM_DIER_CC3IE_Pos                  (3U)                               
#define TIM_DIER_CC3IE_Msk                  (0x1U << TIM_DIER_CC3IE_Pos)       /*!< 0x00000008 */
#define TIM_DIER_CC3IE                      TIM_DIER_CC3IE_Msk                 /*!<Capture/Compare 3 interrupt enable */
#define TIM_DIER_CC4IE_Pos                  (4U)                               
#define TIM_DIER_CC4IE_Msk                  (0x1U << TIM_DIER_CC4IE_Pos)       /*!< 0x00000010 */
#define TIM_DIER_CC4IE                      TIM_DIER_CC4IE_Msk                 /*!<Capture/Compare 4 interrupt enable */
#define TIM_DIER_COMIE_Pos                  (5U)                               
#define TIM_DIER_COMIE_Msk                  (0x1U << TIM_DIER_COMIE_Pos)       /*!< 0x00000020 */
#define TIM_DIER_COMIE                      TIM_DIER_COMIE_Msk                 /*!<COM interrupt enable */
#define TIM_DIER_TIE_Pos                    (6U)                               
#define TIM_DIER_TIE_Msk                    (0x1U << TIM_DIER_TIE_Pos)         /*!< 0x00000040 */
#define TIM_DIER_TIE                        TIM_DIER_TIE_Msk                   /*!<Trigger interrupt enable */
#define TIM_DIER_BIE_Pos                    (7U)                               
#define TIM_DIER_BIE_Msk                    (0x1U << TIM_DIER_BIE_Pos)         /*!< 0x00000080 */
#define TIM_DIER_BIE                        TIM_DIER_BIE_Msk                   /*!<Break interrupt enable */
#define TIM_DIER_UDE_Pos                    (8U)                               
#define TIM_DIER_UDE_Msk                    (0x1U << TIM_DIER_UDE_Pos)         /*!< 0x00000100 */
#define TIM_DIER_UDE                        TIM_DIER_UDE_Msk                   /*!<Update DMA request enable */
#define TIM_DIER_CC1DE_Pos                  (9U)                               
#define TIM_DIER_CC1DE_Msk                  (0x1U << TIM_DIER_CC1DE_Pos)       /*!< 0x00000200 */
#define TIM_DIER_CC1DE                      TIM_DIER_CC1DE_Msk                 /*!<Capture/Compare 1 DMA request enable */
#define TIM_DIER_CC2DE_Pos                  (10U)                              
#define TIM_DIER_CC2DE_Msk                  (0x1U << TIM_DIER_CC2DE_Pos)       /*!< 0x00000400 */
#define TIM_DIER_CC2DE                      TIM_DIER_CC2DE_Msk                 /*!<Capture/Compare 2 DMA request enable */
#define TIM_DIER_CC3DE_Pos                  (11U)                              
#define TIM_DIER_CC3DE_Msk                  (0x1U << TIM_DIER_CC3DE_Pos)       /*!< 0x00000800 */
#define TIM_DIER_CC3DE                      TIM_DIER_CC3DE_Msk                 /*!<Capture/Compare 3 DMA request enable */
#define TIM_DIER_CC4DE_Pos                  (12U)                              
#define TIM_DIER_CC4DE_Msk                  (0x1U << TIM_DIER_CC4DE_Pos)       /*!< 0x00001000 */
#define TIM_DIER_CC4DE                      TIM_DIER_CC4DE_Msk                 /*!<Capture/Compare 4 DMA request enable */
#define TIM_DIER_COMDE_Pos                  (13U)                              
#define TIM_DIER_COMDE_Msk                  (0x1U << TIM_DIER_COMDE_Pos)       /*!< 0x00002000 */
#define TIM_DIER_COMDE                      TIM_DIER_COMDE_Msk                 /*!<COM DMA request enable */
#define TIM_DIER_TDE_Pos                    (14U)                              
#define TIM_DIER_TDE_Msk                    (0x1U << TIM_DIER_TDE_Pos)         /*!< 0x00004000 */
#define TIM_DIER_TDE                        TIM_DIER_TDE_Msk                   /*!<Trigger DMA request enable */

/********************  Bit definition for TIM_SR register  *******************/
#define TIM_SR_UIF_Pos                      (0U)                               
#define TIM_SR_UIF_Msk                      (0x1U << TIM_SR_UIF_Pos)           /*!< 0x00000001 */
#define TIM_SR_UIF                          TIM_SR_UIF_Msk                     /*!<Update interrupt Flag */
#define TIM_SR_CC1IF_Pos                    (1U)                               
#define TIM_SR_CC1IF_Msk                    (0x1U << TIM_SR_CC1IF_Pos)         /*!< 0x00000002 */
#define TIM_SR_CC1IF                        TIM_SR_CC1IF_Msk                   /*!<Capture/Compare 1 interrupt Flag */
#define TIM_SR_CC2IF_Pos                    (2U)                               
#define TIM_SR_CC2IF_Msk                    (0x1U << TIM_SR_CC2IF_Pos)         /*!< 0x00000004 */
#define TIM_SR_CC2IF                        TIM_SR_CC2IF_Msk                   /*!<Capture/Compare 2 interrupt Flag */
#define TIM_SR_CC3IF_Pos                    (3U)                               
#define TIM_SR_CC3IF_Msk                    (0x1U << TIM_SR_CC3IF_Pos)         /*!< 0x00000008 */
#define TIM_SR_CC3IF                        TIM_SR_CC3IF_Msk                   /*!<Capture/Compare 3 interrupt Flag */
#define TIM_SR_CC4IF_Pos                    (4U)                               
#define TIM_SR_CC4IF_Msk                    (0x1U << TIM_SR_CC4IF_Pos)         /*!< 0x00000010 */
#define TIM_SR_CC4IF                        TIM_SR_CC4IF_Msk                   /*!<Capture/Compare 4 interrupt Flag */
#define TIM_SR_COMIF_Pos                    (5U)                               
#define TIM_SR_COMIF_Msk                    (0x1U << TIM_SR_COMIF_Pos)         /*!< 0x00000020 */
#define TIM_SR_COMIF                        TIM_SR_COMIF_Msk                   /*!<COM interrupt Flag */
#define TIM_SR_TIF_Pos                      (6U)                               
#define TIM_SR_TIF_Msk                      (0x1U << TIM_SR_TIF_Pos)           /*!< 0x00000040 */
#define TIM_SR_TIF                          TIM_SR_TIF_Msk                     /*!<Trigger interrupt Flag */
#define TIM_SR_BIF_Pos                      (7U)                               
#define TIM_SR_BIF_Msk                      (0x1U << TIM_SR_BIF_Pos)           /*!< 0x00000080 */
#define TIM_SR_BIF                          TIM_SR_BIF_Msk                     /*!<Break interrupt Flag */
#define TIM_SR_CC1OF_Pos                    (9U)                               
#define TIM_SR_CC1OF_Msk                    (0x1U << TIM_SR_CC1OF_Pos)         /*!< 0x00000200 */
#define TIM_SR_CC1OF                        TIM_SR_CC1OF_Msk                   /*!<Capture/Compare 1 Overcapture Flag */
#define TIM_SR_CC2OF_Pos                    (10U)                              
#define TIM_SR_CC2OF_Msk                    (0x1U << TIM_SR_CC2OF_Pos)         /*!< 0x00000400 */
#define TIM_SR_CC2OF                        TIM_SR_CC2OF_Msk                   /*!<Capture/Compare 2 Overcapture Flag */
#define TIM_SR_CC3OF_Pos                    (11U)                              
#define TIM_SR_CC3OF_Msk                    (0x1U << TIM_SR_CC3OF_Pos)         /*!< 0x00000800 */
#define TIM_SR_CC3OF                        TIM_SR_CC3OF_Msk                   /*!<Capture/Compare 3 Overcapture Flag */
#define TIM_SR_CC4OF_Pos                    (12U)                              
#define TIM_SR_CC4OF_Msk                    (0x1U << TIM_SR_CC4OF_Pos)         /*!< 0x00001000 */
#define TIM_SR_CC4OF                        TIM_SR_CC4OF_Msk                   /*!<Capture/Compare 4 Overcapture Flag */

/*******************  Bit definition for TIM_EGR register  *******************/
#define TIM_EGR_UG_Pos                      (0U)                               
#define TIM_EGR_UG_Msk                      (0x1U << TIM_EGR_UG_Pos)           /*!< 0x00000001 */
#define TIM_EGR_UG                          TIM_EGR_UG_Msk                     /*!<Update Generation */
#define TIM_EGR_CC1G_Pos                    (1U)                               
#define TIM_EGR_CC1G_Msk                    (0x1U << TIM_EGR_CC1G_Pos)         /*!< 0x00000002 */
#define TIM_EGR_CC1G                        TIM_EGR_CC1G_Msk                   /*!<Capture/Compare 1 Generation */
#define TIM_EGR_CC2G_Pos                    (2U)                               
#define TIM_EGR_CC2G_Msk                    (0x1U << TIM_EGR_CC2G_Pos)         /*!< 0x00000004 */
#define TIM_EGR_CC2G                        TIM_EGR_CC2G_Msk                   /*!<Capture/Compare 2 Generation */
#define TIM_EGR_CC3G_Pos                    (3U)                               
#define TIM_EGR_CC3G_Msk                    (0x1U << TIM_EGR_CC3G_Pos)         /*!< 0x00000008 */
#define TIM_EGR_CC3G                        TIM_EGR_CC3G_Msk                   /*!<Capture/Compare 3 Generation */
#define TIM_EGR_CC4G_Pos                    (4U)                               
#define TIM_EGR_CC4G_Msk                    (0x1U << TIM_EGR_CC4G_Pos)         /*!< 0x00000010 */
#define TIM_EGR_CC4G                        TIM_EGR_CC4G_Msk                   /*!<Capture/Compare 4 Generation */
#define TIM_EGR_COMG_Pos                    (5U)                               
#define TIM_EGR_COMG_Msk                    (0x1U << TIM_EGR_COMG_Pos)         /*!< 0x00000020 */
#define TIM_EGR_COMG                        TIM_EGR_COMG_Msk                   /*!<Capture/Compare Control Update Generation */
#define TIM_EGR_TG_Pos                      (6U)                               
#define TIM_EGR_TG_Msk                      (0x1U << TIM_EGR_TG_Pos)           /*!< 0x00000040 */
#define TIM_EGR_TG                          TIM_EGR_TG_Msk                     /*!<Trigger Generation */
#define TIM_EGR_BG_Pos                      (7U)                               
#define TIM_EGR_BG_Msk                      (0x1U << TIM_EGR_BG_Pos)           /*!< 0x00000080 */
#define TIM_EGR_BG                          TIM_EGR_BG_Msk                     /*!<Break Generation */

/******************  Bit definition for TIM_CCMR1 register  ******************/
#define TIM_CCMR1_CC1S_Pos                  (0U)                               
#define TIM_CCMR1_CC1S_Msk                  (0x3U << TIM_CCMR1_CC1S_Pos)       /*!< 0x00000003 */
#define TIM_CCMR1_CC1S                      TIM_CCMR1_CC1S_Msk                 /*!<CC1S[1:0] bits (Capture/Compare 1 Selection) */
#define TIM_CCMR1_CC1S_0                    (0x1U << TIM_CCMR1_CC1S_Pos)       /*!< 0x00000001 */
#define TIM_CCMR1_CC1S_1                    (0x2U << TIM_CCMR1_CC1S_Pos)       /*!< 0x00000002 */

#define TIM_CCMR1_OC1FE_Pos                 (2U)                               
#define TIM_CCMR1_OC1FE_Msk                 (0x1U << TIM_CCMR1_OC1FE_Pos)      /*!< 0x00000004 */
#define TIM_CCMR1_OC1FE                     TIM_CCMR1_OC1FE_Msk                /*!<Output Compare 1 Fast enable */
#define TIM_CCMR1_OC1PE_Pos                 (3U)                               
#define TIM_CCMR1_OC1PE_Msk                 (0x1U << TIM_CCMR1_OC1PE_Pos)      /*!< 0x00000008 */
#define TIM_CCMR1_OC1PE                     TIM_CCMR1_OC1PE_Msk                /*!<Output Compare 1 Preload enable */

#define TIM_CCMR1_OC1M_Pos                  (4U)                               
#define TIM_CCMR1_OC1M_Msk                  (0x7U << TIM_CCMR1_OC1M_Pos)       /*!< 0x00000070 */
#define TIM_CCMR1_OC1M                      TIM_CCMR1_OC1M_Msk                 /*!<OC1M[2:0] bits (Output Compare 1 Mode) */
#define TIM_CCMR1_OC1M_0                    (0x1U << TIM_CCMR1_OC1M_Pos)       /*!< 0x00000010 */
#define TIM_CCMR1_OC1M_1                    (0x2U << TIM_CCMR1_OC1M_Pos)       /*!< 0x00000020 */
#define TIM_CCMR1_OC1M_2                    (0x4U << TIM_CCMR1_OC1M_Pos)       /*!< 0x00000040 */

#define TIM_CCMR1_OC1CE_Pos                 (7U)                               
#define TIM_CCMR1_OC1CE_Msk                 (0x1U << TIM_CCMR1_OC1CE_Pos)      /*!< 0x00000080 */
#define TIM_CCMR1_OC1CE                     TIM_CCMR1_OC1CE_Msk                /*!<Output Compare 1Clear Enable */

#define TIM_CCMR1_CC2S_Pos                  (8U)                               
#define TIM_CCMR1_CC2S_Msk                  (0x3U << TIM_CCMR1_CC2S_Pos)       /*!< 0x00000300 */
#define TIM_CCMR1_CC2S                      TIM_CCMR1_CC2S_Msk                 /*!<CC2S[1:0] bits (Capture/Compare 2 Selection) */
#define TIM_CCMR1_CC2S_0                    (0x1U << TIM_CCMR1_CC2S_Pos)       /*!< 0x00000100 */
#define TIM_CCMR1_CC2S_1                    (0x2U << TIM_CCMR1_CC2S_Pos)       /*!< 0x00000200 */

#define TIM_CCMR1_OC2FE_Pos                 (10U)                              
#define TIM_CCMR1_OC2FE_Msk                 (0x1U << TIM_CCMR1_OC2FE_Pos)      /*!< 0x00000400 */
#define TIM_CCMR1_OC2FE                     TIM_CCMR1_OC2FE_Msk                /*!<Output Compare 2 Fast enable */
#define TIM_CCMR1_OC2PE_Pos                 (11U)                              
#define TIM_CCMR1_OC2PE_Msk                 (0x1U << TIM_CCMR1_OC2PE_Pos)      /*!< 0x00000800 */
#define TIM_CCMR1_OC2PE                     TIM_CCMR1_OC2PE_Msk                /*!<Output Compare 2 Preload enable */

#define TIM_CCMR1_OC2M_Pos                  (12U)                              
#define TIM_CCMR1_OC2M_Msk                  (0x7U << TIM_CCMR1_OC2M_Pos)       /*!< 0x00007000 */
#define TIM_CCMR1_OC2M                      TIM_CCMR1_OC2M_Msk                 /*!<OC2M[2:0] bits (Output Compare 2 Mode) */
#define TIM_CCMR1_OC2M_0                    (0x1U << TIM_CCMR1_OC2M_Pos)       /*!< 0x00001000 */
#define TIM_CCMR1_OC2M_1                    (0x2U << TIM_CCMR1_OC2M_Pos)       /*!< 0x00002000 */
#define TIM_CCMR1_OC2M_2                    (0x4U << TIM_CCMR1_OC2M_Pos)       /*!< 0x00004000 */

#define TIM_CCMR1_OC2CE_Pos                 (15U)                              
#define TIM_CCMR1_OC2CE_Msk                 (0x1U << TIM_CCMR1_OC2CE_Pos)      /*!< 0x00008000 */
#define TIM_CCMR1_OC2CE                     TIM_CCMR1_OC2CE_Msk                /*!<Output Compare 2 Clear Enable */

/*---------------------------------------------------------------------------*/

#define TIM_CCMR1_IC1PSC_Pos                (2U)                               
#define TIM_CCMR1_IC1PSC_Msk                (0x3U << TIM_CCMR1_IC1PSC_Pos)     /*!< 0x0000000C */
#define TIM_CCMR1_IC1PSC                    TIM_CCMR1_IC1PSC_Msk               /*!<IC1PSC[1:0] bits (Input Capture 1 Prescaler) */
#define TIM_CCMR1_IC1PSC_0                  (0x1U << TIM_CCMR1_IC1PSC_Pos)     /*!< 0x00000004 */
#define TIM_CCMR1_IC1PSC_1                  (0x2U << TIM_CCMR1_IC1PSC_Pos)     /*!< 0x00000008 */

#define TIM_CCMR1_IC1F_Pos                  (4U)                               
#define TIM_CCMR1_IC1F_Msk                  (0xFU << TIM_CCMR1_IC1F_Pos)       /*!< 0x000000F0 */
#define TIM_CCMR1_IC1F                      TIM_CCMR1_IC1F_Msk                 /*!<IC1F[3:0] bits (Input Capture 1 Filter) */
#define TIM_CCMR1_IC1F_0                    (0x1U << TIM_CCMR1_IC1F_Pos)       /*!< 0x00000010 */
#define TIM_CCMR1_IC1F_1                    (0x2U << TIM_CCMR1_IC1F_Pos)       /*!< 0x00000020 */
#define TIM_CCMR1_IC1F_2                    (0x4U << TIM_CCMR1_IC1F_Pos)       /*!< 0x00000040 */
#define TIM_CCMR1_IC1F_3                    (0x8U << TIM_CCMR1_IC1F_Pos)       /*!< 0x00000080 */

#define TIM_CCMR1_IC2PSC_Pos                (10U)                              
#define TIM_CCMR1_IC2PSC_Msk                (0x3U << TIM_CCMR1_IC2PSC_Pos)     /*!< 0x00000C00 */
#define TIM_CCMR1_IC2PSC                    TIM_CCMR1_IC2PSC_Msk               /*!<IC2PSC[1:0] bits (Input Capture 2 Prescaler) */
#define TIM_CCMR1_IC2PSC_0                  (0x1U << TIM_CCMR1_IC2PSC_Pos)     /*!< 0x00000400 */
#define TIM_CCMR1_IC2PSC_1                  (0x2U << TIM_CCMR1_IC2PSC_Pos)     /*!< 0x00000800 */

#define TIM_CCMR1_IC2F_Pos                  (12U)                              
#define TIM_CCMR1_IC2F_Msk                  (0xFU << TIM_CCMR1_IC2F_Pos)       /*!< 0x0000F000 */
#define TIM_CCMR1_IC2F                      TIM_CCMR1_IC2F_Msk                 /*!<IC2F[3:0] bits (Input Capture 2 Filter) */
#define TIM_CCMR1_IC2F_0                    (0x1U << TIM_CCMR1_IC2F_Pos)       /*!< 0x00001000 */
#define TIM_CCMR1_IC2F_1                    (0x2U << TIM_CCMR1_IC2F_Pos)       /*!< 0x00002000 */
#define TIM_CCMR1_IC2F_2                    (0x4U << TIM_CCMR1_IC2F_Pos)       /*!< 0x00004000 */
#define TIM_CCMR1_IC2F_3                    (0x8U << TIM_CCMR1_IC2F_Pos)       /*!< 0x00008000 */

/******************  Bit definition for TIM_CCMR2 register  ******************/
#define TIM_CCMR2_CC3S_Pos                  (0U)                               
#define TIM_CCMR2_CC3S_Msk                  (0x3U << TIM_CCMR2_CC3S_Pos)       /*!< 0x00000003 */
#define TIM_CCMR2_CC3S                      TIM_CCMR2_CC3S_Msk                 /*!<CC3S[1:0] bits (Capture/Compare 3 Selection) */
#define TIM_CCMR2_CC3S_0                    (0x1U << TIM_CCMR2_CC3S_Pos)       /*!< 0x00000001 */
#define TIM_CCMR2_CC3S_1                    (0x2U << TIM_CCMR2_CC3S_Pos)       /*!< 0x00000002 */

#define TIM_CCMR2_OC3FE_Pos                 (2U)                               
#define TIM_CCMR2_OC3FE_Msk                 (0x1U << TIM_CCMR2_OC3FE_Pos)      /*!< 0x00000004 */
#define TIM_CCMR2_OC3FE                     TIM_CCMR2_OC3FE_Msk                /*!<Output Compare 3 Fast enable */
#define TIM_CCMR2_OC3PE_Pos                 (3U)                               
#define TIM_CCMR2_OC3PE_Msk                 (0x1U << TIM_CCMR2_OC3PE_Pos)      /*!< 0x00000008 */
#define TIM_CCMR2_OC3PE                     TIM_CCMR2_OC3PE_Msk                /*!<Output Compare 3 Preload enable */

#define TIM_CCMR2_OC3M_Pos                  (4U)                               
#define TIM_CCMR2_OC3M_Msk                  (0x7U << TIM_CCMR2_OC3M_Pos)       /*!< 0x00000070 */
#define TIM_CCMR2_OC3M                      TIM_CCMR2_OC3M_Msk                 /*!<OC3M[2:0] bits (Output Compare 3 Mode) */
#define TIM_CCMR2_OC3M_0                    (0x1U << TIM_CCMR2_OC3M_Pos)       /*!< 0x00000010 */
#define TIM_CCMR2_OC3M_1                    (0x2U << TIM_CCMR2_OC3M_Pos)       /*!< 0x00000020 */
#define TIM_CCMR2_OC3M_2                    (0x4U << TIM_CCMR2_OC3M_Pos)       /*!< 0x00000040 */

#define TIM_CCMR2_OC3CE_Pos                 (7U)                               
#define TIM_CCMR2_OC3CE_Msk                 (0x1U << TIM_CCMR2_OC3CE_Pos)      /*!< 0x00000080 */
#define TIM_CCMR2_OC3CE                     TIM_CCMR2_OC3CE_Msk                /*!<Output Compare 3 Clear Enable */

#define TIM_CCMR2_CC4S_Pos                  (8U)                               
#define TIM_CCMR2_CC4S_Msk                  (0x3U << TIM_CCMR2_CC4S_Pos)       /*!< 0x00000300 */
#define TIM_CCMR2_CC4S                      TIM_CCMR2_CC4S_Msk                 /*!<CC4S[1:0] bits (Capture/Compare 4 Selection) */
#define TIM_CCMR2_CC4S_0                    (0x1U << TIM_CCMR2_CC4S_Pos)       /*!< 0x00000100 */
#define TIM_CCMR2_CC4S_1                    (0x2U << TIM_CCMR2_CC4S_Pos)       /*!< 0x00000200 */

#define TIM_CCMR2_OC4FE_Pos                 (10U)                              
#define TIM_CCMR2_OC4FE_Msk                 (0x1U << TIM_CCMR2_OC4FE_Pos)      /*!< 0x00000400 */
#define TIM_CCMR2_OC4FE                     TIM_CCMR2_OC4FE_Msk                /*!<Output Compare 4 Fast enable */
#define TIM_CCMR2_OC4PE_Pos                 (11U)                              
#define TIM_CCMR2_OC4PE_Msk                 (0x1U << TIM_CCMR2_OC4PE_Pos)      /*!< 0x00000800 */
#define TIM_CCMR2_OC4PE                     TIM_CCMR2_OC4PE_Msk                /*!<Output Compare 4 Preload enable */

#define TIM_CCMR2_OC4M_Pos                  (12U)                              
#define TIM_CCMR2_OC4M_Msk                  (0x7U << TIM_CCMR2_OC4M_Pos)       /*!< 0x00007000 */
#define TIM_CCMR2_OC4M                      TIM_CCMR2_OC4M_Msk                 /*!<OC4M[2:0] bits (Output Compare 4 Mode) */
#define TIM_CCMR2_OC4M_0                    (0x1U << TIM_CCMR2_OC4M_Pos)       /*!< 0x00001000 */
#define TIM_CCMR2_OC4M_1                    (0x2U << TIM_CCMR2_OC4M_Pos)       /*!< 0x00002000 */
#define TIM_CCMR2_OC4M_2                    (0x4U << TIM_CCMR2_OC4M_Pos)       /*!< 0x00004000 */

#define TIM_CCMR2_OC4CE_Pos                 (15U)                              
#define TIM_CCMR2_OC4CE_Msk                 (0x1U << TIM_CCMR2_OC4CE_Pos)      /*!< 0x00008000 */
#define TIM_CCMR2_OC4CE                     TIM_CCMR2_OC4CE_Msk                /*!<Output Compare 4 Clear Enable */

/*---------------------------------------------------------------------------*/

#define TIM_CCMR2_IC3PSC_Pos                (2U)                               
#define TIM_CCMR2_IC3PSC_Msk                (0x3U << TIM_CCMR2_IC3PSC_Pos)     /*!< 0x0000000C */
#define TIM_CCMR2_IC3PSC                    TIM_CCMR2_IC3PSC_Msk               /*!<IC3PSC[1:0] bits (Input Capture 3 Prescaler) */
#define TIM_CCMR2_IC3PSC_0                  (0x1U << TIM_CCMR2_IC3PSC_Pos)     /*!< 0x00000004 */
#define TIM_CCMR2_IC3PSC_1                  (0x2U << TIM_CCMR2_IC3PSC_Pos)     /*!< 0x00000008 */

#define TIM_CCMR2_IC3F_Pos                  (4U)                               
#define TIM_CCMR2_IC3F_Msk                  (0xFU << TIM_CCMR2_IC3F_Pos)       /*!< 0x000000F0 */
#define TIM_CCMR2_IC3F                      TIM_CCMR2_IC3F_Msk                 /*!<IC3F[3:0] bits (Input Capture 3 Filter) */
#define TIM_CCMR2_IC3F_0                    (0x1U << TIM_CCMR2_IC3F_Pos)       /*!< 0x00000010 */
#define TIM_CCMR2_IC3F_1                    (0x2U << TIM_CCMR2_IC3F_Pos)       /*!< 0x00000020 */
#define TIM_CCMR2_IC3F_2                    (0x4U << TIM_CCMR2_IC3F_Pos)       /*!< 0x00000040 */
#define TIM_CCMR2_IC3F_3                    (0x8U << TIM_CCMR2_IC3F_Pos)       /*!< 0x00000080 */

#define TIM_CCMR2_IC4PSC_Pos                (10U)                              
#define TIM_CCMR2_IC4PSC_Msk                (0x3U << TIM_CCMR2_IC4PSC_Pos)     /*!< 0x00000C00 */
#define TIM_CCMR2_IC4PSC                    TIM_CCMR2_IC4PSC_Msk               /*!<IC4PSC[1:0] bits (Input Capture 4 Prescaler) */
#define TIM_CCMR2_IC4PSC_0                  (0x1U << TIM_CCMR2_IC4PSC_Pos)     /*!< 0x00000400 */
#define TIM_CCMR2_IC4PSC_1                  (0x2U << TIM_CCMR2_IC4PSC_Pos)     /*!< 0x00000800 */

#define TIM_CCMR2_IC4F_Pos                  (12U)                              
#define TIM_CCMR2_IC4F_Msk                  (0xFU << TIM_CCMR2_IC4F_Pos)       /*!< 0x0000F000 */
#define TIM_CCMR2_IC4F                      TIM_CCMR2_IC4F_Msk                 /*!<IC4F[3:0] bits (Input Capture 4 Filter) */
#define TIM_CCMR2_IC4F_0                    (0x1U << TIM_CCMR2_IC4F_Pos)       /*!< 0x00001000 */
#define TIM_CCMR2_IC4F_1                    (0x2U << TIM_CCMR2_IC4F_Pos)       /*!< 0x00002000 */
#define TIM_CCMR2_IC4F_2                    (0x4U << TIM_CCMR2_IC4F_Pos)       /*!< 0x00004000 */
#define TIM_CCMR2_IC4F_3                    (0x8U << TIM_CCMR2_IC4F_Pos)       /*!< 0x00008000 */

/*******************  Bit definition for TIM_CCER register  ******************/
#define TIM_CCER_CC1E_Pos                   (0U)                               
#define TIM_CCER_CC1E_Msk                   (0x1U << TIM_CCER_CC1E_Pos)        /*!< 0x00000001 */
#define TIM_CCER_CC1E                       TIM_CCER_CC1E_Msk                  /*!<Capture/Compare 1 output enable */
#define TIM_CCER_CC1P_Pos                   (1U)                               
#define TIM_CCER_CC1P_Msk                   (0x1U << TIM_CCER_CC1P_Pos)        /*!< 0x00000002 */
#define TIM_CCER_CC1P                       TIM_CCER_CC1P_Msk                  /*!<Capture/Compare 1 output Polarity */
#define TIM_CCER_CC1NE_Pos                  (2U)                               
#define TIM_CCER_CC1NE_Msk                  (0x1U << TIM_CCER_CC1NE_Pos)       /*!< 0x00000004 */
#define TIM_CCER_CC1NE                      TIM_CCER_CC1NE_Msk                 /*!<Capture/Compare 1 Complementary output enable */
#define TIM_CCER_CC1NP_Pos                  (3U)                               
#define TIM_CCER_CC1NP_Msk                  (0x1U << TIM_CCER_CC1NP_Pos)       /*!< 0x00000008 */
#define TIM_CCER_CC1NP                      TIM_CCER_CC1NP_Msk                 /*!<Capture/Compare 1 Complementary output Polarity */
#define TIM_CCER_CC2E_Pos                   (4U)                               
#define TIM_CCER_CC2E_Msk                   (0x1U << TIM_CCER_CC2E_Pos)        /*!< 0x00000010 */
#define TIM_CCER_CC2E                       TIM_CCER_CC2E_Msk                  /*!<Capture/Compare 2 output enable */
#define TIM_CCER_CC2P_Pos                   (5U)                               
#define TIM_CCER_CC2P_Msk                   (0x1U << TIM_CCER_CC2P_Pos)        /*!< 0x00000020 */
#define TIM_CCER_CC2P                       TIM_CCER_CC2P_Msk                  /*!<Capture/Compare 2 output Polarity */
#define TIM_CCER_CC2NE_Pos                  (6U)                               
#define TIM_CCER_CC2NE_Msk                  (0x1U << TIM_CCER_CC2NE_Pos)       /*!< 0x00000040 */
#define TIM_CCER_CC2NE                      TIM_CCER_CC2NE_Msk                 /*!<Capture/Compare 2 Complementary output enable */
#define TIM_CCER_CC2NP_Pos                  (7U)                               
#define TIM_CCER_CC2NP_Msk                  (0x1U << TIM_CCER_CC2NP_Pos)       /*!< 0x00000080 */
#define TIM_CCER_CC2NP                      TIM_CCER_CC2NP_Msk                 /*!<Capture/Compare 2 Complementary output Polarity */
#define TIM_CCER_CC3E_Pos                   (8U)                               
#define TIM_CCER_CC3E_Msk                   (0x1U << TIM_CCER_CC3E_Pos)        /*!< 0x00000100 */
#define TIM_CCER_CC3E                       TIM_CCER_CC3E_Msk                  /*!<Capture/Compare 3 output enable */
#define TIM_CCER_CC3P_Pos                   (9U)                               
#define TIM_CCER_CC3P_Msk                   (0x1U << TIM_CCER_CC3P_Pos)        /*!< 0x00000200 */
#define TIM_CCER_CC3P                       TIM_CCER_CC3P_Msk                  /*!<Capture/Compare 3 output Polarity */
#define TIM_CCER_CC3NE_Pos                  (10U)                              
#define TIM_CCER_CC3NE_Msk                  (0x1U << TIM_CCER_CC3NE_Pos)       /*!< 0x00000400 */
#define TIM_CCER_CC3NE                      TIM_CCER_CC3NE_Msk                 /*!<Capture/Compare 3 Complementary output enable */
#define TIM_CCER_CC3NP_Pos                  (11U)                              
#define TIM_CCER_CC3NP_Msk                  (0x1U << TIM_CCER_CC3NP_Pos)       /*!< 0x00000800 */
#define TIM_CCER_CC3NP                      TIM_CCER_CC3NP_Msk                 /*!<Capture/Compare 3 Complementary output Polarity */
#define TIM_CCER_CC4E_Pos                   (12U)                              
#define TIM_CCER_CC4E_Msk                   (0x1U << TIM_CCER_CC4E_Pos)        /*!< 0x00001000 */
#define TIM_CCER_CC4E                       TIM_CCER_CC4E_Msk                  /*!<Capture/Compare 4 output enable */
#define TIM_CCER_CC4P_Pos                   (13U)                              
#define TIM_CCER_CC4P_Msk                   (0x1U << TIM_CCER_CC4P_Pos)        /*!< 0x00002000 */
#define TIM_CCER_CC4P                       TIM_CCER_CC4P_Msk                  /*!<Capture/Compare 4 output Polarity */

/*******************  Bit definition for TIM_CNT register  *******************/
#define TIM_CNT_CNT_Pos                     (0U)                               
#define TIM_CNT_CNT_Msk                     (0xFFFFFFFFU << TIM_CNT_CNT_Pos)   /*!< 0xFFFFFFFF */
#define TIM_CNT_CNT                         TIM_CNT_CNT_Msk                    /*!<Counter Value */

/*******************  Bit definition for TIM_PSC register  *******************/
#define TIM_PSC_PSC_Pos                     (0U)                               
#define TIM_PSC_PSC_Msk                     (0xFFFFU << TIM_PSC_PSC_Pos)       /*!< 0x0000FFFF */
#define TIM_PSC_PSC                         TIM_PSC_PSC_Msk                    /*!<Prescaler Value */

/*******************  Bit definition for TIM_ARR register  *******************/
#define TIM_ARR_ARR_Pos                     (0U)                               
#define TIM_ARR_ARR_Msk                     (0xFFFFFFFFU << TIM_ARR_ARR_Pos)   /*!< 0xFFFFFFFF */
#define TIM_ARR_ARR                         TIM_ARR_ARR_Msk                    /*!<actual auto-reload Value */

/*******************  Bit definition for TIM_RCR register  *******************/
#define TIM_RCR_REP_Pos                     (0U)                               
#define TIM_RCR_REP_Msk                     (0xFFU << TIM_RCR_REP_Pos)         /*!< 0x000000FF */
#define TIM_RCR_REP                         TIM_RCR_REP_Msk                    /*!<Repetition Counter Value */

/*******************  Bit definition for TIM_CCR1 register  ******************/
#define TIM_CCR1_CCR1_Pos                   (0U)                               
#define TIM_CCR1_CCR1_Msk                   (0xFFFFU << TIM_CCR1_CCR1_Pos)     /*!< 0x0000FFFF */
#define TIM_CCR1_CCR1                       TIM_CCR1_CCR1_Msk                  /*!<Capture/Compare 1 Value */

/*******************  Bit definition for TIM_CCR2 register  ******************/
#define TIM_CCR2_CCR2_Pos                   (0U)                               
#define TIM_CCR2_CCR2_Msk                   (0xFFFFU << TIM_CCR2_CCR2_Pos)     /*!< 0x0000FFFF */
#define TIM_CCR2_CCR2                       TIM_CCR2_CCR2_Msk                  /*!<Capture/Compare 2 Value */

/*******************  Bit definition for TIM_CCR3 register  ******************/
#define TIM_CCR3_CCR3_Pos                   (0U)                               
#define TIM_CCR3_CCR3_Msk                   (0xFFFFU << TIM_CCR3_CCR3_Pos)     /*!< 0x0000FFFF */
#define TIM_CCR3_CCR3                       TIM_CCR3_CCR3_Msk                  /*!<Capture/Compare 3 Value */

/*******************  Bit definition for TIM_CCR4 register  ******************/
#define TIM_CCR4_CCR4_Pos                   (0U)                               
#define TIM_CCR4_CCR4_Msk                   (0xFFFFU << TIM_CCR4_CCR4_Pos)     /*!< 0x0000FFFF */
#define TIM_CCR4_CCR4                       TIM_CCR4_CCR4_Msk                  /*!<Capture/Compare 4 Value */

/*******************  Bit definition for TIM_BDTR register  ******************/
#define TIM_BDTR_DTG_Pos                    (0U)                               
#define TIM_BDTR_DTG_Msk                    (0xFFU << TIM_BDTR_DTG_Pos)        /*!< 0x000000FF */
#define TIM_BDTR_DTG                        TIM_BDTR_DTG_Msk                   /*!<DTG[0:7] bits (Dead-Time Generator set-up) */
#define TIM_BDTR_DTG_0                      (0x01U << TIM_BDTR_DTG_Pos)        /*!< 0x00000001 */
#define TIM_BDTR_DTG_1                      (0x02U << TIM_BDTR_DTG_Pos)        /*!< 0x00000002 */
#define TIM_BDTR_DTG_2                      (0x04U << TIM_BDTR_DTG_Pos)        /*!< 0x00000004 */
#define TIM_BDTR_DTG_3                      (0x08U << TIM_BDTR_DTG_Pos)        /*!< 0x00000008 */
#define TIM_BDTR_DTG_4                      (0x10U << TIM_BDTR_DTG_Pos)        /*!< 0x00000010 */
#define TIM_BDTR_DTG_5                      (0x20U << TIM_BDTR_DTG_Pos)        /*!< 0x00000020 */
#define TIM_BDTR_DTG_6                      (0x40U << TIM_BDTR_DTG_Pos)        /*!< 0x00000040 */
#define TIM_BDTR_DTG_7                      (0x80U << TIM_BDTR_DTG_Pos)        /*!< 0x00000080 */

#define TIM_BDTR_LOCK_Pos                   (8U)                               
#define TIM_BDTR_LOCK_Msk                   (0x3U << TIM_BDTR_LOCK_Pos)        /*!< 0x00000300 */
#define TIM_BDTR_LOCK                       TIM_BDTR_LOCK_Msk                  /*!<LOCK[1:0] bits (Lock Configuration) */
#define TIM_BDTR_LOCK_0                     (0x1U << TIM_BDTR_LOCK_Pos)        /*!< 0x00000100 */
#define TIM_BDTR_LOCK_1                     (0x2U << TIM_BDTR_LOCK_Pos)        /*!< 0x00000200 */

#define TIM_BDTR_OSSI_Pos                   (10U)                              
#define TIM_BDTR_OSSI_Msk                   (0x1U << TIM_BDTR_OSSI_Pos)        /*!< 0x00000400 */
#define TIM_BDTR_OSSI                       TIM_BDTR_OSSI_Msk                  /*!<Off-State Selection for Idle mode */
#define TIM_BDTR_OSSR_Pos                   (11U)                              
#define TIM_BDTR_OSSR_Msk                   (0x1U << TIM_BDTR_OSSR_Pos)        /*!< 0x00000800 */
#define TIM_BDTR_OSSR                       TIM_BDTR_OSSR_Msk                  /*!<Off-State Selection for Run mode */
#define TIM_BDTR_BKE_Pos                    (12U)                              
#define TIM_BDTR_BKE_Msk                    (0x1U << TIM_BDTR_BKE_Pos)         /*!< 0x00001000 */
#define TIM_BDTR_BKE                        TIM_BDTR_BKE_Msk                   /*!<Break enable */
#define TIM_BDTR_BKP_Pos                    (13U)                              
#define TIM_BDTR_BKP_Msk                    (0x1U << TIM_BDTR_BKP_Pos)         /*!< 0x00002000 */
#define TIM_BDTR_BKP                        TIM_BDTR_BKP_Msk                   /*!<Break Polarity */
#define TIM_BDTR_AOE_Pos                    (14U)                              
#define TIM_BDTR_AOE_Msk                    (0x1U << TIM_BDTR_AOE_Pos)         /*!< 0x00004000 */
#define TIM_BDTR_AOE                        TIM_BDTR_AOE_Msk                   /*!<Automatic Output enable */
#define TIM_BDTR_MOE_Pos                    (15U)                              
#define TIM_BDTR_MOE_Msk                    (0x1U << TIM_BDTR_MOE_Pos)         /*!< 0x00008000 */
#define TIM_BDTR_MOE                        TIM_BDTR_MOE_Msk                   /*!<Main Output enable */

/*******************  Bit definition for TIM_DCR register  *******************/
#define TIM_DCR_DBA_Pos                     (0U)                               
#define TIM_DCR_DBA_Msk                     (0x1FU << TIM_DCR_DBA_Pos)         /*!< 0x0000001F */
#define TIM_DCR_DBA                         TIM_DCR_DBA_Msk                    /*!<DBA[4:0] bits (DMA Base Address) */
#define TIM_DCR_DBA_0                       (0x01U << TIM_DCR_DBA_Pos)         /*!< 0x00000001 */
#define TIM_DCR_DBA_1                       (0x02U << TIM_DCR_DBA_Pos)         /*!< 0x00000002 */
#define TIM_DCR_DBA_2                       (0x04U << TIM_DCR_DBA_Pos)         /*!< 0x00000004 */
#define TIM_DCR_DBA_3                       (0x08U << TIM_DCR_DBA_Pos)         /*!< 0x00000008 */
#define TIM_DCR_DBA_4                       (0x10U << TIM_DCR_DBA_Pos)         /*!< 0x00000010 */

#define TIM_DCR_DBL_Pos                     (8U)                               
#define TIM_DCR_DBL_Msk                     (0x1FU << TIM_DCR_DBL_Pos)         /*!< 0x00001F00 */
#define TIM_DCR_DBL                         TIM_DCR_DBL_Msk                    /*!<DBL[4:0] bits (DMA Burst Length) */
#define TIM_DCR_DBL_0                       (0x01U << TIM_DCR_DBL_Pos)         /*!< 0x00000100 */
#define TIM_DCR_DBL_1                       (0x02U << TIM_DCR_DBL_Pos)         /*!< 0x00000200 */
#define TIM_DCR_DBL_2                       (0x04U << TIM_DCR_DBL_Pos)         /*!< 0x00000400 */
#define TIM_DCR_DBL_3                       (0x08U << TIM_DCR_DBL_Pos)         /*!< 0x00000800 */
#define TIM_DCR_DBL_4                       (0x10U << TIM_DCR_DBL_Pos)         /*!< 0x00001000 */

/*******************  Bit definition for TIM_DMAR register  ******************/
#define TIM_DMAR_DMAB_Pos                   (0U)                               
#define TIM_DMAR_DMAB_Msk                   (0xFFFFU << TIM_DMAR_DMAB_Pos)     /*!< 0x0000FFFF */
#define TIM_DMAR_DMAB                       TIM_DMAR_DMAB_Msk                  /*!<DMA register for burst accesses */

/******************************************************************************/
/*                                                                            */
/*                             Real-Time Clock                                */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for RTC_INTEN register  ********************/
#define RTC_INTEN_SCIE_Pos                   (0U)                               
#define RTC_INTEN_SCIE_Msk                   (0x1U << RTC_INTEN_SCIE_Pos)        /*!< 0x00000001 */
#define RTC_INTEN_SCIE                       RTC_INTEN_SCIE_Msk                  /*!< Second Interrupt Enable */
#define RTC_INTEN_ALRMIE_Pos                   (1U)                               
#define RTC_INTEN_ALRMIE_Msk                   (0x1U << RTC_INTEN_ALRMIE_Pos)        /*!< 0x00000002 */
#define RTC_INTEN_ALRMIE                       RTC_INTEN_ALRMIE_Msk                  /*!< Alarm Interrupt Enable */
#define RTC_INTEN_OVIE_Pos                    (2U)                               
#define RTC_INTEN_OVIE_Msk                    (0x1U << RTC_INTEN_OVIE_Pos)         /*!< 0x00000004 */
#define RTC_INTEN_OVIE                        RTC_INTEN_OVIE_Msk                   /*!< OverfloW Interrupt Enable */

/*******************  Bit definition for RTC_CTL register  ********************/
#define RTC_CTL_SCIF_Pos                    (0U)                               
#define RTC_CTL_SCIF_Msk                    (0x1U << RTC_CTL_SCIF_Pos)         /*!< 0x00000001 */
#define RTC_CTL_SCIF                        RTC_CTL_SCIF_Msk                   /*!< Second Flag */
#define RTC_CTL_ALRMIF_Pos                    (1U)                               
#define RTC_CTL_ALRMIF_Msk                    (0x1U << RTC_CTL_ALRMIF_Pos)         /*!< 0x00000002 */
#define RTC_CTL_ALRMIF                        RTC_CTL_ALRMIF_Msk                   /*!< Alarm Flag */
#define RTC_CTL_OVIF_Pos                     (2U)                               
#define RTC_CTL_OVIF_Msk                     (0x1U << RTC_CTL_OVIF_Pos)          /*!< 0x00000004 */
#define RTC_CTL_OVIF                         RTC_CTL_OVIF_Msk                    /*!< OverfloW Flag */
#define RTC_CTL_RSYNF_Pos                     (3U)                               
#define RTC_CTL_RSYNF_Msk                     (0x1U << RTC_CTL_RSYNF_Pos)          /*!< 0x00000008 */
#define RTC_CTL_RSYNF                         RTC_CTL_RSYNF_Msk                    /*!< Registers Synchronized Flag */
#define RTC_CTL_CMF_Pos                     (4U)                               
#define RTC_CTL_CMF_Msk                     (0x1U << RTC_CTL_CMF_Pos)          /*!< 0x00000010 */
#define RTC_CTL_CMF                         RTC_CTL_CMF_Msk                    /*!< Configuration Flag */
#define RTC_CTL_LWOFF_Pos                   (5U)                               
#define RTC_CTL_LWOFF_Msk                   (0x1U << RTC_CTL_LWOFF_Pos)        /*!< 0x00000020 */
#define RTC_CTL_LWOFF                       RTC_CTL_LWOFF_Msk                  /*!< RTC operation OFF */

/*******************  Bit definition for RTC_PSCH register  *******************/
#define RTC_PSCH_PSC_Pos                    (0U)                               
#define RTC_PSCH_PSC_Msk                    (0xFU << RTC_PSCH_PSC_Pos)         /*!< 0x0000000F */
#define RTC_PSCH_PSC                        RTC_PSCH_PSC_Msk                   /*!< RTC Prescaler Reload Value High */

/*******************  Bit definition for RTC_PSCL register  *******************/
#define RTC_PSCL_PSC_Pos                    (0U)                               
#define RTC_PSCL_PSC_Msk                    (0xFFFFU << RTC_PSCL_PSC_Pos)      /*!< 0x0000FFFF */
#define RTC_PSCL_PSC                        RTC_PSCL_PSC_Msk                   /*!< RTC Prescaler Reload Value Low */

/*******************  Bit definition for RTC_DIVH register  *******************/
#define RTC_DIVH_DIV_Pos                (0U)                               
#define RTC_DIVH_DIV_Msk                (0xFU << RTC_DIVH_DIV_Pos)     /*!< 0x0000000F */
#define RTC_DIVH_RTC_DIV                    RTC_DIVH_DIV_Msk               /*!< RTC Clock Divider High */

/*******************  Bit definition for RTC_DIVL register  *******************/
#define RTC_DIVL_DIV_Pos                (0U)                               
#define RTC_DIVL_DIV_Msk                (0xFFFFU << RTC_DIVL_DIV_Pos)  /*!< 0x0000FFFF */
#define RTC_DIVL_RTC_DIV                    RTC_DIVL_DIV_Msk               /*!< RTC Clock Divider Low */

/*******************  Bit definition for RTC_CNTH register  *******************/
#define RTC_CNTH_RTC_CNT_Pos                (0U)                               
#define RTC_CNTH_RTC_CNT_Msk                (0xFFFFU << RTC_CNTH_RTC_CNT_Pos)  /*!< 0x0000FFFF */
#define RTC_CNTH_RTC_CNT                    RTC_CNTH_RTC_CNT_Msk               /*!< RTC Counter High */

/*******************  Bit definition for RTC_CNTL register  *******************/
#define RTC_CNTL_RTC_CNT_Pos                (0U)                               
#define RTC_CNTL_RTC_CNT_Msk                (0xFFFFU << RTC_CNTL_RTC_CNT_Pos)  /*!< 0x0000FFFF */
#define RTC_CNTL_RTC_CNT                    RTC_CNTL_RTC_CNT_Msk               /*!< RTC Counter Low */

/*******************  Bit definition for RTC_ALRMH register  *******************/
#define RTC_ALRMH_RTC_ALRM_Pos                (0U)                               
#define RTC_ALRMH_RTC_ALRM_Msk                (0xFFFFU << RTC_ALRMH_RTC_ALRM_Pos)  /*!< 0x0000FFFF */
#define RTC_ALRMH_RTC_ALRM                    RTC_ALRMH_RTC_ALRM_Msk               /*!< RTC Alarm High */

/*******************  Bit definition for RTC_ALRML register  *******************/
#define RTC_ALRML_RTC_ALRM_Pos                (0U)                               
#define RTC_ALRML_RTC_ALRM_Msk                (0xFFFFU << RTC_ALRML_RTC_ALRM_Pos)  /*!< 0x0000FFFF */
#define RTC_ALRML_RTC_ALRM                    RTC_ALRML_RTC_ALRM_Msk               /*!< RTC Alarm Low */

/******************************************************************************/
/*                                                                            */
/*                        FREE WATCHDOG TIMER (FWDGT)                         */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for FWDGT_CTL register  ********************/
#define FWDGT_CTL_CMD_Pos                     (0U)                               
#define FWDGT_CTL_CMD_Msk                     (0xFFFFU << FWDGT_CTL_CMD_Pos)       /*!< 0x0000FFFF */
#define FWDGT_CTL_CMD                         FWDGT_CTL_CMD_Msk                    /*!< Key value (write only, read 0000h) */

/*******************  Bit definition for FWDGT_PSC register  ********************/
#define FWDGT_PSC_PSC_Pos                      (0U)                               
#define FWDGT_PSC_PSC_Msk                      (0x7U << FWDGT_PSC_PSC_Pos)           /*!< 0x00000007 */
#define FWDGT_PSC_PSC                          FWDGT_PSC_PSC_Msk                     /*!< PR[2:0] (Prescaler divider) */
#define FWDGT_PSC_PSC_0                        (0x1U << FWDGT_PSC_PSC_Pos)           /*!< 0x00000001 */
#define FWDGT_PSC_PSC_1                        (0x2U << FWDGT_PSC_PSC_Pos)           /*!< 0x00000002 */
#define FWDGT_PSC_PSC_2                        (0x4U << FWDGT_PSC_PSC_Pos)           /*!< 0x00000004 */

/*******************  Bit definition for FWDGT_RLD register  *******************/
#define FWDGT_RLD_RLD_Pos                     (0U)                               
#define FWDGT_RLD_RLD_Msk                     (0xFFFU << FWDGT_RLD_RLD_Pos)        /*!< 0x00000FFF */
#define FWDGT_RLD_RLD                         FWDGT_RLD_RLD_Msk                    /*!< Watchdog counter reload value */

/*******************  Bit definition for FWDGT_STAT register  ********************/
#define FWDGT_STAT_PUD_Pos                     (0U)                               
#define FWDGT_STAT_PUD_Msk                     (0x1U << FWDGT_STAT_PUD_Pos)          /*!< 0x00000001 */
#define FWDGT_STAT_PUD                         FWDGT_STAT_PUD_Msk                    /*!< Watchdog prescaler value update */
#define FWDGT_STAT_RUD_Pos                     (1U)                               
#define FWDGT_STAT_RUD_Msk                     (0x1U << FWDGT_STAT_RUD_Pos)          /*!< 0x00000002 */
#define FWDGT_STAT_RUD                         FWDGT_STAT_RUD_Msk                    /*!< Watchdog counter reload value update */

/******************************************************************************/
/*                                                                            */
/*                         Window WATCHDOG (WWDG)                             */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for WWDG_CR register  ********************/
#define WWDG_CR_T_Pos                       (0U)                               
#define WWDG_CR_T_Msk                       (0x7FU << WWDG_CR_T_Pos)           /*!< 0x0000007F */
#define WWDG_CR_T                           WWDG_CR_T_Msk                      /*!< T[6:0] bits (7-Bit counter (MSB to LSB)) */
#define WWDG_CR_T_0                         (0x01U << WWDG_CR_T_Pos)           /*!< 0x00000001 */
#define WWDG_CR_T_1                         (0x02U << WWDG_CR_T_Pos)           /*!< 0x00000002 */
#define WWDG_CR_T_2                         (0x04U << WWDG_CR_T_Pos)           /*!< 0x00000004 */
#define WWDG_CR_T_3                         (0x08U << WWDG_CR_T_Pos)           /*!< 0x00000008 */
#define WWDG_CR_T_4                         (0x10U << WWDG_CR_T_Pos)           /*!< 0x00000010 */
#define WWDG_CR_T_5                         (0x20U << WWDG_CR_T_Pos)           /*!< 0x00000020 */
#define WWDG_CR_T_6                         (0x40U << WWDG_CR_T_Pos)           /*!< 0x00000040 */

/* Legacy defines */
#define  WWDG_CR_T0 WWDG_CR_T_0
#define  WWDG_CR_T1 WWDG_CR_T_1
#define  WWDG_CR_T2 WWDG_CR_T_2
#define  WWDG_CR_T3 WWDG_CR_T_3
#define  WWDG_CR_T4 WWDG_CR_T_4
#define  WWDG_CR_T5 WWDG_CR_T_5
#define  WWDG_CR_T6 WWDG_CR_T_6

#define WWDG_CR_WDGA_Pos                    (7U)                               
#define WWDG_CR_WDGA_Msk                    (0x1U << WWDG_CR_WDGA_Pos)         /*!< 0x00000080 */
#define WWDG_CR_WDGA                        WWDG_CR_WDGA_Msk                   /*!< Activation bit */

/*******************  Bit definition for WWDG_CFR register  *******************/
#define WWDG_CFR_W_Pos                      (0U)                               
#define WWDG_CFR_W_Msk                      (0x7FU << WWDG_CFR_W_Pos)          /*!< 0x0000007F */
#define WWDG_CFR_W                          WWDG_CFR_W_Msk                     /*!< W[6:0] bits (7-bit window value) */
#define WWDG_CFR_W_0                        (0x01U << WWDG_CFR_W_Pos)          /*!< 0x00000001 */
#define WWDG_CFR_W_1                        (0x02U << WWDG_CFR_W_Pos)          /*!< 0x00000002 */
#define WWDG_CFR_W_2                        (0x04U << WWDG_CFR_W_Pos)          /*!< 0x00000004 */
#define WWDG_CFR_W_3                        (0x08U << WWDG_CFR_W_Pos)          /*!< 0x00000008 */
#define WWDG_CFR_W_4                        (0x10U << WWDG_CFR_W_Pos)          /*!< 0x00000010 */
#define WWDG_CFR_W_5                        (0x20U << WWDG_CFR_W_Pos)          /*!< 0x00000020 */
#define WWDG_CFR_W_6                        (0x40U << WWDG_CFR_W_Pos)          /*!< 0x00000040 */

/* Legacy defines */
#define  WWDG_CFR_W0 WWDG_CFR_W_0
#define  WWDG_CFR_W1 WWDG_CFR_W_1
#define  WWDG_CFR_W2 WWDG_CFR_W_2
#define  WWDG_CFR_W3 WWDG_CFR_W_3
#define  WWDG_CFR_W4 WWDG_CFR_W_4
#define  WWDG_CFR_W5 WWDG_CFR_W_5
#define  WWDG_CFR_W6 WWDG_CFR_W_6

#define WWDG_CFR_WDGTB_Pos                  (7U)                               
#define WWDG_CFR_WDGTB_Msk                  (0x3U << WWDG_CFR_WDGTB_Pos)       /*!< 0x00000180 */
#define WWDG_CFR_WDGTB                      WWDG_CFR_WDGTB_Msk                 /*!< WDGTB[1:0] bits (Timer Base) */
#define WWDG_CFR_WDGTB_0                    (0x1U << WWDG_CFR_WDGTB_Pos)       /*!< 0x00000080 */
#define WWDG_CFR_WDGTB_1                    (0x2U << WWDG_CFR_WDGTB_Pos)       /*!< 0x00000100 */

/* Legacy defines */
#define  WWDG_CFR_WDGTB0 WWDG_CFR_WDGTB_0
#define  WWDG_CFR_WDGTB1 WWDG_CFR_WDGTB_1

#define WWDG_CFR_EWI_Pos                    (9U)                               
#define WWDG_CFR_EWI_Msk                    (0x1U << WWDG_CFR_EWI_Pos)         /*!< 0x00000200 */
#define WWDG_CFR_EWI                        WWDG_CFR_EWI_Msk                   /*!< Early Wakeup Interrupt */

/*******************  Bit definition for WWDG_SR register  ********************/
#define WWDG_SR_EWIF_Pos                    (0U)                               
#define WWDG_SR_EWIF_Msk                    (0x1U << WWDG_SR_EWIF_Pos)         /*!< 0x00000001 */
#define WWDG_SR_EWIF                        WWDG_SR_EWIF_Msk                   /*!< Early Wakeup Interrupt Flag */

/******************************************************************************/
/*                                                                            */
/*                         Controller Area Network                            */
/*                                                                            */
/******************************************************************************/

/*!< CAN control and status registers */
/*******************  Bit definition for CAN_CTL register  ********************/
#define CAN_CTL_IWMOD_Pos                     (0U)                              
#define CAN_CTL_IWMOD_Msk                     (0x1U << CAN_CTL_IWMOD_Pos)        /*!< 0x00000001 */
#define CAN_CTL_IWMOD                         CAN_CTL_IWMOD_Msk                  /*!< Initialization Request */
#define CAN_CTL_SLPWMOD_Pos                    (1U)                              
#define CAN_CTL_SLPWMOD_Msk                    (0x1U << CAN_CTL_SLPWMOD_Pos)       /*!< 0x00000002 */
#define CAN_CTL_SLPWMOD                        CAN_CTL_SLPWMOD_Msk                 /*!< Sleep Mode Request */
#define CAN_CTL_TFO_Pos                     (2U)                              
#define CAN_CTL_TFO_Msk                     (0x1U << CAN_CTL_TFO_Pos)        /*!< 0x00000004 */
#define CAN_CTL_TFO                         CAN_CTL_TFO_Msk                  /*!< Transmit FIFO Priority */
#define CAN_CTL_RFOD_Pos                     (3U)                              
#define CAN_CTL_RFOD_Msk                     (0x1U << CAN_CTL_RFOD_Pos)        /*!< 0x00000008 */
#define CAN_CTL_RFOD                         CAN_CTL_RFOD_Msk                  /*!< Receive FIFO Locked Mode */
#define CAN_CTL_ARD_Pos                     (4U)                              
#define CAN_CTL_ARD_Msk                     (0x1U << CAN_CTL_ARD_Pos)        /*!< 0x00000010 */
#define CAN_CTL_ARD                         CAN_CTL_ARD_Msk                  /*!< No Automatic Retransmission */
#define CAN_CTL_AWU_Pos                     (5U)                              
#define CAN_CTL_AWU_Msk                     (0x1U << CAN_CTL_AWU_Pos)        /*!< 0x00000020 */
#define CAN_CTL_AWU                         CAN_CTL_AWU_Msk                  /*!< Automatic Wakeup Mode */
#define CAN_CTL_ABOR_Pos                     (6U)                              
#define CAN_CTL_ABOR_Msk                     (0x1U << CAN_CTL_ABOR_Pos)        /*!< 0x00000040 */
#define CAN_CTL_ABOR                         CAN_CTL_ABOR_Msk                  /*!< Automatic Bus-Off Management */
#define CAN_CTL_TTC_Pos                     (7U)                              
#define CAN_CTL_TTC_Msk                     (0x1U << CAN_CTL_TTC_Pos)        /*!< 0x00000080 */
#define CAN_CTL_TTC                         CAN_CTL_TTC_Msk                  /*!< Time Triggered Communication Mode */
#define CAN_CTL_SWRST_Pos                    (15U)                             
#define CAN_CTL_SWRST_Msk                    (0x1U << CAN_CTL_SWRST_Pos)       /*!< 0x00008000 */
#define CAN_CTL_SWRST                        CAN_CTL_SWRST_Msk                 /*!< CAN software master reset */
#define CAN_CTL_DFZ_Pos                      (16U)                             
#define CAN_CTL_DFZ_Msk                      (0x1U << CAN_CTL_DFZ_Pos)         /*!< 0x00010000 */
#define CAN_CTL_DFZ                          CAN_CTL_DFZ_Msk                   /*!< CAN Debug freeze */

/*******************  Bit definition for CAN_STAT register  ********************/
#define CAN_STAT_IWS_Pos                     (0U)                              
#define CAN_STAT_IWS_Msk                     (0x1U << CAN_STAT_IWS_Pos)        /*!< 0x00000001 */
#define CAN_STAT_IWS                         CAN_STAT_IWS_Msk                  /*!< Initialization Acknowledge */
#define CAN_STAT_SLPWS_Pos                     (1U)                              
#define CAN_STAT_SLPWS_Msk                     (0x1U << CAN_STAT_SLPWS_Pos)        /*!< 0x00000002 */
#define CAN_STAT_SLPWS                         CAN_STAT_SLPWS_Msk                  /*!< Sleep Acknowledge */
#define CAN_STAT_ERRIF_Pos                     (2U)                              
#define CAN_STAT_ERRIF_Msk                     (0x1U << CAN_STAT_ERRIF_Pos)        /*!< 0x00000004 */
#define CAN_STAT_ERRIF                         CAN_STAT_ERRIF_Msk                  /*!< Error Interrupt */
#define CAN_STAT_WUIF_Pos                     (3U)                              
#define CAN_STAT_WUIF_Msk                     (0x1U << CAN_STAT_WUIF_Pos)        /*!< 0x00000008 */
#define CAN_STAT_WUIF                         CAN_STAT_WUIF_Msk                  /*!< Wakeup Interrupt */
#define CAN_STAT_SLPWSI_Pos                    (4U)                              
#define CAN_STAT_SLPWSI_Msk                    (0x1U << CAN_STAT_SLPWSI_Pos)       /*!< 0x00000010 */
#define CAN_STAT_SLPWSI                        CAN_STAT_SLPWSI_Msk                 /*!< Sleep Acknowledge Interrupt */
#define CAN_STAT_TS_Pos                      (8U)                              
#define CAN_STAT_TS_Msk                      (0x1U << CAN_STAT_TS_Pos)         /*!< 0x00000100 */
#define CAN_STAT_TS                          CAN_STAT_TS_Msk                   /*!< Transmit Mode */
#define CAN_STAT_RS_Pos                      (9U)                              
#define CAN_STAT_RS_Msk                      (0x1U << CAN_STAT_RS_Pos)         /*!< 0x00000200 */
#define CAN_STAT_RS                          CAN_STAT_RS_Msk                   /*!< Receive Mode */
#define CAN_STAT_LASTRX_Pos                     (10U)                             
#define CAN_STAT_LASTRX_Msk                     (0x1U << CAN_STAT_LASTRX_Pos)        /*!< 0x00000400 */
#define CAN_STAT_LASTRX                         CAN_STAT_LASTRX_Msk                  /*!< Last Sample Point */
#define CAN_STAT_RXL_Pos                       (11U)                             
#define CAN_STAT_RXL_Msk                       (0x1U << CAN_STAT_RXL_Pos)          /*!< 0x00000800 */
#define CAN_STAT_RXL                           CAN_STAT_RXL_Msk                    /*!< CAN Rx Signal */

/*******************  Bit definition for CAN_TSTAT register  ********************/
#define CAN_TSTAT_MTF0_Pos                    (0U)                              
#define CAN_TSTAT_MTF0_Msk                    (0x1U << CAN_TSTAT_MTF0_Pos)       /*!< 0x00000001 */
#define CAN_TSTAT_MTF0                        CAN_TSTAT_MTF0_Msk                 /*!< Request Completed Mailbox0 */
#define CAN_TSTAT_MTFNERR0_Pos                    (1U)                              
#define CAN_TSTAT_MTFNERR0_Msk                    (0x1U << CAN_TSTAT_MTFNERR0_Pos)       /*!< 0x00000002 */
#define CAN_TSTAT_MTFNERR0                        CAN_TSTAT_MTFNERR0_Msk                 /*!< Transmission OK of Mailbox0 */
#define CAN_TSTAT_MAL0_Pos                    (2U)                              
#define CAN_TSTAT_MAL0_Msk                    (0x1U << CAN_TSTAT_MAL0_Pos)       /*!< 0x00000004 */
#define CAN_TSTAT_MAL0                        CAN_TSTAT_MAL0_Msk                 /*!< Arbitration Lost for Mailbox0 */
#define CAN_TSTAT_MTE0_Pos                    (3U)                              
#define CAN_TSTAT_MTE0_Msk                    (0x1U << CAN_TSTAT_MTE0_Pos)       /*!< 0x00000008 */
#define CAN_TSTAT_MTE0                        CAN_TSTAT_MTE0_Msk                 /*!< Transmission Error of Mailbox0 */
#define CAN_TSTAT_MST0_Pos                    (7U)                              
#define CAN_TSTAT_MST0_Msk                    (0x1U << CAN_TSTAT_MST0_Pos)       /*!< 0x00000080 */
#define CAN_TSTAT_MST0                        CAN_TSTAT_MST0_Msk                 /*!< Abort Request for Mailbox0 */
#define CAN_TSTAT_MTF1_Pos                    (8U)                              
#define CAN_TSTAT_MTF1_Msk                    (0x1U << CAN_TSTAT_MTF1_Pos)       /*!< 0x00000100 */
#define CAN_TSTAT_MTF1                        CAN_TSTAT_MTF1_Msk                 /*!< Request Completed Mailbox1 */
#define CAN_TSTAT_MTFNERR1_Pos                    (9U)                              
#define CAN_TSTAT_MTFNERR1_Msk                    (0x1U << CAN_TSTAT_MTFNERR1_Pos)       /*!< 0x00000200 */
#define CAN_TSTAT_MTFNERR1                        CAN_TSTAT_MTFNERR1_Msk                 /*!< Transmission OK of Mailbox1 */
#define CAN_TSTAT_MAL1_Pos                    (10U)                             
#define CAN_TSTAT_MAL1_Msk                    (0x1U << CAN_TSTAT_MAL1_Pos)       /*!< 0x00000400 */
#define CAN_TSTAT_MAL1                        CAN_TSTAT_MAL1_Msk                 /*!< Arbitration Lost for Mailbox1 */
#define CAN_TSTAT_MTE1_Pos                    (11U)                             
#define CAN_TSTAT_MTE1_Msk                    (0x1U << CAN_TSTAT_MTE1_Pos)       /*!< 0x00000800 */
#define CAN_TSTAT_MTE1                        CAN_TSTAT_MTE1_Msk                 /*!< Transmission Error of Mailbox1 */
#define CAN_TSTAT_MST1_Pos                    (15U)                             
#define CAN_TSTAT_MST1_Msk                    (0x1U << CAN_TSTAT_MST1_Pos)       /*!< 0x00008000 */
#define CAN_TSTAT_MST1                        CAN_TSTAT_MST1_Msk                 /*!< Abort Request for Mailbox 1 */
#define CAN_TSTAT_MTF2_Pos                    (16U)                             
#define CAN_TSTAT_MTF2_Msk                    (0x1U << CAN_TSTAT_MTF2_Pos)       /*!< 0x00010000 */
#define CAN_TSTAT_MTF2                        CAN_TSTAT_MTF2_Msk                 /*!< Request Completed Mailbox2 */
#define CAN_TSTAT_MTFNERR2_Pos                    (17U)                             
#define CAN_TSTAT_MTFNERR2_Msk                    (0x1U << CAN_TSTAT_MTFNERR2_Pos)       /*!< 0x00020000 */
#define CAN_TSTAT_MTFNERR2                        CAN_TSTAT_MTFNERR2_Msk                 /*!< Transmission OK of Mailbox 2 */
#define CAN_TSTAT_MAL2_Pos                    (18U)                             
#define CAN_TSTAT_MAL2_Msk                    (0x1U << CAN_TSTAT_MAL2_Pos)       /*!< 0x00040000 */
#define CAN_TSTAT_MAL2                        CAN_TSTAT_MAL2_Msk                 /*!< Arbitration Lost for mailbox 2 */
#define CAN_TSTAT_MTE2_Pos                    (19U)                             
#define CAN_TSTAT_MTE2_Msk                    (0x1U << CAN_TSTAT_MTE2_Pos)       /*!< 0x00080000 */
#define CAN_TSTAT_MTE2                        CAN_TSTAT_MTE2_Msk                 /*!< Transmission Error of Mailbox 2 */
#define CAN_TSTAT_MST2_Pos                    (23U)                             
#define CAN_TSTAT_MST2_Msk                    (0x1U << CAN_TSTAT_MST2_Pos)       /*!< 0x00800000 */
#define CAN_TSTAT_MST2                        CAN_TSTAT_MST2_Msk                 /*!< Abort Request for Mailbox 2 */
#define CAN_TSTAT_NUM_Pos                     (24U)                             
#define CAN_TSTAT_NUM_Msk                     (0x3U << CAN_TSTAT_NUM_Pos)        /*!< 0x03000000 */
#define CAN_TSTAT_NUM                         CAN_TSTAT_NUM_Msk                  /*!< Mailbox Code */

#define CAN_TSTAT_TME_Pos                      (26U)                             
#define CAN_TSTAT_TME_Msk                      (0x7U << CAN_TSTAT_TME_Pos)         /*!< 0x1C000000 */
#define CAN_TSTAT_TME                          CAN_TSTAT_TME_Msk                   /*!< TME[2:0] bits */
#define CAN_TSTAT_TME0_Pos                     (26U)                             
#define CAN_TSTAT_TME0_Msk                     (0x1U << CAN_TSTAT_TME0_Pos)        /*!< 0x04000000 */
#define CAN_TSTAT_TME0                         CAN_TSTAT_TME0_Msk                  /*!< Transmit Mailbox 0 Empty */
#define CAN_TSTAT_TME1_Pos                     (27U)                             
#define CAN_TSTAT_TME1_Msk                     (0x1U << CAN_TSTAT_TME1_Pos)        /*!< 0x08000000 */
#define CAN_TSTAT_TME1                         CAN_TSTAT_TME1_Msk                  /*!< Transmit Mailbox 1 Empty */
#define CAN_TSTAT_TME2_Pos                     (28U)                             
#define CAN_TSTAT_TME2_Msk                     (0x1U << CAN_TSTAT_TME2_Pos)        /*!< 0x10000000 */
#define CAN_TSTAT_TME2                         CAN_TSTAT_TME2_Msk                  /*!< Transmit Mailbox 2 Empty */

#define CAN_TSTAT_TMLS_Pos                      (29U)                             
#define CAN_TSTAT_TMLS_Msk                      (0x7U << CAN_TSTAT_TMLS_Pos)         /*!< 0xE0000000 */
#define CAN_TSTAT_TMLS                          CAN_TSTAT_TMLS_Msk                   /*!< LOW[2:0] bits */
#define CAN_TSTAT_TMLS0_Pos                     (29U)                             
#define CAN_TSTAT_TMLS0_Msk                     (0x1U << CAN_TSTAT_TMLS0_Pos)        /*!< 0x20000000 */
#define CAN_TSTAT_TMLS0                         CAN_TSTAT_TMLS0_Msk                  /*!< Lowest Priority Flag for Mailbox 0 */
#define CAN_TSTAT_TMLS1_Pos                     (30U)                             
#define CAN_TSTAT_TMLS1_Msk                     (0x1U << CAN_TSTAT_TMLS1_Pos)        /*!< 0x40000000 */
#define CAN_TSTAT_TMLS1                         CAN_TSTAT_TMLS1_Msk                  /*!< Lowest Priority Flag for Mailbox 1 */
#define CAN_TSTAT_TMLS2_Pos                     (31U)                             
#define CAN_TSTAT_TMLS2_Msk                     (0x1U << CAN_TSTAT_TMLS2_Pos)        /*!< 0x80000000 */
#define CAN_TSTAT_TMLS2                         CAN_TSTAT_TMLS2_Msk                  /*!< Lowest Priority Flag for Mailbox 2 */

/*******************  Bit definition for CAN_RFIFO0 register  *******************/
#define CAN_RFIFO0_RFL0_Pos                    (0U)                              
#define CAN_RFIFO0_RFL0_Msk                    (0x3U << CAN_RFIFO0_RFL0_Pos)       /*!< 0x00000003 */
#define CAN_RFIFO0_RFL0                        CAN_RFIFO0_RFL0_Msk                 /*!< FIFO 0 Message Pending */
#define CAN_RFIFO0_RFF0_Pos                   (3U)                              
#define CAN_RFIFO0_RFF0_Msk                   (0x1U << CAN_RFIFO0_RFF0_Pos)      /*!< 0x00000008 */
#define CAN_RFIFO0_RFF0                       CAN_RFIFO0_RFF0_Msk                /*!< FIFO 0 Full */
#define CAN_RFIFO0_RFO0_Pos                   (4U)                              
#define CAN_RFIFO0_RFO0_Msk                   (0x1U << CAN_RFIFO0_RFO0_Pos)      /*!< 0x00000010 */
#define CAN_RFIFO0_RFO0                       CAN_RFIFO0_RFO0_Msk                /*!< FIFO 0 Overrun */
#define CAN_RFIFO0_RFD0_Pos                   (5U)                              
#define CAN_RFIFO0_RFD0_Msk                   (0x1U << CAN_RFIFO0_RFD0_Pos)      /*!< 0x00000020 */
#define CAN_RFIFO0_RFD0                       CAN_RFIFO0_RFD0_Msk                /*!< Release FIFO 0 Output Mailbox */

/*******************  Bit definition for CAN_RFIFO1 register  *******************/
#define CAN_RFIFO1_RFL1_Pos                    (0U)                              
#define CAN_RFIFO1_RFL1_Msk                    (0x3U << CAN_RFIFO1_RFL1_Pos)       /*!< 0x00000003 */
#define CAN_RFIFO1_RFL1                        CAN_RFIFO1_RFL1_Msk                 /*!< FIFO 1 Message Pending */
#define CAN_RFIFO1_RFF1_Pos                   (3U)                              
#define CAN_RFIFO1_RFF1_Msk                   (0x1U << CAN_RFIFO1_RFF1_Pos)      /*!< 0x00000008 */
#define CAN_RFIFO1_RFF1                       CAN_RFIFO1_RFF1_Msk                /*!< FIFO 1 Full */
#define CAN_RFIFO1_RFO1_Pos                   (4U)                              
#define CAN_RFIFO1_RFO1_Msk                   (0x1U << CAN_RFIFO1_RFO1_Pos)      /*!< 0x00000010 */
#define CAN_RFIFO1_RFO1                       CAN_RFIFO1_RFO1_Msk                /*!< FIFO 1 Overrun */
#define CAN_RFIFO1_RFD1_Pos                   (5U)                              
#define CAN_RFIFO1_RFD1_Msk                   (0x1U << CAN_RFIFO1_RFD1_Pos)      /*!< 0x00000020 */
#define CAN_RFIFO1_RFD1                       CAN_RFIFO1_RFD1_Msk                /*!< Release FIFO 1 Output Mailbox */

/********************  Bit definition for CAN_INTEN register  *******************/
#define CAN_INTEN_TMEIE_Pos                    (0U)                              
#define CAN_INTEN_TMEIE_Msk                    (0x1U << CAN_INTEN_TMEIE_Pos)       /*!< 0x00000001 */
#define CAN_INTEN_TMEIE                        CAN_INTEN_TMEIE_Msk                 /*!< Transmit Mailbox Empty Interrupt Enable */
#define CAN_INTEN_RFNEIE0_Pos                   (1U)                              
#define CAN_INTEN_RFNEIE0_Msk                   (0x1U << CAN_INTEN_RFNEIE0_Pos)      /*!< 0x00000002 */
#define CAN_INTEN_RFNEIE0                       CAN_INTEN_RFNEIE0_Msk                /*!< FIFO Message Pending Interrupt Enable */
#define CAN_INTEN_RFFIE0_Pos                    (2U)                              
#define CAN_INTEN_RFFIE0_Msk                    (0x1U << CAN_INTEN_RFFIE0_Pos)       /*!< 0x00000004 */
#define CAN_INTEN_RFFIE0                        CAN_INTEN_RFFIE0_Msk                 /*!< FIFO Full Interrupt Enable */
#define CAN_INTEN_RFOIE0_Pos                   (3U)                              
#define CAN_INTEN_RFOIE0_Msk                   (0x1U << CAN_INTEN_RFOIE0_Pos)      /*!< 0x00000008 */
#define CAN_INTEN_RFOIE0                       CAN_INTEN_RFOIE0_Msk                /*!< FIFO Overrun Interrupt Enable */
#define CAN_INTEN_RFNEIE1_Pos                   (4U)                              
#define CAN_INTEN_RFNEIE1_Msk                   (0x1U << CAN_INTEN_RFNEIE1_Pos)      /*!< 0x00000010 */
#define CAN_INTEN_RFNEIE1                       CAN_INTEN_RFNEIE1_Msk                /*!< FIFO Message Pending Interrupt Enable */
#define CAN_INTEN_RFFIE1_Pos                    (5U)                              
#define CAN_INTEN_RFFIE1_Msk                    (0x1U << CAN_INTEN_RFFIE1_Pos)       /*!< 0x00000020 */
#define CAN_INTEN_RFFIE1                        CAN_INTEN_RFFIE1_Msk                 /*!< FIFO Full Interrupt Enable */
#define CAN_INTEN_RFOIE1_Pos                   (6U)                              
#define CAN_INTEN_RFOIE1_Msk                   (0x1U << CAN_INTEN_RFOIE1_Pos)      /*!< 0x00000040 */
#define CAN_INTEN_RFOIE1                       CAN_INTEN_RFOIE1_Msk                /*!< FIFO Overrun Interrupt Enable */
#define CAN_INTEN_WERRIE_Pos                    (8U)                              
#define CAN_INTEN_WERRIE_Msk                    (0x1U << CAN_INTEN_WERRIE_Pos)       /*!< 0x00000100 */
#define CAN_INTEN_WERRIE                        CAN_INTEN_WERRIE_Msk                 /*!< Error Warning Interrupt Enable */
#define CAN_INTEN_PERRIE_Pos                    (9U)                              
#define CAN_INTEN_PERRIE_Msk                    (0x1U << CAN_INTEN_PERRIE_Pos)       /*!< 0x00000200 */
#define CAN_INTEN_PERRIE                        CAN_INTEN_PERRIE_Msk                 /*!< Error Passive Interrupt Enable */
#define CAN_INTEN_BOIE_Pos                    (10U)                             
#define CAN_INTEN_BOIE_Msk                    (0x1U << CAN_INTEN_BOIE_Pos)       /*!< 0x00000400 */
#define CAN_INTEN_BOIE                        CAN_INTEN_BOIE_Msk                 /*!< Bus-Off Interrupt Enable */
#define CAN_INTEN_ERRNIE_Pos                    (11U)                             
#define CAN_INTEN_ERRNIE_Msk                    (0x1U << CAN_INTEN_ERRNIE_Pos)       /*!< 0x00000800 */
#define CAN_INTEN_ERRNIE                        CAN_INTEN_ERRNIE_Msk                 /*!< Last Error Code Interrupt Enable */
#define CAN_INTEN_ERRIE_Pos                    (15U)                             
#define CAN_INTEN_ERRIE_Msk                    (0x1U << CAN_INTEN_ERRIE_Pos)       /*!< 0x00008000 */
#define CAN_INTEN_ERRIE                        CAN_INTEN_ERRIE_Msk                 /*!< Error Interrupt Enable */
#define CAN_INTEN_WIE_Pos                    (16U)                             
#define CAN_INTEN_WIE_Msk                    (0x1U << CAN_INTEN_WIE_Pos)       /*!< 0x00010000 */
#define CAN_INTEN_WIE                        CAN_INTEN_WIE_Msk                 /*!< Wakeup Interrupt Enable */
#define CAN_INTEN_SLPWIE_Pos                    (17U)                             
#define CAN_INTEN_SLPWIE_Msk                    (0x1U << CAN_INTEN_SLPWIE_Pos)       /*!< 0x00020000 */
#define CAN_INTEN_SLPWIE                        CAN_INTEN_SLPWIE_Msk                 /*!< Sleep Interrupt Enable */

/********************  Bit definition for CAN_ERR register  *******************/
#define CAN_ERR_WERR_Pos                     (0U)                              
#define CAN_ERR_WERR_Msk                     (0x1U << CAN_ERR_WERR_Pos)        /*!< 0x00000001 */
#define CAN_ERR_WERR                         CAN_ERR_WERR_Msk                  /*!< Error Warning Flag */
#define CAN_ERR_PERR_Pos                     (1U)                              
#define CAN_ERR_PERR_Msk                     (0x1U << CAN_ERR_PERR_Pos)        /*!< 0x00000002 */
#define CAN_ERR_PERR                         CAN_ERR_PERR_Msk                  /*!< Error Passive Flag */
#define CAN_ERR_BOERR_Pos                     (2U)                              
#define CAN_ERR_BOERR_Msk                     (0x1U << CAN_ERR_BOERR_Pos)        /*!< 0x00000004 */
#define CAN_ERR_BOERR                         CAN_ERR_BOERR_Msk                  /*!< Bus-Off Flag */

#define CAN_ERR_ERRN_Pos                      (4U)                              
#define CAN_ERR_ERRN_Msk                      (0x7U << CAN_ERR_ERRN_Pos)         /*!< 0x00000070 */
#define CAN_ERR_ERRN                          CAN_ERR_ERRN_Msk                   /*!< LEC[2:0] bits (Last Error Code) */
#define CAN_ERR_ERRN_0                        (0x1U << CAN_ERR_ERRN_Pos)         /*!< 0x00000010 */
#define CAN_ERR_ERRN_1                        (0x2U << CAN_ERR_ERRN_Pos)         /*!< 0x00000020 */
#define CAN_ERR_ERRN_2                        (0x4U << CAN_ERR_ERRN_Pos)         /*!< 0x00000040 */

#define CAN_ERR_TECNT_Pos                      (16U)                             
#define CAN_ERR_TECNT_Msk                      (0xFFU << CAN_ERR_TECNT_Pos)        /*!< 0x00FF0000 */
#define CAN_ERR_TECNT                          CAN_ERR_TECNT_Msk                   /*!< Least significant byte of the 9-bit Transmit Error Counter */
#define CAN_ERR_RECNT_Pos                      (24U)                             
#define CAN_ERR_RECNT_Msk                      (0xFFU << CAN_ERR_RECNT_Pos)        /*!< 0xFF000000 */
#define CAN_ERR_RECNT                          CAN_ERR_RECNT_Msk                   /*!< Receive Error Counter */

/*******************  Bit definition for CAN_BT register  ********************/
#define CAN_BT_BAUDPSC_Pos                      (0U)                              
#define CAN_BT_BAUDPSC_Msk                      (0x3FFU << CAN_BT_BAUDPSC_Pos)       /*!< 0x000003FF */
#define CAN_BT_BAUDPSC                          CAN_BT_BAUDPSC_Msk                   /*!<Baud Rate Prescaler */
#define CAN_BT_BS1_Pos                      (16U)                             
#define CAN_BT_BS1_Msk                      (0xFU << CAN_BT_BS1_Pos)         /*!< 0x000F0000 */
#define CAN_BT_BS1                          CAN_BT_BS1_Msk                   /*!<Time Segment 1 */
#define CAN_BT_BS1_0                        (0x1U << CAN_BT_BS1_Pos)         /*!< 0x00010000 */
#define CAN_BT_BS1_1                        (0x2U << CAN_BT_BS1_Pos)         /*!< 0x00020000 */
#define CAN_BT_BS1_2                        (0x4U << CAN_BT_BS1_Pos)         /*!< 0x00040000 */
#define CAN_BT_BS1_3                        (0x8U << CAN_BT_BS1_Pos)         /*!< 0x00080000 */
#define CAN_BT_BS2_Pos                      (20U)                             
#define CAN_BT_BS2_Msk                      (0x7U << CAN_BT_BS2_Pos)         /*!< 0x00700000 */
#define CAN_BT_BS2                          CAN_BT_BS2_Msk                   /*!<Time Segment 2 */
#define CAN_BT_BS2_0                        (0x1U << CAN_BT_BS2_Pos)         /*!< 0x00100000 */
#define CAN_BT_BS2_1                        (0x2U << CAN_BT_BS2_Pos)         /*!< 0x00200000 */
#define CAN_BT_BS2_2                        (0x4U << CAN_BT_BS2_Pos)         /*!< 0x00400000 */
#define CAN_BT_SJW_Pos                      (24U)                             
#define CAN_BT_SJW_Msk                      (0x3U << CAN_BT_SJW_Pos)         /*!< 0x03000000 */
#define CAN_BT_SJW                          CAN_BT_SJW_Msk                   /*!<Resynchronization Jump Width */
#define CAN_BT_SJW_0                        (0x1U << CAN_BT_SJW_Pos)         /*!< 0x01000000 */
#define CAN_BT_SJW_1                        (0x2U << CAN_BT_SJW_Pos)         /*!< 0x02000000 */
#define CAN_BT_LCMOD_Pos                     (30U)                             
#define CAN_BT_LCMOD_Msk                     (0x1U << CAN_BT_LCMOD_Pos)        /*!< 0x40000000 */
#define CAN_BT_LCMOD                         CAN_BT_LCMOD_Msk                  /*!<Loop Back Mode (Debug) */
#define CAN_BT_SCMOD_Pos                     (31U)                             
#define CAN_BT_SCMOD_Msk                     (0x1U << CAN_BT_SCMOD_Pos)        /*!< 0x80000000 */
#define CAN_BT_SCMOD                         CAN_BT_SCMOD_Msk                  /*!<Silent Mode */

/*!< Mailbox registers */
/******************  Bit definition for CAN_TMI0 register  ********************/
#define CAN_TMI0_TEN_Pos                    (0U)                              
#define CAN_TMI0_TEN_Msk                    (0x1U << CAN_TMI0_TEN_Pos)       /*!< 0x00000001 */
#define CAN_TMI0_TEN                        CAN_TMI0_TEN_Msk                 /*!< Transmit Mailbox Request */
#define CAN_TMI0_FT_Pos                     (1U)                              
#define CAN_TMI0_FT_Msk                     (0x1U << CAN_TMI0_FT_Pos)        /*!< 0x00000002 */
#define CAN_TMI0_FT                         CAN_TMI0_FT_Msk                  /*!< Remote Transmission Request */
#define CAN_TMI0_FF_Pos                     (2U)                              
#define CAN_TMI0_FF_Msk                     (0x1U << CAN_TMI0_FF_Pos)        /*!< 0x00000004 */
#define CAN_TMI0_FF                         CAN_TMI0_FF_Msk                  /*!< Identifier Extension */
#define CAN_TMI0_EFID_Pos                    (3U)                              
#define CAN_TMI0_EFID_Msk                    (0x3FFFFU << CAN_TMI0_EFID_Pos)   /*!< 0x001FFFF8 */
#define CAN_TMI0_EFID                        CAN_TMI0_EFID_Msk                 /*!< Extended Identifier */
#define CAN_TMI0_SFID_Pos                    (21U)                             
#define CAN_TMI0_SFID_Msk                    (0x7FFU << CAN_TMI0_SFID_Pos)     /*!< 0xFFE00000 */
#define CAN_TMI0_SFID                        CAN_TMI0_SFID_Msk                 /*!< Standard Identifier or Extended Identifier */

/******************  Bit definition for CAN_TMP0 register  *******************/
#define CAN_TMP0_DLENC_Pos                    (0U)                              
#define CAN_TMP0_DLENC_Msk                    (0xFU << CAN_TMP0_DLENC_Pos)       /*!< 0x0000000F */
#define CAN_TMP0_DLENC                        CAN_TMP0_DLENC_Msk                 /*!< Data Length Code */
#define CAN_TMP0_TSEN_Pos                    (8U)                              
#define CAN_TMP0_TSEN_Msk                    (0x1U << CAN_TMP0_TSEN_Pos)       /*!< 0x00000100 */
#define CAN_TMP0_TSEN                        CAN_TMP0_TSEN_Msk                 /*!< Transmit Global Time */
#define CAN_TMP0_TS_Pos                   (16U)                             
#define CAN_TMP0_TS_Msk                   (0xFFFFU << CAN_TMP0_TS_Pos)   /*!< 0xFFFF0000 */
#define CAN_TMP0_TS                       CAN_TMP0_TS_Msk                /*!< Message Time Stamp */

/******************  Bit definition for CAN_TMDATA00 register  *******************/
#define CAN_TMDATA00_DB0_Pos                  (0U)                              
#define CAN_TMDATA00_DB0_Msk                  (0xFFU << CAN_TMDATA00_DB0_Pos)    /*!< 0x000000FF */
#define CAN_TMDATA00_DB0                      CAN_TMDATA00_DB0_Msk               /*!< Data byte 0 */
#define CAN_TMDATA00_DB1_Pos                  (8U)                              
#define CAN_TMDATA00_DB1_Msk                  (0xFFU << CAN_TMDATA00_DB1_Pos)    /*!< 0x0000FF00 */
#define CAN_TMDATA00_DB1                      CAN_TMDATA00_DB1_Msk               /*!< Data byte 1 */
#define CAN_TMDATA00_DB2_Pos                  (16U)                             
#define CAN_TMDATA00_DB2_Msk                  (0xFFU << CAN_TMDATA00_DB2_Pos)    /*!< 0x00FF0000 */
#define CAN_TMDATA00_DB2                      CAN_TMDATA00_DB2_Msk               /*!< Data byte 2 */
#define CAN_TMDATA00_DB3_Pos                  (24U)                             
#define CAN_TMDATA00_DB3_Msk                  (0xFFU << CAN_TMDATA00_DB3_Pos)    /*!< 0xFF000000 */
#define CAN_TMDATA00_DB3                      CAN_TMDATA00_DB3_Msk               /*!< Data byte 3 */

/******************  Bit definition for CAN_TMDATA10 register  *******************/
#define CAN_TMDATA10_DB4_Pos                  (0U)                              
#define CAN_TMDATA10_DB4_Msk                  (0xFFU << CAN_TMDATA10_DB4_Pos)    /*!< 0x000000FF */
#define CAN_TMDATA10_DB4                      CAN_TMDATA10_DB4_Msk               /*!< Data byte 4 */
#define CAN_TMDATA10_DB5_Pos                  (8U)                              
#define CAN_TMDATA10_DB5_Msk                  (0xFFU << CAN_TMDATA10_DB5_Pos)    /*!< 0x0000FF00 */
#define CAN_TMDATA10_DB5                      CAN_TMDATA10_DB5_Msk               /*!< Data byte 5 */
#define CAN_TMDATA10_DB6_Pos                  (16U)                             
#define CAN_TMDATA10_DB6_Msk                  (0xFFU << CAN_TMDATA10_DB6_Pos)    /*!< 0x00FF0000 */
#define CAN_TMDATA10_DB6                      CAN_TMDATA10_DB6_Msk               /*!< Data byte 6 */
#define CAN_TMDATA10_DB7_Pos                  (24U)                             
#define CAN_TMDATA10_DB7_Msk                  (0xFFU << CAN_TMDATA10_DB7_Pos)    /*!< 0xFF000000 */
#define CAN_TMDATA10_DB7                      CAN_TMDATA10_DB7_Msk               /*!< Data byte 7 */

/*******************  Bit definition for CAN_TMI0 register  *******************/
#define CAN_TMI0_TEN_Pos                    (0U)                              
#define CAN_TMI0_TEN_Msk                    (0x1U << CAN_TMI0_TEN_Pos)       /*!< 0x00000001 */
#define CAN_TMI0_TEN                        CAN_TMI0_TEN_Msk                 /*!< Transmit Mailbox Request */
#define CAN_TMI0_FT_Pos                     (1U)                              
#define CAN_TMI0_FT_Msk                     (0x1U << CAN_TMI0_FT_Pos)        /*!< 0x00000002 */
#define CAN_TMI0_FT                         CAN_TMI0_FT_Msk                  /*!< Remote Transmission Request */
#define CAN_TMI0_FF_Pos                     (2U)                              
#define CAN_TMI0_FF_Msk                     (0x1U << CAN_TMI0_FF_Pos)        /*!< 0x00000004 */
#define CAN_TMI0_FF                         CAN_TMI0_FF_Msk                  /*!< Identifier Extension */
#define CAN_TMI0_EFID_Pos                    (3U)                              
#define CAN_TMI0_EFID_Msk                    (0x3FFFFU << CAN_TMI0_EFID_Pos)   /*!< 0x001FFFF8 */
#define CAN_TMI0_EFID                        CAN_TMI0_EFID_Msk                 /*!< Extended Identifier */
#define CAN_TMI0_SFID_Pos                    (21U)                             
#define CAN_TMI0_SFID_Msk                    (0x7FFU << CAN_TMI0_SFID_Pos)     /*!< 0xFFE00000 */
#define CAN_TMI0_SFID                        CAN_TMI0_SFID_Msk                 /*!< Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_TMP1 register  ******************/
#define CAN_TMP1_DLENC_Pos                    (0U)                              
#define CAN_TMP1_DLENC_Msk                    (0xFU << CAN_TMP1_DLENC_Pos)       /*!< 0x0000000F */
#define CAN_TMP1_DLENC                        CAN_TMP1_DLENC_Msk                 /*!< Data Length Code */
#define CAN_TMP1_TSEN_Pos                    (8U)                              
#define CAN_TMP1_TSEN_Msk                    (0x1U << CAN_TMP1_TSEN_Pos)       /*!< 0x00000100 */
#define CAN_TMP1_TSEN                        CAN_TMP1_TSEN_Msk                 /*!< Transmit Global Time */
#define CAN_TMP1_TS_Pos                   (16U)                             
#define CAN_TMP1_TS_Msk                   (0xFFFFU << CAN_TMP1_TS_Pos)   /*!< 0xFFFF0000 */
#define CAN_TMP1_TS                       CAN_TMP1_TS_Msk                /*!< Message Time Stamp */

/*******************  Bit definition for CAN_TMDATA00 register  ******************/
#define CAN_TMDATA00_DB0_Pos                  (0U)                              
#define CAN_TMDATA00_DB0_Msk                  (0xFFU << CAN_TMDATA00_DB0_Pos)    /*!< 0x000000FF */
#define CAN_TMDATA00_DB0                      CAN_TMDATA00_DB0_Msk               /*!< Data byte 0 */
#define CAN_TMDATA00_DB1_Pos                  (8U)                              
#define CAN_TMDATA00_DB1_Msk                  (0xFFU << CAN_TMDATA00_DB1_Pos)    /*!< 0x0000FF00 */
#define CAN_TMDATA00_DB1                      CAN_TMDATA00_DB1_Msk               /*!< Data byte 1 */
#define CAN_TMDATA00_DB2_Pos                  (16U)                             
#define CAN_TMDATA00_DB2_Msk                  (0xFFU << CAN_TMDATA00_DB2_Pos)    /*!< 0x00FF0000 */
#define CAN_TMDATA00_DB2                      CAN_TMDATA00_DB2_Msk               /*!< Data byte 2 */
#define CAN_TMDATA00_DB3_Pos                  (24U)                             
#define CAN_TMDATA00_DB3_Msk                  (0xFFU << CAN_TMDATA00_DB3_Pos)    /*!< 0xFF000000 */
#define CAN_TMDATA00_DB3                      CAN_TMDATA00_DB3_Msk               /*!< Data byte 3 */

/*******************  Bit definition for CAN_TMDATA11 register  ******************/
#define CAN_TMDATA11_DB4_Pos                  (0U)                              
#define CAN_TMDATA11_DB4_Msk                  (0xFFU << CAN_TMDATA11_DB4_Pos)    /*!< 0x000000FF */
#define CAN_TMDATA11_DB4                      CAN_TMDATA11_DB4_Msk               /*!< Data byte 4 */
#define CAN_TMDATA11_DB5_Pos                  (8U)                              
#define CAN_TMDATA11_DB5_Msk                  (0xFFU << CAN_TMDATA11_DB5_Pos)    /*!< 0x0000FF00 */
#define CAN_TMDATA11_DB5                      CAN_TMDATA11_DB5_Msk               /*!< Data byte 5 */
#define CAN_TMDATA11_DB6_Pos                  (16U)                             
#define CAN_TMDATA11_DB6_Msk                  (0xFFU << CAN_TMDATA11_DB6_Pos)    /*!< 0x00FF0000 */
#define CAN_TMDATA11_DB6                      CAN_TMDATA11_DB6_Msk               /*!< Data byte 6 */
#define CAN_TMDATA11_DB7_Pos                  (24U)                             
#define CAN_TMDATA11_DB7_Msk                  (0xFFU << CAN_TMDATA11_DB7_Pos)    /*!< 0xFF000000 */
#define CAN_TMDATA11_DB7                      CAN_TMDATA11_DB7_Msk               /*!< Data byte 7 */

/*******************  Bit definition for CAN_TMI2 register  *******************/
#define CAN_TMI2_TEN_Pos                    (0U)                              
#define CAN_TMI2_TEN_Msk                    (0x1U << CAN_TMI2_TEN_Pos)       /*!< 0x00000001 */
#define CAN_TMI2_TEN                        CAN_TMI2_TEN_Msk                 /*!< Transmit Mailbox Request */
#define CAN_TMI2_FT_Pos                     (1U)                              
#define CAN_TMI2_FT_Msk                     (0x1U << CAN_TMI2_FT_Pos)        /*!< 0x00000002 */
#define CAN_TMI2_FT                         CAN_TMI2_FT_Msk                  /*!< Remote Transmission Request */
#define CAN_TMI2_FF_Pos                     (2U)                              
#define CAN_TMI2_FF_Msk                     (0x1U << CAN_TMI2_FF_Pos)        /*!< 0x00000004 */
#define CAN_TMI2_FF                         CAN_TMI2_FF_Msk                  /*!< Identifier Extension */
#define CAN_TMI2_EFID_Pos                    (3U)                              
#define CAN_TMI2_EFID_Msk                    (0x3FFFFU << CAN_TMI2_EFID_Pos)   /*!< 0x001FFFF8 */
#define CAN_TMI2_EFID                        CAN_TMI2_EFID_Msk                 /*!< Extended identifier */
#define CAN_TMI2_SFID_Pos                    (21U)                             
#define CAN_TMI2_SFID_Msk                    (0x7FFU << CAN_TMI2_SFID_Pos)     /*!< 0xFFE00000 */
#define CAN_TMI2_SFID                        CAN_TMI2_SFID_Msk                 /*!< Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_TMP2 register  ******************/  
#define CAN_TMP2_DLENC_Pos                    (0U)                              
#define CAN_TMP2_DLENC_Msk                    (0xFU << CAN_TMP2_DLENC_Pos)       /*!< 0x0000000F */
#define CAN_TMP2_DLENC                        CAN_TMP2_DLENC_Msk                 /*!< Data Length Code */
#define CAN_TMP2_TSEN_Pos                    (8U)                              
#define CAN_TMP2_TSEN_Msk                    (0x1U << CAN_TMP2_TSEN_Pos)       /*!< 0x00000100 */
#define CAN_TMP2_TSEN                        CAN_TMP2_TSEN_Msk                 /*!< Transmit Global Time */
#define CAN_TMP2_TS_Pos                   (16U)                             
#define CAN_TMP2_TS_Msk                   (0xFFFFU << CAN_TMP2_TS_Pos)   /*!< 0xFFFF0000 */
#define CAN_TMP2_TS                       CAN_TMP2_TS_Msk                /*!< Message Time Stamp */

/*******************  Bit definition for CAN_TMDATA02 register  ******************/
#define CAN_TMDATA02_DB0_Pos                  (0U)                              
#define CAN_TMDATA02_DB0_Msk                  (0xFFU << CAN_TMDATA02_DB0_Pos)    /*!< 0x000000FF */
#define CAN_TMDATA02_DB0                      CAN_TMDATA02_DB0_Msk               /*!< Data byte 0 */
#define CAN_TMDATA02_DB1_Pos                  (8U)                              
#define CAN_TMDATA02_DB1_Msk                  (0xFFU << CAN_TMDATA02_DB1_Pos)    /*!< 0x0000FF00 */
#define CAN_TMDATA02_DB1                      CAN_TMDATA02_DB1_Msk               /*!< Data byte 1 */
#define CAN_TMDATA02_DB2_Pos                  (16U)                             
#define CAN_TMDATA02_DB2_Msk                  (0xFFU << CAN_TMDATA02_DB2_Pos)    /*!< 0x00FF0000 */
#define CAN_TMDATA02_DB2                      CAN_TMDATA02_DB2_Msk               /*!< Data byte 2 */
#define CAN_TMDATA02_DB3_Pos                  (24U)                             
#define CAN_TMDATA02_DB3_Msk                  (0xFFU << CAN_TMDATA02_DB3_Pos)    /*!< 0xFF000000 */
#define CAN_TMDATA02_DB3                      CAN_TMDATA02_DB3_Msk               /*!< Data byte 3 */

/*******************  Bit definition for CAN_TMDATA12 register  ******************/
#define CAN_TMDATA12_DB4_Pos                  (0U)                              
#define CAN_TMDATA12_DB4_Msk                  (0xFFU << CAN_TMDATA12_DB4_Pos)    /*!< 0x000000FF */
#define CAN_TMDATA12_DB4                      CAN_TMDATA12_DB4_Msk               /*!< Data byte 4 */
#define CAN_TMDATA12_DB5_Pos                  (8U)                              
#define CAN_TMDATA12_DB5_Msk                  (0xFFU << CAN_TMDATA12_DB5_Pos)    /*!< 0x0000FF00 */
#define CAN_TMDATA12_DB5                      CAN_TMDATA12_DB5_Msk               /*!< Data byte 5 */
#define CAN_TMDATA12_DB6_Pos                  (16U)                             
#define CAN_TMDATA12_DB6_Msk                  (0xFFU << CAN_TMDATA12_DB6_Pos)    /*!< 0x00FF0000 */
#define CAN_TMDATA12_DB6                      CAN_TMDATA12_DB6_Msk               /*!< Data byte 6 */
#define CAN_TMDATA12_DB7_Pos                  (24U)                             
#define CAN_TMDATA12_DB7_Msk                  (0xFFU << CAN_TMDATA12_DB7_Pos)    /*!< 0xFF000000 */
#define CAN_TMDATA12_DB7                      CAN_TMDATA12_DB7_Msk               /*!< Data byte 7 */

/*******************  Bit definition for CAN_RFIFOMI0 register  *******************/
#define CAN_RFIFOMI0_FT_Pos                     (1U)                              
#define CAN_RFIFOMI0_FT_Msk                     (0x1U << CAN_RFIFOMI0_FT_Pos)        /*!< 0x00000002 */
#define CAN_RFIFOMI0_FT                         CAN_RFIFOMI0_FT_Msk                  /*!< Remote Transmission Request */
#define CAN_RFIFOMI0_FF_Pos                     (2U)                              
#define CAN_RFIFOMI0_FF_Msk                     (0x1U << CAN_RFIFOMI0_FF_Pos)        /*!< 0x00000004 */
#define CAN_RFIFOMI0_FF                         CAN_RFIFOMI0_FF_Msk                  /*!< Identifier Extension */
#define CAN_RFIFOMI0_EFID_Pos                    (3U)                              
#define CAN_RFIFOMI0_EFID_Msk                    (0x3FFFFU << CAN_RFIFOMI0_EFID_Pos)   /*!< 0x001FFFF8 */
#define CAN_RFIFOMI0_EFID                        CAN_RFIFOMI0_EFID_Msk                 /*!< Extended Identifier */
#define CAN_RFIFOMI0_SFID_Pos                    (21U)                             
#define CAN_RFIFOMI0_SFID_Msk                    (0x7FFU << CAN_RFIFOMI0_SFID_Pos)     /*!< 0xFFE00000 */
#define CAN_RFIFOMI0_SFID                        CAN_RFIFOMI0_SFID_Msk                 /*!< Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_RFIFOMP0 register  ******************/
#define CAN_RFIFOMP0_DLENC_Pos                    (0U)                              
#define CAN_RFIFOMP0_DLENC_Msk                    (0xFU << CAN_RFIFOMP0_DLENC_Pos)       /*!< 0x0000000F */
#define CAN_RFIFOMP0_DLENC                        CAN_RFIFOMP0_DLENC_Msk                 /*!< Data Length Code */
#define CAN_RFIFOMP0_FI_Pos                    (8U)                              
#define CAN_RFIFOMP0_FI_Msk                    (0xFFU << CAN_RFIFOMP0_FI_Pos)      /*!< 0x0000FF00 */
#define CAN_RFIFOMP0_FI                        CAN_RFIFOMP0_FI_Msk                 /*!< Filter Match Index */
#define CAN_RFIFOMP0_TS_Pos                   (16U)                             
#define CAN_RFIFOMP0_TS_Msk                   (0xFFFFU << CAN_RFIFOMP0_TS_Pos)   /*!< 0xFFFF0000 */
#define CAN_RFIFOMP0_TS                       CAN_RFIFOMP0_TS_Msk                /*!< Message Time Stamp */

/*******************  Bit definition for CAN_RFIFOMDATA00 register  ******************/
#define CAN_RFIFOMDATA00_DB0_Pos                  (0U)                              
#define CAN_RFIFOMDATA00_DB0_Msk                  (0xFFU << CAN_RFIFOMDATA00_DB0_Pos)    /*!< 0x000000FF */
#define CAN_RFIFOMDATA00_DB0                      CAN_RFIFOMDATA00_DB0_Msk               /*!< Data byte 0 */
#define CAN_RFIFOMDATA00_DB1_Pos                  (8U)                              
#define CAN_RFIFOMDATA00_DB1_Msk                  (0xFFU << CAN_RFIFOMDATA00_DB1_Pos)    /*!< 0x0000FF00 */
#define CAN_RFIFOMDATA00_DB1                      CAN_RFIFOMDATA00_DB1_Msk               /*!< Data byte 1 */
#define CAN_RFIFOMDATA00_DB2_Pos                  (16U)                             
#define CAN_RFIFOMDATA00_DB2_Msk                  (0xFFU << CAN_RFIFOMDATA00_DB2_Pos)    /*!< 0x00FF0000 */
#define CAN_RFIFOMDATA00_DB2                      CAN_RFIFOMDATA00_DB2_Msk               /*!< Data byte 2 */
#define CAN_RFIFOMDATA00_DB3_Pos                  (24U)                             
#define CAN_RFIFOMDATA00_DB3_Msk                  (0xFFU << CAN_RFIFOMDATA00_DB3_Pos)    /*!< 0xFF000000 */
#define CAN_RFIFOMDATA00_DB3                      CAN_RFIFOMDATA00_DB3_Msk               /*!< Data byte 3 */

/*******************  Bit definition for CAN_RFIFOMDATA10 register  ******************/
#define CAN_RFIFOMDATA10_DB4_Pos                  (0U)                              
#define CAN_RFIFOMDATA10_DB4_Msk                  (0xFFU << CAN_RFIFOMDATA10_DB4_Pos)    /*!< 0x000000FF */
#define CAN_RFIFOMDATA10_DB4                      CAN_RFIFOMDATA10_DB4_Msk               /*!< Data byte 4 */
#define CAN_RFIFOMDATA10_DB5_Pos                  (8U)                              
#define CAN_RFIFOMDATA10_DB5_Msk                  (0xFFU << CAN_RFIFOMDATA10_DB5_Pos)    /*!< 0x0000FF00 */
#define CAN_RFIFOMDATA10_DB5                      CAN_RFIFOMDATA10_DB5_Msk               /*!< Data byte 5 */
#define CAN_RFIFOMDATA10_DB6_Pos                  (16U)                             
#define CAN_RFIFOMDATA10_DB6_Msk                  (0xFFU << CAN_RFIFOMDATA10_DB6_Pos)    /*!< 0x00FF0000 */
#define CAN_RFIFOMDATA10_DB6                      CAN_RFIFOMDATA10_DB6_Msk               /*!< Data byte 6 */
#define CAN_RFIFOMDATA10_DB7_Pos                  (24U)                             
#define CAN_RFIFOMDATA10_DB7_Msk                  (0xFFU << CAN_RFIFOMDATA10_DB7_Pos)    /*!< 0xFF000000 */
#define CAN_RFIFOMDATA10_DB7                      CAN_RFIFOMDATA10_DB7_Msk               /*!< Data byte 7 */

/*******************  Bit definition for CAN_RFIFOMI1 register  *******************/
#define CAN_RFIFOMI1_FT_Pos                     (1U)                              
#define CAN_RFIFOMI1_FT_Msk                     (0x1U << CAN_RFIFOMI1_FT_Pos)        /*!< 0x00000002 */
#define CAN_RFIFOMI1_FT                         CAN_RFIFOMI1_FT_Msk                  /*!< Remote Transmission Request */
#define CAN_RFIFOMI1_FF_Pos                     (2U)                              
#define CAN_RFIFOMI1_FF_Msk                     (0x1U << CAN_RFIFOMI1_FF_Pos)        /*!< 0x00000004 */
#define CAN_RFIFOMI1_FF                         CAN_RFIFOMI1_FF_Msk                  /*!< Identifier Extension */
#define CAN_RFIFOMI1_EFID_Pos                    (3U)                              
#define CAN_RFIFOMI1_EFID_Msk                    (0x3FFFFU << CAN_RFIFOMI1_EFID_Pos)   /*!< 0x001FFFF8 */
#define CAN_RFIFOMI1_EFID                        CAN_RFIFOMI1_EFID_Msk                 /*!< Extended identifier */
#define CAN_RFIFOMI1_SFID_Pos                    (21U)                             
#define CAN_RFIFOMI1_SFID_Msk                    (0x7FFU << CAN_RFIFOMI1_SFID_Pos)     /*!< 0xFFE00000 */
#define CAN_RFIFOMI1_SFID                        CAN_RFIFOMI1_SFID_Msk                 /*!< Standard Identifier or Extended Identifier */

/*******************  Bit definition for CAN_RFIFOMP1 register  ******************/
#define CAN_RFIFOMP1_DLENC_Pos                    (0U)                              
#define CAN_RFIFOMP1_DLENC_Msk                    (0xFU << CAN_RFIFOMP1_DLENC_Pos)       /*!< 0x0000000F */
#define CAN_RFIFOMP1_DLENC                        CAN_RFIFOMP1_DLENC_Msk                 /*!< Data Length Code */
#define CAN_RFIFOMP1_FI_Pos                    (8U)                              
#define CAN_RFIFOMP1_FI_Msk                    (0xFFU << CAN_RFIFOMP1_FI_Pos)      /*!< 0x0000FF00 */
#define CAN_RFIFOMP1_FI                        CAN_RFIFOMP1_FI_Msk                 /*!< Filter Match Index */
#define CAN_RFIFOMP1_TS_Pos                   (16U)                             
#define CAN_RFIFOMP1_TS_Msk                   (0xFFFFU << CAN_RFIFOMP1_TS_Pos)   /*!< 0xFFFF0000 */
#define CAN_RFIFOMP1_TS                       CAN_RFIFOMP1_TS_Msk                /*!< Message Time Stamp */

/*******************  Bit definition for CAN_RFIFOMDATA01 register  ******************/
#define CAN_RFIFOMDATA01_DB0_Pos                  (0U)                              
#define CAN_RFIFOMDATA01_DB0_Msk                  (0xFFU << CAN_RFIFOMDATA01_DB0_Pos)    /*!< 0x000000FF */
#define CAN_RFIFOMDATA01_DB0                      CAN_RFIFOMDATA01_DB0_Msk               /*!< Data byte 0 */
#define CAN_RFIFOMDATA01_DB1_Pos                  (8U)                              
#define CAN_RFIFOMDATA01_DB1_Msk                  (0xFFU << CAN_RFIFOMDATA01_DB1_Pos)    /*!< 0x0000FF00 */
#define CAN_RFIFOMDATA01_DB1                      CAN_RFIFOMDATA01_DB1_Msk               /*!< Data byte 1 */
#define CAN_RFIFOMDATA01_DB2_Pos                  (16U)                             
#define CAN_RFIFOMDATA01_DB2_Msk                  (0xFFU << CAN_RFIFOMDATA01_DB2_Pos)    /*!< 0x00FF0000 */
#define CAN_RFIFOMDATA01_DB2                      CAN_RFIFOMDATA01_DB2_Msk               /*!< Data byte 2 */
#define CAN_RFIFOMDATA01_DB3_Pos                  (24U)                             
#define CAN_RFIFOMDATA01_DB3_Msk                  (0xFFU << CAN_RFIFOMDATA01_DB3_Pos)    /*!< 0xFF000000 */
#define CAN_RFIFOMDATA01_DB3                      CAN_RFIFOMDATA01_DB3_Msk               /*!< Data byte 3 */

/*******************  Bit definition for CAN_RFIFOMDATA11 register  ******************/
#define CAN_RFIFOMDATA11_DB4_Pos                  (0U)                              
#define CAN_RFIFOMDATA11_DB4_Msk                  (0xFFU << CAN_RFIFOMDATA11_DB4_Pos)    /*!< 0x000000FF */
#define CAN_RFIFOMDATA11_DB4                      CAN_RFIFOMDATA11_DB4_Msk               /*!< Data byte 4 */
#define CAN_RFIFOMDATA11_DB5_Pos                  (8U)                              
#define CAN_RFIFOMDATA11_DB5_Msk                  (0xFFU << CAN_RFIFOMDATA11_DB5_Pos)    /*!< 0x0000FF00 */
#define CAN_RFIFOMDATA11_DB5                      CAN_RFIFOMDATA11_DB5_Msk               /*!< Data byte 5 */
#define CAN_RFIFOMDATA11_DB6_Pos                  (16U)                             
#define CAN_RFIFOMDATA11_DB6_Msk                  (0xFFU << CAN_RFIFOMDATA11_DB6_Pos)    /*!< 0x00FF0000 */
#define CAN_RFIFOMDATA11_DB6                      CAN_RFIFOMDATA11_DB6_Msk               /*!< Data byte 6 */
#define CAN_RFIFOMDATA11_DB7_Pos                  (24U)                             
#define CAN_RFIFOMDATA11_DB7_Msk                  (0xFFU << CAN_RFIFOMDATA11_DB7_Pos)    /*!< 0xFF000000 */
#define CAN_RFIFOMDATA11_DB7                      CAN_RFIFOMDATA11_DB7_Msk               /*!< Data byte 7 */

/*!< CAN filter registers */
/*******************  Bit definition for CAN_FCTL register  ********************/
#define CAN_FCTL_FLD_Pos                    (0U)                              
#define CAN_FCTL_FLD_Msk                    (0x1U << CAN_FCTL_FLD_Pos)       /*!< 0x00000001 */
#define CAN_FCTL_FLD                        CAN_FCTL_FLD_Msk                 /*!< Filter Init Mode */
#define CAN_FCTL_HBC1F_Pos                   (8U)                              
#define CAN_FCTL_HBC1F_Msk                   (0x3FU << CAN_FCTL_HBC1F_Pos)     /*!< 0x00003F00 */
#define CAN_FCTL_HBC1F                       CAN_FCTL_HBC1F_Msk                /*!< CAN1 start bank */

/*******************  Bit definition for CAN_FMCFG register  *******************/
#define CAN_FMCFG_FMOD_Pos                     (0U)                              
#define CAN_FMCFG_FMOD_Msk                     (0x3FFFU << CAN_FMCFG_FMOD_Pos)     /*!< 0x00003FFF */
#define CAN_FMCFG_FMOD                         CAN_FMCFG_FMOD_Msk                  /*!< Filter Mode */
#define CAN_FMCFG_FMOD0_Pos                    (0U)                              
#define CAN_FMCFG_FMOD0_Msk                    (0x1U << CAN_FMCFG_FMOD0_Pos)       /*!< 0x00000001 */
#define CAN_FMCFG_FMOD0                        CAN_FMCFG_FMOD0_Msk                 /*!< Filter Init Mode for filter 0 */
#define CAN_FMCFG_FMOD1_Pos                    (1U)                              
#define CAN_FMCFG_FMOD1_Msk                    (0x1U << CAN_FMCFG_FMOD1_Pos)       /*!< 0x00000002 */
#define CAN_FMCFG_FMOD1                        CAN_FMCFG_FMOD1_Msk                 /*!< Filter Init Mode for filter 1 */
#define CAN_FMCFG_FMOD2_Pos                    (2U)                              
#define CAN_FMCFG_FMOD2_Msk                    (0x1U << CAN_FMCFG_FMOD2_Pos)       /*!< 0x00000004 */
#define CAN_FMCFG_FMOD2                        CAN_FMCFG_FMOD2_Msk                 /*!< Filter Init Mode for filter 2 */
#define CAN_FMCFG_FMOD3_Pos                    (3U)                              
#define CAN_FMCFG_FMOD3_Msk                    (0x1U << CAN_FMCFG_FMOD3_Pos)       /*!< 0x00000008 */
#define CAN_FMCFG_FMOD3                        CAN_FMCFG_FMOD3_Msk                 /*!< Filter Init Mode for filter 3 */
#define CAN_FMCFG_FMOD4_Pos                    (4U)                              
#define CAN_FMCFG_FMOD4_Msk                    (0x1U << CAN_FMCFG_FMOD4_Pos)       /*!< 0x00000010 */
#define CAN_FMCFG_FMOD4                        CAN_FMCFG_FMOD4_Msk                 /*!< Filter Init Mode for filter 4 */
#define CAN_FMCFG_FMOD5_Pos                    (5U)                              
#define CAN_FMCFG_FMOD5_Msk                    (0x1U << CAN_FMCFG_FMOD5_Pos)       /*!< 0x00000020 */
#define CAN_FMCFG_FMOD5                        CAN_FMCFG_FMOD5_Msk                 /*!< Filter Init Mode for filter 5 */
#define CAN_FMCFG_FMOD6_Pos                    (6U)                              
#define CAN_FMCFG_FMOD6_Msk                    (0x1U << CAN_FMCFG_FMOD6_Pos)       /*!< 0x00000040 */
#define CAN_FMCFG_FMOD6                        CAN_FMCFG_FMOD6_Msk                 /*!< Filter Init Mode for filter 6 */
#define CAN_FMCFG_FMOD7_Pos                    (7U)                              
#define CAN_FMCFG_FMOD7_Msk                    (0x1U << CAN_FMCFG_FMOD7_Pos)       /*!< 0x00000080 */
#define CAN_FMCFG_FMOD7                        CAN_FMCFG_FMOD7_Msk                 /*!< Filter Init Mode for filter 7 */
#define CAN_FMCFG_FMOD8_Pos                    (8U)                              
#define CAN_FMCFG_FMOD8_Msk                    (0x1U << CAN_FMCFG_FMOD8_Pos)       /*!< 0x00000100 */
#define CAN_FMCFG_FMOD8                        CAN_FMCFG_FMOD8_Msk                 /*!< Filter Init Mode for filter 8 */
#define CAN_FMCFG_FMOD9_Pos                    (9U)                              
#define CAN_FMCFG_FMOD9_Msk                    (0x1U << CAN_FMCFG_FMOD9_Pos)       /*!< 0x00000200 */
#define CAN_FMCFG_FMOD9                        CAN_FMCFG_FMOD9_Msk                 /*!< Filter Init Mode for filter 9 */
#define CAN_FMCFG_FMOD10_Pos                   (10U)                             
#define CAN_FMCFG_FMOD10_Msk                   (0x1U << CAN_FMCFG_FMOD10_Pos)      /*!< 0x00000400 */
#define CAN_FMCFG_FMOD10                       CAN_FMCFG_FMOD10_Msk                /*!< Filter Init Mode for filter 10 */
#define CAN_FMCFG_FMOD11_Pos                   (11U)                             
#define CAN_FMCFG_FMOD11_Msk                   (0x1U << CAN_FMCFG_FMOD11_Pos)      /*!< 0x00000800 */
#define CAN_FMCFG_FMOD11                       CAN_FMCFG_FMOD11_Msk                /*!< Filter Init Mode for filter 11 */
#define CAN_FMCFG_FMOD12_Pos                   (12U)                             
#define CAN_FMCFG_FMOD12_Msk                   (0x1U << CAN_FMCFG_FMOD12_Pos)      /*!< 0x00001000 */
#define CAN_FMCFG_FMOD12                       CAN_FMCFG_FMOD12_Msk                /*!< Filter Init Mode for filter 12 */
#define CAN_FMCFG_FMOD13_Pos                   (13U)                             
#define CAN_FMCFG_FMOD13_Msk                   (0x1U << CAN_FMCFG_FMOD13_Pos)      /*!< 0x00002000 */
#define CAN_FMCFG_FMOD13                       CAN_FMCFG_FMOD13_Msk                /*!< Filter Init Mode for filter 13 */
#define CAN_FMCFG_FMOD14_Pos                   (14U)                             
#define CAN_FMCFG_FMOD14_Msk                   (0x1U << CAN_FMCFG_FMOD14_Pos)      /*!< 0x00004000 */
#define CAN_FMCFG_FMOD14                       CAN_FMCFG_FMOD14_Msk                /*!< Filter Init Mode for filter 14 */
#define CAN_FMCFG_FMOD15_Pos                   (15U)                             
#define CAN_FMCFG_FMOD15_Msk                   (0x1U << CAN_FMCFG_FMOD15_Pos)      /*!< 0x00008000 */
#define CAN_FMCFG_FMOD15                       CAN_FMCFG_FMOD15_Msk                /*!< Filter Init Mode for filter 15 */
#define CAN_FMCFG_FMOD16_Pos                   (16U)                             
#define CAN_FMCFG_FMOD16_Msk                   (0x1U << CAN_FMCFG_FMOD16_Pos)      /*!< 0x00010000 */
#define CAN_FMCFG_FMOD16                       CAN_FMCFG_FMOD16_Msk                /*!< Filter Init Mode for filter 16 */
#define CAN_FMCFG_FMOD17_Pos                   (17U)                             
#define CAN_FMCFG_FMOD17_Msk                   (0x1U << CAN_FMCFG_FMOD17_Pos)      /*!< 0x00020000 */
#define CAN_FMCFG_FMOD17                       CAN_FMCFG_FMOD17_Msk                /*!< Filter Init Mode for filter 17 */
#define CAN_FMCFG_FMOD18_Pos                   (18U)                             
#define CAN_FMCFG_FMOD18_Msk                   (0x1U << CAN_FMCFG_FMOD18_Pos)      /*!< 0x00040000 */
#define CAN_FMCFG_FMOD18                       CAN_FMCFG_FMOD18_Msk                /*!< Filter Init Mode for filter 18 */
#define CAN_FMCFG_FMOD19_Pos                   (19U)                             
#define CAN_FMCFG_FMOD19_Msk                   (0x1U << CAN_FMCFG_FMOD19_Pos)      /*!< 0x00080000 */
#define CAN_FMCFG_FMOD19                       CAN_FMCFG_FMOD19_Msk                /*!< Filter Init Mode for filter 19 */
#define CAN_FMCFG_FMOD20_Pos                   (20U)                             
#define CAN_FMCFG_FMOD20_Msk                   (0x1U << CAN_FMCFG_FMOD20_Pos)      /*!< 0x00100000 */
#define CAN_FMCFG_FMOD20                       CAN_FMCFG_FMOD20_Msk                /*!< Filter Init Mode for filter 20 */
#define CAN_FMCFG_FMOD21_Pos                   (21U)                             
#define CAN_FMCFG_FMOD21_Msk                   (0x1U << CAN_FMCFG_FMOD21_Pos)      /*!< 0x00200000 */
#define CAN_FMCFG_FMOD21                       CAN_FMCFG_FMOD21_Msk                /*!< Filter Init Mode for filter 21 */
#define CAN_FMCFG_FMOD22_Pos                   (22U)                             
#define CAN_FMCFG_FMOD22_Msk                   (0x1U << CAN_FMCFG_FMOD22_Pos)      /*!< 0x00400000 */
#define CAN_FMCFG_FMOD22                       CAN_FMCFG_FMOD22_Msk                /*!< Filter Init Mode for filter 22 */
#define CAN_FMCFG_FMOD23_Pos                   (23U)                             
#define CAN_FMCFG_FMOD23_Msk                   (0x1U << CAN_FMCFG_FMOD23_Pos)      /*!< 0x00800000 */
#define CAN_FMCFG_FMOD23                       CAN_FMCFG_FMOD23_Msk                /*!< Filter Init Mode for filter 23 */
#define CAN_FMCFG_FMOD24_Pos                   (24U)                             
#define CAN_FMCFG_FMOD24_Msk                   (0x1U << CAN_FMCFG_FMOD24_Pos)      /*!< 0x01000000 */
#define CAN_FMCFG_FMOD24                       CAN_FMCFG_FMOD24_Msk                /*!< Filter Init Mode for filter 24 */
#define CAN_FMCFG_FMOD25_Pos                   (25U)                             
#define CAN_FMCFG_FMOD25_Msk                   (0x1U << CAN_FMCFG_FMOD25_Pos)      /*!< 0x02000000 */
#define CAN_FMCFG_FMOD25                       CAN_FMCFG_FMOD25_Msk                /*!< Filter Init Mode for filter 25 */
#define CAN_FMCFG_FMOD26_Pos                   (26U)                             
#define CAN_FMCFG_FMOD26_Msk                   (0x1U << CAN_FMCFG_FMOD26_Pos)      /*!< 0x04000000 */
#define CAN_FMCFG_FMOD26                       CAN_FMCFG_FMOD26_Msk                /*!< Filter Init Mode for filter 26 */
#define CAN_FMCFG_FMOD27_Pos                   (27U)                             
#define CAN_FMCFG_FMOD27_Msk                   (0x1U << CAN_FMCFG_FMOD27_Pos)      /*!< 0x08000000 */
#define CAN_FMCFG_FMOD27                       CAN_FMCFG_FMOD27_Msk                /*!< Filter Init Mode for filter 27 */

/*******************  Bit definition for CAN_FSCFG register  *******************/
#define CAN_FSCFG_FS_Pos                     (0U)                              
#define CAN_FSCFG_FS_Msk                     (0x3FFFU << CAN_FSCFG_FS_Pos)     /*!< 0x00003FFF */
#define CAN_FSCFG_FS                         CAN_FSCFG_FS_Msk                  /*!< Filter Scale Configuration */
#define CAN_FSCFG_FS0_Pos                    (0U)                              
#define CAN_FSCFG_FS0_Msk                    (0x1U << CAN_FSCFG_FS0_Pos)       /*!< 0x00000001 */
#define CAN_FSCFG_FS0                        CAN_FSCFG_FS0_Msk                 /*!< Filter Scale Configuration for filter 0 */
#define CAN_FSCFG_FS1_Pos                    (1U)                              
#define CAN_FSCFG_FS1_Msk                    (0x1U << CAN_FSCFG_FS1_Pos)       /*!< 0x00000002 */
#define CAN_FSCFG_FS1                        CAN_FSCFG_FS1_Msk                 /*!< Filter Scale Configuration for filter 1 */
#define CAN_FSCFG_FS2_Pos                    (2U)                              
#define CAN_FSCFG_FS2_Msk                    (0x1U << CAN_FSCFG_FS2_Pos)       /*!< 0x00000004 */
#define CAN_FSCFG_FS2                        CAN_FSCFG_FS2_Msk                 /*!< Filter Scale Configuration for filter 2 */
#define CAN_FSCFG_FS3_Pos                    (3U)                              
#define CAN_FSCFG_FS3_Msk                    (0x1U << CAN_FSCFG_FS3_Pos)       /*!< 0x00000008 */
#define CAN_FSCFG_FS3                        CAN_FSCFG_FS3_Msk                 /*!< Filter Scale Configuration for filter 3 */
#define CAN_FSCFG_FS4_Pos                    (4U)                              
#define CAN_FSCFG_FS4_Msk                    (0x1U << CAN_FSCFG_FS4_Pos)       /*!< 0x00000010 */
#define CAN_FSCFG_FS4                        CAN_FSCFG_FS4_Msk                 /*!< Filter Scale Configuration for filter 4 */
#define CAN_FSCFG_FS5_Pos                    (5U)                              
#define CAN_FSCFG_FS5_Msk                    (0x1U << CAN_FSCFG_FS5_Pos)       /*!< 0x00000020 */
#define CAN_FSCFG_FS5                        CAN_FSCFG_FS5_Msk                 /*!< Filter Scale Configuration for filter 5 */
#define CAN_FSCFG_FS6_Pos                    (6U)                              
#define CAN_FSCFG_FS6_Msk                    (0x1U << CAN_FSCFG_FS6_Pos)       /*!< 0x00000040 */
#define CAN_FSCFG_FS6                        CAN_FSCFG_FS6_Msk                 /*!< Filter Scale Configuration for filter 6 */
#define CAN_FSCFG_FS7_Pos                    (7U)                              
#define CAN_FSCFG_FS7_Msk                    (0x1U << CAN_FSCFG_FS7_Pos)       /*!< 0x00000080 */
#define CAN_FSCFG_FS7                        CAN_FSCFG_FS7_Msk                 /*!< Filter Scale Configuration for filter 7 */
#define CAN_FSCFG_FS8_Pos                    (8U)                              
#define CAN_FSCFG_FS8_Msk                    (0x1U << CAN_FSCFG_FS8_Pos)       /*!< 0x00000100 */
#define CAN_FSCFG_FS8                        CAN_FSCFG_FS8_Msk                 /*!< Filter Scale Configuration for filter 8 */
#define CAN_FSCFG_FS9_Pos                    (9U)                              
#define CAN_FSCFG_FS9_Msk                    (0x1U << CAN_FSCFG_FS9_Pos)       /*!< 0x00000200 */
#define CAN_FSCFG_FS9                        CAN_FSCFG_FS9_Msk                 /*!< Filter Scale Configuration for filter 9 */
#define CAN_FSCFG_FS10_Pos                   (10U)                             
#define CAN_FSCFG_FS10_Msk                   (0x1U << CAN_FSCFG_FS10_Pos)      /*!< 0x00000400 */
#define CAN_FSCFG_FS10                       CAN_FSCFG_FS10_Msk                /*!< Filter Scale Configuration for filter 10 */
#define CAN_FSCFG_FS11_Pos                   (11U)                             
#define CAN_FSCFG_FS11_Msk                   (0x1U << CAN_FSCFG_FS11_Pos)      /*!< 0x00000800 */
#define CAN_FSCFG_FS11                       CAN_FSCFG_FS11_Msk                /*!< Filter Scale Configuration for filter 11 */
#define CAN_FSCFG_FS12_Pos                   (12U)                             
#define CAN_FSCFG_FS12_Msk                   (0x1U << CAN_FSCFG_FS12_Pos)      /*!< 0x00001000 */
#define CAN_FSCFG_FS12                       CAN_FSCFG_FS12_Msk                /*!< Filter Scale Configuration for filter 12 */
#define CAN_FSCFG_FS13_Pos                   (13U)                             
#define CAN_FSCFG_FS13_Msk                   (0x1U << CAN_FSCFG_FS13_Pos)      /*!< 0x00002000 */
#define CAN_FSCFG_FS13                       CAN_FSCFG_FS13_Msk                /*!< Filter Scale Configuration for filter 13 */
#define CAN_FSCFG_FS14_Pos                   (14U)                             
#define CAN_FSCFG_FS14_Msk                   (0x1U << CAN_FSCFG_FS14_Pos)      /*!< 0x00004000 */
#define CAN_FSCFG_FS14                       CAN_FSCFG_FS14_Msk                /*!< Filter Scale Configuration for filter 14 */
#define CAN_FSCFG_FS15_Pos                   (15U)                             
#define CAN_FSCFG_FS15_Msk                   (0x1U << CAN_FSCFG_FS15_Pos)      /*!< 0x00008000 */
#define CAN_FSCFG_FS15                       CAN_FSCFG_FS15_Msk                /*!< Filter Scale Configuration for filter 15 */
#define CAN_FSCFG_FS16_Pos                   (16U)                             
#define CAN_FSCFG_FS16_Msk                   (0x1U << CAN_FSCFG_FS16_Pos)      /*!< 0x00010000 */
#define CAN_FSCFG_FS16                       CAN_FSCFG_FS16_Msk                /*!< Filter Scale Configuration for filter 16 */
#define CAN_FSCFG_FS17_Pos                   (17U)                             
#define CAN_FSCFG_FS17_Msk                   (0x1U << CAN_FSCFG_FS17_Pos)      /*!< 0x00020000 */
#define CAN_FSCFG_FS17                       CAN_FSCFG_FS17_Msk                /*!< Filter Scale Configuration for filter 17 */
#define CAN_FSCFG_FS18_Pos                   (18U)                             
#define CAN_FSCFG_FS18_Msk                   (0x1U << CAN_FSCFG_FS18_Pos)      /*!< 0x00040000 */
#define CAN_FSCFG_FS18                       CAN_FSCFG_FS18_Msk                /*!< Filter Scale Configuration for filter 18 */
#define CAN_FSCFG_FS19_Pos                   (19U)                             
#define CAN_FSCFG_FS19_Msk                   (0x1U << CAN_FSCFG_FS19_Pos)      /*!< 0x00080000 */
#define CAN_FSCFG_FS19                       CAN_FSCFG_FS19_Msk                /*!< Filter Scale Configuration for filter 19 */
#define CAN_FSCFG_FS20_Pos                   (20U)                             
#define CAN_FSCFG_FS20_Msk                   (0x1U << CAN_FSCFG_FS20_Pos)      /*!< 0x00100000 */
#define CAN_FSCFG_FS20                       CAN_FSCFG_FS20_Msk                /*!< Filter Scale Configuration for filter 20 */
#define CAN_FSCFG_FS21_Pos                   (21U)                             
#define CAN_FSCFG_FS21_Msk                   (0x1U << CAN_FSCFG_FS21_Pos)      /*!< 0x00200000 */
#define CAN_FSCFG_FS21                       CAN_FSCFG_FS21_Msk                /*!< Filter Scale Configuration for filter 21 */
#define CAN_FSCFG_FS22_Pos                   (22U)                             
#define CAN_FSCFG_FS22_Msk                   (0x1U << CAN_FSCFG_FS22_Pos)      /*!< 0x00400000 */
#define CAN_FSCFG_FS22                       CAN_FSCFG_FS22_Msk                /*!< Filter Scale Configuration for filter 22 */
#define CAN_FSCFG_FS23_Pos                   (23U)                             
#define CAN_FSCFG_FS23_Msk                   (0x1U << CAN_FSCFG_FS23_Pos)      /*!< 0x00800000 */
#define CAN_FSCFG_FS23                       CAN_FSCFG_FS23_Msk                /*!< Filter Scale Configuration for filter 23 */
#define CAN_FSCFG_FS24_Pos                   (24U)                             
#define CAN_FSCFG_FS24_Msk                   (0x1U << CAN_FSCFG_FS24_Pos)      /*!< 0x01000000 */
#define CAN_FSCFG_FS24                       CAN_FSCFG_FS24_Msk                /*!< Filter Scale Configuration for filter 24 */
#define CAN_FSCFG_FS25_Pos                   (25U)                             
#define CAN_FSCFG_FS25_Msk                   (0x1U << CAN_FSCFG_FS25_Pos)      /*!< 0x02000000 */
#define CAN_FSCFG_FS25                       CAN_FSCFG_FS25_Msk                /*!< Filter Scale Configuration for filter 25 */
#define CAN_FSCFG_FS26_Pos                   (26U)                             
#define CAN_FSCFG_FS26_Msk                   (0x1U << CAN_FSCFG_FS26_Pos)      /*!< 0x04000000 */
#define CAN_FSCFG_FS26                       CAN_FSCFG_FS26_Msk                /*!< Filter Scale Configuration for filter 26 */
#define CAN_FSCFG_FS27_Pos                   (27U)                             
#define CAN_FSCFG_FS27_Msk                   (0x1U << CAN_FSCFG_FS27_Pos)      /*!< 0x08000000 */
#define CAN_FSCFG_FS27                       CAN_FSCFG_FS27_Msk                /*!< Filter Scale Configuration for filter 27 */

/******************  Bit definition for CAN_FAFIFO register  *******************/
#define CAN_FAFIFO_FAF_Pos                    (0U)                              
#define CAN_FAFIFO_FAF_Msk                    (0x3FFFU << CAN_FAFIFO_FAF_Pos)    /*!< 0x00003FFF */
#define CAN_FAFIFO_FAF                        CAN_FAFIFO_FAF_Msk                 /*!< Filter FIFO Assignment */
#define CAN_FAFIFO_FAF0_Pos                   (0U)                              
#define CAN_FAFIFO_FAF0_Msk                   (0x1U << CAN_FAFIFO_FAF0_Pos)      /*!< 0x00000001 */
#define CAN_FAFIFO_FAF0                       CAN_FAFIFO_FAF0_Msk                /*!< Filter FIFO Assignment for filter 0 */
#define CAN_FAFIFO_FAF1_Pos                   (1U)                              
#define CAN_FAFIFO_FAF1_Msk                   (0x1U << CAN_FAFIFO_FAF1_Pos)      /*!< 0x00000002 */
#define CAN_FAFIFO_FAF1                       CAN_FAFIFO_FAF1_Msk                /*!< Filter FIFO Assignment for filter 1 */
#define CAN_FAFIFO_FAF2_Pos                   (2U)                              
#define CAN_FAFIFO_FAF2_Msk                   (0x1U << CAN_FAFIFO_FAF2_Pos)      /*!< 0x00000004 */
#define CAN_FAFIFO_FAF2                       CAN_FAFIFO_FAF2_Msk                /*!< Filter FIFO Assignment for filter 2 */
#define CAN_FAFIFO_FAF3_Pos                   (3U)                              
#define CAN_FAFIFO_FAF3_Msk                   (0x1U << CAN_FAFIFO_FAF3_Pos)      /*!< 0x00000008 */
#define CAN_FAFIFO_FAF3                       CAN_FAFIFO_FAF3_Msk                /*!< Filter FIFO Assignment for filter 3 */
#define CAN_FAFIFO_FAF4_Pos                   (4U)                              
#define CAN_FAFIFO_FAF4_Msk                   (0x1U << CAN_FAFIFO_FAF4_Pos)      /*!< 0x00000010 */
#define CAN_FAFIFO_FAF4                       CAN_FAFIFO_FAF4_Msk                /*!< Filter FIFO Assignment for filter 4 */
#define CAN_FAFIFO_FAF5_Pos                   (5U)                              
#define CAN_FAFIFO_FAF5_Msk                   (0x1U << CAN_FAFIFO_FAF5_Pos)      /*!< 0x00000020 */
#define CAN_FAFIFO_FAF5                       CAN_FAFIFO_FAF5_Msk                /*!< Filter FIFO Assignment for filter 5 */
#define CAN_FAFIFO_FAF6_Pos                   (6U)                              
#define CAN_FAFIFO_FAF6_Msk                   (0x1U << CAN_FAFIFO_FAF6_Pos)      /*!< 0x00000040 */
#define CAN_FAFIFO_FAF6                       CAN_FAFIFO_FAF6_Msk                /*!< Filter FIFO Assignment for filter 6 */
#define CAN_FAFIFO_FAF7_Pos                   (7U)                              
#define CAN_FAFIFO_FAF7_Msk                   (0x1U << CAN_FAFIFO_FAF7_Pos)      /*!< 0x00000080 */
#define CAN_FAFIFO_FAF7                       CAN_FAFIFO_FAF7_Msk                /*!< Filter FIFO Assignment for filter 7 */
#define CAN_FAFIFO_FAF8_Pos                   (8U)                              
#define CAN_FAFIFO_FAF8_Msk                   (0x1U << CAN_FAFIFO_FAF8_Pos)      /*!< 0x00000100 */
#define CAN_FAFIFO_FAF8                       CAN_FAFIFO_FAF8_Msk                /*!< Filter FIFO Assignment for filter 8 */
#define CAN_FAFIFO_FAF9_Pos                   (9U)                              
#define CAN_FAFIFO_FAF9_Msk                   (0x1U << CAN_FAFIFO_FAF9_Pos)      /*!< 0x00000200 */
#define CAN_FAFIFO_FAF9                       CAN_FAFIFO_FAF9_Msk                /*!< Filter FIFO Assignment for filter 9 */
#define CAN_FAFIFO_FAF10_Pos                  (10U)                             
#define CAN_FAFIFO_FAF10_Msk                  (0x1U << CAN_FAFIFO_FAF10_Pos)     /*!< 0x00000400 */
#define CAN_FAFIFO_FAF10                      CAN_FAFIFO_FAF10_Msk               /*!< Filter FIFO Assignment for filter 10 */
#define CAN_FAFIFO_FAF11_Pos                  (11U)                             
#define CAN_FAFIFO_FAF11_Msk                  (0x1U << CAN_FAFIFO_FAF11_Pos)     /*!< 0x00000800 */
#define CAN_FAFIFO_FAF11                      CAN_FAFIFO_FAF11_Msk               /*!< Filter FIFO Assignment for filter 11 */
#define CAN_FAFIFO_FAF12_Pos                  (12U)                             
#define CAN_FAFIFO_FAF12_Msk                  (0x1U << CAN_FAFIFO_FAF12_Pos)     /*!< 0x00001000 */
#define CAN_FAFIFO_FAF12                      CAN_FAFIFO_FAF12_Msk               /*!< Filter FIFO Assignment for filter 12 */
#define CAN_FAFIFO_FAF13_Pos                  (13U)                             
#define CAN_FAFIFO_FAF13_Msk                  (0x1U << CAN_FAFIFO_FAF13_Pos)     /*!< 0x00002000 */
#define CAN_FAFIFO_FAF13                      CAN_FAFIFO_FAF13_Msk               /*!< Filter FIFO Assignment for filter 13 */
#define CAN_FAFIFO_FAF14_Pos                   (14U)                             
#define CAN_FAFIFO_FAF14_Msk                   (0x1U << CAN_FAFIFO_FAF14_Pos)      /*!< 0x00004000 */
#define CAN_FAFIFO_FAF14                       CAN_FAFIFO_FAF14_Msk                /*!< Filter FIFO Assignment for filter 14 */
#define CAN_FAFIFO_FAF15_Pos                   (15U)                             
#define CAN_FAFIFO_FAF15_Msk                   (0x1U << CAN_FAFIFO_FAF15_Pos)      /*!< 0x00008000 */
#define CAN_FAFIFO_FAF15                       CAN_FAFIFO_FAF15_Msk                /*!< Filter FIFO Assignment for filter 15 */
#define CAN_FAFIFO_FAF16_Pos                   (16U)                             
#define CAN_FAFIFO_FAF16_Msk                   (0x1U << CAN_FAFIFO_FAF16_Pos)      /*!< 0x00010000 */
#define CAN_FAFIFO_FAF16                       CAN_FAFIFO_FAF16_Msk                /*!< Filter FIFO Assignment for filter 16 */
#define CAN_FAFIFO_FAF17_Pos                   (17U)                             
#define CAN_FAFIFO_FAF17_Msk                   (0x1U << CAN_FAFIFO_FAF17_Pos)      /*!< 0x00020000 */
#define CAN_FAFIFO_FAF17                       CAN_FAFIFO_FAF17_Msk                /*!< Filter FIFO Assignment for filter 17 */
#define CAN_FAFIFO_FAF18_Pos                   (18U)                             
#define CAN_FAFIFO_FAF18_Msk                   (0x1U << CAN_FAFIFO_FAF18_Pos)      /*!< 0x00040000 */
#define CAN_FAFIFO_FAF18                       CAN_FAFIFO_FAF18_Msk                /*!< Filter FIFO Assignment for filter 18 */
#define CAN_FAFIFO_FAF19_Pos                   (19U)                             
#define CAN_FAFIFO_FAF19_Msk                   (0x1U << CAN_FAFIFO_FAF19_Pos)      /*!< 0x00080000 */
#define CAN_FAFIFO_FAF19                       CAN_FAFIFO_FAF19_Msk                /*!< Filter FIFO Assignment for filter 19 */
#define CAN_FAFIFO_FAF20_Pos                   (20U)                             
#define CAN_FAFIFO_FAF20_Msk                   (0x1U << CAN_FAFIFO_FAF20_Pos)      /*!< 0x00100000 */
#define CAN_FAFIFO_FAF20                       CAN_FAFIFO_FAF20_Msk                /*!< Filter FIFO Assignment for filter 20 */
#define CAN_FAFIFO_FAF21_Pos                   (21U)                             
#define CAN_FAFIFO_FAF21_Msk                   (0x1U << CAN_FAFIFO_FAF21_Pos)      /*!< 0x00200000 */
#define CAN_FAFIFO_FAF21                       CAN_FAFIFO_FAF21_Msk                /*!< Filter FIFO Assignment for filter 21 */
#define CAN_FAFIFO_FAF22_Pos                   (22U)                             
#define CAN_FAFIFO_FAF22_Msk                   (0x1U << CAN_FAFIFO_FAF22_Pos)      /*!< 0x00400000 */
#define CAN_FAFIFO_FAF22                       CAN_FAFIFO_FAF22_Msk                /*!< Filter FIFO Assignment for filter 22 */
#define CAN_FAFIFO_FAF23_Pos                   (23U)                             
#define CAN_FAFIFO_FAF23_Msk                   (0x1U << CAN_FAFIFO_FAF23_Pos)      /*!< 0x00800000 */
#define CAN_FAFIFO_FAF23                       CAN_FAFIFO_FAF23_Msk                /*!< Filter FIFO Assignment for filter 23 */
#define CAN_FAFIFO_FAF24_Pos                   (24U)                             
#define CAN_FAFIFO_FAF24_Msk                   (0x1U << CAN_FAFIFO_FAF24_Pos)      /*!< 0x01000000 */
#define CAN_FAFIFO_FAF24                       CAN_FAFIFO_FAF24_Msk                /*!< Filter FIFO Assignment for filter 24 */
#define CAN_FAFIFO_FAF25_Pos                   (25U)                             
#define CAN_FAFIFO_FAF25_Msk                   (0x1U << CAN_FAFIFO_FAF25_Pos)      /*!< 0x02000000 */
#define CAN_FAFIFO_FAF25                       CAN_FAFIFO_FAF25_Msk                /*!< Filter FIFO Assignment for filter 25 */
#define CAN_FAFIFO_FAF26_Pos                   (26U)                             
#define CAN_FAFIFO_FAF26_Msk                   (0x1U << CAN_FAFIFO_FAF26_Pos)      /*!< 0x04000000 */
#define CAN_FAFIFO_FAF26                       CAN_FAFIFO_FAF26_Msk                /*!< Filter FIFO Assignment for filter 26 */
#define CAN_FAFIFO_FAF27_Pos                   (27U)                             
#define CAN_FAFIFO_FAF27_Msk                   (0x1U << CAN_FAFIFO_FAF27_Pos)      /*!< 0x08000000 */
#define CAN_FAFIFO_FAF27                       CAN_FAFIFO_FAF27_Msk                /*!< Filter FIFO Assignment for filter 27 */

/*******************  Bit definition for CAN_FW register  *******************/
#define CAN_FW_FW_Pos                    (0U)                              
#define CAN_FW_FW_Msk                    (0x3FFFU << CAN_FW_FW_Pos)    /*!< 0x00003FFF */
#define CAN_FW_FW                        CAN_FW_FW_Msk                 /*!< Filter Active */
#define CAN_FW_FW0_Pos                   (0U)                              
#define CAN_FW_FW0_Msk                   (0x1U << CAN_FW_FW0_Pos)      /*!< 0x00000001 */
#define CAN_FW_FW0                       CAN_FW_FW0_Msk                /*!< Filter 0 Active */
#define CAN_FW_FW1_Pos                   (1U)                              
#define CAN_FW_FW1_Msk                   (0x1U << CAN_FW_FW1_Pos)      /*!< 0x00000002 */
#define CAN_FW_FW1                       CAN_FW_FW1_Msk                /*!< Filter 1 Active */
#define CAN_FW_FW2_Pos                   (2U)                              
#define CAN_FW_FW2_Msk                   (0x1U << CAN_FW_FW2_Pos)      /*!< 0x00000004 */
#define CAN_FW_FW2                       CAN_FW_FW2_Msk                /*!< Filter 2 Active */
#define CAN_FW_FW3_Pos                   (3U)                              
#define CAN_FW_FW3_Msk                   (0x1U << CAN_FW_FW3_Pos)      /*!< 0x00000008 */
#define CAN_FW_FW3                       CAN_FW_FW3_Msk                /*!< Filter 3 Active */
#define CAN_FW_FW4_Pos                   (4U)                              
#define CAN_FW_FW4_Msk                   (0x1U << CAN_FW_FW4_Pos)      /*!< 0x00000010 */
#define CAN_FW_FW4                       CAN_FW_FW4_Msk                /*!< Filter 4 Active */
#define CAN_FW_FW5_Pos                   (5U)                              
#define CAN_FW_FW5_Msk                   (0x1U << CAN_FW_FW5_Pos)      /*!< 0x00000020 */
#define CAN_FW_FW5                       CAN_FW_FW5_Msk                /*!< Filter 5 Active */
#define CAN_FW_FW6_Pos                   (6U)                              
#define CAN_FW_FW6_Msk                   (0x1U << CAN_FW_FW6_Pos)      /*!< 0x00000040 */
#define CAN_FW_FW6                       CAN_FW_FW6_Msk                /*!< Filter 6 Active */
#define CAN_FW_FW7_Pos                   (7U)                              
#define CAN_FW_FW7_Msk                   (0x1U << CAN_FW_FW7_Pos)      /*!< 0x00000080 */
#define CAN_FW_FW7                       CAN_FW_FW7_Msk                /*!< Filter 7 Active */
#define CAN_FW_FW8_Pos                   (8U)                              
#define CAN_FW_FW8_Msk                   (0x1U << CAN_FW_FW8_Pos)      /*!< 0x00000100 */
#define CAN_FW_FW8                       CAN_FW_FW8_Msk                /*!< Filter 8 Active */
#define CAN_FW_FW9_Pos                   (9U)                              
#define CAN_FW_FW9_Msk                   (0x1U << CAN_FW_FW9_Pos)      /*!< 0x00000200 */
#define CAN_FW_FW9                       CAN_FW_FW9_Msk                /*!< Filter 9 Active */
#define CAN_FW_FW10_Pos                  (10U)                             
#define CAN_FW_FW10_Msk                  (0x1U << CAN_FW_FW10_Pos)     /*!< 0x00000400 */
#define CAN_FW_FW10                      CAN_FW_FW10_Msk               /*!< Filter 10 Active */
#define CAN_FW_FW11_Pos                  (11U)                             
#define CAN_FW_FW11_Msk                  (0x1U << CAN_FW_FW11_Pos)     /*!< 0x00000800 */
#define CAN_FW_FW11                      CAN_FW_FW11_Msk               /*!< Filter 11 Active */
#define CAN_FW_FW12_Pos                  (12U)                             
#define CAN_FW_FW12_Msk                  (0x1U << CAN_FW_FW12_Pos)     /*!< 0x00001000 */
#define CAN_FW_FW12                      CAN_FW_FW12_Msk               /*!< Filter 12 Active */
#define CAN_FW_FW13_Pos                  (13U)                             
#define CAN_FW_FW13_Msk                  (0x1U << CAN_FW_FW13_Pos)     /*!< 0x00002000 */
#define CAN_FW_FW13                      CAN_FW_FW13_Msk               /*!< Filter 13 Active */
#define CAN_FW_FW14_Pos                  (14U)                             
#define CAN_FW_FW14_Msk                  (0x1U << CAN_FW_FW14_Pos)     /*!< 0x00004000 */
#define CAN_FW_FW14                      CAN_FW_FW14_Msk               /*!< Filter 14 Active */
#define CAN_FW_FW15_Pos                  (15U)                             
#define CAN_FW_FW15_Msk                  (0x1U << CAN_FW_FW15_Pos)     /*!< 0x00008000 */
#define CAN_FW_FW15                      CAN_FW_FW15_Msk               /*!< Filter 15 Active */
#define CAN_FW_FW16_Pos                  (16U)                             
#define CAN_FW_FW16_Msk                  (0x1U << CAN_FW_FW16_Pos)     /*!< 0x00010000 */
#define CAN_FW_FW16                      CAN_FW_FW16_Msk               /*!< Filter 16 Active */
#define CAN_FW_FW17_Pos                  (17U)                             
#define CAN_FW_FW17_Msk                  (0x1U << CAN_FW_FW17_Pos)     /*!< 0x00020000 */
#define CAN_FW_FW17                      CAN_FW_FW17_Msk               /*!< Filter 17 Active */
#define CAN_FW_FW18_Pos                  (18U)                             
#define CAN_FW_FW18_Msk                  (0x1U << CAN_FW_FW18_Pos)     /*!< 0x00040000 */
#define CAN_FW_FW18                      CAN_FW_FW18_Msk               /*!< Filter 18 Active */
#define CAN_FW_FW19_Pos                  (19U)                             
#define CAN_FW_FW19_Msk                  (0x1U << CAN_FW_FW19_Pos)     /*!< 0x00080000 */
#define CAN_FW_FW19                      CAN_FW_FW19_Msk               /*!< Filter 19 Active */
#define CAN_FW_FW20_Pos                  (20U)                             
#define CAN_FW_FW20_Msk                  (0x1U << CAN_FW_FW20_Pos)     /*!< 0x00100000 */
#define CAN_FW_FW20                      CAN_FW_FW20_Msk               /*!< Filter 20 Active */
#define CAN_FW_FW21_Pos                  (21U)                             
#define CAN_FW_FW21_Msk                  (0x1U << CAN_FW_FW21_Pos)     /*!< 0x00200000 */
#define CAN_FW_FW21                      CAN_FW_FW21_Msk               /*!< Filter 21 Active */
#define CAN_FW_FW22_Pos                  (22U)                             
#define CAN_FW_FW22_Msk                  (0x1U << CAN_FW_FW22_Pos)     /*!< 0x00400000 */
#define CAN_FW_FW22                      CAN_FW_FW22_Msk               /*!< Filter 22 Active */
#define CAN_FW_FW23_Pos                  (23U)                             
#define CAN_FW_FW23_Msk                  (0x1U << CAN_FW_FW23_Pos)     /*!< 0x00800000 */
#define CAN_FW_FW23                      CAN_FW_FW23_Msk               /*!< Filter 23 Active */
#define CAN_FW_FW24_Pos                  (24U)                             
#define CAN_FW_FW24_Msk                  (0x1U << CAN_FW_FW24_Pos)     /*!< 0x01000000 */
#define CAN_FW_FW24                      CAN_FW_FW24_Msk               /*!< Filter 24 Active */
#define CAN_FW_FW25_Pos                  (25U)                             
#define CAN_FW_FW25_Msk                  (0x1U << CAN_FW_FW25_Pos)     /*!< 0x02000000 */
#define CAN_FW_FW25                      CAN_FW_FW25_Msk               /*!< Filter 25 Active */
#define CAN_FW_FW26_Pos                  (26U)                             
#define CAN_FW_FW26_Msk                  (0x1U << CAN_FW_FW26_Pos)     /*!< 0x04000000 */
#define CAN_FW_FW26                      CAN_FW_FW26_Msk               /*!< Filter 26 Active */
#define CAN_FW_FW27_Pos                  (27U)                             
#define CAN_FW_FW27_Msk                  (0x1U << CAN_FW_FW27_Pos)     /*!< 0x08000000 */
#define CAN_FW_FW27                      CAN_FW_FW27_Msk               /*!< Filter 27 Active */

/*******************  Bit definition for CAN_F0R1 register  *******************/
#define CAN_F0DATA0_FD0_Pos                     (0U)                              
#define CAN_F0DATA0_FD0_Msk                     (0x1U << CAN_F0DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F0DATA0_FD0                         CAN_F0DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F0DATA0_FD1_Pos                     (1U)                              
#define CAN_F0DATA0_FD1_Msk                     (0x1U << CAN_F0DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F0DATA0_FD1                         CAN_F0DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F0DATA0_FD2_Pos                     (2U)                              
#define CAN_F0DATA0_FD2_Msk                     (0x1U << CAN_F0DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F0DATA0_FD2                         CAN_F0DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F0DATA0_FD3_Pos                     (3U)                              
#define CAN_F0DATA0_FD3_Msk                     (0x1U << CAN_F0DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F0DATA0_FD3                         CAN_F0DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F0DATA0_FD4_Pos                     (4U)                              
#define CAN_F0DATA0_FD4_Msk                     (0x1U << CAN_F0DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F0DATA0_FD4                         CAN_F0DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F0DATA0_FD5_Pos                     (5U)                              
#define CAN_F0DATA0_FD5_Msk                     (0x1U << CAN_F0DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F0DATA0_FD5                         CAN_F0DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F0DATA0_FD6_Pos                     (6U)                              
#define CAN_F0DATA0_FD6_Msk                     (0x1U << CAN_F0DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F0DATA0_FD6                         CAN_F0DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F0DATA0_FD7_Pos                     (7U)                              
#define CAN_F0DATA0_FD7_Msk                     (0x1U << CAN_F0DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F0DATA0_FD7                         CAN_F0DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F0DATA0_FD8_Pos                     (8U)                              
#define CAN_F0DATA0_FD8_Msk                     (0x1U << CAN_F0DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F0DATA0_FD8                         CAN_F0DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F0DATA0_FD9_Pos                     (9U)                              
#define CAN_F0DATA0_FD9_Msk                     (0x1U << CAN_F0DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F0DATA0_FD9                         CAN_F0DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F0DATA0_FD10_Pos                    (10U)                             
#define CAN_F0DATA0_FD10_Msk                    (0x1U << CAN_F0DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F0DATA0_FD10                        CAN_F0DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F0DATA0_FD11_Pos                    (11U)                             
#define CAN_F0DATA0_FD11_Msk                    (0x1U << CAN_F0DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F0DATA0_FD11                        CAN_F0DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F0DATA0_FD12_Pos                    (12U)                             
#define CAN_F0DATA0_FD12_Msk                    (0x1U << CAN_F0DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F0DATA0_FD12                        CAN_F0DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F0DATA0_FD13_Pos                    (13U)                             
#define CAN_F0DATA0_FD13_Msk                    (0x1U << CAN_F0DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F0DATA0_FD13                        CAN_F0DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F0DATA0_FD14_Pos                    (14U)                             
#define CAN_F0DATA0_FD14_Msk                    (0x1U << CAN_F0DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F0DATA0_FD14                        CAN_F0DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F0DATA0_FD15_Pos                    (15U)                             
#define CAN_F0DATA0_FD15_Msk                    (0x1U << CAN_F0DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F0DATA0_FD15                        CAN_F0DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F0DATA0_FD16_Pos                    (16U)                             
#define CAN_F0DATA0_FD16_Msk                    (0x1U << CAN_F0DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F0DATA0_FD16                        CAN_F0DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F0DATA0_FD17_Pos                    (17U)                             
#define CAN_F0DATA0_FD17_Msk                    (0x1U << CAN_F0DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F0DATA0_FD17                        CAN_F0DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F0DATA0_FD18_Pos                    (18U)                             
#define CAN_F0DATA0_FD18_Msk                    (0x1U << CAN_F0DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F0DATA0_FD18                        CAN_F0DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F0DATA0_FD19_Pos                    (19U)                             
#define CAN_F0DATA0_FD19_Msk                    (0x1U << CAN_F0DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F0DATA0_FD19                        CAN_F0DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F0DATA0_FD20_Pos                    (20U)                             
#define CAN_F0DATA0_FD20_Msk                    (0x1U << CAN_F0DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F0DATA0_FD20                        CAN_F0DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F0DATA0_FD21_Pos                    (21U)                             
#define CAN_F0DATA0_FD21_Msk                    (0x1U << CAN_F0DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F0DATA0_FD21                        CAN_F0DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F0DATA0_FD22_Pos                    (22U)                             
#define CAN_F0DATA0_FD22_Msk                    (0x1U << CAN_F0DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F0DATA0_FD22                        CAN_F0DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F0DATA0_FD23_Pos                    (23U)                             
#define CAN_F0DATA0_FD23_Msk                    (0x1U << CAN_F0DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F0DATA0_FD23                        CAN_F0DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F0DATA0_FD24_Pos                    (24U)                             
#define CAN_F0DATA0_FD24_Msk                    (0x1U << CAN_F0DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F0DATA0_FD24                        CAN_F0DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F0DATA0_FD25_Pos                    (25U)                             
#define CAN_F0DATA0_FD25_Msk                    (0x1U << CAN_F0DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F0DATA0_FD25                        CAN_F0DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F0DATA0_FD26_Pos                    (26U)                             
#define CAN_F0DATA0_FD26_Msk                    (0x1U << CAN_F0DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F0DATA0_FD26                        CAN_F0DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F0DATA0_FD27_Pos                    (27U)                             
#define CAN_F0DATA0_FD27_Msk                    (0x1U << CAN_F0DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F0DATA0_FD27                        CAN_F0DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F0DATA0_FD28_Pos                    (28U)                             
#define CAN_F0DATA0_FD28_Msk                    (0x1U << CAN_F0DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F0DATA0_FD28                        CAN_F0DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F0DATA0_FD29_Pos                    (29U)                             
#define CAN_F0DATA0_FD29_Msk                    (0x1U << CAN_F0DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F0DATA0_FD29                        CAN_F0DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F0DATA0_FD30_Pos                    (30U)                             
#define CAN_F0DATA0_FD30_Msk                    (0x1U << CAN_F0DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F0DATA0_FD30                        CAN_F0DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F0DATA0_FD31_Pos                    (31U)                             
#define CAN_F0DATA0_FD31_Msk                    (0x1U << CAN_F0DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F0DATA0_FD31                        CAN_F0DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F1R1 register  *******************/
#define CAN_F1DATA0_FD0_Pos                     (0U)                              
#define CAN_F1DATA0_FD0_Msk                     (0x1U << CAN_F1DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F1DATA0_FD0                         CAN_F1DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F1DATA0_FD1_Pos                     (1U)                              
#define CAN_F1DATA0_FD1_Msk                     (0x1U << CAN_F1DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F1DATA0_FD1                         CAN_F1DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F1DATA0_FD2_Pos                     (2U)                              
#define CAN_F1DATA0_FD2_Msk                     (0x1U << CAN_F1DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F1DATA0_FD2                         CAN_F1DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F1DATA0_FD3_Pos                     (3U)                              
#define CAN_F1DATA0_FD3_Msk                     (0x1U << CAN_F1DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F1DATA0_FD3                         CAN_F1DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F1DATA0_FD4_Pos                     (4U)                              
#define CAN_F1DATA0_FD4_Msk                     (0x1U << CAN_F1DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F1DATA0_FD4                         CAN_F1DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F1DATA0_FD5_Pos                     (5U)                              
#define CAN_F1DATA0_FD5_Msk                     (0x1U << CAN_F1DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F1DATA0_FD5                         CAN_F1DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F1DATA0_FD6_Pos                     (6U)                              
#define CAN_F1DATA0_FD6_Msk                     (0x1U << CAN_F1DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F1DATA0_FD6                         CAN_F1DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F1DATA0_FD7_Pos                     (7U)                              
#define CAN_F1DATA0_FD7_Msk                     (0x1U << CAN_F1DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F1DATA0_FD7                         CAN_F1DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F1DATA0_FD8_Pos                     (8U)                              
#define CAN_F1DATA0_FD8_Msk                     (0x1U << CAN_F1DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F1DATA0_FD8                         CAN_F1DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F1DATA0_FD9_Pos                     (9U)                              
#define CAN_F1DATA0_FD9_Msk                     (0x1U << CAN_F1DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F1DATA0_FD9                         CAN_F1DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F1DATA0_FD10_Pos                    (10U)                             
#define CAN_F1DATA0_FD10_Msk                    (0x1U << CAN_F1DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F1DATA0_FD10                        CAN_F1DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F1DATA0_FD11_Pos                    (11U)                             
#define CAN_F1DATA0_FD11_Msk                    (0x1U << CAN_F1DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F1DATA0_FD11                        CAN_F1DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F1DATA0_FD12_Pos                    (12U)                             
#define CAN_F1DATA0_FD12_Msk                    (0x1U << CAN_F1DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F1DATA0_FD12                        CAN_F1DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F1DATA0_FD13_Pos                    (13U)                             
#define CAN_F1DATA0_FD13_Msk                    (0x1U << CAN_F1DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F1DATA0_FD13                        CAN_F1DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F1DATA0_FD14_Pos                    (14U)                             
#define CAN_F1DATA0_FD14_Msk                    (0x1U << CAN_F1DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F1DATA0_FD14                        CAN_F1DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F1DATA0_FD15_Pos                    (15U)                             
#define CAN_F1DATA0_FD15_Msk                    (0x1U << CAN_F1DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F1DATA0_FD15                        CAN_F1DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F1DATA0_FD16_Pos                    (16U)                             
#define CAN_F1DATA0_FD16_Msk                    (0x1U << CAN_F1DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F1DATA0_FD16                        CAN_F1DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F1DATA0_FD17_Pos                    (17U)                             
#define CAN_F1DATA0_FD17_Msk                    (0x1U << CAN_F1DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F1DATA0_FD17                        CAN_F1DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F1DATA0_FD18_Pos                    (18U)                             
#define CAN_F1DATA0_FD18_Msk                    (0x1U << CAN_F1DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F1DATA0_FD18                        CAN_F1DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F1DATA0_FD19_Pos                    (19U)                             
#define CAN_F1DATA0_FD19_Msk                    (0x1U << CAN_F1DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F1DATA0_FD19                        CAN_F1DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F1DATA0_FD20_Pos                    (20U)                             
#define CAN_F1DATA0_FD20_Msk                    (0x1U << CAN_F1DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F1DATA0_FD20                        CAN_F1DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F1DATA0_FD21_Pos                    (21U)                             
#define CAN_F1DATA0_FD21_Msk                    (0x1U << CAN_F1DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F1DATA0_FD21                        CAN_F1DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F1DATA0_FD22_Pos                    (22U)                             
#define CAN_F1DATA0_FD22_Msk                    (0x1U << CAN_F1DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F1DATA0_FD22                        CAN_F1DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F1DATA0_FD23_Pos                    (23U)                             
#define CAN_F1DATA0_FD23_Msk                    (0x1U << CAN_F1DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F1DATA0_FD23                        CAN_F1DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F1DATA0_FD24_Pos                    (24U)                             
#define CAN_F1DATA0_FD24_Msk                    (0x1U << CAN_F1DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F1DATA0_FD24                        CAN_F1DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F1DATA0_FD25_Pos                    (25U)                             
#define CAN_F1DATA0_FD25_Msk                    (0x1U << CAN_F1DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F1DATA0_FD25                        CAN_F1DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F1DATA0_FD26_Pos                    (26U)                             
#define CAN_F1DATA0_FD26_Msk                    (0x1U << CAN_F1DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F1DATA0_FD26                        CAN_F1DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F1DATA0_FD27_Pos                    (27U)                             
#define CAN_F1DATA0_FD27_Msk                    (0x1U << CAN_F1DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F1DATA0_FD27                        CAN_F1DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F1DATA0_FD28_Pos                    (28U)                             
#define CAN_F1DATA0_FD28_Msk                    (0x1U << CAN_F1DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F1DATA0_FD28                        CAN_F1DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F1DATA0_FD29_Pos                    (29U)                             
#define CAN_F1DATA0_FD29_Msk                    (0x1U << CAN_F1DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F1DATA0_FD29                        CAN_F1DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F1DATA0_FD30_Pos                    (30U)                             
#define CAN_F1DATA0_FD30_Msk                    (0x1U << CAN_F1DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F1DATA0_FD30                        CAN_F1DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F1DATA0_FD31_Pos                    (31U)                             
#define CAN_F1DATA0_FD31_Msk                    (0x1U << CAN_F1DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F1DATA0_FD31                        CAN_F1DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F2R1 register  *******************/
#define CAN_F2DATA0_FD0_Pos                     (0U)                              
#define CAN_F2DATA0_FD0_Msk                     (0x1U << CAN_F2DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F2DATA0_FD0                         CAN_F2DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F2DATA0_FD1_Pos                     (1U)                              
#define CAN_F2DATA0_FD1_Msk                     (0x1U << CAN_F2DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F2DATA0_FD1                         CAN_F2DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F2DATA0_FD2_Pos                     (2U)                              
#define CAN_F2DATA0_FD2_Msk                     (0x1U << CAN_F2DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F2DATA0_FD2                         CAN_F2DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F2DATA0_FD3_Pos                     (3U)                              
#define CAN_F2DATA0_FD3_Msk                     (0x1U << CAN_F2DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F2DATA0_FD3                         CAN_F2DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F2DATA0_FD4_Pos                     (4U)                              
#define CAN_F2DATA0_FD4_Msk                     (0x1U << CAN_F2DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F2DATA0_FD4                         CAN_F2DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F2DATA0_FD5_Pos                     (5U)                              
#define CAN_F2DATA0_FD5_Msk                     (0x1U << CAN_F2DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F2DATA0_FD5                         CAN_F2DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F2DATA0_FD6_Pos                     (6U)                              
#define CAN_F2DATA0_FD6_Msk                     (0x1U << CAN_F2DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F2DATA0_FD6                         CAN_F2DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F2DATA0_FD7_Pos                     (7U)                              
#define CAN_F2DATA0_FD7_Msk                     (0x1U << CAN_F2DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F2DATA0_FD7                         CAN_F2DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F2DATA0_FD8_Pos                     (8U)                              
#define CAN_F2DATA0_FD8_Msk                     (0x1U << CAN_F2DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F2DATA0_FD8                         CAN_F2DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F2DATA0_FD9_Pos                     (9U)                              
#define CAN_F2DATA0_FD9_Msk                     (0x1U << CAN_F2DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F2DATA0_FD9                         CAN_F2DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F2DATA0_FD10_Pos                    (10U)                             
#define CAN_F2DATA0_FD10_Msk                    (0x1U << CAN_F2DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F2DATA0_FD10                        CAN_F2DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F2DATA0_FD11_Pos                    (11U)                             
#define CAN_F2DATA0_FD11_Msk                    (0x1U << CAN_F2DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F2DATA0_FD11                        CAN_F2DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F2DATA0_FD12_Pos                    (12U)                             
#define CAN_F2DATA0_FD12_Msk                    (0x1U << CAN_F2DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F2DATA0_FD12                        CAN_F2DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F2DATA0_FD13_Pos                    (13U)                             
#define CAN_F2DATA0_FD13_Msk                    (0x1U << CAN_F2DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F2DATA0_FD13                        CAN_F2DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F2DATA0_FD14_Pos                    (14U)                             
#define CAN_F2DATA0_FD14_Msk                    (0x1U << CAN_F2DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F2DATA0_FD14                        CAN_F2DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F2DATA0_FD15_Pos                    (15U)                             
#define CAN_F2DATA0_FD15_Msk                    (0x1U << CAN_F2DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F2DATA0_FD15                        CAN_F2DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F2DATA0_FD16_Pos                    (16U)                             
#define CAN_F2DATA0_FD16_Msk                    (0x1U << CAN_F2DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F2DATA0_FD16                        CAN_F2DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F2DATA0_FD17_Pos                    (17U)                             
#define CAN_F2DATA0_FD17_Msk                    (0x1U << CAN_F2DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F2DATA0_FD17                        CAN_F2DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F2DATA0_FD18_Pos                    (18U)                             
#define CAN_F2DATA0_FD18_Msk                    (0x1U << CAN_F2DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F2DATA0_FD18                        CAN_F2DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F2DATA0_FD19_Pos                    (19U)                             
#define CAN_F2DATA0_FD19_Msk                    (0x1U << CAN_F2DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F2DATA0_FD19                        CAN_F2DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F2DATA0_FD20_Pos                    (20U)                             
#define CAN_F2DATA0_FD20_Msk                    (0x1U << CAN_F2DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F2DATA0_FD20                        CAN_F2DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F2DATA0_FD21_Pos                    (21U)                             
#define CAN_F2DATA0_FD21_Msk                    (0x1U << CAN_F2DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F2DATA0_FD21                        CAN_F2DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F2DATA0_FD22_Pos                    (22U)                             
#define CAN_F2DATA0_FD22_Msk                    (0x1U << CAN_F2DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F2DATA0_FD22                        CAN_F2DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F2DATA0_FD23_Pos                    (23U)                             
#define CAN_F2DATA0_FD23_Msk                    (0x1U << CAN_F2DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F2DATA0_FD23                        CAN_F2DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F2DATA0_FD24_Pos                    (24U)                             
#define CAN_F2DATA0_FD24_Msk                    (0x1U << CAN_F2DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F2DATA0_FD24                        CAN_F2DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F2DATA0_FD25_Pos                    (25U)                             
#define CAN_F2DATA0_FD25_Msk                    (0x1U << CAN_F2DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F2DATA0_FD25                        CAN_F2DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F2DATA0_FD26_Pos                    (26U)                             
#define CAN_F2DATA0_FD26_Msk                    (0x1U << CAN_F2DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F2DATA0_FD26                        CAN_F2DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F2DATA0_FD27_Pos                    (27U)                             
#define CAN_F2DATA0_FD27_Msk                    (0x1U << CAN_F2DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F2DATA0_FD27                        CAN_F2DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F2DATA0_FD28_Pos                    (28U)                             
#define CAN_F2DATA0_FD28_Msk                    (0x1U << CAN_F2DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F2DATA0_FD28                        CAN_F2DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F2DATA0_FD29_Pos                    (29U)                             
#define CAN_F2DATA0_FD29_Msk                    (0x1U << CAN_F2DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F2DATA0_FD29                        CAN_F2DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F2DATA0_FD30_Pos                    (30U)                             
#define CAN_F2DATA0_FD30_Msk                    (0x1U << CAN_F2DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F2DATA0_FD30                        CAN_F2DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F2DATA0_FD31_Pos                    (31U)                             
#define CAN_F2DATA0_FD31_Msk                    (0x1U << CAN_F2DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F2DATA0_FD31                        CAN_F2DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F3R1 register  *******************/
#define CAN_F3DATA0_FD0_Pos                     (0U)                              
#define CAN_F3DATA0_FD0_Msk                     (0x1U << CAN_F3DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F3DATA0_FD0                         CAN_F3DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F3DATA0_FD1_Pos                     (1U)                              
#define CAN_F3DATA0_FD1_Msk                     (0x1U << CAN_F3DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F3DATA0_FD1                         CAN_F3DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F3DATA0_FD2_Pos                     (2U)                              
#define CAN_F3DATA0_FD2_Msk                     (0x1U << CAN_F3DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F3DATA0_FD2                         CAN_F3DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F3DATA0_FD3_Pos                     (3U)                              
#define CAN_F3DATA0_FD3_Msk                     (0x1U << CAN_F3DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F3DATA0_FD3                         CAN_F3DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F3DATA0_FD4_Pos                     (4U)                              
#define CAN_F3DATA0_FD4_Msk                     (0x1U << CAN_F3DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F3DATA0_FD4                         CAN_F3DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F3DATA0_FD5_Pos                     (5U)                              
#define CAN_F3DATA0_FD5_Msk                     (0x1U << CAN_F3DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F3DATA0_FD5                         CAN_F3DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F3DATA0_FD6_Pos                     (6U)                              
#define CAN_F3DATA0_FD6_Msk                     (0x1U << CAN_F3DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F3DATA0_FD6                         CAN_F3DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F3DATA0_FD7_Pos                     (7U)                              
#define CAN_F3DATA0_FD7_Msk                     (0x1U << CAN_F3DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F3DATA0_FD7                         CAN_F3DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F3DATA0_FD8_Pos                     (8U)                              
#define CAN_F3DATA0_FD8_Msk                     (0x1U << CAN_F3DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F3DATA0_FD8                         CAN_F3DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F3DATA0_FD9_Pos                     (9U)                              
#define CAN_F3DATA0_FD9_Msk                     (0x1U << CAN_F3DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F3DATA0_FD9                         CAN_F3DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F3DATA0_FD10_Pos                    (10U)                             
#define CAN_F3DATA0_FD10_Msk                    (0x1U << CAN_F3DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F3DATA0_FD10                        CAN_F3DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F3DATA0_FD11_Pos                    (11U)                             
#define CAN_F3DATA0_FD11_Msk                    (0x1U << CAN_F3DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F3DATA0_FD11                        CAN_F3DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F3DATA0_FD12_Pos                    (12U)                             
#define CAN_F3DATA0_FD12_Msk                    (0x1U << CAN_F3DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F3DATA0_FD12                        CAN_F3DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F3DATA0_FD13_Pos                    (13U)                             
#define CAN_F3DATA0_FD13_Msk                    (0x1U << CAN_F3DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F3DATA0_FD13                        CAN_F3DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F3DATA0_FD14_Pos                    (14U)                             
#define CAN_F3DATA0_FD14_Msk                    (0x1U << CAN_F3DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F3DATA0_FD14                        CAN_F3DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F3DATA0_FD15_Pos                    (15U)                             
#define CAN_F3DATA0_FD15_Msk                    (0x1U << CAN_F3DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F3DATA0_FD15                        CAN_F3DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F3DATA0_FD16_Pos                    (16U)                             
#define CAN_F3DATA0_FD16_Msk                    (0x1U << CAN_F3DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F3DATA0_FD16                        CAN_F3DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F3DATA0_FD17_Pos                    (17U)                             
#define CAN_F3DATA0_FD17_Msk                    (0x1U << CAN_F3DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F3DATA0_FD17                        CAN_F3DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F3DATA0_FD18_Pos                    (18U)                             
#define CAN_F3DATA0_FD18_Msk                    (0x1U << CAN_F3DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F3DATA0_FD18                        CAN_F3DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F3DATA0_FD19_Pos                    (19U)                             
#define CAN_F3DATA0_FD19_Msk                    (0x1U << CAN_F3DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F3DATA0_FD19                        CAN_F3DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F3DATA0_FD20_Pos                    (20U)                             
#define CAN_F3DATA0_FD20_Msk                    (0x1U << CAN_F3DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F3DATA0_FD20                        CAN_F3DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F3DATA0_FD21_Pos                    (21U)                             
#define CAN_F3DATA0_FD21_Msk                    (0x1U << CAN_F3DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F3DATA0_FD21                        CAN_F3DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F3DATA0_FD22_Pos                    (22U)                             
#define CAN_F3DATA0_FD22_Msk                    (0x1U << CAN_F3DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F3DATA0_FD22                        CAN_F3DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F3DATA0_FD23_Pos                    (23U)                             
#define CAN_F3DATA0_FD23_Msk                    (0x1U << CAN_F3DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F3DATA0_FD23                        CAN_F3DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F3DATA0_FD24_Pos                    (24U)                             
#define CAN_F3DATA0_FD24_Msk                    (0x1U << CAN_F3DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F3DATA0_FD24                        CAN_F3DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F3DATA0_FD25_Pos                    (25U)                             
#define CAN_F3DATA0_FD25_Msk                    (0x1U << CAN_F3DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F3DATA0_FD25                        CAN_F3DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F3DATA0_FD26_Pos                    (26U)                             
#define CAN_F3DATA0_FD26_Msk                    (0x1U << CAN_F3DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F3DATA0_FD26                        CAN_F3DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F3DATA0_FD27_Pos                    (27U)                             
#define CAN_F3DATA0_FD27_Msk                    (0x1U << CAN_F3DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F3DATA0_FD27                        CAN_F3DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F3DATA0_FD28_Pos                    (28U)                             
#define CAN_F3DATA0_FD28_Msk                    (0x1U << CAN_F3DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F3DATA0_FD28                        CAN_F3DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F3DATA0_FD29_Pos                    (29U)                             
#define CAN_F3DATA0_FD29_Msk                    (0x1U << CAN_F3DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F3DATA0_FD29                        CAN_F3DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F3DATA0_FD30_Pos                    (30U)                             
#define CAN_F3DATA0_FD30_Msk                    (0x1U << CAN_F3DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F3DATA0_FD30                        CAN_F3DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F3DATA0_FD31_Pos                    (31U)                             
#define CAN_F3DATA0_FD31_Msk                    (0x1U << CAN_F3DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F3DATA0_FD31                        CAN_F3DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F4R1 register  *******************/
#define CAN_F4DATA0_FD0_Pos                     (0U)                              
#define CAN_F4DATA0_FD0_Msk                     (0x1U << CAN_F4DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F4DATA0_FD0                         CAN_F4DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F4DATA0_FD1_Pos                     (1U)                              
#define CAN_F4DATA0_FD1_Msk                     (0x1U << CAN_F4DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F4DATA0_FD1                         CAN_F4DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F4DATA0_FD2_Pos                     (2U)                              
#define CAN_F4DATA0_FD2_Msk                     (0x1U << CAN_F4DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F4DATA0_FD2                         CAN_F4DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F4DATA0_FD3_Pos                     (3U)                              
#define CAN_F4DATA0_FD3_Msk                     (0x1U << CAN_F4DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F4DATA0_FD3                         CAN_F4DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F4DATA0_FD4_Pos                     (4U)                              
#define CAN_F4DATA0_FD4_Msk                     (0x1U << CAN_F4DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F4DATA0_FD4                         CAN_F4DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F4DATA0_FD5_Pos                     (5U)                              
#define CAN_F4DATA0_FD5_Msk                     (0x1U << CAN_F4DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F4DATA0_FD5                         CAN_F4DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F4DATA0_FD6_Pos                     (6U)                              
#define CAN_F4DATA0_FD6_Msk                     (0x1U << CAN_F4DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F4DATA0_FD6                         CAN_F4DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F4DATA0_FD7_Pos                     (7U)                              
#define CAN_F4DATA0_FD7_Msk                     (0x1U << CAN_F4DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F4DATA0_FD7                         CAN_F4DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F4DATA0_FD8_Pos                     (8U)                              
#define CAN_F4DATA0_FD8_Msk                     (0x1U << CAN_F4DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F4DATA0_FD8                         CAN_F4DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F4DATA0_FD9_Pos                     (9U)                              
#define CAN_F4DATA0_FD9_Msk                     (0x1U << CAN_F4DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F4DATA0_FD9                         CAN_F4DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F4DATA0_FD10_Pos                    (10U)                             
#define CAN_F4DATA0_FD10_Msk                    (0x1U << CAN_F4DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F4DATA0_FD10                        CAN_F4DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F4DATA0_FD11_Pos                    (11U)                             
#define CAN_F4DATA0_FD11_Msk                    (0x1U << CAN_F4DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F4DATA0_FD11                        CAN_F4DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F4DATA0_FD12_Pos                    (12U)                             
#define CAN_F4DATA0_FD12_Msk                    (0x1U << CAN_F4DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F4DATA0_FD12                        CAN_F4DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F4DATA0_FD13_Pos                    (13U)                             
#define CAN_F4DATA0_FD13_Msk                    (0x1U << CAN_F4DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F4DATA0_FD13                        CAN_F4DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F4DATA0_FD14_Pos                    (14U)                             
#define CAN_F4DATA0_FD14_Msk                    (0x1U << CAN_F4DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F4DATA0_FD14                        CAN_F4DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F4DATA0_FD15_Pos                    (15U)                             
#define CAN_F4DATA0_FD15_Msk                    (0x1U << CAN_F4DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F4DATA0_FD15                        CAN_F4DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F4DATA0_FD16_Pos                    (16U)                             
#define CAN_F4DATA0_FD16_Msk                    (0x1U << CAN_F4DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F4DATA0_FD16                        CAN_F4DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F4DATA0_FD17_Pos                    (17U)                             
#define CAN_F4DATA0_FD17_Msk                    (0x1U << CAN_F4DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F4DATA0_FD17                        CAN_F4DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F4DATA0_FD18_Pos                    (18U)                             
#define CAN_F4DATA0_FD18_Msk                    (0x1U << CAN_F4DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F4DATA0_FD18                        CAN_F4DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F4DATA0_FD19_Pos                    (19U)                             
#define CAN_F4DATA0_FD19_Msk                    (0x1U << CAN_F4DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F4DATA0_FD19                        CAN_F4DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F4DATA0_FD20_Pos                    (20U)                             
#define CAN_F4DATA0_FD20_Msk                    (0x1U << CAN_F4DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F4DATA0_FD20                        CAN_F4DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F4DATA0_FD21_Pos                    (21U)                             
#define CAN_F4DATA0_FD21_Msk                    (0x1U << CAN_F4DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F4DATA0_FD21                        CAN_F4DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F4DATA0_FD22_Pos                    (22U)                             
#define CAN_F4DATA0_FD22_Msk                    (0x1U << CAN_F4DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F4DATA0_FD22                        CAN_F4DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F4DATA0_FD23_Pos                    (23U)                             
#define CAN_F4DATA0_FD23_Msk                    (0x1U << CAN_F4DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F4DATA0_FD23                        CAN_F4DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F4DATA0_FD24_Pos                    (24U)                             
#define CAN_F4DATA0_FD24_Msk                    (0x1U << CAN_F4DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F4DATA0_FD24                        CAN_F4DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F4DATA0_FD25_Pos                    (25U)                             
#define CAN_F4DATA0_FD25_Msk                    (0x1U << CAN_F4DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F4DATA0_FD25                        CAN_F4DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F4DATA0_FD26_Pos                    (26U)                             
#define CAN_F4DATA0_FD26_Msk                    (0x1U << CAN_F4DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F4DATA0_FD26                        CAN_F4DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F4DATA0_FD27_Pos                    (27U)                             
#define CAN_F4DATA0_FD27_Msk                    (0x1U << CAN_F4DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F4DATA0_FD27                        CAN_F4DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F4DATA0_FD28_Pos                    (28U)                             
#define CAN_F4DATA0_FD28_Msk                    (0x1U << CAN_F4DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F4DATA0_FD28                        CAN_F4DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F4DATA0_FD29_Pos                    (29U)                             
#define CAN_F4DATA0_FD29_Msk                    (0x1U << CAN_F4DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F4DATA0_FD29                        CAN_F4DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F4DATA0_FD30_Pos                    (30U)                             
#define CAN_F4DATA0_FD30_Msk                    (0x1U << CAN_F4DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F4DATA0_FD30                        CAN_F4DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F4DATA0_FD31_Pos                    (31U)                             
#define CAN_F4DATA0_FD31_Msk                    (0x1U << CAN_F4DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F4DATA0_FD31                        CAN_F4DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F5R1 register  *******************/
#define CAN_F5DATA0_FD0_Pos                     (0U)                              
#define CAN_F5DATA0_FD0_Msk                     (0x1U << CAN_F5DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F5DATA0_FD0                         CAN_F5DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F5DATA0_FD1_Pos                     (1U)                              
#define CAN_F5DATA0_FD1_Msk                     (0x1U << CAN_F5DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F5DATA0_FD1                         CAN_F5DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F5DATA0_FD2_Pos                     (2U)                              
#define CAN_F5DATA0_FD2_Msk                     (0x1U << CAN_F5DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F5DATA0_FD2                         CAN_F5DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F5DATA0_FD3_Pos                     (3U)                              
#define CAN_F5DATA0_FD3_Msk                     (0x1U << CAN_F5DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F5DATA0_FD3                         CAN_F5DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F5DATA0_FD4_Pos                     (4U)                              
#define CAN_F5DATA0_FD4_Msk                     (0x1U << CAN_F5DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F5DATA0_FD4                         CAN_F5DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F5DATA0_FD5_Pos                     (5U)                              
#define CAN_F5DATA0_FD5_Msk                     (0x1U << CAN_F5DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F5DATA0_FD5                         CAN_F5DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F5DATA0_FD6_Pos                     (6U)                              
#define CAN_F5DATA0_FD6_Msk                     (0x1U << CAN_F5DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F5DATA0_FD6                         CAN_F5DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F5DATA0_FD7_Pos                     (7U)                              
#define CAN_F5DATA0_FD7_Msk                     (0x1U << CAN_F5DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F5DATA0_FD7                         CAN_F5DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F5DATA0_FD8_Pos                     (8U)                              
#define CAN_F5DATA0_FD8_Msk                     (0x1U << CAN_F5DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F5DATA0_FD8                         CAN_F5DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F5DATA0_FD9_Pos                     (9U)                              
#define CAN_F5DATA0_FD9_Msk                     (0x1U << CAN_F5DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F5DATA0_FD9                         CAN_F5DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F5DATA0_FD10_Pos                    (10U)                             
#define CAN_F5DATA0_FD10_Msk                    (0x1U << CAN_F5DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F5DATA0_FD10                        CAN_F5DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F5DATA0_FD11_Pos                    (11U)                             
#define CAN_F5DATA0_FD11_Msk                    (0x1U << CAN_F5DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F5DATA0_FD11                        CAN_F5DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F5DATA0_FD12_Pos                    (12U)                             
#define CAN_F5DATA0_FD12_Msk                    (0x1U << CAN_F5DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F5DATA0_FD12                        CAN_F5DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F5DATA0_FD13_Pos                    (13U)                             
#define CAN_F5DATA0_FD13_Msk                    (0x1U << CAN_F5DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F5DATA0_FD13                        CAN_F5DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F5DATA0_FD14_Pos                    (14U)                             
#define CAN_F5DATA0_FD14_Msk                    (0x1U << CAN_F5DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F5DATA0_FD14                        CAN_F5DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F5DATA0_FD15_Pos                    (15U)                             
#define CAN_F5DATA0_FD15_Msk                    (0x1U << CAN_F5DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F5DATA0_FD15                        CAN_F5DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F5DATA0_FD16_Pos                    (16U)                             
#define CAN_F5DATA0_FD16_Msk                    (0x1U << CAN_F5DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F5DATA0_FD16                        CAN_F5DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F5DATA0_FD17_Pos                    (17U)                             
#define CAN_F5DATA0_FD17_Msk                    (0x1U << CAN_F5DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F5DATA0_FD17                        CAN_F5DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F5DATA0_FD18_Pos                    (18U)                             
#define CAN_F5DATA0_FD18_Msk                    (0x1U << CAN_F5DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F5DATA0_FD18                        CAN_F5DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F5DATA0_FD19_Pos                    (19U)                             
#define CAN_F5DATA0_FD19_Msk                    (0x1U << CAN_F5DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F5DATA0_FD19                        CAN_F5DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F5DATA0_FD20_Pos                    (20U)                             
#define CAN_F5DATA0_FD20_Msk                    (0x1U << CAN_F5DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F5DATA0_FD20                        CAN_F5DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F5DATA0_FD21_Pos                    (21U)                             
#define CAN_F5DATA0_FD21_Msk                    (0x1U << CAN_F5DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F5DATA0_FD21                        CAN_F5DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F5DATA0_FD22_Pos                    (22U)                             
#define CAN_F5DATA0_FD22_Msk                    (0x1U << CAN_F5DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F5DATA0_FD22                        CAN_F5DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F5DATA0_FD23_Pos                    (23U)                             
#define CAN_F5DATA0_FD23_Msk                    (0x1U << CAN_F5DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F5DATA0_FD23                        CAN_F5DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F5DATA0_FD24_Pos                    (24U)                             
#define CAN_F5DATA0_FD24_Msk                    (0x1U << CAN_F5DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F5DATA0_FD24                        CAN_F5DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F5DATA0_FD25_Pos                    (25U)                             
#define CAN_F5DATA0_FD25_Msk                    (0x1U << CAN_F5DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F5DATA0_FD25                        CAN_F5DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F5DATA0_FD26_Pos                    (26U)                             
#define CAN_F5DATA0_FD26_Msk                    (0x1U << CAN_F5DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F5DATA0_FD26                        CAN_F5DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F5DATA0_FD27_Pos                    (27U)                             
#define CAN_F5DATA0_FD27_Msk                    (0x1U << CAN_F5DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F5DATA0_FD27                        CAN_F5DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F5DATA0_FD28_Pos                    (28U)                             
#define CAN_F5DATA0_FD28_Msk                    (0x1U << CAN_F5DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F5DATA0_FD28                        CAN_F5DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F5DATA0_FD29_Pos                    (29U)                             
#define CAN_F5DATA0_FD29_Msk                    (0x1U << CAN_F5DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F5DATA0_FD29                        CAN_F5DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F5DATA0_FD30_Pos                    (30U)                             
#define CAN_F5DATA0_FD30_Msk                    (0x1U << CAN_F5DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F5DATA0_FD30                        CAN_F5DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F5DATA0_FD31_Pos                    (31U)                             
#define CAN_F5DATA0_FD31_Msk                    (0x1U << CAN_F5DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F5DATA0_FD31                        CAN_F5DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F6R1 register  *******************/
#define CAN_F6DATA0_FD0_Pos                     (0U)                              
#define CAN_F6DATA0_FD0_Msk                     (0x1U << CAN_F6DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F6DATA0_FD0                         CAN_F6DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F6DATA0_FD1_Pos                     (1U)                              
#define CAN_F6DATA0_FD1_Msk                     (0x1U << CAN_F6DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F6DATA0_FD1                         CAN_F6DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F6DATA0_FD2_Pos                     (2U)                              
#define CAN_F6DATA0_FD2_Msk                     (0x1U << CAN_F6DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F6DATA0_FD2                         CAN_F6DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F6DATA0_FD3_Pos                     (3U)                              
#define CAN_F6DATA0_FD3_Msk                     (0x1U << CAN_F6DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F6DATA0_FD3                         CAN_F6DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F6DATA0_FD4_Pos                     (4U)                              
#define CAN_F6DATA0_FD4_Msk                     (0x1U << CAN_F6DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F6DATA0_FD4                         CAN_F6DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F6DATA0_FD5_Pos                     (5U)                              
#define CAN_F6DATA0_FD5_Msk                     (0x1U << CAN_F6DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F6DATA0_FD5                         CAN_F6DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F6DATA0_FD6_Pos                     (6U)                              
#define CAN_F6DATA0_FD6_Msk                     (0x1U << CAN_F6DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F6DATA0_FD6                         CAN_F6DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F6DATA0_FD7_Pos                     (7U)                              
#define CAN_F6DATA0_FD7_Msk                     (0x1U << CAN_F6DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F6DATA0_FD7                         CAN_F6DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F6DATA0_FD8_Pos                     (8U)                              
#define CAN_F6DATA0_FD8_Msk                     (0x1U << CAN_F6DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F6DATA0_FD8                         CAN_F6DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F6DATA0_FD9_Pos                     (9U)                              
#define CAN_F6DATA0_FD9_Msk                     (0x1U << CAN_F6DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F6DATA0_FD9                         CAN_F6DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F6DATA0_FD10_Pos                    (10U)                             
#define CAN_F6DATA0_FD10_Msk                    (0x1U << CAN_F6DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F6DATA0_FD10                        CAN_F6DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F6DATA0_FD11_Pos                    (11U)                             
#define CAN_F6DATA0_FD11_Msk                    (0x1U << CAN_F6DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F6DATA0_FD11                        CAN_F6DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F6DATA0_FD12_Pos                    (12U)                             
#define CAN_F6DATA0_FD12_Msk                    (0x1U << CAN_F6DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F6DATA0_FD12                        CAN_F6DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F6DATA0_FD13_Pos                    (13U)                             
#define CAN_F6DATA0_FD13_Msk                    (0x1U << CAN_F6DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F6DATA0_FD13                        CAN_F6DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F6DATA0_FD14_Pos                    (14U)                             
#define CAN_F6DATA0_FD14_Msk                    (0x1U << CAN_F6DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F6DATA0_FD14                        CAN_F6DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F6DATA0_FD15_Pos                    (15U)                             
#define CAN_F6DATA0_FD15_Msk                    (0x1U << CAN_F6DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F6DATA0_FD15                        CAN_F6DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F6DATA0_FD16_Pos                    (16U)                             
#define CAN_F6DATA0_FD16_Msk                    (0x1U << CAN_F6DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F6DATA0_FD16                        CAN_F6DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F6DATA0_FD17_Pos                    (17U)                             
#define CAN_F6DATA0_FD17_Msk                    (0x1U << CAN_F6DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F6DATA0_FD17                        CAN_F6DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F6DATA0_FD18_Pos                    (18U)                             
#define CAN_F6DATA0_FD18_Msk                    (0x1U << CAN_F6DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F6DATA0_FD18                        CAN_F6DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F6DATA0_FD19_Pos                    (19U)                             
#define CAN_F6DATA0_FD19_Msk                    (0x1U << CAN_F6DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F6DATA0_FD19                        CAN_F6DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F6DATA0_FD20_Pos                    (20U)                             
#define CAN_F6DATA0_FD20_Msk                    (0x1U << CAN_F6DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F6DATA0_FD20                        CAN_F6DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F6DATA0_FD21_Pos                    (21U)                             
#define CAN_F6DATA0_FD21_Msk                    (0x1U << CAN_F6DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F6DATA0_FD21                        CAN_F6DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F6DATA0_FD22_Pos                    (22U)                             
#define CAN_F6DATA0_FD22_Msk                    (0x1U << CAN_F6DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F6DATA0_FD22                        CAN_F6DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F6DATA0_FD23_Pos                    (23U)                             
#define CAN_F6DATA0_FD23_Msk                    (0x1U << CAN_F6DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F6DATA0_FD23                        CAN_F6DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F6DATA0_FD24_Pos                    (24U)                             
#define CAN_F6DATA0_FD24_Msk                    (0x1U << CAN_F6DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F6DATA0_FD24                        CAN_F6DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F6DATA0_FD25_Pos                    (25U)                             
#define CAN_F6DATA0_FD25_Msk                    (0x1U << CAN_F6DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F6DATA0_FD25                        CAN_F6DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F6DATA0_FD26_Pos                    (26U)                             
#define CAN_F6DATA0_FD26_Msk                    (0x1U << CAN_F6DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F6DATA0_FD26                        CAN_F6DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F6DATA0_FD27_Pos                    (27U)                             
#define CAN_F6DATA0_FD27_Msk                    (0x1U << CAN_F6DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F6DATA0_FD27                        CAN_F6DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F6DATA0_FD28_Pos                    (28U)                             
#define CAN_F6DATA0_FD28_Msk                    (0x1U << CAN_F6DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F6DATA0_FD28                        CAN_F6DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F6DATA0_FD29_Pos                    (29U)                             
#define CAN_F6DATA0_FD29_Msk                    (0x1U << CAN_F6DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F6DATA0_FD29                        CAN_F6DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F6DATA0_FD30_Pos                    (30U)                             
#define CAN_F6DATA0_FD30_Msk                    (0x1U << CAN_F6DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F6DATA0_FD30                        CAN_F6DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F6DATA0_FD31_Pos                    (31U)                             
#define CAN_F6DATA0_FD31_Msk                    (0x1U << CAN_F6DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F6DATA0_FD31                        CAN_F6DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F7R1 register  *******************/
#define CAN_F7DATA0_FD0_Pos                     (0U)                              
#define CAN_F7DATA0_FD0_Msk                     (0x1U << CAN_F7DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F7DATA0_FD0                         CAN_F7DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F7DATA0_FD1_Pos                     (1U)                              
#define CAN_F7DATA0_FD1_Msk                     (0x1U << CAN_F7DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F7DATA0_FD1                         CAN_F7DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F7DATA0_FD2_Pos                     (2U)                              
#define CAN_F7DATA0_FD2_Msk                     (0x1U << CAN_F7DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F7DATA0_FD2                         CAN_F7DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F7DATA0_FD3_Pos                     (3U)                              
#define CAN_F7DATA0_FD3_Msk                     (0x1U << CAN_F7DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F7DATA0_FD3                         CAN_F7DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F7DATA0_FD4_Pos                     (4U)                              
#define CAN_F7DATA0_FD4_Msk                     (0x1U << CAN_F7DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F7DATA0_FD4                         CAN_F7DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F7DATA0_FD5_Pos                     (5U)                              
#define CAN_F7DATA0_FD5_Msk                     (0x1U << CAN_F7DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F7DATA0_FD5                         CAN_F7DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F7DATA0_FD6_Pos                     (6U)                              
#define CAN_F7DATA0_FD6_Msk                     (0x1U << CAN_F7DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F7DATA0_FD6                         CAN_F7DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F7DATA0_FD7_Pos                     (7U)                              
#define CAN_F7DATA0_FD7_Msk                     (0x1U << CAN_F7DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F7DATA0_FD7                         CAN_F7DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F7DATA0_FD8_Pos                     (8U)                              
#define CAN_F7DATA0_FD8_Msk                     (0x1U << CAN_F7DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F7DATA0_FD8                         CAN_F7DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F7DATA0_FD9_Pos                     (9U)                              
#define CAN_F7DATA0_FD9_Msk                     (0x1U << CAN_F7DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F7DATA0_FD9                         CAN_F7DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F7DATA0_FD10_Pos                    (10U)                             
#define CAN_F7DATA0_FD10_Msk                    (0x1U << CAN_F7DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F7DATA0_FD10                        CAN_F7DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F7DATA0_FD11_Pos                    (11U)                             
#define CAN_F7DATA0_FD11_Msk                    (0x1U << CAN_F7DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F7DATA0_FD11                        CAN_F7DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F7DATA0_FD12_Pos                    (12U)                             
#define CAN_F7DATA0_FD12_Msk                    (0x1U << CAN_F7DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F7DATA0_FD12                        CAN_F7DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F7DATA0_FD13_Pos                    (13U)                             
#define CAN_F7DATA0_FD13_Msk                    (0x1U << CAN_F7DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F7DATA0_FD13                        CAN_F7DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F7DATA0_FD14_Pos                    (14U)                             
#define CAN_F7DATA0_FD14_Msk                    (0x1U << CAN_F7DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F7DATA0_FD14                        CAN_F7DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F7DATA0_FD15_Pos                    (15U)                             
#define CAN_F7DATA0_FD15_Msk                    (0x1U << CAN_F7DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F7DATA0_FD15                        CAN_F7DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F7DATA0_FD16_Pos                    (16U)                             
#define CAN_F7DATA0_FD16_Msk                    (0x1U << CAN_F7DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F7DATA0_FD16                        CAN_F7DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F7DATA0_FD17_Pos                    (17U)                             
#define CAN_F7DATA0_FD17_Msk                    (0x1U << CAN_F7DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F7DATA0_FD17                        CAN_F7DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F7DATA0_FD18_Pos                    (18U)                             
#define CAN_F7DATA0_FD18_Msk                    (0x1U << CAN_F7DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F7DATA0_FD18                        CAN_F7DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F7DATA0_FD19_Pos                    (19U)                             
#define CAN_F7DATA0_FD19_Msk                    (0x1U << CAN_F7DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F7DATA0_FD19                        CAN_F7DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F7DATA0_FD20_Pos                    (20U)                             
#define CAN_F7DATA0_FD20_Msk                    (0x1U << CAN_F7DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F7DATA0_FD20                        CAN_F7DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F7DATA0_FD21_Pos                    (21U)                             
#define CAN_F7DATA0_FD21_Msk                    (0x1U << CAN_F7DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F7DATA0_FD21                        CAN_F7DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F7DATA0_FD22_Pos                    (22U)                             
#define CAN_F7DATA0_FD22_Msk                    (0x1U << CAN_F7DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F7DATA0_FD22                        CAN_F7DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F7DATA0_FD23_Pos                    (23U)                             
#define CAN_F7DATA0_FD23_Msk                    (0x1U << CAN_F7DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F7DATA0_FD23                        CAN_F7DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F7DATA0_FD24_Pos                    (24U)                             
#define CAN_F7DATA0_FD24_Msk                    (0x1U << CAN_F7DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F7DATA0_FD24                        CAN_F7DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F7DATA0_FD25_Pos                    (25U)                             
#define CAN_F7DATA0_FD25_Msk                    (0x1U << CAN_F7DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F7DATA0_FD25                        CAN_F7DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F7DATA0_FD26_Pos                    (26U)                             
#define CAN_F7DATA0_FD26_Msk                    (0x1U << CAN_F7DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F7DATA0_FD26                        CAN_F7DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F7DATA0_FD27_Pos                    (27U)                             
#define CAN_F7DATA0_FD27_Msk                    (0x1U << CAN_F7DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F7DATA0_FD27                        CAN_F7DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F7DATA0_FD28_Pos                    (28U)                             
#define CAN_F7DATA0_FD28_Msk                    (0x1U << CAN_F7DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F7DATA0_FD28                        CAN_F7DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F7DATA0_FD29_Pos                    (29U)                             
#define CAN_F7DATA0_FD29_Msk                    (0x1U << CAN_F7DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F7DATA0_FD29                        CAN_F7DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F7DATA0_FD30_Pos                    (30U)                             
#define CAN_F7DATA0_FD30_Msk                    (0x1U << CAN_F7DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F7DATA0_FD30                        CAN_F7DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F7DATA0_FD31_Pos                    (31U)                             
#define CAN_F7DATA0_FD31_Msk                    (0x1U << CAN_F7DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F7DATA0_FD31                        CAN_F7DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F8R1 register  *******************/
#define CAN_F8DATA0_FD0_Pos                     (0U)                              
#define CAN_F8DATA0_FD0_Msk                     (0x1U << CAN_F8DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F8DATA0_FD0                         CAN_F8DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F8DATA0_FD1_Pos                     (1U)                              
#define CAN_F8DATA0_FD1_Msk                     (0x1U << CAN_F8DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F8DATA0_FD1                         CAN_F8DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F8DATA0_FD2_Pos                     (2U)                              
#define CAN_F8DATA0_FD2_Msk                     (0x1U << CAN_F8DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F8DATA0_FD2                         CAN_F8DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F8DATA0_FD3_Pos                     (3U)                              
#define CAN_F8DATA0_FD3_Msk                     (0x1U << CAN_F8DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F8DATA0_FD3                         CAN_F8DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F8DATA0_FD4_Pos                     (4U)                              
#define CAN_F8DATA0_FD4_Msk                     (0x1U << CAN_F8DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F8DATA0_FD4                         CAN_F8DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F8DATA0_FD5_Pos                     (5U)                              
#define CAN_F8DATA0_FD5_Msk                     (0x1U << CAN_F8DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F8DATA0_FD5                         CAN_F8DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F8DATA0_FD6_Pos                     (6U)                              
#define CAN_F8DATA0_FD6_Msk                     (0x1U << CAN_F8DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F8DATA0_FD6                         CAN_F8DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F8DATA0_FD7_Pos                     (7U)                              
#define CAN_F8DATA0_FD7_Msk                     (0x1U << CAN_F8DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F8DATA0_FD7                         CAN_F8DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F8DATA0_FD8_Pos                     (8U)                              
#define CAN_F8DATA0_FD8_Msk                     (0x1U << CAN_F8DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F8DATA0_FD8                         CAN_F8DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F8DATA0_FD9_Pos                     (9U)                              
#define CAN_F8DATA0_FD9_Msk                     (0x1U << CAN_F8DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F8DATA0_FD9                         CAN_F8DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F8DATA0_FD10_Pos                    (10U)                             
#define CAN_F8DATA0_FD10_Msk                    (0x1U << CAN_F8DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F8DATA0_FD10                        CAN_F8DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F8DATA0_FD11_Pos                    (11U)                             
#define CAN_F8DATA0_FD11_Msk                    (0x1U << CAN_F8DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F8DATA0_FD11                        CAN_F8DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F8DATA0_FD12_Pos                    (12U)                             
#define CAN_F8DATA0_FD12_Msk                    (0x1U << CAN_F8DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F8DATA0_FD12                        CAN_F8DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F8DATA0_FD13_Pos                    (13U)                             
#define CAN_F8DATA0_FD13_Msk                    (0x1U << CAN_F8DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F8DATA0_FD13                        CAN_F8DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F8DATA0_FD14_Pos                    (14U)                             
#define CAN_F8DATA0_FD14_Msk                    (0x1U << CAN_F8DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F8DATA0_FD14                        CAN_F8DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F8DATA0_FD15_Pos                    (15U)                             
#define CAN_F8DATA0_FD15_Msk                    (0x1U << CAN_F8DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F8DATA0_FD15                        CAN_F8DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F8DATA0_FD16_Pos                    (16U)                             
#define CAN_F8DATA0_FD16_Msk                    (0x1U << CAN_F8DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F8DATA0_FD16                        CAN_F8DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F8DATA0_FD17_Pos                    (17U)                             
#define CAN_F8DATA0_FD17_Msk                    (0x1U << CAN_F8DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F8DATA0_FD17                        CAN_F8DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F8DATA0_FD18_Pos                    (18U)                             
#define CAN_F8DATA0_FD18_Msk                    (0x1U << CAN_F8DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F8DATA0_FD18                        CAN_F8DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F8DATA0_FD19_Pos                    (19U)                             
#define CAN_F8DATA0_FD19_Msk                    (0x1U << CAN_F8DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F8DATA0_FD19                        CAN_F8DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F8DATA0_FD20_Pos                    (20U)                             
#define CAN_F8DATA0_FD20_Msk                    (0x1U << CAN_F8DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F8DATA0_FD20                        CAN_F8DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F8DATA0_FD21_Pos                    (21U)                             
#define CAN_F8DATA0_FD21_Msk                    (0x1U << CAN_F8DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F8DATA0_FD21                        CAN_F8DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F8DATA0_FD22_Pos                    (22U)                             
#define CAN_F8DATA0_FD22_Msk                    (0x1U << CAN_F8DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F8DATA0_FD22                        CAN_F8DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F8DATA0_FD23_Pos                    (23U)                             
#define CAN_F8DATA0_FD23_Msk                    (0x1U << CAN_F8DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F8DATA0_FD23                        CAN_F8DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F8DATA0_FD24_Pos                    (24U)                             
#define CAN_F8DATA0_FD24_Msk                    (0x1U << CAN_F8DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F8DATA0_FD24                        CAN_F8DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F8DATA0_FD25_Pos                    (25U)                             
#define CAN_F8DATA0_FD25_Msk                    (0x1U << CAN_F8DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F8DATA0_FD25                        CAN_F8DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F8DATA0_FD26_Pos                    (26U)                             
#define CAN_F8DATA0_FD26_Msk                    (0x1U << CAN_F8DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F8DATA0_FD26                        CAN_F8DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F8DATA0_FD27_Pos                    (27U)                             
#define CAN_F8DATA0_FD27_Msk                    (0x1U << CAN_F8DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F8DATA0_FD27                        CAN_F8DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F8DATA0_FD28_Pos                    (28U)                             
#define CAN_F8DATA0_FD28_Msk                    (0x1U << CAN_F8DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F8DATA0_FD28                        CAN_F8DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F8DATA0_FD29_Pos                    (29U)                             
#define CAN_F8DATA0_FD29_Msk                    (0x1U << CAN_F8DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F8DATA0_FD29                        CAN_F8DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F8DATA0_FD30_Pos                    (30U)                             
#define CAN_F8DATA0_FD30_Msk                    (0x1U << CAN_F8DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F8DATA0_FD30                        CAN_F8DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F8DATA0_FD31_Pos                    (31U)                             
#define CAN_F8DATA0_FD31_Msk                    (0x1U << CAN_F8DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F8DATA0_FD31                        CAN_F8DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F9R1 register  *******************/
#define CAN_F9DATA0_FD0_Pos                     (0U)                              
#define CAN_F9DATA0_FD0_Msk                     (0x1U << CAN_F9DATA0_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F9DATA0_FD0                         CAN_F9DATA0_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F9DATA0_FD1_Pos                     (1U)                              
#define CAN_F9DATA0_FD1_Msk                     (0x1U << CAN_F9DATA0_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F9DATA0_FD1                         CAN_F9DATA0_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F9DATA0_FD2_Pos                     (2U)                              
#define CAN_F9DATA0_FD2_Msk                     (0x1U << CAN_F9DATA0_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F9DATA0_FD2                         CAN_F9DATA0_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F9DATA0_FD3_Pos                     (3U)                              
#define CAN_F9DATA0_FD3_Msk                     (0x1U << CAN_F9DATA0_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F9DATA0_FD3                         CAN_F9DATA0_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F9DATA0_FD4_Pos                     (4U)                              
#define CAN_F9DATA0_FD4_Msk                     (0x1U << CAN_F9DATA0_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F9DATA0_FD4                         CAN_F9DATA0_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F9DATA0_FD5_Pos                     (5U)                              
#define CAN_F9DATA0_FD5_Msk                     (0x1U << CAN_F9DATA0_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F9DATA0_FD5                         CAN_F9DATA0_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F9DATA0_FD6_Pos                     (6U)                              
#define CAN_F9DATA0_FD6_Msk                     (0x1U << CAN_F9DATA0_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F9DATA0_FD6                         CAN_F9DATA0_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F9DATA0_FD7_Pos                     (7U)                              
#define CAN_F9DATA0_FD7_Msk                     (0x1U << CAN_F9DATA0_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F9DATA0_FD7                         CAN_F9DATA0_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F9DATA0_FD8_Pos                     (8U)                              
#define CAN_F9DATA0_FD8_Msk                     (0x1U << CAN_F9DATA0_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F9DATA0_FD8                         CAN_F9DATA0_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F9DATA0_FD9_Pos                     (9U)                              
#define CAN_F9DATA0_FD9_Msk                     (0x1U << CAN_F9DATA0_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F9DATA0_FD9                         CAN_F9DATA0_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F9DATA0_FD10_Pos                    (10U)                             
#define CAN_F9DATA0_FD10_Msk                    (0x1U << CAN_F9DATA0_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F9DATA0_FD10                        CAN_F9DATA0_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F9DATA0_FD11_Pos                    (11U)                             
#define CAN_F9DATA0_FD11_Msk                    (0x1U << CAN_F9DATA0_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F9DATA0_FD11                        CAN_F9DATA0_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F9DATA0_FD12_Pos                    (12U)                             
#define CAN_F9DATA0_FD12_Msk                    (0x1U << CAN_F9DATA0_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F9DATA0_FD12                        CAN_F9DATA0_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F9DATA0_FD13_Pos                    (13U)                             
#define CAN_F9DATA0_FD13_Msk                    (0x1U << CAN_F9DATA0_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F9DATA0_FD13                        CAN_F9DATA0_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F9DATA0_FD14_Pos                    (14U)                             
#define CAN_F9DATA0_FD14_Msk                    (0x1U << CAN_F9DATA0_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F9DATA0_FD14                        CAN_F9DATA0_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F9DATA0_FD15_Pos                    (15U)                             
#define CAN_F9DATA0_FD15_Msk                    (0x1U << CAN_F9DATA0_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F9DATA0_FD15                        CAN_F9DATA0_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F9DATA0_FD16_Pos                    (16U)                             
#define CAN_F9DATA0_FD16_Msk                    (0x1U << CAN_F9DATA0_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F9DATA0_FD16                        CAN_F9DATA0_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F9DATA0_FD17_Pos                    (17U)                             
#define CAN_F9DATA0_FD17_Msk                    (0x1U << CAN_F9DATA0_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F9DATA0_FD17                        CAN_F9DATA0_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F9DATA0_FD18_Pos                    (18U)                             
#define CAN_F9DATA0_FD18_Msk                    (0x1U << CAN_F9DATA0_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F9DATA0_FD18                        CAN_F9DATA0_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F9DATA0_FD19_Pos                    (19U)                             
#define CAN_F9DATA0_FD19_Msk                    (0x1U << CAN_F9DATA0_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F9DATA0_FD19                        CAN_F9DATA0_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F9DATA0_FD20_Pos                    (20U)                             
#define CAN_F9DATA0_FD20_Msk                    (0x1U << CAN_F9DATA0_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F9DATA0_FD20                        CAN_F9DATA0_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F9DATA0_FD21_Pos                    (21U)                             
#define CAN_F9DATA0_FD21_Msk                    (0x1U << CAN_F9DATA0_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F9DATA0_FD21                        CAN_F9DATA0_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F9DATA0_FD22_Pos                    (22U)                             
#define CAN_F9DATA0_FD22_Msk                    (0x1U << CAN_F9DATA0_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F9DATA0_FD22                        CAN_F9DATA0_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F9DATA0_FD23_Pos                    (23U)                             
#define CAN_F9DATA0_FD23_Msk                    (0x1U << CAN_F9DATA0_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F9DATA0_FD23                        CAN_F9DATA0_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F9DATA0_FD24_Pos                    (24U)                             
#define CAN_F9DATA0_FD24_Msk                    (0x1U << CAN_F9DATA0_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F9DATA0_FD24                        CAN_F9DATA0_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F9DATA0_FD25_Pos                    (25U)                             
#define CAN_F9DATA0_FD25_Msk                    (0x1U << CAN_F9DATA0_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F9DATA0_FD25                        CAN_F9DATA0_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F9DATA0_FD26_Pos                    (26U)                             
#define CAN_F9DATA0_FD26_Msk                    (0x1U << CAN_F9DATA0_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F9DATA0_FD26                        CAN_F9DATA0_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F9DATA0_FD27_Pos                    (27U)                             
#define CAN_F9DATA0_FD27_Msk                    (0x1U << CAN_F9DATA0_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F9DATA0_FD27                        CAN_F9DATA0_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F9DATA0_FD28_Pos                    (28U)                             
#define CAN_F9DATA0_FD28_Msk                    (0x1U << CAN_F9DATA0_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F9DATA0_FD28                        CAN_F9DATA0_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F9DATA0_FD29_Pos                    (29U)                             
#define CAN_F9DATA0_FD29_Msk                    (0x1U << CAN_F9DATA0_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F9DATA0_FD29                        CAN_F9DATA0_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F9DATA0_FD30_Pos                    (30U)                             
#define CAN_F9DATA0_FD30_Msk                    (0x1U << CAN_F9DATA0_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F9DATA0_FD30                        CAN_F9DATA0_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F9DATA0_FD31_Pos                    (31U)                             
#define CAN_F9DATA0_FD31_Msk                    (0x1U << CAN_F9DATA0_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F9DATA0_FD31                        CAN_F9DATA0_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F10R1 register  ******************/
#define CAN_F10DATA0_FD0_Pos                    (0U)                              
#define CAN_F10DATA0_FD0_Msk                    (0x1U << CAN_F10DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F10DATA0_FD0                        CAN_F10DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F10DATA0_FD1_Pos                    (1U)                              
#define CAN_F10DATA0_FD1_Msk                    (0x1U << CAN_F10DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F10DATA0_FD1                        CAN_F10DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F10DATA0_FD2_Pos                    (2U)                              
#define CAN_F10DATA0_FD2_Msk                    (0x1U << CAN_F10DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F10DATA0_FD2                        CAN_F10DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F10DATA0_FD3_Pos                    (3U)                              
#define CAN_F10DATA0_FD3_Msk                    (0x1U << CAN_F10DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F10DATA0_FD3                        CAN_F10DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F10DATA0_FD4_Pos                    (4U)                              
#define CAN_F10DATA0_FD4_Msk                    (0x1U << CAN_F10DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F10DATA0_FD4                        CAN_F10DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F10DATA0_FD5_Pos                    (5U)                              
#define CAN_F10DATA0_FD5_Msk                    (0x1U << CAN_F10DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F10DATA0_FD5                        CAN_F10DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F10DATA0_FD6_Pos                    (6U)                              
#define CAN_F10DATA0_FD6_Msk                    (0x1U << CAN_F10DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F10DATA0_FD6                        CAN_F10DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F10DATA0_FD7_Pos                    (7U)                              
#define CAN_F10DATA0_FD7_Msk                    (0x1U << CAN_F10DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F10DATA0_FD7                        CAN_F10DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F10DATA0_FD8_Pos                    (8U)                              
#define CAN_F10DATA0_FD8_Msk                    (0x1U << CAN_F10DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F10DATA0_FD8                        CAN_F10DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F10DATA0_FD9_Pos                    (9U)                              
#define CAN_F10DATA0_FD9_Msk                    (0x1U << CAN_F10DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F10DATA0_FD9                        CAN_F10DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F10DATA0_FD10_Pos                   (10U)                             
#define CAN_F10DATA0_FD10_Msk                   (0x1U << CAN_F10DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F10DATA0_FD10                       CAN_F10DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F10DATA0_FD11_Pos                   (11U)                             
#define CAN_F10DATA0_FD11_Msk                   (0x1U << CAN_F10DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F10DATA0_FD11                       CAN_F10DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F10DATA0_FD12_Pos                   (12U)                             
#define CAN_F10DATA0_FD12_Msk                   (0x1U << CAN_F10DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F10DATA0_FD12                       CAN_F10DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F10DATA0_FD13_Pos                   (13U)                             
#define CAN_F10DATA0_FD13_Msk                   (0x1U << CAN_F10DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F10DATA0_FD13                       CAN_F10DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F10DATA0_FD14_Pos                   (14U)                             
#define CAN_F10DATA0_FD14_Msk                   (0x1U << CAN_F10DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F10DATA0_FD14                       CAN_F10DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F10DATA0_FD15_Pos                   (15U)                             
#define CAN_F10DATA0_FD15_Msk                   (0x1U << CAN_F10DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F10DATA0_FD15                       CAN_F10DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F10DATA0_FD16_Pos                   (16U)                             
#define CAN_F10DATA0_FD16_Msk                   (0x1U << CAN_F10DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F10DATA0_FD16                       CAN_F10DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F10DATA0_FD17_Pos                   (17U)                             
#define CAN_F10DATA0_FD17_Msk                   (0x1U << CAN_F10DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F10DATA0_FD17                       CAN_F10DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F10DATA0_FD18_Pos                   (18U)                             
#define CAN_F10DATA0_FD18_Msk                   (0x1U << CAN_F10DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F10DATA0_FD18                       CAN_F10DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F10DATA0_FD19_Pos                   (19U)                             
#define CAN_F10DATA0_FD19_Msk                   (0x1U << CAN_F10DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F10DATA0_FD19                       CAN_F10DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F10DATA0_FD20_Pos                   (20U)                             
#define CAN_F10DATA0_FD20_Msk                   (0x1U << CAN_F10DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F10DATA0_FD20                       CAN_F10DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F10DATA0_FD21_Pos                   (21U)                             
#define CAN_F10DATA0_FD21_Msk                   (0x1U << CAN_F10DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F10DATA0_FD21                       CAN_F10DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F10DATA0_FD22_Pos                   (22U)                             
#define CAN_F10DATA0_FD22_Msk                   (0x1U << CAN_F10DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F10DATA0_FD22                       CAN_F10DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F10DATA0_FD23_Pos                   (23U)                             
#define CAN_F10DATA0_FD23_Msk                   (0x1U << CAN_F10DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F10DATA0_FD23                       CAN_F10DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F10DATA0_FD24_Pos                   (24U)                             
#define CAN_F10DATA0_FD24_Msk                   (0x1U << CAN_F10DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F10DATA0_FD24                       CAN_F10DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F10DATA0_FD25_Pos                   (25U)                             
#define CAN_F10DATA0_FD25_Msk                   (0x1U << CAN_F10DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F10DATA0_FD25                       CAN_F10DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F10DATA0_FD26_Pos                   (26U)                             
#define CAN_F10DATA0_FD26_Msk                   (0x1U << CAN_F10DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F10DATA0_FD26                       CAN_F10DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F10DATA0_FD27_Pos                   (27U)                             
#define CAN_F10DATA0_FD27_Msk                   (0x1U << CAN_F10DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F10DATA0_FD27                       CAN_F10DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F10DATA0_FD28_Pos                   (28U)                             
#define CAN_F10DATA0_FD28_Msk                   (0x1U << CAN_F10DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F10DATA0_FD28                       CAN_F10DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F10DATA0_FD29_Pos                   (29U)                             
#define CAN_F10DATA0_FD29_Msk                   (0x1U << CAN_F10DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F10DATA0_FD29                       CAN_F10DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F10DATA0_FD30_Pos                   (30U)                             
#define CAN_F10DATA0_FD30_Msk                   (0x1U << CAN_F10DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F10DATA0_FD30                       CAN_F10DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F10DATA0_FD31_Pos                   (31U)                             
#define CAN_F10DATA0_FD31_Msk                   (0x1U << CAN_F10DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F10DATA0_FD31                       CAN_F10DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F11R1 register  ******************/
#define CAN_F11DATA0_FD0_Pos                    (0U)                              
#define CAN_F11DATA0_FD0_Msk                    (0x1U << CAN_F11DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F11DATA0_FD0                        CAN_F11DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F11DATA0_FD1_Pos                    (1U)                              
#define CAN_F11DATA0_FD1_Msk                    (0x1U << CAN_F11DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F11DATA0_FD1                        CAN_F11DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F11DATA0_FD2_Pos                    (2U)                              
#define CAN_F11DATA0_FD2_Msk                    (0x1U << CAN_F11DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F11DATA0_FD2                        CAN_F11DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F11DATA0_FD3_Pos                    (3U)                              
#define CAN_F11DATA0_FD3_Msk                    (0x1U << CAN_F11DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F11DATA0_FD3                        CAN_F11DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F11DATA0_FD4_Pos                    (4U)                              
#define CAN_F11DATA0_FD4_Msk                    (0x1U << CAN_F11DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F11DATA0_FD4                        CAN_F11DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F11DATA0_FD5_Pos                    (5U)                              
#define CAN_F11DATA0_FD5_Msk                    (0x1U << CAN_F11DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F11DATA0_FD5                        CAN_F11DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F11DATA0_FD6_Pos                    (6U)                              
#define CAN_F11DATA0_FD6_Msk                    (0x1U << CAN_F11DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F11DATA0_FD6                        CAN_F11DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F11DATA0_FD7_Pos                    (7U)                              
#define CAN_F11DATA0_FD7_Msk                    (0x1U << CAN_F11DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F11DATA0_FD7                        CAN_F11DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F11DATA0_FD8_Pos                    (8U)                              
#define CAN_F11DATA0_FD8_Msk                    (0x1U << CAN_F11DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F11DATA0_FD8                        CAN_F11DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F11DATA0_FD9_Pos                    (9U)                              
#define CAN_F11DATA0_FD9_Msk                    (0x1U << CAN_F11DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F11DATA0_FD9                        CAN_F11DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F11DATA0_FD10_Pos                   (10U)                             
#define CAN_F11DATA0_FD10_Msk                   (0x1U << CAN_F11DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F11DATA0_FD10                       CAN_F11DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F11DATA0_FD11_Pos                   (11U)                             
#define CAN_F11DATA0_FD11_Msk                   (0x1U << CAN_F11DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F11DATA0_FD11                       CAN_F11DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F11DATA0_FD12_Pos                   (12U)                             
#define CAN_F11DATA0_FD12_Msk                   (0x1U << CAN_F11DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F11DATA0_FD12                       CAN_F11DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F11DATA0_FD13_Pos                   (13U)                             
#define CAN_F11DATA0_FD13_Msk                   (0x1U << CAN_F11DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F11DATA0_FD13                       CAN_F11DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F11DATA0_FD14_Pos                   (14U)                             
#define CAN_F11DATA0_FD14_Msk                   (0x1U << CAN_F11DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F11DATA0_FD14                       CAN_F11DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F11DATA0_FD15_Pos                   (15U)                             
#define CAN_F11DATA0_FD15_Msk                   (0x1U << CAN_F11DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F11DATA0_FD15                       CAN_F11DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F11DATA0_FD16_Pos                   (16U)                             
#define CAN_F11DATA0_FD16_Msk                   (0x1U << CAN_F11DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F11DATA0_FD16                       CAN_F11DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F11DATA0_FD17_Pos                   (17U)                             
#define CAN_F11DATA0_FD17_Msk                   (0x1U << CAN_F11DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F11DATA0_FD17                       CAN_F11DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F11DATA0_FD18_Pos                   (18U)                             
#define CAN_F11DATA0_FD18_Msk                   (0x1U << CAN_F11DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F11DATA0_FD18                       CAN_F11DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F11DATA0_FD19_Pos                   (19U)                             
#define CAN_F11DATA0_FD19_Msk                   (0x1U << CAN_F11DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F11DATA0_FD19                       CAN_F11DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F11DATA0_FD20_Pos                   (20U)                             
#define CAN_F11DATA0_FD20_Msk                   (0x1U << CAN_F11DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F11DATA0_FD20                       CAN_F11DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F11DATA0_FD21_Pos                   (21U)                             
#define CAN_F11DATA0_FD21_Msk                   (0x1U << CAN_F11DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F11DATA0_FD21                       CAN_F11DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F11DATA0_FD22_Pos                   (22U)                             
#define CAN_F11DATA0_FD22_Msk                   (0x1U << CAN_F11DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F11DATA0_FD22                       CAN_F11DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F11DATA0_FD23_Pos                   (23U)                             
#define CAN_F11DATA0_FD23_Msk                   (0x1U << CAN_F11DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F11DATA0_FD23                       CAN_F11DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F11DATA0_FD24_Pos                   (24U)                             
#define CAN_F11DATA0_FD24_Msk                   (0x1U << CAN_F11DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F11DATA0_FD24                       CAN_F11DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F11DATA0_FD25_Pos                   (25U)                             
#define CAN_F11DATA0_FD25_Msk                   (0x1U << CAN_F11DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F11DATA0_FD25                       CAN_F11DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F11DATA0_FD26_Pos                   (26U)                             
#define CAN_F11DATA0_FD26_Msk                   (0x1U << CAN_F11DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F11DATA0_FD26                       CAN_F11DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F11DATA0_FD27_Pos                   (27U)                             
#define CAN_F11DATA0_FD27_Msk                   (0x1U << CAN_F11DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F11DATA0_FD27                       CAN_F11DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F11DATA0_FD28_Pos                   (28U)                             
#define CAN_F11DATA0_FD28_Msk                   (0x1U << CAN_F11DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F11DATA0_FD28                       CAN_F11DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F11DATA0_FD29_Pos                   (29U)                             
#define CAN_F11DATA0_FD29_Msk                   (0x1U << CAN_F11DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F11DATA0_FD29                       CAN_F11DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F11DATA0_FD30_Pos                   (30U)                             
#define CAN_F11DATA0_FD30_Msk                   (0x1U << CAN_F11DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F11DATA0_FD30                       CAN_F11DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F11DATA0_FD31_Pos                   (31U)                             
#define CAN_F11DATA0_FD31_Msk                   (0x1U << CAN_F11DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F11DATA0_FD31                       CAN_F11DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F12R1 register  ******************/
#define CAN_F12DATA0_FD0_Pos                    (0U)                              
#define CAN_F12DATA0_FD0_Msk                    (0x1U << CAN_F12DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F12DATA0_FD0                        CAN_F12DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F12DATA0_FD1_Pos                    (1U)                              
#define CAN_F12DATA0_FD1_Msk                    (0x1U << CAN_F12DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F12DATA0_FD1                        CAN_F12DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F12DATA0_FD2_Pos                    (2U)                              
#define CAN_F12DATA0_FD2_Msk                    (0x1U << CAN_F12DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F12DATA0_FD2                        CAN_F12DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F12DATA0_FD3_Pos                    (3U)                              
#define CAN_F12DATA0_FD3_Msk                    (0x1U << CAN_F12DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F12DATA0_FD3                        CAN_F12DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F12DATA0_FD4_Pos                    (4U)                              
#define CAN_F12DATA0_FD4_Msk                    (0x1U << CAN_F12DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F12DATA0_FD4                        CAN_F12DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F12DATA0_FD5_Pos                    (5U)                              
#define CAN_F12DATA0_FD5_Msk                    (0x1U << CAN_F12DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F12DATA0_FD5                        CAN_F12DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F12DATA0_FD6_Pos                    (6U)                              
#define CAN_F12DATA0_FD6_Msk                    (0x1U << CAN_F12DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F12DATA0_FD6                        CAN_F12DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F12DATA0_FD7_Pos                    (7U)                              
#define CAN_F12DATA0_FD7_Msk                    (0x1U << CAN_F12DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F12DATA0_FD7                        CAN_F12DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F12DATA0_FD8_Pos                    (8U)                              
#define CAN_F12DATA0_FD8_Msk                    (0x1U << CAN_F12DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F12DATA0_FD8                        CAN_F12DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F12DATA0_FD9_Pos                    (9U)                              
#define CAN_F12DATA0_FD9_Msk                    (0x1U << CAN_F12DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F12DATA0_FD9                        CAN_F12DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F12DATA0_FD10_Pos                   (10U)                             
#define CAN_F12DATA0_FD10_Msk                   (0x1U << CAN_F12DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F12DATA0_FD10                       CAN_F12DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F12DATA0_FD11_Pos                   (11U)                             
#define CAN_F12DATA0_FD11_Msk                   (0x1U << CAN_F12DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F12DATA0_FD11                       CAN_F12DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F12DATA0_FD12_Pos                   (12U)                             
#define CAN_F12DATA0_FD12_Msk                   (0x1U << CAN_F12DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F12DATA0_FD12                       CAN_F12DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F12DATA0_FD13_Pos                   (13U)                             
#define CAN_F12DATA0_FD13_Msk                   (0x1U << CAN_F12DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F12DATA0_FD13                       CAN_F12DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F12DATA0_FD14_Pos                   (14U)                             
#define CAN_F12DATA0_FD14_Msk                   (0x1U << CAN_F12DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F12DATA0_FD14                       CAN_F12DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F12DATA0_FD15_Pos                   (15U)                             
#define CAN_F12DATA0_FD15_Msk                   (0x1U << CAN_F12DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F12DATA0_FD15                       CAN_F12DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F12DATA0_FD16_Pos                   (16U)                             
#define CAN_F12DATA0_FD16_Msk                   (0x1U << CAN_F12DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F12DATA0_FD16                       CAN_F12DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F12DATA0_FD17_Pos                   (17U)                             
#define CAN_F12DATA0_FD17_Msk                   (0x1U << CAN_F12DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F12DATA0_FD17                       CAN_F12DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F12DATA0_FD18_Pos                   (18U)                             
#define CAN_F12DATA0_FD18_Msk                   (0x1U << CAN_F12DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F12DATA0_FD18                       CAN_F12DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F12DATA0_FD19_Pos                   (19U)                             
#define CAN_F12DATA0_FD19_Msk                   (0x1U << CAN_F12DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F12DATA0_FD19                       CAN_F12DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F12DATA0_FD20_Pos                   (20U)                             
#define CAN_F12DATA0_FD20_Msk                   (0x1U << CAN_F12DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F12DATA0_FD20                       CAN_F12DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F12DATA0_FD21_Pos                   (21U)                             
#define CAN_F12DATA0_FD21_Msk                   (0x1U << CAN_F12DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F12DATA0_FD21                       CAN_F12DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F12DATA0_FD22_Pos                   (22U)                             
#define CAN_F12DATA0_FD22_Msk                   (0x1U << CAN_F12DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F12DATA0_FD22                       CAN_F12DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F12DATA0_FD23_Pos                   (23U)                             
#define CAN_F12DATA0_FD23_Msk                   (0x1U << CAN_F12DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F12DATA0_FD23                       CAN_F12DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F12DATA0_FD24_Pos                   (24U)                             
#define CAN_F12DATA0_FD24_Msk                   (0x1U << CAN_F12DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F12DATA0_FD24                       CAN_F12DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F12DATA0_FD25_Pos                   (25U)                             
#define CAN_F12DATA0_FD25_Msk                   (0x1U << CAN_F12DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F12DATA0_FD25                       CAN_F12DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F12DATA0_FD26_Pos                   (26U)                             
#define CAN_F12DATA0_FD26_Msk                   (0x1U << CAN_F12DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F12DATA0_FD26                       CAN_F12DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F12DATA0_FD27_Pos                   (27U)                             
#define CAN_F12DATA0_FD27_Msk                   (0x1U << CAN_F12DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F12DATA0_FD27                       CAN_F12DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F12DATA0_FD28_Pos                   (28U)                             
#define CAN_F12DATA0_FD28_Msk                   (0x1U << CAN_F12DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F12DATA0_FD28                       CAN_F12DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F12DATA0_FD29_Pos                   (29U)                             
#define CAN_F12DATA0_FD29_Msk                   (0x1U << CAN_F12DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F12DATA0_FD29                       CAN_F12DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F12DATA0_FD30_Pos                   (30U)                             
#define CAN_F12DATA0_FD30_Msk                   (0x1U << CAN_F12DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F12DATA0_FD30                       CAN_F12DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F12DATA0_FD31_Pos                   (31U)                             
#define CAN_F12DATA0_FD31_Msk                   (0x1U << CAN_F12DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F12DATA0_FD31                       CAN_F12DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F13R1 register  ******************/
#define CAN_F13DATA0_FD0_Pos                    (0U)                              
#define CAN_F13DATA0_FD0_Msk                    (0x1U << CAN_F13DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F13DATA0_FD0                        CAN_F13DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F13DATA0_FD1_Pos                    (1U)                              
#define CAN_F13DATA0_FD1_Msk                    (0x1U << CAN_F13DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F13DATA0_FD1                        CAN_F13DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F13DATA0_FD2_Pos                    (2U)                              
#define CAN_F13DATA0_FD2_Msk                    (0x1U << CAN_F13DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F13DATA0_FD2                        CAN_F13DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F13DATA0_FD3_Pos                    (3U)                              
#define CAN_F13DATA0_FD3_Msk                    (0x1U << CAN_F13DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F13DATA0_FD3                        CAN_F13DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F13DATA0_FD4_Pos                    (4U)                              
#define CAN_F13DATA0_FD4_Msk                    (0x1U << CAN_F13DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F13DATA0_FD4                        CAN_F13DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F13DATA0_FD5_Pos                    (5U)                              
#define CAN_F13DATA0_FD5_Msk                    (0x1U << CAN_F13DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F13DATA0_FD5                        CAN_F13DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F13DATA0_FD6_Pos                    (6U)                              
#define CAN_F13DATA0_FD6_Msk                    (0x1U << CAN_F13DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F13DATA0_FD6                        CAN_F13DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F13DATA0_FD7_Pos                    (7U)                              
#define CAN_F13DATA0_FD7_Msk                    (0x1U << CAN_F13DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F13DATA0_FD7                        CAN_F13DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F13DATA0_FD8_Pos                    (8U)                              
#define CAN_F13DATA0_FD8_Msk                    (0x1U << CAN_F13DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F13DATA0_FD8                        CAN_F13DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F13DATA0_FD9_Pos                    (9U)                              
#define CAN_F13DATA0_FD9_Msk                    (0x1U << CAN_F13DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F13DATA0_FD9                        CAN_F13DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F13DATA0_FD10_Pos                   (10U)                             
#define CAN_F13DATA0_FD10_Msk                   (0x1U << CAN_F13DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F13DATA0_FD10                       CAN_F13DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F13DATA0_FD11_Pos                   (11U)                             
#define CAN_F13DATA0_FD11_Msk                   (0x1U << CAN_F13DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F13DATA0_FD11                       CAN_F13DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F13DATA0_FD12_Pos                   (12U)                             
#define CAN_F13DATA0_FD12_Msk                   (0x1U << CAN_F13DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F13DATA0_FD12                       CAN_F13DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F13DATA0_FD13_Pos                   (13U)                             
#define CAN_F13DATA0_FD13_Msk                   (0x1U << CAN_F13DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F13DATA0_FD13                       CAN_F13DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F13DATA0_FD14_Pos                   (14U)                             
#define CAN_F13DATA0_FD14_Msk                   (0x1U << CAN_F13DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F13DATA0_FD14                       CAN_F13DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F13DATA0_FD15_Pos                   (15U)                             
#define CAN_F13DATA0_FD15_Msk                   (0x1U << CAN_F13DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F13DATA0_FD15                       CAN_F13DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F13DATA0_FD16_Pos                   (16U)                             
#define CAN_F13DATA0_FD16_Msk                   (0x1U << CAN_F13DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F13DATA0_FD16                       CAN_F13DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F13DATA0_FD17_Pos                   (17U)                             
#define CAN_F13DATA0_FD17_Msk                   (0x1U << CAN_F13DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F13DATA0_FD17                       CAN_F13DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F13DATA0_FD18_Pos                   (18U)                             
#define CAN_F13DATA0_FD18_Msk                   (0x1U << CAN_F13DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F13DATA0_FD18                       CAN_F13DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F13DATA0_FD19_Pos                   (19U)                             
#define CAN_F13DATA0_FD19_Msk                   (0x1U << CAN_F13DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F13DATA0_FD19                       CAN_F13DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F13DATA0_FD20_Pos                   (20U)                             
#define CAN_F13DATA0_FD20_Msk                   (0x1U << CAN_F13DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F13DATA0_FD20                       CAN_F13DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F13DATA0_FD21_Pos                   (21U)                             
#define CAN_F13DATA0_FD21_Msk                   (0x1U << CAN_F13DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F13DATA0_FD21                       CAN_F13DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F13DATA0_FD22_Pos                   (22U)                             
#define CAN_F13DATA0_FD22_Msk                   (0x1U << CAN_F13DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F13DATA0_FD22                       CAN_F13DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F13DATA0_FD23_Pos                   (23U)                             
#define CAN_F13DATA0_FD23_Msk                   (0x1U << CAN_F13DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F13DATA0_FD23                       CAN_F13DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F13DATA0_FD24_Pos                   (24U)                             
#define CAN_F13DATA0_FD24_Msk                   (0x1U << CAN_F13DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F13DATA0_FD24                       CAN_F13DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F13DATA0_FD25_Pos                   (25U)                             
#define CAN_F13DATA0_FD25_Msk                   (0x1U << CAN_F13DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F13DATA0_FD25                       CAN_F13DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F13DATA0_FD26_Pos                   (26U)                             
#define CAN_F13DATA0_FD26_Msk                   (0x1U << CAN_F13DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F13DATA0_FD26                       CAN_F13DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F13DATA0_FD27_Pos                   (27U)                             
#define CAN_F13DATA0_FD27_Msk                   (0x1U << CAN_F13DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F13DATA0_FD27                       CAN_F13DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F13DATA0_FD28_Pos                   (28U)                             
#define CAN_F13DATA0_FD28_Msk                   (0x1U << CAN_F13DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F13DATA0_FD28                       CAN_F13DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F13DATA0_FD29_Pos                   (29U)                             
#define CAN_F13DATA0_FD29_Msk                   (0x1U << CAN_F13DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F13DATA0_FD29                       CAN_F13DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F13DATA0_FD30_Pos                   (30U)                             
#define CAN_F13DATA0_FD30_Msk                   (0x1U << CAN_F13DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F13DATA0_FD30                       CAN_F13DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F13DATA0_FD31_Pos                   (31U)                             
#define CAN_F13DATA0_FD31_Msk                   (0x1U << CAN_F13DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F13DATA0_FD31                       CAN_F13DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F14R1 register  ******************/
#define CAN_F14DATA0_FD0_Pos                    (0U)                              
#define CAN_F14DATA0_FD0_Msk                    (0x1U << CAN_F14DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F14DATA0_FD0                        CAN_F14DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F14DATA0_FD1_Pos                    (1U)                              
#define CAN_F14DATA0_FD1_Msk                    (0x1U << CAN_F14DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F14DATA0_FD1                        CAN_F14DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F14DATA0_FD2_Pos                    (2U)                              
#define CAN_F14DATA0_FD2_Msk                    (0x1U << CAN_F14DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F14DATA0_FD2                        CAN_F14DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F14DATA0_FD3_Pos                    (3U)                              
#define CAN_F14DATA0_FD3_Msk                    (0x1U << CAN_F14DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F14DATA0_FD3                        CAN_F14DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F14DATA0_FD4_Pos                    (4U)                              
#define CAN_F14DATA0_FD4_Msk                    (0x1U << CAN_F14DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F14DATA0_FD4                        CAN_F14DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F14DATA0_FD5_Pos                    (5U)                              
#define CAN_F14DATA0_FD5_Msk                    (0x1U << CAN_F14DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F14DATA0_FD5                        CAN_F14DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F14DATA0_FD6_Pos                    (6U)                              
#define CAN_F14DATA0_FD6_Msk                    (0x1U << CAN_F14DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F14DATA0_FD6                        CAN_F14DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F14DATA0_FD7_Pos                    (7U)                              
#define CAN_F14DATA0_FD7_Msk                    (0x1U << CAN_F14DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F14DATA0_FD7                        CAN_F14DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F14DATA0_FD8_Pos                    (8U)                              
#define CAN_F14DATA0_FD8_Msk                    (0x1U << CAN_F14DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F14DATA0_FD8                        CAN_F14DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F14DATA0_FD9_Pos                    (9U)                              
#define CAN_F14DATA0_FD9_Msk                    (0x1U << CAN_F14DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F14DATA0_FD9                        CAN_F14DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F14DATA0_FD10_Pos                   (10U)                             
#define CAN_F14DATA0_FD10_Msk                   (0x1U << CAN_F14DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F14DATA0_FD10                       CAN_F14DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F14DATA0_FD11_Pos                   (11U)                             
#define CAN_F14DATA0_FD11_Msk                   (0x1U << CAN_F14DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F14DATA0_FD11                       CAN_F14DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F14DATA0_FD12_Pos                   (12U)                             
#define CAN_F14DATA0_FD12_Msk                   (0x1U << CAN_F14DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F14DATA0_FD12                       CAN_F14DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F14DATA0_FD13_Pos                   (13U)                             
#define CAN_F14DATA0_FD13_Msk                   (0x1U << CAN_F14DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F14DATA0_FD13                       CAN_F14DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F14DATA0_FD14_Pos                   (14U)                             
#define CAN_F14DATA0_FD14_Msk                   (0x1U << CAN_F14DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F14DATA0_FD14                       CAN_F14DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F14DATA0_FD15_Pos                   (15U)                             
#define CAN_F14DATA0_FD15_Msk                   (0x1U << CAN_F14DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F14DATA0_FD15                       CAN_F14DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F14DATA0_FD16_Pos                   (16U)                             
#define CAN_F14DATA0_FD16_Msk                   (0x1U << CAN_F14DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F14DATA0_FD16                       CAN_F14DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F14DATA0_FD17_Pos                   (17U)                             
#define CAN_F14DATA0_FD17_Msk                   (0x1U << CAN_F14DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F14DATA0_FD17                       CAN_F14DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F14DATA0_FD18_Pos                   (18U)                             
#define CAN_F14DATA0_FD18_Msk                   (0x1U << CAN_F14DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F14DATA0_FD18                       CAN_F14DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F14DATA0_FD19_Pos                   (19U)                             
#define CAN_F14DATA0_FD19_Msk                   (0x1U << CAN_F14DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F14DATA0_FD19                       CAN_F14DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F14DATA0_FD20_Pos                   (20U)                             
#define CAN_F14DATA0_FD20_Msk                   (0x1U << CAN_F14DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F14DATA0_FD20                       CAN_F14DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F14DATA0_FD21_Pos                   (21U)                             
#define CAN_F14DATA0_FD21_Msk                   (0x1U << CAN_F14DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F14DATA0_FD21                       CAN_F14DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F14DATA0_FD22_Pos                   (22U)                             
#define CAN_F14DATA0_FD22_Msk                   (0x1U << CAN_F14DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F14DATA0_FD22                       CAN_F14DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F14DATA0_FD23_Pos                   (23U)                             
#define CAN_F14DATA0_FD23_Msk                   (0x1U << CAN_F14DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F14DATA0_FD23                       CAN_F14DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F14DATA0_FD24_Pos                   (24U)                             
#define CAN_F14DATA0_FD24_Msk                   (0x1U << CAN_F14DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F14DATA0_FD24                       CAN_F14DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F14DATA0_FD25_Pos                   (25U)                             
#define CAN_F14DATA0_FD25_Msk                   (0x1U << CAN_F14DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F14DATA0_FD25                       CAN_F14DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F14DATA0_FD26_Pos                   (26U)                             
#define CAN_F14DATA0_FD26_Msk                   (0x1U << CAN_F14DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F14DATA0_FD26                       CAN_F14DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F14DATA0_FD27_Pos                   (27U)                             
#define CAN_F14DATA0_FD27_Msk                   (0x1U << CAN_F14DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F14DATA0_FD27                       CAN_F14DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F14DATA0_FD28_Pos                   (28U)                             
#define CAN_F14DATA0_FD28_Msk                   (0x1U << CAN_F14DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F14DATA0_FD28                       CAN_F14DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F14DATA0_FD29_Pos                   (29U)                             
#define CAN_F14DATA0_FD29_Msk                   (0x1U << CAN_F14DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F14DATA0_FD29                       CAN_F14DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F14DATA0_FD30_Pos                   (30U)                             
#define CAN_F14DATA0_FD30_Msk                   (0x1U << CAN_F14DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F14DATA0_FD30                       CAN_F14DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F14DATA0_FD31_Pos                   (31U)                             
#define CAN_F14DATA0_FD31_Msk                   (0x1U << CAN_F14DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F14DATA0_FD31                       CAN_F14DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F15R1 register  ******************/
#define CAN_F15DATA0_FD0_Pos                    (0U)                              
#define CAN_F15DATA0_FD0_Msk                    (0x1U << CAN_F15DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F15DATA0_FD0                        CAN_F15DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F15DATA0_FD1_Pos                    (1U)                              
#define CAN_F15DATA0_FD1_Msk                    (0x1U << CAN_F15DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F15DATA0_FD1                        CAN_F15DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F15DATA0_FD2_Pos                    (2U)                              
#define CAN_F15DATA0_FD2_Msk                    (0x1U << CAN_F15DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F15DATA0_FD2                        CAN_F15DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F15DATA0_FD3_Pos                    (3U)                              
#define CAN_F15DATA0_FD3_Msk                    (0x1U << CAN_F15DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F15DATA0_FD3                        CAN_F15DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F15DATA0_FD4_Pos                    (4U)                              
#define CAN_F15DATA0_FD4_Msk                    (0x1U << CAN_F15DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F15DATA0_FD4                        CAN_F15DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F15DATA0_FD5_Pos                    (5U)                              
#define CAN_F15DATA0_FD5_Msk                    (0x1U << CAN_F15DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F15DATA0_FD5                        CAN_F15DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F15DATA0_FD6_Pos                    (6U)                              
#define CAN_F15DATA0_FD6_Msk                    (0x1U << CAN_F15DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F15DATA0_FD6                        CAN_F15DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F15DATA0_FD7_Pos                    (7U)                              
#define CAN_F15DATA0_FD7_Msk                    (0x1U << CAN_F15DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F15DATA0_FD7                        CAN_F15DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F15DATA0_FD8_Pos                    (8U)                              
#define CAN_F15DATA0_FD8_Msk                    (0x1U << CAN_F15DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F15DATA0_FD8                        CAN_F15DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F15DATA0_FD9_Pos                    (9U)                              
#define CAN_F15DATA0_FD9_Msk                    (0x1U << CAN_F15DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F15DATA0_FD9                        CAN_F15DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F15DATA0_FD10_Pos                   (10U)                             
#define CAN_F15DATA0_FD10_Msk                   (0x1U << CAN_F15DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F15DATA0_FD10                       CAN_F15DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F15DATA0_FD11_Pos                   (11U)                             
#define CAN_F15DATA0_FD11_Msk                   (0x1U << CAN_F15DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F15DATA0_FD11                       CAN_F15DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F15DATA0_FD12_Pos                   (12U)                             
#define CAN_F15DATA0_FD12_Msk                   (0x1U << CAN_F15DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F15DATA0_FD12                       CAN_F15DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F15DATA0_FD13_Pos                   (13U)                             
#define CAN_F15DATA0_FD13_Msk                   (0x1U << CAN_F15DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F15DATA0_FD13                       CAN_F15DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F15DATA0_FD14_Pos                   (14U)                             
#define CAN_F15DATA0_FD14_Msk                   (0x1U << CAN_F15DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F15DATA0_FD14                       CAN_F15DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F15DATA0_FD15_Pos                   (15U)                             
#define CAN_F15DATA0_FD15_Msk                   (0x1U << CAN_F15DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F15DATA0_FD15                       CAN_F15DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F15DATA0_FD16_Pos                   (16U)                             
#define CAN_F15DATA0_FD16_Msk                   (0x1U << CAN_F15DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F15DATA0_FD16                       CAN_F15DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F15DATA0_FD17_Pos                   (17U)                             
#define CAN_F15DATA0_FD17_Msk                   (0x1U << CAN_F15DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F15DATA0_FD17                       CAN_F15DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F15DATA0_FD18_Pos                   (18U)                             
#define CAN_F15DATA0_FD18_Msk                   (0x1U << CAN_F15DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F15DATA0_FD18                       CAN_F15DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F15DATA0_FD19_Pos                   (19U)                             
#define CAN_F15DATA0_FD19_Msk                   (0x1U << CAN_F15DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F15DATA0_FD19                       CAN_F15DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F15DATA0_FD20_Pos                   (20U)                             
#define CAN_F15DATA0_FD20_Msk                   (0x1U << CAN_F15DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F15DATA0_FD20                       CAN_F15DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F15DATA0_FD21_Pos                   (21U)                             
#define CAN_F15DATA0_FD21_Msk                   (0x1U << CAN_F15DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F15DATA0_FD21                       CAN_F15DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F15DATA0_FD22_Pos                   (22U)                             
#define CAN_F15DATA0_FD22_Msk                   (0x1U << CAN_F15DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F15DATA0_FD22                       CAN_F15DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F15DATA0_FD23_Pos                   (23U)                             
#define CAN_F15DATA0_FD23_Msk                   (0x1U << CAN_F15DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F15DATA0_FD23                       CAN_F15DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F15DATA0_FD24_Pos                   (24U)                             
#define CAN_F15DATA0_FD24_Msk                   (0x1U << CAN_F15DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F15DATA0_FD24                       CAN_F15DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F15DATA0_FD25_Pos                   (25U)                             
#define CAN_F15DATA0_FD25_Msk                   (0x1U << CAN_F15DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F15DATA0_FD25                       CAN_F15DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F15DATA0_FD26_Pos                   (26U)                             
#define CAN_F15DATA0_FD26_Msk                   (0x1U << CAN_F15DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F15DATA0_FD26                       CAN_F15DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F15DATA0_FD27_Pos                   (27U)                             
#define CAN_F15DATA0_FD27_Msk                   (0x1U << CAN_F15DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F15DATA0_FD27                       CAN_F15DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F15DATA0_FD28_Pos                   (28U)                             
#define CAN_F15DATA0_FD28_Msk                   (0x1U << CAN_F15DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F15DATA0_FD28                       CAN_F15DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F15DATA0_FD29_Pos                   (29U)                             
#define CAN_F15DATA0_FD29_Msk                   (0x1U << CAN_F15DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F15DATA0_FD29                       CAN_F15DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F15DATA0_FD30_Pos                   (30U)                             
#define CAN_F15DATA0_FD30_Msk                   (0x1U << CAN_F15DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F15DATA0_FD30                       CAN_F15DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F15DATA0_FD31_Pos                   (31U)                             
#define CAN_F15DATA0_FD31_Msk                   (0x1U << CAN_F15DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F15DATA0_FD31                       CAN_F15DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F16R1 register  ******************/
#define CAN_F16DATA0_FD0_Pos                    (0U)                              
#define CAN_F16DATA0_FD0_Msk                    (0x1U << CAN_F16DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F16DATA0_FD0                        CAN_F16DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F16DATA0_FD1_Pos                    (1U)                              
#define CAN_F16DATA0_FD1_Msk                    (0x1U << CAN_F16DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F16DATA0_FD1                        CAN_F16DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F16DATA0_FD2_Pos                    (2U)                              
#define CAN_F16DATA0_FD2_Msk                    (0x1U << CAN_F16DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F16DATA0_FD2                        CAN_F16DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F16DATA0_FD3_Pos                    (3U)                              
#define CAN_F16DATA0_FD3_Msk                    (0x1U << CAN_F16DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F16DATA0_FD3                        CAN_F16DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F16DATA0_FD4_Pos                    (4U)                              
#define CAN_F16DATA0_FD4_Msk                    (0x1U << CAN_F16DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F16DATA0_FD4                        CAN_F16DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F16DATA0_FD5_Pos                    (5U)                              
#define CAN_F16DATA0_FD5_Msk                    (0x1U << CAN_F16DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F16DATA0_FD5                        CAN_F16DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F16DATA0_FD6_Pos                    (6U)                              
#define CAN_F16DATA0_FD6_Msk                    (0x1U << CAN_F16DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F16DATA0_FD6                        CAN_F16DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F16DATA0_FD7_Pos                    (7U)                              
#define CAN_F16DATA0_FD7_Msk                    (0x1U << CAN_F16DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F16DATA0_FD7                        CAN_F16DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F16DATA0_FD8_Pos                    (8U)                              
#define CAN_F16DATA0_FD8_Msk                    (0x1U << CAN_F16DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F16DATA0_FD8                        CAN_F16DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F16DATA0_FD9_Pos                    (9U)                              
#define CAN_F16DATA0_FD9_Msk                    (0x1U << CAN_F16DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F16DATA0_FD9                        CAN_F16DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F16DATA0_FD10_Pos                   (10U)                             
#define CAN_F16DATA0_FD10_Msk                   (0x1U << CAN_F16DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F16DATA0_FD10                       CAN_F16DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F16DATA0_FD11_Pos                   (11U)                             
#define CAN_F16DATA0_FD11_Msk                   (0x1U << CAN_F16DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F16DATA0_FD11                       CAN_F16DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F16DATA0_FD12_Pos                   (12U)                             
#define CAN_F16DATA0_FD12_Msk                   (0x1U << CAN_F16DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F16DATA0_FD12                       CAN_F16DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F16DATA0_FD13_Pos                   (13U)                             
#define CAN_F16DATA0_FD13_Msk                   (0x1U << CAN_F16DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F16DATA0_FD13                       CAN_F16DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F16DATA0_FD14_Pos                   (14U)                             
#define CAN_F16DATA0_FD14_Msk                   (0x1U << CAN_F16DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F16DATA0_FD14                       CAN_F16DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F16DATA0_FD15_Pos                   (15U)                             
#define CAN_F16DATA0_FD15_Msk                   (0x1U << CAN_F16DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F16DATA0_FD15                       CAN_F16DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F16DATA0_FD16_Pos                   (16U)                             
#define CAN_F16DATA0_FD16_Msk                   (0x1U << CAN_F16DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F16DATA0_FD16                       CAN_F16DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F16DATA0_FD17_Pos                   (17U)                             
#define CAN_F16DATA0_FD17_Msk                   (0x1U << CAN_F16DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F16DATA0_FD17                       CAN_F16DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F16DATA0_FD18_Pos                   (18U)                             
#define CAN_F16DATA0_FD18_Msk                   (0x1U << CAN_F16DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F16DATA0_FD18                       CAN_F16DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F16DATA0_FD19_Pos                   (19U)                             
#define CAN_F16DATA0_FD19_Msk                   (0x1U << CAN_F16DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F16DATA0_FD19                       CAN_F16DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F16DATA0_FD20_Pos                   (20U)                             
#define CAN_F16DATA0_FD20_Msk                   (0x1U << CAN_F16DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F16DATA0_FD20                       CAN_F16DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F16DATA0_FD21_Pos                   (21U)                             
#define CAN_F16DATA0_FD21_Msk                   (0x1U << CAN_F16DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F16DATA0_FD21                       CAN_F16DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F16DATA0_FD22_Pos                   (22U)                             
#define CAN_F16DATA0_FD22_Msk                   (0x1U << CAN_F16DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F16DATA0_FD22                       CAN_F16DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F16DATA0_FD23_Pos                   (23U)                             
#define CAN_F16DATA0_FD23_Msk                   (0x1U << CAN_F16DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F16DATA0_FD23                       CAN_F16DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F16DATA0_FD24_Pos                   (24U)                             
#define CAN_F16DATA0_FD24_Msk                   (0x1U << CAN_F16DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F16DATA0_FD24                       CAN_F16DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F16DATA0_FD25_Pos                   (25U)                             
#define CAN_F16DATA0_FD25_Msk                   (0x1U << CAN_F16DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F16DATA0_FD25                       CAN_F16DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F16DATA0_FD26_Pos                   (26U)                             
#define CAN_F16DATA0_FD26_Msk                   (0x1U << CAN_F16DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F16DATA0_FD26                       CAN_F16DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F16DATA0_FD27_Pos                   (27U)                             
#define CAN_F16DATA0_FD27_Msk                   (0x1U << CAN_F16DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F16DATA0_FD27                       CAN_F16DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F16DATA0_FD28_Pos                   (28U)                             
#define CAN_F16DATA0_FD28_Msk                   (0x1U << CAN_F16DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F16DATA0_FD28                       CAN_F16DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F16DATA0_FD29_Pos                   (29U)                             
#define CAN_F16DATA0_FD29_Msk                   (0x1U << CAN_F16DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F16DATA0_FD29                       CAN_F16DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F16DATA0_FD30_Pos                   (30U)                             
#define CAN_F16DATA0_FD30_Msk                   (0x1U << CAN_F16DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F16DATA0_FD30                       CAN_F16DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F16DATA0_FD31_Pos                   (31U)                             
#define CAN_F16DATA0_FD31_Msk                   (0x1U << CAN_F16DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F16DATA0_FD31                       CAN_F16DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F17R1 register  ******************/
#define CAN_F17DATA0_FD0_Pos                    (0U)                              
#define CAN_F17DATA0_FD0_Msk                    (0x1U << CAN_F17DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F17DATA0_FD0                        CAN_F17DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F17DATA0_FD1_Pos                    (1U)                              
#define CAN_F17DATA0_FD1_Msk                    (0x1U << CAN_F17DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F17DATA0_FD1                        CAN_F17DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F17DATA0_FD2_Pos                    (2U)                              
#define CAN_F17DATA0_FD2_Msk                    (0x1U << CAN_F17DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F17DATA0_FD2                        CAN_F17DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F17DATA0_FD3_Pos                    (3U)                              
#define CAN_F17DATA0_FD3_Msk                    (0x1U << CAN_F17DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F17DATA0_FD3                        CAN_F17DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F17DATA0_FD4_Pos                    (4U)                              
#define CAN_F17DATA0_FD4_Msk                    (0x1U << CAN_F17DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F17DATA0_FD4                        CAN_F17DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F17DATA0_FD5_Pos                    (5U)                              
#define CAN_F17DATA0_FD5_Msk                    (0x1U << CAN_F17DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F17DATA0_FD5                        CAN_F17DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F17DATA0_FD6_Pos                    (6U)                              
#define CAN_F17DATA0_FD6_Msk                    (0x1U << CAN_F17DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F17DATA0_FD6                        CAN_F17DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F17DATA0_FD7_Pos                    (7U)                              
#define CAN_F17DATA0_FD7_Msk                    (0x1U << CAN_F17DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F17DATA0_FD7                        CAN_F17DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F17DATA0_FD8_Pos                    (8U)                              
#define CAN_F17DATA0_FD8_Msk                    (0x1U << CAN_F17DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F17DATA0_FD8                        CAN_F17DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F17DATA0_FD9_Pos                    (9U)                              
#define CAN_F17DATA0_FD9_Msk                    (0x1U << CAN_F17DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F17DATA0_FD9                        CAN_F17DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F17DATA0_FD10_Pos                   (10U)                             
#define CAN_F17DATA0_FD10_Msk                   (0x1U << CAN_F17DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F17DATA0_FD10                       CAN_F17DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F17DATA0_FD11_Pos                   (11U)                             
#define CAN_F17DATA0_FD11_Msk                   (0x1U << CAN_F17DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F17DATA0_FD11                       CAN_F17DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F17DATA0_FD12_Pos                   (12U)                             
#define CAN_F17DATA0_FD12_Msk                   (0x1U << CAN_F17DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F17DATA0_FD12                       CAN_F17DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F17DATA0_FD13_Pos                   (13U)                             
#define CAN_F17DATA0_FD13_Msk                   (0x1U << CAN_F17DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F17DATA0_FD13                       CAN_F17DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F17DATA0_FD14_Pos                   (14U)                             
#define CAN_F17DATA0_FD14_Msk                   (0x1U << CAN_F17DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F17DATA0_FD14                       CAN_F17DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F17DATA0_FD15_Pos                   (15U)                             
#define CAN_F17DATA0_FD15_Msk                   (0x1U << CAN_F17DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F17DATA0_FD15                       CAN_F17DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F17DATA0_FD16_Pos                   (16U)                             
#define CAN_F17DATA0_FD16_Msk                   (0x1U << CAN_F17DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F17DATA0_FD16                       CAN_F17DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F17DATA0_FD17_Pos                   (17U)                             
#define CAN_F17DATA0_FD17_Msk                   (0x1U << CAN_F17DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F17DATA0_FD17                       CAN_F17DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F17DATA0_FD18_Pos                   (18U)                             
#define CAN_F17DATA0_FD18_Msk                   (0x1U << CAN_F17DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F17DATA0_FD18                       CAN_F17DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F17DATA0_FD19_Pos                   (19U)                             
#define CAN_F17DATA0_FD19_Msk                   (0x1U << CAN_F17DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F17DATA0_FD19                       CAN_F17DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F17DATA0_FD20_Pos                   (20U)                             
#define CAN_F17DATA0_FD20_Msk                   (0x1U << CAN_F17DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F17DATA0_FD20                       CAN_F17DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F17DATA0_FD21_Pos                   (21U)                             
#define CAN_F17DATA0_FD21_Msk                   (0x1U << CAN_F17DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F17DATA0_FD21                       CAN_F17DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F17DATA0_FD22_Pos                   (22U)                             
#define CAN_F17DATA0_FD22_Msk                   (0x1U << CAN_F17DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F17DATA0_FD22                       CAN_F17DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F17DATA0_FD23_Pos                   (23U)                             
#define CAN_F17DATA0_FD23_Msk                   (0x1U << CAN_F17DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F17DATA0_FD23                       CAN_F17DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F17DATA0_FD24_Pos                   (24U)                             
#define CAN_F17DATA0_FD24_Msk                   (0x1U << CAN_F17DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F17DATA0_FD24                       CAN_F17DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F17DATA0_FD25_Pos                   (25U)                             
#define CAN_F17DATA0_FD25_Msk                   (0x1U << CAN_F17DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F17DATA0_FD25                       CAN_F17DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F17DATA0_FD26_Pos                   (26U)                             
#define CAN_F17DATA0_FD26_Msk                   (0x1U << CAN_F17DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F17DATA0_FD26                       CAN_F17DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F17DATA0_FD27_Pos                   (27U)                             
#define CAN_F17DATA0_FD27_Msk                   (0x1U << CAN_F17DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F17DATA0_FD27                       CAN_F17DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F17DATA0_FD28_Pos                   (28U)                             
#define CAN_F17DATA0_FD28_Msk                   (0x1U << CAN_F17DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F17DATA0_FD28                       CAN_F17DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F17DATA0_FD29_Pos                   (29U)                             
#define CAN_F17DATA0_FD29_Msk                   (0x1U << CAN_F17DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F17DATA0_FD29                       CAN_F17DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F17DATA0_FD30_Pos                   (30U)                             
#define CAN_F17DATA0_FD30_Msk                   (0x1U << CAN_F17DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F17DATA0_FD30                       CAN_F17DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F17DATA0_FD31_Pos                   (31U)                             
#define CAN_F17DATA0_FD31_Msk                   (0x1U << CAN_F17DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F17DATA0_FD31                       CAN_F17DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F18R1 register  ******************/
#define CAN_F18DATA0_FD0_Pos                    (0U)                              
#define CAN_F18DATA0_FD0_Msk                    (0x1U << CAN_F18DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F18DATA0_FD0                        CAN_F18DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F18DATA0_FD1_Pos                    (1U)                              
#define CAN_F18DATA0_FD1_Msk                    (0x1U << CAN_F18DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F18DATA0_FD1                        CAN_F18DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F18DATA0_FD2_Pos                    (2U)                              
#define CAN_F18DATA0_FD2_Msk                    (0x1U << CAN_F18DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F18DATA0_FD2                        CAN_F18DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F18DATA0_FD3_Pos                    (3U)                              
#define CAN_F18DATA0_FD3_Msk                    (0x1U << CAN_F18DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F18DATA0_FD3                        CAN_F18DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F18DATA0_FD4_Pos                    (4U)                              
#define CAN_F18DATA0_FD4_Msk                    (0x1U << CAN_F18DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F18DATA0_FD4                        CAN_F18DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F18DATA0_FD5_Pos                    (5U)                              
#define CAN_F18DATA0_FD5_Msk                    (0x1U << CAN_F18DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F18DATA0_FD5                        CAN_F18DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F18DATA0_FD6_Pos                    (6U)                              
#define CAN_F18DATA0_FD6_Msk                    (0x1U << CAN_F18DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F18DATA0_FD6                        CAN_F18DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F18DATA0_FD7_Pos                    (7U)                              
#define CAN_F18DATA0_FD7_Msk                    (0x1U << CAN_F18DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F18DATA0_FD7                        CAN_F18DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F18DATA0_FD8_Pos                    (8U)                              
#define CAN_F18DATA0_FD8_Msk                    (0x1U << CAN_F18DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F18DATA0_FD8                        CAN_F18DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F18DATA0_FD9_Pos                    (9U)                              
#define CAN_F18DATA0_FD9_Msk                    (0x1U << CAN_F18DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F18DATA0_FD9                        CAN_F18DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F18DATA0_FD10_Pos                   (10U)                             
#define CAN_F18DATA0_FD10_Msk                   (0x1U << CAN_F18DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F18DATA0_FD10                       CAN_F18DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F18DATA0_FD11_Pos                   (11U)                             
#define CAN_F18DATA0_FD11_Msk                   (0x1U << CAN_F18DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F18DATA0_FD11                       CAN_F18DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F18DATA0_FD12_Pos                   (12U)                             
#define CAN_F18DATA0_FD12_Msk                   (0x1U << CAN_F18DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F18DATA0_FD12                       CAN_F18DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F18DATA0_FD13_Pos                   (13U)                             
#define CAN_F18DATA0_FD13_Msk                   (0x1U << CAN_F18DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F18DATA0_FD13                       CAN_F18DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F18DATA0_FD14_Pos                   (14U)                             
#define CAN_F18DATA0_FD14_Msk                   (0x1U << CAN_F18DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F18DATA0_FD14                       CAN_F18DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F18DATA0_FD15_Pos                   (15U)                             
#define CAN_F18DATA0_FD15_Msk                   (0x1U << CAN_F18DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F18DATA0_FD15                       CAN_F18DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F18DATA0_FD16_Pos                   (16U)                             
#define CAN_F18DATA0_FD16_Msk                   (0x1U << CAN_F18DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F18DATA0_FD16                       CAN_F18DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F18DATA0_FD17_Pos                   (17U)                             
#define CAN_F18DATA0_FD17_Msk                   (0x1U << CAN_F18DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F18DATA0_FD17                       CAN_F18DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F18DATA0_FD18_Pos                   (18U)                             
#define CAN_F18DATA0_FD18_Msk                   (0x1U << CAN_F18DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F18DATA0_FD18                       CAN_F18DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F18DATA0_FD19_Pos                   (19U)                             
#define CAN_F18DATA0_FD19_Msk                   (0x1U << CAN_F18DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F18DATA0_FD19                       CAN_F18DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F18DATA0_FD20_Pos                   (20U)                             
#define CAN_F18DATA0_FD20_Msk                   (0x1U << CAN_F18DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F18DATA0_FD20                       CAN_F18DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F18DATA0_FD21_Pos                   (21U)                             
#define CAN_F18DATA0_FD21_Msk                   (0x1U << CAN_F18DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F18DATA0_FD21                       CAN_F18DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F18DATA0_FD22_Pos                   (22U)                             
#define CAN_F18DATA0_FD22_Msk                   (0x1U << CAN_F18DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F18DATA0_FD22                       CAN_F18DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F18DATA0_FD23_Pos                   (23U)                             
#define CAN_F18DATA0_FD23_Msk                   (0x1U << CAN_F18DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F18DATA0_FD23                       CAN_F18DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F18DATA0_FD24_Pos                   (24U)                             
#define CAN_F18DATA0_FD24_Msk                   (0x1U << CAN_F18DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F18DATA0_FD24                       CAN_F18DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F18DATA0_FD25_Pos                   (25U)                             
#define CAN_F18DATA0_FD25_Msk                   (0x1U << CAN_F18DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F18DATA0_FD25                       CAN_F18DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F18DATA0_FD26_Pos                   (26U)                             
#define CAN_F18DATA0_FD26_Msk                   (0x1U << CAN_F18DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F18DATA0_FD26                       CAN_F18DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F18DATA0_FD27_Pos                   (27U)                             
#define CAN_F18DATA0_FD27_Msk                   (0x1U << CAN_F18DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F18DATA0_FD27                       CAN_F18DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F18DATA0_FD28_Pos                   (28U)                             
#define CAN_F18DATA0_FD28_Msk                   (0x1U << CAN_F18DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F18DATA0_FD28                       CAN_F18DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F18DATA0_FD29_Pos                   (29U)                             
#define CAN_F18DATA0_FD29_Msk                   (0x1U << CAN_F18DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F18DATA0_FD29                       CAN_F18DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F18DATA0_FD30_Pos                   (30U)                             
#define CAN_F18DATA0_FD30_Msk                   (0x1U << CAN_F18DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F18DATA0_FD30                       CAN_F18DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F18DATA0_FD31_Pos                   (31U)                             
#define CAN_F18DATA0_FD31_Msk                   (0x1U << CAN_F18DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F18DATA0_FD31                       CAN_F18DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F19R1 register  ******************/
#define CAN_F19DATA0_FD0_Pos                    (0U)                              
#define CAN_F19DATA0_FD0_Msk                    (0x1U << CAN_F19DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F19DATA0_FD0                        CAN_F19DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F19DATA0_FD1_Pos                    (1U)                              
#define CAN_F19DATA0_FD1_Msk                    (0x1U << CAN_F19DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F19DATA0_FD1                        CAN_F19DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F19DATA0_FD2_Pos                    (2U)                              
#define CAN_F19DATA0_FD2_Msk                    (0x1U << CAN_F19DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F19DATA0_FD2                        CAN_F19DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F19DATA0_FD3_Pos                    (3U)                              
#define CAN_F19DATA0_FD3_Msk                    (0x1U << CAN_F19DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F19DATA0_FD3                        CAN_F19DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F19DATA0_FD4_Pos                    (4U)                              
#define CAN_F19DATA0_FD4_Msk                    (0x1U << CAN_F19DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F19DATA0_FD4                        CAN_F19DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F19DATA0_FD5_Pos                    (5U)                              
#define CAN_F19DATA0_FD5_Msk                    (0x1U << CAN_F19DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F19DATA0_FD5                        CAN_F19DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F19DATA0_FD6_Pos                    (6U)                              
#define CAN_F19DATA0_FD6_Msk                    (0x1U << CAN_F19DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F19DATA0_FD6                        CAN_F19DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F19DATA0_FD7_Pos                    (7U)                              
#define CAN_F19DATA0_FD7_Msk                    (0x1U << CAN_F19DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F19DATA0_FD7                        CAN_F19DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F19DATA0_FD8_Pos                    (8U)                              
#define CAN_F19DATA0_FD8_Msk                    (0x1U << CAN_F19DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F19DATA0_FD8                        CAN_F19DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F19DATA0_FD9_Pos                    (9U)                              
#define CAN_F19DATA0_FD9_Msk                    (0x1U << CAN_F19DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F19DATA0_FD9                        CAN_F19DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F19DATA0_FD10_Pos                   (10U)                             
#define CAN_F19DATA0_FD10_Msk                   (0x1U << CAN_F19DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F19DATA0_FD10                       CAN_F19DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F19DATA0_FD11_Pos                   (11U)                             
#define CAN_F19DATA0_FD11_Msk                   (0x1U << CAN_F19DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F19DATA0_FD11                       CAN_F19DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F19DATA0_FD12_Pos                   (12U)                             
#define CAN_F19DATA0_FD12_Msk                   (0x1U << CAN_F19DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F19DATA0_FD12                       CAN_F19DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F19DATA0_FD13_Pos                   (13U)                             
#define CAN_F19DATA0_FD13_Msk                   (0x1U << CAN_F19DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F19DATA0_FD13                       CAN_F19DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F19DATA0_FD14_Pos                   (14U)                             
#define CAN_F19DATA0_FD14_Msk                   (0x1U << CAN_F19DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F19DATA0_FD14                       CAN_F19DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F19DATA0_FD15_Pos                   (15U)                             
#define CAN_F19DATA0_FD15_Msk                   (0x1U << CAN_F19DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F19DATA0_FD15                       CAN_F19DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F19DATA0_FD16_Pos                   (16U)                             
#define CAN_F19DATA0_FD16_Msk                   (0x1U << CAN_F19DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F19DATA0_FD16                       CAN_F19DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F19DATA0_FD17_Pos                   (17U)                             
#define CAN_F19DATA0_FD17_Msk                   (0x1U << CAN_F19DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F19DATA0_FD17                       CAN_F19DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F19DATA0_FD18_Pos                   (18U)                             
#define CAN_F19DATA0_FD18_Msk                   (0x1U << CAN_F19DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F19DATA0_FD18                       CAN_F19DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F19DATA0_FD19_Pos                   (19U)                             
#define CAN_F19DATA0_FD19_Msk                   (0x1U << CAN_F19DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F19DATA0_FD19                       CAN_F19DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F19DATA0_FD20_Pos                   (20U)                             
#define CAN_F19DATA0_FD20_Msk                   (0x1U << CAN_F19DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F19DATA0_FD20                       CAN_F19DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F19DATA0_FD21_Pos                   (21U)                             
#define CAN_F19DATA0_FD21_Msk                   (0x1U << CAN_F19DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F19DATA0_FD21                       CAN_F19DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F19DATA0_FD22_Pos                   (22U)                             
#define CAN_F19DATA0_FD22_Msk                   (0x1U << CAN_F19DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F19DATA0_FD22                       CAN_F19DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F19DATA0_FD23_Pos                   (23U)                             
#define CAN_F19DATA0_FD23_Msk                   (0x1U << CAN_F19DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F19DATA0_FD23                       CAN_F19DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F19DATA0_FD24_Pos                   (24U)                             
#define CAN_F19DATA0_FD24_Msk                   (0x1U << CAN_F19DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F19DATA0_FD24                       CAN_F19DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F19DATA0_FD25_Pos                   (25U)                             
#define CAN_F19DATA0_FD25_Msk                   (0x1U << CAN_F19DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F19DATA0_FD25                       CAN_F19DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F19DATA0_FD26_Pos                   (26U)                             
#define CAN_F19DATA0_FD26_Msk                   (0x1U << CAN_F19DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F19DATA0_FD26                       CAN_F19DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F19DATA0_FD27_Pos                   (27U)                             
#define CAN_F19DATA0_FD27_Msk                   (0x1U << CAN_F19DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F19DATA0_FD27                       CAN_F19DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F19DATA0_FD28_Pos                   (28U)                             
#define CAN_F19DATA0_FD28_Msk                   (0x1U << CAN_F19DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F19DATA0_FD28                       CAN_F19DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F19DATA0_FD29_Pos                   (29U)                             
#define CAN_F19DATA0_FD29_Msk                   (0x1U << CAN_F19DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F19DATA0_FD29                       CAN_F19DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F19DATA0_FD30_Pos                   (30U)                             
#define CAN_F19DATA0_FD30_Msk                   (0x1U << CAN_F19DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F19DATA0_FD30                       CAN_F19DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F19DATA0_FD31_Pos                   (31U)                             
#define CAN_F19DATA0_FD31_Msk                   (0x1U << CAN_F19DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F19DATA0_FD31                       CAN_F19DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F20R1 register  ******************/
#define CAN_F20DATA0_FD0_Pos                    (0U)                              
#define CAN_F20DATA0_FD0_Msk                    (0x1U << CAN_F20DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F20DATA0_FD0                        CAN_F20DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F20DATA0_FD1_Pos                    (1U)                              
#define CAN_F20DATA0_FD1_Msk                    (0x1U << CAN_F20DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F20DATA0_FD1                        CAN_F20DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F20DATA0_FD2_Pos                    (2U)                              
#define CAN_F20DATA0_FD2_Msk                    (0x1U << CAN_F20DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F20DATA0_FD2                        CAN_F20DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F20DATA0_FD3_Pos                    (3U)                              
#define CAN_F20DATA0_FD3_Msk                    (0x1U << CAN_F20DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F20DATA0_FD3                        CAN_F20DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F20DATA0_FD4_Pos                    (4U)                              
#define CAN_F20DATA0_FD4_Msk                    (0x1U << CAN_F20DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F20DATA0_FD4                        CAN_F20DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F20DATA0_FD5_Pos                    (5U)                              
#define CAN_F20DATA0_FD5_Msk                    (0x1U << CAN_F20DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F20DATA0_FD5                        CAN_F20DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F20DATA0_FD6_Pos                    (6U)                              
#define CAN_F20DATA0_FD6_Msk                    (0x1U << CAN_F20DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F20DATA0_FD6                        CAN_F20DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F20DATA0_FD7_Pos                    (7U)                              
#define CAN_F20DATA0_FD7_Msk                    (0x1U << CAN_F20DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F20DATA0_FD7                        CAN_F20DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F20DATA0_FD8_Pos                    (8U)                              
#define CAN_F20DATA0_FD8_Msk                    (0x1U << CAN_F20DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F20DATA0_FD8                        CAN_F20DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F20DATA0_FD9_Pos                    (9U)                              
#define CAN_F20DATA0_FD9_Msk                    (0x1U << CAN_F20DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F20DATA0_FD9                        CAN_F20DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F20DATA0_FD10_Pos                   (10U)                             
#define CAN_F20DATA0_FD10_Msk                   (0x1U << CAN_F20DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F20DATA0_FD10                       CAN_F20DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F20DATA0_FD11_Pos                   (11U)                             
#define CAN_F20DATA0_FD11_Msk                   (0x1U << CAN_F20DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F20DATA0_FD11                       CAN_F20DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F20DATA0_FD12_Pos                   (12U)                             
#define CAN_F20DATA0_FD12_Msk                   (0x1U << CAN_F20DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F20DATA0_FD12                       CAN_F20DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F20DATA0_FD13_Pos                   (13U)                             
#define CAN_F20DATA0_FD13_Msk                   (0x1U << CAN_F20DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F20DATA0_FD13                       CAN_F20DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F20DATA0_FD14_Pos                   (14U)                             
#define CAN_F20DATA0_FD14_Msk                   (0x1U << CAN_F20DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F20DATA0_FD14                       CAN_F20DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F20DATA0_FD15_Pos                   (15U)                             
#define CAN_F20DATA0_FD15_Msk                   (0x1U << CAN_F20DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F20DATA0_FD15                       CAN_F20DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F20DATA0_FD16_Pos                   (16U)                             
#define CAN_F20DATA0_FD16_Msk                   (0x1U << CAN_F20DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F20DATA0_FD16                       CAN_F20DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F20DATA0_FD17_Pos                   (17U)                             
#define CAN_F20DATA0_FD17_Msk                   (0x1U << CAN_F20DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F20DATA0_FD17                       CAN_F20DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F20DATA0_FD18_Pos                   (18U)                             
#define CAN_F20DATA0_FD18_Msk                   (0x1U << CAN_F20DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F20DATA0_FD18                       CAN_F20DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F20DATA0_FD19_Pos                   (19U)                             
#define CAN_F20DATA0_FD19_Msk                   (0x1U << CAN_F20DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F20DATA0_FD19                       CAN_F20DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F20DATA0_FD20_Pos                   (20U)                             
#define CAN_F20DATA0_FD20_Msk                   (0x1U << CAN_F20DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F20DATA0_FD20                       CAN_F20DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F20DATA0_FD21_Pos                   (21U)                             
#define CAN_F20DATA0_FD21_Msk                   (0x1U << CAN_F20DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F20DATA0_FD21                       CAN_F20DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F20DATA0_FD22_Pos                   (22U)                             
#define CAN_F20DATA0_FD22_Msk                   (0x1U << CAN_F20DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F20DATA0_FD22                       CAN_F20DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F20DATA0_FD23_Pos                   (23U)                             
#define CAN_F20DATA0_FD23_Msk                   (0x1U << CAN_F20DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F20DATA0_FD23                       CAN_F20DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F20DATA0_FD24_Pos                   (24U)                             
#define CAN_F20DATA0_FD24_Msk                   (0x1U << CAN_F20DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F20DATA0_FD24                       CAN_F20DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F20DATA0_FD25_Pos                   (25U)                             
#define CAN_F20DATA0_FD25_Msk                   (0x1U << CAN_F20DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F20DATA0_FD25                       CAN_F20DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F20DATA0_FD26_Pos                   (26U)                             
#define CAN_F20DATA0_FD26_Msk                   (0x1U << CAN_F20DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F20DATA0_FD26                       CAN_F20DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F20DATA0_FD27_Pos                   (27U)                             
#define CAN_F20DATA0_FD27_Msk                   (0x1U << CAN_F20DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F20DATA0_FD27                       CAN_F20DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F20DATA0_FD28_Pos                   (28U)                             
#define CAN_F20DATA0_FD28_Msk                   (0x1U << CAN_F20DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F20DATA0_FD28                       CAN_F20DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F20DATA0_FD29_Pos                   (29U)                             
#define CAN_F20DATA0_FD29_Msk                   (0x1U << CAN_F20DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F20DATA0_FD29                       CAN_F20DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F20DATA0_FD30_Pos                   (30U)                             
#define CAN_F20DATA0_FD30_Msk                   (0x1U << CAN_F20DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F20DATA0_FD30                       CAN_F20DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F20DATA0_FD31_Pos                   (31U)                             
#define CAN_F20DATA0_FD31_Msk                   (0x1U << CAN_F20DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F20DATA0_FD31                       CAN_F20DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F21R1 register  ******************/
#define CAN_F21DATA0_FD0_Pos                    (0U)                              
#define CAN_F21DATA0_FD0_Msk                    (0x1U << CAN_F21DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F21DATA0_FD0                        CAN_F21DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F21DATA0_FD1_Pos                    (1U)                              
#define CAN_F21DATA0_FD1_Msk                    (0x1U << CAN_F21DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F21DATA0_FD1                        CAN_F21DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F21DATA0_FD2_Pos                    (2U)                              
#define CAN_F21DATA0_FD2_Msk                    (0x1U << CAN_F21DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F21DATA0_FD2                        CAN_F21DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F21DATA0_FD3_Pos                    (3U)                              
#define CAN_F21DATA0_FD3_Msk                    (0x1U << CAN_F21DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F21DATA0_FD3                        CAN_F21DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F21DATA0_FD4_Pos                    (4U)                              
#define CAN_F21DATA0_FD4_Msk                    (0x1U << CAN_F21DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F21DATA0_FD4                        CAN_F21DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F21DATA0_FD5_Pos                    (5U)                              
#define CAN_F21DATA0_FD5_Msk                    (0x1U << CAN_F21DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F21DATA0_FD5                        CAN_F21DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F21DATA0_FD6_Pos                    (6U)                              
#define CAN_F21DATA0_FD6_Msk                    (0x1U << CAN_F21DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F21DATA0_FD6                        CAN_F21DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F21DATA0_FD7_Pos                    (7U)                              
#define CAN_F21DATA0_FD7_Msk                    (0x1U << CAN_F21DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F21DATA0_FD7                        CAN_F21DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F21DATA0_FD8_Pos                    (8U)                              
#define CAN_F21DATA0_FD8_Msk                    (0x1U << CAN_F21DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F21DATA0_FD8                        CAN_F21DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F21DATA0_FD9_Pos                    (9U)                              
#define CAN_F21DATA0_FD9_Msk                    (0x1U << CAN_F21DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F21DATA0_FD9                        CAN_F21DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F21DATA0_FD10_Pos                   (10U)                             
#define CAN_F21DATA0_FD10_Msk                   (0x1U << CAN_F21DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F21DATA0_FD10                       CAN_F21DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F21DATA0_FD11_Pos                   (11U)                             
#define CAN_F21DATA0_FD11_Msk                   (0x1U << CAN_F21DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F21DATA0_FD11                       CAN_F21DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F21DATA0_FD12_Pos                   (12U)                             
#define CAN_F21DATA0_FD12_Msk                   (0x1U << CAN_F21DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F21DATA0_FD12                       CAN_F21DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F21DATA0_FD13_Pos                   (13U)                             
#define CAN_F21DATA0_FD13_Msk                   (0x1U << CAN_F21DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F21DATA0_FD13                       CAN_F21DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F21DATA0_FD14_Pos                   (14U)                             
#define CAN_F21DATA0_FD14_Msk                   (0x1U << CAN_F21DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F21DATA0_FD14                       CAN_F21DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F21DATA0_FD15_Pos                   (15U)                             
#define CAN_F21DATA0_FD15_Msk                   (0x1U << CAN_F21DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F21DATA0_FD15                       CAN_F21DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F21DATA0_FD16_Pos                   (16U)                             
#define CAN_F21DATA0_FD16_Msk                   (0x1U << CAN_F21DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F21DATA0_FD16                       CAN_F21DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F21DATA0_FD17_Pos                   (17U)                             
#define CAN_F21DATA0_FD17_Msk                   (0x1U << CAN_F21DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F21DATA0_FD17                       CAN_F21DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F21DATA0_FD18_Pos                   (18U)                             
#define CAN_F21DATA0_FD18_Msk                   (0x1U << CAN_F21DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F21DATA0_FD18                       CAN_F21DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F21DATA0_FD19_Pos                   (19U)                             
#define CAN_F21DATA0_FD19_Msk                   (0x1U << CAN_F21DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F21DATA0_FD19                       CAN_F21DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F21DATA0_FD20_Pos                   (20U)                             
#define CAN_F21DATA0_FD20_Msk                   (0x1U << CAN_F21DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F21DATA0_FD20                       CAN_F21DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F21DATA0_FD21_Pos                   (21U)                             
#define CAN_F21DATA0_FD21_Msk                   (0x1U << CAN_F21DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F21DATA0_FD21                       CAN_F21DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F21DATA0_FD22_Pos                   (22U)                             
#define CAN_F21DATA0_FD22_Msk                   (0x1U << CAN_F21DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F21DATA0_FD22                       CAN_F21DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F21DATA0_FD23_Pos                   (23U)                             
#define CAN_F21DATA0_FD23_Msk                   (0x1U << CAN_F21DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F21DATA0_FD23                       CAN_F21DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F21DATA0_FD24_Pos                   (24U)                             
#define CAN_F21DATA0_FD24_Msk                   (0x1U << CAN_F21DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F21DATA0_FD24                       CAN_F21DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F21DATA0_FD25_Pos                   (25U)                             
#define CAN_F21DATA0_FD25_Msk                   (0x1U << CAN_F21DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F21DATA0_FD25                       CAN_F21DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F21DATA0_FD26_Pos                   (26U)                             
#define CAN_F21DATA0_FD26_Msk                   (0x1U << CAN_F21DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F21DATA0_FD26                       CAN_F21DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F21DATA0_FD27_Pos                   (27U)                             
#define CAN_F21DATA0_FD27_Msk                   (0x1U << CAN_F21DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F21DATA0_FD27                       CAN_F21DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F21DATA0_FD28_Pos                   (28U)                             
#define CAN_F21DATA0_FD28_Msk                   (0x1U << CAN_F21DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F21DATA0_FD28                       CAN_F21DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F21DATA0_FD29_Pos                   (29U)                             
#define CAN_F21DATA0_FD29_Msk                   (0x1U << CAN_F21DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F21DATA0_FD29                       CAN_F21DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F21DATA0_FD30_Pos                   (30U)                             
#define CAN_F21DATA0_FD30_Msk                   (0x1U << CAN_F21DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F21DATA0_FD30                       CAN_F21DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F21DATA0_FD31_Pos                   (31U)                             
#define CAN_F21DATA0_FD31_Msk                   (0x1U << CAN_F21DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F21DATA0_FD31                       CAN_F21DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F22R1 register  ******************/
#define CAN_F22DATA0_FD0_Pos                    (0U)                              
#define CAN_F22DATA0_FD0_Msk                    (0x1U << CAN_F22DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F22DATA0_FD0                        CAN_F22DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F22DATA0_FD1_Pos                    (1U)                              
#define CAN_F22DATA0_FD1_Msk                    (0x1U << CAN_F22DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F22DATA0_FD1                        CAN_F22DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F22DATA0_FD2_Pos                    (2U)                              
#define CAN_F22DATA0_FD2_Msk                    (0x1U << CAN_F22DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F22DATA0_FD2                        CAN_F22DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F22DATA0_FD3_Pos                    (3U)                              
#define CAN_F22DATA0_FD3_Msk                    (0x1U << CAN_F22DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F22DATA0_FD3                        CAN_F22DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F22DATA0_FD4_Pos                    (4U)                              
#define CAN_F22DATA0_FD4_Msk                    (0x1U << CAN_F22DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F22DATA0_FD4                        CAN_F22DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F22DATA0_FD5_Pos                    (5U)                              
#define CAN_F22DATA0_FD5_Msk                    (0x1U << CAN_F22DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F22DATA0_FD5                        CAN_F22DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F22DATA0_FD6_Pos                    (6U)                              
#define CAN_F22DATA0_FD6_Msk                    (0x1U << CAN_F22DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F22DATA0_FD6                        CAN_F22DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F22DATA0_FD7_Pos                    (7U)                              
#define CAN_F22DATA0_FD7_Msk                    (0x1U << CAN_F22DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F22DATA0_FD7                        CAN_F22DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F22DATA0_FD8_Pos                    (8U)                              
#define CAN_F22DATA0_FD8_Msk                    (0x1U << CAN_F22DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F22DATA0_FD8                        CAN_F22DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F22DATA0_FD9_Pos                    (9U)                              
#define CAN_F22DATA0_FD9_Msk                    (0x1U << CAN_F22DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F22DATA0_FD9                        CAN_F22DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F22DATA0_FD10_Pos                   (10U)                             
#define CAN_F22DATA0_FD10_Msk                   (0x1U << CAN_F22DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F22DATA0_FD10                       CAN_F22DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F22DATA0_FD11_Pos                   (11U)                             
#define CAN_F22DATA0_FD11_Msk                   (0x1U << CAN_F22DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F22DATA0_FD11                       CAN_F22DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F22DATA0_FD12_Pos                   (12U)                             
#define CAN_F22DATA0_FD12_Msk                   (0x1U << CAN_F22DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F22DATA0_FD12                       CAN_F22DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F22DATA0_FD13_Pos                   (13U)                             
#define CAN_F22DATA0_FD13_Msk                   (0x1U << CAN_F22DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F22DATA0_FD13                       CAN_F22DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F22DATA0_FD14_Pos                   (14U)                             
#define CAN_F22DATA0_FD14_Msk                   (0x1U << CAN_F22DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F22DATA0_FD14                       CAN_F22DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F22DATA0_FD15_Pos                   (15U)                             
#define CAN_F22DATA0_FD15_Msk                   (0x1U << CAN_F22DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F22DATA0_FD15                       CAN_F22DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F22DATA0_FD16_Pos                   (16U)                             
#define CAN_F22DATA0_FD16_Msk                   (0x1U << CAN_F22DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F22DATA0_FD16                       CAN_F22DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F22DATA0_FD17_Pos                   (17U)                             
#define CAN_F22DATA0_FD17_Msk                   (0x1U << CAN_F22DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F22DATA0_FD17                       CAN_F22DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F22DATA0_FD18_Pos                   (18U)                             
#define CAN_F22DATA0_FD18_Msk                   (0x1U << CAN_F22DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F22DATA0_FD18                       CAN_F22DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F22DATA0_FD19_Pos                   (19U)                             
#define CAN_F22DATA0_FD19_Msk                   (0x1U << CAN_F22DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F22DATA0_FD19                       CAN_F22DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F22DATA0_FD20_Pos                   (20U)                             
#define CAN_F22DATA0_FD20_Msk                   (0x1U << CAN_F22DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F22DATA0_FD20                       CAN_F22DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F22DATA0_FD21_Pos                   (21U)                             
#define CAN_F22DATA0_FD21_Msk                   (0x1U << CAN_F22DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F22DATA0_FD21                       CAN_F22DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F22DATA0_FD22_Pos                   (22U)                             
#define CAN_F22DATA0_FD22_Msk                   (0x1U << CAN_F22DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F22DATA0_FD22                       CAN_F22DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F22DATA0_FD23_Pos                   (23U)                             
#define CAN_F22DATA0_FD23_Msk                   (0x1U << CAN_F22DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F22DATA0_FD23                       CAN_F22DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F22DATA0_FD24_Pos                   (24U)                             
#define CAN_F22DATA0_FD24_Msk                   (0x1U << CAN_F22DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F22DATA0_FD24                       CAN_F22DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F22DATA0_FD25_Pos                   (25U)                             
#define CAN_F22DATA0_FD25_Msk                   (0x1U << CAN_F22DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F22DATA0_FD25                       CAN_F22DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F22DATA0_FD26_Pos                   (26U)                             
#define CAN_F22DATA0_FD26_Msk                   (0x1U << CAN_F22DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F22DATA0_FD26                       CAN_F22DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F22DATA0_FD27_Pos                   (27U)                             
#define CAN_F22DATA0_FD27_Msk                   (0x1U << CAN_F22DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F22DATA0_FD27                       CAN_F22DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F22DATA0_FD28_Pos                   (28U)                             
#define CAN_F22DATA0_FD28_Msk                   (0x1U << CAN_F22DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F22DATA0_FD28                       CAN_F22DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F22DATA0_FD29_Pos                   (29U)                             
#define CAN_F22DATA0_FD29_Msk                   (0x1U << CAN_F22DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F22DATA0_FD29                       CAN_F22DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F22DATA0_FD30_Pos                   (30U)                             
#define CAN_F22DATA0_FD30_Msk                   (0x1U << CAN_F22DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F22DATA0_FD30                       CAN_F22DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F22DATA0_FD31_Pos                   (31U)                             
#define CAN_F22DATA0_FD31_Msk                   (0x1U << CAN_F22DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F22DATA0_FD31                       CAN_F22DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F23R1 register  ******************/
#define CAN_F23DATA0_FD0_Pos                    (0U)                              
#define CAN_F23DATA0_FD0_Msk                    (0x1U << CAN_F23DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F23DATA0_FD0                        CAN_F23DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F23DATA0_FD1_Pos                    (1U)                              
#define CAN_F23DATA0_FD1_Msk                    (0x1U << CAN_F23DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F23DATA0_FD1                        CAN_F23DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F23DATA0_FD2_Pos                    (2U)                              
#define CAN_F23DATA0_FD2_Msk                    (0x1U << CAN_F23DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F23DATA0_FD2                        CAN_F23DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F23DATA0_FD3_Pos                    (3U)                              
#define CAN_F23DATA0_FD3_Msk                    (0x1U << CAN_F23DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F23DATA0_FD3                        CAN_F23DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F23DATA0_FD4_Pos                    (4U)                              
#define CAN_F23DATA0_FD4_Msk                    (0x1U << CAN_F23DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F23DATA0_FD4                        CAN_F23DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F23DATA0_FD5_Pos                    (5U)                              
#define CAN_F23DATA0_FD5_Msk                    (0x1U << CAN_F23DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F23DATA0_FD5                        CAN_F23DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F23DATA0_FD6_Pos                    (6U)                              
#define CAN_F23DATA0_FD6_Msk                    (0x1U << CAN_F23DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F23DATA0_FD6                        CAN_F23DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F23DATA0_FD7_Pos                    (7U)                              
#define CAN_F23DATA0_FD7_Msk                    (0x1U << CAN_F23DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F23DATA0_FD7                        CAN_F23DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F23DATA0_FD8_Pos                    (8U)                              
#define CAN_F23DATA0_FD8_Msk                    (0x1U << CAN_F23DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F23DATA0_FD8                        CAN_F23DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F23DATA0_FD9_Pos                    (9U)                              
#define CAN_F23DATA0_FD9_Msk                    (0x1U << CAN_F23DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F23DATA0_FD9                        CAN_F23DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F23DATA0_FD10_Pos                   (10U)                             
#define CAN_F23DATA0_FD10_Msk                   (0x1U << CAN_F23DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F23DATA0_FD10                       CAN_F23DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F23DATA0_FD11_Pos                   (11U)                             
#define CAN_F23DATA0_FD11_Msk                   (0x1U << CAN_F23DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F23DATA0_FD11                       CAN_F23DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F23DATA0_FD12_Pos                   (12U)                             
#define CAN_F23DATA0_FD12_Msk                   (0x1U << CAN_F23DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F23DATA0_FD12                       CAN_F23DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F23DATA0_FD13_Pos                   (13U)                             
#define CAN_F23DATA0_FD13_Msk                   (0x1U << CAN_F23DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F23DATA0_FD13                       CAN_F23DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F23DATA0_FD14_Pos                   (14U)                             
#define CAN_F23DATA0_FD14_Msk                   (0x1U << CAN_F23DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F23DATA0_FD14                       CAN_F23DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F23DATA0_FD15_Pos                   (15U)                             
#define CAN_F23DATA0_FD15_Msk                   (0x1U << CAN_F23DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F23DATA0_FD15                       CAN_F23DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F23DATA0_FD16_Pos                   (16U)                             
#define CAN_F23DATA0_FD16_Msk                   (0x1U << CAN_F23DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F23DATA0_FD16                       CAN_F23DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F23DATA0_FD17_Pos                   (17U)                             
#define CAN_F23DATA0_FD17_Msk                   (0x1U << CAN_F23DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F23DATA0_FD17                       CAN_F23DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F23DATA0_FD18_Pos                   (18U)                             
#define CAN_F23DATA0_FD18_Msk                   (0x1U << CAN_F23DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F23DATA0_FD18                       CAN_F23DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F23DATA0_FD19_Pos                   (19U)                             
#define CAN_F23DATA0_FD19_Msk                   (0x1U << CAN_F23DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F23DATA0_FD19                       CAN_F23DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F23DATA0_FD20_Pos                   (20U)                             
#define CAN_F23DATA0_FD20_Msk                   (0x1U << CAN_F23DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F23DATA0_FD20                       CAN_F23DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F23DATA0_FD21_Pos                   (21U)                             
#define CAN_F23DATA0_FD21_Msk                   (0x1U << CAN_F23DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F23DATA0_FD21                       CAN_F23DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F23DATA0_FD22_Pos                   (22U)                             
#define CAN_F23DATA0_FD22_Msk                   (0x1U << CAN_F23DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F23DATA0_FD22                       CAN_F23DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F23DATA0_FD23_Pos                   (23U)                             
#define CAN_F23DATA0_FD23_Msk                   (0x1U << CAN_F23DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F23DATA0_FD23                       CAN_F23DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F23DATA0_FD24_Pos                   (24U)                             
#define CAN_F23DATA0_FD24_Msk                   (0x1U << CAN_F23DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F23DATA0_FD24                       CAN_F23DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F23DATA0_FD25_Pos                   (25U)                             
#define CAN_F23DATA0_FD25_Msk                   (0x1U << CAN_F23DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F23DATA0_FD25                       CAN_F23DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F23DATA0_FD26_Pos                   (26U)                             
#define CAN_F23DATA0_FD26_Msk                   (0x1U << CAN_F23DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F23DATA0_FD26                       CAN_F23DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F23DATA0_FD27_Pos                   (27U)                             
#define CAN_F23DATA0_FD27_Msk                   (0x1U << CAN_F23DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F23DATA0_FD27                       CAN_F23DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F23DATA0_FD28_Pos                   (28U)                             
#define CAN_F23DATA0_FD28_Msk                   (0x1U << CAN_F23DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F23DATA0_FD28                       CAN_F23DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F23DATA0_FD29_Pos                   (29U)                             
#define CAN_F23DATA0_FD29_Msk                   (0x1U << CAN_F23DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F23DATA0_FD29                       CAN_F23DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F23DATA0_FD30_Pos                   (30U)                             
#define CAN_F23DATA0_FD30_Msk                   (0x1U << CAN_F23DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F23DATA0_FD30                       CAN_F23DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F23DATA0_FD31_Pos                   (31U)                             
#define CAN_F23DATA0_FD31_Msk                   (0x1U << CAN_F23DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F23DATA0_FD31                       CAN_F23DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F24R1 register  ******************/
#define CAN_F24DATA0_FD0_Pos                    (0U)                              
#define CAN_F24DATA0_FD0_Msk                    (0x1U << CAN_F24DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F24DATA0_FD0                        CAN_F24DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F24DATA0_FD1_Pos                    (1U)                              
#define CAN_F24DATA0_FD1_Msk                    (0x1U << CAN_F24DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F24DATA0_FD1                        CAN_F24DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F24DATA0_FD2_Pos                    (2U)                              
#define CAN_F24DATA0_FD2_Msk                    (0x1U << CAN_F24DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F24DATA0_FD2                        CAN_F24DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F24DATA0_FD3_Pos                    (3U)                              
#define CAN_F24DATA0_FD3_Msk                    (0x1U << CAN_F24DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F24DATA0_FD3                        CAN_F24DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F24DATA0_FD4_Pos                    (4U)                              
#define CAN_F24DATA0_FD4_Msk                    (0x1U << CAN_F24DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F24DATA0_FD4                        CAN_F24DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F24DATA0_FD5_Pos                    (5U)                              
#define CAN_F24DATA0_FD5_Msk                    (0x1U << CAN_F24DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F24DATA0_FD5                        CAN_F24DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F24DATA0_FD6_Pos                    (6U)                              
#define CAN_F24DATA0_FD6_Msk                    (0x1U << CAN_F24DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F24DATA0_FD6                        CAN_F24DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F24DATA0_FD7_Pos                    (7U)                              
#define CAN_F24DATA0_FD7_Msk                    (0x1U << CAN_F24DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F24DATA0_FD7                        CAN_F24DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F24DATA0_FD8_Pos                    (8U)                              
#define CAN_F24DATA0_FD8_Msk                    (0x1U << CAN_F24DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F24DATA0_FD8                        CAN_F24DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F24DATA0_FD9_Pos                    (9U)                              
#define CAN_F24DATA0_FD9_Msk                    (0x1U << CAN_F24DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F24DATA0_FD9                        CAN_F24DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F24DATA0_FD10_Pos                   (10U)                             
#define CAN_F24DATA0_FD10_Msk                   (0x1U << CAN_F24DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F24DATA0_FD10                       CAN_F24DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F24DATA0_FD11_Pos                   (11U)                             
#define CAN_F24DATA0_FD11_Msk                   (0x1U << CAN_F24DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F24DATA0_FD11                       CAN_F24DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F24DATA0_FD12_Pos                   (12U)                             
#define CAN_F24DATA0_FD12_Msk                   (0x1U << CAN_F24DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F24DATA0_FD12                       CAN_F24DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F24DATA0_FD13_Pos                   (13U)                             
#define CAN_F24DATA0_FD13_Msk                   (0x1U << CAN_F24DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F24DATA0_FD13                       CAN_F24DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F24DATA0_FD14_Pos                   (14U)                             
#define CAN_F24DATA0_FD14_Msk                   (0x1U << CAN_F24DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F24DATA0_FD14                       CAN_F24DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F24DATA0_FD15_Pos                   (15U)                             
#define CAN_F24DATA0_FD15_Msk                   (0x1U << CAN_F24DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F24DATA0_FD15                       CAN_F24DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F24DATA0_FD16_Pos                   (16U)                             
#define CAN_F24DATA0_FD16_Msk                   (0x1U << CAN_F24DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F24DATA0_FD16                       CAN_F24DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F24DATA0_FD17_Pos                   (17U)                             
#define CAN_F24DATA0_FD17_Msk                   (0x1U << CAN_F24DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F24DATA0_FD17                       CAN_F24DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F24DATA0_FD18_Pos                   (18U)                             
#define CAN_F24DATA0_FD18_Msk                   (0x1U << CAN_F24DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F24DATA0_FD18                       CAN_F24DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F24DATA0_FD19_Pos                   (19U)                             
#define CAN_F24DATA0_FD19_Msk                   (0x1U << CAN_F24DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F24DATA0_FD19                       CAN_F24DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F24DATA0_FD20_Pos                   (20U)                             
#define CAN_F24DATA0_FD20_Msk                   (0x1U << CAN_F24DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F24DATA0_FD20                       CAN_F24DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F24DATA0_FD21_Pos                   (21U)                             
#define CAN_F24DATA0_FD21_Msk                   (0x1U << CAN_F24DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F24DATA0_FD21                       CAN_F24DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F24DATA0_FD22_Pos                   (22U)                             
#define CAN_F24DATA0_FD22_Msk                   (0x1U << CAN_F24DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F24DATA0_FD22                       CAN_F24DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F24DATA0_FD23_Pos                   (23U)                             
#define CAN_F24DATA0_FD23_Msk                   (0x1U << CAN_F24DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F24DATA0_FD23                       CAN_F24DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F24DATA0_FD24_Pos                   (24U)                             
#define CAN_F24DATA0_FD24_Msk                   (0x1U << CAN_F24DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F24DATA0_FD24                       CAN_F24DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F24DATA0_FD25_Pos                   (25U)                             
#define CAN_F24DATA0_FD25_Msk                   (0x1U << CAN_F24DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F24DATA0_FD25                       CAN_F24DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F24DATA0_FD26_Pos                   (26U)                             
#define CAN_F24DATA0_FD26_Msk                   (0x1U << CAN_F24DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F24DATA0_FD26                       CAN_F24DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F24DATA0_FD27_Pos                   (27U)                             
#define CAN_F24DATA0_FD27_Msk                   (0x1U << CAN_F24DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F24DATA0_FD27                       CAN_F24DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F24DATA0_FD28_Pos                   (28U)                             
#define CAN_F24DATA0_FD28_Msk                   (0x1U << CAN_F24DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F24DATA0_FD28                       CAN_F24DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F24DATA0_FD29_Pos                   (29U)                             
#define CAN_F24DATA0_FD29_Msk                   (0x1U << CAN_F24DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F24DATA0_FD29                       CAN_F24DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F24DATA0_FD30_Pos                   (30U)                             
#define CAN_F24DATA0_FD30_Msk                   (0x1U << CAN_F24DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F24DATA0_FD30                       CAN_F24DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F24DATA0_FD31_Pos                   (31U)                             
#define CAN_F24DATA0_FD31_Msk                   (0x1U << CAN_F24DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F24DATA0_FD31                       CAN_F24DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F25R1 register  ******************/
#define CAN_F25DATA0_FD0_Pos                    (0U)                              
#define CAN_F25DATA0_FD0_Msk                    (0x1U << CAN_F25DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F25DATA0_FD0                        CAN_F25DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F25DATA0_FD1_Pos                    (1U)                              
#define CAN_F25DATA0_FD1_Msk                    (0x1U << CAN_F25DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F25DATA0_FD1                        CAN_F25DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F25DATA0_FD2_Pos                    (2U)                              
#define CAN_F25DATA0_FD2_Msk                    (0x1U << CAN_F25DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F25DATA0_FD2                        CAN_F25DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F25DATA0_FD3_Pos                    (3U)                              
#define CAN_F25DATA0_FD3_Msk                    (0x1U << CAN_F25DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F25DATA0_FD3                        CAN_F25DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F25DATA0_FD4_Pos                    (4U)                              
#define CAN_F25DATA0_FD4_Msk                    (0x1U << CAN_F25DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F25DATA0_FD4                        CAN_F25DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F25DATA0_FD5_Pos                    (5U)                              
#define CAN_F25DATA0_FD5_Msk                    (0x1U << CAN_F25DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F25DATA0_FD5                        CAN_F25DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F25DATA0_FD6_Pos                    (6U)                              
#define CAN_F25DATA0_FD6_Msk                    (0x1U << CAN_F25DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F25DATA0_FD6                        CAN_F25DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F25DATA0_FD7_Pos                    (7U)                              
#define CAN_F25DATA0_FD7_Msk                    (0x1U << CAN_F25DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F25DATA0_FD7                        CAN_F25DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F25DATA0_FD8_Pos                    (8U)                              
#define CAN_F25DATA0_FD8_Msk                    (0x1U << CAN_F25DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F25DATA0_FD8                        CAN_F25DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F25DATA0_FD9_Pos                    (9U)                              
#define CAN_F25DATA0_FD9_Msk                    (0x1U << CAN_F25DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F25DATA0_FD9                        CAN_F25DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F25DATA0_FD10_Pos                   (10U)                             
#define CAN_F25DATA0_FD10_Msk                   (0x1U << CAN_F25DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F25DATA0_FD10                       CAN_F25DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F25DATA0_FD11_Pos                   (11U)                             
#define CAN_F25DATA0_FD11_Msk                   (0x1U << CAN_F25DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F25DATA0_FD11                       CAN_F25DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F25DATA0_FD12_Pos                   (12U)                             
#define CAN_F25DATA0_FD12_Msk                   (0x1U << CAN_F25DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F25DATA0_FD12                       CAN_F25DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F25DATA0_FD13_Pos                   (13U)                             
#define CAN_F25DATA0_FD13_Msk                   (0x1U << CAN_F25DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F25DATA0_FD13                       CAN_F25DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F25DATA0_FD14_Pos                   (14U)                             
#define CAN_F25DATA0_FD14_Msk                   (0x1U << CAN_F25DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F25DATA0_FD14                       CAN_F25DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F25DATA0_FD15_Pos                   (15U)                             
#define CAN_F25DATA0_FD15_Msk                   (0x1U << CAN_F25DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F25DATA0_FD15                       CAN_F25DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F25DATA0_FD16_Pos                   (16U)                             
#define CAN_F25DATA0_FD16_Msk                   (0x1U << CAN_F25DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F25DATA0_FD16                       CAN_F25DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F25DATA0_FD17_Pos                   (17U)                             
#define CAN_F25DATA0_FD17_Msk                   (0x1U << CAN_F25DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F25DATA0_FD17                       CAN_F25DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F25DATA0_FD18_Pos                   (18U)                             
#define CAN_F25DATA0_FD18_Msk                   (0x1U << CAN_F25DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F25DATA0_FD18                       CAN_F25DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F25DATA0_FD19_Pos                   (19U)                             
#define CAN_F25DATA0_FD19_Msk                   (0x1U << CAN_F25DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F25DATA0_FD19                       CAN_F25DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F25DATA0_FD20_Pos                   (20U)                             
#define CAN_F25DATA0_FD20_Msk                   (0x1U << CAN_F25DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F25DATA0_FD20                       CAN_F25DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F25DATA0_FD21_Pos                   (21U)                             
#define CAN_F25DATA0_FD21_Msk                   (0x1U << CAN_F25DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F25DATA0_FD21                       CAN_F25DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F25DATA0_FD22_Pos                   (22U)                             
#define CAN_F25DATA0_FD22_Msk                   (0x1U << CAN_F25DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F25DATA0_FD22                       CAN_F25DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F25DATA0_FD23_Pos                   (23U)                             
#define CAN_F25DATA0_FD23_Msk                   (0x1U << CAN_F25DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F25DATA0_FD23                       CAN_F25DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F25DATA0_FD24_Pos                   (24U)                             
#define CAN_F25DATA0_FD24_Msk                   (0x1U << CAN_F25DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F25DATA0_FD24                       CAN_F25DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F25DATA0_FD25_Pos                   (25U)                             
#define CAN_F25DATA0_FD25_Msk                   (0x1U << CAN_F25DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F25DATA0_FD25                       CAN_F25DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F25DATA0_FD26_Pos                   (26U)                             
#define CAN_F25DATA0_FD26_Msk                   (0x1U << CAN_F25DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F25DATA0_FD26                       CAN_F25DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F25DATA0_FD27_Pos                   (27U)                             
#define CAN_F25DATA0_FD27_Msk                   (0x1U << CAN_F25DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F25DATA0_FD27                       CAN_F25DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F25DATA0_FD28_Pos                   (28U)                             
#define CAN_F25DATA0_FD28_Msk                   (0x1U << CAN_F25DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F25DATA0_FD28                       CAN_F25DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F25DATA0_FD29_Pos                   (29U)                             
#define CAN_F25DATA0_FD29_Msk                   (0x1U << CAN_F25DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F25DATA0_FD29                       CAN_F25DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F25DATA0_FD30_Pos                   (30U)                             
#define CAN_F25DATA0_FD30_Msk                   (0x1U << CAN_F25DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F25DATA0_FD30                       CAN_F25DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F25DATA0_FD31_Pos                   (31U)                             
#define CAN_F25DATA0_FD31_Msk                   (0x1U << CAN_F25DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F25DATA0_FD31                       CAN_F25DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F26R1 register  ******************/
#define CAN_F26DATA0_FD0_Pos                    (0U)                              
#define CAN_F26DATA0_FD0_Msk                    (0x1U << CAN_F26DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F26DATA0_FD0                        CAN_F26DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F26DATA0_FD1_Pos                    (1U)                              
#define CAN_F26DATA0_FD1_Msk                    (0x1U << CAN_F26DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F26DATA0_FD1                        CAN_F26DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F26DATA0_FD2_Pos                    (2U)                              
#define CAN_F26DATA0_FD2_Msk                    (0x1U << CAN_F26DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F26DATA0_FD2                        CAN_F26DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F26DATA0_FD3_Pos                    (3U)                              
#define CAN_F26DATA0_FD3_Msk                    (0x1U << CAN_F26DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F26DATA0_FD3                        CAN_F26DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F26DATA0_FD4_Pos                    (4U)                              
#define CAN_F26DATA0_FD4_Msk                    (0x1U << CAN_F26DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F26DATA0_FD4                        CAN_F26DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F26DATA0_FD5_Pos                    (5U)                              
#define CAN_F26DATA0_FD5_Msk                    (0x1U << CAN_F26DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F26DATA0_FD5                        CAN_F26DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F26DATA0_FD6_Pos                    (6U)                              
#define CAN_F26DATA0_FD6_Msk                    (0x1U << CAN_F26DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F26DATA0_FD6                        CAN_F26DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F26DATA0_FD7_Pos                    (7U)                              
#define CAN_F26DATA0_FD7_Msk                    (0x1U << CAN_F26DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F26DATA0_FD7                        CAN_F26DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F26DATA0_FD8_Pos                    (8U)                              
#define CAN_F26DATA0_FD8_Msk                    (0x1U << CAN_F26DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F26DATA0_FD8                        CAN_F26DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F26DATA0_FD9_Pos                    (9U)                              
#define CAN_F26DATA0_FD9_Msk                    (0x1U << CAN_F26DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F26DATA0_FD9                        CAN_F26DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F26DATA0_FD10_Pos                   (10U)                             
#define CAN_F26DATA0_FD10_Msk                   (0x1U << CAN_F26DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F26DATA0_FD10                       CAN_F26DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F26DATA0_FD11_Pos                   (11U)                             
#define CAN_F26DATA0_FD11_Msk                   (0x1U << CAN_F26DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F26DATA0_FD11                       CAN_F26DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F26DATA0_FD12_Pos                   (12U)                             
#define CAN_F26DATA0_FD12_Msk                   (0x1U << CAN_F26DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F26DATA0_FD12                       CAN_F26DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F26DATA0_FD13_Pos                   (13U)                             
#define CAN_F26DATA0_FD13_Msk                   (0x1U << CAN_F26DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F26DATA0_FD13                       CAN_F26DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F26DATA0_FD14_Pos                   (14U)                             
#define CAN_F26DATA0_FD14_Msk                   (0x1U << CAN_F26DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F26DATA0_FD14                       CAN_F26DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F26DATA0_FD15_Pos                   (15U)                             
#define CAN_F26DATA0_FD15_Msk                   (0x1U << CAN_F26DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F26DATA0_FD15                       CAN_F26DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F26DATA0_FD16_Pos                   (16U)                             
#define CAN_F26DATA0_FD16_Msk                   (0x1U << CAN_F26DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F26DATA0_FD16                       CAN_F26DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F26DATA0_FD17_Pos                   (17U)                             
#define CAN_F26DATA0_FD17_Msk                   (0x1U << CAN_F26DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F26DATA0_FD17                       CAN_F26DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F26DATA0_FD18_Pos                   (18U)                             
#define CAN_F26DATA0_FD18_Msk                   (0x1U << CAN_F26DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F26DATA0_FD18                       CAN_F26DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F26DATA0_FD19_Pos                   (19U)                             
#define CAN_F26DATA0_FD19_Msk                   (0x1U << CAN_F26DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F26DATA0_FD19                       CAN_F26DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F26DATA0_FD20_Pos                   (20U)                             
#define CAN_F26DATA0_FD20_Msk                   (0x1U << CAN_F26DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F26DATA0_FD20                       CAN_F26DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F26DATA0_FD21_Pos                   (21U)                             
#define CAN_F26DATA0_FD21_Msk                   (0x1U << CAN_F26DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F26DATA0_FD21                       CAN_F26DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F26DATA0_FD22_Pos                   (22U)                             
#define CAN_F26DATA0_FD22_Msk                   (0x1U << CAN_F26DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F26DATA0_FD22                       CAN_F26DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F26DATA0_FD23_Pos                   (23U)                             
#define CAN_F26DATA0_FD23_Msk                   (0x1U << CAN_F26DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F26DATA0_FD23                       CAN_F26DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F26DATA0_FD24_Pos                   (24U)                             
#define CAN_F26DATA0_FD24_Msk                   (0x1U << CAN_F26DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F26DATA0_FD24                       CAN_F26DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F26DATA0_FD25_Pos                   (25U)                             
#define CAN_F26DATA0_FD25_Msk                   (0x1U << CAN_F26DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F26DATA0_FD25                       CAN_F26DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F26DATA0_FD26_Pos                   (26U)                             
#define CAN_F26DATA0_FD26_Msk                   (0x1U << CAN_F26DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F26DATA0_FD26                       CAN_F26DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F26DATA0_FD27_Pos                   (27U)                             
#define CAN_F26DATA0_FD27_Msk                   (0x1U << CAN_F26DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F26DATA0_FD27                       CAN_F26DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F26DATA0_FD28_Pos                   (28U)                             
#define CAN_F26DATA0_FD28_Msk                   (0x1U << CAN_F26DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F26DATA0_FD28                       CAN_F26DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F26DATA0_FD29_Pos                   (29U)                             
#define CAN_F26DATA0_FD29_Msk                   (0x1U << CAN_F26DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F26DATA0_FD29                       CAN_F26DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F26DATA0_FD30_Pos                   (30U)                             
#define CAN_F26DATA0_FD30_Msk                   (0x1U << CAN_F26DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F26DATA0_FD30                       CAN_F26DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F26DATA0_FD31_Pos                   (31U)                             
#define CAN_F26DATA0_FD31_Msk                   (0x1U << CAN_F26DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F26DATA0_FD31                       CAN_F26DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F27R1 register  ******************/
#define CAN_F27DATA0_FD0_Pos                    (0U)                              
#define CAN_F27DATA0_FD0_Msk                    (0x1U << CAN_F27DATA0_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F27DATA0_FD0                        CAN_F27DATA0_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F27DATA0_FD1_Pos                    (1U)                              
#define CAN_F27DATA0_FD1_Msk                    (0x1U << CAN_F27DATA0_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F27DATA0_FD1                        CAN_F27DATA0_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F27DATA0_FD2_Pos                    (2U)                              
#define CAN_F27DATA0_FD2_Msk                    (0x1U << CAN_F27DATA0_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F27DATA0_FD2                        CAN_F27DATA0_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F27DATA0_FD3_Pos                    (3U)                              
#define CAN_F27DATA0_FD3_Msk                    (0x1U << CAN_F27DATA0_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F27DATA0_FD3                        CAN_F27DATA0_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F27DATA0_FD4_Pos                    (4U)                              
#define CAN_F27DATA0_FD4_Msk                    (0x1U << CAN_F27DATA0_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F27DATA0_FD4                        CAN_F27DATA0_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F27DATA0_FD5_Pos                    (5U)                              
#define CAN_F27DATA0_FD5_Msk                    (0x1U << CAN_F27DATA0_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F27DATA0_FD5                        CAN_F27DATA0_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F27DATA0_FD6_Pos                    (6U)                              
#define CAN_F27DATA0_FD6_Msk                    (0x1U << CAN_F27DATA0_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F27DATA0_FD6                        CAN_F27DATA0_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F27DATA0_FD7_Pos                    (7U)                              
#define CAN_F27DATA0_FD7_Msk                    (0x1U << CAN_F27DATA0_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F27DATA0_FD7                        CAN_F27DATA0_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F27DATA0_FD8_Pos                    (8U)                              
#define CAN_F27DATA0_FD8_Msk                    (0x1U << CAN_F27DATA0_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F27DATA0_FD8                        CAN_F27DATA0_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F27DATA0_FD9_Pos                    (9U)                              
#define CAN_F27DATA0_FD9_Msk                    (0x1U << CAN_F27DATA0_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F27DATA0_FD9                        CAN_F27DATA0_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F27DATA0_FD10_Pos                   (10U)                             
#define CAN_F27DATA0_FD10_Msk                   (0x1U << CAN_F27DATA0_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F27DATA0_FD10                       CAN_F27DATA0_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F27DATA0_FD11_Pos                   (11U)                             
#define CAN_F27DATA0_FD11_Msk                   (0x1U << CAN_F27DATA0_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F27DATA0_FD11                       CAN_F27DATA0_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F27DATA0_FD12_Pos                   (12U)                             
#define CAN_F27DATA0_FD12_Msk                   (0x1U << CAN_F27DATA0_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F27DATA0_FD12                       CAN_F27DATA0_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F27DATA0_FD13_Pos                   (13U)                             
#define CAN_F27DATA0_FD13_Msk                   (0x1U << CAN_F27DATA0_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F27DATA0_FD13                       CAN_F27DATA0_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F27DATA0_FD14_Pos                   (14U)                             
#define CAN_F27DATA0_FD14_Msk                   (0x1U << CAN_F27DATA0_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F27DATA0_FD14                       CAN_F27DATA0_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F27DATA0_FD15_Pos                   (15U)                             
#define CAN_F27DATA0_FD15_Msk                   (0x1U << CAN_F27DATA0_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F27DATA0_FD15                       CAN_F27DATA0_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F27DATA0_FD16_Pos                   (16U)                             
#define CAN_F27DATA0_FD16_Msk                   (0x1U << CAN_F27DATA0_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F27DATA0_FD16                       CAN_F27DATA0_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F27DATA0_FD17_Pos                   (17U)                             
#define CAN_F27DATA0_FD17_Msk                   (0x1U << CAN_F27DATA0_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F27DATA0_FD17                       CAN_F27DATA0_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F27DATA0_FD18_Pos                   (18U)                             
#define CAN_F27DATA0_FD18_Msk                   (0x1U << CAN_F27DATA0_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F27DATA0_FD18                       CAN_F27DATA0_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F27DATA0_FD19_Pos                   (19U)                             
#define CAN_F27DATA0_FD19_Msk                   (0x1U << CAN_F27DATA0_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F27DATA0_FD19                       CAN_F27DATA0_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F27DATA0_FD20_Pos                   (20U)                             
#define CAN_F27DATA0_FD20_Msk                   (0x1U << CAN_F27DATA0_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F27DATA0_FD20                       CAN_F27DATA0_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F27DATA0_FD21_Pos                   (21U)                             
#define CAN_F27DATA0_FD21_Msk                   (0x1U << CAN_F27DATA0_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F27DATA0_FD21                       CAN_F27DATA0_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F27DATA0_FD22_Pos                   (22U)                             
#define CAN_F27DATA0_FD22_Msk                   (0x1U << CAN_F27DATA0_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F27DATA0_FD22                       CAN_F27DATA0_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F27DATA0_FD23_Pos                   (23U)                             
#define CAN_F27DATA0_FD23_Msk                   (0x1U << CAN_F27DATA0_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F27DATA0_FD23                       CAN_F27DATA0_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F27DATA0_FD24_Pos                   (24U)                             
#define CAN_F27DATA0_FD24_Msk                   (0x1U << CAN_F27DATA0_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F27DATA0_FD24                       CAN_F27DATA0_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F27DATA0_FD25_Pos                   (25U)                             
#define CAN_F27DATA0_FD25_Msk                   (0x1U << CAN_F27DATA0_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F27DATA0_FD25                       CAN_F27DATA0_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F27DATA0_FD26_Pos                   (26U)                             
#define CAN_F27DATA0_FD26_Msk                   (0x1U << CAN_F27DATA0_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F27DATA0_FD26                       CAN_F27DATA0_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F27DATA0_FD27_Pos                   (27U)                             
#define CAN_F27DATA0_FD27_Msk                   (0x1U << CAN_F27DATA0_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F27DATA0_FD27                       CAN_F27DATA0_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F27DATA0_FD28_Pos                   (28U)                             
#define CAN_F27DATA0_FD28_Msk                   (0x1U << CAN_F27DATA0_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F27DATA0_FD28                       CAN_F27DATA0_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F27DATA0_FD29_Pos                   (29U)                             
#define CAN_F27DATA0_FD29_Msk                   (0x1U << CAN_F27DATA0_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F27DATA0_FD29                       CAN_F27DATA0_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F27DATA0_FD30_Pos                   (30U)                             
#define CAN_F27DATA0_FD30_Msk                   (0x1U << CAN_F27DATA0_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F27DATA0_FD30                       CAN_F27DATA0_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F27DATA0_FD31_Pos                   (31U)                             
#define CAN_F27DATA0_FD31_Msk                   (0x1U << CAN_F27DATA0_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F27DATA0_FD31                       CAN_F27DATA0_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F0R2 register  *******************/
#define CAN_F0DATA1_FD0_Pos                     (0U)                              
#define CAN_F0DATA1_FD0_Msk                     (0x1U << CAN_F0DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F0DATA1_FD0                         CAN_F0DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F0DATA1_FD1_Pos                     (1U)                              
#define CAN_F0DATA1_FD1_Msk                     (0x1U << CAN_F0DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F0DATA1_FD1                         CAN_F0DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F0DATA1_FD2_Pos                     (2U)                              
#define CAN_F0DATA1_FD2_Msk                     (0x1U << CAN_F0DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F0DATA1_FD2                         CAN_F0DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F0DATA1_FD3_Pos                     (3U)                              
#define CAN_F0DATA1_FD3_Msk                     (0x1U << CAN_F0DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F0DATA1_FD3                         CAN_F0DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F0DATA1_FD4_Pos                     (4U)                              
#define CAN_F0DATA1_FD4_Msk                     (0x1U << CAN_F0DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F0DATA1_FD4                         CAN_F0DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F0DATA1_FD5_Pos                     (5U)                              
#define CAN_F0DATA1_FD5_Msk                     (0x1U << CAN_F0DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F0DATA1_FD5                         CAN_F0DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F0DATA1_FD6_Pos                     (6U)                              
#define CAN_F0DATA1_FD6_Msk                     (0x1U << CAN_F0DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F0DATA1_FD6                         CAN_F0DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F0DATA1_FD7_Pos                     (7U)                              
#define CAN_F0DATA1_FD7_Msk                     (0x1U << CAN_F0DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F0DATA1_FD7                         CAN_F0DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F0DATA1_FD8_Pos                     (8U)                              
#define CAN_F0DATA1_FD8_Msk                     (0x1U << CAN_F0DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F0DATA1_FD8                         CAN_F0DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F0DATA1_FD9_Pos                     (9U)                              
#define CAN_F0DATA1_FD9_Msk                     (0x1U << CAN_F0DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F0DATA1_FD9                         CAN_F0DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F0DATA1_FD10_Pos                    (10U)                             
#define CAN_F0DATA1_FD10_Msk                    (0x1U << CAN_F0DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F0DATA1_FD10                        CAN_F0DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F0DATA1_FD11_Pos                    (11U)                             
#define CAN_F0DATA1_FD11_Msk                    (0x1U << CAN_F0DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F0DATA1_FD11                        CAN_F0DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F0DATA1_FD12_Pos                    (12U)                             
#define CAN_F0DATA1_FD12_Msk                    (0x1U << CAN_F0DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F0DATA1_FD12                        CAN_F0DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F0DATA1_FD13_Pos                    (13U)                             
#define CAN_F0DATA1_FD13_Msk                    (0x1U << CAN_F0DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F0DATA1_FD13                        CAN_F0DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F0DATA1_FD14_Pos                    (14U)                             
#define CAN_F0DATA1_FD14_Msk                    (0x1U << CAN_F0DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F0DATA1_FD14                        CAN_F0DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F0DATA1_FD15_Pos                    (15U)                             
#define CAN_F0DATA1_FD15_Msk                    (0x1U << CAN_F0DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F0DATA1_FD15                        CAN_F0DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F0DATA1_FD16_Pos                    (16U)                             
#define CAN_F0DATA1_FD16_Msk                    (0x1U << CAN_F0DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F0DATA1_FD16                        CAN_F0DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F0DATA1_FD17_Pos                    (17U)                             
#define CAN_F0DATA1_FD17_Msk                    (0x1U << CAN_F0DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F0DATA1_FD17                        CAN_F0DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F0DATA1_FD18_Pos                    (18U)                             
#define CAN_F0DATA1_FD18_Msk                    (0x1U << CAN_F0DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F0DATA1_FD18                        CAN_F0DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F0DATA1_FD19_Pos                    (19U)                             
#define CAN_F0DATA1_FD19_Msk                    (0x1U << CAN_F0DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F0DATA1_FD19                        CAN_F0DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F0DATA1_FD20_Pos                    (20U)                             
#define CAN_F0DATA1_FD20_Msk                    (0x1U << CAN_F0DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F0DATA1_FD20                        CAN_F0DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F0DATA1_FD21_Pos                    (21U)                             
#define CAN_F0DATA1_FD21_Msk                    (0x1U << CAN_F0DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F0DATA1_FD21                        CAN_F0DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F0DATA1_FD22_Pos                    (22U)                             
#define CAN_F0DATA1_FD22_Msk                    (0x1U << CAN_F0DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F0DATA1_FD22                        CAN_F0DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F0DATA1_FD23_Pos                    (23U)                             
#define CAN_F0DATA1_FD23_Msk                    (0x1U << CAN_F0DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F0DATA1_FD23                        CAN_F0DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F0DATA1_FD24_Pos                    (24U)                             
#define CAN_F0DATA1_FD24_Msk                    (0x1U << CAN_F0DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F0DATA1_FD24                        CAN_F0DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F0DATA1_FD25_Pos                    (25U)                             
#define CAN_F0DATA1_FD25_Msk                    (0x1U << CAN_F0DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F0DATA1_FD25                        CAN_F0DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F0DATA1_FD26_Pos                    (26U)                             
#define CAN_F0DATA1_FD26_Msk                    (0x1U << CAN_F0DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F0DATA1_FD26                        CAN_F0DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F0DATA1_FD27_Pos                    (27U)                             
#define CAN_F0DATA1_FD27_Msk                    (0x1U << CAN_F0DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F0DATA1_FD27                        CAN_F0DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F0DATA1_FD28_Pos                    (28U)                             
#define CAN_F0DATA1_FD28_Msk                    (0x1U << CAN_F0DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F0DATA1_FD28                        CAN_F0DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F0DATA1_FD29_Pos                    (29U)                             
#define CAN_F0DATA1_FD29_Msk                    (0x1U << CAN_F0DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F0DATA1_FD29                        CAN_F0DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F0DATA1_FD30_Pos                    (30U)                             
#define CAN_F0DATA1_FD30_Msk                    (0x1U << CAN_F0DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F0DATA1_FD30                        CAN_F0DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F0DATA1_FD31_Pos                    (31U)                             
#define CAN_F0DATA1_FD31_Msk                    (0x1U << CAN_F0DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F0DATA1_FD31                        CAN_F0DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F1R2 register  *******************/
#define CAN_F1DATA1_FD0_Pos                     (0U)                              
#define CAN_F1DATA1_FD0_Msk                     (0x1U << CAN_F1DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F1DATA1_FD0                         CAN_F1DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F1DATA1_FD1_Pos                     (1U)                              
#define CAN_F1DATA1_FD1_Msk                     (0x1U << CAN_F1DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F1DATA1_FD1                         CAN_F1DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F1DATA1_FD2_Pos                     (2U)                              
#define CAN_F1DATA1_FD2_Msk                     (0x1U << CAN_F1DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F1DATA1_FD2                         CAN_F1DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F1DATA1_FD3_Pos                     (3U)                              
#define CAN_F1DATA1_FD3_Msk                     (0x1U << CAN_F1DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F1DATA1_FD3                         CAN_F1DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F1DATA1_FD4_Pos                     (4U)                              
#define CAN_F1DATA1_FD4_Msk                     (0x1U << CAN_F1DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F1DATA1_FD4                         CAN_F1DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F1DATA1_FD5_Pos                     (5U)                              
#define CAN_F1DATA1_FD5_Msk                     (0x1U << CAN_F1DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F1DATA1_FD5                         CAN_F1DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F1DATA1_FD6_Pos                     (6U)                              
#define CAN_F1DATA1_FD6_Msk                     (0x1U << CAN_F1DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F1DATA1_FD6                         CAN_F1DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F1DATA1_FD7_Pos                     (7U)                              
#define CAN_F1DATA1_FD7_Msk                     (0x1U << CAN_F1DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F1DATA1_FD7                         CAN_F1DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F1DATA1_FD8_Pos                     (8U)                              
#define CAN_F1DATA1_FD8_Msk                     (0x1U << CAN_F1DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F1DATA1_FD8                         CAN_F1DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F1DATA1_FD9_Pos                     (9U)                              
#define CAN_F1DATA1_FD9_Msk                     (0x1U << CAN_F1DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F1DATA1_FD9                         CAN_F1DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F1DATA1_FD10_Pos                    (10U)                             
#define CAN_F1DATA1_FD10_Msk                    (0x1U << CAN_F1DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F1DATA1_FD10                        CAN_F1DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F1DATA1_FD11_Pos                    (11U)                             
#define CAN_F1DATA1_FD11_Msk                    (0x1U << CAN_F1DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F1DATA1_FD11                        CAN_F1DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F1DATA1_FD12_Pos                    (12U)                             
#define CAN_F1DATA1_FD12_Msk                    (0x1U << CAN_F1DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F1DATA1_FD12                        CAN_F1DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F1DATA1_FD13_Pos                    (13U)                             
#define CAN_F1DATA1_FD13_Msk                    (0x1U << CAN_F1DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F1DATA1_FD13                        CAN_F1DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F1DATA1_FD14_Pos                    (14U)                             
#define CAN_F1DATA1_FD14_Msk                    (0x1U << CAN_F1DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F1DATA1_FD14                        CAN_F1DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F1DATA1_FD15_Pos                    (15U)                             
#define CAN_F1DATA1_FD15_Msk                    (0x1U << CAN_F1DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F1DATA1_FD15                        CAN_F1DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F1DATA1_FD16_Pos                    (16U)                             
#define CAN_F1DATA1_FD16_Msk                    (0x1U << CAN_F1DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F1DATA1_FD16                        CAN_F1DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F1DATA1_FD17_Pos                    (17U)                             
#define CAN_F1DATA1_FD17_Msk                    (0x1U << CAN_F1DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F1DATA1_FD17                        CAN_F1DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F1DATA1_FD18_Pos                    (18U)                             
#define CAN_F1DATA1_FD18_Msk                    (0x1U << CAN_F1DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F1DATA1_FD18                        CAN_F1DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F1DATA1_FD19_Pos                    (19U)                             
#define CAN_F1DATA1_FD19_Msk                    (0x1U << CAN_F1DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F1DATA1_FD19                        CAN_F1DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F1DATA1_FD20_Pos                    (20U)                             
#define CAN_F1DATA1_FD20_Msk                    (0x1U << CAN_F1DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F1DATA1_FD20                        CAN_F1DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F1DATA1_FD21_Pos                    (21U)                             
#define CAN_F1DATA1_FD21_Msk                    (0x1U << CAN_F1DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F1DATA1_FD21                        CAN_F1DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F1DATA1_FD22_Pos                    (22U)                             
#define CAN_F1DATA1_FD22_Msk                    (0x1U << CAN_F1DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F1DATA1_FD22                        CAN_F1DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F1DATA1_FD23_Pos                    (23U)                             
#define CAN_F1DATA1_FD23_Msk                    (0x1U << CAN_F1DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F1DATA1_FD23                        CAN_F1DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F1DATA1_FD24_Pos                    (24U)                             
#define CAN_F1DATA1_FD24_Msk                    (0x1U << CAN_F1DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F1DATA1_FD24                        CAN_F1DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F1DATA1_FD25_Pos                    (25U)                             
#define CAN_F1DATA1_FD25_Msk                    (0x1U << CAN_F1DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F1DATA1_FD25                        CAN_F1DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F1DATA1_FD26_Pos                    (26U)                             
#define CAN_F1DATA1_FD26_Msk                    (0x1U << CAN_F1DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F1DATA1_FD26                        CAN_F1DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F1DATA1_FD27_Pos                    (27U)                             
#define CAN_F1DATA1_FD27_Msk                    (0x1U << CAN_F1DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F1DATA1_FD27                        CAN_F1DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F1DATA1_FD28_Pos                    (28U)                             
#define CAN_F1DATA1_FD28_Msk                    (0x1U << CAN_F1DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F1DATA1_FD28                        CAN_F1DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F1DATA1_FD29_Pos                    (29U)                             
#define CAN_F1DATA1_FD29_Msk                    (0x1U << CAN_F1DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F1DATA1_FD29                        CAN_F1DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F1DATA1_FD30_Pos                    (30U)                             
#define CAN_F1DATA1_FD30_Msk                    (0x1U << CAN_F1DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F1DATA1_FD30                        CAN_F1DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F1DATA1_FD31_Pos                    (31U)                             
#define CAN_F1DATA1_FD31_Msk                    (0x1U << CAN_F1DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F1DATA1_FD31                        CAN_F1DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F2R2 register  *******************/
#define CAN_F2DATA1_FD0_Pos                     (0U)                              
#define CAN_F2DATA1_FD0_Msk                     (0x1U << CAN_F2DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F2DATA1_FD0                         CAN_F2DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F2DATA1_FD1_Pos                     (1U)                              
#define CAN_F2DATA1_FD1_Msk                     (0x1U << CAN_F2DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F2DATA1_FD1                         CAN_F2DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F2DATA1_FD2_Pos                     (2U)                              
#define CAN_F2DATA1_FD2_Msk                     (0x1U << CAN_F2DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F2DATA1_FD2                         CAN_F2DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F2DATA1_FD3_Pos                     (3U)                              
#define CAN_F2DATA1_FD3_Msk                     (0x1U << CAN_F2DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F2DATA1_FD3                         CAN_F2DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F2DATA1_FD4_Pos                     (4U)                              
#define CAN_F2DATA1_FD4_Msk                     (0x1U << CAN_F2DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F2DATA1_FD4                         CAN_F2DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F2DATA1_FD5_Pos                     (5U)                              
#define CAN_F2DATA1_FD5_Msk                     (0x1U << CAN_F2DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F2DATA1_FD5                         CAN_F2DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F2DATA1_FD6_Pos                     (6U)                              
#define CAN_F2DATA1_FD6_Msk                     (0x1U << CAN_F2DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F2DATA1_FD6                         CAN_F2DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F2DATA1_FD7_Pos                     (7U)                              
#define CAN_F2DATA1_FD7_Msk                     (0x1U << CAN_F2DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F2DATA1_FD7                         CAN_F2DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F2DATA1_FD8_Pos                     (8U)                              
#define CAN_F2DATA1_FD8_Msk                     (0x1U << CAN_F2DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F2DATA1_FD8                         CAN_F2DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F2DATA1_FD9_Pos                     (9U)                              
#define CAN_F2DATA1_FD9_Msk                     (0x1U << CAN_F2DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F2DATA1_FD9                         CAN_F2DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F2DATA1_FD10_Pos                    (10U)                             
#define CAN_F2DATA1_FD10_Msk                    (0x1U << CAN_F2DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F2DATA1_FD10                        CAN_F2DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F2DATA1_FD11_Pos                    (11U)                             
#define CAN_F2DATA1_FD11_Msk                    (0x1U << CAN_F2DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F2DATA1_FD11                        CAN_F2DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F2DATA1_FD12_Pos                    (12U)                             
#define CAN_F2DATA1_FD12_Msk                    (0x1U << CAN_F2DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F2DATA1_FD12                        CAN_F2DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F2DATA1_FD13_Pos                    (13U)                             
#define CAN_F2DATA1_FD13_Msk                    (0x1U << CAN_F2DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F2DATA1_FD13                        CAN_F2DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F2DATA1_FD14_Pos                    (14U)                             
#define CAN_F2DATA1_FD14_Msk                    (0x1U << CAN_F2DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F2DATA1_FD14                        CAN_F2DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F2DATA1_FD15_Pos                    (15U)                             
#define CAN_F2DATA1_FD15_Msk                    (0x1U << CAN_F2DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F2DATA1_FD15                        CAN_F2DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F2DATA1_FD16_Pos                    (16U)                             
#define CAN_F2DATA1_FD16_Msk                    (0x1U << CAN_F2DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F2DATA1_FD16                        CAN_F2DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F2DATA1_FD17_Pos                    (17U)                             
#define CAN_F2DATA1_FD17_Msk                    (0x1U << CAN_F2DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F2DATA1_FD17                        CAN_F2DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F2DATA1_FD18_Pos                    (18U)                             
#define CAN_F2DATA1_FD18_Msk                    (0x1U << CAN_F2DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F2DATA1_FD18                        CAN_F2DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F2DATA1_FD19_Pos                    (19U)                             
#define CAN_F2DATA1_FD19_Msk                    (0x1U << CAN_F2DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F2DATA1_FD19                        CAN_F2DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F2DATA1_FD20_Pos                    (20U)                             
#define CAN_F2DATA1_FD20_Msk                    (0x1U << CAN_F2DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F2DATA1_FD20                        CAN_F2DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F2DATA1_FD21_Pos                    (21U)                             
#define CAN_F2DATA1_FD21_Msk                    (0x1U << CAN_F2DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F2DATA1_FD21                        CAN_F2DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F2DATA1_FD22_Pos                    (22U)                             
#define CAN_F2DATA1_FD22_Msk                    (0x1U << CAN_F2DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F2DATA1_FD22                        CAN_F2DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F2DATA1_FD23_Pos                    (23U)                             
#define CAN_F2DATA1_FD23_Msk                    (0x1U << CAN_F2DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F2DATA1_FD23                        CAN_F2DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F2DATA1_FD24_Pos                    (24U)                             
#define CAN_F2DATA1_FD24_Msk                    (0x1U << CAN_F2DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F2DATA1_FD24                        CAN_F2DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F2DATA1_FD25_Pos                    (25U)                             
#define CAN_F2DATA1_FD25_Msk                    (0x1U << CAN_F2DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F2DATA1_FD25                        CAN_F2DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F2DATA1_FD26_Pos                    (26U)                             
#define CAN_F2DATA1_FD26_Msk                    (0x1U << CAN_F2DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F2DATA1_FD26                        CAN_F2DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F2DATA1_FD27_Pos                    (27U)                             
#define CAN_F2DATA1_FD27_Msk                    (0x1U << CAN_F2DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F2DATA1_FD27                        CAN_F2DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F2DATA1_FD28_Pos                    (28U)                             
#define CAN_F2DATA1_FD28_Msk                    (0x1U << CAN_F2DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F2DATA1_FD28                        CAN_F2DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F2DATA1_FD29_Pos                    (29U)                             
#define CAN_F2DATA1_FD29_Msk                    (0x1U << CAN_F2DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F2DATA1_FD29                        CAN_F2DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F2DATA1_FD30_Pos                    (30U)                             
#define CAN_F2DATA1_FD30_Msk                    (0x1U << CAN_F2DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F2DATA1_FD30                        CAN_F2DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F2DATA1_FD31_Pos                    (31U)                             
#define CAN_F2DATA1_FD31_Msk                    (0x1U << CAN_F2DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F2DATA1_FD31                        CAN_F2DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F3R2 register  *******************/
#define CAN_F3DATA1_FD0_Pos                     (0U)                              
#define CAN_F3DATA1_FD0_Msk                     (0x1U << CAN_F3DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F3DATA1_FD0                         CAN_F3DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F3DATA1_FD1_Pos                     (1U)                              
#define CAN_F3DATA1_FD1_Msk                     (0x1U << CAN_F3DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F3DATA1_FD1                         CAN_F3DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F3DATA1_FD2_Pos                     (2U)                              
#define CAN_F3DATA1_FD2_Msk                     (0x1U << CAN_F3DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F3DATA1_FD2                         CAN_F3DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F3DATA1_FD3_Pos                     (3U)                              
#define CAN_F3DATA1_FD3_Msk                     (0x1U << CAN_F3DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F3DATA1_FD3                         CAN_F3DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F3DATA1_FD4_Pos                     (4U)                              
#define CAN_F3DATA1_FD4_Msk                     (0x1U << CAN_F3DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F3DATA1_FD4                         CAN_F3DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F3DATA1_FD5_Pos                     (5U)                              
#define CAN_F3DATA1_FD5_Msk                     (0x1U << CAN_F3DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F3DATA1_FD5                         CAN_F3DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F3DATA1_FD6_Pos                     (6U)                              
#define CAN_F3DATA1_FD6_Msk                     (0x1U << CAN_F3DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F3DATA1_FD6                         CAN_F3DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F3DATA1_FD7_Pos                     (7U)                              
#define CAN_F3DATA1_FD7_Msk                     (0x1U << CAN_F3DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F3DATA1_FD7                         CAN_F3DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F3DATA1_FD8_Pos                     (8U)                              
#define CAN_F3DATA1_FD8_Msk                     (0x1U << CAN_F3DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F3DATA1_FD8                         CAN_F3DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F3DATA1_FD9_Pos                     (9U)                              
#define CAN_F3DATA1_FD9_Msk                     (0x1U << CAN_F3DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F3DATA1_FD9                         CAN_F3DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F3DATA1_FD10_Pos                    (10U)                             
#define CAN_F3DATA1_FD10_Msk                    (0x1U << CAN_F3DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F3DATA1_FD10                        CAN_F3DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F3DATA1_FD11_Pos                    (11U)                             
#define CAN_F3DATA1_FD11_Msk                    (0x1U << CAN_F3DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F3DATA1_FD11                        CAN_F3DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F3DATA1_FD12_Pos                    (12U)                             
#define CAN_F3DATA1_FD12_Msk                    (0x1U << CAN_F3DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F3DATA1_FD12                        CAN_F3DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F3DATA1_FD13_Pos                    (13U)                             
#define CAN_F3DATA1_FD13_Msk                    (0x1U << CAN_F3DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F3DATA1_FD13                        CAN_F3DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F3DATA1_FD14_Pos                    (14U)                             
#define CAN_F3DATA1_FD14_Msk                    (0x1U << CAN_F3DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F3DATA1_FD14                        CAN_F3DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F3DATA1_FD15_Pos                    (15U)                             
#define CAN_F3DATA1_FD15_Msk                    (0x1U << CAN_F3DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F3DATA1_FD15                        CAN_F3DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F3DATA1_FD16_Pos                    (16U)                             
#define CAN_F3DATA1_FD16_Msk                    (0x1U << CAN_F3DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F3DATA1_FD16                        CAN_F3DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F3DATA1_FD17_Pos                    (17U)                             
#define CAN_F3DATA1_FD17_Msk                    (0x1U << CAN_F3DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F3DATA1_FD17                        CAN_F3DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F3DATA1_FD18_Pos                    (18U)                             
#define CAN_F3DATA1_FD18_Msk                    (0x1U << CAN_F3DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F3DATA1_FD18                        CAN_F3DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F3DATA1_FD19_Pos                    (19U)                             
#define CAN_F3DATA1_FD19_Msk                    (0x1U << CAN_F3DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F3DATA1_FD19                        CAN_F3DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F3DATA1_FD20_Pos                    (20U)                             
#define CAN_F3DATA1_FD20_Msk                    (0x1U << CAN_F3DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F3DATA1_FD20                        CAN_F3DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F3DATA1_FD21_Pos                    (21U)                             
#define CAN_F3DATA1_FD21_Msk                    (0x1U << CAN_F3DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F3DATA1_FD21                        CAN_F3DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F3DATA1_FD22_Pos                    (22U)                             
#define CAN_F3DATA1_FD22_Msk                    (0x1U << CAN_F3DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F3DATA1_FD22                        CAN_F3DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F3DATA1_FD23_Pos                    (23U)                             
#define CAN_F3DATA1_FD23_Msk                    (0x1U << CAN_F3DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F3DATA1_FD23                        CAN_F3DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F3DATA1_FD24_Pos                    (24U)                             
#define CAN_F3DATA1_FD24_Msk                    (0x1U << CAN_F3DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F3DATA1_FD24                        CAN_F3DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F3DATA1_FD25_Pos                    (25U)                             
#define CAN_F3DATA1_FD25_Msk                    (0x1U << CAN_F3DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F3DATA1_FD25                        CAN_F3DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F3DATA1_FD26_Pos                    (26U)                             
#define CAN_F3DATA1_FD26_Msk                    (0x1U << CAN_F3DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F3DATA1_FD26                        CAN_F3DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F3DATA1_FD27_Pos                    (27U)                             
#define CAN_F3DATA1_FD27_Msk                    (0x1U << CAN_F3DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F3DATA1_FD27                        CAN_F3DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F3DATA1_FD28_Pos                    (28U)                             
#define CAN_F3DATA1_FD28_Msk                    (0x1U << CAN_F3DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F3DATA1_FD28                        CAN_F3DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F3DATA1_FD29_Pos                    (29U)                             
#define CAN_F3DATA1_FD29_Msk                    (0x1U << CAN_F3DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F3DATA1_FD29                        CAN_F3DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F3DATA1_FD30_Pos                    (30U)                             
#define CAN_F3DATA1_FD30_Msk                    (0x1U << CAN_F3DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F3DATA1_FD30                        CAN_F3DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F3DATA1_FD31_Pos                    (31U)                             
#define CAN_F3DATA1_FD31_Msk                    (0x1U << CAN_F3DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F3DATA1_FD31                        CAN_F3DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F4R2 register  *******************/
#define CAN_F4DATA1_FD0_Pos                     (0U)                              
#define CAN_F4DATA1_FD0_Msk                     (0x1U << CAN_F4DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F4DATA1_FD0                         CAN_F4DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F4DATA1_FD1_Pos                     (1U)                              
#define CAN_F4DATA1_FD1_Msk                     (0x1U << CAN_F4DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F4DATA1_FD1                         CAN_F4DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F4DATA1_FD2_Pos                     (2U)                              
#define CAN_F4DATA1_FD2_Msk                     (0x1U << CAN_F4DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F4DATA1_FD2                         CAN_F4DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F4DATA1_FD3_Pos                     (3U)                              
#define CAN_F4DATA1_FD3_Msk                     (0x1U << CAN_F4DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F4DATA1_FD3                         CAN_F4DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F4DATA1_FD4_Pos                     (4U)                              
#define CAN_F4DATA1_FD4_Msk                     (0x1U << CAN_F4DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F4DATA1_FD4                         CAN_F4DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F4DATA1_FD5_Pos                     (5U)                              
#define CAN_F4DATA1_FD5_Msk                     (0x1U << CAN_F4DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F4DATA1_FD5                         CAN_F4DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F4DATA1_FD6_Pos                     (6U)                              
#define CAN_F4DATA1_FD6_Msk                     (0x1U << CAN_F4DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F4DATA1_FD6                         CAN_F4DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F4DATA1_FD7_Pos                     (7U)                              
#define CAN_F4DATA1_FD7_Msk                     (0x1U << CAN_F4DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F4DATA1_FD7                         CAN_F4DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F4DATA1_FD8_Pos                     (8U)                              
#define CAN_F4DATA1_FD8_Msk                     (0x1U << CAN_F4DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F4DATA1_FD8                         CAN_F4DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F4DATA1_FD9_Pos                     (9U)                              
#define CAN_F4DATA1_FD9_Msk                     (0x1U << CAN_F4DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F4DATA1_FD9                         CAN_F4DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F4DATA1_FD10_Pos                    (10U)                             
#define CAN_F4DATA1_FD10_Msk                    (0x1U << CAN_F4DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F4DATA1_FD10                        CAN_F4DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F4DATA1_FD11_Pos                    (11U)                             
#define CAN_F4DATA1_FD11_Msk                    (0x1U << CAN_F4DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F4DATA1_FD11                        CAN_F4DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F4DATA1_FD12_Pos                    (12U)                             
#define CAN_F4DATA1_FD12_Msk                    (0x1U << CAN_F4DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F4DATA1_FD12                        CAN_F4DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F4DATA1_FD13_Pos                    (13U)                             
#define CAN_F4DATA1_FD13_Msk                    (0x1U << CAN_F4DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F4DATA1_FD13                        CAN_F4DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F4DATA1_FD14_Pos                    (14U)                             
#define CAN_F4DATA1_FD14_Msk                    (0x1U << CAN_F4DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F4DATA1_FD14                        CAN_F4DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F4DATA1_FD15_Pos                    (15U)                             
#define CAN_F4DATA1_FD15_Msk                    (0x1U << CAN_F4DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F4DATA1_FD15                        CAN_F4DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F4DATA1_FD16_Pos                    (16U)                             
#define CAN_F4DATA1_FD16_Msk                    (0x1U << CAN_F4DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F4DATA1_FD16                        CAN_F4DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F4DATA1_FD17_Pos                    (17U)                             
#define CAN_F4DATA1_FD17_Msk                    (0x1U << CAN_F4DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F4DATA1_FD17                        CAN_F4DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F4DATA1_FD18_Pos                    (18U)                             
#define CAN_F4DATA1_FD18_Msk                    (0x1U << CAN_F4DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F4DATA1_FD18                        CAN_F4DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F4DATA1_FD19_Pos                    (19U)                             
#define CAN_F4DATA1_FD19_Msk                    (0x1U << CAN_F4DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F4DATA1_FD19                        CAN_F4DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F4DATA1_FD20_Pos                    (20U)                             
#define CAN_F4DATA1_FD20_Msk                    (0x1U << CAN_F4DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F4DATA1_FD20                        CAN_F4DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F4DATA1_FD21_Pos                    (21U)                             
#define CAN_F4DATA1_FD21_Msk                    (0x1U << CAN_F4DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F4DATA1_FD21                        CAN_F4DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F4DATA1_FD22_Pos                    (22U)                             
#define CAN_F4DATA1_FD22_Msk                    (0x1U << CAN_F4DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F4DATA1_FD22                        CAN_F4DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F4DATA1_FD23_Pos                    (23U)                             
#define CAN_F4DATA1_FD23_Msk                    (0x1U << CAN_F4DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F4DATA1_FD23                        CAN_F4DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F4DATA1_FD24_Pos                    (24U)                             
#define CAN_F4DATA1_FD24_Msk                    (0x1U << CAN_F4DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F4DATA1_FD24                        CAN_F4DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F4DATA1_FD25_Pos                    (25U)                             
#define CAN_F4DATA1_FD25_Msk                    (0x1U << CAN_F4DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F4DATA1_FD25                        CAN_F4DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F4DATA1_FD26_Pos                    (26U)                             
#define CAN_F4DATA1_FD26_Msk                    (0x1U << CAN_F4DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F4DATA1_FD26                        CAN_F4DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F4DATA1_FD27_Pos                    (27U)                             
#define CAN_F4DATA1_FD27_Msk                    (0x1U << CAN_F4DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F4DATA1_FD27                        CAN_F4DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F4DATA1_FD28_Pos                    (28U)                             
#define CAN_F4DATA1_FD28_Msk                    (0x1U << CAN_F4DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F4DATA1_FD28                        CAN_F4DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F4DATA1_FD29_Pos                    (29U)                             
#define CAN_F4DATA1_FD29_Msk                    (0x1U << CAN_F4DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F4DATA1_FD29                        CAN_F4DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F4DATA1_FD30_Pos                    (30U)                             
#define CAN_F4DATA1_FD30_Msk                    (0x1U << CAN_F4DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F4DATA1_FD30                        CAN_F4DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F4DATA1_FD31_Pos                    (31U)                             
#define CAN_F4DATA1_FD31_Msk                    (0x1U << CAN_F4DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F4DATA1_FD31                        CAN_F4DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F5R2 register  *******************/
#define CAN_F5DATA1_FD0_Pos                     (0U)                              
#define CAN_F5DATA1_FD0_Msk                     (0x1U << CAN_F5DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F5DATA1_FD0                         CAN_F5DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F5DATA1_FD1_Pos                     (1U)                              
#define CAN_F5DATA1_FD1_Msk                     (0x1U << CAN_F5DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F5DATA1_FD1                         CAN_F5DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F5DATA1_FD2_Pos                     (2U)                              
#define CAN_F5DATA1_FD2_Msk                     (0x1U << CAN_F5DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F5DATA1_FD2                         CAN_F5DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F5DATA1_FD3_Pos                     (3U)                              
#define CAN_F5DATA1_FD3_Msk                     (0x1U << CAN_F5DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F5DATA1_FD3                         CAN_F5DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F5DATA1_FD4_Pos                     (4U)                              
#define CAN_F5DATA1_FD4_Msk                     (0x1U << CAN_F5DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F5DATA1_FD4                         CAN_F5DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F5DATA1_FD5_Pos                     (5U)                              
#define CAN_F5DATA1_FD5_Msk                     (0x1U << CAN_F5DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F5DATA1_FD5                         CAN_F5DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F5DATA1_FD6_Pos                     (6U)                              
#define CAN_F5DATA1_FD6_Msk                     (0x1U << CAN_F5DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F5DATA1_FD6                         CAN_F5DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F5DATA1_FD7_Pos                     (7U)                              
#define CAN_F5DATA1_FD7_Msk                     (0x1U << CAN_F5DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F5DATA1_FD7                         CAN_F5DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F5DATA1_FD8_Pos                     (8U)                              
#define CAN_F5DATA1_FD8_Msk                     (0x1U << CAN_F5DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F5DATA1_FD8                         CAN_F5DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F5DATA1_FD9_Pos                     (9U)                              
#define CAN_F5DATA1_FD9_Msk                     (0x1U << CAN_F5DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F5DATA1_FD9                         CAN_F5DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F5DATA1_FD10_Pos                    (10U)                             
#define CAN_F5DATA1_FD10_Msk                    (0x1U << CAN_F5DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F5DATA1_FD10                        CAN_F5DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F5DATA1_FD11_Pos                    (11U)                             
#define CAN_F5DATA1_FD11_Msk                    (0x1U << CAN_F5DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F5DATA1_FD11                        CAN_F5DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F5DATA1_FD12_Pos                    (12U)                             
#define CAN_F5DATA1_FD12_Msk                    (0x1U << CAN_F5DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F5DATA1_FD12                        CAN_F5DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F5DATA1_FD13_Pos                    (13U)                             
#define CAN_F5DATA1_FD13_Msk                    (0x1U << CAN_F5DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F5DATA1_FD13                        CAN_F5DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F5DATA1_FD14_Pos                    (14U)                             
#define CAN_F5DATA1_FD14_Msk                    (0x1U << CAN_F5DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F5DATA1_FD14                        CAN_F5DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F5DATA1_FD15_Pos                    (15U)                             
#define CAN_F5DATA1_FD15_Msk                    (0x1U << CAN_F5DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F5DATA1_FD15                        CAN_F5DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F5DATA1_FD16_Pos                    (16U)                             
#define CAN_F5DATA1_FD16_Msk                    (0x1U << CAN_F5DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F5DATA1_FD16                        CAN_F5DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F5DATA1_FD17_Pos                    (17U)                             
#define CAN_F5DATA1_FD17_Msk                    (0x1U << CAN_F5DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F5DATA1_FD17                        CAN_F5DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F5DATA1_FD18_Pos                    (18U)                             
#define CAN_F5DATA1_FD18_Msk                    (0x1U << CAN_F5DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F5DATA1_FD18                        CAN_F5DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F5DATA1_FD19_Pos                    (19U)                             
#define CAN_F5DATA1_FD19_Msk                    (0x1U << CAN_F5DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F5DATA1_FD19                        CAN_F5DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F5DATA1_FD20_Pos                    (20U)                             
#define CAN_F5DATA1_FD20_Msk                    (0x1U << CAN_F5DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F5DATA1_FD20                        CAN_F5DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F5DATA1_FD21_Pos                    (21U)                             
#define CAN_F5DATA1_FD21_Msk                    (0x1U << CAN_F5DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F5DATA1_FD21                        CAN_F5DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F5DATA1_FD22_Pos                    (22U)                             
#define CAN_F5DATA1_FD22_Msk                    (0x1U << CAN_F5DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F5DATA1_FD22                        CAN_F5DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F5DATA1_FD23_Pos                    (23U)                             
#define CAN_F5DATA1_FD23_Msk                    (0x1U << CAN_F5DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F5DATA1_FD23                        CAN_F5DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F5DATA1_FD24_Pos                    (24U)                             
#define CAN_F5DATA1_FD24_Msk                    (0x1U << CAN_F5DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F5DATA1_FD24                        CAN_F5DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F5DATA1_FD25_Pos                    (25U)                             
#define CAN_F5DATA1_FD25_Msk                    (0x1U << CAN_F5DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F5DATA1_FD25                        CAN_F5DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F5DATA1_FD26_Pos                    (26U)                             
#define CAN_F5DATA1_FD26_Msk                    (0x1U << CAN_F5DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F5DATA1_FD26                        CAN_F5DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F5DATA1_FD27_Pos                    (27U)                             
#define CAN_F5DATA1_FD27_Msk                    (0x1U << CAN_F5DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F5DATA1_FD27                        CAN_F5DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F5DATA1_FD28_Pos                    (28U)                             
#define CAN_F5DATA1_FD28_Msk                    (0x1U << CAN_F5DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F5DATA1_FD28                        CAN_F5DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F5DATA1_FD29_Pos                    (29U)                             
#define CAN_F5DATA1_FD29_Msk                    (0x1U << CAN_F5DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F5DATA1_FD29                        CAN_F5DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F5DATA1_FD30_Pos                    (30U)                             
#define CAN_F5DATA1_FD30_Msk                    (0x1U << CAN_F5DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F5DATA1_FD30                        CAN_F5DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F5DATA1_FD31_Pos                    (31U)                             
#define CAN_F5DATA1_FD31_Msk                    (0x1U << CAN_F5DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F5DATA1_FD31                        CAN_F5DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F6R2 register  *******************/
#define CAN_F6DATA1_FD0_Pos                     (0U)                              
#define CAN_F6DATA1_FD0_Msk                     (0x1U << CAN_F6DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F6DATA1_FD0                         CAN_F6DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F6DATA1_FD1_Pos                     (1U)                              
#define CAN_F6DATA1_FD1_Msk                     (0x1U << CAN_F6DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F6DATA1_FD1                         CAN_F6DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F6DATA1_FD2_Pos                     (2U)                              
#define CAN_F6DATA1_FD2_Msk                     (0x1U << CAN_F6DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F6DATA1_FD2                         CAN_F6DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F6DATA1_FD3_Pos                     (3U)                              
#define CAN_F6DATA1_FD3_Msk                     (0x1U << CAN_F6DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F6DATA1_FD3                         CAN_F6DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F6DATA1_FD4_Pos                     (4U)                              
#define CAN_F6DATA1_FD4_Msk                     (0x1U << CAN_F6DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F6DATA1_FD4                         CAN_F6DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F6DATA1_FD5_Pos                     (5U)                              
#define CAN_F6DATA1_FD5_Msk                     (0x1U << CAN_F6DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F6DATA1_FD5                         CAN_F6DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F6DATA1_FD6_Pos                     (6U)                              
#define CAN_F6DATA1_FD6_Msk                     (0x1U << CAN_F6DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F6DATA1_FD6                         CAN_F6DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F6DATA1_FD7_Pos                     (7U)                              
#define CAN_F6DATA1_FD7_Msk                     (0x1U << CAN_F6DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F6DATA1_FD7                         CAN_F6DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F6DATA1_FD8_Pos                     (8U)                              
#define CAN_F6DATA1_FD8_Msk                     (0x1U << CAN_F6DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F6DATA1_FD8                         CAN_F6DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F6DATA1_FD9_Pos                     (9U)                              
#define CAN_F6DATA1_FD9_Msk                     (0x1U << CAN_F6DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F6DATA1_FD9                         CAN_F6DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F6DATA1_FD10_Pos                    (10U)                             
#define CAN_F6DATA1_FD10_Msk                    (0x1U << CAN_F6DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F6DATA1_FD10                        CAN_F6DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F6DATA1_FD11_Pos                    (11U)                             
#define CAN_F6DATA1_FD11_Msk                    (0x1U << CAN_F6DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F6DATA1_FD11                        CAN_F6DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F6DATA1_FD12_Pos                    (12U)                             
#define CAN_F6DATA1_FD12_Msk                    (0x1U << CAN_F6DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F6DATA1_FD12                        CAN_F6DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F6DATA1_FD13_Pos                    (13U)                             
#define CAN_F6DATA1_FD13_Msk                    (0x1U << CAN_F6DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F6DATA1_FD13                        CAN_F6DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F6DATA1_FD14_Pos                    (14U)                             
#define CAN_F6DATA1_FD14_Msk                    (0x1U << CAN_F6DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F6DATA1_FD14                        CAN_F6DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F6DATA1_FD15_Pos                    (15U)                             
#define CAN_F6DATA1_FD15_Msk                    (0x1U << CAN_F6DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F6DATA1_FD15                        CAN_F6DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F6DATA1_FD16_Pos                    (16U)                             
#define CAN_F6DATA1_FD16_Msk                    (0x1U << CAN_F6DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F6DATA1_FD16                        CAN_F6DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F6DATA1_FD17_Pos                    (17U)                             
#define CAN_F6DATA1_FD17_Msk                    (0x1U << CAN_F6DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F6DATA1_FD17                        CAN_F6DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F6DATA1_FD18_Pos                    (18U)                             
#define CAN_F6DATA1_FD18_Msk                    (0x1U << CAN_F6DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F6DATA1_FD18                        CAN_F6DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F6DATA1_FD19_Pos                    (19U)                             
#define CAN_F6DATA1_FD19_Msk                    (0x1U << CAN_F6DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F6DATA1_FD19                        CAN_F6DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F6DATA1_FD20_Pos                    (20U)                             
#define CAN_F6DATA1_FD20_Msk                    (0x1U << CAN_F6DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F6DATA1_FD20                        CAN_F6DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F6DATA1_FD21_Pos                    (21U)                             
#define CAN_F6DATA1_FD21_Msk                    (0x1U << CAN_F6DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F6DATA1_FD21                        CAN_F6DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F6DATA1_FD22_Pos                    (22U)                             
#define CAN_F6DATA1_FD22_Msk                    (0x1U << CAN_F6DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F6DATA1_FD22                        CAN_F6DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F6DATA1_FD23_Pos                    (23U)                             
#define CAN_F6DATA1_FD23_Msk                    (0x1U << CAN_F6DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F6DATA1_FD23                        CAN_F6DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F6DATA1_FD24_Pos                    (24U)                             
#define CAN_F6DATA1_FD24_Msk                    (0x1U << CAN_F6DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F6DATA1_FD24                        CAN_F6DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F6DATA1_FD25_Pos                    (25U)                             
#define CAN_F6DATA1_FD25_Msk                    (0x1U << CAN_F6DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F6DATA1_FD25                        CAN_F6DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F6DATA1_FD26_Pos                    (26U)                             
#define CAN_F6DATA1_FD26_Msk                    (0x1U << CAN_F6DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F6DATA1_FD26                        CAN_F6DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F6DATA1_FD27_Pos                    (27U)                             
#define CAN_F6DATA1_FD27_Msk                    (0x1U << CAN_F6DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F6DATA1_FD27                        CAN_F6DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F6DATA1_FD28_Pos                    (28U)                             
#define CAN_F6DATA1_FD28_Msk                    (0x1U << CAN_F6DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F6DATA1_FD28                        CAN_F6DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F6DATA1_FD29_Pos                    (29U)                             
#define CAN_F6DATA1_FD29_Msk                    (0x1U << CAN_F6DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F6DATA1_FD29                        CAN_F6DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F6DATA1_FD30_Pos                    (30U)                             
#define CAN_F6DATA1_FD30_Msk                    (0x1U << CAN_F6DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F6DATA1_FD30                        CAN_F6DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F6DATA1_FD31_Pos                    (31U)                             
#define CAN_F6DATA1_FD31_Msk                    (0x1U << CAN_F6DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F6DATA1_FD31                        CAN_F6DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F7R2 register  *******************/
#define CAN_F7DATA1_FD0_Pos                     (0U)                              
#define CAN_F7DATA1_FD0_Msk                     (0x1U << CAN_F7DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F7DATA1_FD0                         CAN_F7DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F7DATA1_FD1_Pos                     (1U)                              
#define CAN_F7DATA1_FD1_Msk                     (0x1U << CAN_F7DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F7DATA1_FD1                         CAN_F7DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F7DATA1_FD2_Pos                     (2U)                              
#define CAN_F7DATA1_FD2_Msk                     (0x1U << CAN_F7DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F7DATA1_FD2                         CAN_F7DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F7DATA1_FD3_Pos                     (3U)                              
#define CAN_F7DATA1_FD3_Msk                     (0x1U << CAN_F7DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F7DATA1_FD3                         CAN_F7DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F7DATA1_FD4_Pos                     (4U)                              
#define CAN_F7DATA1_FD4_Msk                     (0x1U << CAN_F7DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F7DATA1_FD4                         CAN_F7DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F7DATA1_FD5_Pos                     (5U)                              
#define CAN_F7DATA1_FD5_Msk                     (0x1U << CAN_F7DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F7DATA1_FD5                         CAN_F7DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F7DATA1_FD6_Pos                     (6U)                              
#define CAN_F7DATA1_FD6_Msk                     (0x1U << CAN_F7DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F7DATA1_FD6                         CAN_F7DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F7DATA1_FD7_Pos                     (7U)                              
#define CAN_F7DATA1_FD7_Msk                     (0x1U << CAN_F7DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F7DATA1_FD7                         CAN_F7DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F7DATA1_FD8_Pos                     (8U)                              
#define CAN_F7DATA1_FD8_Msk                     (0x1U << CAN_F7DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F7DATA1_FD8                         CAN_F7DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F7DATA1_FD9_Pos                     (9U)                              
#define CAN_F7DATA1_FD9_Msk                     (0x1U << CAN_F7DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F7DATA1_FD9                         CAN_F7DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F7DATA1_FD10_Pos                    (10U)                             
#define CAN_F7DATA1_FD10_Msk                    (0x1U << CAN_F7DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F7DATA1_FD10                        CAN_F7DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F7DATA1_FD11_Pos                    (11U)                             
#define CAN_F7DATA1_FD11_Msk                    (0x1U << CAN_F7DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F7DATA1_FD11                        CAN_F7DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F7DATA1_FD12_Pos                    (12U)                             
#define CAN_F7DATA1_FD12_Msk                    (0x1U << CAN_F7DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F7DATA1_FD12                        CAN_F7DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F7DATA1_FD13_Pos                    (13U)                             
#define CAN_F7DATA1_FD13_Msk                    (0x1U << CAN_F7DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F7DATA1_FD13                        CAN_F7DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F7DATA1_FD14_Pos                    (14U)                             
#define CAN_F7DATA1_FD14_Msk                    (0x1U << CAN_F7DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F7DATA1_FD14                        CAN_F7DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F7DATA1_FD15_Pos                    (15U)                             
#define CAN_F7DATA1_FD15_Msk                    (0x1U << CAN_F7DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F7DATA1_FD15                        CAN_F7DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F7DATA1_FD16_Pos                    (16U)                             
#define CAN_F7DATA1_FD16_Msk                    (0x1U << CAN_F7DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F7DATA1_FD16                        CAN_F7DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F7DATA1_FD17_Pos                    (17U)                             
#define CAN_F7DATA1_FD17_Msk                    (0x1U << CAN_F7DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F7DATA1_FD17                        CAN_F7DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F7DATA1_FD18_Pos                    (18U)                             
#define CAN_F7DATA1_FD18_Msk                    (0x1U << CAN_F7DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F7DATA1_FD18                        CAN_F7DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F7DATA1_FD19_Pos                    (19U)                             
#define CAN_F7DATA1_FD19_Msk                    (0x1U << CAN_F7DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F7DATA1_FD19                        CAN_F7DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F7DATA1_FD20_Pos                    (20U)                             
#define CAN_F7DATA1_FD20_Msk                    (0x1U << CAN_F7DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F7DATA1_FD20                        CAN_F7DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F7DATA1_FD21_Pos                    (21U)                             
#define CAN_F7DATA1_FD21_Msk                    (0x1U << CAN_F7DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F7DATA1_FD21                        CAN_F7DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F7DATA1_FD22_Pos                    (22U)                             
#define CAN_F7DATA1_FD22_Msk                    (0x1U << CAN_F7DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F7DATA1_FD22                        CAN_F7DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F7DATA1_FD23_Pos                    (23U)                             
#define CAN_F7DATA1_FD23_Msk                    (0x1U << CAN_F7DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F7DATA1_FD23                        CAN_F7DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F7DATA1_FD24_Pos                    (24U)                             
#define CAN_F7DATA1_FD24_Msk                    (0x1U << CAN_F7DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F7DATA1_FD24                        CAN_F7DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F7DATA1_FD25_Pos                    (25U)                             
#define CAN_F7DATA1_FD25_Msk                    (0x1U << CAN_F7DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F7DATA1_FD25                        CAN_F7DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F7DATA1_FD26_Pos                    (26U)                             
#define CAN_F7DATA1_FD26_Msk                    (0x1U << CAN_F7DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F7DATA1_FD26                        CAN_F7DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F7DATA1_FD27_Pos                    (27U)                             
#define CAN_F7DATA1_FD27_Msk                    (0x1U << CAN_F7DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F7DATA1_FD27                        CAN_F7DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F7DATA1_FD28_Pos                    (28U)                             
#define CAN_F7DATA1_FD28_Msk                    (0x1U << CAN_F7DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F7DATA1_FD28                        CAN_F7DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F7DATA1_FD29_Pos                    (29U)                             
#define CAN_F7DATA1_FD29_Msk                    (0x1U << CAN_F7DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F7DATA1_FD29                        CAN_F7DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F7DATA1_FD30_Pos                    (30U)                             
#define CAN_F7DATA1_FD30_Msk                    (0x1U << CAN_F7DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F7DATA1_FD30                        CAN_F7DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F7DATA1_FD31_Pos                    (31U)                             
#define CAN_F7DATA1_FD31_Msk                    (0x1U << CAN_F7DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F7DATA1_FD31                        CAN_F7DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F8R2 register  *******************/
#define CAN_F8DATA1_FD0_Pos                     (0U)                              
#define CAN_F8DATA1_FD0_Msk                     (0x1U << CAN_F8DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F8DATA1_FD0                         CAN_F8DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F8DATA1_FD1_Pos                     (1U)                              
#define CAN_F8DATA1_FD1_Msk                     (0x1U << CAN_F8DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F8DATA1_FD1                         CAN_F8DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F8DATA1_FD2_Pos                     (2U)                              
#define CAN_F8DATA1_FD2_Msk                     (0x1U << CAN_F8DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F8DATA1_FD2                         CAN_F8DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F8DATA1_FD3_Pos                     (3U)                              
#define CAN_F8DATA1_FD3_Msk                     (0x1U << CAN_F8DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F8DATA1_FD3                         CAN_F8DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F8DATA1_FD4_Pos                     (4U)                              
#define CAN_F8DATA1_FD4_Msk                     (0x1U << CAN_F8DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F8DATA1_FD4                         CAN_F8DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F8DATA1_FD5_Pos                     (5U)                              
#define CAN_F8DATA1_FD5_Msk                     (0x1U << CAN_F8DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F8DATA1_FD5                         CAN_F8DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F8DATA1_FD6_Pos                     (6U)                              
#define CAN_F8DATA1_FD6_Msk                     (0x1U << CAN_F8DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F8DATA1_FD6                         CAN_F8DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F8DATA1_FD7_Pos                     (7U)                              
#define CAN_F8DATA1_FD7_Msk                     (0x1U << CAN_F8DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F8DATA1_FD7                         CAN_F8DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F8DATA1_FD8_Pos                     (8U)                              
#define CAN_F8DATA1_FD8_Msk                     (0x1U << CAN_F8DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F8DATA1_FD8                         CAN_F8DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F8DATA1_FD9_Pos                     (9U)                              
#define CAN_F8DATA1_FD9_Msk                     (0x1U << CAN_F8DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F8DATA1_FD9                         CAN_F8DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F8DATA1_FD10_Pos                    (10U)                             
#define CAN_F8DATA1_FD10_Msk                    (0x1U << CAN_F8DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F8DATA1_FD10                        CAN_F8DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F8DATA1_FD11_Pos                    (11U)                             
#define CAN_F8DATA1_FD11_Msk                    (0x1U << CAN_F8DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F8DATA1_FD11                        CAN_F8DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F8DATA1_FD12_Pos                    (12U)                             
#define CAN_F8DATA1_FD12_Msk                    (0x1U << CAN_F8DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F8DATA1_FD12                        CAN_F8DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F8DATA1_FD13_Pos                    (13U)                             
#define CAN_F8DATA1_FD13_Msk                    (0x1U << CAN_F8DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F8DATA1_FD13                        CAN_F8DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F8DATA1_FD14_Pos                    (14U)                             
#define CAN_F8DATA1_FD14_Msk                    (0x1U << CAN_F8DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F8DATA1_FD14                        CAN_F8DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F8DATA1_FD15_Pos                    (15U)                             
#define CAN_F8DATA1_FD15_Msk                    (0x1U << CAN_F8DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F8DATA1_FD15                        CAN_F8DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F8DATA1_FD16_Pos                    (16U)                             
#define CAN_F8DATA1_FD16_Msk                    (0x1U << CAN_F8DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F8DATA1_FD16                        CAN_F8DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F8DATA1_FD17_Pos                    (17U)                             
#define CAN_F8DATA1_FD17_Msk                    (0x1U << CAN_F8DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F8DATA1_FD17                        CAN_F8DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F8DATA1_FD18_Pos                    (18U)                             
#define CAN_F8DATA1_FD18_Msk                    (0x1U << CAN_F8DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F8DATA1_FD18                        CAN_F8DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F8DATA1_FD19_Pos                    (19U)                             
#define CAN_F8DATA1_FD19_Msk                    (0x1U << CAN_F8DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F8DATA1_FD19                        CAN_F8DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F8DATA1_FD20_Pos                    (20U)                             
#define CAN_F8DATA1_FD20_Msk                    (0x1U << CAN_F8DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F8DATA1_FD20                        CAN_F8DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F8DATA1_FD21_Pos                    (21U)                             
#define CAN_F8DATA1_FD21_Msk                    (0x1U << CAN_F8DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F8DATA1_FD21                        CAN_F8DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F8DATA1_FD22_Pos                    (22U)                             
#define CAN_F8DATA1_FD22_Msk                    (0x1U << CAN_F8DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F8DATA1_FD22                        CAN_F8DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F8DATA1_FD23_Pos                    (23U)                             
#define CAN_F8DATA1_FD23_Msk                    (0x1U << CAN_F8DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F8DATA1_FD23                        CAN_F8DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F8DATA1_FD24_Pos                    (24U)                             
#define CAN_F8DATA1_FD24_Msk                    (0x1U << CAN_F8DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F8DATA1_FD24                        CAN_F8DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F8DATA1_FD25_Pos                    (25U)                             
#define CAN_F8DATA1_FD25_Msk                    (0x1U << CAN_F8DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F8DATA1_FD25                        CAN_F8DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F8DATA1_FD26_Pos                    (26U)                             
#define CAN_F8DATA1_FD26_Msk                    (0x1U << CAN_F8DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F8DATA1_FD26                        CAN_F8DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F8DATA1_FD27_Pos                    (27U)                             
#define CAN_F8DATA1_FD27_Msk                    (0x1U << CAN_F8DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F8DATA1_FD27                        CAN_F8DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F8DATA1_FD28_Pos                    (28U)                             
#define CAN_F8DATA1_FD28_Msk                    (0x1U << CAN_F8DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F8DATA1_FD28                        CAN_F8DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F8DATA1_FD29_Pos                    (29U)                             
#define CAN_F8DATA1_FD29_Msk                    (0x1U << CAN_F8DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F8DATA1_FD29                        CAN_F8DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F8DATA1_FD30_Pos                    (30U)                             
#define CAN_F8DATA1_FD30_Msk                    (0x1U << CAN_F8DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F8DATA1_FD30                        CAN_F8DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F8DATA1_FD31_Pos                    (31U)                             
#define CAN_F8DATA1_FD31_Msk                    (0x1U << CAN_F8DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F8DATA1_FD31                        CAN_F8DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F9R2 register  *******************/
#define CAN_F9DATA1_FD0_Pos                     (0U)                              
#define CAN_F9DATA1_FD0_Msk                     (0x1U << CAN_F9DATA1_FD0_Pos)        /*!< 0x00000001 */
#define CAN_F9DATA1_FD0                         CAN_F9DATA1_FD0_Msk                  /*!< Filter bit 0 */
#define CAN_F9DATA1_FD1_Pos                     (1U)                              
#define CAN_F9DATA1_FD1_Msk                     (0x1U << CAN_F9DATA1_FD1_Pos)        /*!< 0x00000002 */
#define CAN_F9DATA1_FD1                         CAN_F9DATA1_FD1_Msk                  /*!< Filter bit 1 */
#define CAN_F9DATA1_FD2_Pos                     (2U)                              
#define CAN_F9DATA1_FD2_Msk                     (0x1U << CAN_F9DATA1_FD2_Pos)        /*!< 0x00000004 */
#define CAN_F9DATA1_FD2                         CAN_F9DATA1_FD2_Msk                  /*!< Filter bit 2 */
#define CAN_F9DATA1_FD3_Pos                     (3U)                              
#define CAN_F9DATA1_FD3_Msk                     (0x1U << CAN_F9DATA1_FD3_Pos)        /*!< 0x00000008 */
#define CAN_F9DATA1_FD3                         CAN_F9DATA1_FD3_Msk                  /*!< Filter bit 3 */
#define CAN_F9DATA1_FD4_Pos                     (4U)                              
#define CAN_F9DATA1_FD4_Msk                     (0x1U << CAN_F9DATA1_FD4_Pos)        /*!< 0x00000010 */
#define CAN_F9DATA1_FD4                         CAN_F9DATA1_FD4_Msk                  /*!< Filter bit 4 */
#define CAN_F9DATA1_FD5_Pos                     (5U)                              
#define CAN_F9DATA1_FD5_Msk                     (0x1U << CAN_F9DATA1_FD5_Pos)        /*!< 0x00000020 */
#define CAN_F9DATA1_FD5                         CAN_F9DATA1_FD5_Msk                  /*!< Filter bit 5 */
#define CAN_F9DATA1_FD6_Pos                     (6U)                              
#define CAN_F9DATA1_FD6_Msk                     (0x1U << CAN_F9DATA1_FD6_Pos)        /*!< 0x00000040 */
#define CAN_F9DATA1_FD6                         CAN_F9DATA1_FD6_Msk                  /*!< Filter bit 6 */
#define CAN_F9DATA1_FD7_Pos                     (7U)                              
#define CAN_F9DATA1_FD7_Msk                     (0x1U << CAN_F9DATA1_FD7_Pos)        /*!< 0x00000080 */
#define CAN_F9DATA1_FD7                         CAN_F9DATA1_FD7_Msk                  /*!< Filter bit 7 */
#define CAN_F9DATA1_FD8_Pos                     (8U)                              
#define CAN_F9DATA1_FD8_Msk                     (0x1U << CAN_F9DATA1_FD8_Pos)        /*!< 0x00000100 */
#define CAN_F9DATA1_FD8                         CAN_F9DATA1_FD8_Msk                  /*!< Filter bit 8 */
#define CAN_F9DATA1_FD9_Pos                     (9U)                              
#define CAN_F9DATA1_FD9_Msk                     (0x1U << CAN_F9DATA1_FD9_Pos)        /*!< 0x00000200 */
#define CAN_F9DATA1_FD9                         CAN_F9DATA1_FD9_Msk                  /*!< Filter bit 9 */
#define CAN_F9DATA1_FD10_Pos                    (10U)                             
#define CAN_F9DATA1_FD10_Msk                    (0x1U << CAN_F9DATA1_FD10_Pos)       /*!< 0x00000400 */
#define CAN_F9DATA1_FD10                        CAN_F9DATA1_FD10_Msk                 /*!< Filter bit 10 */
#define CAN_F9DATA1_FD11_Pos                    (11U)                             
#define CAN_F9DATA1_FD11_Msk                    (0x1U << CAN_F9DATA1_FD11_Pos)       /*!< 0x00000800 */
#define CAN_F9DATA1_FD11                        CAN_F9DATA1_FD11_Msk                 /*!< Filter bit 11 */
#define CAN_F9DATA1_FD12_Pos                    (12U)                             
#define CAN_F9DATA1_FD12_Msk                    (0x1U << CAN_F9DATA1_FD12_Pos)       /*!< 0x00001000 */
#define CAN_F9DATA1_FD12                        CAN_F9DATA1_FD12_Msk                 /*!< Filter bit 12 */
#define CAN_F9DATA1_FD13_Pos                    (13U)                             
#define CAN_F9DATA1_FD13_Msk                    (0x1U << CAN_F9DATA1_FD13_Pos)       /*!< 0x00002000 */
#define CAN_F9DATA1_FD13                        CAN_F9DATA1_FD13_Msk                 /*!< Filter bit 13 */
#define CAN_F9DATA1_FD14_Pos                    (14U)                             
#define CAN_F9DATA1_FD14_Msk                    (0x1U << CAN_F9DATA1_FD14_Pos)       /*!< 0x00004000 */
#define CAN_F9DATA1_FD14                        CAN_F9DATA1_FD14_Msk                 /*!< Filter bit 14 */
#define CAN_F9DATA1_FD15_Pos                    (15U)                             
#define CAN_F9DATA1_FD15_Msk                    (0x1U << CAN_F9DATA1_FD15_Pos)       /*!< 0x00008000 */
#define CAN_F9DATA1_FD15                        CAN_F9DATA1_FD15_Msk                 /*!< Filter bit 15 */
#define CAN_F9DATA1_FD16_Pos                    (16U)                             
#define CAN_F9DATA1_FD16_Msk                    (0x1U << CAN_F9DATA1_FD16_Pos)       /*!< 0x00010000 */
#define CAN_F9DATA1_FD16                        CAN_F9DATA1_FD16_Msk                 /*!< Filter bit 16 */
#define CAN_F9DATA1_FD17_Pos                    (17U)                             
#define CAN_F9DATA1_FD17_Msk                    (0x1U << CAN_F9DATA1_FD17_Pos)       /*!< 0x00020000 */
#define CAN_F9DATA1_FD17                        CAN_F9DATA1_FD17_Msk                 /*!< Filter bit 17 */
#define CAN_F9DATA1_FD18_Pos                    (18U)                             
#define CAN_F9DATA1_FD18_Msk                    (0x1U << CAN_F9DATA1_FD18_Pos)       /*!< 0x00040000 */
#define CAN_F9DATA1_FD18                        CAN_F9DATA1_FD18_Msk                 /*!< Filter bit 18 */
#define CAN_F9DATA1_FD19_Pos                    (19U)                             
#define CAN_F9DATA1_FD19_Msk                    (0x1U << CAN_F9DATA1_FD19_Pos)       /*!< 0x00080000 */
#define CAN_F9DATA1_FD19                        CAN_F9DATA1_FD19_Msk                 /*!< Filter bit 19 */
#define CAN_F9DATA1_FD20_Pos                    (20U)                             
#define CAN_F9DATA1_FD20_Msk                    (0x1U << CAN_F9DATA1_FD20_Pos)       /*!< 0x00100000 */
#define CAN_F9DATA1_FD20                        CAN_F9DATA1_FD20_Msk                 /*!< Filter bit 20 */
#define CAN_F9DATA1_FD21_Pos                    (21U)                             
#define CAN_F9DATA1_FD21_Msk                    (0x1U << CAN_F9DATA1_FD21_Pos)       /*!< 0x00200000 */
#define CAN_F9DATA1_FD21                        CAN_F9DATA1_FD21_Msk                 /*!< Filter bit 21 */
#define CAN_F9DATA1_FD22_Pos                    (22U)                             
#define CAN_F9DATA1_FD22_Msk                    (0x1U << CAN_F9DATA1_FD22_Pos)       /*!< 0x00400000 */
#define CAN_F9DATA1_FD22                        CAN_F9DATA1_FD22_Msk                 /*!< Filter bit 22 */
#define CAN_F9DATA1_FD23_Pos                    (23U)                             
#define CAN_F9DATA1_FD23_Msk                    (0x1U << CAN_F9DATA1_FD23_Pos)       /*!< 0x00800000 */
#define CAN_F9DATA1_FD23                        CAN_F9DATA1_FD23_Msk                 /*!< Filter bit 23 */
#define CAN_F9DATA1_FD24_Pos                    (24U)                             
#define CAN_F9DATA1_FD24_Msk                    (0x1U << CAN_F9DATA1_FD24_Pos)       /*!< 0x01000000 */
#define CAN_F9DATA1_FD24                        CAN_F9DATA1_FD24_Msk                 /*!< Filter bit 24 */
#define CAN_F9DATA1_FD25_Pos                    (25U)                             
#define CAN_F9DATA1_FD25_Msk                    (0x1U << CAN_F9DATA1_FD25_Pos)       /*!< 0x02000000 */
#define CAN_F9DATA1_FD25                        CAN_F9DATA1_FD25_Msk                 /*!< Filter bit 25 */
#define CAN_F9DATA1_FD26_Pos                    (26U)                             
#define CAN_F9DATA1_FD26_Msk                    (0x1U << CAN_F9DATA1_FD26_Pos)       /*!< 0x04000000 */
#define CAN_F9DATA1_FD26                        CAN_F9DATA1_FD26_Msk                 /*!< Filter bit 26 */
#define CAN_F9DATA1_FD27_Pos                    (27U)                             
#define CAN_F9DATA1_FD27_Msk                    (0x1U << CAN_F9DATA1_FD27_Pos)       /*!< 0x08000000 */
#define CAN_F9DATA1_FD27                        CAN_F9DATA1_FD27_Msk                 /*!< Filter bit 27 */
#define CAN_F9DATA1_FD28_Pos                    (28U)                             
#define CAN_F9DATA1_FD28_Msk                    (0x1U << CAN_F9DATA1_FD28_Pos)       /*!< 0x10000000 */
#define CAN_F9DATA1_FD28                        CAN_F9DATA1_FD28_Msk                 /*!< Filter bit 28 */
#define CAN_F9DATA1_FD29_Pos                    (29U)                             
#define CAN_F9DATA1_FD29_Msk                    (0x1U << CAN_F9DATA1_FD29_Pos)       /*!< 0x20000000 */
#define CAN_F9DATA1_FD29                        CAN_F9DATA1_FD29_Msk                 /*!< Filter bit 29 */
#define CAN_F9DATA1_FD30_Pos                    (30U)                             
#define CAN_F9DATA1_FD30_Msk                    (0x1U << CAN_F9DATA1_FD30_Pos)       /*!< 0x40000000 */
#define CAN_F9DATA1_FD30                        CAN_F9DATA1_FD30_Msk                 /*!< Filter bit 30 */
#define CAN_F9DATA1_FD31_Pos                    (31U)                             
#define CAN_F9DATA1_FD31_Msk                    (0x1U << CAN_F9DATA1_FD31_Pos)       /*!< 0x80000000 */
#define CAN_F9DATA1_FD31                        CAN_F9DATA1_FD31_Msk                 /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F10R2 register  ******************/
#define CAN_F10DATA1_FD0_Pos                    (0U)                              
#define CAN_F10DATA1_FD0_Msk                    (0x1U << CAN_F10DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F10DATA1_FD0                        CAN_F10DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F10DATA1_FD1_Pos                    (1U)                              
#define CAN_F10DATA1_FD1_Msk                    (0x1U << CAN_F10DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F10DATA1_FD1                        CAN_F10DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F10DATA1_FD2_Pos                    (2U)                              
#define CAN_F10DATA1_FD2_Msk                    (0x1U << CAN_F10DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F10DATA1_FD2                        CAN_F10DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F10DATA1_FD3_Pos                    (3U)                              
#define CAN_F10DATA1_FD3_Msk                    (0x1U << CAN_F10DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F10DATA1_FD3                        CAN_F10DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F10DATA1_FD4_Pos                    (4U)                              
#define CAN_F10DATA1_FD4_Msk                    (0x1U << CAN_F10DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F10DATA1_FD4                        CAN_F10DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F10DATA1_FD5_Pos                    (5U)                              
#define CAN_F10DATA1_FD5_Msk                    (0x1U << CAN_F10DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F10DATA1_FD5                        CAN_F10DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F10DATA1_FD6_Pos                    (6U)                              
#define CAN_F10DATA1_FD6_Msk                    (0x1U << CAN_F10DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F10DATA1_FD6                        CAN_F10DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F10DATA1_FD7_Pos                    (7U)                              
#define CAN_F10DATA1_FD7_Msk                    (0x1U << CAN_F10DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F10DATA1_FD7                        CAN_F10DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F10DATA1_FD8_Pos                    (8U)                              
#define CAN_F10DATA1_FD8_Msk                    (0x1U << CAN_F10DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F10DATA1_FD8                        CAN_F10DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F10DATA1_FD9_Pos                    (9U)                              
#define CAN_F10DATA1_FD9_Msk                    (0x1U << CAN_F10DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F10DATA1_FD9                        CAN_F10DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F10DATA1_FD10_Pos                   (10U)                             
#define CAN_F10DATA1_FD10_Msk                   (0x1U << CAN_F10DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F10DATA1_FD10                       CAN_F10DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F10DATA1_FD11_Pos                   (11U)                             
#define CAN_F10DATA1_FD11_Msk                   (0x1U << CAN_F10DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F10DATA1_FD11                       CAN_F10DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F10DATA1_FD12_Pos                   (12U)                             
#define CAN_F10DATA1_FD12_Msk                   (0x1U << CAN_F10DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F10DATA1_FD12                       CAN_F10DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F10DATA1_FD13_Pos                   (13U)                             
#define CAN_F10DATA1_FD13_Msk                   (0x1U << CAN_F10DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F10DATA1_FD13                       CAN_F10DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F10DATA1_FD14_Pos                   (14U)                             
#define CAN_F10DATA1_FD14_Msk                   (0x1U << CAN_F10DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F10DATA1_FD14                       CAN_F10DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F10DATA1_FD15_Pos                   (15U)                             
#define CAN_F10DATA1_FD15_Msk                   (0x1U << CAN_F10DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F10DATA1_FD15                       CAN_F10DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F10DATA1_FD16_Pos                   (16U)                             
#define CAN_F10DATA1_FD16_Msk                   (0x1U << CAN_F10DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F10DATA1_FD16                       CAN_F10DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F10DATA1_FD17_Pos                   (17U)                             
#define CAN_F10DATA1_FD17_Msk                   (0x1U << CAN_F10DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F10DATA1_FD17                       CAN_F10DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F10DATA1_FD18_Pos                   (18U)                             
#define CAN_F10DATA1_FD18_Msk                   (0x1U << CAN_F10DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F10DATA1_FD18                       CAN_F10DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F10DATA1_FD19_Pos                   (19U)                             
#define CAN_F10DATA1_FD19_Msk                   (0x1U << CAN_F10DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F10DATA1_FD19                       CAN_F10DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F10DATA1_FD20_Pos                   (20U)                             
#define CAN_F10DATA1_FD20_Msk                   (0x1U << CAN_F10DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F10DATA1_FD20                       CAN_F10DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F10DATA1_FD21_Pos                   (21U)                             
#define CAN_F10DATA1_FD21_Msk                   (0x1U << CAN_F10DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F10DATA1_FD21                       CAN_F10DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F10DATA1_FD22_Pos                   (22U)                             
#define CAN_F10DATA1_FD22_Msk                   (0x1U << CAN_F10DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F10DATA1_FD22                       CAN_F10DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F10DATA1_FD23_Pos                   (23U)                             
#define CAN_F10DATA1_FD23_Msk                   (0x1U << CAN_F10DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F10DATA1_FD23                       CAN_F10DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F10DATA1_FD24_Pos                   (24U)                             
#define CAN_F10DATA1_FD24_Msk                   (0x1U << CAN_F10DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F10DATA1_FD24                       CAN_F10DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F10DATA1_FD25_Pos                   (25U)                             
#define CAN_F10DATA1_FD25_Msk                   (0x1U << CAN_F10DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F10DATA1_FD25                       CAN_F10DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F10DATA1_FD26_Pos                   (26U)                             
#define CAN_F10DATA1_FD26_Msk                   (0x1U << CAN_F10DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F10DATA1_FD26                       CAN_F10DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F10DATA1_FD27_Pos                   (27U)                             
#define CAN_F10DATA1_FD27_Msk                   (0x1U << CAN_F10DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F10DATA1_FD27                       CAN_F10DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F10DATA1_FD28_Pos                   (28U)                             
#define CAN_F10DATA1_FD28_Msk                   (0x1U << CAN_F10DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F10DATA1_FD28                       CAN_F10DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F10DATA1_FD29_Pos                   (29U)                             
#define CAN_F10DATA1_FD29_Msk                   (0x1U << CAN_F10DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F10DATA1_FD29                       CAN_F10DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F10DATA1_FD30_Pos                   (30U)                             
#define CAN_F10DATA1_FD30_Msk                   (0x1U << CAN_F10DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F10DATA1_FD30                       CAN_F10DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F10DATA1_FD31_Pos                   (31U)                             
#define CAN_F10DATA1_FD31_Msk                   (0x1U << CAN_F10DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F10DATA1_FD31                       CAN_F10DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F11R2 register  ******************/
#define CAN_F11DATA1_FD0_Pos                    (0U)                              
#define CAN_F11DATA1_FD0_Msk                    (0x1U << CAN_F11DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F11DATA1_FD0                        CAN_F11DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F11DATA1_FD1_Pos                    (1U)                              
#define CAN_F11DATA1_FD1_Msk                    (0x1U << CAN_F11DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F11DATA1_FD1                        CAN_F11DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F11DATA1_FD2_Pos                    (2U)                              
#define CAN_F11DATA1_FD2_Msk                    (0x1U << CAN_F11DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F11DATA1_FD2                        CAN_F11DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F11DATA1_FD3_Pos                    (3U)                              
#define CAN_F11DATA1_FD3_Msk                    (0x1U << CAN_F11DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F11DATA1_FD3                        CAN_F11DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F11DATA1_FD4_Pos                    (4U)                              
#define CAN_F11DATA1_FD4_Msk                    (0x1U << CAN_F11DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F11DATA1_FD4                        CAN_F11DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F11DATA1_FD5_Pos                    (5U)                              
#define CAN_F11DATA1_FD5_Msk                    (0x1U << CAN_F11DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F11DATA1_FD5                        CAN_F11DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F11DATA1_FD6_Pos                    (6U)                              
#define CAN_F11DATA1_FD6_Msk                    (0x1U << CAN_F11DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F11DATA1_FD6                        CAN_F11DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F11DATA1_FD7_Pos                    (7U)                              
#define CAN_F11DATA1_FD7_Msk                    (0x1U << CAN_F11DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F11DATA1_FD7                        CAN_F11DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F11DATA1_FD8_Pos                    (8U)                              
#define CAN_F11DATA1_FD8_Msk                    (0x1U << CAN_F11DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F11DATA1_FD8                        CAN_F11DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F11DATA1_FD9_Pos                    (9U)                              
#define CAN_F11DATA1_FD9_Msk                    (0x1U << CAN_F11DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F11DATA1_FD9                        CAN_F11DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F11DATA1_FD10_Pos                   (10U)                             
#define CAN_F11DATA1_FD10_Msk                   (0x1U << CAN_F11DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F11DATA1_FD10                       CAN_F11DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F11DATA1_FD11_Pos                   (11U)                             
#define CAN_F11DATA1_FD11_Msk                   (0x1U << CAN_F11DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F11DATA1_FD11                       CAN_F11DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F11DATA1_FD12_Pos                   (12U)                             
#define CAN_F11DATA1_FD12_Msk                   (0x1U << CAN_F11DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F11DATA1_FD12                       CAN_F11DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F11DATA1_FD13_Pos                   (13U)                             
#define CAN_F11DATA1_FD13_Msk                   (0x1U << CAN_F11DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F11DATA1_FD13                       CAN_F11DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F11DATA1_FD14_Pos                   (14U)                             
#define CAN_F11DATA1_FD14_Msk                   (0x1U << CAN_F11DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F11DATA1_FD14                       CAN_F11DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F11DATA1_FD15_Pos                   (15U)                             
#define CAN_F11DATA1_FD15_Msk                   (0x1U << CAN_F11DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F11DATA1_FD15                       CAN_F11DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F11DATA1_FD16_Pos                   (16U)                             
#define CAN_F11DATA1_FD16_Msk                   (0x1U << CAN_F11DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F11DATA1_FD16                       CAN_F11DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F11DATA1_FD17_Pos                   (17U)                             
#define CAN_F11DATA1_FD17_Msk                   (0x1U << CAN_F11DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F11DATA1_FD17                       CAN_F11DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F11DATA1_FD18_Pos                   (18U)                             
#define CAN_F11DATA1_FD18_Msk                   (0x1U << CAN_F11DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F11DATA1_FD18                       CAN_F11DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F11DATA1_FD19_Pos                   (19U)                             
#define CAN_F11DATA1_FD19_Msk                   (0x1U << CAN_F11DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F11DATA1_FD19                       CAN_F11DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F11DATA1_FD20_Pos                   (20U)                             
#define CAN_F11DATA1_FD20_Msk                   (0x1U << CAN_F11DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F11DATA1_FD20                       CAN_F11DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F11DATA1_FD21_Pos                   (21U)                             
#define CAN_F11DATA1_FD21_Msk                   (0x1U << CAN_F11DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F11DATA1_FD21                       CAN_F11DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F11DATA1_FD22_Pos                   (22U)                             
#define CAN_F11DATA1_FD22_Msk                   (0x1U << CAN_F11DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F11DATA1_FD22                       CAN_F11DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F11DATA1_FD23_Pos                   (23U)                             
#define CAN_F11DATA1_FD23_Msk                   (0x1U << CAN_F11DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F11DATA1_FD23                       CAN_F11DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F11DATA1_FD24_Pos                   (24U)                             
#define CAN_F11DATA1_FD24_Msk                   (0x1U << CAN_F11DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F11DATA1_FD24                       CAN_F11DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F11DATA1_FD25_Pos                   (25U)                             
#define CAN_F11DATA1_FD25_Msk                   (0x1U << CAN_F11DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F11DATA1_FD25                       CAN_F11DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F11DATA1_FD26_Pos                   (26U)                             
#define CAN_F11DATA1_FD26_Msk                   (0x1U << CAN_F11DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F11DATA1_FD26                       CAN_F11DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F11DATA1_FD27_Pos                   (27U)                             
#define CAN_F11DATA1_FD27_Msk                   (0x1U << CAN_F11DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F11DATA1_FD27                       CAN_F11DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F11DATA1_FD28_Pos                   (28U)                             
#define CAN_F11DATA1_FD28_Msk                   (0x1U << CAN_F11DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F11DATA1_FD28                       CAN_F11DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F11DATA1_FD29_Pos                   (29U)                             
#define CAN_F11DATA1_FD29_Msk                   (0x1U << CAN_F11DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F11DATA1_FD29                       CAN_F11DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F11DATA1_FD30_Pos                   (30U)                             
#define CAN_F11DATA1_FD30_Msk                   (0x1U << CAN_F11DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F11DATA1_FD30                       CAN_F11DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F11DATA1_FD31_Pos                   (31U)                             
#define CAN_F11DATA1_FD31_Msk                   (0x1U << CAN_F11DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F11DATA1_FD31                       CAN_F11DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F12R2 register  ******************/
#define CAN_F12DATA1_FD0_Pos                    (0U)                              
#define CAN_F12DATA1_FD0_Msk                    (0x1U << CAN_F12DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F12DATA1_FD0                        CAN_F12DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F12DATA1_FD1_Pos                    (1U)                              
#define CAN_F12DATA1_FD1_Msk                    (0x1U << CAN_F12DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F12DATA1_FD1                        CAN_F12DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F12DATA1_FD2_Pos                    (2U)                              
#define CAN_F12DATA1_FD2_Msk                    (0x1U << CAN_F12DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F12DATA1_FD2                        CAN_F12DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F12DATA1_FD3_Pos                    (3U)                              
#define CAN_F12DATA1_FD3_Msk                    (0x1U << CAN_F12DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F12DATA1_FD3                        CAN_F12DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F12DATA1_FD4_Pos                    (4U)                              
#define CAN_F12DATA1_FD4_Msk                    (0x1U << CAN_F12DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F12DATA1_FD4                        CAN_F12DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F12DATA1_FD5_Pos                    (5U)                              
#define CAN_F12DATA1_FD5_Msk                    (0x1U << CAN_F12DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F12DATA1_FD5                        CAN_F12DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F12DATA1_FD6_Pos                    (6U)                              
#define CAN_F12DATA1_FD6_Msk                    (0x1U << CAN_F12DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F12DATA1_FD6                        CAN_F12DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F12DATA1_FD7_Pos                    (7U)                              
#define CAN_F12DATA1_FD7_Msk                    (0x1U << CAN_F12DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F12DATA1_FD7                        CAN_F12DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F12DATA1_FD8_Pos                    (8U)                              
#define CAN_F12DATA1_FD8_Msk                    (0x1U << CAN_F12DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F12DATA1_FD8                        CAN_F12DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F12DATA1_FD9_Pos                    (9U)                              
#define CAN_F12DATA1_FD9_Msk                    (0x1U << CAN_F12DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F12DATA1_FD9                        CAN_F12DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F12DATA1_FD10_Pos                   (10U)                             
#define CAN_F12DATA1_FD10_Msk                   (0x1U << CAN_F12DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F12DATA1_FD10                       CAN_F12DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F12DATA1_FD11_Pos                   (11U)                             
#define CAN_F12DATA1_FD11_Msk                   (0x1U << CAN_F12DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F12DATA1_FD11                       CAN_F12DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F12DATA1_FD12_Pos                   (12U)                             
#define CAN_F12DATA1_FD12_Msk                   (0x1U << CAN_F12DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F12DATA1_FD12                       CAN_F12DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F12DATA1_FD13_Pos                   (13U)                             
#define CAN_F12DATA1_FD13_Msk                   (0x1U << CAN_F12DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F12DATA1_FD13                       CAN_F12DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F12DATA1_FD14_Pos                   (14U)                             
#define CAN_F12DATA1_FD14_Msk                   (0x1U << CAN_F12DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F12DATA1_FD14                       CAN_F12DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F12DATA1_FD15_Pos                   (15U)                             
#define CAN_F12DATA1_FD15_Msk                   (0x1U << CAN_F12DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F12DATA1_FD15                       CAN_F12DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F12DATA1_FD16_Pos                   (16U)                             
#define CAN_F12DATA1_FD16_Msk                   (0x1U << CAN_F12DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F12DATA1_FD16                       CAN_F12DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F12DATA1_FD17_Pos                   (17U)                             
#define CAN_F12DATA1_FD17_Msk                   (0x1U << CAN_F12DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F12DATA1_FD17                       CAN_F12DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F12DATA1_FD18_Pos                   (18U)                             
#define CAN_F12DATA1_FD18_Msk                   (0x1U << CAN_F12DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F12DATA1_FD18                       CAN_F12DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F12DATA1_FD19_Pos                   (19U)                             
#define CAN_F12DATA1_FD19_Msk                   (0x1U << CAN_F12DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F12DATA1_FD19                       CAN_F12DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F12DATA1_FD20_Pos                   (20U)                             
#define CAN_F12DATA1_FD20_Msk                   (0x1U << CAN_F12DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F12DATA1_FD20                       CAN_F12DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F12DATA1_FD21_Pos                   (21U)                             
#define CAN_F12DATA1_FD21_Msk                   (0x1U << CAN_F12DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F12DATA1_FD21                       CAN_F12DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F12DATA1_FD22_Pos                   (22U)                             
#define CAN_F12DATA1_FD22_Msk                   (0x1U << CAN_F12DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F12DATA1_FD22                       CAN_F12DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F12DATA1_FD23_Pos                   (23U)                             
#define CAN_F12DATA1_FD23_Msk                   (0x1U << CAN_F12DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F12DATA1_FD23                       CAN_F12DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F12DATA1_FD24_Pos                   (24U)                             
#define CAN_F12DATA1_FD24_Msk                   (0x1U << CAN_F12DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F12DATA1_FD24                       CAN_F12DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F12DATA1_FD25_Pos                   (25U)                             
#define CAN_F12DATA1_FD25_Msk                   (0x1U << CAN_F12DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F12DATA1_FD25                       CAN_F12DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F12DATA1_FD26_Pos                   (26U)                             
#define CAN_F12DATA1_FD26_Msk                   (0x1U << CAN_F12DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F12DATA1_FD26                       CAN_F12DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F12DATA1_FD27_Pos                   (27U)                             
#define CAN_F12DATA1_FD27_Msk                   (0x1U << CAN_F12DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F12DATA1_FD27                       CAN_F12DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F12DATA1_FD28_Pos                   (28U)                             
#define CAN_F12DATA1_FD28_Msk                   (0x1U << CAN_F12DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F12DATA1_FD28                       CAN_F12DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F12DATA1_FD29_Pos                   (29U)                             
#define CAN_F12DATA1_FD29_Msk                   (0x1U << CAN_F12DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F12DATA1_FD29                       CAN_F12DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F12DATA1_FD30_Pos                   (30U)                             
#define CAN_F12DATA1_FD30_Msk                   (0x1U << CAN_F12DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F12DATA1_FD30                       CAN_F12DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F12DATA1_FD31_Pos                   (31U)                             
#define CAN_F12DATA1_FD31_Msk                   (0x1U << CAN_F12DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F12DATA1_FD31                       CAN_F12DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F13R2 register  ******************/
#define CAN_F13DATA1_FD0_Pos                    (0U)                              
#define CAN_F13DATA1_FD0_Msk                    (0x1U << CAN_F13DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F13DATA1_FD0                        CAN_F13DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F13DATA1_FD1_Pos                    (1U)                              
#define CAN_F13DATA1_FD1_Msk                    (0x1U << CAN_F13DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F13DATA1_FD1                        CAN_F13DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F13DATA1_FD2_Pos                    (2U)                              
#define CAN_F13DATA1_FD2_Msk                    (0x1U << CAN_F13DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F13DATA1_FD2                        CAN_F13DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F13DATA1_FD3_Pos                    (3U)                              
#define CAN_F13DATA1_FD3_Msk                    (0x1U << CAN_F13DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F13DATA1_FD3                        CAN_F13DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F13DATA1_FD4_Pos                    (4U)                              
#define CAN_F13DATA1_FD4_Msk                    (0x1U << CAN_F13DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F13DATA1_FD4                        CAN_F13DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F13DATA1_FD5_Pos                    (5U)                              
#define CAN_F13DATA1_FD5_Msk                    (0x1U << CAN_F13DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F13DATA1_FD5                        CAN_F13DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F13DATA1_FD6_Pos                    (6U)                              
#define CAN_F13DATA1_FD6_Msk                    (0x1U << CAN_F13DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F13DATA1_FD6                        CAN_F13DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F13DATA1_FD7_Pos                    (7U)                              
#define CAN_F13DATA1_FD7_Msk                    (0x1U << CAN_F13DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F13DATA1_FD7                        CAN_F13DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F13DATA1_FD8_Pos                    (8U)                              
#define CAN_F13DATA1_FD8_Msk                    (0x1U << CAN_F13DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F13DATA1_FD8                        CAN_F13DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F13DATA1_FD9_Pos                    (9U)                              
#define CAN_F13DATA1_FD9_Msk                    (0x1U << CAN_F13DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F13DATA1_FD9                        CAN_F13DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F13DATA1_FD10_Pos                   (10U)                             
#define CAN_F13DATA1_FD10_Msk                   (0x1U << CAN_F13DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F13DATA1_FD10                       CAN_F13DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F13DATA1_FD11_Pos                   (11U)                             
#define CAN_F13DATA1_FD11_Msk                   (0x1U << CAN_F13DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F13DATA1_FD11                       CAN_F13DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F13DATA1_FD12_Pos                   (12U)                             
#define CAN_F13DATA1_FD12_Msk                   (0x1U << CAN_F13DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F13DATA1_FD12                       CAN_F13DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F13DATA1_FD13_Pos                   (13U)                             
#define CAN_F13DATA1_FD13_Msk                   (0x1U << CAN_F13DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F13DATA1_FD13                       CAN_F13DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F13DATA1_FD14_Pos                   (14U)                             
#define CAN_F13DATA1_FD14_Msk                   (0x1U << CAN_F13DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F13DATA1_FD14                       CAN_F13DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F13DATA1_FD15_Pos                   (15U)                             
#define CAN_F13DATA1_FD15_Msk                   (0x1U << CAN_F13DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F13DATA1_FD15                       CAN_F13DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F13DATA1_FD16_Pos                   (16U)                             
#define CAN_F13DATA1_FD16_Msk                   (0x1U << CAN_F13DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F13DATA1_FD16                       CAN_F13DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F13DATA1_FD17_Pos                   (17U)                             
#define CAN_F13DATA1_FD17_Msk                   (0x1U << CAN_F13DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F13DATA1_FD17                       CAN_F13DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F13DATA1_FD18_Pos                   (18U)                             
#define CAN_F13DATA1_FD18_Msk                   (0x1U << CAN_F13DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F13DATA1_FD18                       CAN_F13DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F13DATA1_FD19_Pos                   (19U)                             
#define CAN_F13DATA1_FD19_Msk                   (0x1U << CAN_F13DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F13DATA1_FD19                       CAN_F13DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F13DATA1_FD20_Pos                   (20U)                             
#define CAN_F13DATA1_FD20_Msk                   (0x1U << CAN_F13DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F13DATA1_FD20                       CAN_F13DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F13DATA1_FD21_Pos                   (21U)                             
#define CAN_F13DATA1_FD21_Msk                   (0x1U << CAN_F13DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F13DATA1_FD21                       CAN_F13DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F13DATA1_FD22_Pos                   (22U)                             
#define CAN_F13DATA1_FD22_Msk                   (0x1U << CAN_F13DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F13DATA1_FD22                       CAN_F13DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F13DATA1_FD23_Pos                   (23U)                             
#define CAN_F13DATA1_FD23_Msk                   (0x1U << CAN_F13DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F13DATA1_FD23                       CAN_F13DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F13DATA1_FD24_Pos                   (24U)                             
#define CAN_F13DATA1_FD24_Msk                   (0x1U << CAN_F13DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F13DATA1_FD24                       CAN_F13DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F13DATA1_FD25_Pos                   (25U)                             
#define CAN_F13DATA1_FD25_Msk                   (0x1U << CAN_F13DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F13DATA1_FD25                       CAN_F13DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F13DATA1_FD26_Pos                   (26U)                             
#define CAN_F13DATA1_FD26_Msk                   (0x1U << CAN_F13DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F13DATA1_FD26                       CAN_F13DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F13DATA1_FD27_Pos                   (27U)                             
#define CAN_F13DATA1_FD27_Msk                   (0x1U << CAN_F13DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F13DATA1_FD27                       CAN_F13DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F13DATA1_FD28_Pos                   (28U)                             
#define CAN_F13DATA1_FD28_Msk                   (0x1U << CAN_F13DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F13DATA1_FD28                       CAN_F13DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F13DATA1_FD29_Pos                   (29U)                             
#define CAN_F13DATA1_FD29_Msk                   (0x1U << CAN_F13DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F13DATA1_FD29                       CAN_F13DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F13DATA1_FD30_Pos                   (30U)                             
#define CAN_F13DATA1_FD30_Msk                   (0x1U << CAN_F13DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F13DATA1_FD30                       CAN_F13DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F13DATA1_FD31_Pos                   (31U)                             
#define CAN_F13DATA1_FD31_Msk                   (0x1U << CAN_F13DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F13DATA1_FD31                       CAN_F13DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F14R2 register  ******************/
#define CAN_F14DATA1_FD0_Pos                    (0U)                              
#define CAN_F14DATA1_FD0_Msk                    (0x1U << CAN_F14DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F14DATA1_FD0                        CAN_F14DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F14DATA1_FD1_Pos                    (1U)                              
#define CAN_F14DATA1_FD1_Msk                    (0x1U << CAN_F14DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F14DATA1_FD1                        CAN_F14DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F14DATA1_FD2_Pos                    (2U)                              
#define CAN_F14DATA1_FD2_Msk                    (0x1U << CAN_F14DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F14DATA1_FD2                        CAN_F14DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F14DATA1_FD3_Pos                    (3U)                              
#define CAN_F14DATA1_FD3_Msk                    (0x1U << CAN_F14DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F14DATA1_FD3                        CAN_F14DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F14DATA1_FD4_Pos                    (4U)                              
#define CAN_F14DATA1_FD4_Msk                    (0x1U << CAN_F14DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F14DATA1_FD4                        CAN_F14DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F14DATA1_FD5_Pos                    (5U)                              
#define CAN_F14DATA1_FD5_Msk                    (0x1U << CAN_F14DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F14DATA1_FD5                        CAN_F14DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F14DATA1_FD6_Pos                    (6U)                              
#define CAN_F14DATA1_FD6_Msk                    (0x1U << CAN_F14DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F14DATA1_FD6                        CAN_F14DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F14DATA1_FD7_Pos                    (7U)                              
#define CAN_F14DATA1_FD7_Msk                    (0x1U << CAN_F14DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F14DATA1_FD7                        CAN_F14DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F14DATA1_FD8_Pos                    (8U)                              
#define CAN_F14DATA1_FD8_Msk                    (0x1U << CAN_F14DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F14DATA1_FD8                        CAN_F14DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F14DATA1_FD9_Pos                    (9U)                              
#define CAN_F14DATA1_FD9_Msk                    (0x1U << CAN_F14DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F14DATA1_FD9                        CAN_F14DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F14DATA1_FD10_Pos                   (10U)                             
#define CAN_F14DATA1_FD10_Msk                   (0x1U << CAN_F14DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F14DATA1_FD10                       CAN_F14DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F14DATA1_FD11_Pos                   (11U)                             
#define CAN_F14DATA1_FD11_Msk                   (0x1U << CAN_F14DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F14DATA1_FD11                       CAN_F14DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F14DATA1_FD12_Pos                   (12U)                             
#define CAN_F14DATA1_FD12_Msk                   (0x1U << CAN_F14DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F14DATA1_FD12                       CAN_F14DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F14DATA1_FD13_Pos                   (13U)                             
#define CAN_F14DATA1_FD13_Msk                   (0x1U << CAN_F14DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F14DATA1_FD13                       CAN_F14DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F14DATA1_FD14_Pos                   (14U)                             
#define CAN_F14DATA1_FD14_Msk                   (0x1U << CAN_F14DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F14DATA1_FD14                       CAN_F14DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F14DATA1_FD15_Pos                   (15U)                             
#define CAN_F14DATA1_FD15_Msk                   (0x1U << CAN_F14DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F14DATA1_FD15                       CAN_F14DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F14DATA1_FD16_Pos                   (16U)                             
#define CAN_F14DATA1_FD16_Msk                   (0x1U << CAN_F14DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F14DATA1_FD16                       CAN_F14DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F14DATA1_FD17_Pos                   (17U)                             
#define CAN_F14DATA1_FD17_Msk                   (0x1U << CAN_F14DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F14DATA1_FD17                       CAN_F14DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F14DATA1_FD18_Pos                   (18U)                             
#define CAN_F14DATA1_FD18_Msk                   (0x1U << CAN_F14DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F14DATA1_FD18                       CAN_F14DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F14DATA1_FD19_Pos                   (19U)                             
#define CAN_F14DATA1_FD19_Msk                   (0x1U << CAN_F14DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F14DATA1_FD19                       CAN_F14DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F14DATA1_FD20_Pos                   (20U)                             
#define CAN_F14DATA1_FD20_Msk                   (0x1U << CAN_F14DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F14DATA1_FD20                       CAN_F14DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F14DATA1_FD21_Pos                   (21U)                             
#define CAN_F14DATA1_FD21_Msk                   (0x1U << CAN_F14DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F14DATA1_FD21                       CAN_F14DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F14DATA1_FD22_Pos                   (22U)                             
#define CAN_F14DATA1_FD22_Msk                   (0x1U << CAN_F14DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F14DATA1_FD22                       CAN_F14DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F14DATA1_FD23_Pos                   (23U)                             
#define CAN_F14DATA1_FD23_Msk                   (0x1U << CAN_F14DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F14DATA1_FD23                       CAN_F14DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F14DATA1_FD24_Pos                   (24U)                             
#define CAN_F14DATA1_FD24_Msk                   (0x1U << CAN_F14DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F14DATA1_FD24                       CAN_F14DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F14DATA1_FD25_Pos                   (25U)                             
#define CAN_F14DATA1_FD25_Msk                   (0x1U << CAN_F14DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F14DATA1_FD25                       CAN_F14DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F14DATA1_FD26_Pos                   (26U)                             
#define CAN_F14DATA1_FD26_Msk                   (0x1U << CAN_F14DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F14DATA1_FD26                       CAN_F14DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F14DATA1_FD27_Pos                   (27U)                             
#define CAN_F14DATA1_FD27_Msk                   (0x1U << CAN_F14DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F14DATA1_FD27                       CAN_F14DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F14DATA1_FD28_Pos                   (28U)                             
#define CAN_F14DATA1_FD28_Msk                   (0x1U << CAN_F14DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F14DATA1_FD28                       CAN_F14DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F14DATA1_FD29_Pos                   (29U)                             
#define CAN_F14DATA1_FD29_Msk                   (0x1U << CAN_F14DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F14DATA1_FD29                       CAN_F14DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F14DATA1_FD30_Pos                   (30U)                             
#define CAN_F14DATA1_FD30_Msk                   (0x1U << CAN_F14DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F14DATA1_FD30                       CAN_F14DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F14DATA1_FD31_Pos                   (31U)                             
#define CAN_F14DATA1_FD31_Msk                   (0x1U << CAN_F14DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F14DATA1_FD31                       CAN_F14DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F15R2 register  ******************/
#define CAN_F15DATA1_FD0_Pos                    (0U)                              
#define CAN_F15DATA1_FD0_Msk                    (0x1U << CAN_F15DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F15DATA1_FD0                        CAN_F15DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F15DATA1_FD1_Pos                    (1U)                              
#define CAN_F15DATA1_FD1_Msk                    (0x1U << CAN_F15DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F15DATA1_FD1                        CAN_F15DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F15DATA1_FD2_Pos                    (2U)                              
#define CAN_F15DATA1_FD2_Msk                    (0x1U << CAN_F15DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F15DATA1_FD2                        CAN_F15DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F15DATA1_FD3_Pos                    (3U)                              
#define CAN_F15DATA1_FD3_Msk                    (0x1U << CAN_F15DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F15DATA1_FD3                        CAN_F15DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F15DATA1_FD4_Pos                    (4U)                              
#define CAN_F15DATA1_FD4_Msk                    (0x1U << CAN_F15DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F15DATA1_FD4                        CAN_F15DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F15DATA1_FD5_Pos                    (5U)                              
#define CAN_F15DATA1_FD5_Msk                    (0x1U << CAN_F15DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F15DATA1_FD5                        CAN_F15DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F15DATA1_FD6_Pos                    (6U)                              
#define CAN_F15DATA1_FD6_Msk                    (0x1U << CAN_F15DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F15DATA1_FD6                        CAN_F15DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F15DATA1_FD7_Pos                    (7U)                              
#define CAN_F15DATA1_FD7_Msk                    (0x1U << CAN_F15DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F15DATA1_FD7                        CAN_F15DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F15DATA1_FD8_Pos                    (8U)                              
#define CAN_F15DATA1_FD8_Msk                    (0x1U << CAN_F15DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F15DATA1_FD8                        CAN_F15DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F15DATA1_FD9_Pos                    (9U)                              
#define CAN_F15DATA1_FD9_Msk                    (0x1U << CAN_F15DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F15DATA1_FD9                        CAN_F15DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F15DATA1_FD10_Pos                   (10U)                             
#define CAN_F15DATA1_FD10_Msk                   (0x1U << CAN_F15DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F15DATA1_FD10                       CAN_F15DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F15DATA1_FD11_Pos                   (11U)                             
#define CAN_F15DATA1_FD11_Msk                   (0x1U << CAN_F15DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F15DATA1_FD11                       CAN_F15DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F15DATA1_FD12_Pos                   (12U)                             
#define CAN_F15DATA1_FD12_Msk                   (0x1U << CAN_F15DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F15DATA1_FD12                       CAN_F15DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F15DATA1_FD13_Pos                   (13U)                             
#define CAN_F15DATA1_FD13_Msk                   (0x1U << CAN_F15DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F15DATA1_FD13                       CAN_F15DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F15DATA1_FD14_Pos                   (14U)                             
#define CAN_F15DATA1_FD14_Msk                   (0x1U << CAN_F15DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F15DATA1_FD14                       CAN_F15DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F15DATA1_FD15_Pos                   (15U)                             
#define CAN_F15DATA1_FD15_Msk                   (0x1U << CAN_F15DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F15DATA1_FD15                       CAN_F15DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F15DATA1_FD16_Pos                   (16U)                             
#define CAN_F15DATA1_FD16_Msk                   (0x1U << CAN_F15DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F15DATA1_FD16                       CAN_F15DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F15DATA1_FD17_Pos                   (17U)                             
#define CAN_F15DATA1_FD17_Msk                   (0x1U << CAN_F15DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F15DATA1_FD17                       CAN_F15DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F15DATA1_FD18_Pos                   (18U)                             
#define CAN_F15DATA1_FD18_Msk                   (0x1U << CAN_F15DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F15DATA1_FD18                       CAN_F15DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F15DATA1_FD19_Pos                   (19U)                             
#define CAN_F15DATA1_FD19_Msk                   (0x1U << CAN_F15DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F15DATA1_FD19                       CAN_F15DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F15DATA1_FD20_Pos                   (20U)                             
#define CAN_F15DATA1_FD20_Msk                   (0x1U << CAN_F15DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F15DATA1_FD20                       CAN_F15DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F15DATA1_FD21_Pos                   (21U)                             
#define CAN_F15DATA1_FD21_Msk                   (0x1U << CAN_F15DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F15DATA1_FD21                       CAN_F15DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F15DATA1_FD22_Pos                   (22U)                             
#define CAN_F15DATA1_FD22_Msk                   (0x1U << CAN_F15DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F15DATA1_FD22                       CAN_F15DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F15DATA1_FD23_Pos                   (23U)                             
#define CAN_F15DATA1_FD23_Msk                   (0x1U << CAN_F15DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F15DATA1_FD23                       CAN_F15DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F15DATA1_FD24_Pos                   (24U)                             
#define CAN_F15DATA1_FD24_Msk                   (0x1U << CAN_F15DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F15DATA1_FD24                       CAN_F15DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F15DATA1_FD25_Pos                   (25U)                             
#define CAN_F15DATA1_FD25_Msk                   (0x1U << CAN_F15DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F15DATA1_FD25                       CAN_F15DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F15DATA1_FD26_Pos                   (26U)                             
#define CAN_F15DATA1_FD26_Msk                   (0x1U << CAN_F15DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F15DATA1_FD26                       CAN_F15DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F15DATA1_FD27_Pos                   (27U)                             
#define CAN_F15DATA1_FD27_Msk                   (0x1U << CAN_F15DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F15DATA1_FD27                       CAN_F15DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F15DATA1_FD28_Pos                   (28U)                             
#define CAN_F15DATA1_FD28_Msk                   (0x1U << CAN_F15DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F15DATA1_FD28                       CAN_F15DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F15DATA1_FD29_Pos                   (29U)                             
#define CAN_F15DATA1_FD29_Msk                   (0x1U << CAN_F15DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F15DATA1_FD29                       CAN_F15DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F15DATA1_FD30_Pos                   (30U)                             
#define CAN_F15DATA1_FD30_Msk                   (0x1U << CAN_F15DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F15DATA1_FD30                       CAN_F15DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F15DATA1_FD31_Pos                   (31U)                             
#define CAN_F15DATA1_FD31_Msk                   (0x1U << CAN_F15DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F15DATA1_FD31                       CAN_F15DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F16R2 register  ******************/
#define CAN_F16DATA1_FD0_Pos                    (0U)                              
#define CAN_F16DATA1_FD0_Msk                    (0x1U << CAN_F16DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F16DATA1_FD0                        CAN_F16DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F16DATA1_FD1_Pos                    (1U)                              
#define CAN_F16DATA1_FD1_Msk                    (0x1U << CAN_F16DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F16DATA1_FD1                        CAN_F16DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F16DATA1_FD2_Pos                    (2U)                              
#define CAN_F16DATA1_FD2_Msk                    (0x1U << CAN_F16DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F16DATA1_FD2                        CAN_F16DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F16DATA1_FD3_Pos                    (3U)                              
#define CAN_F16DATA1_FD3_Msk                    (0x1U << CAN_F16DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F16DATA1_FD3                        CAN_F16DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F16DATA1_FD4_Pos                    (4U)                              
#define CAN_F16DATA1_FD4_Msk                    (0x1U << CAN_F16DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F16DATA1_FD4                        CAN_F16DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F16DATA1_FD5_Pos                    (5U)                              
#define CAN_F16DATA1_FD5_Msk                    (0x1U << CAN_F16DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F16DATA1_FD5                        CAN_F16DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F16DATA1_FD6_Pos                    (6U)                              
#define CAN_F16DATA1_FD6_Msk                    (0x1U << CAN_F16DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F16DATA1_FD6                        CAN_F16DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F16DATA1_FD7_Pos                    (7U)                              
#define CAN_F16DATA1_FD7_Msk                    (0x1U << CAN_F16DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F16DATA1_FD7                        CAN_F16DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F16DATA1_FD8_Pos                    (8U)                              
#define CAN_F16DATA1_FD8_Msk                    (0x1U << CAN_F16DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F16DATA1_FD8                        CAN_F16DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F16DATA1_FD9_Pos                    (9U)                              
#define CAN_F16DATA1_FD9_Msk                    (0x1U << CAN_F16DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F16DATA1_FD9                        CAN_F16DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F16DATA1_FD10_Pos                   (10U)                             
#define CAN_F16DATA1_FD10_Msk                   (0x1U << CAN_F16DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F16DATA1_FD10                       CAN_F16DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F16DATA1_FD11_Pos                   (11U)                             
#define CAN_F16DATA1_FD11_Msk                   (0x1U << CAN_F16DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F16DATA1_FD11                       CAN_F16DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F16DATA1_FD12_Pos                   (12U)                             
#define CAN_F16DATA1_FD12_Msk                   (0x1U << CAN_F16DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F16DATA1_FD12                       CAN_F16DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F16DATA1_FD13_Pos                   (13U)                             
#define CAN_F16DATA1_FD13_Msk                   (0x1U << CAN_F16DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F16DATA1_FD13                       CAN_F16DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F16DATA1_FD14_Pos                   (14U)                             
#define CAN_F16DATA1_FD14_Msk                   (0x1U << CAN_F16DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F16DATA1_FD14                       CAN_F16DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F16DATA1_FD15_Pos                   (15U)                             
#define CAN_F16DATA1_FD15_Msk                   (0x1U << CAN_F16DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F16DATA1_FD15                       CAN_F16DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F16DATA1_FD16_Pos                   (16U)                             
#define CAN_F16DATA1_FD16_Msk                   (0x1U << CAN_F16DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F16DATA1_FD16                       CAN_F16DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F16DATA1_FD17_Pos                   (17U)                             
#define CAN_F16DATA1_FD17_Msk                   (0x1U << CAN_F16DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F16DATA1_FD17                       CAN_F16DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F16DATA1_FD18_Pos                   (18U)                             
#define CAN_F16DATA1_FD18_Msk                   (0x1U << CAN_F16DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F16DATA1_FD18                       CAN_F16DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F16DATA1_FD19_Pos                   (19U)                             
#define CAN_F16DATA1_FD19_Msk                   (0x1U << CAN_F16DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F16DATA1_FD19                       CAN_F16DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F16DATA1_FD20_Pos                   (20U)                             
#define CAN_F16DATA1_FD20_Msk                   (0x1U << CAN_F16DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F16DATA1_FD20                       CAN_F16DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F16DATA1_FD21_Pos                   (21U)                             
#define CAN_F16DATA1_FD21_Msk                   (0x1U << CAN_F16DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F16DATA1_FD21                       CAN_F16DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F16DATA1_FD22_Pos                   (22U)                             
#define CAN_F16DATA1_FD22_Msk                   (0x1U << CAN_F16DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F16DATA1_FD22                       CAN_F16DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F16DATA1_FD23_Pos                   (23U)                             
#define CAN_F16DATA1_FD23_Msk                   (0x1U << CAN_F16DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F16DATA1_FD23                       CAN_F16DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F16DATA1_FD24_Pos                   (24U)                             
#define CAN_F16DATA1_FD24_Msk                   (0x1U << CAN_F16DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F16DATA1_FD24                       CAN_F16DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F16DATA1_FD25_Pos                   (25U)                             
#define CAN_F16DATA1_FD25_Msk                   (0x1U << CAN_F16DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F16DATA1_FD25                       CAN_F16DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F16DATA1_FD26_Pos                   (26U)                             
#define CAN_F16DATA1_FD26_Msk                   (0x1U << CAN_F16DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F16DATA1_FD26                       CAN_F16DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F16DATA1_FD27_Pos                   (27U)                             
#define CAN_F16DATA1_FD27_Msk                   (0x1U << CAN_F16DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F16DATA1_FD27                       CAN_F16DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F16DATA1_FD28_Pos                   (28U)                             
#define CAN_F16DATA1_FD28_Msk                   (0x1U << CAN_F16DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F16DATA1_FD28                       CAN_F16DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F16DATA1_FD29_Pos                   (29U)                             
#define CAN_F16DATA1_FD29_Msk                   (0x1U << CAN_F16DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F16DATA1_FD29                       CAN_F16DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F16DATA1_FD30_Pos                   (30U)                             
#define CAN_F16DATA1_FD30_Msk                   (0x1U << CAN_F16DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F16DATA1_FD30                       CAN_F16DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F16DATA1_FD31_Pos                   (31U)                             
#define CAN_F16DATA1_FD31_Msk                   (0x1U << CAN_F16DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F16DATA1_FD31                       CAN_F16DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F17R2 register  ******************/
#define CAN_F17DATA1_FD0_Pos                    (0U)                              
#define CAN_F17DATA1_FD0_Msk                    (0x1U << CAN_F17DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F17DATA1_FD0                        CAN_F17DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F17DATA1_FD1_Pos                    (1U)                              
#define CAN_F17DATA1_FD1_Msk                    (0x1U << CAN_F17DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F17DATA1_FD1                        CAN_F17DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F17DATA1_FD2_Pos                    (2U)                              
#define CAN_F17DATA1_FD2_Msk                    (0x1U << CAN_F17DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F17DATA1_FD2                        CAN_F17DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F17DATA1_FD3_Pos                    (3U)                              
#define CAN_F17DATA1_FD3_Msk                    (0x1U << CAN_F17DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F17DATA1_FD3                        CAN_F17DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F17DATA1_FD4_Pos                    (4U)                              
#define CAN_F17DATA1_FD4_Msk                    (0x1U << CAN_F17DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F17DATA1_FD4                        CAN_F17DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F17DATA1_FD5_Pos                    (5U)                              
#define CAN_F17DATA1_FD5_Msk                    (0x1U << CAN_F17DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F17DATA1_FD5                        CAN_F17DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F17DATA1_FD6_Pos                    (6U)                              
#define CAN_F17DATA1_FD6_Msk                    (0x1U << CAN_F17DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F17DATA1_FD6                        CAN_F17DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F17DATA1_FD7_Pos                    (7U)                              
#define CAN_F17DATA1_FD7_Msk                    (0x1U << CAN_F17DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F17DATA1_FD7                        CAN_F17DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F17DATA1_FD8_Pos                    (8U)                              
#define CAN_F17DATA1_FD8_Msk                    (0x1U << CAN_F17DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F17DATA1_FD8                        CAN_F17DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F17DATA1_FD9_Pos                    (9U)                              
#define CAN_F17DATA1_FD9_Msk                    (0x1U << CAN_F17DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F17DATA1_FD9                        CAN_F17DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F17DATA1_FD10_Pos                   (10U)                             
#define CAN_F17DATA1_FD10_Msk                   (0x1U << CAN_F17DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F17DATA1_FD10                       CAN_F17DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F17DATA1_FD11_Pos                   (11U)                             
#define CAN_F17DATA1_FD11_Msk                   (0x1U << CAN_F17DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F17DATA1_FD11                       CAN_F17DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F17DATA1_FD12_Pos                   (12U)                             
#define CAN_F17DATA1_FD12_Msk                   (0x1U << CAN_F17DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F17DATA1_FD12                       CAN_F17DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F17DATA1_FD13_Pos                   (13U)                             
#define CAN_F17DATA1_FD13_Msk                   (0x1U << CAN_F17DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F17DATA1_FD13                       CAN_F17DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F17DATA1_FD14_Pos                   (14U)                             
#define CAN_F17DATA1_FD14_Msk                   (0x1U << CAN_F17DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F17DATA1_FD14                       CAN_F17DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F17DATA1_FD15_Pos                   (15U)                             
#define CAN_F17DATA1_FD15_Msk                   (0x1U << CAN_F17DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F17DATA1_FD15                       CAN_F17DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F17DATA1_FD16_Pos                   (16U)                             
#define CAN_F17DATA1_FD16_Msk                   (0x1U << CAN_F17DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F17DATA1_FD16                       CAN_F17DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F17DATA1_FD17_Pos                   (17U)                             
#define CAN_F17DATA1_FD17_Msk                   (0x1U << CAN_F17DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F17DATA1_FD17                       CAN_F17DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F17DATA1_FD18_Pos                   (18U)                             
#define CAN_F17DATA1_FD18_Msk                   (0x1U << CAN_F17DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F17DATA1_FD18                       CAN_F17DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F17DATA1_FD19_Pos                   (19U)                             
#define CAN_F17DATA1_FD19_Msk                   (0x1U << CAN_F17DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F17DATA1_FD19                       CAN_F17DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F17DATA1_FD20_Pos                   (20U)                             
#define CAN_F17DATA1_FD20_Msk                   (0x1U << CAN_F17DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F17DATA1_FD20                       CAN_F17DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F17DATA1_FD21_Pos                   (21U)                             
#define CAN_F17DATA1_FD21_Msk                   (0x1U << CAN_F17DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F17DATA1_FD21                       CAN_F17DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F17DATA1_FD22_Pos                   (22U)                             
#define CAN_F17DATA1_FD22_Msk                   (0x1U << CAN_F17DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F17DATA1_FD22                       CAN_F17DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F17DATA1_FD23_Pos                   (23U)                             
#define CAN_F17DATA1_FD23_Msk                   (0x1U << CAN_F17DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F17DATA1_FD23                       CAN_F17DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F17DATA1_FD24_Pos                   (24U)                             
#define CAN_F17DATA1_FD24_Msk                   (0x1U << CAN_F17DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F17DATA1_FD24                       CAN_F17DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F17DATA1_FD25_Pos                   (25U)                             
#define CAN_F17DATA1_FD25_Msk                   (0x1U << CAN_F17DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F17DATA1_FD25                       CAN_F17DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F17DATA1_FD26_Pos                   (26U)                             
#define CAN_F17DATA1_FD26_Msk                   (0x1U << CAN_F17DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F17DATA1_FD26                       CAN_F17DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F17DATA1_FD27_Pos                   (27U)                             
#define CAN_F17DATA1_FD27_Msk                   (0x1U << CAN_F17DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F17DATA1_FD27                       CAN_F17DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F17DATA1_FD28_Pos                   (28U)                             
#define CAN_F17DATA1_FD28_Msk                   (0x1U << CAN_F17DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F17DATA1_FD28                       CAN_F17DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F17DATA1_FD29_Pos                   (29U)                             
#define CAN_F17DATA1_FD29_Msk                   (0x1U << CAN_F17DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F17DATA1_FD29                       CAN_F17DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F17DATA1_FD30_Pos                   (30U)                             
#define CAN_F17DATA1_FD30_Msk                   (0x1U << CAN_F17DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F17DATA1_FD30                       CAN_F17DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F17DATA1_FD31_Pos                   (31U)                             
#define CAN_F17DATA1_FD31_Msk                   (0x1U << CAN_F17DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F17DATA1_FD31                       CAN_F17DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F18R2 register  ******************/
#define CAN_F18DATA1_FD0_Pos                    (0U)                              
#define CAN_F18DATA1_FD0_Msk                    (0x1U << CAN_F18DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F18DATA1_FD0                        CAN_F18DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F18DATA1_FD1_Pos                    (1U)                              
#define CAN_F18DATA1_FD1_Msk                    (0x1U << CAN_F18DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F18DATA1_FD1                        CAN_F18DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F18DATA1_FD2_Pos                    (2U)                              
#define CAN_F18DATA1_FD2_Msk                    (0x1U << CAN_F18DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F18DATA1_FD2                        CAN_F18DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F18DATA1_FD3_Pos                    (3U)                              
#define CAN_F18DATA1_FD3_Msk                    (0x1U << CAN_F18DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F18DATA1_FD3                        CAN_F18DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F18DATA1_FD4_Pos                    (4U)                              
#define CAN_F18DATA1_FD4_Msk                    (0x1U << CAN_F18DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F18DATA1_FD4                        CAN_F18DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F18DATA1_FD5_Pos                    (5U)                              
#define CAN_F18DATA1_FD5_Msk                    (0x1U << CAN_F18DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F18DATA1_FD5                        CAN_F18DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F18DATA1_FD6_Pos                    (6U)                              
#define CAN_F18DATA1_FD6_Msk                    (0x1U << CAN_F18DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F18DATA1_FD6                        CAN_F18DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F18DATA1_FD7_Pos                    (7U)                              
#define CAN_F18DATA1_FD7_Msk                    (0x1U << CAN_F18DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F18DATA1_FD7                        CAN_F18DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F18DATA1_FD8_Pos                    (8U)                              
#define CAN_F18DATA1_FD8_Msk                    (0x1U << CAN_F18DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F18DATA1_FD8                        CAN_F18DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F18DATA1_FD9_Pos                    (9U)                              
#define CAN_F18DATA1_FD9_Msk                    (0x1U << CAN_F18DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F18DATA1_FD9                        CAN_F18DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F18DATA1_FD10_Pos                   (10U)                             
#define CAN_F18DATA1_FD10_Msk                   (0x1U << CAN_F18DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F18DATA1_FD10                       CAN_F18DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F18DATA1_FD11_Pos                   (11U)                             
#define CAN_F18DATA1_FD11_Msk                   (0x1U << CAN_F18DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F18DATA1_FD11                       CAN_F18DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F18DATA1_FD12_Pos                   (12U)                             
#define CAN_F18DATA1_FD12_Msk                   (0x1U << CAN_F18DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F18DATA1_FD12                       CAN_F18DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F18DATA1_FD13_Pos                   (13U)                             
#define CAN_F18DATA1_FD13_Msk                   (0x1U << CAN_F18DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F18DATA1_FD13                       CAN_F18DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F18DATA1_FD14_Pos                   (14U)                             
#define CAN_F18DATA1_FD14_Msk                   (0x1U << CAN_F18DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F18DATA1_FD14                       CAN_F18DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F18DATA1_FD15_Pos                   (15U)                             
#define CAN_F18DATA1_FD15_Msk                   (0x1U << CAN_F18DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F18DATA1_FD15                       CAN_F18DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F18DATA1_FD16_Pos                   (16U)                             
#define CAN_F18DATA1_FD16_Msk                   (0x1U << CAN_F18DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F18DATA1_FD16                       CAN_F18DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F18DATA1_FD17_Pos                   (17U)                             
#define CAN_F18DATA1_FD17_Msk                   (0x1U << CAN_F18DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F18DATA1_FD17                       CAN_F18DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F18DATA1_FD18_Pos                   (18U)                             
#define CAN_F18DATA1_FD18_Msk                   (0x1U << CAN_F18DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F18DATA1_FD18                       CAN_F18DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F18DATA1_FD19_Pos                   (19U)                             
#define CAN_F18DATA1_FD19_Msk                   (0x1U << CAN_F18DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F18DATA1_FD19                       CAN_F18DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F18DATA1_FD20_Pos                   (20U)                             
#define CAN_F18DATA1_FD20_Msk                   (0x1U << CAN_F18DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F18DATA1_FD20                       CAN_F18DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F18DATA1_FD21_Pos                   (21U)                             
#define CAN_F18DATA1_FD21_Msk                   (0x1U << CAN_F18DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F18DATA1_FD21                       CAN_F18DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F18DATA1_FD22_Pos                   (22U)                             
#define CAN_F18DATA1_FD22_Msk                   (0x1U << CAN_F18DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F18DATA1_FD22                       CAN_F18DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F18DATA1_FD23_Pos                   (23U)                             
#define CAN_F18DATA1_FD23_Msk                   (0x1U << CAN_F18DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F18DATA1_FD23                       CAN_F18DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F18DATA1_FD24_Pos                   (24U)                             
#define CAN_F18DATA1_FD24_Msk                   (0x1U << CAN_F18DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F18DATA1_FD24                       CAN_F18DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F18DATA1_FD25_Pos                   (25U)                             
#define CAN_F18DATA1_FD25_Msk                   (0x1U << CAN_F18DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F18DATA1_FD25                       CAN_F18DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F18DATA1_FD26_Pos                   (26U)                             
#define CAN_F18DATA1_FD26_Msk                   (0x1U << CAN_F18DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F18DATA1_FD26                       CAN_F18DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F18DATA1_FD27_Pos                   (27U)                             
#define CAN_F18DATA1_FD27_Msk                   (0x1U << CAN_F18DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F18DATA1_FD27                       CAN_F18DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F18DATA1_FD28_Pos                   (28U)                             
#define CAN_F18DATA1_FD28_Msk                   (0x1U << CAN_F18DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F18DATA1_FD28                       CAN_F18DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F18DATA1_FD29_Pos                   (29U)                             
#define CAN_F18DATA1_FD29_Msk                   (0x1U << CAN_F18DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F18DATA1_FD29                       CAN_F18DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F18DATA1_FD30_Pos                   (30U)                             
#define CAN_F18DATA1_FD30_Msk                   (0x1U << CAN_F18DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F18DATA1_FD30                       CAN_F18DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F18DATA1_FD31_Pos                   (31U)                             
#define CAN_F18DATA1_FD31_Msk                   (0x1U << CAN_F18DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F18DATA1_FD31                       CAN_F18DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F19R2 register  ******************/
#define CAN_F19DATA1_FD0_Pos                    (0U)                              
#define CAN_F19DATA1_FD0_Msk                    (0x1U << CAN_F19DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F19DATA1_FD0                        CAN_F19DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F19DATA1_FD1_Pos                    (1U)                              
#define CAN_F19DATA1_FD1_Msk                    (0x1U << CAN_F19DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F19DATA1_FD1                        CAN_F19DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F19DATA1_FD2_Pos                    (2U)                              
#define CAN_F19DATA1_FD2_Msk                    (0x1U << CAN_F19DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F19DATA1_FD2                        CAN_F19DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F19DATA1_FD3_Pos                    (3U)                              
#define CAN_F19DATA1_FD3_Msk                    (0x1U << CAN_F19DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F19DATA1_FD3                        CAN_F19DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F19DATA1_FD4_Pos                    (4U)                              
#define CAN_F19DATA1_FD4_Msk                    (0x1U << CAN_F19DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F19DATA1_FD4                        CAN_F19DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F19DATA1_FD5_Pos                    (5U)                              
#define CAN_F19DATA1_FD5_Msk                    (0x1U << CAN_F19DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F19DATA1_FD5                        CAN_F19DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F19DATA1_FD6_Pos                    (6U)                              
#define CAN_F19DATA1_FD6_Msk                    (0x1U << CAN_F19DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F19DATA1_FD6                        CAN_F19DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F19DATA1_FD7_Pos                    (7U)                              
#define CAN_F19DATA1_FD7_Msk                    (0x1U << CAN_F19DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F19DATA1_FD7                        CAN_F19DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F19DATA1_FD8_Pos                    (8U)                              
#define CAN_F19DATA1_FD8_Msk                    (0x1U << CAN_F19DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F19DATA1_FD8                        CAN_F19DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F19DATA1_FD9_Pos                    (9U)                              
#define CAN_F19DATA1_FD9_Msk                    (0x1U << CAN_F19DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F19DATA1_FD9                        CAN_F19DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F19DATA1_FD10_Pos                   (10U)                             
#define CAN_F19DATA1_FD10_Msk                   (0x1U << CAN_F19DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F19DATA1_FD10                       CAN_F19DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F19DATA1_FD11_Pos                   (11U)                             
#define CAN_F19DATA1_FD11_Msk                   (0x1U << CAN_F19DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F19DATA1_FD11                       CAN_F19DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F19DATA1_FD12_Pos                   (12U)                             
#define CAN_F19DATA1_FD12_Msk                   (0x1U << CAN_F19DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F19DATA1_FD12                       CAN_F19DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F19DATA1_FD13_Pos                   (13U)                             
#define CAN_F19DATA1_FD13_Msk                   (0x1U << CAN_F19DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F19DATA1_FD13                       CAN_F19DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F19DATA1_FD14_Pos                   (14U)                             
#define CAN_F19DATA1_FD14_Msk                   (0x1U << CAN_F19DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F19DATA1_FD14                       CAN_F19DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F19DATA1_FD15_Pos                   (15U)                             
#define CAN_F19DATA1_FD15_Msk                   (0x1U << CAN_F19DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F19DATA1_FD15                       CAN_F19DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F19DATA1_FD16_Pos                   (16U)                             
#define CAN_F19DATA1_FD16_Msk                   (0x1U << CAN_F19DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F19DATA1_FD16                       CAN_F19DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F19DATA1_FD17_Pos                   (17U)                             
#define CAN_F19DATA1_FD17_Msk                   (0x1U << CAN_F19DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F19DATA1_FD17                       CAN_F19DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F19DATA1_FD18_Pos                   (18U)                             
#define CAN_F19DATA1_FD18_Msk                   (0x1U << CAN_F19DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F19DATA1_FD18                       CAN_F19DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F19DATA1_FD19_Pos                   (19U)                             
#define CAN_F19DATA1_FD19_Msk                   (0x1U << CAN_F19DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F19DATA1_FD19                       CAN_F19DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F19DATA1_FD20_Pos                   (20U)                             
#define CAN_F19DATA1_FD20_Msk                   (0x1U << CAN_F19DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F19DATA1_FD20                       CAN_F19DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F19DATA1_FD21_Pos                   (21U)                             
#define CAN_F19DATA1_FD21_Msk                   (0x1U << CAN_F19DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F19DATA1_FD21                       CAN_F19DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F19DATA1_FD22_Pos                   (22U)                             
#define CAN_F19DATA1_FD22_Msk                   (0x1U << CAN_F19DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F19DATA1_FD22                       CAN_F19DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F19DATA1_FD23_Pos                   (23U)                             
#define CAN_F19DATA1_FD23_Msk                   (0x1U << CAN_F19DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F19DATA1_FD23                       CAN_F19DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F19DATA1_FD24_Pos                   (24U)                             
#define CAN_F19DATA1_FD24_Msk                   (0x1U << CAN_F19DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F19DATA1_FD24                       CAN_F19DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F19DATA1_FD25_Pos                   (25U)                             
#define CAN_F19DATA1_FD25_Msk                   (0x1U << CAN_F19DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F19DATA1_FD25                       CAN_F19DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F19DATA1_FD26_Pos                   (26U)                             
#define CAN_F19DATA1_FD26_Msk                   (0x1U << CAN_F19DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F19DATA1_FD26                       CAN_F19DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F19DATA1_FD27_Pos                   (27U)                             
#define CAN_F19DATA1_FD27_Msk                   (0x1U << CAN_F19DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F19DATA1_FD27                       CAN_F19DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F19DATA1_FD28_Pos                   (28U)                             
#define CAN_F19DATA1_FD28_Msk                   (0x1U << CAN_F19DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F19DATA1_FD28                       CAN_F19DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F19DATA1_FD29_Pos                   (29U)                             
#define CAN_F19DATA1_FD29_Msk                   (0x1U << CAN_F19DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F19DATA1_FD29                       CAN_F19DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F19DATA1_FD30_Pos                   (30U)                             
#define CAN_F19DATA1_FD30_Msk                   (0x1U << CAN_F19DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F19DATA1_FD30                       CAN_F19DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F19DATA1_FD31_Pos                   (31U)                             
#define CAN_F19DATA1_FD31_Msk                   (0x1U << CAN_F19DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F19DATA1_FD31                       CAN_F19DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F20R2 register  ******************/
#define CAN_F20DATA1_FD0_Pos                    (0U)                              
#define CAN_F20DATA1_FD0_Msk                    (0x1U << CAN_F20DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F20DATA1_FD0                        CAN_F20DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F20DATA1_FD1_Pos                    (1U)                              
#define CAN_F20DATA1_FD1_Msk                    (0x1U << CAN_F20DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F20DATA1_FD1                        CAN_F20DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F20DATA1_FD2_Pos                    (2U)                              
#define CAN_F20DATA1_FD2_Msk                    (0x1U << CAN_F20DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F20DATA1_FD2                        CAN_F20DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F20DATA1_FD3_Pos                    (3U)                              
#define CAN_F20DATA1_FD3_Msk                    (0x1U << CAN_F20DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F20DATA1_FD3                        CAN_F20DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F20DATA1_FD4_Pos                    (4U)                              
#define CAN_F20DATA1_FD4_Msk                    (0x1U << CAN_F20DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F20DATA1_FD4                        CAN_F20DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F20DATA1_FD5_Pos                    (5U)                              
#define CAN_F20DATA1_FD5_Msk                    (0x1U << CAN_F20DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F20DATA1_FD5                        CAN_F20DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F20DATA1_FD6_Pos                    (6U)                              
#define CAN_F20DATA1_FD6_Msk                    (0x1U << CAN_F20DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F20DATA1_FD6                        CAN_F20DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F20DATA1_FD7_Pos                    (7U)                              
#define CAN_F20DATA1_FD7_Msk                    (0x1U << CAN_F20DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F20DATA1_FD7                        CAN_F20DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F20DATA1_FD8_Pos                    (8U)                              
#define CAN_F20DATA1_FD8_Msk                    (0x1U << CAN_F20DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F20DATA1_FD8                        CAN_F20DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F20DATA1_FD9_Pos                    (9U)                              
#define CAN_F20DATA1_FD9_Msk                    (0x1U << CAN_F20DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F20DATA1_FD9                        CAN_F20DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F20DATA1_FD10_Pos                   (10U)                             
#define CAN_F20DATA1_FD10_Msk                   (0x1U << CAN_F20DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F20DATA1_FD10                       CAN_F20DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F20DATA1_FD11_Pos                   (11U)                             
#define CAN_F20DATA1_FD11_Msk                   (0x1U << CAN_F20DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F20DATA1_FD11                       CAN_F20DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F20DATA1_FD12_Pos                   (12U)                             
#define CAN_F20DATA1_FD12_Msk                   (0x1U << CAN_F20DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F20DATA1_FD12                       CAN_F20DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F20DATA1_FD13_Pos                   (13U)                             
#define CAN_F20DATA1_FD13_Msk                   (0x1U << CAN_F20DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F20DATA1_FD13                       CAN_F20DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F20DATA1_FD14_Pos                   (14U)                             
#define CAN_F20DATA1_FD14_Msk                   (0x1U << CAN_F20DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F20DATA1_FD14                       CAN_F20DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F20DATA1_FD15_Pos                   (15U)                             
#define CAN_F20DATA1_FD15_Msk                   (0x1U << CAN_F20DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F20DATA1_FD15                       CAN_F20DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F20DATA1_FD16_Pos                   (16U)                             
#define CAN_F20DATA1_FD16_Msk                   (0x1U << CAN_F20DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F20DATA1_FD16                       CAN_F20DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F20DATA1_FD17_Pos                   (17U)                             
#define CAN_F20DATA1_FD17_Msk                   (0x1U << CAN_F20DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F20DATA1_FD17                       CAN_F20DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F20DATA1_FD18_Pos                   (18U)                             
#define CAN_F20DATA1_FD18_Msk                   (0x1U << CAN_F20DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F20DATA1_FD18                       CAN_F20DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F20DATA1_FD19_Pos                   (19U)                             
#define CAN_F20DATA1_FD19_Msk                   (0x1U << CAN_F20DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F20DATA1_FD19                       CAN_F20DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F20DATA1_FD20_Pos                   (20U)                             
#define CAN_F20DATA1_FD20_Msk                   (0x1U << CAN_F20DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F20DATA1_FD20                       CAN_F20DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F20DATA1_FD21_Pos                   (21U)                             
#define CAN_F20DATA1_FD21_Msk                   (0x1U << CAN_F20DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F20DATA1_FD21                       CAN_F20DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F20DATA1_FD22_Pos                   (22U)                             
#define CAN_F20DATA1_FD22_Msk                   (0x1U << CAN_F20DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F20DATA1_FD22                       CAN_F20DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F20DATA1_FD23_Pos                   (23U)                             
#define CAN_F20DATA1_FD23_Msk                   (0x1U << CAN_F20DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F20DATA1_FD23                       CAN_F20DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F20DATA1_FD24_Pos                   (24U)                             
#define CAN_F20DATA1_FD24_Msk                   (0x1U << CAN_F20DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F20DATA1_FD24                       CAN_F20DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F20DATA1_FD25_Pos                   (25U)                             
#define CAN_F20DATA1_FD25_Msk                   (0x1U << CAN_F20DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F20DATA1_FD25                       CAN_F20DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F20DATA1_FD26_Pos                   (26U)                             
#define CAN_F20DATA1_FD26_Msk                   (0x1U << CAN_F20DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F20DATA1_FD26                       CAN_F20DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F20DATA1_FD27_Pos                   (27U)                             
#define CAN_F20DATA1_FD27_Msk                   (0x1U << CAN_F20DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F20DATA1_FD27                       CAN_F20DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F20DATA1_FD28_Pos                   (28U)                             
#define CAN_F20DATA1_FD28_Msk                   (0x1U << CAN_F20DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F20DATA1_FD28                       CAN_F20DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F20DATA1_FD29_Pos                   (29U)                             
#define CAN_F20DATA1_FD29_Msk                   (0x1U << CAN_F20DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F20DATA1_FD29                       CAN_F20DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F20DATA1_FD30_Pos                   (30U)                             
#define CAN_F20DATA1_FD30_Msk                   (0x1U << CAN_F20DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F20DATA1_FD30                       CAN_F20DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F20DATA1_FD31_Pos                   (31U)                             
#define CAN_F20DATA1_FD31_Msk                   (0x1U << CAN_F20DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F20DATA1_FD31                       CAN_F20DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F21R2 register  ******************/
#define CAN_F21DATA1_FD0_Pos                    (0U)                              
#define CAN_F21DATA1_FD0_Msk                    (0x1U << CAN_F21DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F21DATA1_FD0                        CAN_F21DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F21DATA1_FD1_Pos                    (1U)                              
#define CAN_F21DATA1_FD1_Msk                    (0x1U << CAN_F21DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F21DATA1_FD1                        CAN_F21DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F21DATA1_FD2_Pos                    (2U)                              
#define CAN_F21DATA1_FD2_Msk                    (0x1U << CAN_F21DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F21DATA1_FD2                        CAN_F21DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F21DATA1_FD3_Pos                    (3U)                              
#define CAN_F21DATA1_FD3_Msk                    (0x1U << CAN_F21DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F21DATA1_FD3                        CAN_F21DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F21DATA1_FD4_Pos                    (4U)                              
#define CAN_F21DATA1_FD4_Msk                    (0x1U << CAN_F21DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F21DATA1_FD4                        CAN_F21DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F21DATA1_FD5_Pos                    (5U)                              
#define CAN_F21DATA1_FD5_Msk                    (0x1U << CAN_F21DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F21DATA1_FD5                        CAN_F21DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F21DATA1_FD6_Pos                    (6U)                              
#define CAN_F21DATA1_FD6_Msk                    (0x1U << CAN_F21DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F21DATA1_FD6                        CAN_F21DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F21DATA1_FD7_Pos                    (7U)                              
#define CAN_F21DATA1_FD7_Msk                    (0x1U << CAN_F21DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F21DATA1_FD7                        CAN_F21DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F21DATA1_FD8_Pos                    (8U)                              
#define CAN_F21DATA1_FD8_Msk                    (0x1U << CAN_F21DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F21DATA1_FD8                        CAN_F21DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F21DATA1_FD9_Pos                    (9U)                              
#define CAN_F21DATA1_FD9_Msk                    (0x1U << CAN_F21DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F21DATA1_FD9                        CAN_F21DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F21DATA1_FD10_Pos                   (10U)                             
#define CAN_F21DATA1_FD10_Msk                   (0x1U << CAN_F21DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F21DATA1_FD10                       CAN_F21DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F21DATA1_FD11_Pos                   (11U)                             
#define CAN_F21DATA1_FD11_Msk                   (0x1U << CAN_F21DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F21DATA1_FD11                       CAN_F21DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F21DATA1_FD12_Pos                   (12U)                             
#define CAN_F21DATA1_FD12_Msk                   (0x1U << CAN_F21DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F21DATA1_FD12                       CAN_F21DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F21DATA1_FD13_Pos                   (13U)                             
#define CAN_F21DATA1_FD13_Msk                   (0x1U << CAN_F21DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F21DATA1_FD13                       CAN_F21DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F21DATA1_FD14_Pos                   (14U)                             
#define CAN_F21DATA1_FD14_Msk                   (0x1U << CAN_F21DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F21DATA1_FD14                       CAN_F21DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F21DATA1_FD15_Pos                   (15U)                             
#define CAN_F21DATA1_FD15_Msk                   (0x1U << CAN_F21DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F21DATA1_FD15                       CAN_F21DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F21DATA1_FD16_Pos                   (16U)                             
#define CAN_F21DATA1_FD16_Msk                   (0x1U << CAN_F21DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F21DATA1_FD16                       CAN_F21DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F21DATA1_FD17_Pos                   (17U)                             
#define CAN_F21DATA1_FD17_Msk                   (0x1U << CAN_F21DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F21DATA1_FD17                       CAN_F21DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F21DATA1_FD18_Pos                   (18U)                             
#define CAN_F21DATA1_FD18_Msk                   (0x1U << CAN_F21DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F21DATA1_FD18                       CAN_F21DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F21DATA1_FD19_Pos                   (19U)                             
#define CAN_F21DATA1_FD19_Msk                   (0x1U << CAN_F21DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F21DATA1_FD19                       CAN_F21DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F21DATA1_FD20_Pos                   (20U)                             
#define CAN_F21DATA1_FD20_Msk                   (0x1U << CAN_F21DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F21DATA1_FD20                       CAN_F21DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F21DATA1_FD21_Pos                   (21U)                             
#define CAN_F21DATA1_FD21_Msk                   (0x1U << CAN_F21DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F21DATA1_FD21                       CAN_F21DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F21DATA1_FD22_Pos                   (22U)                             
#define CAN_F21DATA1_FD22_Msk                   (0x1U << CAN_F21DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F21DATA1_FD22                       CAN_F21DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F21DATA1_FD23_Pos                   (23U)                             
#define CAN_F21DATA1_FD23_Msk                   (0x1U << CAN_F21DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F21DATA1_FD23                       CAN_F21DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F21DATA1_FD24_Pos                   (24U)                             
#define CAN_F21DATA1_FD24_Msk                   (0x1U << CAN_F21DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F21DATA1_FD24                       CAN_F21DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F21DATA1_FD25_Pos                   (25U)                             
#define CAN_F21DATA1_FD25_Msk                   (0x1U << CAN_F21DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F21DATA1_FD25                       CAN_F21DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F21DATA1_FD26_Pos                   (26U)                             
#define CAN_F21DATA1_FD26_Msk                   (0x1U << CAN_F21DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F21DATA1_FD26                       CAN_F21DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F21DATA1_FD27_Pos                   (27U)                             
#define CAN_F21DATA1_FD27_Msk                   (0x1U << CAN_F21DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F21DATA1_FD27                       CAN_F21DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F21DATA1_FD28_Pos                   (28U)                             
#define CAN_F21DATA1_FD28_Msk                   (0x1U << CAN_F21DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F21DATA1_FD28                       CAN_F21DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F21DATA1_FD29_Pos                   (29U)                             
#define CAN_F21DATA1_FD29_Msk                   (0x1U << CAN_F21DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F21DATA1_FD29                       CAN_F21DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F21DATA1_FD30_Pos                   (30U)                             
#define CAN_F21DATA1_FD30_Msk                   (0x1U << CAN_F21DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F21DATA1_FD30                       CAN_F21DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F21DATA1_FD31_Pos                   (31U)                             
#define CAN_F21DATA1_FD31_Msk                   (0x1U << CAN_F21DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F21DATA1_FD31                       CAN_F21DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F22R2 register  ******************/
#define CAN_F22DATA1_FD0_Pos                    (0U)                              
#define CAN_F22DATA1_FD0_Msk                    (0x1U << CAN_F22DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F22DATA1_FD0                        CAN_F22DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F22DATA1_FD1_Pos                    (1U)                              
#define CAN_F22DATA1_FD1_Msk                    (0x1U << CAN_F22DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F22DATA1_FD1                        CAN_F22DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F22DATA1_FD2_Pos                    (2U)                              
#define CAN_F22DATA1_FD2_Msk                    (0x1U << CAN_F22DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F22DATA1_FD2                        CAN_F22DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F22DATA1_FD3_Pos                    (3U)                              
#define CAN_F22DATA1_FD3_Msk                    (0x1U << CAN_F22DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F22DATA1_FD3                        CAN_F22DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F22DATA1_FD4_Pos                    (4U)                              
#define CAN_F22DATA1_FD4_Msk                    (0x1U << CAN_F22DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F22DATA1_FD4                        CAN_F22DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F22DATA1_FD5_Pos                    (5U)                              
#define CAN_F22DATA1_FD5_Msk                    (0x1U << CAN_F22DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F22DATA1_FD5                        CAN_F22DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F22DATA1_FD6_Pos                    (6U)                              
#define CAN_F22DATA1_FD6_Msk                    (0x1U << CAN_F22DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F22DATA1_FD6                        CAN_F22DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F22DATA1_FD7_Pos                    (7U)                              
#define CAN_F22DATA1_FD7_Msk                    (0x1U << CAN_F22DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F22DATA1_FD7                        CAN_F22DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F22DATA1_FD8_Pos                    (8U)                              
#define CAN_F22DATA1_FD8_Msk                    (0x1U << CAN_F22DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F22DATA1_FD8                        CAN_F22DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F22DATA1_FD9_Pos                    (9U)                              
#define CAN_F22DATA1_FD9_Msk                    (0x1U << CAN_F22DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F22DATA1_FD9                        CAN_F22DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F22DATA1_FD10_Pos                   (10U)                             
#define CAN_F22DATA1_FD10_Msk                   (0x1U << CAN_F22DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F22DATA1_FD10                       CAN_F22DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F22DATA1_FD11_Pos                   (11U)                             
#define CAN_F22DATA1_FD11_Msk                   (0x1U << CAN_F22DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F22DATA1_FD11                       CAN_F22DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F22DATA1_FD12_Pos                   (12U)                             
#define CAN_F22DATA1_FD12_Msk                   (0x1U << CAN_F22DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F22DATA1_FD12                       CAN_F22DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F22DATA1_FD13_Pos                   (13U)                             
#define CAN_F22DATA1_FD13_Msk                   (0x1U << CAN_F22DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F22DATA1_FD13                       CAN_F22DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F22DATA1_FD14_Pos                   (14U)                             
#define CAN_F22DATA1_FD14_Msk                   (0x1U << CAN_F22DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F22DATA1_FD14                       CAN_F22DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F22DATA1_FD15_Pos                   (15U)                             
#define CAN_F22DATA1_FD15_Msk                   (0x1U << CAN_F22DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F22DATA1_FD15                       CAN_F22DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F22DATA1_FD16_Pos                   (16U)                             
#define CAN_F22DATA1_FD16_Msk                   (0x1U << CAN_F22DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F22DATA1_FD16                       CAN_F22DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F22DATA1_FD17_Pos                   (17U)                             
#define CAN_F22DATA1_FD17_Msk                   (0x1U << CAN_F22DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F22DATA1_FD17                       CAN_F22DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F22DATA1_FD18_Pos                   (18U)                             
#define CAN_F22DATA1_FD18_Msk                   (0x1U << CAN_F22DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F22DATA1_FD18                       CAN_F22DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F22DATA1_FD19_Pos                   (19U)                             
#define CAN_F22DATA1_FD19_Msk                   (0x1U << CAN_F22DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F22DATA1_FD19                       CAN_F22DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F22DATA1_FD20_Pos                   (20U)                             
#define CAN_F22DATA1_FD20_Msk                   (0x1U << CAN_F22DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F22DATA1_FD20                       CAN_F22DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F22DATA1_FD21_Pos                   (21U)                             
#define CAN_F22DATA1_FD21_Msk                   (0x1U << CAN_F22DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F22DATA1_FD21                       CAN_F22DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F22DATA1_FD22_Pos                   (22U)                             
#define CAN_F22DATA1_FD22_Msk                   (0x1U << CAN_F22DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F22DATA1_FD22                       CAN_F22DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F22DATA1_FD23_Pos                   (23U)                             
#define CAN_F22DATA1_FD23_Msk                   (0x1U << CAN_F22DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F22DATA1_FD23                       CAN_F22DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F22DATA1_FD24_Pos                   (24U)                             
#define CAN_F22DATA1_FD24_Msk                   (0x1U << CAN_F22DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F22DATA1_FD24                       CAN_F22DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F22DATA1_FD25_Pos                   (25U)                             
#define CAN_F22DATA1_FD25_Msk                   (0x1U << CAN_F22DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F22DATA1_FD25                       CAN_F22DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F22DATA1_FD26_Pos                   (26U)                             
#define CAN_F22DATA1_FD26_Msk                   (0x1U << CAN_F22DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F22DATA1_FD26                       CAN_F22DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F22DATA1_FD27_Pos                   (27U)                             
#define CAN_F22DATA1_FD27_Msk                   (0x1U << CAN_F22DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F22DATA1_FD27                       CAN_F22DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F22DATA1_FD28_Pos                   (28U)                             
#define CAN_F22DATA1_FD28_Msk                   (0x1U << CAN_F22DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F22DATA1_FD28                       CAN_F22DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F22DATA1_FD29_Pos                   (29U)                             
#define CAN_F22DATA1_FD29_Msk                   (0x1U << CAN_F22DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F22DATA1_FD29                       CAN_F22DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F22DATA1_FD30_Pos                   (30U)                             
#define CAN_F22DATA1_FD30_Msk                   (0x1U << CAN_F22DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F22DATA1_FD30                       CAN_F22DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F22DATA1_FD31_Pos                   (31U)                             
#define CAN_F22DATA1_FD31_Msk                   (0x1U << CAN_F22DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F22DATA1_FD31                       CAN_F22DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F23R2 register  ******************/
#define CAN_F23DATA1_FD0_Pos                    (0U)                              
#define CAN_F23DATA1_FD0_Msk                    (0x1U << CAN_F23DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F23DATA1_FD0                        CAN_F23DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F23DATA1_FD1_Pos                    (1U)                              
#define CAN_F23DATA1_FD1_Msk                    (0x1U << CAN_F23DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F23DATA1_FD1                        CAN_F23DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F23DATA1_FD2_Pos                    (2U)                              
#define CAN_F23DATA1_FD2_Msk                    (0x1U << CAN_F23DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F23DATA1_FD2                        CAN_F23DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F23DATA1_FD3_Pos                    (3U)                              
#define CAN_F23DATA1_FD3_Msk                    (0x1U << CAN_F23DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F23DATA1_FD3                        CAN_F23DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F23DATA1_FD4_Pos                    (4U)                              
#define CAN_F23DATA1_FD4_Msk                    (0x1U << CAN_F23DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F23DATA1_FD4                        CAN_F23DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F23DATA1_FD5_Pos                    (5U)                              
#define CAN_F23DATA1_FD5_Msk                    (0x1U << CAN_F23DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F23DATA1_FD5                        CAN_F23DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F23DATA1_FD6_Pos                    (6U)                              
#define CAN_F23DATA1_FD6_Msk                    (0x1U << CAN_F23DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F23DATA1_FD6                        CAN_F23DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F23DATA1_FD7_Pos                    (7U)                              
#define CAN_F23DATA1_FD7_Msk                    (0x1U << CAN_F23DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F23DATA1_FD7                        CAN_F23DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F23DATA1_FD8_Pos                    (8U)                              
#define CAN_F23DATA1_FD8_Msk                    (0x1U << CAN_F23DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F23DATA1_FD8                        CAN_F23DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F23DATA1_FD9_Pos                    (9U)                              
#define CAN_F23DATA1_FD9_Msk                    (0x1U << CAN_F23DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F23DATA1_FD9                        CAN_F23DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F23DATA1_FD10_Pos                   (10U)                             
#define CAN_F23DATA1_FD10_Msk                   (0x1U << CAN_F23DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F23DATA1_FD10                       CAN_F23DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F23DATA1_FD11_Pos                   (11U)                             
#define CAN_F23DATA1_FD11_Msk                   (0x1U << CAN_F23DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F23DATA1_FD11                       CAN_F23DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F23DATA1_FD12_Pos                   (12U)                             
#define CAN_F23DATA1_FD12_Msk                   (0x1U << CAN_F23DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F23DATA1_FD12                       CAN_F23DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F23DATA1_FD13_Pos                   (13U)                             
#define CAN_F23DATA1_FD13_Msk                   (0x1U << CAN_F23DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F23DATA1_FD13                       CAN_F23DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F23DATA1_FD14_Pos                   (14U)                             
#define CAN_F23DATA1_FD14_Msk                   (0x1U << CAN_F23DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F23DATA1_FD14                       CAN_F23DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F23DATA1_FD15_Pos                   (15U)                             
#define CAN_F23DATA1_FD15_Msk                   (0x1U << CAN_F23DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F23DATA1_FD15                       CAN_F23DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F23DATA1_FD16_Pos                   (16U)                             
#define CAN_F23DATA1_FD16_Msk                   (0x1U << CAN_F23DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F23DATA1_FD16                       CAN_F23DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F23DATA1_FD17_Pos                   (17U)                             
#define CAN_F23DATA1_FD17_Msk                   (0x1U << CAN_F23DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F23DATA1_FD17                       CAN_F23DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F23DATA1_FD18_Pos                   (18U)                             
#define CAN_F23DATA1_FD18_Msk                   (0x1U << CAN_F23DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F23DATA1_FD18                       CAN_F23DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F23DATA1_FD19_Pos                   (19U)                             
#define CAN_F23DATA1_FD19_Msk                   (0x1U << CAN_F23DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F23DATA1_FD19                       CAN_F23DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F23DATA1_FD20_Pos                   (20U)                             
#define CAN_F23DATA1_FD20_Msk                   (0x1U << CAN_F23DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F23DATA1_FD20                       CAN_F23DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F23DATA1_FD21_Pos                   (21U)                             
#define CAN_F23DATA1_FD21_Msk                   (0x1U << CAN_F23DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F23DATA1_FD21                       CAN_F23DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F23DATA1_FD22_Pos                   (22U)                             
#define CAN_F23DATA1_FD22_Msk                   (0x1U << CAN_F23DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F23DATA1_FD22                       CAN_F23DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F23DATA1_FD23_Pos                   (23U)                             
#define CAN_F23DATA1_FD23_Msk                   (0x1U << CAN_F23DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F23DATA1_FD23                       CAN_F23DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F23DATA1_FD24_Pos                   (24U)                             
#define CAN_F23DATA1_FD24_Msk                   (0x1U << CAN_F23DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F23DATA1_FD24                       CAN_F23DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F23DATA1_FD25_Pos                   (25U)                             
#define CAN_F23DATA1_FD25_Msk                   (0x1U << CAN_F23DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F23DATA1_FD25                       CAN_F23DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F23DATA1_FD26_Pos                   (26U)                             
#define CAN_F23DATA1_FD26_Msk                   (0x1U << CAN_F23DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F23DATA1_FD26                       CAN_F23DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F23DATA1_FD27_Pos                   (27U)                             
#define CAN_F23DATA1_FD27_Msk                   (0x1U << CAN_F23DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F23DATA1_FD27                       CAN_F23DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F23DATA1_FD28_Pos                   (28U)                             
#define CAN_F23DATA1_FD28_Msk                   (0x1U << CAN_F23DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F23DATA1_FD28                       CAN_F23DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F23DATA1_FD29_Pos                   (29U)                             
#define CAN_F23DATA1_FD29_Msk                   (0x1U << CAN_F23DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F23DATA1_FD29                       CAN_F23DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F23DATA1_FD30_Pos                   (30U)                             
#define CAN_F23DATA1_FD30_Msk                   (0x1U << CAN_F23DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F23DATA1_FD30                       CAN_F23DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F23DATA1_FD31_Pos                   (31U)                             
#define CAN_F23DATA1_FD31_Msk                   (0x1U << CAN_F23DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F23DATA1_FD31                       CAN_F23DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F24R2 register  ******************/
#define CAN_F24DATA1_FD0_Pos                    (0U)                              
#define CAN_F24DATA1_FD0_Msk                    (0x1U << CAN_F24DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F24DATA1_FD0                        CAN_F24DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F24DATA1_FD1_Pos                    (1U)                              
#define CAN_F24DATA1_FD1_Msk                    (0x1U << CAN_F24DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F24DATA1_FD1                        CAN_F24DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F24DATA1_FD2_Pos                    (2U)                              
#define CAN_F24DATA1_FD2_Msk                    (0x1U << CAN_F24DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F24DATA1_FD2                        CAN_F24DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F24DATA1_FD3_Pos                    (3U)                              
#define CAN_F24DATA1_FD3_Msk                    (0x1U << CAN_F24DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F24DATA1_FD3                        CAN_F24DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F24DATA1_FD4_Pos                    (4U)                              
#define CAN_F24DATA1_FD4_Msk                    (0x1U << CAN_F24DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F24DATA1_FD4                        CAN_F24DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F24DATA1_FD5_Pos                    (5U)                              
#define CAN_F24DATA1_FD5_Msk                    (0x1U << CAN_F24DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F24DATA1_FD5                        CAN_F24DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F24DATA1_FD6_Pos                    (6U)                              
#define CAN_F24DATA1_FD6_Msk                    (0x1U << CAN_F24DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F24DATA1_FD6                        CAN_F24DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F24DATA1_FD7_Pos                    (7U)                              
#define CAN_F24DATA1_FD7_Msk                    (0x1U << CAN_F24DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F24DATA1_FD7                        CAN_F24DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F24DATA1_FD8_Pos                    (8U)                              
#define CAN_F24DATA1_FD8_Msk                    (0x1U << CAN_F24DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F24DATA1_FD8                        CAN_F24DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F24DATA1_FD9_Pos                    (9U)                              
#define CAN_F24DATA1_FD9_Msk                    (0x1U << CAN_F24DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F24DATA1_FD9                        CAN_F24DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F24DATA1_FD10_Pos                   (10U)                             
#define CAN_F24DATA1_FD10_Msk                   (0x1U << CAN_F24DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F24DATA1_FD10                       CAN_F24DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F24DATA1_FD11_Pos                   (11U)                             
#define CAN_F24DATA1_FD11_Msk                   (0x1U << CAN_F24DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F24DATA1_FD11                       CAN_F24DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F24DATA1_FD12_Pos                   (12U)                             
#define CAN_F24DATA1_FD12_Msk                   (0x1U << CAN_F24DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F24DATA1_FD12                       CAN_F24DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F24DATA1_FD13_Pos                   (13U)                             
#define CAN_F24DATA1_FD13_Msk                   (0x1U << CAN_F24DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F24DATA1_FD13                       CAN_F24DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F24DATA1_FD14_Pos                   (14U)                             
#define CAN_F24DATA1_FD14_Msk                   (0x1U << CAN_F24DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F24DATA1_FD14                       CAN_F24DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F24DATA1_FD15_Pos                   (15U)                             
#define CAN_F24DATA1_FD15_Msk                   (0x1U << CAN_F24DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F24DATA1_FD15                       CAN_F24DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F24DATA1_FD16_Pos                   (16U)                             
#define CAN_F24DATA1_FD16_Msk                   (0x1U << CAN_F24DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F24DATA1_FD16                       CAN_F24DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F24DATA1_FD17_Pos                   (17U)                             
#define CAN_F24DATA1_FD17_Msk                   (0x1U << CAN_F24DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F24DATA1_FD17                       CAN_F24DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F24DATA1_FD18_Pos                   (18U)                             
#define CAN_F24DATA1_FD18_Msk                   (0x1U << CAN_F24DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F24DATA1_FD18                       CAN_F24DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F24DATA1_FD19_Pos                   (19U)                             
#define CAN_F24DATA1_FD19_Msk                   (0x1U << CAN_F24DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F24DATA1_FD19                       CAN_F24DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F24DATA1_FD20_Pos                   (20U)                             
#define CAN_F24DATA1_FD20_Msk                   (0x1U << CAN_F24DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F24DATA1_FD20                       CAN_F24DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F24DATA1_FD21_Pos                   (21U)                             
#define CAN_F24DATA1_FD21_Msk                   (0x1U << CAN_F24DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F24DATA1_FD21                       CAN_F24DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F24DATA1_FD22_Pos                   (22U)                             
#define CAN_F24DATA1_FD22_Msk                   (0x1U << CAN_F24DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F24DATA1_FD22                       CAN_F24DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F24DATA1_FD23_Pos                   (23U)                             
#define CAN_F24DATA1_FD23_Msk                   (0x1U << CAN_F24DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F24DATA1_FD23                       CAN_F24DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F24DATA1_FD24_Pos                   (24U)                             
#define CAN_F24DATA1_FD24_Msk                   (0x1U << CAN_F24DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F24DATA1_FD24                       CAN_F24DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F24DATA1_FD25_Pos                   (25U)                             
#define CAN_F24DATA1_FD25_Msk                   (0x1U << CAN_F24DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F24DATA1_FD25                       CAN_F24DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F24DATA1_FD26_Pos                   (26U)                             
#define CAN_F24DATA1_FD26_Msk                   (0x1U << CAN_F24DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F24DATA1_FD26                       CAN_F24DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F24DATA1_FD27_Pos                   (27U)                             
#define CAN_F24DATA1_FD27_Msk                   (0x1U << CAN_F24DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F24DATA1_FD27                       CAN_F24DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F24DATA1_FD28_Pos                   (28U)                             
#define CAN_F24DATA1_FD28_Msk                   (0x1U << CAN_F24DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F24DATA1_FD28                       CAN_F24DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F24DATA1_FD29_Pos                   (29U)                             
#define CAN_F24DATA1_FD29_Msk                   (0x1U << CAN_F24DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F24DATA1_FD29                       CAN_F24DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F24DATA1_FD30_Pos                   (30U)                             
#define CAN_F24DATA1_FD30_Msk                   (0x1U << CAN_F24DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F24DATA1_FD30                       CAN_F24DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F24DATA1_FD31_Pos                   (31U)                             
#define CAN_F24DATA1_FD31_Msk                   (0x1U << CAN_F24DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F24DATA1_FD31                       CAN_F24DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F25R2 register  ******************/
#define CAN_F25DATA1_FD0_Pos                    (0U)                              
#define CAN_F25DATA1_FD0_Msk                    (0x1U << CAN_F25DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F25DATA1_FD0                        CAN_F25DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F25DATA1_FD1_Pos                    (1U)                              
#define CAN_F25DATA1_FD1_Msk                    (0x1U << CAN_F25DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F25DATA1_FD1                        CAN_F25DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F25DATA1_FD2_Pos                    (2U)                              
#define CAN_F25DATA1_FD2_Msk                    (0x1U << CAN_F25DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F25DATA1_FD2                        CAN_F25DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F25DATA1_FD3_Pos                    (3U)                              
#define CAN_F25DATA1_FD3_Msk                    (0x1U << CAN_F25DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F25DATA1_FD3                        CAN_F25DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F25DATA1_FD4_Pos                    (4U)                              
#define CAN_F25DATA1_FD4_Msk                    (0x1U << CAN_F25DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F25DATA1_FD4                        CAN_F25DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F25DATA1_FD5_Pos                    (5U)                              
#define CAN_F25DATA1_FD5_Msk                    (0x1U << CAN_F25DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F25DATA1_FD5                        CAN_F25DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F25DATA1_FD6_Pos                    (6U)                              
#define CAN_F25DATA1_FD6_Msk                    (0x1U << CAN_F25DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F25DATA1_FD6                        CAN_F25DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F25DATA1_FD7_Pos                    (7U)                              
#define CAN_F25DATA1_FD7_Msk                    (0x1U << CAN_F25DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F25DATA1_FD7                        CAN_F25DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F25DATA1_FD8_Pos                    (8U)                              
#define CAN_F25DATA1_FD8_Msk                    (0x1U << CAN_F25DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F25DATA1_FD8                        CAN_F25DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F25DATA1_FD9_Pos                    (9U)                              
#define CAN_F25DATA1_FD9_Msk                    (0x1U << CAN_F25DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F25DATA1_FD9                        CAN_F25DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F25DATA1_FD10_Pos                   (10U)                             
#define CAN_F25DATA1_FD10_Msk                   (0x1U << CAN_F25DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F25DATA1_FD10                       CAN_F25DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F25DATA1_FD11_Pos                   (11U)                             
#define CAN_F25DATA1_FD11_Msk                   (0x1U << CAN_F25DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F25DATA1_FD11                       CAN_F25DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F25DATA1_FD12_Pos                   (12U)                             
#define CAN_F25DATA1_FD12_Msk                   (0x1U << CAN_F25DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F25DATA1_FD12                       CAN_F25DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F25DATA1_FD13_Pos                   (13U)                             
#define CAN_F25DATA1_FD13_Msk                   (0x1U << CAN_F25DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F25DATA1_FD13                       CAN_F25DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F25DATA1_FD14_Pos                   (14U)                             
#define CAN_F25DATA1_FD14_Msk                   (0x1U << CAN_F25DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F25DATA1_FD14                       CAN_F25DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F25DATA1_FD15_Pos                   (15U)                             
#define CAN_F25DATA1_FD15_Msk                   (0x1U << CAN_F25DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F25DATA1_FD15                       CAN_F25DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F25DATA1_FD16_Pos                   (16U)                             
#define CAN_F25DATA1_FD16_Msk                   (0x1U << CAN_F25DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F25DATA1_FD16                       CAN_F25DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F25DATA1_FD17_Pos                   (17U)                             
#define CAN_F25DATA1_FD17_Msk                   (0x1U << CAN_F25DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F25DATA1_FD17                       CAN_F25DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F25DATA1_FD18_Pos                   (18U)                             
#define CAN_F25DATA1_FD18_Msk                   (0x1U << CAN_F25DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F25DATA1_FD18                       CAN_F25DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F25DATA1_FD19_Pos                   (19U)                             
#define CAN_F25DATA1_FD19_Msk                   (0x1U << CAN_F25DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F25DATA1_FD19                       CAN_F25DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F25DATA1_FD20_Pos                   (20U)                             
#define CAN_F25DATA1_FD20_Msk                   (0x1U << CAN_F25DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F25DATA1_FD20                       CAN_F25DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F25DATA1_FD21_Pos                   (21U)                             
#define CAN_F25DATA1_FD21_Msk                   (0x1U << CAN_F25DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F25DATA1_FD21                       CAN_F25DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F25DATA1_FD22_Pos                   (22U)                             
#define CAN_F25DATA1_FD22_Msk                   (0x1U << CAN_F25DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F25DATA1_FD22                       CAN_F25DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F25DATA1_FD23_Pos                   (23U)                             
#define CAN_F25DATA1_FD23_Msk                   (0x1U << CAN_F25DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F25DATA1_FD23                       CAN_F25DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F25DATA1_FD24_Pos                   (24U)                             
#define CAN_F25DATA1_FD24_Msk                   (0x1U << CAN_F25DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F25DATA1_FD24                       CAN_F25DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F25DATA1_FD25_Pos                   (25U)                             
#define CAN_F25DATA1_FD25_Msk                   (0x1U << CAN_F25DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F25DATA1_FD25                       CAN_F25DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F25DATA1_FD26_Pos                   (26U)                             
#define CAN_F25DATA1_FD26_Msk                   (0x1U << CAN_F25DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F25DATA1_FD26                       CAN_F25DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F25DATA1_FD27_Pos                   (27U)                             
#define CAN_F25DATA1_FD27_Msk                   (0x1U << CAN_F25DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F25DATA1_FD27                       CAN_F25DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F25DATA1_FD28_Pos                   (28U)                             
#define CAN_F25DATA1_FD28_Msk                   (0x1U << CAN_F25DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F25DATA1_FD28                       CAN_F25DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F25DATA1_FD29_Pos                   (29U)                             
#define CAN_F25DATA1_FD29_Msk                   (0x1U << CAN_F25DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F25DATA1_FD29                       CAN_F25DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F25DATA1_FD30_Pos                   (30U)                             
#define CAN_F25DATA1_FD30_Msk                   (0x1U << CAN_F25DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F25DATA1_FD30                       CAN_F25DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F25DATA1_FD31_Pos                   (31U)                             
#define CAN_F25DATA1_FD31_Msk                   (0x1U << CAN_F25DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F25DATA1_FD31                       CAN_F25DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F26R2 register  ******************/
#define CAN_F26DATA1_FD0_Pos                    (0U)                              
#define CAN_F26DATA1_FD0_Msk                    (0x1U << CAN_F26DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F26DATA1_FD0                        CAN_F26DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F26DATA1_FD1_Pos                    (1U)                              
#define CAN_F26DATA1_FD1_Msk                    (0x1U << CAN_F26DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F26DATA1_FD1                        CAN_F26DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F26DATA1_FD2_Pos                    (2U)                              
#define CAN_F26DATA1_FD2_Msk                    (0x1U << CAN_F26DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F26DATA1_FD2                        CAN_F26DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F26DATA1_FD3_Pos                    (3U)                              
#define CAN_F26DATA1_FD3_Msk                    (0x1U << CAN_F26DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F26DATA1_FD3                        CAN_F26DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F26DATA1_FD4_Pos                    (4U)                              
#define CAN_F26DATA1_FD4_Msk                    (0x1U << CAN_F26DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F26DATA1_FD4                        CAN_F26DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F26DATA1_FD5_Pos                    (5U)                              
#define CAN_F26DATA1_FD5_Msk                    (0x1U << CAN_F26DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F26DATA1_FD5                        CAN_F26DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F26DATA1_FD6_Pos                    (6U)                              
#define CAN_F26DATA1_FD6_Msk                    (0x1U << CAN_F26DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F26DATA1_FD6                        CAN_F26DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F26DATA1_FD7_Pos                    (7U)                              
#define CAN_F26DATA1_FD7_Msk                    (0x1U << CAN_F26DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F26DATA1_FD7                        CAN_F26DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F26DATA1_FD8_Pos                    (8U)                              
#define CAN_F26DATA1_FD8_Msk                    (0x1U << CAN_F26DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F26DATA1_FD8                        CAN_F26DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F26DATA1_FD9_Pos                    (9U)                              
#define CAN_F26DATA1_FD9_Msk                    (0x1U << CAN_F26DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F26DATA1_FD9                        CAN_F26DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F26DATA1_FD10_Pos                   (10U)                             
#define CAN_F26DATA1_FD10_Msk                   (0x1U << CAN_F26DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F26DATA1_FD10                       CAN_F26DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F26DATA1_FD11_Pos                   (11U)                             
#define CAN_F26DATA1_FD11_Msk                   (0x1U << CAN_F26DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F26DATA1_FD11                       CAN_F26DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F26DATA1_FD12_Pos                   (12U)                             
#define CAN_F26DATA1_FD12_Msk                   (0x1U << CAN_F26DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F26DATA1_FD12                       CAN_F26DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F26DATA1_FD13_Pos                   (13U)                             
#define CAN_F26DATA1_FD13_Msk                   (0x1U << CAN_F26DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F26DATA1_FD13                       CAN_F26DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F26DATA1_FD14_Pos                   (14U)                             
#define CAN_F26DATA1_FD14_Msk                   (0x1U << CAN_F26DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F26DATA1_FD14                       CAN_F26DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F26DATA1_FD15_Pos                   (15U)                             
#define CAN_F26DATA1_FD15_Msk                   (0x1U << CAN_F26DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F26DATA1_FD15                       CAN_F26DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F26DATA1_FD16_Pos                   (16U)                             
#define CAN_F26DATA1_FD16_Msk                   (0x1U << CAN_F26DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F26DATA1_FD16                       CAN_F26DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F26DATA1_FD17_Pos                   (17U)                             
#define CAN_F26DATA1_FD17_Msk                   (0x1U << CAN_F26DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F26DATA1_FD17                       CAN_F26DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F26DATA1_FD18_Pos                   (18U)                             
#define CAN_F26DATA1_FD18_Msk                   (0x1U << CAN_F26DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F26DATA1_FD18                       CAN_F26DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F26DATA1_FD19_Pos                   (19U)                             
#define CAN_F26DATA1_FD19_Msk                   (0x1U << CAN_F26DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F26DATA1_FD19                       CAN_F26DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F26DATA1_FD20_Pos                   (20U)                             
#define CAN_F26DATA1_FD20_Msk                   (0x1U << CAN_F26DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F26DATA1_FD20                       CAN_F26DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F26DATA1_FD21_Pos                   (21U)                             
#define CAN_F26DATA1_FD21_Msk                   (0x1U << CAN_F26DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F26DATA1_FD21                       CAN_F26DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F26DATA1_FD22_Pos                   (22U)                             
#define CAN_F26DATA1_FD22_Msk                   (0x1U << CAN_F26DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F26DATA1_FD22                       CAN_F26DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F26DATA1_FD23_Pos                   (23U)                             
#define CAN_F26DATA1_FD23_Msk                   (0x1U << CAN_F26DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F26DATA1_FD23                       CAN_F26DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F26DATA1_FD24_Pos                   (24U)                             
#define CAN_F26DATA1_FD24_Msk                   (0x1U << CAN_F26DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F26DATA1_FD24                       CAN_F26DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F26DATA1_FD25_Pos                   (25U)                             
#define CAN_F26DATA1_FD25_Msk                   (0x1U << CAN_F26DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F26DATA1_FD25                       CAN_F26DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F26DATA1_FD26_Pos                   (26U)                             
#define CAN_F26DATA1_FD26_Msk                   (0x1U << CAN_F26DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F26DATA1_FD26                       CAN_F26DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F26DATA1_FD27_Pos                   (27U)                             
#define CAN_F26DATA1_FD27_Msk                   (0x1U << CAN_F26DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F26DATA1_FD27                       CAN_F26DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F26DATA1_FD28_Pos                   (28U)                             
#define CAN_F26DATA1_FD28_Msk                   (0x1U << CAN_F26DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F26DATA1_FD28                       CAN_F26DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F26DATA1_FD29_Pos                   (29U)                             
#define CAN_F26DATA1_FD29_Msk                   (0x1U << CAN_F26DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F26DATA1_FD29                       CAN_F26DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F26DATA1_FD30_Pos                   (30U)                             
#define CAN_F26DATA1_FD30_Msk                   (0x1U << CAN_F26DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F26DATA1_FD30                       CAN_F26DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F26DATA1_FD31_Pos                   (31U)                             
#define CAN_F26DATA1_FD31_Msk                   (0x1U << CAN_F26DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F26DATA1_FD31                       CAN_F26DATA1_FD31_Msk                /*!< Filter bit 31 */

/*******************  Bit definition for CAN_F27R2 register  ******************/
#define CAN_F27DATA1_FD0_Pos                    (0U)                              
#define CAN_F27DATA1_FD0_Msk                    (0x1U << CAN_F27DATA1_FD0_Pos)       /*!< 0x00000001 */
#define CAN_F27DATA1_FD0                        CAN_F27DATA1_FD0_Msk                 /*!< Filter bit 0 */
#define CAN_F27DATA1_FD1_Pos                    (1U)                              
#define CAN_F27DATA1_FD1_Msk                    (0x1U << CAN_F27DATA1_FD1_Pos)       /*!< 0x00000002 */
#define CAN_F27DATA1_FD1                        CAN_F27DATA1_FD1_Msk                 /*!< Filter bit 1 */
#define CAN_F27DATA1_FD2_Pos                    (2U)                              
#define CAN_F27DATA1_FD2_Msk                    (0x1U << CAN_F27DATA1_FD2_Pos)       /*!< 0x00000004 */
#define CAN_F27DATA1_FD2                        CAN_F27DATA1_FD2_Msk                 /*!< Filter bit 2 */
#define CAN_F27DATA1_FD3_Pos                    (3U)                              
#define CAN_F27DATA1_FD3_Msk                    (0x1U << CAN_F27DATA1_FD3_Pos)       /*!< 0x00000008 */
#define CAN_F27DATA1_FD3                        CAN_F27DATA1_FD3_Msk                 /*!< Filter bit 3 */
#define CAN_F27DATA1_FD4_Pos                    (4U)                              
#define CAN_F27DATA1_FD4_Msk                    (0x1U << CAN_F27DATA1_FD4_Pos)       /*!< 0x00000010 */
#define CAN_F27DATA1_FD4                        CAN_F27DATA1_FD4_Msk                 /*!< Filter bit 4 */
#define CAN_F27DATA1_FD5_Pos                    (5U)                              
#define CAN_F27DATA1_FD5_Msk                    (0x1U << CAN_F27DATA1_FD5_Pos)       /*!< 0x00000020 */
#define CAN_F27DATA1_FD5                        CAN_F27DATA1_FD5_Msk                 /*!< Filter bit 5 */
#define CAN_F27DATA1_FD6_Pos                    (6U)                              
#define CAN_F27DATA1_FD6_Msk                    (0x1U << CAN_F27DATA1_FD6_Pos)       /*!< 0x00000040 */
#define CAN_F27DATA1_FD6                        CAN_F27DATA1_FD6_Msk                 /*!< Filter bit 6 */
#define CAN_F27DATA1_FD7_Pos                    (7U)                              
#define CAN_F27DATA1_FD7_Msk                    (0x1U << CAN_F27DATA1_FD7_Pos)       /*!< 0x00000080 */
#define CAN_F27DATA1_FD7                        CAN_F27DATA1_FD7_Msk                 /*!< Filter bit 7 */
#define CAN_F27DATA1_FD8_Pos                    (8U)                              
#define CAN_F27DATA1_FD8_Msk                    (0x1U << CAN_F27DATA1_FD8_Pos)       /*!< 0x00000100 */
#define CAN_F27DATA1_FD8                        CAN_F27DATA1_FD8_Msk                 /*!< Filter bit 8 */
#define CAN_F27DATA1_FD9_Pos                    (9U)                              
#define CAN_F27DATA1_FD9_Msk                    (0x1U << CAN_F27DATA1_FD9_Pos)       /*!< 0x00000200 */
#define CAN_F27DATA1_FD9                        CAN_F27DATA1_FD9_Msk                 /*!< Filter bit 9 */
#define CAN_F27DATA1_FD10_Pos                   (10U)                             
#define CAN_F27DATA1_FD10_Msk                   (0x1U << CAN_F27DATA1_FD10_Pos)      /*!< 0x00000400 */
#define CAN_F27DATA1_FD10                       CAN_F27DATA1_FD10_Msk                /*!< Filter bit 10 */
#define CAN_F27DATA1_FD11_Pos                   (11U)                             
#define CAN_F27DATA1_FD11_Msk                   (0x1U << CAN_F27DATA1_FD11_Pos)      /*!< 0x00000800 */
#define CAN_F27DATA1_FD11                       CAN_F27DATA1_FD11_Msk                /*!< Filter bit 11 */
#define CAN_F27DATA1_FD12_Pos                   (12U)                             
#define CAN_F27DATA1_FD12_Msk                   (0x1U << CAN_F27DATA1_FD12_Pos)      /*!< 0x00001000 */
#define CAN_F27DATA1_FD12                       CAN_F27DATA1_FD12_Msk                /*!< Filter bit 12 */
#define CAN_F27DATA1_FD13_Pos                   (13U)                             
#define CAN_F27DATA1_FD13_Msk                   (0x1U << CAN_F27DATA1_FD13_Pos)      /*!< 0x00002000 */
#define CAN_F27DATA1_FD13                       CAN_F27DATA1_FD13_Msk                /*!< Filter bit 13 */
#define CAN_F27DATA1_FD14_Pos                   (14U)                             
#define CAN_F27DATA1_FD14_Msk                   (0x1U << CAN_F27DATA1_FD14_Pos)      /*!< 0x00004000 */
#define CAN_F27DATA1_FD14                       CAN_F27DATA1_FD14_Msk                /*!< Filter bit 14 */
#define CAN_F27DATA1_FD15_Pos                   (15U)                             
#define CAN_F27DATA1_FD15_Msk                   (0x1U << CAN_F27DATA1_FD15_Pos)      /*!< 0x00008000 */
#define CAN_F27DATA1_FD15                       CAN_F27DATA1_FD15_Msk                /*!< Filter bit 15 */
#define CAN_F27DATA1_FD16_Pos                   (16U)                             
#define CAN_F27DATA1_FD16_Msk                   (0x1U << CAN_F27DATA1_FD16_Pos)      /*!< 0x00010000 */
#define CAN_F27DATA1_FD16                       CAN_F27DATA1_FD16_Msk                /*!< Filter bit 16 */
#define CAN_F27DATA1_FD17_Pos                   (17U)                             
#define CAN_F27DATA1_FD17_Msk                   (0x1U << CAN_F27DATA1_FD17_Pos)      /*!< 0x00020000 */
#define CAN_F27DATA1_FD17                       CAN_F27DATA1_FD17_Msk                /*!< Filter bit 17 */
#define CAN_F27DATA1_FD18_Pos                   (18U)                             
#define CAN_F27DATA1_FD18_Msk                   (0x1U << CAN_F27DATA1_FD18_Pos)      /*!< 0x00040000 */
#define CAN_F27DATA1_FD18                       CAN_F27DATA1_FD18_Msk                /*!< Filter bit 18 */
#define CAN_F27DATA1_FD19_Pos                   (19U)                             
#define CAN_F27DATA1_FD19_Msk                   (0x1U << CAN_F27DATA1_FD19_Pos)      /*!< 0x00080000 */
#define CAN_F27DATA1_FD19                       CAN_F27DATA1_FD19_Msk                /*!< Filter bit 19 */
#define CAN_F27DATA1_FD20_Pos                   (20U)                             
#define CAN_F27DATA1_FD20_Msk                   (0x1U << CAN_F27DATA1_FD20_Pos)      /*!< 0x00100000 */
#define CAN_F27DATA1_FD20                       CAN_F27DATA1_FD20_Msk                /*!< Filter bit 20 */
#define CAN_F27DATA1_FD21_Pos                   (21U)                             
#define CAN_F27DATA1_FD21_Msk                   (0x1U << CAN_F27DATA1_FD21_Pos)      /*!< 0x00200000 */
#define CAN_F27DATA1_FD21                       CAN_F27DATA1_FD21_Msk                /*!< Filter bit 21 */
#define CAN_F27DATA1_FD22_Pos                   (22U)                             
#define CAN_F27DATA1_FD22_Msk                   (0x1U << CAN_F27DATA1_FD22_Pos)      /*!< 0x00400000 */
#define CAN_F27DATA1_FD22                       CAN_F27DATA1_FD22_Msk                /*!< Filter bit 22 */
#define CAN_F27DATA1_FD23_Pos                   (23U)                             
#define CAN_F27DATA1_FD23_Msk                   (0x1U << CAN_F27DATA1_FD23_Pos)      /*!< 0x00800000 */
#define CAN_F27DATA1_FD23                       CAN_F27DATA1_FD23_Msk                /*!< Filter bit 23 */
#define CAN_F27DATA1_FD24_Pos                   (24U)                             
#define CAN_F27DATA1_FD24_Msk                   (0x1U << CAN_F27DATA1_FD24_Pos)      /*!< 0x01000000 */
#define CAN_F27DATA1_FD24                       CAN_F27DATA1_FD24_Msk                /*!< Filter bit 24 */
#define CAN_F27DATA1_FD25_Pos                   (25U)                             
#define CAN_F27DATA1_FD25_Msk                   (0x1U << CAN_F27DATA1_FD25_Pos)      /*!< 0x02000000 */
#define CAN_F27DATA1_FD25                       CAN_F27DATA1_FD25_Msk                /*!< Filter bit 25 */
#define CAN_F27DATA1_FD26_Pos                   (26U)                             
#define CAN_F27DATA1_FD26_Msk                   (0x1U << CAN_F27DATA1_FD26_Pos)      /*!< 0x04000000 */
#define CAN_F27DATA1_FD26                       CAN_F27DATA1_FD26_Msk                /*!< Filter bit 26 */
#define CAN_F27DATA1_FD27_Pos                   (27U)                             
#define CAN_F27DATA1_FD27_Msk                   (0x1U << CAN_F27DATA1_FD27_Pos)      /*!< 0x08000000 */
#define CAN_F27DATA1_FD27                       CAN_F27DATA1_FD27_Msk                /*!< Filter bit 27 */
#define CAN_F27DATA1_FD28_Pos                   (28U)                             
#define CAN_F27DATA1_FD28_Msk                   (0x1U << CAN_F27DATA1_FD28_Pos)      /*!< 0x10000000 */
#define CAN_F27DATA1_FD28                       CAN_F27DATA1_FD28_Msk                /*!< Filter bit 28 */
#define CAN_F27DATA1_FD29_Pos                   (29U)                             
#define CAN_F27DATA1_FD29_Msk                   (0x1U << CAN_F27DATA1_FD29_Pos)      /*!< 0x20000000 */
#define CAN_F27DATA1_FD29                       CAN_F27DATA1_FD29_Msk                /*!< Filter bit 29 */
#define CAN_F27DATA1_FD30_Pos                   (30U)                             
#define CAN_F27DATA1_FD30_Msk                   (0x1U << CAN_F27DATA1_FD30_Pos)      /*!< 0x40000000 */
#define CAN_F27DATA1_FD30                       CAN_F27DATA1_FD30_Msk                /*!< Filter bit 30 */
#define CAN_F27DATA1_FD31_Pos                   (31U)                             
#define CAN_F27DATA1_FD31_Msk                   (0x1U << CAN_F27DATA1_FD31_Pos)      /*!< 0x80000000 */
#define CAN_F27DATA1_FD31                       CAN_F27DATA1_FD31_Msk                /*!< Filter bit 31 */

/******************************************************************************/
/*                                                                            */
/*                        Serial Peripheral Interface                         */
/*                                                                            */
/******************************************************************************/
/*
 * @brief Specific device feature definitions (not present on all devices in the GD32VF103 serie)
 */
#define SPI_I2S_SUPPORT       /*!< I2S support */
#define I2S2_I2S3_CLOCK_FEATURE
/*******************  Bit definition for SPI_CTL0 register  ********************/
#define SPI_CTL0_CKPH_Pos                    (0U)                               
#define SPI_CTL0_CKPH_Msk                    (0x1U << SPI_CTL0_CKPH_Pos)         /*!< 0x00000001 */
#define SPI_CTL0_CKPH                        SPI_CTL0_CKPH_Msk                   /*!< Clock Phase */
#define SPI_CTL0_CKPL_Pos                    (1U)                               
#define SPI_CTL0_CKPL_Msk                    (0x1U << SPI_CTL0_CKPL_Pos)         /*!< 0x00000002 */
#define SPI_CTL0_CKPL                        SPI_CTL0_CKPL_Msk                   /*!< Clock Polarity */
#define SPI_CTL0_MSTMOD_Pos                    (2U)                               
#define SPI_CTL0_MSTMOD_Msk                    (0x1U << SPI_CTL0_MSTMOD_Pos)         /*!< 0x00000004 */
#define SPI_CTL0_MSTMOD                        SPI_CTL0_MSTMOD_Msk                   /*!< Master Selection */

#define SPI_CTL0_PSC_Pos                      (3U)                               
#define SPI_CTL0_PSC_Msk                      (0x7U << SPI_CTL0_PSC_Pos)           /*!< 0x00000038 */
#define SPI_CTL0_PSC                          SPI_CTL0_PSC_Msk                     /*!< BR[2:0] bits (Baud Rate Control) */
#define SPI_CTL0_PSC_0                        (0x1U << SPI_CTL0_PSC_Pos)           /*!< 0x00000008 */
#define SPI_CTL0_PSC_1                        (0x2U << SPI_CTL0_PSC_Pos)           /*!< 0x00000010 */
#define SPI_CTL0_PSC_2                        (0x4U << SPI_CTL0_PSC_Pos)           /*!< 0x00000020 */

#define SPI_CTL0_SPIEN_Pos                     (6U)                               
#define SPI_CTL0_SPIEN_Msk                     (0x1U << SPI_CTL0_SPIEN_Pos)          /*!< 0x00000040 */
#define SPI_CTL0_SPIEN                         SPI_CTL0_SPIEN_Msk                    /*!< SPI Enable */
#define SPI_CTL0_LF_Pos                (7U)                               
#define SPI_CTL0_LF_Msk                (0x1U << SPI_CTL0_LF_Pos)     /*!< 0x00000080 */
#define SPI_CTL0_LF                    SPI_CTL0_LF_Msk               /*!< Frame Format */
#define SPI_CTL0_SWNSS_Pos                     (8U)                               
#define SPI_CTL0_SWNSS_Msk                     (0x1U << SPI_CTL0_SWNSS_Pos)          /*!< 0x00000100 */
#define SPI_CTL0_SWNSS                         SPI_CTL0_SWNSS_Msk                    /*!< Internal slave select */
#define SPI_CTL0_SWNSSEN_Pos                     (9U)                               
#define SPI_CTL0_SWNSSEN_Msk                     (0x1U << SPI_CTL0_SWNSSEN_Pos)          /*!< 0x00000200 */
#define SPI_CTL0_SWNSSEN                         SPI_CTL0_SWNSSEN_Msk                    /*!< Software slave management */
#define SPI_CTL0_RO_Pos                  (10U)                              
#define SPI_CTL0_RO_Msk                  (0x1U << SPI_CTL0_RO_Pos)       /*!< 0x00000400 */
#define SPI_CTL0_RO                      SPI_CTL0_RO_Msk                 /*!< Receive only */
#define SPI_CTL0_FF16_Pos                     (11U)                              
#define SPI_CTL0_FF16_Msk                     (0x1U << SPI_CTL0_FF16_Pos)          /*!< 0x00000800 */
#define SPI_CTL0_FF16                         SPI_CTL0_FF16_Msk                    /*!< Data Frame Format */
#define SPI_CTL0_CRCNT_Pos                 (12U)                              
#define SPI_CTL0_CRCNT_Msk                 (0x1U << SPI_CTL0_CRCNT_Pos)      /*!< 0x00001000 */
#define SPI_CTL0_CRCNT                     SPI_CTL0_CRCNT_Msk                /*!< Transmit CRC next */
#define SPI_CTL0_CRCEN_Pos                   (13U)                              
#define SPI_CTL0_CRCEN_Msk                   (0x1U << SPI_CTL0_CRCEN_Pos)        /*!< 0x00002000 */
#define SPI_CTL0_CRCEN                       SPI_CTL0_CRCEN_Msk                  /*!< Hardware CRC calculation enable */
#define SPI_CTL0_BDOEN_Pos                  (14U)                              
#define SPI_CTL0_BDOEN_Msk                  (0x1U << SPI_CTL0_BDOEN_Pos)       /*!< 0x00004000 */
#define SPI_CTL0_BDOEN                      SPI_CTL0_BDOEN_Msk                 /*!< Output enable in bidirectional mode */
#define SPI_CTL0_BDEN_Pos                (15U)                              
#define SPI_CTL0_BDEN_Msk                (0x1U << SPI_CTL0_BDEN_Pos)     /*!< 0x00008000 */
#define SPI_CTL0_BDEN                    SPI_CTL0_BDEN_Msk               /*!< Bidirectional data mode enable */

/*******************  Bit definition for SPI_CTL1 register  ********************/
#define SPI_CTL1_DMAREN_Pos                  (0U)                               
#define SPI_CTL1_DMAREN_Msk                  (0x1U << SPI_CTL1_DMAREN_Pos)      /*!< 0x00000001 */
#define SPI_CTL1_DMAREN                      SPI_CTL1_DMAREN_Msk                /*!< Rx Buffer DMA Enable */
#define SPI_CTL1_DMATEN_Pos                  (1U)                               
#define SPI_CTL1_DMATEN_Msk                  (0x1U << SPI_CTL1_DMATEN_Pos)      /*!< 0x00000002 */
#define SPI_CTL1_DMATEN                      SPI_CTL1_DMATEN_Msk                /*!< Tx Buffer DMA Enable */
#define SPI_CTL1_NSSDRV_Pos                  (2U)                               
#define SPI_CTL1_NSSDRV_Msk                  (0x1U << SPI_CTL1_NSSDRV_Pos)       /*!< 0x00000004 */
#define SPI_CTL1_NSSDRV                      SPI_CTL1_NSSDRV_Msk                 /*!< SS Output Enable */
#define SPI_CTL1_NSSP_Pos                    (3U)                               
#define SPI_CTL1_NSSP_Msk                    (0x1U << SPI_CTL1_NSSP_Pos)         /*!< 0x00000004 */
#define SPI_CTL1_NSSP                        SPI_CTL1_NSSP_Msk                   /*!< SS Output Enable */
#define SPI_CTL1_TMOD_Pos                    (4U)                               
#define SPI_CTL1_TMOD_Msk                    (0x1U << SPI_CTL1_TMOD_Pos)         /*!< 0x00000004 */
#define SPI_CTL1_TMOD                        SPI_CTL1_TMOD_Msk                   /*!< SS Output Enable */
#define SPI_CTL1_ERRIE_Pos                   (5U)                               
#define SPI_CTL1_ERRIE_Msk                   (0x1U << SPI_CTL1_ERRIE_Pos)        /*!< 0x00000020 */
#define SPI_CTL1_ERRIE                       SPI_CTL1_ERRIE_Msk                  /*!< Error Interrupt Enable */
#define SPI_CTL1_RBNEIE_Pos                  (6U)                               
#define SPI_CTL1_RBNEIE_Msk                  (0x1U << SPI_CTL1_RBNEIE_Pos)       /*!< 0x00000040 */
#define SPI_CTL1_RBNEIE                      SPI_CTL1_RBNEIE_Msk                 /*!< RX buffer Not Empty Interrupt Enable */
#define SPI_CTL1_TBEIE_Pos                   (7U)                               
#define SPI_CTL1_TBEIE_Msk                   (0x1U << SPI_CTL1_TBEIE_Pos)        /*!< 0x00000080 */
#define SPI_CTL1_TBEIE                       SPI_CTL1_TBEIE_Msk                  /*!< Tx buffer Empty Interrupt Enable */

/********************  Bit definition for SPI_STAT register  ********************/
#define SPI_STAT_RBNE_Pos                     (0U)                               
#define SPI_STAT_RBNE_Msk                     (0x1U << SPI_STAT_RBNE_Pos)          /*!< 0x00000001 */
#define SPI_STAT_RBNE                         SPI_STAT_RBNE_Msk                    /*!< Receive buffer Not Empty */
#define SPI_STAT_TBE_Pos                      (1U)                               
#define SPI_STAT_TBE_Msk                      (0x1U << SPI_STAT_TBE_Pos)           /*!< 0x00000002 */
#define SPI_STAT_TBE                          SPI_STAT_TBE_Msk                     /*!< Transmit buffer Empty */
#define SPI_STAT_I2SCH_Pos                   (2U)                               
#define SPI_STAT_I2SCH_Msk                   (0x1U << SPI_STAT_I2SCH_Pos)        /*!< 0x00000004 */
#define SPI_STAT_I2SCH                       SPI_STAT_I2SCH_Msk                  /*!< Channel side */
#define SPI_STAT_TXURERR_Pos                      (3U)                               
#define SPI_STAT_TXURERR_Msk                      (0x1U << SPI_STAT_TXURERR_Pos)           /*!< 0x00000008 */
#define SPI_STAT_TXURERR                          SPI_STAT_TXURERR_Msk                     /*!< Underrun flag */
#define SPI_STAT_CRCERR_Pos                   (4U)                               
#define SPI_STAT_CRCERR_Msk                   (0x1U << SPI_STAT_CRCERR_Pos)        /*!< 0x00000010 */
#define SPI_STAT_CRCERR                       SPI_STAT_CRCERR_Msk                  /*!< CRC Error flag */
#define SPI_STAT_CONFERR_Pos                     (5U)                               
#define SPI_STAT_CONFERR_Msk                     (0x1U << SPI_STAT_CONFERR_Pos)          /*!< 0x00000020 */
#define SPI_STAT_CONFERR                         SPI_STAT_CONFERR_Msk                    /*!< Mode fault */
#define SPI_STAT_RXORERR_Pos                      (6U)                               
#define SPI_STAT_RXORERR_Msk                      (0x1U << SPI_STAT_RXORERR_Pos)           /*!< 0x00000040 */
#define SPI_STAT_RXORERR                          SPI_STAT_RXORERR_Msk                     /*!< Overrun flag */
#define SPI_STAT_TRANS_Pos                      (7U)                               
#define SPI_STAT_TRANS_Msk                      (0x1U << SPI_STAT_TRANS_Pos)           /*!< 0x00000080 */
#define SPI_STAT_TRANS                          SPI_STAT_TRANS_Msk                     /*!< Busy flag */
#define SPI_STAT_FERR_Pos                      (8U)                               
#define SPI_STAT_FERR_Msk                      (0x1U << SPI_STAT_FERR_Pos)           /*!< 0x00000080 */
#define SPI_STAT_FERR                          SPI_STAT_FERR_Msk                     /*!< Busy flag */

/********************  Bit definition for SPI_DR register  ********************/
#define SPI_DATA_SPI_DATA_Pos                       (0U)                               
#define SPI_DATA_SPI_DATA_Msk                       (0xFFFFU << SPI_DATA_SPI_DATA_Pos)         /*!< 0x0000FFFF */
#define SPI_DATA_SPI_DATA                           SPI_DATA_SPI_DATA_Msk                      /*!< Data Register */

/*******************  Bit definition for SPI_CRCPOLY register  ******************/
#define SPI_CRCPOLY_CRCPOLY_Pos               (0U)                               
#define SPI_CRCPOLY_CRCPOLY_Msk               (0xFFFFU << SPI_CRCPOLY_CRCPOLY_Pos) /*!< 0x0000FFFF */
#define SPI_CRCPOLY_CRCPOLY                   SPI_CRCPOLY_CRCPOLY_Msk              /*!< CRC polynomial register */

/******************  Bit definition for SPI_RXCRCR register  ******************/
#define SPI_RCRC_RCRC_Pos                (0U)                               
#define SPI_RCRC_RCRC_Msk                (0xFFFFU << SPI_RCRC_RCRC_Pos)  /*!< 0x0000FFFF */
#define SPI_RCRC_RCRC                    SPI_RCRC_RCRC_Msk               /*!< Rx CRC Register */

/******************  Bit definition for SPI_TXCRCR register  ******************/
#define SPI_TCRC_TCRC_Pos                (0U)                               
#define SPI_TCRC_TCRC_Msk                (0xFFFFU << SPI_TCRC_TCRC_Pos)  /*!< 0x0000FFFF */
#define SPI_TCRC_TCRC                    SPI_TCRC_TCRC_Msk               /*!< Tx CRC Register */

/******************  Bit definition for SPI_I2SCTL register  *****************/
#define SPI_I2SCTL_CHLEN_Pos               (0U)                               
#define SPI_I2SCTL_CHLEN_Msk               (0x1U << SPI_I2SCTL_CHLEN_Pos)    /*!< 0x00000001 */
#define SPI_I2SCTL_CHLEN                   SPI_I2SCTL_CHLEN_Msk              /*!< Channel length (number of bits per audio channel) */

#define SPI_I2SCTL_DTLEN_Pos              (1U)                               
#define SPI_I2SCTL_DTLEN_Msk              (0x3U << SPI_I2SCTL_DTLEN_Pos)   /*!< 0x00000006 */
#define SPI_I2SCTL_DTLEN                  SPI_I2SCTL_DTLEN_Msk             /*!< DATLEN[1:0] bits (Data length to be transferred) */
#define SPI_I2SCTL_DTLEN_0                (0x1U << SPI_I2SCTL_DTLEN_Pos)   /*!< 0x00000002 */
#define SPI_I2SCTL_DTLEN_1                (0x2U << SPI_I2SCTL_DTLEN_Pos)   /*!< 0x00000004 */

#define SPI_I2SCTL_CKPL_Pos               (3U)                               
#define SPI_I2SCTL_CKPL_Msk               (0x1U << SPI_I2SCTL_CKPL_Pos)    /*!< 0x00000008 */
#define SPI_I2SCTL_CKPL                   SPI_I2SCTL_CKPL_Msk              /*!< steady state clock polarity */

#define SPI_I2SCTL_I2SSTD_Pos              (4U)                               
#define SPI_I2SCTL_I2SSTD_Msk              (0x3U << SPI_I2SCTL_I2SSTD_Pos)   /*!< 0x00000030 */
#define SPI_I2SCTL_I2SSTD                  SPI_I2SCTL_I2SSTD_Msk             /*!< I2SSTD[1:0] bits (I2S standard selection) */
#define SPI_I2SCTL_I2SSTD_0                (0x1U << SPI_I2SCTL_I2SSTD_Pos)   /*!< 0x00000010 */
#define SPI_I2SCTL_I2SSTD_1                (0x2U << SPI_I2SCTL_I2SSTD_Pos)   /*!< 0x00000020 */

#define SPI_I2SCTL_PCMSMOD_Pos             (7U)                               
#define SPI_I2SCTL_PCMSMOD_Msk             (0x1U << SPI_I2SCTL_PCMSMOD_Pos)  /*!< 0x00000080 */
#define SPI_I2SCTL_PCMSMOD                 SPI_I2SCTL_PCMSMOD_Msk            /*!< PCM frame synchronization */

#define SPI_I2SCTL_I2SOPMOD_Pos              (8U)                               
#define SPI_I2SCTL_I2SOPMOD_Msk              (0x3U << SPI_I2SCTL_I2SOPMOD_Pos)   /*!< 0x00000300 */
#define SPI_I2SCTL_I2SOPMOD                  SPI_I2SCTL_I2SOPMOD_Msk             /*!< I2SCFG[1:0] bits (I2S configuration mode) */
#define SPI_I2SCTL_I2SOPMOD_0                (0x1U << SPI_I2SCTL_I2SOPMOD_Pos)   /*!< 0x00000100 */
#define SPI_I2SCTL_I2SOPMOD_1                (0x2U << SPI_I2SCTL_I2SOPMOD_Pos)   /*!< 0x00000200 */

#define SPI_I2SCTL_I2SEN_Pos                (10U)                              
#define SPI_I2SCTL_I2SEN_Msk                (0x1U << SPI_I2SCTL_I2SEN_Pos)     /*!< 0x00000400 */
#define SPI_I2SCTL_I2SEN                    SPI_I2SCTL_I2SEN_Msk               /*!< I2S Enable */
#define SPI_I2SCTL_I2SSEL_Pos              (11U)                              
#define SPI_I2SCTL_I2SSEL_Msk              (0x1U << SPI_I2SCTL_I2SSEL_Pos)   /*!< 0x00000800 */
#define SPI_I2SCTL_I2SSEL                  SPI_I2SCTL_I2SSEL_Msk             /*!< I2S mode selection */

/******************  Bit definition for SPI_I2SPSC register  *******************/
#define SPI_I2SPSC_DIV_Pos                (0U)                               
#define SPI_I2SPSC_DIV_Msk                (0xFFU << SPI_I2SPSC_DIV_Pos)    /*!< 0x000000FF */
#define SPI_I2SPSC_DIV                    SPI_I2SPSC_DIV_Msk               /*!< I2S Linear prescaler */
#define SPI_I2SPSC_OF_Pos                   (8U)                               
#define SPI_I2SPSC_OF_Msk                   (0x1U << SPI_I2SPSC_OF_Pos)        /*!< 0x00000100 */
#define SPI_I2SPSC_OF                       SPI_I2SPSC_OF_Msk                  /*!< Odd factor for the prescaler */
#define SPI_I2SPSC_MCKOEN_Pos                 (9U)                               
#define SPI_I2SPSC_MCKOEN_Msk                 (0x1U << SPI_I2SPSC_MCKOEN_Pos)      /*!< 0x00000200 */
#define SPI_I2SPSC_MCKOEN                     SPI_I2SPSC_MCKOEN_Msk                /*!< Master Clock Output Enable */

/******************************************************************************/
/*                                                                            */
/*                      Inter-integrated Circuit Interface                    */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for I2C_CTL0 register  ********************/
#define I2C_CTL0_I2CEN_Pos                      (0U)                               
#define I2C_CTL0_I2CEN_Msk                      (0x1U << I2C_CTL0_I2CEN_Pos)           /*!< 0x00000001 */
#define I2C_CTL0_I2CEN                          I2C_CTL0_I2CEN_Msk                     /*!< Peripheral Enable */
#define I2C_CTL0_SMBEN_Pos                   (1U)                               
#define I2C_CTL0_SMBEN_Msk                   (0x1U << I2C_CTL0_SMBEN_Pos)        /*!< 0x00000002 */
#define I2C_CTL0_SMBEN                       I2C_CTL0_SMBEN_Msk                  /*!< SMBus Mode */
#define I2C_CTL0_SMBSEL_Pos                 (3U)                               
#define I2C_CTL0_SMBSEL_Msk                 (0x1U << I2C_CTL0_SMBSEL_Pos)      /*!< 0x00000008 */
#define I2C_CTL0_SMBSEL                     I2C_CTL0_SMBSEL_Msk                /*!< SMBus Type */
#define I2C_CTL0_ARPEN_Pos                   (4U)                               
#define I2C_CTL0_ARPEN_Msk                   (0x1U << I2C_CTL0_ARPEN_Pos)        /*!< 0x00000010 */
#define I2C_CTL0_ARPEN                       I2C_CTL0_ARPEN_Msk                  /*!< ARP Enable */
#define I2C_CTL0_PECTRANSEN_Pos                   (5U)                               
#define I2C_CTL0_PECTRANSEN_Msk                   (0x1U << I2C_CTL0_PECTRANSEN_Pos)        /*!< 0x00000020 */
#define I2C_CTL0_PECTRANSEN                       I2C_CTL0_PECTRANSEN_Msk                  /*!< PEC Enable */
#define I2C_CTL0_GCEN_Pos                    (6U)                               
#define I2C_CTL0_GCEN_Msk                    (0x1U << I2C_CTL0_GCEN_Pos)         /*!< 0x00000040 */
#define I2C_CTL0_GCEN                        I2C_CTL0_GCEN_Msk                   /*!< General Call Enable */
#define I2C_CTL0_SS_Pos               (7U)                               
#define I2C_CTL0_SS_Msk               (0x1U << I2C_CTL0_SS_Pos)    /*!< 0x00000080 */
#define I2C_CTL0_SS                   I2C_CTL0_SS_Msk              /*!< Clock Stretching Disable (Slave mode) */
#define I2C_CTL0_START_Pos                   (8U)                               
#define I2C_CTL0_START_Msk                   (0x1U << I2C_CTL0_START_Pos)        /*!< 0x00000100 */
#define I2C_CTL0_START                       I2C_CTL0_START_Msk                  /*!< Start Generation */
#define I2C_CTL0_STOP_Pos                    (9U)                               
#define I2C_CTL0_STOP_Msk                    (0x1U << I2C_CTL0_STOP_Pos)         /*!< 0x00000200 */
#define I2C_CTL0_STOP                        I2C_CTL0_STOP_Msk                   /*!< Stop Generation */
#define I2C_CTL0_ACKEN_Pos                     (10U)                              
#define I2C_CTL0_ACKEN_Msk                     (0x1U << I2C_CTL0_ACKEN_Pos)          /*!< 0x00000400 */
#define I2C_CTL0_ACKEN                         I2C_CTL0_ACKEN_Msk                    /*!< Acknowledge Enable */
#define I2C_CTL0_POAP_Pos                     (11U)                              
#define I2C_CTL0_POAP_Msk                     (0x1U << I2C_CTL0_POAP_Pos)          /*!< 0x00000800 */
#define I2C_CTL0_POAP                         I2C_CTL0_POAP_Msk                    /*!< Acknowledge/PEC Position (for data reception) */
#define I2C_CTL0_I2CENC_Pos                     (12U)                              
#define I2C_CTL0_I2CENC_Msk                     (0x1U << I2C_CTL0_I2CENC_Pos)          /*!< 0x00001000 */
#define I2C_CTL0_I2CENC                         I2C_CTL0_I2CENC_Msk                    /*!< Packet Error Checking */
#define I2C_CTL0_SALT_Pos                   (13U)                              
#define I2C_CTL0_SALT_Msk                   (0x1U << I2C_CTL0_SALT_Pos)        /*!< 0x00002000 */
#define I2C_CTL0_SALT                       I2C_CTL0_SALT_Msk                  /*!< SMBus Alert */
#define I2C_CTL0_SRESET_Pos                   (15U)                              
#define I2C_CTL0_SRESET_Msk                   (0x1U << I2C_CTL0_SRESET_Pos)        /*!< 0x00008000 */
#define I2C_CTL0_SRESET                       I2C_CTL0_SRESET_Msk                  /*!< Software Reset */

/*******************  Bit definition for I2C_CTL1 register  ********************/
#define I2C_CTL1_I2CCLK_Pos                    (0U)                               
#define I2C_CTL1_I2CCLK_Msk                    (0x3FU << I2C_CTL1_I2CCLK_Pos)        /*!< 0x0000003F */
#define I2C_CTL1_I2CCLK                        I2C_CTL1_I2CCLK_Msk                   /*!< FREQ[5:0] bits (Peripheral Clock Frequency) */
#define I2C_CTL1_I2CCLK_0                      (0x01U << I2C_CTL1_I2CCLK_Pos)        /*!< 0x00000001 */
#define I2C_CTL1_I2CCLK_1                      (0x02U << I2C_CTL1_I2CCLK_Pos)        /*!< 0x00000002 */
#define I2C_CTL1_I2CCLK_2                      (0x04U << I2C_CTL1_I2CCLK_Pos)        /*!< 0x00000004 */
#define I2C_CTL1_I2CCLK_3                      (0x08U << I2C_CTL1_I2CCLK_Pos)        /*!< 0x00000008 */
#define I2C_CTL1_I2CCLK_4                      (0x10U << I2C_CTL1_I2CCLK_Pos)        /*!< 0x00000010 */
#define I2C_CTL1_I2CCLK_5                      (0x20U << I2C_CTL1_I2CCLK_Pos)        /*!< 0x00000020 */

#define I2C_CTL1_ERRIE_Pos                 (8U)                               
#define I2C_CTL1_ERRIE_Msk                 (0x1U << I2C_CTL1_ERRIE_Pos)      /*!< 0x00000100 */
#define I2C_CTL1_ERRIE                     I2C_CTL1_ERRIE_Msk                /*!< Error Interrupt Enable */
#define I2C_CTL1_EVIE_Pos                 (9U)                               
#define I2C_CTL1_EVIE_Msk                 (0x1U << I2C_CTL1_EVIE_Pos)      /*!< 0x00000200 */
#define I2C_CTL1_EVIE                     I2C_CTL1_EVIE_Msk                /*!< Event Interrupt Enable */
#define I2C_CTL1_BUFIE_Pos                 (10U)                              
#define I2C_CTL1_BUFIE_Msk                 (0x1U << I2C_CTL1_BUFIE_Pos)      /*!< 0x00000400 */
#define I2C_CTL1_BUFIE                     I2C_CTL1_BUFIE_Msk                /*!< Buffer Interrupt Enable */
#define I2C_CTL1_DMAON_Pos                   (11U)                              
#define I2C_CTL1_DMAON_Msk                   (0x1U << I2C_CTL1_DMAON_Pos)        /*!< 0x00000800 */
#define I2C_CTL1_DMAON                       I2C_CTL1_DMAON_Msk                  /*!< DMA Requests Enable */
#define I2C_CTL1_DMALST_Pos                    (12U)                              
#define I2C_CTL1_DMALST_Msk                    (0x1U << I2C_CTL1_DMALST_Pos)         /*!< 0x00001000 */
#define I2C_CTL1_DMALST                        I2C_CTL1_DMALST_Msk                   /*!< DMA Last Transfer */

/*******************  Bit definition for I2C_SADDR0 register  *******************/
#define I2C_SADDR0_ADDRESS1_7                     0x000000FEU             /*!< Interface Address */
#define I2C_SADDR0_ADDRESS8_9                     0x00000300U             /*!< Interface Address */

#define I2C_SADDR0_ADDRESS0_Pos                   (0U)                               
#define I2C_SADDR0_ADDRESS0_Msk                   (0x1U << I2C_SADDR0_ADDRESS0_Pos)        /*!< 0x00000001 */
#define I2C_SADDR0_ADDRESS0                       I2C_SADDR0_ADDRESS0_Msk                  /*!< Bit 0 */
#define I2C_SADDR0_ADDRESS1_Pos                   (1U)                               
#define I2C_SADDR0_ADDRESS1_Msk                   (0x1U << I2C_SADDR0_ADDRESS1_Pos)        /*!< 0x00000002 */
#define I2C_SADDR0_ADDRESS1                       I2C_SADDR0_ADDRESS1_Msk                  /*!< Bit 1 */
#define I2C_SADDR0_ADDRESS2_Pos                   (2U)                               
#define I2C_SADDR0_ADDRESS2_Msk                   (0x1U << I2C_SADDR0_ADDRESS2_Pos)        /*!< 0x00000004 */
#define I2C_SADDR0_ADDRESS2                       I2C_SADDR0_ADDRESS2_Msk                  /*!< Bit 2 */
#define I2C_SADDR0_ADDRESS3_Pos                   (3U)                               
#define I2C_SADDR0_ADDRESS3_Msk                   (0x1U << I2C_SADDR0_ADDRESS3_Pos)        /*!< 0x00000008 */
#define I2C_SADDR0_ADDRESS3                       I2C_SADDR0_ADDRESS3_Msk                  /*!< Bit 3 */
#define I2C_SADDR0_ADDRESS4_Pos                   (4U)                               
#define I2C_SADDR0_ADDRESS4_Msk                   (0x1U << I2C_SADDR0_ADDRESS4_Pos)        /*!< 0x00000010 */
#define I2C_SADDR0_ADDRESS4                       I2C_SADDR0_ADDRESS4_Msk                  /*!< Bit 4 */
#define I2C_SADDR0_ADDRESS5_Pos                   (5U)                               
#define I2C_SADDR0_ADDRESS5_Msk                   (0x1U << I2C_SADDR0_ADDRESS5_Pos)        /*!< 0x00000020 */
#define I2C_SADDR0_ADDRESS5                       I2C_SADDR0_ADDRESS5_Msk                  /*!< Bit 5 */
#define I2C_SADDR0_ADDRESS6_Pos                   (6U)                               
#define I2C_SADDR0_ADDRESS6_Msk                   (0x1U << I2C_SADDR0_ADDRESS6_Pos)        /*!< 0x00000040 */
#define I2C_SADDR0_ADDRESS6                       I2C_SADDR0_ADDRESS6_Msk                  /*!< Bit 6 */
#define I2C_SADDR0_ADDRESS7_Pos                   (7U)                               
#define I2C_SADDR0_ADDRESS7_Msk                   (0x1U << I2C_SADDR0_ADDRESS7_Pos)        /*!< 0x00000080 */
#define I2C_SADDR0_ADDRESS7                       I2C_SADDR0_ADDRESS7_Msk                  /*!< Bit 7 */
#define I2C_SADDR0_ADDRESS8_Pos                   (8U)                               
#define I2C_SADDR0_ADDRESS8_Msk                   (0x1U << I2C_SADDR0_ADDRESS8_Pos)        /*!< 0x00000100 */
#define I2C_SADDR0_ADDRESS8                       I2C_SADDR0_ADDRESS8_Msk                  /*!< Bit 8 */
#define I2C_SADDR0_ADDRESS9_Pos                   (9U)                               
#define I2C_SADDR0_ADDRESS9_Msk                   (0x1U << I2C_SADDR0_ADDRESS9_Pos)        /*!< 0x00000200 */
#define I2C_SADDR0_ADDRESS9                       I2C_SADDR0_ADDRESS9_Msk                  /*!< Bit 9 */

#define I2C_SADDR0_ADDRESSMODE_Pos                (15U)                              
#define I2C_SADDR0_ADDRESSMODE_Msk                (0x1U << I2C_SADDR0_ADDRESSMODE_Pos)     /*!< 0x00008000 */
#define I2C_SADDR0_ADDRESSMODE                    I2C_SADDR0_ADDRESSMODE_Msk               /*!< Addressing Mode (Slave mode) */

/*******************  Bit definition for I2C_SADDR1 register  *******************/
#define I2C_SADDR1_DUADEN_Pos                 (0U)                               
#define I2C_SADDR1_DUADEN_Msk                 (0x1U << I2C_SADDR1_DUADEN_Pos)      /*!< 0x00000001 */
#define I2C_SADDR1_DUADEN                     I2C_SADDR1_DUADEN_Msk                /*!< Dual addressing mode enable */
#define I2C_SADDR1_ADDRESS2_Pos                   (1U)                               
#define I2C_SADDR1_ADDRESS2_Msk                   (0x7FU << I2C_SADDR1_ADDRESS2_Pos)       /*!< 0x000000FE */
#define I2C_SADDR1_ADDRESS2                       I2C_SADDR1_ADDRESS2_Msk                  /*!< Interface address */

/********************  Bit definition for I2C_DATA register  ********************/
#define I2C_DATA_TRB_Pos             (0U)                                         
#define I2C_DATA_TRB_Msk             (0xFFU << I2C_DATA_TRB_Pos)                     /*!< 0x000000FF */
#define I2C_DATA_TRB                 I2C_DATA_TRB_Msk                                /*!< 8-bit Data Register         */

/*******************  Bit definition for I2C_STAT0 register  ********************/
#define I2C_STAT0_SBSEND_Pos                      (0U)                               
#define I2C_STAT0_SBSEND_Msk                      (0x1U << I2C_STAT0_SBSEND_Pos)           /*!< 0x00000001 */
#define I2C_STAT0_SBSEND                          I2C_STAT0_SBSEND_Msk                     /*!< Start Bit (Master mode) */
#define I2C_STAT0_ADDSEND_Pos                    (1U)                               
#define I2C_STAT0_ADDSEND_Msk                    (0x1U << I2C_STAT0_ADDSEND_Pos)         /*!< 0x00000002 */
#define I2C_STAT0_ADDSEND                        I2C_STAT0_ADDSEND_Msk                   /*!< Address sent (master mode)/matched (slave mode) */
#define I2C_STAT0_BTC_Pos                     (2U)                               
#define I2C_STAT0_BTC_Msk                     (0x1U << I2C_STAT0_BTC_Pos)          /*!< 0x00000004 */
#define I2C_STAT0_BTC                         I2C_STAT0_BTC_Msk                    /*!< Byte Transfer Finished */
#define I2C_STAT0_ADD10SEND_Pos                   (3U)                               
#define I2C_STAT0_ADD10SEND_Msk                   (0x1U << I2C_STAT0_ADD10SEND_Pos)        /*!< 0x00000008 */
#define I2C_STAT0_ADD10SEND                       I2C_STAT0_ADD10SEND_Msk                  /*!< 10-bit header sent (Master mode) */
#define I2C_STAT0_STPDET_Pos                   (4U)                               
#define I2C_STAT0_STPDET_Msk                   (0x1U << I2C_STAT0_STPDET_Pos)        /*!< 0x00000010 */
#define I2C_STAT0_STPDET                       I2C_STAT0_STPDET_Msk                  /*!< Stop detection (Slave mode) */
#define I2C_STAT0_RBNE_Pos                    (6U)                               
#define I2C_STAT0_RBNE_Msk                    (0x1U << I2C_STAT0_RBNE_Pos)         /*!< 0x00000040 */
#define I2C_STAT0_RBNE                        I2C_STAT0_RBNE_Msk                   /*!< Data Register not Empty (receivers) */
#define I2C_STAT0_TBE_Pos                     (7U)                               
#define I2C_STAT0_TBE_Msk                     (0x1U << I2C_STAT0_TBE_Pos)          /*!< 0x00000080 */
#define I2C_STAT0_TBE                         I2C_STAT0_TBE_Msk                    /*!< Data Register Empty (transmitters) */
#define I2C_STAT0_BERR_Pos                    (8U)                               
#define I2C_STAT0_BERR_Msk                    (0x1U << I2C_STAT0_BERR_Pos)         /*!< 0x00000100 */
#define I2C_STAT0_BERR                        I2C_STAT0_BERR_Msk                   /*!< Bus Error */
#define I2C_STAT0_LOSTARB_Pos                    (9U)                               
#define I2C_STAT0_LOSTARB_Msk                    (0x1U << I2C_STAT0_LOSTARB_Pos)         /*!< 0x00000200 */
#define I2C_STAT0_LOSTARB                        I2C_STAT0_LOSTARB_Msk                   /*!< Arbitration Lost (master mode) */
#define I2C_STAT0_AERR_Pos                      (10U)                              
#define I2C_STAT0_AERR_Msk                      (0x1U << I2C_STAT0_AERR_Pos)           /*!< 0x00000400 */
#define I2C_STAT0_AERR                          I2C_STAT0_AERR_Msk                     /*!< Acknowledge Failure */
#define I2C_STAT0_OUERR_Pos                     (11U)                              
#define I2C_STAT0_OUERR_Msk                     (0x1U << I2C_STAT0_OUERR_Pos)          /*!< 0x00000800 */
#define I2C_STAT0_OUERR                         I2C_STAT0_OUERR_Msk                    /*!< Overrun/Underrun */
#define I2C_STAT0_PECERR_Pos                  (12U)                              
#define I2C_STAT0_PECERR_Msk                  (0x1U << I2C_STAT0_PECERR_Pos)       /*!< 0x00001000 */
#define I2C_STAT0_PECERR                      I2C_STAT0_PECERR_Msk                 /*!< PEC Error in reception */
#define I2C_STAT0_SMBTO_Pos                 (14U)                              
#define I2C_STAT0_SMBTO_Msk                 (0x1U << I2C_STAT0_SMBTO_Pos)      /*!< 0x00004000 */
#define I2C_STAT0_SMBTO                     I2C_STAT0_SMBTO_Msk                /*!< Timeout or Tlow Error */
#define I2C_STAT0_SMBALT_Pos                (15U)                              
#define I2C_STAT0_SMBALT_Msk                (0x1U << I2C_STAT0_SMBALT_Pos)     /*!< 0x00008000 */
#define I2C_STAT0_SMBALT                    I2C_STAT0_SMBALT_Msk               /*!< SMBus Alert */

/*******************  Bit definition for I2C_STAT1 register  ********************/
#define I2C_STAT1_MASTER_Pos                     (0U)                               
#define I2C_STAT1_MASTER_Msk                     (0x1U << I2C_STAT1_MASTER_Pos)          /*!< 0x00000001 */
#define I2C_STAT1_MASTER                         I2C_STAT1_MASTER_Msk                    /*!< Master/Slave */
#define I2C_STAT1_I2CBSY_Pos                    (1U)                               
#define I2C_STAT1_I2CBSY_Msk                    (0x1U << I2C_STAT1_I2CBSY_Pos)         /*!< 0x00000002 */
#define I2C_STAT1_I2CBSY                        I2C_STAT1_I2CBSY_Msk                   /*!< Bus Busy */
#define I2C_STAT1_TR_Pos                     (2U)                               
#define I2C_STAT1_TR_Msk                     (0x1U << I2C_STAT1_TR_Pos)          /*!< 0x00000004 */
#define I2C_STAT1_TR                         I2C_STAT1_TR_Msk                    /*!< Transmitter/Receiver */
#define I2C_STAT1_RXGC_Pos                 (4U)                               
#define I2C_STAT1_RXGC_Msk                 (0x1U << I2C_STAT1_RXGC_Pos)      /*!< 0x00000010 */
#define I2C_STAT1_RXGC                     I2C_STAT1_RXGC_Msk                /*!< General Call Address (Slave mode) */
#define I2C_STAT1_DEFSMB_Pos              (5U)                               
#define I2C_STAT1_DEFSMB_Msk              (0x1U << I2C_STAT1_DEFSMB_Pos)   /*!< 0x00000020 */
#define I2C_STAT1_DEFSMB                  I2C_STAT1_DEFSMB_Msk             /*!< SMBus Device Default Address (Slave mode) */
#define I2C_STAT1_HSTSMB_Pos                 (6U)                               
#define I2C_STAT1_HSTSMB_Msk                 (0x1U << I2C_STAT1_HSTSMB_Pos)      /*!< 0x00000040 */
#define I2C_STAT1_HSTSMB                     I2C_STAT1_HSTSMB_Msk                /*!< SMBus Host Header (Slave mode) */
#define I2C_STAT1_DUMODF_Pos                   (7U)                               
#define I2C_STAT1_DUMODF_Msk                   (0x1U << I2C_STAT1_DUMODF_Pos)        /*!< 0x00000080 */
#define I2C_STAT1_DUMODF                       I2C_STAT1_DUMODF_Msk                  /*!< Dual Flag (Slave mode) */
#define I2C_STAT1_PECV_Pos                     (8U)                               
#define I2C_STAT1_PECV_Msk                     (0xFFU << I2C_STAT1_PECV_Pos)         /*!< 0x0000FF00 */
#define I2C_STAT1_PECV                         I2C_STAT1_PECV_Msk                    /*!< Packet Error Checking Register */

/*******************  Bit definition for I2C_CKCFG register  ********************/
#define I2C_CKCFG_CLKC_Pos                     (0U)                               
#define I2C_CKCFG_CLKC_Msk                     (0xFFFU << I2C_CKCFG_CLKC_Pos)        /*!< 0x00000FFF */
#define I2C_CKCFG_CLKC                         I2C_CKCFG_CLKC_Msk                    /*!< Clock Control Register in Fast/Standard mode (Master mode) */
#define I2C_CKCFG_DTCY_Pos                    (14U)                              
#define I2C_CKCFG_DTCY_Msk                    (0x1U << I2C_CKCFG_DTCY_Pos)         /*!< 0x00004000 */
#define I2C_CKCFG_DTCY                        I2C_CKCFG_DTCY_Msk                   /*!< Fast Mode Duty Cycle */
#define I2C_CKCFG_FAST_Pos                      (15U)                              
#define I2C_CKCFG_FAST_Msk                      (0x1U << I2C_CKCFG_FAST_Pos)           /*!< 0x00008000 */
#define I2C_CKCFG_FAST                          I2C_CKCFG_FAST_Msk                     /*!< I2C Master Mode Selection */

/******************  Bit definition for I2C_RT register  *******************/
#define I2C_RT_RISETIME_Pos                 (0U)                               
#define I2C_RT_RISETIME_Msk                 (0x3FU << I2C_RT_RISETIME_Pos)     /*!< 0x0000003F */
#define I2C_RT_RISETIME                     I2C_RT_RISETIME_Msk                /*!< Maximum Rise Time in Fast/Standard mode (Master mode) */

/******************  Bit definition for I2C_FMPCFG register  *******************/
#define I2C_FMPCFG_FMPEN_Pos                 (0U)                               
#define I2C_FMPCFG_FMPEN_Msk                 (0x1U << I2C_FMPCFG_FMPEN_Pos)     /*!< 0x0000003F */
#define I2C_FMPCFG_FMPEN                     I2C_FMPCFG_FMPEN_Msk             /*!< Fast mode plus enable */

/******************************************************************************/
/*                                                                            */
/*         Universal Synchronous Asynchronous Receiver Transmitter            */
/*                                                                            */
/******************************************************************************/

/*******************  Bit definition for USART_STAT register  *******************/
#define USART_STAT_PERR_Pos                     (0U)                               
#define USART_STAT_PERR_Msk                     (0x1U << USART_STAT_PERR_Pos)          /*!< 0x00000001 */
#define USART_STAT_PERR                         USART_STAT_PERR_Msk                    /*!< Parity Error */
#define USART_STAT_FERR_Pos                     (1U)                               
#define USART_STAT_FERR_Msk                     (0x1U << USART_STAT_FERR_Pos)          /*!< 0x00000002 */
#define USART_STAT_FERR                         USART_STAT_FERR_Msk                    /*!< Framing Error */
#define USART_STAT_NERR_Pos                     (2U)                               
#define USART_STAT_NERR_Msk                     (0x1U << USART_STAT_NERR_Pos)          /*!< 0x00000004 */
#define USART_STAT_NERR                         USART_STAT_NERR_Msk                    /*!< Noise Error Flag */
#define USART_STAT_ORERR_Pos                    (3U)                               
#define USART_STAT_ORERR_Msk                    (0x1U << USART_STAT_ORERR_Pos)         /*!< 0x00000008 */
#define USART_STAT_ORERR                        USART_STAT_ORERR_Msk                   /*!< OverRun Error */
#define USART_STAT_IDLEF_Pos                   (4U)                               
#define USART_STAT_IDLEF_Msk                   (0x1U << USART_STAT_IDLEF_Pos)        /*!< 0x00000010 */
#define USART_STAT_IDLEF                       USART_STAT_IDLEF_Msk                  /*!< IDLE line detected */
#define USART_STAT_RBNE_Pos                   (5U)                               
#define USART_STAT_RBNE_Msk                   (0x1U << USART_STAT_RBNE_Pos)        /*!< 0x00000020 */
#define USART_STAT_RBNE                       USART_STAT_RBNE_Msk                  /*!< Read Data Register Not Empty */
#define USART_STAT_TC_Pos                     (6U)                               
#define USART_STAT_TC_Msk                     (0x1U << USART_STAT_TC_Pos)          /*!< 0x00000040 */
#define USART_STAT_TC                         USART_STAT_TC_Msk                    /*!< Transmission Complete */
#define USART_STAT_TBE_Pos                    (7U)                               
#define USART_STAT_TBE_Msk                    (0x1U << USART_STAT_TBE_Pos)         /*!< 0x00000080 */
#define USART_STAT_TBE                        USART_STAT_TBE_Msk                   /*!< Transmit Data Register Empty */
#define USART_STAT_LBDF_Pos                    (8U)                               
#define USART_STAT_LBDF_Msk                    (0x1U << USART_STAT_LBDF_Pos)         /*!< 0x00000100 */
#define USART_STAT_LBDF                        USART_STAT_LBDF_Msk                   /*!< LIN Break Detection Flag */
#define USART_STAT_CTSF_Pos                    (9U)                               
#define USART_STAT_CTSF_Msk                    (0x1U << USART_STAT_CTSF_Pos)         /*!< 0x00000200 */
#define USART_STAT_CTSF                        USART_STAT_CTSF_Msk                   /*!< CTS Flag */

/*******************  Bit definition for USART_DATA register  *******************/
#define USART_DATA_DATA_Pos                     (0U)                               
#define USART_DATA_DATA_Msk                     (0x1FFU << USART_DATA_DATA_Pos)        /*!< 0x000001FF */
#define USART_DATA_DATA                         USART_DATA_DATA_Msk                    /*!< Data value */

/******************  Bit definition for USART_BAUD register  *******************/
#define USART_BAUD_FRADIV_Pos          (0U)                               
#define USART_BAUD_FRADIV_Msk          (0xFU << USART_BAUD_FRADIV_Pos) /*!< 0x0000000F */
#define USART_BAUD_FRADIV              USART_BAUD_FRADIV_Msk         /*!< Fraction of USARTDIV */
#define USART_BAUD_INTDIV_Pos          (4U)                               
#define USART_BAUD_INTDIV_Msk          (0xFFFU << USART_BAUD_INTDIV_Pos) /*!< 0x0000FFF0 */
#define USART_BAUD_INTDIV              USART_BAUD_INTDIV_Msk         /*!< Mantissa of USARTDIV */

/******************  Bit definition for USART_CTL0 register  *******************/
#define USART_CTL0_SBKCMD_Pos                   (0U)                               
#define USART_CTL0_SBKCMD_Msk                   (0x1U << USART_CTL0_SBKCMD_Pos)        /*!< 0x00000001 */
#define USART_CTL0_SBKCMD                       USART_CTL0_SBKCMD_Msk                  /*!< Send Break */
#define USART_CTL0_RWU_Pos                   (1U)                               
#define USART_CTL0_RWU_Msk                   (0x1U << USART_CTL0_RWU_Pos)        /*!< 0x00000002 */
#define USART_CTL0_RWU                       USART_CTL0_RWU_Msk                  /*!< Receiver wakeup */
#define USART_CTL0_REN_Pos                    (2U)                               
#define USART_CTL0_REN_Msk                    (0x1U << USART_CTL0_REN_Pos)         /*!< 0x00000004 */
#define USART_CTL0_REN                        USART_CTL0_REN_Msk                   /*!< Receiver Enable */
#define USART_CTL0_TEN_Pos                    (3U)                               
#define USART_CTL0_TEN_Msk                    (0x1U << USART_CTL0_TEN_Pos)         /*!< 0x00000008 */
#define USART_CTL0_TEN                        USART_CTL0_TEN_Msk                   /*!< Transmitter Enable */
#define USART_CTL0_IDLEIE_Pos                (4U)                               
#define USART_CTL0_IDLEIE_Msk                (0x1U << USART_CTL0_IDLEIE_Pos)     /*!< 0x00000010 */
#define USART_CTL0_IDLEIE                    USART_CTL0_IDLEIE_Msk               /*!< IDLE Interrupt Enable */
#define USART_CTL0_RBNEIE_Pos                (5U)                               
#define USART_CTL0_RBNEIE_Msk                (0x1U << USART_CTL0_RBNEIE_Pos)     /*!< 0x00000020 */
#define USART_CTL0_RBNEIE                    USART_CTL0_RBNEIE_Msk               /*!< RXNE Interrupt Enable */
#define USART_CTL0_TCIE_Pos                  (6U)                               
#define USART_CTL0_TCIE_Msk                  (0x1U << USART_CTL0_TCIE_Pos)       /*!< 0x00000040 */
#define USART_CTL0_TCIE                      USART_CTL0_TCIE_Msk                 /*!< Transmission Complete Interrupt Enable */
#define USART_CTL0_TBEIE_Pos                 (7U)                               
#define USART_CTL0_TBEIE_Msk                 (0x1U << USART_CTL0_TBEIE_Pos)      /*!< 0x00000080 */
#define USART_CTL0_TBEIE                     USART_CTL0_TBEIE_Msk                /*!< PE Interrupt Enable */
#define USART_CTL0_PERRIE_Pos                  (8U)                               
#define USART_CTL0_PERRIE_Msk                  (0x1U << USART_CTL0_PERRIE_Pos)       /*!< 0x00000100 */
#define USART_CTL0_PERRIE                      USART_CTL0_PERRIE_Msk                 /*!< PE Interrupt Enable */
#define USART_CTL0_PM_Pos                    (9U)                               
#define USART_CTL0_PM_Msk                    (0x1U << USART_CTL0_PM_Pos)         /*!< 0x00000200 */
#define USART_CTL0_PM                        USART_CTL0_PM_Msk                   /*!< Parity Selection */
#define USART_CTL0_PCEN_Pos                   (10U)                              
#define USART_CTL0_PCEN_Msk                   (0x1U << USART_CTL0_PCEN_Pos)        /*!< 0x00000400 */
#define USART_CTL0_PCEN                       USART_CTL0_PCEN_Msk                  /*!< Parity Control Enable */
#define USART_CTL0_WM_Pos                  (11U)                              
#define USART_CTL0_WM_Msk                  (0x1U << USART_CTL0_WM_Pos)       /*!< 0x00000800 */
#define USART_CTL0_WM                      USART_CTL0_WM_Msk                 /*!< Wakeup method */
#define USART_CTL0_WL_Pos                     (12U)                              
#define USART_CTL0_WL_Msk                     (0x1U << USART_CTL0_WL_Pos)          /*!< 0x00001000 */
#define USART_CTL0_WL                         USART_CTL0_WL_Msk                    /*!< Word length */
#define USART_CTL0_UEN_Pos                    (13U)                              
#define USART_CTL0_UEN_Msk                    (0x1U << USART_CTL0_UEN_Pos)         /*!< 0x00002000 */
#define USART_CTL0_UEN                        USART_CTL0_UEN_Msk                   /*!< USART Enable */

/******************  Bit definition for USART_CTL1 register  *******************/
#define USART_CTL1_ADDR_Pos                   (0U)                               
#define USART_CTL1_ADDR_Msk                   (0xFU << USART_CTL1_ADDR_Pos)        /*!< 0x0000000F */
#define USART_CTL1_ADDR                       USART_CTL1_ADDR_Msk                  /*!< Address of the USART node */
#define USART_CTL1_LBLEN_Pos                  (5U)                               
#define USART_CTL1_LBLEN_Msk                  (0x1U << USART_CTL1_LBLEN_Pos)       /*!< 0x00000020 */
#define U