// =====================================================================
//  $Id: TDataElement.cc,v 1.3 2003/07/30 16:17:50 goiwai Exp $
//  $Name: CLDAQ-1-07-01 $
//
//  $Log: TDataElement.cc,v $
//  Revision 1.3  2003/07/30 16:17:50  goiwai
//  ե˥ߥåȥĤ뤳Ȥˤޤ.
//
// =====================================================================
#include "TDataElement.hh"
#include "TOutputObjectStream.hh"
#include "TOutputObjectFile.hh"
#include "TOutputObjectSocket.hh"
#include "TOutputObjectSharedMemory.hh"

TDataElement::TDataElement( Telement_t type, const Tstring& id )
  : TStreamableObject( tObjectDataElement, id ),
    theData( 0 ), theElementType( type ), theNumberOfPrimitives( 0 )
{;}

TDataElement::TDataElement( Tvoid* data, Telement_t type, const Tstring& id, Tint ndata )
  : TStreamableObject( tObjectDataElement, id ), 
    theData( 0 ), theElementType( type ), theNumberOfPrimitives( ndata )
{
  allocateDataSpace( data );
}

TDataElement::TDataElement( const TDataElement& right )
  : TStreamableObject( right ),
    theData( 0 ), theElementType( right.theElementType ),
    theNumberOfPrimitives( right.theNumberOfPrimitives )
{
  allocateDataSpace( right.theData );
}

TDataElement::~TDataElement()
{
  freeDataSpace();
}

Tint TDataElement::Record( TOutputObjectStream* output )
{
  Tint datasize = 0;
  Tstream_t streamtype = output -> GetStreamType();

  switch ( streamtype ) {
    case tFileStream:
      datasize = record( (TOutputObjectFile*)output );
      break;
    case tSocketStream:
      datasize = record( (TOutputObjectSocket*)output );
      break;
    case tSharedMemoryStream:
      datasize = record( (TOutputObjectSharedMemory*)output );
      break;
    case tUnknownStream:
    default:
      Tcerr << "TDataElement::Record: unknown stream type." << Tendl;
      return( 0 );
  }

  return( datasize );
}

Tint TDataElement::GetRecordSize()
{
  Tsize_t nchar = theID.size() + 1;
  // record size + object type + object id + element type + number of primitives
  Tint total = Tsizeof( Tint ) + Tsizeof( Tobject_t ) + Tsizeof( Tsize_t ) + ( Tsizeof( Tchar ) * nchar ) + Tsizeof( Telement_t ) + Tsizeof( Tint );

  switch ( theElementType ) {
    case tTypeInt:
      total += ( Tsizeof( Tint ) ) * theNumberOfPrimitives;
      break;
    case tTypeUnsignedInt:
      total += ( Tsizeof( TUint ) ) * theNumberOfPrimitives;
      break;
    case tTypeWord:
    case tTypeUnsignedShort:
      total += ( Tsizeof( TUshort ) ) * theNumberOfPrimitives;
      break;
    case tTypeDouble:
      total += ( Tsizeof( Tdouble ) ) * theNumberOfPrimitives;
      break;
    case tTypeFloat:
      total += ( Tsizeof( Tfloat ) ) * theNumberOfPrimitives;
      break;
    case tTypeShort:
      total += ( Tsizeof( Tshort ) ) * theNumberOfPrimitives;
      break;
    case tTypeLong:
      total += ( Tsizeof( Tlong ) ) * theNumberOfPrimitives;
      break;
    case tTypeUnsignedLong:
      total += ( Tsizeof( TUlong ) ) * theNumberOfPrimitives;
      break;
    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	total += ( Tsizeof( Tsize_t ) );
	Tsize_t ss = ( ( (Tstring*)theData )[ i ] ).size() + 1;
	total += ( Tsizeof( Tchar ) ) * ss;
      }
      break;
    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }
  return( (Tint)total );
}

const TDataElement& TDataElement::operator=( const TDataElement& right )
{
  *( (TStreamableObject*)this ) = *( (TStreamableObject*)(&right) );

  freeDataSpace();

  theElementType = right.theElementType;
  theNumberOfPrimitives = right.theNumberOfPrimitives;

  allocateDataSpace( right.theData );

  return( *this );
}

Tbool TDataElement::operator==( const TDataElement& right ) const
{
  Tbool ret = Ttrue;
  ret &= ( *((TStreamableObject*)this) == *( (TStreamableObject*)(&right) ) );
  ret &= ( theElementType == right.theElementType );
  ret &= ( theNumberOfPrimitives == right.theNumberOfPrimitives );

  if ( ret != Tfalse ) {
    switch ( theElementType ) {
      case tTypeInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tint*)theData)[ i ] == ((Tint*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUint*)theData)[ i ] == ((TUint*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedShort:
      case tTypeWord:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUshort*)theData)[ i ] == ((TUshort*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeDouble:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tdouble*)theData)[ i ] == ((Tdouble*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeFloat:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tfloat*)theData)[ i ] == ((Tfloat*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeShort:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tshort*)theData)[ i ] == ((Tshort*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tlong*)theData)[ i ] == ((Tlong*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUlong*)theData)[ i ] == ((TUlong*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeString:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tstring*)theData)[ i ] == ((Tstring*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnknown:
      case tTypeObject:
      default:
	ret &= Tfalse;
	break;
    }
  }
  return( ret );
}

inline Tbool TDataElement::operator!=( const TDataElement& right ) const
{
  Tbool ret = Tfalse;
  ret |= ( *((TStreamableObject*)this) != *((TStreamableObject*)(&right)) );
  ret |= ( theElementType != right.theElementType );
  ret |= ( theNumberOfPrimitives != right.theNumberOfPrimitives );


  if ( ret == Tfalse ) {
    switch ( theElementType ) {
      case tTypeInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tint*)theData)[ i ] != ((Tint*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnsignedInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((TUint*)theData)[ i ] != ((TUint*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnsignedShort:
      case tTypeWord:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((TUshort*)theData)[ i ] != ((TUshort*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeDouble:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tdouble*)theData)[ i ] != ((Tdouble*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeFloat:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tfloat*)theData)[ i ] != ((Tfloat*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeShort:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tshort*)theData)[ i ] != ((Tshort*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tlong*)theData)[ i ] != ((Tlong*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnsignedLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((TUlong*)theData)[ i ] != ((TUlong*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeString:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tstring*)theData)[ i ] != ((Tstring*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnknown:
      case tTypeObject:
      default:
	ret |= Ttrue;
	break;
    }
  }
  return( ret );
}

Tint TDataElement::record( TOutputObjectFile* ofile )
{
  Tsize_t datasize = 0;
  static const Tsize_t nmemb = 1;
  static const Tstring head = "TDataElement::record";
  Tsize_t ss = 0;


  // write record size
  Tint recsize = GetRecordSize();
  ss = fwrite( &recsize, Tsizeof( Tint ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb ) {
    perror( head.c_str() );
  } else {
    datasize += ss * Tsizeof( Tint );
  }



  // write object type
  ss = fwrite( &theObjectType, Tsizeof( Tobject_t ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb ) {
    perror( head.c_str() );
  } else {
    datasize += ss * Tsizeof( Tobject_t );
  }


  // write object ID
  Tchar charbuf;
  Tsize_t nchar = theID.size() + 1;
  ss = fwrite( &nchar, Tsizeof( Tsize_t ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb ) {
    perror( head.c_str() );
  } else {
    datasize += ss * Tsizeof( Tsize_t );
  }
  for ( Tsize_t i = 0; i < nchar; i ++ ) {
    if ( i == ( nchar - 1 ) ) {
      charbuf = '\0';
    } else {
      charbuf = theID[ i ];
    }
    ss = fwrite( &charbuf, Tsizeof( Tchar ), nmemb, ofile -> GetFileStream() );
    if ( ss != nmemb ) {
      perror( head.c_str() );
    } else {
      datasize += ss * Tsizeof( Tchar );
    }
  }


  // write element type
  ss = fwrite( &theElementType, Tsizeof( Telement_t ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb ) {
    perror( head.c_str() );
  } else {
    datasize += ss * Tsizeof( Telement_t );
  }


  // write number of primitives
  ss = fwrite( &theNumberOfPrimitives, Tsizeof( Tint ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb ) {
    perror( head.c_str() );
  } else {
    datasize += Tsizeof( Tint );
  }


  Tstring strbuf;
  switch ( theElementType ) {

    case tTypeInt:
      Tint ii;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ii = ( (Tint*)theData )[ i ];
	ss = fwrite( &ii, Tsizeof( Tint ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( Tint );
      }
      break;


    case tTypeUnsignedInt:
      TUint ui;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ui = ( (TUint*)theData )[ i ];
	ss = fwrite( &ui, Tsizeof( TUint ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( TUint );
      }
      break;


    case tTypeUnsignedShort:
    case tTypeWord:
      TUshort us;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	us = ( (TUshort*)theData )[ i ];
	ss = fwrite( &us, Tsizeof( TUshort ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( TUshort );
      }
      break;


    case tTypeDouble:
      Tdouble dd;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	dd = ( (Tdouble*)theData )[ i ];
	ss = fwrite( &dd, Tsizeof( Tdouble ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( Tdouble );
      }
      break;


    case tTypeFloat:
      Tfloat ff;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ff = ( (Tfloat*)theData )[ i ];
	ss = fwrite( &ff, Tsizeof( Tfloat ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( Tfloat );
      }
      break;


    case tTypeShort:
      Tshort ww;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ww = ( (Tshort*)theData )[ i ];
	ss = fwrite( &ww, Tsizeof( Tshort ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( Tshort );
      }
      break;


    case tTypeLong:
      Tlong ll;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ll = ( (Tlong*)theData )[ i ];
	ss = fwrite( &ll, Tsizeof( Tlong ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( Tlong );
      }
      break;


    case tTypeUnsignedLong:
      TUlong ul;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ul = ( (TUlong*)theData )[ i ];
	ss = fwrite( &ul, Tsizeof( TUlong ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( TUlong );
      }
      break;


    case tTypeString:
      Tchar cc;
      Tsize_t nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];

	nchar = strbuf.size() + 1;
	ss = fwrite( &nchar, Tsizeof( Tsize_t ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( head.c_str() );
	else
	  datasize += ss * Tsizeof( Tsize_t );


	for ( Tsize_t j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) )
	    cc = '\0';
	  else
	    cc = strbuf[ j ];
	  ss = fwrite( &cc, Tsizeof( Tchar ), nmemb, ofile -> GetFileStream() );
	  if ( ss != nmemb ) {
	    perror( head.c_str() );
	  } else {
	    datasize += ss * Tsizeof( Tchar );
	  }
	}
      }
      break;


    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }

  return( (Tint)datasize );
}

Tint TDataElement::record( TOutputObjectSocket* osocket )
{
  Tsize_t datasize = 0;
  static const Tstring head = "TDataElement::record";


  // send record size
  Tint recsize = GetRecordSize();
  if ( send( osocket -> GetServerDescriptor(), &recsize, Tsizeof( Tint ), 0 ) != Tsizeof( Tint ) ) {
    perror( head.c_str() );
  } else {
    datasize += Tsizeof( Tint );
  }


  // send object type
  if ( send( osocket -> GetServerDescriptor(), &theObjectType, Tsizeof( Tobject_t ), 0 ) != Tsizeof( Tobject_t ) ) {
    perror( head.c_str() );
  } else {
    datasize += Tsizeof( Tobject_t );
  }


  // send object ID
  Tchar charbuf;
  Tsize_t nchar = theID.size() + 1;
  if ( send( osocket -> GetServerDescriptor(), &nchar, Tsizeof( Tsize_t ), 0 ) != Tsizeof( Tsize_t ) ) {
    perror( head.c_str() );
  } else {
    datasize += Tsizeof( Tsize_t );
  }
  for ( Tsize_t i = 0; i < nchar; i ++ ) {
    if ( i == ( nchar - 1 ) ) {
      charbuf = '\0';
    } else {
      charbuf = theID[ i ];
    }
    if ( send( osocket -> GetServerDescriptor(), &charbuf, Tsizeof( Tchar ), 0 ) != Tsizeof( Tchar ) ) {
      perror( head.c_str() );
    } else {
      datasize += Tsizeof( Tchar );
    }
  }


  // send element type
  if ( send( osocket -> GetServerDescriptor(), &theElementType, Tsizeof( Telement_t ), 0 ) != Tsizeof( Telement_t ) ) {
    perror( head.c_str() );
  } else {
    datasize += Tsizeof( Telement_t );
  }


  // send number of primitives
  if ( send( osocket -> GetServerDescriptor(), &theNumberOfPrimitives, Tsizeof( Tint ), 0 ) != Tsizeof( Tint ) ) {
    perror( head.c_str() );
  } else {
    datasize += Tsizeof( Tint );
  }




  Tstring strbuf;
  switch ( theElementType ) {


    case tTypeInt:
      Tint ii;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ii = ( (Tint*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ii, Tsizeof( Tint ), 0 ) != Tsizeof( Tint ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( Tint );
	}
      }
      break;


    case tTypeUnsignedInt:
      TUint ui;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ui = ( (TUint*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ui, Tsizeof( TUint ), 0 ) != Tsizeof( TUint ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( TUint );
	}
      }
      break;



    case tTypeUnsignedShort:
    case tTypeWord:
      TUshort us;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	us = ( (TUshort*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &us, Tsizeof( TUshort ), 0 ) != Tsizeof( TUshort ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( TUshort );
	}
      }
      break;


    case tTypeDouble:
      Tdouble dd;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	dd = ( (Tdouble*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &dd, Tsizeof( Tdouble ), 0 ) != Tsizeof( Tdouble ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( Tdouble );
	}
      }
      break;


    case tTypeFloat:
      Tfloat ff;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ff = ( (Tfloat*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ff, Tsizeof( Tfloat ), 0 ) != Tsizeof( Tfloat ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( Tfloat );
	}
      }
      break;


    case tTypeShort:
      Tshort ww;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ww = ( (TUshort*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ww, Tsizeof( Tshort ), 0 ) != Tsizeof( Tshort ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( Tshort );
	}
      }
      break;



    case tTypeLong:
      Tlong ll;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ll = ( (Tlong*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ll, Tsizeof( Tlong ), 0 ) != Tsizeof( Tlong ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( Tlong );
	}
      }
      break;


    case tTypeUnsignedLong:
      TUlong ul;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ul = ( (TUlong*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ul, Tsizeof( TUlong ), 0 ) != Tsizeof( TUlong ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( TUlong );
	}
      }
      break;



    case tTypeString:
      Tchar cc;
      Tsize_t nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];
	nchar = strbuf.size() + 1;
	if ( send( osocket -> GetServerDescriptor(), &nchar, Tsizeof( Tsize_t ), 0 ) != Tsizeof( Tsize_t ) ) {
	  perror( head.c_str() );
	} else {
	  datasize += Tsizeof( Tsize_t );
	}
	for ( Tsize_t j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) ) {
	    cc = '\0';
	  } else {
	    cc = strbuf[ j ];
	  }
	  if ( send( osocket -> GetServerDescriptor(), &cc, Tsizeof( Tchar ), 0 ) != Tsizeof( Tchar ) ) {
	    perror( head.c_str() );
	  } else {
	    datasize += Tsizeof( Tchar );
	  }
	}
      }
      break;



    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }
  
  return( (Tint)datasize );
}

Tint TDataElement::record( TOutputObjectSharedMemory* omemory )
{
  Tsize_t datasize = 0;
  static const Tstring head = "TDataElement::record";
  Tvoid* ptr = omemory -> GetAddress();


  // write record size
  *( (Tint*)ptr ) = GetRecordSize();
  datasize += Tsizeof( Tint );
  ( (Tint*)ptr ) ++;


  // write object type
  *( (Tobject_t*)ptr ) = theObjectType;
  datasize += Tsizeof( Tobject_t );
  ( (Tobject_t*)ptr ) ++;


  // write object ID
  Tchar charbuf;
  Tsize_t nchar = theID.size() + 1;
  *( (Tsize_t*)ptr ) = nchar;
  datasize += Tsizeof( Tsize_t );
  ( (Tsize_t*)ptr ) ++;
  for ( Tsize_t i = 0; i < nchar; i ++ ) {
    if ( i == ( nchar - 1 ) ) {
      charbuf = '\0';
    } else {
      charbuf = theID[ i ];
    }
    *( (Tchar*)ptr ) = charbuf;
    datasize += Tsizeof( Tchar );
    ( (Tchar*)ptr ) ++;
  }


  // write element type
  *( (Telement_t*)ptr ) = theElementType;
  datasize += Tsizeof( Telement_t );
  ( (Telement_t*)ptr ) ++;


  // write number of primitives
  *( (Tint*)ptr ) = theNumberOfPrimitives;
  datasize += Tsizeof( Tint );
  ( (Tint*)ptr ) ++;


  Tstring strbuf;
  switch ( theElementType ) {


    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tint*)ptr ) = ( (Tint*)theData )[ i ];
	datasize += Tsizeof( Tint );
	( (Tint*)ptr ) ++;
      }
      break;


    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (TUint*)ptr ) = ( (TUint*)theData )[ i ];
	datasize += Tsizeof( TUint );
	( (TUint*)ptr ) ++;
      }
      break;


    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (TUshort*)ptr ) = ( (TUshort*)theData )[ i ];
	datasize += Tsizeof( TUshort );
	( (TUshort*)ptr ) ++;
      }
      break;


    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tdouble*)ptr ) = ( (Tdouble*)theData )[ i ];
	datasize += Tsizeof( Tdouble );
	( (Tdouble*)ptr ) ++;
      }
      break;


    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tfloat*)ptr ) = ( (Tfloat*)theData )[ i ];
	datasize += Tsizeof( Tfloat );
	( (Tdouble*)ptr ) ++;
      }
      break;


    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tshort*)ptr ) = ( (Tshort*)theData )[ i ];
	datasize += Tsizeof( Tshort );
	( (Tshort*)ptr ) ++;
      }
      break;


    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tlong*)ptr ) = ( (Tlong*)theData )[ i ];
	datasize += Tsizeof( Tlong );
	( (Tlong*)ptr ) ++;
      }
      break;


    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (TUlong*)ptr ) = ( (TUlong*)theData )[ i ];
	datasize += Tsizeof( TUlong );
	( (TUlong*)ptr ) ++;
      }
      break;



    case tTypeString:
      Tchar cc;
      Tsize_t nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];
	nchar = strbuf.size() + 1;


	// write Number of characters
	*( (Tsize_t*)ptr ) = nchar;
	datasize += Tsizeof( Tsize_t );
	( (Tsize_t*)ptr ) ++;

	for ( Tsize_t j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) )
	    cc = '\0';
	  else
	    cc = strbuf[ j ];

	  // write a character
	  *( (Tchar*)ptr ) = cc;
	  datasize += Tsizeof( Tchar );
	  ( (Tchar*)ptr ) ++;
	}
      }
      break;


    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }

  omemory -> SetAddress( ptr );
  
  return( (Tint)datasize );
}

Tostream& operator<<( Tostream& tos, const TDataElement& right )
{

  tos << "Data Element(" << right.theObjectType << "), "
      << "ID: " << right.theID
      << ", Type: " << right.theElementType;

  switch ( right.theElementType ) {


    case tTypeInt:
      tos << Tslash << "Tint(" << Tsizeof( Tint ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tint*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 ) {
	  tos << ", ";
        }
      }
      break;


    case tTypeUnsignedInt:
      tos << Tslash << "TUint(" << Tsizeof( TUint ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (TUint*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 ) {
	  tos << ", ";
        }
      }
      break;



    case tTypeWord:
    case tTypeUnsignedShort:
      tos << Tslash << "TUshort(" << Tsizeof( TUshort ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (TUshort*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 ) {
	  tos << ", ";
        }
      }
      break;



    case tTypeDouble:
      tos << Tslash << "Tdouble(" << Tsizeof( Tdouble ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tdouble*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 ) {
	  tos << ", ";
        }
      }
      break;



    case tTypeFloat:
      tos << Tslash << "Tfloat(" << Tsizeof( Tfloat ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tfloat*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << ", ";
      }
      break;



    case tTypeShort:
      tos << Tslash << "Tshort(" << Tsizeof( Tshort ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tshort*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 ) {
	  tos << ", ";
        }
      }
      break;



    case tTypeLong:
      tos << Tslash << "Tlong(" << Tsizeof( Tlong ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tlong*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 ) {
	  tos << ", ";
        }
      }
      break;


    case tTypeUnsignedLong:
      tos << Tslash << "TUlong(" << Tsizeof( TUlong ) << "b), Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (TUlong*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << ", ";
      }
      break;



    case tTypeString:
      tos << Tslash << "Tstring, Data("
          << right.theNumberOfPrimitives << "): ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tstring*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 ) {
	  tos << ", ";
        }
      }
      break;


    case tTypeUnknown:
    case tTypeObject:
    default:
      tos << Tslash << "Unknown, Data: unknown(or undefined) data.";
      break;
  }

  return( tos );
}

Tvoid TDataElement::freeDataSpace()
{
  if ( theData != 0 ) {
    switch ( theElementType ) {

      case tTypeInt:
	delete [] (Tint*)theData;
	break;

      case tTypeUnsignedInt:
	delete [] (TUint*)theData;
	break;

      case tTypeWord:
      case tTypeUnsignedShort:
	delete [] (TUshort*)theData;
	break;

      case tTypeDouble:
	delete [] (Tdouble*)theData;
	break;

      case tTypeFloat:
	delete [] (Tfloat*)theData;
	break;

      case tTypeShort:
	delete [] (Tshort*)theData;
	break;

      case tTypeLong:
	delete [] (Tlong*)theData;
	break;

      case tTypeUnsignedLong:
	delete [] (TUlong*)theData;
	break;

      case tTypeString:
	delete [] (Tstring*)theData;
	break;

      case tTypeUnknown:
      case tTypeObject:
      default:
	break;
    }
  }
  theData = 0;
  theNumberOfPrimitives = 0;
  return;
}

Tvoid TDataElement::allocateDataSpace( Tvoid* data )
{
  if ( data == 0 )
    return;

  switch ( theElementType ) {

    case tTypeInt:
      theData = new Tint[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tint*)(theData) )[ i ] = *( (Tint*)data );
	( (Tint*)data ) ++;
      }
      break;


    case tTypeUnsignedInt:
      theData = new TUint[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (TUint*)(theData) )[ i ] = *( (TUint*)data );
	( (TUint*)data ) ++;
      }
      break;

    case tTypeWord:
    case tTypeUnsignedShort:
      theData = new TUshort[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
  	( (TUshort*)(theData) )[ i ] = *( (TUshort*)data );
  	( (TUshort*)data ) ++;
      }
      break;

    case tTypeDouble:
      theData = new Tdouble[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tdouble*)(theData) )[ i ] = *( (Tdouble*)data );
	( (Tdouble*)data ) ++;
      }
      break;

    case tTypeFloat:
      theData = new Tfloat[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tfloat*)(theData) )[ i ] = *( (Tfloat*)data );
	( (Tfloat*)data ) ++;
      }
      break;

    case tTypeShort:
      theData = new Tshort[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
  	( (Tshort*)(theData) )[ i ] = *( (Tshort*)data );
  	( (Tshort*)data ) ++;
      }
      break;

    case tTypeLong:
      theData = new Tlong[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tlong*)(theData) )[ i ] = *( (Tlong*)data );
	( (Tlong*)data ) ++;
      }
      break;

    case tTypeUnsignedLong:
      theData = new TUlong[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (TUlong*)(theData) )[ i ] = *( (TUlong*)data );
	( (TUlong*)data ) ++;
      }
      break;

    case tTypeString:
      theData = new Tstring[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tstring*)(theData) )[ i ] = *( (Tstring*)data );
	( (Tstring*)data ) ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }
  return;
}

Tvoid TDataElement::Clear()
{
  freeDataSpace();
  theID = TunknownID;
  theElementType = tTypeUnknown;
  return;
}

Tint TDataElement::Serialize( Tvoid* buffer )
{
  Tsize_t datasize = 0;

  // write record size
  *( (Tint*)buffer ) = GetRecordSize();
  ( (Tint*)buffer ) ++;
  datasize += Tsizeof( Tint );


  // write object type
  *( (Tobject_t*)buffer ) = theObjectType;
  ( (Tobject_t*)buffer ) ++;
  datasize += Tsizeof( Tobject_t );


  // write object ID
  Tchar charbuf;
  Tsize_t nchar = theID.size() + 1;
  *( (Tsize_t*)buffer ) = nchar;
  ( (Tsize_t*)buffer ) ++;
  datasize += Tsizeof( Tsize_t );
  for ( Tsize_t i = 0; i < nchar; i ++ ) {
    if ( i == ( nchar - 1 ) ) {
      charbuf = '\0';
    } else {
      charbuf = theID[ i ];
    }
    *( (Tchar*)buffer ) = charbuf;
    ( (Tchar*)buffer ) ++;
    datasize += Tsizeof( Tchar );
  }


  // write element type
  *( (Telement_t*)buffer ) = theElementType;
  ( (Telement_t*)buffer ) ++;
  datasize += Tsizeof( Telement_t );


  // write number of primitives
  *( (Tint*)buffer ) = theNumberOfPrimitives;
  ( (Tint*)buffer ) ++;
  datasize += Tsizeof( Tint );


  // write primitives
  Tstring strbuf;
  switch ( theElementType ) {

    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tint*)buffer ) = ( (Tint*)theData )[ i ];
	( (Tint*)buffer ) ++;
	datasize += Tsizeof( Tint );
      }
      break;


    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (TUint*)buffer ) = ( (TUint*)theData )[ i ];
	( (TUint*)buffer ) ++;
	datasize += Tsizeof( TUint );
      }
      break;


    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (TUshort*)buffer ) = ( (TUshort*)theData )[ i ];
	( (TUshort*)buffer ) ++;
	datasize += Tsizeof( TUshort );
      }
      break;


    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tdouble*)buffer ) = ( (Tdouble*)theData )[ i ];
	( (Tdouble*)buffer ) ++;
	datasize += Tsizeof( Tdouble );
      }
      break;


    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tfloat*)buffer ) = ( (Tfloat*)theData )[ i ];
	( (Tfloat*)buffer ) ++;
	datasize += Tsizeof( Tfloat );
      }
      break;


    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tshort*)buffer ) = ( (Tshort*)theData )[ i ];
	( (Tshort*)buffer ) ++;
	datasize += Tsizeof( Tshort );
      }
      break;


    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (Tlong*)buffer ) = ( (Tlong*)theData )[ i ];
	( (Tlong*)buffer ) ++;
	datasize += Tsizeof( Tlong );
      }
      break;


    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	*( (TUlong*)buffer ) = ( (TUlong*)theData )[ i ];
	( (TUlong*)buffer ) ++;
	datasize += Tsizeof( TUlong );
      }
      break;


    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];
	nchar = strbuf.size() + 1;
	*( (Tsize_t*)buffer ) = nchar;
	( (Tsize_t*)buffer ) ++;
	datasize += Tsizeof( Tsize_t );

	for ( Tsize_t j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) ) {
	    charbuf = '\0';
	  } else {
	    charbuf = strbuf[ j ];
	  }
	  *( (Tchar*)buffer ) = charbuf;
	  ( (Tchar*)buffer ) ++;
	  datasize += Tsizeof( Tchar );
	}
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }

  return( (Tint)datasize );
}

Tint TDataElement::StorePrimitives( Tint* buffer, Tint narray ) const
{
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";
  Tint nfilled = 0;

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tint)strtol( (((Tstring*)theData )[ i ]).c_str(), 0, 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << ")." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( Tdouble* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tdouble)strtod( (((Tstring*)theData )[ i ]).c_str(), 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( Tfloat* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tfloat)strtof( (((Tstring*)theData )[ i ]).c_str(), 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( TUshort* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUshort)strtoul( (((Tstring*)theData )[ i ]).c_str(), 0, 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( Tshort* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tshort)strtol( (((Tstring*)theData )[ i ]).c_str(), 0, 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( Tlong* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (Tlong)strtol( (((Tstring*)theData )[ i ]).c_str(), 0, 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( TUlong* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUlong)strtoul( (((Tstring*)theData )[ i ]).c_str(), 0, 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( TUint* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (Tint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (TUint*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (TUshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (Tdouble*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (Tfloat*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (Tshort*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (Tlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)( (TUlong*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = (TUint)strtoul( (((Tstring*)theData )[ i ]).c_str(), 0, 0 );
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}

Tint TDataElement::StorePrimitives( Tstring* buffer, Tint narray ) const
{
  Tint nfilled = 0;
  static const Tint digits = 0;
  static const Tstring error = "TDataElement::StorePrimitives: not enough space.";

  switch ( theElementType ) {
    case tTypeInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = itostr( ( (Tint*)theData )[ i ], digits );
        nfilled ++;
      }
      break;

    case tTypeUnsignedInt:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = ultostr( (TUlong)( (TUint*)theData )[ i ], digits );
        nfilled ++;
      }
      break;

    case tTypeUnsignedShort:
    case tTypeWord:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = ultostr( (TUlong)( (TUshort*)theData )[ i ], digits );
        nfilled ++;
      }
      break;

    case tTypeDouble:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = dtostr( ( (Tdouble*)theData )[ i ] );
        nfilled ++;
      }
      break;

    case tTypeFloat:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = ftostr( ( (Tfloat*)theData )[ i ] );
        nfilled ++;
      }
      break;

    case tTypeShort:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = itostr( (Tint)( (Tshort*)theData )[ i ], digits );
        nfilled ++;
      }
      break;

    case tTypeLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = ltostr( ( (Tlong*)theData )[ i ], digits );
        nfilled ++;
      }
      break;

    case tTypeUnsignedLong:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = ultostr( ( (TUlong*)theData )[ i ], digits );
        nfilled ++;
      }
      break;

    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
        if ( narray != 0 && narray == i ) {
          Tcerr << error << Tendl;
          break;
        }
        buffer[ i ] = ( (Tstring*)theData )[ i ];
        nfilled ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      static const Tstring head = "TDataElement::StorePrimitives: ";
      Tcerr << head << "unresolved type information(" << theElementType << "." << Tendl;
      Tcerr << head << "nothing to do." << Tendl;
      break;
  }

  return( nfilled );
}
