%% -------------------------------------------------------------------
%%
%% Copyright (c) 2014 Basho Technologies, Inc.  All Rights Reserved.
%%
%%   This Source Code Form is subject to the terms of the Mozilla Public
%%   License, v. 2.0. If a copy of the MPL was not distributed with this
%%   file, You can obtain one at http://mozilla.org/MPL/2.0/.
%%
%% -------------------------------------------------------------------
%% @doc Accessor function for cloudi_x_exometer data structures
%%
%% This module uses the cloudi_x_exprecs transform (see <a href="https://github.com/uwiger/cloudi_x_parse_trans/tree/master/doc/cloudi_x_exprecs.md">cloudi_x_exprecs</a>)
%% to generate accessor functions for cloudi_x_exometer data structures.
%%
%% Note that the `value' attribute in `cloudi_x_exometer_entry{}' records may not
%% represent the true value of the metric, since cloudi_x_exometer entries often
%% have structured values, or are represented as CRDTs for update efficiency.
%%
%% @end
-module(cloudi_x_exometer_info).

-export([status/1,
         pp/1,
         pp_lookup/1,
         pp_find/1,
         pp_select/1]).

-include("cloudi_x_exometer.hrl").
-include_lib("cloudi_x_parse_trans/include/cloudi_x_exprecs.hrl").

-export_type([pp/0]).

-export_records([cloudi_x_exometer_entry]).

-type pp() :: {atom(), [{atom(), any()}]}.

-spec status(cloudi_x_exometer:entry()) -> enabled | disabled.
%% @doc Return the operational status of the given cloudi_x_exometer entry.
%%
%% The `status' attribute is overloaded in the `#cloudi_x_exometer_entry{}' record.
%% This function extracts the correct status (`enabled | disabled').
%% @end
status(#cloudi_x_exometer_entry{status = St}) ->
    cloudi_x_exometer_util:get_status(St).

-spec pp(tuple() | list()) -> pp() | [pp() | any()].
%% @doc Pretty-print a record, or list containing records.
%%
%% This function pretty-prints a record as `{RecordName, [{Attr,Value}]}',
%% or, if the input is a list, recognizes records and pretty-prints them,
%% leaving other data structures unchanged.
%% @end
pp(L) when is_list(L) ->
    [pp(X) || X <- L];
pp(X) ->
    case '#is_record-'(X) of
        true ->
            RecName = element(1,X),
            {RecName, lists:zip(
                        '#info-'(RecName,fields),
                        pp(tl(tuple_to_list(X))))};
        false ->
            if is_tuple(X) ->
                    list_to_tuple(pp(tuple_to_list(X)));
               true ->
                    X
            end
    end.

-spec pp_lookup(cloudi_x_exometer:name()) -> pp() | undefined.
%% @doc Performs a lookup by name of entry and pretty-prints the result.
%%
%% This function returns `undefined' if the entry cannot be found.
%% @end
pp_lookup(Name) ->
    case cloudi_x_exometer:info(Name, entry) of
        undefined ->
            undefined;
        Entry ->
            pp(Entry)
    end.

-spec pp_find(list()) -> [pp()].
%% @doc Performs `cloudi_x_exometer:find_entries(Path) & returns pretty-printed result.
%%
%% This function calls `cloudi_x_exometer:find_entries(Path)', retrieves the entry
%% for each matching metric, and calls `pp(Entry)' for each entry.
%% @end
pp_find(Path) ->
    pp([cloudi_x_exometer:info(M, entry) || {M,_,_} <- cloudi_x_exometer:find_entries(Path)]).

-spec pp_select(ets:match_spec()) -> [pp()].
%% @doc Performs `cloudi_x_exometer:select(Pattern) & returns pretty-printed result.
%%
%% This function calls `cloudi_x_exometer:select(Pattern)', retrieves the entry
%% for each matching metric, and calls `pp(Entry)' for each entry.
%%
%% Note that the match body of the select pattern must produce the full
%% `{Name, Type, Status}' object, e.g. by specifying <code>['$_']</code>.
%% @end
pp_select(Pat) ->
    pp([cloudi_x_exometer:info(M, entry) || {M,_,_} <- cloudi_x_exometer:select(Pat)]).
