//-*-Mode:C++;coding:utf-8;tab-width:4;c-basic-offset:4;indent-tabs-mode:()-*-
// ex: set ft=cpp fenc=utf-8 sts=4 ts=4 sw=4 et nomod:
//
// MIT License
//
// Copyright (c) 2009-2017 Michael Truog <mjtruog at gmail dot com>
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//
#include <stdint.h>
#include <string>
#include <boost/lexical_cast.hpp>
#include <gmp.h>
#include "piqpr8_gmp_verify.hpp"

class HexadecimalPiTestData
{
    public:
        static bool verify(std::string const & digit,
                           std::string const & pi)
        {
            switch (digit.size())
            {
                case 1:
                case 2:
                case 3:
                case 4:
                case 5:
                    {
                        uint32_t const indexStart =
                            boost::lexical_cast<uint32_t>(digit);
                        if (indexStart > 65536)
                        {
                            return (verify(digit, pi,
                                           "74951",
                                           digits_74951_to_75000,
                                           50) &&
                                    verify(digit, pi,
                                           "99951",
                                           digits_99951_to_100000,
                                           50));
                        }
                        uint32_t const indexEnd = 
                            std::min<size_t>(65536, indexStart + pi.size() - 1);
                        uint32_t const count = indexEnd - indexStart + 1;
                        return (pi.compare(0, count, digits_1_to_65536,
                                           indexStart, count) == 0);
                    }
                case 6:     // 10^5
                case 7:     // 10^6
                    return (verify(digit, pi,
                                   "1000000",
                                   digits_1000000_to_1000013,
                                   14) &&
                            verify(digit, pi,
                                   "10000000",
                                   digits_10000000_to_10000013,
                                   14));
                case 8:     // 10^7
                    return (verify(digit, pi,
                                   "10000000",
                                   digits_10000000_to_10000013,
                                   14) &&
                            verify(digit, pi,
                                   "100000000",
                                   digits_100000000_to_100000013,
                                   14));
                case 9:     // 10^8
                    return (verify(digit, pi,
                                   "100000000",
                                   digits_100000000_to_100000013,
                                   14) &&
                            verify(digit, pi,
                                   "1000000000",
                                   digits_1000000000_to_1000000013,
                                   14));
                case 10:    // 10^9
                    return (verify(digit, pi,
                                   "1000000000",
                                   digits_1000000000_to_1000000013,
                                   14) &&
                            verify(digit, pi,
                                   "10000000000",
                                   digits_10000000000_to_10000000013,
                                   14));
                case 11:    // 10^10
                    return (verify(digit, pi,
                                   "10000000000",
                                   digits_10000000000_to_10000000013,
                                   14) &&
                            verify(digit, pi,
                                   "100000000000",
                                   digits_100000000000_to_100000000013,
                                   14));
                case 12:    // 10^11
                    return  verify(digit, pi,
                                   "100000000000",
                                   digits_100000000000_to_100000000013,
                                   14);
                case 13:    // 10^12
                    return  verify(digit, pi,
                                   "1250000000000",
                                   digits_1250000000000_to_1250000000013,
                                   14);
                case 15:    // 10^14
                    return  verify(digit, pi,
                                   "250000000000000",
                                   digits_250000000000000_to_250000000000013,
                                   14);
                default:
                    // any missing data is automatically valid
                    return true;
            }
        }
    private:
        static bool verify(std::string const & digit,
                           std::string const & pi,
                           char const * verifyDigit,
                           char const * verifyPi,
                           size_t length)
        {
            mpz_t indexStart;
            mpz_t indexStartOffset;
            mpz_t indexEnd;
            mpz_init(indexStart);
            mpz_init(indexStartOffset);
            mpz_init(indexEnd);
            if (mpz_set_str(indexStart, digit.c_str(), 10) < 0)
                return false;
            if (mpz_set_str(indexStartOffset, verifyDigit, 10) < 0)
                return false;
            if (mpz_cmp(indexStart, indexStartOffset) >= 0)
            {
                mpz_sub(indexStart, indexStart, indexStartOffset);
                if (mpz_cmp_ui(indexStart, length) > 0)
                    return true; // beyond verifyPi
                mpz_add_ui(indexEnd, indexStart, pi.size());
                if (mpz_fits_ulong_p(indexEnd) == 0)
                    return false;
                size_t const end = mpz_get_ui(indexEnd);
                // end >= pi.size()

                // index into verifyPi
                size_t const position = end - pi.size();
                size_t const count = std::min(pi.size(), length - position);
                return (pi.compare(0, count, verifyPi, position, count) == 0);
            }
            else
            {
                mpz_sub(indexStart, indexStart, indexStartOffset);
                mpz_add_ui(indexEnd, indexStart, pi.size());
                if (mpz_cmp_ui(indexEnd, 0) <= 0)
                    return true; // before verifyPi
                if (mpz_fits_ulong_p(indexEnd) == 0)
                    return false;
                size_t const end = mpz_get_ui(indexEnd);
                // end < pi.size()

                // index into pi
                size_t const position = pi.size() - end;
                size_t const count = std::min(end, length);
                return (pi.compare(position, count, verifyPi, 0, count) == 0);
            }
        }

        // data collected for verifying the algorithm

        // data collected from:
        // David H. Bailey, "The BBP Algorithm for Pi", 8 Sept 2006
        // http://crd.lbl.gov/~dhbailey/dhbpapers/bbp-alg.pdf
        static char const digits_1000000_to_1000013[];
        static char const digits_10000000_to_10000013[];
        static char const digits_100000000_to_100000013[];
        static char const digits_1000000000_to_1000000013[];
        static char const digits_10000000000_to_10000000013[];
        static char const digits_100000000000_to_100000000013[];
        static char const digits_1250000000000_to_1250000000013[];
        static char const digits_250000000000000_to_250000000000013[];

        // data collected from:
        // http://www.numberworld.org/digits/Pi/
        static char const digits_74951_to_75000[];
        static char const digits_99951_to_100000[];

        // data collected from:
        // http://www.jjj.de/hfloat/digits/pihex64k.gz
        static char const digits_1_to_65536[];

};

bool bbp_pi_verify(std::string const & digit, std::string const & pi)
{
    return HexadecimalPiTestData::verify(digit, pi);
}

char const HexadecimalPiTestData::digits_1000000_to_1000013[] =
    "26c65e52cb4593";
char const HexadecimalPiTestData::digits_10000000_to_10000013[] =
    "17af5863efed8d";
char const HexadecimalPiTestData::digits_100000000_to_100000013[] =
    "ecb840e21926ec";
char const HexadecimalPiTestData::digits_1000000000_to_1000000013[] =
    "85895585a0428b";
char const HexadecimalPiTestData::digits_10000000000_to_10000000013[] =
    "921c73c6838fb2";
char const HexadecimalPiTestData::digits_100000000000_to_100000000013[] =
    "9c381872d27596";
char const HexadecimalPiTestData::digits_1250000000000_to_1250000000013[] =
    "07e45733cc790b";
char const HexadecimalPiTestData::digits_250000000000000_to_250000000000013[] =
    "e6216b069cb6c1";
char const HexadecimalPiTestData::digits_74951_to_75000[] =
    "a3327fd065f0c4d23f82f48a26bb03abbd9f66e01b4f748a88";
char const HexadecimalPiTestData::digits_99951_to_100000[] =
    "443388751069558b3e62e612bc302ec487aa9a6ea22673c1a5";
char const HexadecimalPiTestData::digits_1_to_65536[] = ".\
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\
";
