%%% Copyright (C) 2011 - Anton Lebedevich.  All rights reserved.

-module(cloudi_x_epgsqli).

-export([start_link/0,
         connect/1, connect/2, connect/3, connect/4, connect/5,
         close/1,
         get_parameter/2,
         set_notice_receiver/2,
         get_cmd_status/1,
         squery/2,
         equery/2, equery/3,
         prepared_query/3,
         parse/2, parse/3, parse/4,
         describe/2, describe/3,
         bind/3, bind/4,
         execute/2, execute/3, execute/4,
         execute_batch/2,
         close/2, close/3,
         sync/1,
         cancel/1]).

-include("cloudi_x_epgsql.hrl").

%% -- client interface --

start_link() ->
    cloudi_x_epgsql_sock:start_link().

connect(Opts) ->
    {ok, C} = cloudi_x_epgsql_sock:start_link(),
    call_connect(C, Opts).

connect(Host, Opts) ->
    connect(Host, os:getenv("USER"), "", Opts).

connect(Host, Username, Opts) ->
    connect(Host, Username, "", Opts).

connect(Host, Username, Password, Opts) ->
    {ok, C} = cloudi_x_epgsql_sock:start_link(),
    connect(C, Host, Username, Password, Opts).

-spec connect(cloudi_x_epgsql:connection(), inet:ip_address() | inet:hostname(),
              string(), string(), cloudi_x_epgsql:connect_opts()) -> reference().
connect(C, Host, Username, Password, Opts) ->
    Opts1 = maps:merge(cloudi_x_epgsql:to_map(Opts),
                       #{host => Host,
                         username => Username,
                         password => Password}),
    call_connect(C, Opts1).

call_connect(C, Opts) ->
    Opts1 = cloudi_x_epgsql_cmd_connect:opts_hide_password(cloudi_x_epgsql:to_map(Opts)),
    cloudi_x_epgsqla:complete_connect(
      C, incremental(C, cloudi_x_epgsql_cmd_connect, Opts1), Opts1).


-spec close(cloudi_x_epgsql:connection()) -> ok.
close(C) ->
    cloudi_x_epgsql_sock:close(C).

-spec get_parameter(cloudi_x_epgsql:connection(), binary()) -> binary() | undefined.
get_parameter(C, Name) ->
    cloudi_x_epgsql_sock:get_parameter(C, Name).

-spec set_notice_receiver(cloudi_x_epgsql:connection(), undefined | pid() | atom()) ->
                                 {ok, Previous :: pid() | atom()}.
set_notice_receiver(C, PidOrName) ->
    cloudi_x_epgsql_sock:set_notice_receiver(C, PidOrName).

-spec get_cmd_status(cloudi_x_epgsql:connection()) -> {ok, Status} when
          Status :: undefined | atom() | {atom(), integer()}.
get_cmd_status(C) ->
    cloudi_x_epgsql_sock:get_cmd_status(C).

-spec squery(cloudi_x_epgsql:connection(), cloudi_x_epgsql:sql_query()) -> reference().
squery(C, Sql) ->
    incremental(C, cloudi_x_epgsql_cmd_squery, Sql).

-spec equery(cloudi_x_epgsql:connection(), cloudi_x_epgsql:statement()) -> reference().
equery(C, Statement) ->
    equery(C, Statement, []).

-spec equery(cloudi_x_epgsql:connection(), cloudi_x_epgsql:statement(), [cloudi_x_epgsql:typed_param()]) -> reference().
equery(C, Statement, TypedParameters) ->
    incremental(C, cloudi_x_epgsql_cmd_equery, {Statement, TypedParameters}).

-spec prepared_query(cloudi_x_epgsql:connection(), cloudi_x_epgsql:statement(), [cloudi_x_epgsql:typed_param()]) -> reference().
prepared_query(C, Statement, TypedParameters) ->
    incremental(C, cloudi_x_epgsql_cmd_prepared_query, {Statement, TypedParameters}).

-spec parse(cloudi_x_epgsql:connection(), cloudi_x_epgsql:sql_query()) -> reference().
parse(C, Sql) ->
    parse(C, "", Sql, []).

-spec parse(cloudi_x_epgsql:connection(), cloudi_x_epgsql:sql_query(), [cloudi_x_epgsql:epgsql_type()]) -> reference().
parse(C, Sql, Types) ->
    parse(C, "", Sql, Types).

-spec parse(cloudi_x_epgsql:connection(), iolist(), cloudi_x_epgsql:sql_query(), [cloudi_x_epgsql:epgsql_type()]) -> reference().
parse(C, Name, Sql, Types) ->
    incremental(C, cloudi_x_epgsql_cmd_parse, {Name, Sql, Types}).

bind(C, Statement, Parameters) ->
    bind(C, Statement, "", Parameters).

-spec bind(cloudi_x_epgsql:connection(), cloudi_x_epgsql:statement(), string(), [cloudi_x_epgsql:bind_param()]) -> reference().
bind(C, Statement, PortalName, Parameters) ->
    incremental(C, cloudi_x_epgsql_cmd_bind, {Statement, PortalName, Parameters}).

execute(C, S) ->
    execute(C, S, "", 0).

execute(C, S, N) ->
    execute(C, S, "", N).

-spec execute(cloudi_x_epgsql:connection(), cloudi_x_epgsql:statement(), string(), non_neg_integer()) -> reference().
execute(C, Statement, PortalName, MaxRows) ->
    incremental(C, cloudi_x_epgsql_cmd_execute, {Statement, PortalName, MaxRows}).

-spec execute_batch(cloudi_x_epgsql:connection(), [{cloudi_x_epgsql:statement(), [cloudi_x_epgsql:bind_param()]}]) -> reference().
execute_batch(C, Batch) ->
    incremental(C, cloudi_x_epgsql_cmd_batch, Batch).

describe(C, #statement{name = Name}) ->
    describe(C, statement, Name).

describe(C, statement, Name) ->
    incremental(C, cloudi_x_epgsql_cmd_describe_statement, Name);

describe(C, portal, Name) ->
    incremental(C, cloudi_x_epgsql_cmd_describe_portal, Name).

close(C, #statement{name = Name}) ->
    close(C, statement, Name).

close(C, Type, Name) ->
    incremental(C, cloudi_x_epgsql_cmd_close, {Type, Name}).

sync(C) ->
    incremental(C, cloudi_x_epgsql_cmd_sync, []).

-spec cancel(cloudi_x_epgsql:connection()) -> ok.
cancel(C) ->
    cloudi_x_epgsql_sock:cancel(C).


%% -- internal functions --

incremental(C, Command, Args) ->
    cloudi_x_epgsql_sock:async_command(C, incremental, Command, Args).
